<?php

/**
 * Bookings and Appointment Plugin for WooCommerce
 *
 * Class for integrating Dokan Products with Bookings & Appointment Plugin
 *
 * @author   Tyche Softwares
 * @package  BKAP/Vendors/WC-Vendors
 * @version  4.8.0
 * @category Classes
 */

/**
 * Class for listing the bookings on respective pages
 *
 * @since 4.6.0
 */
class bkap_wc_vendors {

	/**
	 * Default Constructor
	 *
	 * @since 4.6.0
	 */
	public function __construct() {

		// include the files
		add_action( 'init', array( &$this, 'bkap_wcv_include_files' ), 6 );

		// add the booking menu in NAV
		add_filter( 'wcv_pro_dashboard_urls', array( &$this, 'bkap_add_menu' ), 10, 1 );
		add_filter( 'wcv_dashboard_pages_nav', array( &$this, 'bkap_modify_menu' ), 10, 1 );

		// add the custom pages
		add_filter( 'wcv_dashboard_custom_pages', array( &$this, 'bkap_booking_menu' ), 10, 1 );

		// View Bookings Data Export
		add_action( 'wp', array( &$this, 'bkap_download_booking_files' ) );

		add_filter( 'bkap_display_multiple_modals', array( &$this, 'bkap_wc_vendors_enable_modals' ) );

		add_action( 'bkap_wc_vendors_booking_list', array( &$this, 'bkap_wc_vendors_load_modals' ), 10, 4 );

		add_action( 'admin_init', array( &$this, 'bkap_remove_menus' ) );

		add_filter( 'woocommerce_email_recipient_bkap_new_booking', array( &$this, 'bkap_change_vendor_emails' ), 10, 3 );

		add_filter( 'bkap_show_add_day_button', array( &$this, 'bkap_show_add_day_button' ), 10, 1 );


	} // construct

	/**
	 * Hide Add Day button in Edit Booking Modal on View Booking for WC Vendors
	 *
	 * @param string $status Show add day button.
	 *
	 * @since 5.7.0
	 */
	public function bkap_show_add_day_button( $status ) {

		if ( isset( $_GET['custom'] ) && 'bkap-booking' === $_GET['custom'] ) {
			$status = false;
		}
		return $status;
	}

	/**
	 * Change New Booking Email Recipient to Vendor's Email Address.
	 *
	 * @param string $recipient Recipient Email Address.
	 * @param obj    $object Shop Order Object.
	 * @param obj    $email_obj Email Object.
	 *
	 * @since 5.5.2
	 */
	public function bkap_change_vendor_emails( $recipient, $object, $email_obj ) {

		$product_id     = $object->product_id;
		$vendor_id      = get_post_field( 'post_author', $product_id );
		$user           = get_user_by( 'ID', $vendor_id );
		$send_to_vendor = apply_filters( 'bkap_confirmation_emails_to_vendor', true );
		if ( $send_to_vendor && '' !== $user->user_email && ( is_plugin_active( 'wc-vendors/class-wc-vendors.php' ) || is_plugin_active( 'dokan-pro/dokan-pro.php' ) ) ) {
			$recipient = $user->user_email;
		}

		return $recipient;
	}

	/**
	 * Include files as needed
	 *
	 * @since 4.6.0
	 */
	function bkap_wcv_include_files() {
		// product page Booking tab file
		include_once 'product.php';
	}

	/**
	 * Add the Booking menu to the Vendor dashboard
	 *
	 * @param array $pages Current Pages displayed in the menu
	 * @return array
	 * @since 4.6.0
	 */
	function bkap_add_menu( $pages ) {

		$pages['bkap-booking'] = array(
			'label'   => __( 'Bookings', 'woocommerce-booking' ),
			'slug'    => 'bkap-booking',
			'actions' => array(),
			'custom'  => 'bkap-booking',
		);

		return $pages;
	}

	/**
	 * Add the Booking menu to the Vendor dashboard
	 *
	 * @param array $pages Current Pages displayed in the menu
	 * @return array
	 */
	function bkap_modify_menu( $pages ) {

		$pages['bkap-booking'] = array(
			'label'   => __( 'Bookings', 'woocommerce-booking' ),
			'slug'    => 'bkap-booking?custom=bkap-booking',
			'id'      => 'bkap-booking',
			'actions' => array(),
			'custom'  => 'bkap-booking',
		);

		return $pages;
	}

	/**
	 * Add the templates
	 *
	 * @param array $menu Current Menu Array
	 *
	 * @return array Menu array modified to containg Booking Page details
	 * @since 4.6.0
	 */
	function bkap_booking_menu( $menu ) {
		$menu['bkap-booking'] = array(
			'slug'          => 'bkap-booking?custom=bkap-booking',
			'label'         => __( 'Bookings', 'woocommerce-booking' ),
			'template_name' => 'bkap-wcv-view-bookings',
			'base_dir'      => BKAP_BOOKINGS_TEMPLATE_PATH . 'vendors-integration/wc-vendors/',
			'args'          => array(),
			'actions'       => array(),
			'parent'        => 'bkap-booking',
		);

		$menu['calendar-view'] = array(
			'slug'          => 'bkap-booking?custom=calendar-view',
			'label'         => __( 'Bookings', 'woocommerce-booking' ),
			'template_name' => 'bkap-wcv-calendar-view',
			'base_dir'      => BKAP_BOOKINGS_TEMPLATE_PATH . 'vendors-integration/wc-vendors/',
			'args'          => array(),
			'actions'       => array(),
		);

		// Uncomment this when we make Resources compatible with Vendors
		/*
				$menu[ 'bkap-resources' ] = array(
			'slug'			=> 'bkap-booking?custom=bkap-resources',
			'label'			=> __('Bookings', 'woocommerce-booking' ),
			'template_name' => 'bkap-wcv-resources',
			'base_dir'      => BKAP_BOOKINGS_TEMPLATE_PATH . 'vendors-integration/wc-vendors/',
			'args'          => array(),
			'actions'		=> array(),
		); */

		return $menu;
	}

	/**
	 * View Bookings Data Export
	 * Print & CSV
	 *
	 * @since 4.6.0
	 */
	function bkap_download_booking_files() {

		if ( isset( $_GET['custom'] ) && ( $_GET['custom'] == 'bkap-print' || $_GET['custom'] ) == 'bkap-csv' ) {

			$current_page = $_GET['custom'];

			$additional_args = array(
				'meta_key'   => '_bkap_vendor_id',
				'meta_value' => get_current_user_id(),
			);
			$data            = bkap_common::bkap_get_bookings( '', $additional_args );

			if ( isset( $current_page ) && $current_page === 'bkap-csv' ) {
				BKAP_Bookings_View::bkap_download_csv_file( $data );
			} elseif ( isset( $current_page ) && $current_page === 'bkap-print' ) {
				BKAP_Bookings_View::bkap_download_print_file( $data );
			}
		}
	}

	/**
	 * Loads modal template for editing the bookings for a Vendor from Vendor Dashboard
	 *
	 * @param int    $booking_id Booking Post ID
	 * @param Object $booking_post WP Post Booking
	 * @since 4.6.0
	 */

	public function bkap_wc_vendors_load_modals( $booking_id, $booking_post, $booking_details, $item_no ) {

		$product_id = $booking_post['product_id'];

		$variation_id = '';

		$page_type = 'view-order';

		$localized_array = array(
			'bkap_booking_params' => $booking_details,
			'bkap_cart_item'      => '',
			'bkap_cart_item_key'  => $booking_post['order_item_id'] . $item_no,
			'bkap_order_id'       => $booking_post['order_id'],
			'bkap_page_type'      => $page_type,
			'bkap_booking_id'     => $booking_id,
		);

		// Additional data for addons
		$additional_addon_data = '';// bkap_common::bkap_get_cart_item_addon_data( $cart_item );

		bkap_edit_bookings_class::bkap_load_template(
			$booking_details,
			wc_get_product( $product_id ),
			$product_id,
			$localized_array,
			$booking_post['order_item_id'] . $item_no,
			$variation_id, // $booking_post['variation_id'],
			$additional_addon_data
		);
	}

	/**
	 * Enable Global Params to be set for Modals to load on View Bookings
	 *
	 * @param bool $display Status indicating presence of multiple products for booking
	 * @return bool True if multiple products present
	 * @since 4.6.0
	 */
	public function bkap_wc_vendors_enable_modals( $display ) {

		if ( isset( $_GET['custom'] ) && $_GET['custom'] === 'bkap-booking' ) {
			return $display = true;
		} else {
			return $display;
		}
	}

	/**
	 * Remove the booking menu from the vendor admin dashboard
	 *
	 * @since 4.6.0
	 */
	function bkap_remove_menus() {

		if ( current_user_can( 'vendor' ) && ! is_multisite() ) {
			remove_menu_page( 'edit.php?post_type=bkap_booking' );
		}
	}
} // end of class

$bkap_wc_vendors = new bkap_wc_vendors();

