<?php

/**
 * Bookings and Appointment Plugin for WooCommerce
 *
 * Class for integrating Dokan with Bookings & Appointment Plugin
 *
 * @author   Tyche Softwares
 * @package  BKAP/Vendors/Dokan
 * @version  4.6.0
 * @category Classes
 */

if ( ! class_exists( 'bkap_dokan_class' ) ) {

	/**
	 * Class for Integrating Booking & Appointment plugin for WooCommerce with Dokan
	 *
	 * @since    4.6.0
	 */
	class bkap_dokan_class {

		/**
		 * Default Constructor. Include dependency files and hook functions to actions & filters
		 *
		 * @since 4.6.0
		 */
		public function __construct() {

			self::bkap_dokan_include_dependencies();

			add_filter( 'dokan_get_dashboard_nav', array( &$this, 'bkap_dokan_add_booking_nav' ) );

			add_filter( 'dokan_query_var_filter', array( &$this, 'bkap_dokan_query_var_filter' ) );

			add_action( 'dokan_rewrite_rules_loaded', array( $this, 'bkap_add_rewrite_rules' ) );

			add_action( 'dokan_load_custom_template', array( &$this, 'bkap_dokan_include_template' ), 10, 1 );

			add_action( 'admin_init', array( &$this, 'bkap_remove_menu' ) );

			// Allow vendors to edit global holidays for their products.
			add_action( 'wp_ajax_bkap_dokan_vendor_global_availability', array( &$this, 'bkap_dokan_vendor_global_availability' ) );
			add_action( 'wp_ajax_bkap_dokan_display_availability', array( &$this, 'bkap_dokan_display_availability' ) );
			add_action( 'wp_ajax_bkap_dokan_vendor_global_availability_delete', array( &$this, 'bkap_dokan_vendor_global_availability_delete' ) );
		}

		/**
		 * Remove Booking Menu for Vendors Admin Dashboard
		 *
		 * @since 4.6.0
		 */
		public function bkap_remove_menu() {

			if ( current_user_can( 'seller' ) ) {
				remove_menu_page( 'edit.php?post_type=bkap_booking' );
			}
		}

		/**
		 * Include dependent files
		 *
		 * @since 4.6.0
		 */
		public static function bkap_dokan_include_dependencies() {

			include_once BKAP_VENDORS_INCLUDES_PATH . 'dokan/class-bkap-dokan-products.php';
			include_once BKAP_VENDORS_INCLUDES_PATH . 'dokan/class-bkap-dokan-orders.php';
			include_once BKAP_VENDORS_INCLUDES_PATH . 'dokan/class-bkap-dokan-calendar.php';
		}

		/**
		 * Add Booking Menu to vendors Dashboard on Frontend
		 *
		 * @param array $urls Array containing existing Menu URLs.
		 * @return array URL Array.
		 * @since 4.6.0
		 */
		public function bkap_dokan_add_booking_nav( $urls ) {

			$urls['bkap_dokan_booking'] = array(
				'title' => __( 'Booking', 'woocommerce-booking' ),
				'icon'  => '<i class="wp-menu-image dashicons-before dashicons-calendar-alt"></i>',
				'url'   => dokan_get_navigation_url( 'bkap_dokan_booking' ),
				'pos'   => '51',
			);

			return $urls;
		}

		/**
		 * Add Booking Query var to the existing query vars
		 *
		 * @param array $url_array Array of URL.
		 * @return array Array of URL after modification.
		 * @since 4.6.0
		 */
		public function bkap_dokan_query_var_filter( $url_array ) {
			$url_array[] = 'bkap_dokan_booking';

			return $url_array;
		}

		/**
		 * Add rewrite rules for Booking Links
		 *
		 * @since 4.6.0
		 */
		public function bkap_add_rewrite_rules() {
			flush_rewrite_rules( true );
		}

		/**
		 * Display the base template for Booking Menu
		 *
		 * @param array $query_vars Query Vars.
		 * @since 4.6.0
		 */
		public function bkap_dokan_include_template( $query_vars ) {

			if ( isset( $query_vars['bkap_dokan_booking'] ) ) {

				$plugin_version_number = get_option( 'woocommerce_booking_db_version' );
				$ajax_url              = get_admin_url() . 'admin-ajax.php';

				wc_get_template(
					'dokan/bkap-dokan-vendor-global-availability.php',
					array(),
					'woocommerce-booking/',
					BKAP_VENDORS_TEMPLATE_PATH
				);

				wc_get_template(
					'dokan/bkap-dokan-booking.php',
					array(),
					'woocommerce-booking/',
					BKAP_VENDORS_TEMPLATE_PATH
				);

				// Include the JS & CSS files that we need.
				bkap_load_scripts_class::bkap_load_dokan_calendar_view_scripts_css();
				bkap_load_scripts_class::bkap_load_dokan_calendar_view_scripts_js( $plugin_version_number, $ajax_url );
			}
		}

		/**
		 * Display the Availability for Vendors in Booking->Calendar View.
		 *
		 * @since 5.0.0
		 */
		public function bkap_dokan_display_availability() {
			ob_start();
			wc_get_template(
				'dokan/bkap-dokan-vendor-global-availability.php',
				array(),
				'woocommerce-booking/',
				BKAP_VENDORS_TEMPLATE_PATH
			);
			ob_flush();
		}

		/**
		 * Allow the vendor to delete global holidays for their products
		 * from Booking->Calendar View.
		 *
		 * @since 5.0.0
		 */
		public function bkap_dokan_vendor_global_availability_delete() {

			$id = isset( $_POST['id'] ) ? sanitize_text_field( wp_unslash( $_POST['id'] ) ) : 0; // phpcs:ignore WordPress.Security.NonceVerification

			if ( $id > 0 ) {

				global $wpdb;

				// Get the Vendor ID.
				$vendor_id = get_current_user_id();

				// update the User meta record.
				$existing_vendor_data = get_user_meta( $vendor_id, '_bkap_vendor_holidays', true );

				if ( is_array( $existing_vendor_data ) && count( $existing_vendor_data ) > 0 ) {
					foreach ( $existing_vendor_data as $availability_id => $vendor_availability ) {
						if ( $id == $vendor_availability['id'] ) {
							unset( $existing_vendor_data[ $availability_id ] );
							break;
						}
					}
				}
				update_user_meta( $vendor_id, '_bkap_vendor_holidays', $existing_vendor_data );

				// update the data for all the products.
				$get_product_ids = $wpdb->get_col( // phpcs:ignore
					$wpdb->prepare(
						'SELECT ID FROM `' . $wpdb->prefix . 'posts` WHERE post_author = %d AND post_type = %s',
						$vendor_id,
						'product'
					)
				);

				if ( is_array( $get_product_ids ) && count( $get_product_ids ) > 0 ) {
					foreach ( $get_product_ids as $product_id ) {

						// If the product is bookable.
						if ( bkap_common::bkap_get_bookable_status( $product_id ) ) {
							$get_existing = get_post_meta( $product_id, '_bkap_holiday_ranges', true );

							if ( is_array( $get_existing ) && count( $get_existing ) > 0 ) {
								foreach ( $get_existing as $existing_prd_id => $existing_prd_data ) {
									if ( $id == $existing_prd_data['id'] ) {
										unset( $get_existing[ $existing_prd_id ] );
										break;
									}
								}

								update_post_meta( $product_id, '_bkap_holiday_ranges', $get_existing );
							}
						}
					}
				}
			}
			die();
		}

		/**
		 * Add/Edit global availability for Vendors from Booking->Calendar View.
		 *
		 * @since 5.0.0
		 */
		public function bkap_dokan_vendor_global_availability() {

			global $wpdb;

			$id = 0;

			$start_date  = isset( $_POST['start_date'] ) ? sanitize_text_field( wp_unslash( $_POST['start_date'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification
			$end_date    = isset( $_POST['end_date'] ) ? sanitize_text_field( wp_unslash( $_POST['end_date'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification
			$update_type = isset( $_POST['update_type'] ) ? sanitize_text_field( wp_unslash( $_POST['update_type'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification
			$title       = isset( $_POST['title'] ) ? sanitize_text_field( wp_unslash( $_POST['title'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification

			// Get the Vendor ID.
			$vendor_id = get_current_user_id();
			// update the User Meta record.
			$existing_vendor_data = get_user_meta( $vendor_id, '_bkap_vendor_holidays', true );

			if ( 'edit' === $update_type ) {
				$id = isset( $_POST['update_id'] ) ? sanitize_text_field( wp_unslash( $_POST['update_id'] ) ) : 0; // phpcs:ignore WordPress.Security.NonceVerification
			} else {
				// get the new ID.
				if ( is_array( $existing_vendor_data ) && count( $existing_vendor_data ) > 0 ) {
					$id = count( $existing_vendor_data ) + 1;
				} else {
					$existing_vendor_data = array();
					$id                   = 1;
				}
			}

			// Verify that we have the start & end date & ID > 0.
			if ( '' === $start_date || '' === $end_date || 0 === $id ) {
				echo 'failed';
				die();
			}

			// Format the start & end.
			$new_range = array(
				'id'             => $id,
				'start'          => date( 'j-n-Y', strtotime( $start_date ) ), //phpcs:ignore
				'end'            => date( 'j-n-Y', strtotime( $end_date ) ), //phpcs:ignore
				'years_to_recur' => '',
				'range_type'     => 'custom_range',
				'range_name'     => $title,
			);

			$get_product_ids = $wpdb->get_col( // phpcs:ignore
				$wpdb->prepare(
					'SELECT ID FROM `' . $wpdb->prefix . 'posts` WHERE post_author = %d AND post_type = %s',
					$vendor_id,
					'product'
				)
			);

			$modified = false;
			if ( is_array( $get_product_ids ) && count( $get_product_ids ) > 0 ) {
				foreach ( $get_product_ids as $product_id ) {

					// If the product is bookable.
					if ( bkap_common::bkap_get_bookable_status( $product_id ) ) {
						$get_existing = get_post_meta( $product_id, '_bkap_holiday_ranges', true );

						if ( ! is_array( $get_existing ) ) {
							$get_existing = array();
						}

						// If we're editing, modify an existing entry.
						if ( 'edit' === $update_type ) {

							foreach ( $get_existing as $existing_prd_id => $existing_prd_data ) {
								if ( $id == $existing_prd_data['id'] ) {
									$get_existing[ $existing_prd_id ] = $new_range;
									break;
								}
							}
						} else { // else add a new entry.
							array_push( $get_existing, $new_range );
						}

						update_post_meta( $product_id, '_bkap_holiday_ranges', $get_existing );
						$modified = true;
					}
				}
			}

			// Update the User Meta record.
			if ( $modified ) {

				if ( 'edit' === $update_type ) { // update the existing entry.
					foreach ( $existing_vendor_data as $availability_id => $vendor_availability ) {
						if ( $id == $vendor_availability['id'] ) {
							$existing_vendor_data[ $availability_id ] = $new_range;
							break;
						}
					}
				} else { // create a new record.
					array_push( $existing_vendor_data, $new_range );
				}

				update_user_meta( $vendor_id, '_bkap_vendor_holidays', $existing_vendor_data );

				echo 'success';
			}
			die();
		}
	}
}

return new bkap_dokan_class();
