<?php
/**
 * Bookings and Appointment Plugin for WooCommerce
 *
 * Menu page for sending manual reminder emails and setting automatic reminders for bookings
 *
 * @author      Tyche Softwares
 * @package     BKAP/Menus
 * @since       2.0
 * @category    Classes
 */

if ( ! class_exists( 'Bkap_Send_Reminder' ) ) {

	/**
	 * Class Bkap_Send_Reminder
	 */
	class Bkap_Send_Reminder {

		/**
		 * Constructor
		 */
		public function __construct() {
			add_action( 'admin_init', array( $this, 'bkap_send_automatic_reminder' ), 10 );
			add_filter( 'woocommerce_screen_ids', array( $this, 'bkap_add_screen_id' ) );
			add_action( 'bkap_auto_reminder_emails', array( $this, 'bkap_send_auto_reminder_emails' ) );
		}

		/**
		 * Settings screen id for Send reminder page.
		 *
		 * @param array $screen_ids Array of Screen Ids.
		 * @since 4.10.0
		 */
		public static function bkap_add_screen_id( $screen_ids ) {

			$screen_ids[] = 'bkap_booking_page_booking_reminder_page';
			return $screen_ids;
		}

		/**
		 * Callback for Reminder Settings section.
		 */
		public static function bkap_reminder_settings_section_callback() {}

		/**
		 * Add a page in the Bookings menu to send reminder emails
		 *
		 * @since 4.10.0
		 */
		public static function bkap_add_reminder_page() {

			$format       = bkap_common::bkap_get_date_format();
			$current_date = date( $format ); // phpcs:ignore

			if ( ! empty( $_POST ) ) {
				$product_ids = isset( $_POST['bkap_reminder_product_id'] ) && '' != $_POST['bkap_reminder_product_id'] ? $_POST['bkap_reminder_product_id'] : '';
				$booking_ids = isset( $_POST['bkap_reminder_booking_id'] ) && '' != $_POST['bkap_reminder_booking_id'] ? $_POST['bkap_reminder_booking_id'] : '';
				$order_ids   = isset( $_POST['bkap_reminder_order_id'] ) && '' != $_POST['bkap_reminder_order_id'] ? $_POST['bkap_reminder_order_id'] : '';

				$subject  = isset( $_POST['bkap_reminder_subject'] ) && '' != $_POST['bkap_reminder_subject'] ? $_POST['bkap_reminder_subject'] : 'Booking Reminder';
				$message  = isset( $_POST['bkap_reminder_message'] ) && '' != $_POST['bkap_reminder_message'] ? $_POST['bkap_reminder_message'] : '';
				$mailer   = WC()->mailer();
				$reminder = $mailer->emails['BKAP_Email_Booking_Reminder'];
				$success  = __( 'Reminder sent successfully', 'woocommerce-booking' );

				if ( is_array( $product_ids ) && ! empty( $product_ids ) ) {

					foreach ( $product_ids as $product_id ) {

						$bookings = bkap_common::bkap_get_bookings_by_product( $product_id );
						foreach ( $bookings as $id => $booking ) {
							$reminder->trigger( $booking->get_item_id(), $subject, $message );
							echo '<div class="updated fade"><p>' . $success . '</p></div>'; // phpcs:ignore
						}
					}
				}

				if ( is_array( $booking_ids ) && ! empty( $booking_ids ) ) {

					foreach ( $booking_ids as $booking_id ) {
						$booking    = new BKAP_Booking( $booking_id );
						$start_date = $booking->get_start_date();

						if ( strtotime( $start_date ) > strtotime( $current_date ) ) {
							$reminder->trigger( $booking->get_item_id(), $subject, $message );
							echo '<div class="updated fade"><p>' . $success . '</p></div>'; // phpcs:ignore
						}
					}
				}

				if ( is_array( $order_ids ) && ! empty( $order_ids ) ) {

					foreach ( $order_ids as $order_id ) {
						$order_bookings = bkap_common::get_booking_ids_from_order_id( $order_id );

						foreach ( $order_bookings as $booking_id ) {
							$booking    = new BKAP_Booking( $booking_id );
							$start_date = $booking->get_start_date();

							if ( strtotime( $start_date ) > strtotime( $current_date ) ) {
								$reminder->trigger( $booking->get_item_id(), $subject, $message );
								echo '<div class="updated fade"><p>' . $success . '</p></div>'; // phpcs:ignore
							}
						}
					}
				}
			}

			$bookable_products = bkap_common::ts_get_all_bookable_products();
			$all_booking_ids   = array();
			$bookings          = bkap_common::bkap_get_bookings( array( 'paid', 'confirmed' ) );

			foreach ( $bookings as $key => $value ) {
				array_push( $all_booking_ids, $value->get_id() );
			}

			$all_order_ids = bkap_common::bkap_get_orders_with_bookings();

			wc_get_template(
				'bkap-reminder-email-view.php',
				array(
					'bookable_products' => $bookable_products,
					'booking_ids'       => $all_booking_ids,
					'order_ids'         => array_unique( $all_order_ids ),
				),
				'woocommerce-booking/',
				BKAP_BOOKINGS_TEMPLATE_PATH
			);

			do_action( 'bkap_sms_reminder_settings' );
		}

		/**
		 * Add a setting for automatic reminders to set the number of days
		 *
		 * @since 4.10.0
		 */
		public static function bkap_send_automatic_reminder() {

			add_settings_section(
				'bkap_reminder_section',
				'',
				array( 'Bkap_Send_Reminder', 'bkap_reminder_settings_section_callback' ),
				'booking_reminder_page'
			);

			add_settings_field(
				'reminder_email_before_days',
				__( 'Number of days for reminder before booking date', 'woocommerce-booking' ),
				array( 'Bkap_Send_Reminder', 'reminder_email_before_days_callback' ),
				'booking_reminder_page',
				'bkap_reminder_section',
				array( __( 'Send the reminder email X number of days before booking date', 'woocommerce-booking' ) )
			);

			register_setting(
				'bkap_reminder_settings',
				'bkap_reminder_settings',
				array( 'Bkap_Send_Reminder', 'bkap_reminder_settings_callback' )
			);
		}

		/**
		 * Callback function for the automatic reminder settings
		 *
		 * @param array $args Argument Array.
		 * @since 4.10.0
		 */
		public static function reminder_email_before_days_callback( $args ) {

			$saved_settings = json_decode( get_option( 'bkap_reminder_settings' ) );
			$number_of_days = 0;

			if ( isset( $saved_settings->reminder_email_before_days ) && 0 !== (int) $saved_settings->reminder_email_before_days ) {
				$number_of_days = $saved_settings->reminder_email_before_days;

				if ( $number_of_days > 0 ) {
					if ( ! wp_next_scheduled( 'bkap_auto_reminder_emails' ) ) {
						wp_schedule_event( time(), 'daily', 'bkap_auto_reminder_emails' );
					}
				} else {
					wp_clear_scheduled_hook( 'bkap_auto_reminder_emails' );
				}
			}

			echo '<input type="number" name="bkap_reminder_settings[reminder_email_before_days]" id="reminder_email_before_days" value="' . esc_attr( $number_of_days ) . '"/>';
			$html = '<label for="reminder_email_before_days"> ' . $args[0] . '</label>';
			echo $html; // phpcs:ignore
		}

		/**
		 * Save the Automatic reminder Settings
		 *
		 * @param array $input - Settings on the page.
		 * @return string $bkap_reminder_settings - JSON.
		 * @since 4.10.0
		 */
		public static function bkap_reminder_settings_callback( $input ) {
			$bkap_reminder_settings_callback = wp_json_encode( $input );
			return $bkap_reminder_settings_callback;
		}

		/**
		 * Scheduled event for the automatic reminder emails
		 *
		 * @since 4.10.0
		 */
		public static function bkap_send_auto_reminder_emails() {

			$booking_posts     = bkap_get_future_bookings();
			$current_date      = date( 'Y-m-d' ); // phpcs:ignore
			$current_date_time = strtotime( $current_date );
			$saved_settings    = json_decode( get_option( 'bkap_reminder_settings' ) );
			$number_of_days    = $saved_settings->reminder_email_before_days;
			$number_of_days    = absint( $number_of_days );
			$mailer            = WC()->mailer();
			$reminder          = $mailer->emails['BKAP_Email_Booking_Reminder'];
			$twilio_details    = bkap_get_sms_settings(); // Getting SMS settings.

			foreach ( $booking_posts as $key => $value ) {
				$booking      = new BKAP_Booking( $value->ID );
				$booking_date = strtotime( date( 'Y-m-d', strtotime( $booking->get_start() ) ) ); // phpcs:ignore
				$interval     = ( $booking_date - $current_date_time ); // booking date - current date time.
				if ( $interval == absint( $number_of_days * 86400 ) ) { // phpcs:ignore
					$item_id = $booking->get_item_id();
					$reminder->trigger( $item_id );
					if ( is_array( $twilio_details ) ) {
						Bkap_SMS_settings::bkap_send_automatic_sms_reminder( $booking, $twilio_details, $item_id );
					}
				}
			}
		}

		/**
		 * Ajax call for the Send Reminder action on Edit Booking page
		 *
		 * @since 4.10.0
		 */
		public static function bkap_send_reminder_action() {

			$booking_id = $_POST['booking_id']; // phpcs:ignore
			$booking    = new BKAP_Booking( $booking_id );
			$item_id    = $booking->get_item_id();
			$mailer     = WC()->mailer();
			$reminder   = $mailer->emails['BKAP_Email_Booking_Reminder'];

			$reminder->trigger( $item_id );

			$twilio_details = bkap_get_sms_settings(); // Getting SMS settings.
			if ( is_array( $twilio_details ) ) {
				Bkap_SMS_settings::bkap_send_automatic_sms_reminder( $booking, $twilio_details, $item_id );
			}
			wp_die();
		}

		/**
		 * Ajax call for saving the email draft on Manual Reminder page
		 *
		 * @since 4.10.0
		 */
		public static function bkap_save_reminder_message() {

			$message = $_POST['message']; // phpcs:ignore
			$subject = $_POST['subject']; // phpcs:ignore

			if ( isset( $message ) && '' !== $message ) {
				update_option( 'reminder_message', $message );
			}

			if ( isset( $subject ) && '' !== $subject ) {
				update_option( 'reminder_subject', $subject );
			}
		}
	}
	new Bkap_Send_Reminder();
}
