<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class phive_royalmail_woocommerce_shipping_method extends WC_Shipping_Method {
	
	/**
	 * @var $default_boxes
	 */
	private $default_boxes;

	/**
	 * @var $found_rates
	 */
	private $found_rates;

	/**
	 * @var $services
	 */
	private $services;

	/**
	 * @var $debug
	 */
	public $debug;

	/**
	 * @var $packing_method
	 */
	public $packing_method;

	/**
	 * @var $conversion_rate
	 */
	public $conversion_rate;

	/**
	 * @var $boxes
	 */
	public $boxes;

	/**
	 * @var $custom_services
	 */
	public $custom_services;

	/**
	 * @var $offer_rates
	 */
	public $offer_rates;

	/**
	 * @var $box_max_weight
	 */
	public $box_max_weight;

	/**
	 * @var $weight_pack_process
	 */
	public $weight_pack_process;

	/**
	 * @var $dimension_unit;
	 */
	public $dimension_unit;

	/**
	 * @var $weight_unit;
	 */
	public $weight_unit;

	/**
	 * @var $labelapi_dimension_unit
	 */
	public $labelapi_dimension_unit;

	/**
	 * @var $labelapi_weight_unit
	 */
	public $labelapi_weight_unit;

	/**
	 * @var $min_amount
	 */
	public $min_amount;

	/**
	 * @var $royalmail_currency
	 */
	public $royalmail_currency;

	/**
	 * @var $ordered_services
	 */
	public $ordered_services;


	public function __construct() {
		$this->id					= PHive_Royal_Mail_ID;

		$this->method_title			= __( 'Royal Mail', 'ph-shipping-royalmail' );
		$this->method_description 	= __( 'Obtains Royal Mail Rates', 'ph-shipping-royalmail' );
		$this->init();
		$this->services				= array_merge( include( 'royalmail/data-rates.php' ), include( 'parcelforce/data-rates.php' ) ) ;
	}


	
	private function init() {
		// Load the settings.
		$this->init_form_fields();
		$this->init_settings();

		$this->enabled				= isset( $this->settings['enabled'] ) ? $this->settings['enabled'] : $this->enabled;
		$this->title				= $this->get_option( 'title', $this->method_title );
		$this->availability			= isset( $this->settings['availability'] ) ? $this->settings['availability'] : 'all';
		$this->countries			= isset( $this->settings['countries'] ) ? $this->settings['countries'] : array();
		$this->debug				= ( $bool = $this->get_option( 'debug' ) ) && $bool == 'yes' ? true : false;
		
		$this->packing_method		= $this->get_option( 'packing_method', 'per_item' );
		$this->conversion_rate		= ! empty( $this->settings['conversion_rate'] ) ? $this->settings['conversion_rate'] : '';
		$this->boxes				= $this->get_option( 'boxes', array( ));
		$this->custom_services		= $this->get_option( 'services', array( ));
		$this->offer_rates			= $this->get_option( 'offer_rates', 'all' );
		$this->box_max_weight		= $this->get_option( 'box_max_weight' );
		$this->weight_pack_process	= $this->get_option( 'weight_pack_process' );
		
		
		$this->dimension_unit		= 'cm';
		$this->weight_unit			= 'kg';
		$this->labelapi_dimension_unit	= 'CM';
		$this->labelapi_weight_unit = 'KG';
		// $this->default_boxes		= include( 'data-ph-box-sizes-cm.php' );
		// $default_boxes		= include( 'royalmail/data-boxes.php' );
		$default_boxes = include('royalmail/data-boxes.php');
		$this->default_boxes = array_merge($default_boxes['domestic'], $default_boxes['international']);

		$this->min_amount	  		= isset( $this->settings['min_amount'] ) ? $this->settings['min_amount'] : 0;

		$this->royalmail_currency 	= apply_filters('ph_change_woocommerce_currency_for_multicurrency', get_woocommerce_currency() );

		if( empty($this->conversion_rate) )
		{
			$this->conversion_rate 	= 1;
			$this->conversion_rate 	= apply_filters('ph_change_conversion_rate_for_multicurrency', $this->conversion_rate, $this->royalmail_currency );
		}

		add_action( 'woocommerce_update_options_shipping_' . $this->id, array( $this, 'process_admin_options' ) );
	}

	/**
	 * is_available function.
	 *
	 * @param array $package
	 * @return bool
	 */
	public function is_available( $package ) {
		if ( "no" === $this->enabled ) {
			return false;
		}

		if ( 'specific' === $this->availability ) {
			if ( is_array( $this->countries ) && ! in_array( $package['destination']['country'], $this->countries ) ) {
				return false;
			}
		} elseif ( 'excluding' === $this->availability ) {
			if ( is_array( $this->countries ) && ( in_array( $package['destination']['country'], $this->countries ) || ! $package['destination']['country'] ) ) {
				return false;
			}
		}
		
		$has_met_min_amount = false;
		
		if(!method_exists(WC()->cart, 'get_displayed_subtotal')){// WC version below 2.6
			$total = WC()->cart->subtotal;
		}else{

			$total = WC()->cart->get_displayed_subtotal();

			if( version_compare( WC()->version, '4.4', '<' ) ) {
				$tax_display 	= WC()->cart->tax_display_cart;
			} else {
				$tax_display 	= WC()->cart->get_tax_price_display_mode();
			}

			if ( 'incl' === $tax_display ) {
				$total = $total - ( WC()->cart->get_cart_discount_total() + WC()->cart->get_cart_discount_tax_total() );
			} else {
				$total = $total - WC()->cart->get_cart_discount_total();
			}
		}
		
		if( $total < 0 )
		{		
			$total = 0;
		}

		if ( $total >= $this->min_amount ) {
			$has_met_min_amount = true;
		}
		$is_available	= $has_met_min_amount;
		
		return apply_filters( 'woocommerce_shipping_' . $this->id . '_is_available', $is_available, $package );
	}

	public function init_form_fields() {
		$this->form_fields  = include( 'data-ph-settings.php' );
	}

	public function generate_services_html() {
		ob_start();
		include( 'html-ph-services.php' );
		return ob_get_clean();
	}

	public function generate_box_packing_html() {
		ob_start();
		include( 'html-ph-box-packing.php' );
		return ob_get_clean();
	}

	public function validate_box_packing_field( $key ) {
		$box_type	 		= isset( $_POST['box_type'] ) ? $_POST['box_type'] : array();
		$boxes_name			 = isset( $_POST['boxes_name'] ) ? $_POST['boxes_name'] : array();
		$boxes_length	 	= isset( $_POST['boxes_length'] ) ? $_POST['boxes_length'] : array();
		$boxes_width	  	= isset( $_POST['boxes_width'] ) ? $_POST['boxes_width'] : array();
		$boxes_height	 	= isset( $_POST['boxes_height'] ) ? $_POST['boxes_height'] : array();
		
		$boxes_max_weight 	= isset( $_POST['boxes_max_weight'] ) ? $_POST['boxes_max_weight'] :  array();
		$boxes_enabled		= isset( $_POST['boxes_enabled'] ) ? $_POST['boxes_enabled'] : array();

		$boxes = array();
		if ( ! empty( $boxes_length ) && sizeof( $boxes_length ) > 0 ) {
			for ( $i = 0; $i <= max( array_keys( $boxes_length ) ); $i ++ ) {

				if ( ! isset( $boxes_length[ $i ] ) )
					continue;

				// if ( $boxes_length[ $i ] && $boxes_width[ $i ] && $boxes_height[ $i ] ) {

					$boxes[] = array(
						'box_type'	=> isset( $box_type[ $i ] ) ? $box_type[ $i ] : '',
						'name'		=> strval($boxes_name[$i]),
						'length'	=> floatval( $boxes_length[ $i ] ),
						'width'		=> floatval( $boxes_width[ $i ] ),
						'height'	=> floatval( $boxes_height[ $i ] ),

						/* Old version compatibility: If inner dimensions are not provided, assume outer dimensions as inner.*/
						'inner_length'	=> isset( $boxes_inner_length[ $i ] ) ? floatval( $boxes_inner_length[ $i ] ) : floatval( $boxes_length[ $i ] ),
						'inner_width'	=> isset( $boxes_inner_width[ $i ] ) ? floatval( $boxes_inner_width[ $i ] ) : floatval( $boxes_width[ $i ] ), 
						'inner_height'	=> isset( $boxes_inner_height[ $i ] ) ? floatval( $boxes_inner_height[ $i ] ) : floatval( $boxes_height[ $i ] ),
						
						'box_weight'	=> floatval( isset($boxes_box_weight[ $i ]) && !empty($boxes_box_weight[ $i ] ? $boxes_box_weight[ $i ] : 0) ),
						'max_weight'	=> floatval( $boxes_max_weight[ $i ] ),
						'enabled'	=> isset( $boxes_enabled[ $i ] ) ? true : false
					);
				// }
			}
		}
		foreach ( $this->default_boxes as $box ) {
			$boxes[ $box['id'] ] = array(
				'enabled' => isset( $boxes_enabled[ $box['id'] ] ) ? true : false
			);
		}
		return $boxes;
	}
	
	public function validate_services_field( $key ) {
		$services		 = array();
		$posted_services  = isset($_POST['royalmail_service']) ? $_POST['royalmail_service'] : array();

		foreach ( $posted_services as $code => $settings ) {
			$services[ $code ] = array(
				'name'			   => wc_clean( $settings['name'] ),
				'order'			  => wc_clean( $settings['order'] ),
				'enabled'			=> isset( $settings['enabled'] ) ? true : false,
				'adjustment'		 => wc_clean( $settings['adjustment'] ),
				'adjustment_percent' => str_replace( '%', '', wc_clean( $settings['adjustment_percent'] ) )
			);
		}

		return $services;
	}

	public function phive_do_packing( $package, $packing_method='' ) {
		if( empty($packing_method) ){
			$packing_method = $this->packing_method;
		}

		switch ( $packing_method ) {
			case 'box_packing' :
				$phive_packages = $this->box_shipping( $package );
			break;
			case 'weight_based' :
				$phive_packages = $this->weight_based_shipping( $package );
			break;
			case 'per_item' :
			default :
				$phive_packages = $this->per_item_shipping( $package );
			break;
		}
		return $phive_packages;
	}

	private function per_item_shipping( $package ) {
		$to_ship  = array();
		$group_id = 1;

		// Get weight of order
		foreach ( $package['contents'] as $item_id => $values ) {
			$values['data'] = $this->phive_load_product( $values['data'] );
			if ( ! $values['data']->needs_shipping() ) {
				$this->debug( sprintf( __( 'Product # is virtual. Skipping.', 'ph-shipping-royalmail' ), $item_id ), 'error' );
				continue;
			}
			
			$skip_product = apply_filters('phive_shipping_skip_product',false, $values, $package['contents']);
			if($skip_product){
				continue;
			}

			if ( ! $values['data']->get_weight() ) {
				$this->debug( sprintf( __( 'Product # is missing weight. Aborting.', 'ph-shipping-royalmail' ), $item_id ), 'error' );
				return;
			}

			$group = array();

			$group = array(
				'weight'		=> array(
					'value' => $this->round_up( wc_get_weight( $values['data']->get_weight(), $this->weight_unit ), 2 ),
					'units' => $this->labelapi_weight_unit,
				),
				'cost' => array(
					'amount'   => round( ( $values['data']->get_price() / $this->conversion_rate ), 2 ),
					'currency' => $this->royalmail_currency,
				),
				'packed_products' => array( $values['data'] )
			);

			$custom_field_index=1;

			$custom_field = '';
			while( $custom_field_index==1 || !empty( $custom_field ) ){
				$length = $width = $height = $weight =0;
				if($values['data']->get_type()=='simple'){
					$length = $custom_field_index == 1 ? $values['data']->get_length() : get_post_meta( $values['data']->id, '_length'.$custom_field_index, true );
					$height = $custom_field_index == 1 ? $values['data']->get_height() : get_post_meta( $values['data']->id, '_height'.$custom_field_index, true );
					$width 	= $custom_field_index == 1 ? $values['data']->get_width()  : get_post_meta( $values['data']->id, '_width' .$custom_field_index, true );
					$weight = $custom_field_index == 1 ? $values['data']->get_weight() : get_post_meta( $values['data']->id, '_weight'.$custom_field_index, true );
				}

				$dimensions = array( 
					!empty($length) ? $length : $values['data']->get_length(),
					!empty($width)  ? $width  : $values['data']->get_width(),
					!empty($height) ? $height : $values['data']->get_height()
				);
				if ( $dimensions[0] && $dimensions[1] && $dimensions[2] ) {
					sort( $dimensions );
	
					$group['dimensions'] = array(
						'length' => max( 1, round( wc_get_dimension( $dimensions[2], $this->dimension_unit ), 0 ) ),
						'width'  => max( 1, round( wc_get_dimension( $dimensions[1], $this->dimension_unit ), 0 ) ),
						'height' => max( 1, round( wc_get_dimension( $dimensions[0], $this->dimension_unit ), 0 ) ),
						'units'  => $this->labelapi_dimension_unit
					);
					$group['weight'] = array(
						'value' => $this->round_up( wc_get_weight( !empty($weight) ? $weight : $values['data']->get_weight(), $this->weight_unit ), 2 ),
						'units' => $this->labelapi_weight_unit
					);
				}
				$custom_field_index++;
				$custom_field = get_post_meta( $values['data']->id, '_weight'.$custom_field_index, true );

				for ( $i=0; $i < $values['quantity'] ; $i++){
					$to_ship[] = $group;
				}

				$group_id++;
			}
		}
		return $to_ship;
	}


	private function round_up( $value, $precision=2 ) { 
		$pow = pow ( 10, $precision ); 
		return ( ceil ( $pow * $value ) + ceil ( $pow * $value - ceil ( $pow * $value ) ) ) / $pow; 
	}
	
	private function box_shipping( $package ) {
		if ( ! class_exists( 'PH_Boxpack' ) ) {
			include_once 'class-ph-packing.php';
		}

		$boxpack = new PH_Boxpack();

		// Merge default boxes
		foreach ( $this->default_boxes as $key => $box ) {
			$box['enabled'] = isset( $this->boxes[ $box['id'] ]['enabled'] ) ? $this->boxes[ $box['id'] ]['enabled'] : true;
			$this->boxes[] = $box;
		}
		
		$special_boxes = array();
		// Define boxes
		foreach ( $this->boxes as $key => $box ) {
			if ( ! is_numeric( $key ) ) {
				continue;
			}

			if ( ! $box['enabled'] ) {
				continue;
			}

			if( empty($box['length']) || empty($box['width']) || empty($box['height']) ){
				$special_boxes[] = $box;
				continue;
			}

			$newbox = $boxpack->add_box( $box['length'], $box['width'], $box['height'], 0 );

			if ( isset( $box['id'] ) ) {
				$newbox->set_id( current( explode( ':', $box['id'] ) ) );
			}

			if ( $box['max_weight'] ) {
				$newbox->set_max_weight( $box['max_weight'] );
			}
		}

		// Add items
		foreach ( $package['contents'] as $item_id => $values ) {
			$values['data'] = $this->phive_load_product( $values['data'] );
			if ( ! $values['data']->needs_shipping() ) {
				$this->debug( sprintf( __( 'Product # is virtual. Skipping.', 'ph-shipping-royalmail' ), $item_id ), 'error' );
				continue;
			}
			
			$skip_product = apply_filters('phive_shipping_skip_product',false, $values, $package['contents']);
			if($skip_product){
				continue;
			}

			$custom_field_index=1;
			$custom_field = '';
			
			while($custom_field_index==1 || !empty( $custom_field ) ){
				$length = $width = $height = $weight = 0;
				if($values['data']->get_type()=='simple'){
					$length = $custom_field_index == 1 ? $values['data']->get_length() : get_post_meta( $values['data']->id, '_length'.$custom_field_index, true );
					$height = $custom_field_index == 1 ? $values['data']->get_height() : get_post_meta( $values['data']->id, '_height'.$custom_field_index, true );
					$width 	= $custom_field_index == 1 ? $values['data']->get_width()  : get_post_meta( $values['data']->id, '_width' .$custom_field_index, true );
					$weight = $custom_field_index == 1 ? $values['data']->get_weight() : get_post_meta( $values['data']->id, '_weight'.$custom_field_index, true );
				}
				$dimensions = array( 
					!empty($length) ? $length : $values['data']->get_length(),
					!empty($width)  ? $width  : $values['data']->get_width(),
					!empty($height) ? $height : $values['data']->get_height()
				);
				if( empty($weight) ){
					$weight = $values['data']->get_weight();
				}

				if ( $dimensions[0] && $dimensions[1] && $dimensions[2] && $weight ) {	
					for ( $i = 0; $i < $values['quantity']; $i ++ ) {
						$boxpack->add_item(
							wc_get_dimension( $dimensions[2], $this->dimension_unit ),
							wc_get_dimension( $dimensions[1], $this->dimension_unit ),
							wc_get_dimension( $dimensions[0], $this->dimension_unit ),
							round(wc_get_weight( $weight, $this->weight_unit ),4),
							$values['data']->get_price(),
							array(
								'data' => $values['data']
							)
						);
					}
	
				} else {
					$this->debug( sprintf( __( 'Product #%s is missing dimensions. Aborting.', 'ph-shipping-royalmail' ), $item_id ), 'error' );
					return;
				}
				$custom_field_index++;
				$custom_field = get_post_meta( $values['data']->id, '_weight'.$custom_field_index, true );
			}
		}

		// Pack it
		$boxpack->pack();
		$packages = $boxpack->get_packages();
		$to_ship  = array();
		$group_id = 1;

		// include_once('royalmail/class-special-boxes-packing.php');
		foreach ($packages as $package) {
			if ( $package->unpacked === true ) {
				$this->debug( 'Unpacked Item' );
			} else {

			}
		}

		// $special_pack = new phive_class_special_boxes_packing();
		// $special_pack -> add_boxes($special_boxes);
		foreach ( $packages as $package ) {
			$this->debug( 'Packed ' . $package->id );
			

			$dimensions = array( $package->length, $package->width, $package->height );

			sort( $dimensions );
			
			// Insurance amount of box $boxinsuredprice
			$boxinsuredprice = 0;
			if( ! empty($package->packed) ) {
				foreach( $package->packed as $box_item){
					$boxinsuredprice += $box_item->meta['data']->get_price();
				}
			}

			$group = array(
				'weight' => array(
					'value' => $this->round_up( $package->weight, 4 ),
					'units' => $this->labelapi_weight_unit
				),
				'dimensions'		=> array(
					'length' => max( 1, round( $dimensions[2], 0 ) ),
					'width'  => max( 1, round( $dimensions[1], 0 ) ),
					'height' => max( 1, round( $dimensions[0], 0 ) ),
					'units'  => $this->labelapi_dimension_unit
				),
				'cost'	  => array(
					'amount'   => round( ( $boxinsuredprice / $this->conversion_rate ), 2 ),
					'currency' => $this->royalmail_currency,
				),
				'packed_products' => array(),
				'package_id'	  => $package->id
			);

			if ( ! empty( $package->packed ) && is_array( $package->packed ) ) {
				foreach ( $package->packed as $packed ) {
					$group['packed_products'][] = $packed->get_meta( 'data' );
				}
			}
			$to_ship[] = $group;

			$group_id++;
		}

		return $to_ship;
	}
	
	private function weight_based_shipping( $package ){
		if ( ! class_exists( 'WeightPack' ) ) {
			include_once 'weight_pack/class-ph-weight-packing.php';
		}
		
		$weight_pack = new WeightPack($this->weight_pack_process);
		$weight_pack->set_max_weight($this->box_max_weight);
		
		foreach ( $package['contents'] as $item_id => $values ) {
			$weight = null;
			$values['data'] = $this->phive_load_product( $values['data'] );

			if ( ! $values['data']->needs_shipping() ) {
				$this->debug( sprintf( __( 'Product # is virtual. Skipping.', 'ph-shipping-royalmail' ), $item_id ), 'error' );
				continue;
			}
			
			$skip_product = apply_filters('phive_shipping_skip_product', false, $values, $package['contents']);
			if($skip_product){
				continue;
			}

			$custom_field_index=1;
			$custom_field = '';
			while($custom_field_index==1 || !empty( $custom_field )){
				$weight = 0;
				if($values['data']->get_type()=='simple'){
					$weight = $custom_field_index == 1 ? $values['data']->get_weight() : get_post_meta( $values['data']->id, '_weight'.$custom_field_index, true );
				}

				if( empty($weight) ){
					$weight = $values['data']->get_weight();
				}

				if( !empty($weight) ){
					$weight_pack->add_item( wc_get_weight( $weight, $this->weight_unit ), $values['data'], $values['quantity'] );
				}else{
					$this->debug( sprintf( __( 'Product # is missing weight. Aborting.', 'ph-shipping-royalmail' ), $item_id ), 'error' );
					return;
				}
				$custom_field_index++;
				$custom_field = get_post_meta( $values['data']->id, '_weight'.$custom_field_index, true );
			}
		}		
		
		$pack	=	$weight_pack->pack_items();
		
		$errors	=	$pack->get_errors();
		if( !empty($errors) ){
			//do nothing
			return;
		} else {
			$boxes		=	$pack->get_packed_boxes();
			$unpacked_items	=	$pack->get_unpacked_items();
			
			$to_ship  				= 	array();
			$group_id 				= 	1;		
			$group					= 	array();
			$packed_products		=	array();
			$insured_value			=	0;
			
			
			$packages	=	array_merge( $boxes, $unpacked_items ); // merge items if unpacked are allowed
			foreach($packages as $package){
				
				$insured_value	=	0;
				foreach($package['items'] as $item){						
					$insured_value		= $insured_value + $item->get_price();
				}
				$group = array(
					'GroupNumber'	   => $group_id,
					'GroupPackageCount' => 1,
					'weight' => array(
						'value' => $this->round_up($package['weight'],2),
						'units' => $this->labelapi_weight_unit
					),
					'packed_products' => $package['items']
				);
				$group['cost'] = array(
					'amount'   => round( ( $insured_value / $this->conversion_rate ), 2 ),
					'currency' => $this->royalmail_currency,
				);

				$to_ship[] = $group;
			}
		}

		return $to_ship;
	}

	private function ph_get_packed_packages( $packages, $packing_method='' ){
		$phive_packages	=	array();
		foreach($packages as $package){
			$package	= apply_filters( 'phive_customize_package_on_cart_and_checkout', $package );	// Customize the packages if cart contains bundled products
			$phive_packs	= $this->phive_do_packing( $package, $packing_method );
			if(is_array($phive_packs)){
				foreach ($phive_packs as $key => &$pack) {
					$pack['destination'] = $package['destination'];
				}
			
				$phive_packages	=	array_merge($phive_packages, $phive_packs);
			}
		}
		return $phive_packages;
	}
	public function calculate_shipping( $package = array() ) {
		// Clear rates
		$this->found_rates = array();

		// Debugging
		$this->debug( __( 'ROYAL MAIL debug mode is on - to hide these messages, turn debug mode off in the settings.', 'ph-shipping-royalmail' ) );
		
		$packages = apply_filters('phive_filter_package_address', array($package) , 'origin_address');
		
		$phive_packages = $this->ph_get_packed_packages($packages);
		
		include_once('royalmail/royalmail-rate-calculator.php');
		include_once('parcelforce/parcelforce-rate-calculator.php');

		$royalmail_calculator 	= new phive_royalmail_rate_calculator();
		$parcelforce_calculator = new phive_parcelforce_rate_calculator();
		$special_service = $this->get_option( 'options', array() );

		$royalmail_rates 	= $royalmail_calculator->calculate_rate( $this->ph_get_packed_packages($packages), $special_service);
		$parcelfroce_rates 	= $parcelforce_calculator->calculate_rate( $this->ph_get_packed_packages($packages, 'weight_based') );
		if( empty($royalmail_rates) ){
			$royalmail_rates = array();
		}
		if( empty($parcelfroce_rates) ){
			$parcelfroce_rates = array();
		}

		$rates = array_merge( $royalmail_rates, $parcelfroce_rates );

		if( empty($rates) ){
			return;
		}

		$unavailable_services 	= array();
		$services_by_country 	= include('ph-service-unavailability-by-country.php');
		$destination_country 	= $package['destination']['country'];

		if( is_array($services_by_country) && array_key_exists($destination_country, $services_by_country) && is_array($services_by_country[$destination_country]) ) {
			$unavailable_services = $services_by_country[$destination_country];
		}

		foreach ($rates as $rate) {
			if( isset($this->custom_services[ $rate['id'] ]['enabled']) && $this->custom_services[ $rate['id'] ]['enabled'] != 1 ){
				continue;
			}

			if( !empty($unavailable_services) && in_array($rate['id'], $unavailable_services) ) {
				continue;
			}
			
			$this->prepare_rate( $rate['code'], $rate['id'], $rate['name'], $rate['cost'] );
		}

		// Ensure rates were found for all packages
		$packages_to_quote_count = sizeof( $phive_packages );

		if ( $this->found_rates ) {
			foreach ( $this->found_rates as $key => $value ) {
				if ( $value['packages'] < $packages_to_quote_count ) {
					unset( $this->found_rates[ $key ] );
				}
			}
		}
		$this->add_found_rates();		
	}

	private function prepare_rate( $rate_code, $rate_id, $rate_name, $rate_cost, $rate_name_extra='' ) {

		// Name adjustment
		if ( ! empty( $this->custom_services[ $rate_code ]['name'] ) ) {
			$rate_name = $this->custom_services[ $rate_code ]['name'] . $rate_name_extra;
		}

		// Cost adjustment %
		if ( ! empty( $this->custom_services[ $rate_code ]['adjustment_percent'] ) ) {
			$rate_cost = $rate_cost + ( $rate_cost * ( floatval( $this->custom_services[ $rate_code ]['adjustment_percent'] ) / 100 ) );
		}
		// Cost adjustment
		if ( ! empty( $this->custom_services[ $rate_code ]['adjustment'] ) ) {
			$rate_cost = $rate_cost + floatval( $this->custom_services[ $rate_code ]['adjustment'] );
		}

		// Enabled check
		if ( isset( $this->custom_services[ $rate_code ] ) && empty( $this->custom_services[ $rate_code ]['enabled'] ) ) {
			return;
		}

		// Merging
		if ( isset( $this->found_rates[ $rate_id ] ) ) {
			$rate_cost = $rate_cost + $this->found_rates[ $rate_id ]['cost'];
			$packages  = 1 + $this->found_rates[ $rate_id ]['packages'];
		} else {
			$packages  = 1;
		}

		// Sort
		if ( isset( $this->custom_services[ $rate_code ]['order'] ) ) {
			$sort = $this->custom_services[ $rate_code ]['order'];
		} else {
			$sort = 999;
		}

		$this->found_rates[ $rate_id ] = array(
			'id'	   => $rate_id,
			'label'	=> $rate_name,
			'cost'	 => $rate_cost,
			'sort'	 => $sort,
			'packages' => $packages,
		);
	}

	public function add_found_rates() {
		if ( $this->found_rates ) {
			
			if( $this->conversion_rate ) {
				foreach ( $this->found_rates as $key => $rate ) {
					$this->found_rates[ $key ][ 'cost' ] = $rate[ 'cost' ] * $this->conversion_rate;
				}
			}

			if ( $this->offer_rates == 'all' ) {

				uasort( $this->found_rates, array( $this, 'sort_rates' ) );

				foreach ( $this->found_rates as $key => $rate ) {
					$this->add_rate( $rate );
				}
			} else {
				$cheapest_rate = '';

				foreach ( $this->found_rates as $key => $rate ) {
					if ( ! $cheapest_rate || $cheapest_rate['cost'] > $rate['cost'] ) {
						$cheapest_rate = $rate;
					}
				}

				$cheapest_rate['label'] = $this->title;

				$this->add_rate( $cheapest_rate );
			}
		}
	}

	public function sort_rates( $a, $b ) {
		if ( $a['sort'] == $b['sort'] ) return 0;
		return ( $a['sort'] < $b['sort'] ) ? -1 : 1;
	}

	private function phive_load_product( $product ){
		if( !$product ){
			return false;
		}
		if( !class_exists('phive_product') ){
			include_once('ph-legacy-handler.php');
		}
		if($product instanceof phive_product){
			return $product;
		}
		return new phive_product( $product );
	}

	public function debug( $message, $type = 'notice' ) {
		if ( $this->debug ) {
			wc_add_notice( $message, $type );
		}
	}
}
new phive_royalmail_woocommerce_shipping_method;
