<?php
namespace memberpress\courses\lib;

if(!defined('ABSPATH')) {die('You are not allowed to call this page directly.');}

use memberpress\courses as base;

class Db {
  public $prefix, $sections, $user_progress, $questions, $attempts, $answers;

  public function __construct() {
    global $wpdb;

    $this->prefix = $wpdb->prefix . base\SLUG_KEY . '_';

    // Tables
    $this->sections = "{$this->prefix}sections";
    $this->user_progress = "{$this->prefix}user_progress";
    $this->questions = "{$this->prefix}questions";
    $this->attempts = "{$this->prefix}attempts";
    $this->answers = "{$this->prefix}answers";
  }

  /**
   * Get the Db instance
   *
   * @param bool $force Force creation of a fresh instance
   * @return Db
   */
  public static function fetch($force = false) {
    static $db;

    if(!isset($db) || $force) {
      $db = new Db();
    }

    return apply_filters(base\SLUG_KEY.'_fetch_db', $db);
  }

  public function upgrade() {
    global $wpdb;

    static $upgrade_already_running;

    if(isset($upgrade_already_running) && true===$upgrade_already_running) {
      return;
    }
    else {
      $upgrade_already_running = true;
    }

    $old_db_version = get_option(base\SLUG_KEY.'_db_version');

    if(base\DB_VERSION > $old_db_version) {
      // Ensure our big queries can run in an upgrade
      $wpdb->query('SET SQL_BIG_SELECTS=1'); //This may be getting set back to 0 when SET MAX_JOIN_SIZE is executed
      $wpdb->query('SET MAX_JOIN_SIZE=18446744073709551615');

      $this->before_upgrade($old_db_version);

      // This was introduced in WordPress 3.5
      // $char_col = $wpdb->get_charset_collate(); //This doesn't work for most non english setups
      $char_col = "";
      $collation = $wpdb->get_row("SHOW FULL COLUMNS FROM {$wpdb->posts} WHERE field = 'post_content'");

      if(isset($collation->Collation)) {
        $charset = explode('_', $collation->Collation);

        if(is_array($charset) && count($charset) > 1) {
          $charset = $charset[0]; //Get the charset from the collation
          $char_col = "DEFAULT CHARACTER SET {$charset} COLLATE {$collation->Collation}";
        }
      }

      //Fine we'll try it your way this time
      if(empty($char_col)) { $char_col = $wpdb->get_charset_collate(); }

      require_once(ABSPATH . 'wp-admin/includes/upgrade.php');

      $sections = "
      CREATE TABLE {$this->sections} (
          `id` bigint(20) NOT NULL auto_increment,
          `title` text NOT NULL,
          `description` longtext NOT NULL,
          `course_id` bigint(20) NOT NULL,
          `section_order` int(11) DEFAULT 0,
          `created_at` datetime NOT NULL,
          `uuid` varchar(40) NOT NULL,
          PRIMARY KEY  (id),
          KEY `course_id` (`course_id`),
          KEY `section_order` (`section_order`),
          KEY `uuid` (`uuid`)
        ) {$char_col};";
      dbDelta($sections);

      $user_progress = "
      CREATE TABLE {$this->user_progress} (
          `id` bigint(20) NOT NULL auto_increment,
          `user_id` bigint(20) NOT NULL,
          `lesson_id` bigint(20) NOT NULL,
          `course_id` bigint(20) NOT NULL,
          `progress` decimal(5,2) DEFAULT 100.00,
          `created_at` datetime NOT NULL,
          `completed_at` datetime,
          PRIMARY KEY  (id),
          KEY `user_id` (`user_id`),
          KEY `lesson_id` (`lesson_id`),
          KEY `course_id` (`course_id`),
          KEY `completed_at` (`completed_at`),
          UNIQUE KEY `user_lesson_course` (`user_id`,`lesson_id`,`course_id`)
        ) {$char_col};";
      dbDelta($user_progress);

      $questions = "
      CREATE TABLE {$this->questions} (
          `id` bigint(20) unsigned NOT NULL auto_increment,
          `quiz_id` bigint(20) unsigned,
          `number` int(11) unsigned,
          `text` longtext,
          `options` longtext,
          `answer` text,
          `type` text,
          `required` bool,
          `points` int(11),
          `feedback` longtext,
          `settings` longtext,
          PRIMARY KEY  (id),
          KEY `quiz_id` (`quiz_id`),
          UNIQUE KEY `quiz_question` (`quiz_id`,`id`)
        ) $char_col;";
      dbDelta($questions);

      $attempts = "
      CREATE TABLE {$this->attempts} (
          `id` bigint(20) unsigned NOT NULL auto_increment,
          `quiz_id` bigint(20) unsigned,
          `user_id` bigint(20) unsigned,
          `points_awarded` int(11) NOT NULL DEFAULT 0,
          `points_possible` int(11) NOT NULL DEFAULT 0,
          `score` int(11) NOT NULL DEFAULT 0,
          `status` varchar(64) NOT NULL DEFAULT 'draft',
          `started_at` datetime,
          `finished_at` datetime,
          PRIMARY KEY  (id),
          KEY `quiz_id` (`quiz_id`),
          KEY `user_id` (`user_id`),
          KEY `status` (`status`)
        ) $char_col;";
      dbDelta($attempts);

      $answers = "
      CREATE TABLE {$this->answers} (
          `id` bigint(20) unsigned NOT NULL auto_increment,
          `attempt_id` bigint(20) unsigned,
          `question_id` bigint(20) unsigned,
          `answer` longtext,
          `points_possible` int(11) NOT NULL DEFAULT 0,
          `points_awarded` int(11) NOT NULL DEFAULT 0,
          `grader` bigint(20) unsigned,
          `answered_at` datetime NOT NULL,
          `graded_at` datetime,
          PRIMARY KEY  (id),
          KEY `attempt_id` (`attempt_id`),
          KEY `question_id` (`question_id`),
          UNIQUE KEY `attempt_question` (`attempt_id`,`question_id`)
        ) $char_col;";
      dbDelta($answers);

      $this->after_upgrade($old_db_version);

      /***** SAVE DB VERSION *****/
      //Let's only run this query if we're actually updating
      update_option(base\SLUG_KEY.'_db_version', base\DB_VERSION);
    }
  }

  public function before_upgrade($curr_db_version) {
    // Nothing yet
  }

  public function after_upgrade($curr_db_version) {
    flush_rewrite_rules();
  }

  public function create_record($table, $args, $record_created_at = true) {
    global $wpdb;
    $cols = array();
    $vars = array();
    $values = array();

    $i = 0;
    foreach($args as $key => $value) {
      if ($key == 'id' && empty($value)) {
        // To prevent issue with SQL MODE NO_AUTO_VALUE_ON_ZERO
        // See the notes on MP Commit: https://github.com/caseproof/memberpress/commit/55d2f9d69a6adc73ced127d226fd6cba6cca0b9c
        continue;
      }

      if($key == 'created_at' && $record_created_at && empty($value)) { continue; }

      $cols[$i] = $key;
      if(is_numeric($value) and preg_match('!\.!',$value)) {
        $vars[$i] = '%f';
      }
      else if(is_int($value) or is_numeric($value) or is_bool($value)) {
        $vars[$i] = '%d';
      }
      else {
        $vars[$i] = '%s';
      }

      if(is_bool($value)) {
        $values[$i] = $value ? 1 : 0;
      }
      else {
        $values[$i] = $value;
      }

      $i++;
    }

    if($record_created_at && (!isset($args['created_at']) || empty($args['created_at']))) {
      $cols[$i] = 'created_at';
      $vars[$i] = $wpdb->prepare('%s',Utils::db_now());
      $i++;
    }

    if(empty($cols)) {
      return false;
    }

    $cols_str = implode(',',$cols);
    $vars_str = implode(',',$vars);

    $query = "INSERT INTO {$table} ( {$cols_str} ) VALUES ( {$vars_str} )";
    if(empty($values)) {
      $query = esc_sql( $query );
    }
    else {
      $query = $wpdb->prepare( $query, $values );
    }

    $query_results = $wpdb->query($query);

    if($query_results)
      return $wpdb->insert_id;
    else
      return false;
  }

  public function update_record( $table, $id, $args )
  {
    global $wpdb;

    if(empty($args) or empty($id))
      return false;

    $set = '';
    $values = array();
    foreach($args as $key => $value)
    {
      if(empty($set))
        $set .= ' SET';
      else
        $set .= ',';

      $set .= " {$key}=";

      if(is_numeric($value) and preg_match('!\.!',$value))
        $set .= "%f";
      else if(is_int($value) or is_numeric($value) or is_bool($value))
        $set .= "%d";
      else
        $set .= "%s";

      if(is_bool($value))
        $values[] = $value ? 1 : 0;
      else if (is_array($value))
        $values[] = maybe_serialize($value);
      else
        $values[] = $value;
    }

    $values[] = $id;
    $query = "UPDATE {$table}{$set} WHERE id=%d";

    if( empty($values) ) {
      $query = esc_sql( $query );
    }
    else {
      $query = $wpdb->prepare( $query, $values );
    }

    if($wpdb->query($query)) {
      return $id;
    }
    else {
      return false;
    }
  }

  public function delete_records($table, $args)
  {
    global $wpdb;

    extract(Db::get_where_clause_and_values( $args ));

    $query = "DELETE FROM {$table}{$where}";

    if( empty($values) ) {
      $query = esc_sql( $query );
    }
    else {
      $query = $wpdb->prepare( $query, $values );
    }

    return $wpdb->query($query);
  }

  public function get_count($table, $args=array(), $joins=array()) {
    global $wpdb;
    $join = '';

    if(!empty($joins)) {
      foreach($joins as $join_clause) {
        $join .= " {$join_clause}";
      }
    }

    extract(Db::get_where_clause_and_values( $args ));

    $query = "SELECT COUNT(*) FROM {$table}{$join}{$where}";

    if( empty($values) ) {
      $query = esc_sql( $query );
    }
    else {
      $query = $wpdb->prepare( $query, $values );
    }

    return $wpdb->get_var($query);
  }

  public function get_where_clause_and_values( $args ) {
    $args = (array)$args;

    $where = '';
    $values = array();
    foreach($args as $key => $value)
    {
      if(!empty($where))
        $where .= ' AND';
      else
        $where .= ' WHERE';

      $where .= " {$key}=";

      if(is_numeric($value) and preg_match('!\.!',$value))
        $where .= "%f";
      else if(is_int($value) or is_numeric($value) or is_bool($value))
        $where .= "%d";
      else
        $where .= "%s";

      if(is_bool($value))
        $values[] = $value ? 1 : 0;
      else
        $values[] = $value;
    }

    return compact('where','values');
  }

  public function get_one_model($model, $args=array()) {
    $table = $this->get_table_for_model($model);

    $rec = $this->get_one_record($table, $args);

    if(!empty($rec)) {
      $obj = new $model();
      $obj->load_from_array($rec);
      return $obj;
    }

    return $rec;
  }

  public function get_one_record($table, $args=array())
  {
    global $wpdb;

    extract(Db::get_where_clause_and_values( $args ));

    $query = "SELECT * FROM {$table}{$where} LIMIT 1";

    if( empty($values) ) {
      $query = esc_sql( $query );
    }
    else {
      $query = $wpdb->prepare( $query, $values );
    }

    return $wpdb->get_row($query);
  }

  public function get_models($model, $order_by='', $limit='', $args=array()) {
    $table = $this->get_table_for_model($model);
    $recs = $this->get_records($table, $args, $order_by, $limit);

    $models = array();
    foreach($recs as $rec) {
      $obj = new $model();
      $obj->load_from_array($rec);
      $models[] = $obj;
    }

    return $models;
  }

  public function get_records($table, $args=array(), $order_by='', $limit='', $joins=array(), $return_type=OBJECT) {
    global $wpdb;

    extract(Db::get_where_clause_and_values( $args ));
    $join = '';

    if(!empty($order_by)) {
      $order_by = " ORDER BY {$order_by}";
    }

    if(!empty($limit)) {
      $limit = " LIMIT {$limit}";
    }

    if(!empty($joins)) {
      foreach($joins as $join_clause) {
        $join .= " {$join_clause}";
      }
    }

    $query = "SELECT * FROM {$table}{$join}{$where}{$order_by}{$limit}";

    if(empty($values)) {
      $query = esc_sql($query);
    }
    else {
      $query = $wpdb->prepare($query, $values);
    }

    return $wpdb->get_results($query, $return_type);
  }

  public function get_col($table, $col, $args=array(), $order_by='', $limit='') {
    global $wpdb;

    extract(Db::get_where_clause_and_values($args));
    if(!empty($order_by)) { $order_by = " ORDER BY {$order_by}"; }
    if(!empty($limit)) { $limit = " LIMIT {$limit}"; }

    $query = "SELECT {$table}.{$col} FROM {$table}{$where}{$order_by}{$limit}";

    if(!empty($values)) {
      $query = $wpdb->prepare($query, $values);
    }
    return $wpdb->get_col($query);
  }

  public function prepare_array($item_type,$values) {
    return implode(
      ',',
      array_map(
        function($value) use ($item_type) {
          global $wpdb;
          return $wpdb->prepare($item_type, $value);
        },
        $values
      )
    );
  }

  /* Built to work with WordPress' built in WP_List_Table class */
  public static function list_table( $cols,
                                     $from,
                                     $joins=array(),
                                     $args=array(),
                                     $order_by='',
                                     $order='',
                                     $paged='',
                                     $search='',
                                     $perpage=10,
                                     $search_cols=array()) {
    global $wpdb;

    // Setup selects
    $col_str_array = array();
    foreach( $cols as $col => $code ) {
      $col_str_array[] = "{$code} AS {$col}";
    }

    $col_str = implode(", ",$col_str_array);

    // Setup Joins
    if(!empty($joins)) {
      $join_str = " " . implode( " ", $joins );
    }
    else {
      $join_str = '';
    }

    $args_str = implode(' AND ', $args);

    /* -- Ordering parameters -- */
    //Parameters that are going to be used to order the result
    $order_by = (!empty($order_by) and !empty($order)) ? ( $order_by = ' ORDER BY ' . $order_by . ' ' . $order ) : '';

    //Page Number
    if(empty($paged) or !is_numeric($paged) or $paged<=0 ){ $paged=1; }

    $limit = '';
    //adjust the query to take pagination into account
    if(!empty($paged) and !empty($perpage)) {
      $offset=($paged-1)*$perpage;
      $limit = ' LIMIT '.(int)$offset.','.(int)$perpage;
    }

    // Searching
    $search_str = "";
    $searches = array();
    if(!empty($search)) {
      if(empty($search_cols)) {
        $search_cols = $cols;
      }

      foreach($search_cols as $code) {
        $like = '%' . $wpdb->esc_like($search) . '%';
        $searches[] = $wpdb->prepare("{$code} LIKE %s", $like);
      }

      if(!empty($searches)) {
        $search_str = implode(' OR ', $searches);
      }
    }

    $conditions = "";

    // Pull Searching into where
    if(!empty($args)) {
      if(!empty($searches)) {
        $conditions = " WHERE $args_str AND ({$search_str})";
      }
      else {
        $conditions = " WHERE $args_str";
      }
    }
    else {
      if(!empty($searches)) {
        $conditions = " WHERE {$search_str}";
      }
    }

    $query = "SELECT {$col_str} FROM {$from}{$join_str}{$conditions}{$order_by}{$limit}";
    $total_query = "SELECT COUNT(*) FROM {$from}{$join_str}{$conditions}";

    //Allows us to run the bazillion JOINS we use on the list tables
    $wpdb->query("SET SQL_BIG_SELECTS=1");

    $results = $wpdb->get_results($query);
    $count = $wpdb->get_var($total_query);

    return array( 'results' => $results, 'count' => $count );
  }

  public function get_table_for_model($model) {
    global $wpdb;
    $class_name = \wp_unslash(preg_replace('/^' . wp_slash(base\MODELS_NAMESPACE) . '(.*)/', '$1', $model));
    $table = Utils::snakecase($class_name);

    // TODO: We need to get true inflections working here eventually ...
    // Only append an s if it doesn't end in s
    if(!preg_match('/s$/', $table))
      $table .= 's';

    return "{$this->prefix}{$table}";
  }

  /**
  * Light weight query to check if record exists
  * @return true|false
  */
  public function record_exists($table, $args = array()) {
    global $wpdb;

    extract(Db::get_where_clause_and_values($args));

    $query = "SELECT 1 AS `exists` FROM {$table}{$where} LIMIT 1";

    if( empty($values) ) {
      $query = esc_sql( $query );
    }
    else {
      $query = $wpdb->prepare( $query, $values );
    }

    $record_exists = $wpdb->get_var($query);

    return $record_exists === "1" ? true : false;
  }

  public function table_exists($table) {
    global $wpdb;
    $q = $wpdb->prepare('SHOW TABLES LIKE %s', $table);
    $table_res = $wpdb->get_var($q);
    return ($table_res == $table);
  }

  public function table_empty($table) {
    return ($this->get_count($table) <= 0);
  }

  public function column_exists($table, $column) {
    global $wpdb;
    $q = $wpdb->prepare("SHOW COLUMNS FROM {$table} LIKE %s", $column);
    $res = $wpdb->get_col($q);
    return (count($res) > 0);
  }
}
