// #region [Imports] ===================================================================================================

// Libraries
import React, { useEffect, useState } from "react";
import { Link } from "react-router-dom";
import { bindActionCreators } from "redux";
import { connect } from "react-redux";
import { Row, Col, Form, InputNumber, Button, Skeleton, message } from "antd";

// CSS
import "./index.scss";

// Components
import PointsWorth from "../PointsBalance/PointsWorth";

// Actions
import { PointsBalanceActions } from "../../store/actions/balance";

// Types
import IStore from "../../types/store";
import IPointsBalance from "../../types/balance";

// Helpers
import { getPathPrefix } from "../../helpers/utils";

// #endregion [Imports]

// #region [Variables] =================================================================================================

declare var lpfwMyPoints: any;

const { readPointsBalance, redeemPoints } = PointsBalanceActions;
const pathPrefix = getPathPrefix();

// #endregion [Variables]

// #region [Interfaces]=================================================================================================

interface IActions {
    readPointsBalance: typeof readPointsBalance;
    redeemPoints: typeof redeemPoints;
}

interface IProps {
    balance: IPointsBalance;
    actions: IActions;
}

interface IFieldValues {
    points: number;
    worth: number;
}

// #endregion [Interfaces]

// #region [Component] =================================================================================================

const PointsRedeem = (props: IProps) => {
    const { balance, actions } = props;
    const {
        labels,
        currency_ratio,
        redeem_ratio,
        currency_symbol,
        coupon_expire_period,
        minimum_points_redeem,
    } = lpfwMyPoints;
    const [pointsValue, setPointsValue]: [number|string, any] = useState(0);
    const [worthValue, setWorthValue]: [number|string, any] = useState(0);
    const [loading, setLoading]: [boolean, any] = useState(false);

    useEffect(() => {
        actions.readPointsBalance({});
    }, []);

    if (!balance) {
        return <Skeleton active />;
    }

    const calculateAmountFromPoints = (value: string | number | undefined) => {
        if (typeof value === "undefined") return;
        value = typeof value === "string" ? parseInt(value) : value;
        if (value < minimum_points_redeem) return;

        let points: number =
            typeof value === "string" ? parseInt(value) : value;

        points = Math.min(points, balance.points);
        let temp = (points / redeem_ratio) * currency_ratio;

        if ( isNaN(temp) ) {
            temp = 0;
            points = 0;
        }

        setWorthValue(temp.toFixed(2));
        setPointsValue(parseInt(points.toString()));
    };

    const calculatePointsFromAmount = (value: string | number | undefined) => {
        if (typeof value === "undefined") return;
        let worth = typeof value === "string" ? parseFloat(value) : value;
        if (worth < 0) return;

        let temp: number = Math.floor((worth / currency_ratio) * redeem_ratio);

        temp = Math.min(temp, balance.points);
        worth = (temp / redeem_ratio) * currency_ratio;

        if ( isNaN( temp ) ) {
            worth = 0;
            temp = 0;
        }

        setPointsValue(parseInt(temp.toString()));
        setWorthValue(worth.toFixed(2));
    };

    const redeemPoints = () => {
        setLoading(true);
        actions.redeemPoints({
            points: pointsValue,
            successCB: (response: any) => {
                setWorthValue(0.0);
                setPointsValue(0);
                setLoading(false);
                message.success( response.data.message );
            },
            failCB: ({error}) => {
                setLoading(false);
                message.error( error.response.data.message );
            }
        });
    };

    return (
        <div className="points-redeem">
            <h3>{labels.redeem_points}</h3>
            <p>{labels.redeem_desc}</p>
            <PointsWorth balance={balance} />
            <Row className="form-wrap" gutter={10}>
                <Col span={11}>
                    <label>{labels.enter_points}:</label>
                    <InputNumber
                        min={minimum_points_redeem}
                        max={balance.points}
                        step={1}
                        value={pointsValue}
                        onChange={calculateAmountFromPoints}
                    />
                </Col>
                <Col span={1}>
                    <span className="equal-sign">{`=`}</span>
                </Col>
                <Col span={11}>
                    <label>
                        {labels.enter_amount} ({currency_symbol}):
                    </label>
                    <InputNumber
                        min={0}
                        step={(1/redeem_ratio) * currency_ratio}
                        value={worthValue}
                        onChange={calculatePointsFromAmount}
                    />
                </Col>
                <Col span={24}>
                    <Button
                        loading={loading}
                        type="primary"
                        size="large"
                        onClick={redeemPoints}
                        disabled={!pointsValue || minimum_points_redeem > pointsValue}
                    >
                        {labels.redeem_button}
                    </Button>
                    <span className="view-redeemed-link">
                        <Link to={`${pathPrefix}lpfw-my-points/?tab=balance`}>
                            {labels.view_redeemed}
                        </Link>
                    </span>
                </Col>
            </Row>
            {0 < coupon_expire_period ? (
                <p
                    className="expiry-note"
                    dangerouslySetInnerHTML={{ __html: labels.expire_notice }}
                />
            ) : null}
        </div>
    );
};

const mapStateToProps = (store: IStore) => ({ balance: store.balance });

const mapDispatchToProps = (dispatch: any) => ({
    actions: bindActionCreators({ readPointsBalance, redeemPoints }, dispatch),
});

export default connect(mapStateToProps, mapDispatchToProps)(PointsRedeem);

// #endregion [Component]
