// #region [Imports] ===================================================================================================

// Libraries
import React from "react";
import ICustomer from "../../types/customer";

// Actions
import { CustomerDataActions } from "../../store/actions/customer";

// #endregion [Imports]

// #region [Variables] =================================================================================================

declare var acfwAdminApp: any;
declare var acfwpElements: any;

const {
    element: { useState, useEffect },
    antd: { Card, Select, Spin, message },
    axiosInstance,
    redux: { bindActionCreators, connect },
} = acfwpElements;
const { Option } = Select;

const { setStoreCustomer } = CustomerDataActions;

// #endregion [Variables]

// #region [Interfaces] ================================================================================================

interface IActions {
    setStoreCustomer: typeof setStoreCustomer;
}

interface IProps {
    customer: ICustomer;
    actions: IActions;
}

// #endregion [Interfaces]

// #region [Component] =================================================================================================

const SearchCustomers = (props: IProps) => {
    const { customer, actions } = props;
    const {
        loyalty_program: { labels },
    } = acfwAdminApp;
    const { search_customers, name_or_email } = labels;
    const [selectedOption, setSelectedOption]: [string, any] = useState("");
    const [searchResults, setSearchResults]: [ICustomer[], any] = useState([]);
    const [searchTimeout, setSearchTimeout]: [any, any] = useState(null);

    /**
     * Set selected option state when customer redux state is already present.
     */
    useEffect(() => {
        if (!customer) return;

        const index = searchResults.findIndex((c) => c.id === customer.id);
        if (index < 0) setSearchResults([...searchResults, customer]);
        setSelectedOption(`${customer.name} <${customer.email}>`);
    }, [customer]);

    /**
     * Schedule customer search as a timeout to prevent duplicate searches while user still typing.
     *
     * @param value
     */
    const setSearchUsersTimeout = (value: string) => {
        if (searchTimeout) {
            clearTimeout(searchTimeout);
            setSearchResults(null);
        }

        if (value) setSearchTimeout(setTimeout(() => searchUsers(value), 1000));
    };

    /**
     * Callback to do actual customer search.
     *
     * @param value
     */
    const searchUsers = async (value: string) => {
        try {
            const response = await axiosInstance.get(
                `loyalty-program/v1/customers?search=${value}`
            );

            setSearchResults(response.data);
            setSearchTimeout(null);
        } catch (error: any) {
            message.error(error.response.data.message);
        }
    };

    /**
     * Set customer to redux store.
     * 
     * @param id 
     */
    const setCustomer = (id: string) => {
        const index = searchResults.findIndex((c) => c.id === parseInt(id));
        const selectedCustomer = index > -1 ? searchResults[index] : null;
        
        // make sure that the selected customer is not null and we're not selecting the current customer.
        if (selectedCustomer && selectedCustomer?.id !== customer?.id)
            actions.setStoreCustomer({ data: selectedCustomer });
    };

    /**
     * Clear search results, selected option state and also the customer data.
     * NOTE: we are hooking via onChange callback instead of onClear as the latter is not working somehow.
     * 
     * @param value 
     */
    const clearSearchData = (value: string) => {

        if (value) return;

        setSelectedOption("");
        setSearchResults([]);
        setSearchTimeout(null);
        actions.setStoreCustomer({ data: null});
    };

    return (
        <Card title={search_customers}>
            <h4>{name_or_email}</h4>
            <Select
                onSearch={setSearchUsersTimeout}
                style={{ width: "100%", maxWidth: "447px" }}
                value={selectedOption}
                filterOption={false}
                showSearch
                allowClear
                notFoundContent={searchTimeout ? <Spin size="small" /> : null}
                onSelect={setCustomer}
                onChange={clearSearchData}
            >
                {searchResults
                    ? searchResults.map((c) => (
                          <Option key={c.id}>{`${c.name} <${c.email}>`}</Option>
                      ))
                    : null}
            </Select>
        </Card>
    );
};

const mapStateToProps = (store: any) => ({ customer: store.customer });

const mapDispatchToProps = (dispatch: any) => ({
    actions: bindActionCreators({ setStoreCustomer }, dispatch),
});

export default connect(mapStateToProps, mapDispatchToProps)(SearchCustomers);

// #endregion [Component]
