<?php
namespace LPFW\Models;

use LPFW\Abstracts\Abstract_Main_Plugin_Class;
use LPFW\Helpers\Helper_Functions;
use LPFW\Helpers\Plugin_Constants;
use LPFW\Interfaces\Initiable_Interface;
use LPFW\Interfaces\Model_Interface;

if (!defined('ABSPATH')) {
    exit;
}
// Exit if accessed directly

/**
 * Model that houses the logic of extending the coupon system of woocommerce.
 * It houses the logic of handling coupon url.
 * Public Model.
 *
 * @since 1.0
 */
class User_Points implements Model_Interface, Initiable_Interface
{

    /*
    |--------------------------------------------------------------------------
    | Class Properties
    |--------------------------------------------------------------------------
     */

    /**
     * Property that holds the single main instance of URL_Coupon.
     *
     * @since 1.0
     * @access private
     * @var User_Points
     */
    private static $_instance;

    /**
     * Model that houses all the plugin constants.
     *
     * @since 1.0
     * @access private
     * @var Plugin_Constants
     */
    private $_constants;

    /**
     * Property that houses all the helper functions of the plugin.
     *
     * @since 1.0
     * @access private
     * @var Helper_Functions
     */
    private $_helper_functions;

    /*
    |--------------------------------------------------------------------------
    | Class Methods
    |--------------------------------------------------------------------------
     */

    /**
     * Class constructor.
     *
     * @since 1.0
     * @access public
     *
     * @param Abstract_Main_Plugin_Class $main_plugin      Main plugin object.
     * @param Plugin_Constants           $constants        Plugin constants object.
     * @param Helper_Functions           $helper_functions Helper functions object.
     */
    public function __construct(Abstract_Main_Plugin_Class $main_plugin, Plugin_Constants $constants, Helper_Functions $helper_functions)
    {
        $this->_constants        = $constants;
        $this->_helper_functions = $helper_functions;

        $main_plugin->add_to_all_plugin_models($this);
        $main_plugin->add_to_public_models($this);
    }

    /**
     * Ensure that only one instance of this class is loaded or can be loaded ( Singleton Pattern ).
     *
     * @since 1.0
     * @access public
     *
     * @param Abstract_Main_Plugin_Class $main_plugin      Main plugin object.
     * @param Plugin_Constants           $constants        Plugin constants object.
     * @param Helper_Functions           $helper_functions Helper functions object.
     * @return User_Points
     */
    public static function get_instance(Abstract_Main_Plugin_Class $main_plugin, Plugin_Constants $constants, Helper_Functions $helper_functions)
    {
        if (!self::$_instance instanceof self) {
            self::$_instance = new self($main_plugin, $constants, $helper_functions);
        }

        return self::$_instance;
    }

    /*
    |--------------------------------------------------------------------------
    | Redeem methods
    |--------------------------------------------------------------------------
     */

    /**
     * Redeem points for user by converting points to a coupon only usable by the user.
     *
     * @since 1.0
     * @access public
     *
     * @param int $points  Points to redeem.
     * @param int $user_id User ID.
     */
    public function redeem_points_for_user($points, $user_id)
    {
        $user_points = LPFW()->Calculate->get_user_total_points($user_id);
        $min_points  = (int) $this->_helper_functions->get_option($this->_constants->MINIMUM_POINTS_REDEEM, '0');

        if (!$points || $points > $user_points || $points < $min_points) {
            return;
        }

        $coupon = $this->_create_user_redeem_coupon($points, $user_id);

        if ($coupon instanceof \WC_Coupon) {
            LPFW()->Entries->insert_entry($user_id, 'redeem', 'coupon', $points, $coupon->get_id());
        }

        return $coupon;
    }

    /**
     * Create user redeem coupon.
     *
     * @since 1.0
     * @access private
     *
     * @param int $points  Points to redeem.
     * @param int $user_id User ID.
     * @return WC_Coupon Advanced coupon object.
     */
    private function _create_user_redeem_coupon($points, $user_id)
    {
        $code   = $user_id . $this->_helper_functions->random_str(6);
        $coupon = new \WC_Coupon($code);
        $amount = LPFW()->Calculate->calculate_redeem_points_worth($points, false);

        if (!$amount) {
            return;
        }

        $coupon->set_id(0);
        $coupon->set_code($code);
        $coupon->set_discount_type('fixed_cart');
        $coupon->set_amount($amount);
        $coupon->set_id($coupon->save());

        if ($coupon_id = $coupon->get_id()) {

            update_post_meta($coupon_id, $this->_constants->META_PREFIX . 'loyalty_program_user', $user_id);
            update_post_meta($coupon_id, $this->_constants->META_PREFIX . 'loyalty_program_points', $points);
            update_post_meta($coupon_id, 'usage_limit', 1);

            $datetime = $this->_get_reedemed_coupon_schedule_expire();
            $format   = is_object($datetime) ? $datetime->format('Y-m-d') : '';

            update_post_meta($coupon_id, 'expiry_date', $format);
            if (is_object($datetime)) {
                update_post_meta($coupon_id, 'date_expires', $datetime->getTimestamp());
            }

            $this->_save_with_default_redeemed_coupon_category($coupon_id);
        }

        $coupon->save_meta_data();

        return $coupon;
    }

    /**
     * Get schedule expire for redeemed coupon based on settings.
     *
     * @since 1.0
     * @access private
     *
     * @return DateTime Expiry date time object.
     */
    private function _get_reedemed_coupon_schedule_expire()
    {
        $expire_period = (int) get_option($this->_constants->COUPON_EXPIRE_PERIOD, 365);

        if (!$expire_period) {
            return;
        }

        $timezone  = new \DateTimeZone($this->_helper_functions->get_site_current_timezone());
        $datetime  = new \DateTime("today", $timezone);
        $timestamp = $datetime->getTimestamp() + ($expire_period * DAY_IN_SECONDS);

        $datetime->setTimestamp($timestamp);

        return $datetime;
    }

    /**
     * Save coupon with default coupon category.
     *
     * @since 1.0
     * @access private
     *
     * @param int $coupon_id Coupon ID.
     */
    private function _save_with_default_redeemed_coupon_category($coupon_id)
    {
        $default_category = (int) get_option($this->_constants->DEFAULT_REDEEM_COUPON_CAT);

        // create the default term if it doesn't exist
        if (!term_exists($default_category, $this->_constants->COUPON_CAT_TAXONOMY)) {

            $default_cat_name = __('Redeemed', 'loyalty-program-for-woocommerce');
            wp_insert_term($default_cat_name, $this->_constants->COUPON_CAT_TAXONOMY);

            $default_term = get_term_by('name', $default_cat_name, $this->_constants->COUPON_CAT_TAXONOMY);
            update_option($this->_constants->DEFAULT_REDEEM_COUPON_CAT, $default_term->term_id);

        } else {
            $default_term = get_term_by('id', $default_category, $this->_constants->COUPON_CAT_TAXONOMY);
        }

        wp_set_post_terms($coupon_id, $default_term->term_id, $this->_constants->COUPON_CAT_TAXONOMY);
    }

    /**
     * Apply redeemed coupon to cart.
     *
     * @since 1.0
     * @access public
     */
    public function apply_redeemed_coupon_to_cart()
    {
        if ((!is_cart() && !is_checkout()) || !isset($_GET['lpfw_coupon'])) {
            return;
        }

        $coupon = sanitize_text_field($_GET['lpfw_coupon']);

        // Initialize cart session
        WC()->session->set_customer_session_cookie(true);

        // Apply coupon to cart
        WC()->cart->apply_coupon($coupon);

        wp_redirect(wc_get_cart_url());
        exit();
    }

    /*
    |--------------------------------------------------------------------------
    | User related methods
    |--------------------------------------------------------------------------
     */

    /**
     * Get user redeemed coupons
     *
     * @since 1.0
     * @access public
     *
     * @global wpdb $wpdb Object that contains a set of functions used to interact with a database.
     *
     * @param int $user_id User ID.
     * @param int $page    Page number.
     * @return array User redeemed coupons.
     */
    public function get_user_redeemed_coupons($user_id, $page = 1)
    {
        global $wpdb;

        $timezone = new \DateTimeZone($this->_helper_functions->get_site_current_timezone());
        $datetime = new \DateTime("today", $timezone);
        $today    = $datetime->format('U');

        $lp_entries_db = $wpdb->prefix . $this->_constants->DB_TABLE_NAME;
        $user_id       = absint(esc_sql($user_id));
        $offset        = $page ? ($page - 1) * 10 : 0;

        $query = "SELECT object_id AS ID, posts.post_title AS code, amount.meta_value AS amount,
            posts.post_date_gmt AS date, entry_amount AS points, coupon_expire.meta_value AS date_expire
            FROM $lp_entries_db
            INNER JOIN $wpdb->posts AS posts ON ( posts.ID = object_id )
            INNER JOIN $wpdb->postmeta AS amount ON ( amount.post_id = object_id AND amount.meta_key = 'coupon_amount' )
            INNER JOIN $wpdb->postmeta AS usage_count ON ( usage_count.post_id = object_id AND usage_count.meta_key = 'usage_count' )
            INNER JOIN $wpdb->postmeta AS coupon_expire ON ( coupon_expire.post_id = object_id AND coupon_expire.meta_key = 'date_expires' )
            WHERE user_id = $user_id
                AND entry_type = 'redeem'
                AND posts.post_status = 'publish'
                AND posts.post_type = 'shop_coupon'
                AND usage_count.meta_value = 0
                AND ( coupon_expire.meta_value = '' OR coupon_expire.meta_value IS NULL OR coupon_expire.meta_value > $today )
                GROUP BY object_id
                ORDER BY posts.post_date DESC
                LIMIT {$offset}, 10
        ";

        $data = $wpdb->get_results($query);

        return $data;
    }

    /**
     * Get user total number of redemeed coupons.
     *
     * @since 1.0
     * @access public
     *
     * @param int $user_id User ID.
     * @return in Number of coupons.
     */
    public function get_user_redeem_coupons_total($user_id)
    {
        global $wpdb;

        $timezone = new \DateTimeZone($this->_helper_functions->get_site_current_timezone());
        $datetime = new \DateTime("today", $timezone);
        $today    = $datetime->format('U');

        $lp_entries_db = $wpdb->prefix . $this->_constants->DB_TABLE_NAME;
        $user_id       = absint(esc_sql($user_id));

        $query = "SELECT entry_id FROM $lp_entries_db
            INNER JOIN $wpdb->posts AS posts ON ( posts.ID = object_id )
            INNER JOIN $wpdb->postmeta AS amount ON ( amount.post_id = object_id AND amount.meta_key = 'coupon_amount' )
            INNER JOIN $wpdb->postmeta AS usage_count ON ( usage_count.post_id = object_id AND usage_count.meta_key = 'usage_count' )
            INNER JOIN $wpdb->postmeta AS coupon_expire ON ( coupon_expire.post_id = object_id AND coupon_expire.meta_key = 'date_expires' )
            WHERE user_id = $user_id
                AND entry_type = 'redeem'
                AND posts.post_status = 'publish'
                AND posts.post_type = 'shop_coupon'
                AND usage_count.meta_value = 0
                AND ( coupon_expire.meta_value = '' OR coupon_expire.meta_value IS NULL OR coupon_expire.meta_value > $today )
                GROUP BY object_id
                ORDER BY posts.post_date DESC
        ";

        return count($wpdb->get_col($query));
    }

    /*
    |--------------------------------------------------------------------------
    | AJAX Functions
    |--------------------------------------------------------------------------
     */

    /**
     * AJAX Redeem points for user.
     *
     * @since 1.0
     * @access public
     */
    public function ajax_redeem_points_for_user()
    {
        if (!defined('DOING_AJAX') || !DOING_AJAX) {
            $response = array('status' => 'fail', 'error_msg' => __('Invalid AJAX call', 'loyalty-program-for-woocommerce'));
        } elseif (!isset($_POST['_wpnonce']) || !wp_verify_nonce($_POST['_wpnonce'], 'acfw_redeem_points_for_user')) {
            $response = array('status' => 'fail', 'error_msg' => __('You are not allowed to do this', 'loyalty-program-for-woocommerce'));
        } elseif (!isset($_POST['redeem_points']) || !$_POST['redeem_points']) {
            $response = array('status' => 'fail', 'error_msg' => __('Redemption failed. Please make sure that you have sufficient points or that the points redeemed is above the set minimum.', 'loyalty-program-for-woocommerce'));
        } else {

            $points = intval($_POST['redeem_points']);
            $user   = wp_get_current_user();

            $coupon = $this->redeem_points_for_user($points, $user->ID);

            if ($coupon instanceof \WC_Coupon) {

                $points   = (int) LPFW()->Calculate->get_user_total_points($user->ID);
                $response = array(
                    'status'  => 'success',
                    'message' => __('Points successfully redeemed.', 'loyalty-program-for-woocommerce'),
                    'code'    => $coupon->get_code(),
                    'amount'  => wc_price($coupon->get_amount()),
                    'date'    => $coupon->get_date_created()->date_i18n('F j, Y g:i a'),
                    'points'  => $points,
                    'action'  => get_permalink($coupon->get_id()),
                    'worth'   => wc_price(LPFW()->Calculate->calculate_redeem_points_worth($points)),
                );

            } else {
                $response = array('status' => 'fail', 'error_msg' => __('Redemption failed. Please make sure that you have sufficient points or that the points redeemed is above the set minimum.', 'loyalty-program-for-woocommerce'));
            }

        }

        @header('Content-Type: application/json; charset=' . get_option('blog_charset'));
        echo wp_json_encode($response);
        wp_die();
    }

    /*
    |--------------------------------------------------------------------------
    | Fulfill implemented interface contracts
    |--------------------------------------------------------------------------
     */

    /**
     * Execute codes that needs to run plugin activation.
     *
     * @since 1.0
     * @access public
     * @implements LPFW\Interfaces\Initializable_Interface
     */
    public function initialize()
    {
        add_action('wp_ajax_acfw_redeem_points_for_user', array($this, 'ajax_redeem_points_for_user'));
    }

    /**
     * Execute User_Points class.
     *
     * @since 1.0
     * @access public
     * @inherit LPFW\Interfaces\Model_Interface
     */
    public function run()
    {
        add_action('template_redirect', array($this, 'apply_redeemed_coupon_to_cart'));
    }

}
