<?php
namespace LPFW\Models\REST_API;

use LPFW\Abstracts\Abstract_Main_Plugin_Class;
use LPFW\Helpers\Helper_Functions;
use LPFW\Helpers\Plugin_Constants;
use LPFW\Interfaces\Model_Interface;

if (!defined('ABSPATH')) {
    exit;
}
// Exit if accessed directly

/**
 * Model that houses the Settings module logic.
 * Public Model.
 *
 * @since 1.2
 */
class API_Dashboard implements Model_Interface
{

    /*
    |--------------------------------------------------------------------------
    | Class Properties
    |--------------------------------------------------------------------------
     */

    /**
     * Property that holds the single main instance of URL_Coupon.
     *
     * @since 1.2
     * @access private
     * @var Cart_Conditions
     */
    private static $_instance;

    /**
     * Model that houses all the plugin constants.
     *
     * @since 1.2
     * @access private
     * @var Plugin_Constants
     */
    private $_constants;

    /**
     * Property that houses all the helper functions of the plugin.
     *
     * @since 1.2
     * @access private
     * @var Helper_Functions
     */
    private $_helper_functions;

    /**
     * Property that houses the ACFW_Settings instance.
     *
     * @since 1.2
     * @access private
     * @var ACFW_Settings
     */
    private $_acfw_settings;

    /**
     * Custom REST API base.
     *
     * @since 1.2
     * @access private
     * @var string
     */
    private $_base = 'dashboard';

    /**
     * Property that holds all settings sections.
     *
     * @since 1.2
     * @access private
     * @var array
     */
    private $_settings_sections;

    /**
     * Property that holds all settings sections options.
     *
     * @since 1.2
     * @access private
     * @var array
     */
    private $_sections_options;

    /*
    |--------------------------------------------------------------------------
    | Class Methods
    |--------------------------------------------------------------------------
     */

    /**
     * Class constructor.
     *
     * @since 1.2
     * @access public
     *
     * @param Abstract_Main_Plugin_Class $main_plugin      Main plugin object.
     * @param Plugin_Constants           $constants        Plugin constants object.
     * @param Helper_Functions           $helper_functions Helper functions object.
     */
    public function __construct(Abstract_Main_Plugin_Class $main_plugin, Plugin_Constants $constants, Helper_Functions $helper_functions)
    {
        $this->_constants        = $constants;
        $this->_helper_functions = $helper_functions;

        $main_plugin->add_to_all_plugin_models($this);
        $main_plugin->add_to_public_models($this);

    }

    /**
     * Ensure that only one instance of this class is loaded or can be loaded ( Singleton Pattern ).
     *
     * @since 1.2
     * @access public
     *
     * @param Abstract_Main_Plugin_Class $main_plugin      Main plugin object.
     * @param Plugin_Constants           $constants        Plugin constants object.
     * @param Helper_Functions           $helper_functions Helper functions object.
     * @return Cart_Conditions
     */
    public static function get_instance(Abstract_Main_Plugin_Class $main_plugin, Plugin_Constants $constants, Helper_Functions $helper_functions)
    {
        if (!self::$_instance instanceof self) {
            self::$_instance = new self($main_plugin, $constants, $helper_functions);
        }

        return self::$_instance;

    }

    /*
    |--------------------------------------------------------------------------
    | Routes.
    |--------------------------------------------------------------------------
     */

    /**
     * Register settings API routes.
     *
     * @since 1.2
     * @access public
     */
    public function register_routes()
    {
        \register_rest_route(
            $this->_constants->REST_API_NAMESPACE,
            '/' . $this->_base,
            array(
                array(
                    'methods'             => \WP_REST_Server::READABLE,
                    'permission_callback' => array($this, 'get_admin_permissions_check'),
                    'callback'            => array($this, 'get_dashboard_data'),
                ),
            )
        );

        do_action('acfw_after_register_routes');
    }

    /*
    |--------------------------------------------------------------------------
    | Permissions.
    |--------------------------------------------------------------------------
     */

    /**
     * Checks if a given request has access to read list of settings options.
     *
     * @since 1.2
     * @access public
     *
     * @param WP_REST_Request $request Full details about the request.
     * @return true|WP_Error True if the request has read access, WP_Error object otherwise.
     */
    public function get_admin_permissions_check($request)
    {
        if (!current_user_can('manage_woocommerce')) {
            return new \WP_Error('rest_forbidden_context', $this->_constants->FORBIDDEN_API_ENDPOINT_TEXT, array('status' => \rest_authorization_required_code()));
        }

        return apply_filters('acfw_get_dashboard_admin_permissions_check', $this->_helper_functions->check_if_valid_api_request($request));
    }

    /*
    |--------------------------------------------------------------------------
    | Getter methods.
    |--------------------------------------------------------------------------
     */

    /**
     * Get dashboard data.
     *
     * @since 1.0
     * @access public
     *
     * @param WP_REST_Request $request Full details about the request.
     * @return WP_REST_Response|WP_Error Response object on success, or WP_Error object on failure.
     */
    public function get_dashboard_data($request)
    {
        $response = \rest_ensure_response($this->_query_dashboard_data());

        return apply_filters('lpfw_api_points_status', $response);
    }

    /*
    |--------------------------------------------------------------------------
    | Queries.
    |--------------------------------------------------------------------------
     */

    /**
     * Calculate points status data with provided raw data from _acfw_loyalprog_entries database.
     * Expected properties: user_id, entry_type, entry_action, entry_amount.
     *
     * @since 1.0
     * @access public
     *
     * @param array $raw_data List of raw data.
     * @return array Calculated points data.
     */
    public function calculate_points_status_data($raw_data)
    {
        $status = array(
            'total'     => 0,
            'unclaimed' => 0,
            'claimed'   => 0,
            'expired'   => 0,
            'deducted'  => 0,
        );
        $sources         = array();
        $customers       = array();
        $claimed_actions = array('coupon');

        foreach ($raw_data as $row) {

            $cid    = absint($row['user_id']);
            $type   = $row['entry_type'];
            $action = $row['entry_action'];
            $amount = intval($row['entry_amount']);

            // increment points statuses.
            if ($type === 'earn') {
                $status['total'] += $amount;
            }

            if ('redeem' === $type) {

                // increment for claimed, expired and deducted points statuses.
                switch ($action) {
                    case 'coupon':
                        $status['claimed'] += $amount;
                        break;
                    case 'expire':
                        $status['expired'] += $amount;
                        break;
                    case 'admin_adjust':
                        $status['deducted'] += $amount;
                        break;
                }

            } else { // handle unclaimed/earned points

                // create source entry for action.
                if (!isset($sources[$action])) {
                    $sources[$action] = 0;
                }

                // increment points sources by action
                $sources[$action] += $amount;
            }

            // create entry for customer
            if (!isset($customers[$cid])) {
                $customers[$cid] = 0;
            }

            // increment/decrement points for customer
            if ($type === 'earn') {
                $customers[$cid] += $amount;
            } else {
                $customers[$cid] -= $amount;
            }
        }

        // calculate unclaimed points.
        $status['unclaimed'] = $status['total'] - $status['claimed'] - $status['expired'] - $status['deducted'];

        return array(
            'status'    => $this->_format_points_status_data($status),
            'sources'   => $this->_format_points_sources_data($sources),
            'customers' => $this->_format_customer_points_data($customers),
        );
    }

    /**
     * Query points status data.
     *
     * @since 1.0
     * @access private
     *
     * @return array Dashboard data.
     */
    private function _query_dashboard_data()
    {
        global $wpdb;

        $raw_data = $wpdb->get_results(
            "SELECT e.user_id,e.entry_type,e.entry_action,e.entry_amount
            FROM {$wpdb->prefix}acfw_loyalprog_entries AS e
            INNER JOIN {$wpdb->users} AS u ON (u.ID = e.user_id)
            WHERE 1
            ", ARRAY_A);

        return $this->calculate_points_status_data($raw_data);
    }

    /**
     * Format points status data.
     *
     * @since 1.0
     * @access private
     *
     * @param array $status Raw points status data.
     * @return array Formated points status data.
     */
    private function _format_points_status_data($status)
    {
        return array(
            array(
                'label'  => __('Total Points (All Time)', 'loyalty-program-for-woocommerce'),
                'points' => $status['total'],
                'value'  => $this->_get_points_price_worth($status['total']),
            ),
            array(
                'label'  => __('Unclaimed Points', 'loyalty-program-for-woocommerce'),
                'points' => $status['unclaimed'],
                'value'  => $this->_get_points_price_worth($status['unclaimed']),
            ),
            array(
                'label'  => __('Claimed Points', 'loyalty-program-for-woocommerce'),
                'points' => $status['claimed'],
                'value'  => $this->_get_points_price_worth($status['claimed']),
            ),
            array(
                'label'  => __('Expired Points', 'loyalty-program-for-woocommerce'),
                'points' => $status['expired'],
                'value'  => $this->_get_points_price_worth($status['expired']),
            ),
            array(
                'label'  => __('Deducted', 'loyalty-program-for-woocommerce'),
                'points' => $status['deducted'],
                'value'  => $this->_get_points_price_worth($status['deducted']),
            ),
        );
    }

    /**
     * Format points sources data.
     *
     * @since 1.0
     * @access private
     *
     * @param array $sources Raw points sources data.
     * @return array Formatted points sources data.
     */
    private function _format_points_sources_data($sources)
    {
        return array(
            array(
                'label'  => __('Purchasing products', 'loyalty-program-for-woocommerce'),
                'points' => $this->_get_points_by_source('buy_product', $sources),
            ),
            array(
                'label'  => __('Leaving a product review', 'loyalty-program-for-woocommerce'),
                'points' => $this->_get_points_by_source('product_review', $sources),
            ),
            array(
                'label'  => __('Commenting on a blog post', 'loyalty-program-for-woocommerce'),
                'points' => $this->_get_points_by_source('blog_comment', $sources),
            ),
            array(
                'label'  => __('Registering as a user/customer', 'loyalty-program-for-woocommerce'),
                'points' => $this->_get_points_by_source('user_register', $sources),
            ),
            array(
                'label'  => __('After completing first order', 'loyalty-program-for-woocommerce'),
                'points' => $this->_get_points_by_source('first_order', $sources),
            ),
            array(
                'label'  => __('Spending over a certain amount', 'loyalty-program-for-woocommerce'),
                'points' => $this->_get_points_by_source('high_spend', $sources),
            ),
            array(
                'label'  => __('Extra points during a period', 'loyalty-program-for-woocommerce'),
                'points' => $this->_get_points_by_source('within_period', $sources),
            ),
            array(
                'label'  => __('Admin Adjustment (increased)', 'loyalty-program-for-woocommerce'),
                'points' => $this->_get_points_by_source('admin_adjust', $sources),
            ),
        );
    }

    /**
     * Get points by source slug.
     *
     * @since 1.0
     * @access private
     */
    private function _get_points_by_source($slug, $sources)
    {
        return isset($sources[$slug]) ? $sources[$slug] : 0;
    }

    /**
     * Format customer points data.
     *
     * @since 1.0
     * @access private
     *
     * @param array $customers Raw customer points data.
     * @return array Formatted customer points data.
     */
    private function _format_customer_points_data($customers)
    {
        $data = array();

        // sort customer points descendingly.
        uasort($customers, function ($a, $b) {
            if ($a == $b) {
                return 0;
            }

            return ($a > $b) ? -1 : 1;
        });

        // limit only to 10 customers.
        $customers = array_slice($customers, 0, 10, true);

        foreach ($customers as $cid => $points) {
            $customer = new \WC_Customer($cid);
            $data[]   = array(
                'id'     => $cid,
                'name'   => $this->_helper_functions->get_customer_name($customer),
                'email'  => $this->_helper_functions->get_customer_email($customer),
                'points' => $points,
            );
        }

        return $data;
    }

    /*
    |--------------------------------------------------------------------------
    | Utilities
    |--------------------------------------------------------------------------
     */

    /**
     * Get points price worth.
     *
     * @since 1.0
     * @access private
     *
     * @param int $points Points amount
     * @param string Pirce worth.
     */
    private function _get_points_price_worth($points)
    {
        $amount = LPFW()->Calculate->calculate_redeem_points_worth($points);

        return $this->_helper_functions->api_wc_price($amount);
    }

    /*
    |--------------------------------------------------------------------------
    | Fulfill implemented interface contracts
    |--------------------------------------------------------------------------
     */

    /**
     * Execute Settings class.
     *
     * @since 1.2
     * @access public
     * @inherit LPFW\Interfaces\Model_Interface
     */
    public function run()
    {
        add_action('rest_api_init', array($this, 'register_routes'));
    }

}
