<?php
namespace LPFW\Models\REST_API;

use LPFW\Abstracts\Abstract_Main_Plugin_Class;
use LPFW\Helpers\Helper_Functions;
use LPFW\Helpers\Plugin_Constants;
use LPFW\Interfaces\Model_Interface;

if (!defined('ABSPATH')) {
    exit;
}
// Exit if accessed directly

/**
 * Model that houses the Settings module logic.
 * Public Model.
 *
 * @since 1.2
 */
class API_Customers implements Model_Interface
{

    /*
    |--------------------------------------------------------------------------
    | Class Properties
    |--------------------------------------------------------------------------
     */

    /**
     * Property that holds the single main instance of URL_Coupon.
     *
     * @since 1.2
     * @access private
     * @var Cart_Conditions
     */
    private static $_instance;

    /**
     * Model that houses all the plugin constants.
     *
     * @since 1.2
     * @access private
     * @var Plugin_Constants
     */
    private $_constants;

    /**
     * Property that houses all the helper functions of the plugin.
     *
     * @since 1.2
     * @access private
     * @var Helper_Functions
     */
    private $_helper_functions;

    /**
     * Custom REST API base.
     *
     * @since 1.2
     * @access private
     * @var string
     */
    private $_base = 'customers';

    /**
     * Property that holds all settings sections.
     *
     * @since 1.2
     * @access private
     * @var array
     */
    private $_settings_sections;

    /**
     * Property that holds all settings sections options.
     *
     * @since 1.2
     * @access private
     * @var array
     */
    private $_sections_options;

    /*
    |--------------------------------------------------------------------------
    | Class Methods
    |--------------------------------------------------------------------------
     */

    /**
     * Class constructor.
     *
     * @since 1.2
     * @access public
     *
     * @param Abstract_Main_Plugin_Class $main_plugin      Main plugin object.
     * @param Plugin_Constants           $constants        Plugin constants object.
     * @param Helper_Functions           $helper_functions Helper functions object.
     */
    public function __construct(Abstract_Main_Plugin_Class $main_plugin, Plugin_Constants $constants, Helper_Functions $helper_functions)
    {
        $this->_constants        = $constants;
        $this->_helper_functions = $helper_functions;

        $main_plugin->add_to_all_plugin_models($this);
        $main_plugin->add_to_public_models($this);

    }

    /**
     * Ensure that only one instance of this class is loaded or can be loaded ( Singleton Pattern ).
     *
     * @since 1.2
     * @access public
     *
     * @param Abstract_Main_Plugin_Class $main_plugin      Main plugin object.
     * @param Plugin_Constants           $constants        Plugin constants object.
     * @param Helper_Functions           $helper_functions Helper functions object.
     * @return Cart_Conditions
     */
    public static function get_instance(Abstract_Main_Plugin_Class $main_plugin, Plugin_Constants $constants, Helper_Functions $helper_functions)
    {
        if (!self::$_instance instanceof self) {
            self::$_instance = new self($main_plugin, $constants, $helper_functions);
        }

        return self::$_instance;

    }

    /*
    |--------------------------------------------------------------------------
    | Routes.
    |--------------------------------------------------------------------------
     */

    /**
     * Register settings API routes.
     *
     * @since 1.0
     * @access public
     */
    public function register_routes()
    {
        \register_rest_route(
            $this->_constants->REST_API_NAMESPACE,
            '/' . $this->_base,
            array(
                array(
                    'methods'             => \WP_REST_Server::READABLE,
                    'permission_callback' => array($this, 'get_admin_permissions_check'),
                    'callback'            => array($this, 'api_search_customers'),
                ),
            )
        );

        \register_rest_route(
            $this->_constants->REST_API_NAMESPACE,
            '/' . $this->_base . '/status/(?P<id>[\w]+)',
            array(
                array(
                    'methods'             => \WP_REST_Server::READABLE,
                    'permission_callback' => array($this, 'get_admin_permissions_check'),
                    'callback'            => array($this, 'get_customer_points_status'),
                ),
            )
        );

        \register_rest_route(
            $this->_constants->REST_API_NAMESPACE,
            '/' . $this->_base . '/history/(?P<id>[\w]+)',
            array(
                array(
                    'methods'             => \WP_REST_Server::READABLE,
                    'permission_callback' => array($this, 'get_admin_permissions_check'),
                    'callback'            => array($this, 'get_customer_points_history'),
                ),
            )
        );

        \register_rest_route(
            $this->_constants->REST_API_NAMESPACE,
            '/' . $this->_base . '/points/(?P<id>[\w]+)',
            array(
                array(
                    'methods'             => \WP_REST_Server::CREATABLE,
                    'permission_callback' => array($this, 'get_admin_permissions_check'),
                    'callback'            => array($this, 'adjust_customer_points'),
                ),
            )
        );
    }

    /*
    |--------------------------------------------------------------------------
    | Permissions.
    |--------------------------------------------------------------------------
     */

    /**
     * Checks if a given request has access to read list of settings options.
     *
     * @since 1.0
     * @access public
     *
     * @param WP_REST_Request $request Full details about the request.
     * @return true|WP_Error True if the request has read access, WP_Error object otherwise.
     */
    public function get_admin_permissions_check($request)
    {
        if (!current_user_can('manage_woocommerce')) {
            return new \WP_Error('rest_forbidden_context', $this->_constants->FORBIDDEN_API_ENDPOINT_TEXT, array('status' => \rest_authorization_required_code()));
        }

        return apply_filters('lpfw_get_customers_admin_permissions_check', $this->_helper_functions->check_if_valid_api_request($request));
    }

    /*
    |--------------------------------------------------------------------------
    | Requests handlers.
    |--------------------------------------------------------------------------
     */

    /**
     * Retrieves a collection of customers.
     *
     * @since 1.0
     * @access public
     *
     * @param WP_REST_Request $request Full details about the request.
     * @return WP_REST_Response|WP_Error Response object on success, or WP_Error object on failure.
     */
    public function api_search_customers($request)
    {
        $search   = sanitize_text_field($request->get_param('search'));
        $response = \rest_ensure_response($this->_query_customers($search));

        return apply_filters('lpfw_api_search_customers', $response);
    }

    /**
     * Retrieves customer points status data.
     *
     * @since 1.0
     * @access public
     *
     * @param WP_REST_Request $request Full details about the request.
     * @return WP_REST_Response|WP_Error Response object on success, or WP_Error object on failure.
     */
    public function get_customer_points_status($request)
    {
        $cid      = absint($request['id']);
        $response = \rest_ensure_response($this->_query_customer_points_status_and_sources($cid));

        return apply_filters('lpfw_api_points_status', $response);
    }

    /**
     * Retrieves customer points history data.
     *
     * @since 1.0
     * @access public
     *
     * @param WP_REST_Request $request Full details about the request.
     * @return WP_REST_Response|WP_Error Response object on success, or WP_Error object on failure.
     */
    public function get_customer_points_history($request)
    {
        $cid      = absint($request['id']);
        $page     = absint($request->get_param('page'));
        $response = \rest_ensure_response($this->query_customer_points_history($cid, $page));

        if (!$page || $page === 1) {
            $response->header('X-TOTAL', $this->query_total_customer_points_history($cid));
        }

        return apply_filters('lpfw_api_get_customer_points_history', $response);
    }

    /**
     * Adjust customer points request handler.
     *
     * @since 1.0
     * @access public
     *
     * @param WP_REST_Request $request Full details about the request.
     * @return WP_REST_Response|WP_Error Response object on success, or WP_Error object on failure.
     */
    public function adjust_customer_points($request)
    {
        $cid           = absint($request['id']);
        $temp          = sanitize_text_field($request->get_param('type'));
        $type          = $temp == 'increase' ? 'earn' : 'redeem';
        $user_points   = (int) \LPFW()->Calculate->get_user_total_points($cid);
        $adjust_points = absint($request->get_param('points'));
        $adjust_points = $temp == 'increase' ? $adjust_points : min($user_points, $adjust_points);

        $entry_id = \LPFW()->Entries->insert_entry($cid, $type, 'admin_adjust', $adjust_points, get_current_user_id());
        if (!$entry_id) {
            return new \WP_Error('error_adjusting_points', __('Failed adjusting points for customer.', 'loyalty-program-for-woocommerce'));
        }

        $update   = $this->_query_customer_points_status_and_sources($cid);
        $response = \rest_ensure_response(array(
            'message' => __('Successfully adjusted points for customer.', 'loyalty-program-for-woocommerce'),
            'status'  => $update['status'],
            'sources' => $update['sources'],
        ));

        return apply_filters('lpfw_api_adjust_customer_points', $response);
    }

    /*
    |--------------------------------------------------------------------------
    | Queries
    |--------------------------------------------------------------------------
     */

    /**
     * Query customers.
     *
     * @since 1.0
     * @access private
     *
     * @param string $search Search query.
     * @return array List of customers.
     */
    private function _query_customers($search)
    {
        global $wpdb;

        $concat_search = 'billing_|nickname|first_name|last_name';
        $query         = "SELECT c.*,
            GROUP_CONCAT( IF(cm.meta_key REGEXP '{$concat_search}', cm.meta_key, null) ORDER BY cm.meta_key DESC SEPARATOR '||' ) AS meta_keys,
            GROUP_CONCAT( IF(cm.meta_key REGEXP '{$concat_search}', IFNULL(cm.meta_value, ''), null) ORDER BY cm.meta_key DESC SEPARATOR '||' ) AS meta_values
            FROM {$wpdb->users} AS c
            INNER JOIN {$wpdb->usermeta} AS cm ON (c.ID = cm.user_id)
            WHERE 1
            GROUP BY c.ID
            HAVING (c.ID LIKE '%{$search}%'
                OR c.user_login LIKE '%{$search}%'
                OR c.user_nicename LIKE '%{$search}%'
                OR c.display_name LIKE '%{$search}'
                OR c.user_email LIKE '%{$search}%'
                OR meta_values LIKE '%{$search}%')
            ORDER BY c.user_registered DESC
        ";

        return array_map(function ($cid) {
            $customer = new \WC_Customer($cid);
            return array(
                'id'    => absint($cid),
                'name'  => $this->_helper_functions->get_customer_name($customer),
                'email' => $this->_helper_functions->get_customer_email($customer),
            );
        }, $wpdb->get_col($query));
    }

    /**
     * Query customer points status and sources data.
     *
     * @since 1.0
     * @access private
     *
     * @param int $cid Customer ID.
     * @return array Points status and sources data.
     */
    private function _query_customer_points_status_and_sources($cid)
    {
        global $wpdb;

        $raw_data = $wpdb->get_results(
            "SELECT user_id,entry_type,entry_action,entry_amount
                FROM {$wpdb->prefix}acfw_loyalprog_entries
                WHERE user_id = {$cid}
            ",
            ARRAY_A
        );

        $data = LPFW()->API_Dashboard->calculate_points_status_data($raw_data);
        unset($data['customers']);

        return $data;
    }

    /**
     * Query customer points history data.
     *
     * @since 1.0
     * @access private
     *
     * @param int $cid  Customer ID.
     * @param int $page Page number.
     * @return array Points history data.
     */
    public function query_customer_points_history($cid, $page = 1)
    {
        global $wpdb;

        $offset = $page ? ($page - 1) * 10 : 0;
        $query  = "SELECT entry_id,user_id,entry_date,entry_type,entry_action,entry_amount,object_id
            FROM {$wpdb->prefix}acfw_loyalprog_entries
            WHERE user_id = {$cid}
            ORDER BY entry_date DESC
            LIMIT {$offset}, 10
        ";

        $entries = $wpdb->get_results($query, ARRAY_A);

        $activity_labels = array(
            'buy_product'    => __('Purchasing products', 'loyalty-program-for-woocommerce'),
            'blog_comment'   => __('Leaving a blog comment', 'loyalty-program-for-woocommerce'),
            'product_review' => __('Leaving a product review', 'loyalty-program-for-woocommerce'),
            'user_register'  => __('Registering as a user/customer', 'loyalty-program-for-woocommerce'),
            'first_order'    => __('After completing first order', 'loyalty-program-for-woocommerce'),
            'high_spend'     => __('Spending over a certain amount', 'loyalty-program-for-woocommerce'),
            'within_period'  => __('Extra points during a period', 'loyalty-program-for-woocommerce'),
            'coupon'         => __('Redeem Coupon', 'loyalty-program-for-woocommerce'),
            'admin_increase' => __('Admin Adjustment (increase)', 'loyalty-program-for-woocommerce'),
            'admin_decrease' => __('Admin Adjustment (decrease)', 'loyalty-program-for-woocommerce'),
            'expire'         => __('Points expired', 'loyalty-program-for-woocommerce'),
        );

        return array_map(function ($e) use ($activity_labels) {

            $activity  = $e['entry_type'] === 'redeem' && !$e['entry_action'] ? 'coupon' : $e['entry_action'];
            $object_id = !$e['object_id'] && $activity === 'user_register' ? absint($e['user_id']) : absint($e['object_id']);
            $rel_link  = $this->_get_related_object_link($activity, $object_id);
            $rel_label = $this->_get_related_object_label($activity, $object_id, $e['entry_type']);

            // get proper activity label for admin adjust.
            if ($activity === 'admin_adjust') {
                $activity = $e['entry_type'] === 'redeem' ? 'admin_decrease' : 'admin_increase';
            }

            return array(
                'id'        => absint($e['entry_id']),
                'object_id' => $object_id,
                'action'    => $activity,
                'date'      => $this->_format_entry_date($e['entry_date']),
                'activity'  => $activity_labels[$activity],
                'points'    => intval($e['entry_amount']),
                'rel_link'  => $rel_link,
                'rel_label' => $rel_label,
            );

        }, $entries);
    }

    /**
     * Query customer points history data.
     *
     * @since 1.0
     * @access private
     *
     * @param int $cid  Customer ID.
     * @return int Total point entries.
     */
    public function query_total_customer_points_history($cid)
    {
        global $wpdb;

        return (int) $wpdb->get_var("SELECT COUNT(entry_id) FROM {$wpdb->prefix}acfw_loyalprog_entries WHERE user_id = {$cid}");
    }

    /*
    |--------------------------------------------------------------------------
    | Utilities
    |--------------------------------------------------------------------------
     */

    /**
     * Format entry date.
     *
     * @since 1.0
     * @access private
     *
     * @param string $utc_date UTC date (mysql format).
     * @return string Date with display format.
     */
    private function _format_entry_date($utc_date)
    {
        $utczone  = new \DateTimeZone('UTC');
        $timezone = new \DateTimeZone($this->_helper_functions->get_site_current_timezone());
        $datetime = \DateTime::createFromFormat('Y-m-d H:i:s', $utc_date, $utczone);

        $datetime->setTimezone($timezone);

        return $datetime->format('m/d/Y H:i:s');
    }

    /**
     * Get related object link.
     *
     * @since 1.0
     * @access private
     *
     * @param string $activity  Activity type.
     * @param int    $object_id Object ID.
     * @return string Object link.
     */
    private function _get_related_object_link($activity, $object_id)
    {
        switch ($activity) {

            case 'buy_product':
            case 'first_order':
            case 'high_spend':
            case 'within_period':
            case 'coupon':
                return sprintf('%spost.php?post=%s&action=edit', admin_url(), $object_id);

            case 'blog_comment':
            case 'product_review':
                return sprintf('%scomment.php?action=editcomment&c=%s', admin_url(), $object_id);

            case 'user_register':
            case 'admin_adjust':
                return sprintf('%suser-edit.php?user_id=%s', admin_url(), $object_id);

            case 'expire':
                return '';
        }
    }

    /**
     * Get related object label.
     *
     * @since 1.0
     * @access private
     *
     * @param string $activity  Activity type.
     * @param int    $object_id Object ID.
     * @return string Label.
     */
    private function _get_related_object_label($activity, $object_id)
    {
        switch ($activity) {

            case 'buy_product':
            case 'first_order':
            case 'high_spend':
            case 'within_period':
                return __('View Order', 'loyalty-program-for-woocommerce');

            case 'coupon':
                return __('View Coupon', 'loyalty-program-for-woocommerce');

            case 'blog_comment':
                return __('View Comment', 'loyalty-program-for-woocommerce');

            case 'product_review':
                return __('View Review', 'loyalty-program-for-woocommerce');

            case 'user_register':
                return __('View User', 'loyalty-program-for-woocommerce');

            case 'admin_adjust':
                $admin = $this->_helper_functions->get_customer_name($object_id);
                return sprintf(__('Admin: %s', 'loyalty-program-for-woocommerce'), $admin);

            case 'expire':
                return '—';
        }
    }

    /*
    |--------------------------------------------------------------------------
    | Fulfill implemented interface contracts
    |--------------------------------------------------------------------------
     */

    /**
     * Execute Settings class.
     *
     * @since 1.2
     * @access public
     * @inherit LPFW\Interfaces\Model_Interface
     */
    public function run()
    {
        add_action('rest_api_init', array($this, 'register_routes'));
    }

}
