<?php
namespace LPFW\Models;

use LPFW\Abstracts\Abstract_Main_Plugin_Class;

use LPFW\Interfaces\Model_Interface;
use LPFW\Interfaces\Initiable_Interface;

use LPFW\Helpers\Plugin_Constants;
use LPFW\Helpers\Helper_Functions;

if ( !defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Model that houses the logic of extending the coupon system of woocommerce.
 * It houses the logic of handling coupon url.
 * Public Model.
 *
 * @since 1.0
 */
class Entries implements Model_Interface , Initiable_Interface {

    /*
    |--------------------------------------------------------------------------
    | Class Properties
    |--------------------------------------------------------------------------
    */

    /**
     * Property that holds the single main instance of URL_Coupon.
     *
     * @since 1.0
     * @access private
     * @var Entries
     */
    private static $_instance;

    /**
     * Model that houses all the plugin constants.
     *
     * @since 1.0
     * @access private
     * @var Plugin_Constants
     */
    private $_constants;

    /**
     * Property that houses all the helper functions of the plugin.
     *
     * @since 1.0
     * @access private
     * @var Helper_Functions
     */
    private $_helper_functions;

    /*
    |--------------------------------------------------------------------------
    | Class Methods
    |--------------------------------------------------------------------------
    */

    /**
     * Class constructor.
     *
     * @since 1.0
     * @access public
     *
     * @param Abstract_Main_Plugin_Class $main_plugin      Main plugin object.
     * @param Plugin_Constants           $constants        Plugin constants object.
     * @param Helper_Functions           $helper_functions Helper functions object.
     */
    public function __construct( Abstract_Main_Plugin_Class $main_plugin , Plugin_Constants $constants , Helper_Functions $helper_functions ) {

        $this->_constants        = $constants;
        $this->_helper_functions = $helper_functions;

        $main_plugin->add_to_all_plugin_models( $this );
        $main_plugin->add_to_public_models( $this );

    }

    /**
     * Ensure that only one instance of this class is loaded or can be loaded ( Singleton Pattern ).
     *
     * @since 1.0
     * @access public
     *
     * @param Abstract_Main_Plugin_Class $main_plugin      Main plugin object.
     * @param Plugin_Constants           $constants        Plugin constants object.
     * @param Helper_Functions           $helper_functions Helper functions object.
     * @return Entries
     */
    public static function get_instance( Abstract_Main_Plugin_Class $main_plugin , Plugin_Constants $constants , Helper_Functions $helper_functions ) {

        if ( !self::$_instance instanceof self )
            self::$_instance = new self( $main_plugin , $constants , $helper_functions );

        return self::$_instance;

    }




    /*
    |--------------------------------------------------------------------------
    | CRUD Methods
    |--------------------------------------------------------------------------
    */

    /**
     * Insert points entry.
     * 
     * @since 1.0
     * @access public
     * 
     * @global wpdb $wpdb Object that contains a set of functions used to interact with a database.
     * 
     * @param int    $user_id   User id.
     * @param string $type      Entry type (redeem or earn).
     * @param string $action    Entry action.
     * @param int    $amount    Entry amount.
     * @param int    $object_id Related object ID (posts, order, comments, etc.).
     * @return bool True if inserted, false otherwise.
     */
    public function insert_entry( $user_id , $type , $action , $amount , $object_id = 0 ) {

        global $wpdb;

        $lp_entries_db = $wpdb->prefix . $this->_constants->DB_TABLE_NAME;

        $check = $wpdb->insert( 
            $lp_entries_db,
            array(
                'user_id'      => $user_id,
                'entry_date'   => current_time( 'mysql' , true ),
                'entry_type'   => $type,
                'entry_action' => $action,
                'entry_amount' => $amount,
                'object_id'    => $object_id
            )
        );

        return $check ? $wpdb->insert_id : 0;
    }

    /**
     * Update points entry.
     * 
     * @since 1.9
     * @access public
     * 
     * @global wpdb $wpdb Object that contains a set of functions used to interact with a database.
     * 
     * @param int   $entry_id Entry ID.
     * @param array $changes  List of entry changes.
     * @return bool True if updated, false otherwise.
     */
    public function update_entry( $entry_id , $changes = array() ) {

        global $wpdb;

        if ( ! $entry_id ) return;

        $lp_entries_db = $wpdb->prefix . $this->_constants->DB_TABLE_NAME;
        $verified      = array();

        if ( isset( $changes[ 'user_id' ] ) && gettype( $changes[ 'user_id' ] ) == 'integer' && $changes[ 'user_id' ] )
            $verified[ 'user_id' ] = $changes[ 'user_id' ];

        if ( isset( $changes[ 'entry_date' ] ) && $changes[ 'entry_date' ] )
            $verified[ 'entry_date' ] = $changes[ 'entry_date' ];

        if ( isset( $changes[ 'entry_type' ] ) && $changes[ 'entry_type' ] )
            $verified[ 'entry_type' ] = $changes[ 'entry_type' ];

        if ( isset( $changes[ 'entry_action' ] ) && $changes[ 'entry_action' ] )
            $verified[ 'entry_action' ] = $changes[ 'entry_action' ];

        if ( isset( $changes[ 'entry_amount' ] ) && gettype( $changes[ 'entry_amount' ] ) == 'integer' && $changes[ 'entry_amount' ] )
            $verified[ 'entry_amount' ] = $changes[ 'entry_amount' ];

        return $wpdb->update(
            $lp_entries_db,
            $verified,
            array( 'entry_id' => absint( $entry_id ) )
        );
    }

    /**
     * Delete points entry.
     * 
     * @since 1.9
     * @access public
     * 
     * @global wpdb $wpdb Object that contains a set of functions used to interact with a database.
     * 
     * @param int $entry_id Entry ID.
     * @return bool True if deleted, false otherwise.
     */
    public function delete_entry( $entry_id ) {

        global $wpdb;

        if ( ! $entry_id ) return;

        $lp_entries_db = $wpdb->prefix . $this->_constants->DB_TABLE_NAME;

        return $wpdb->delete( $lp_entries_db , array( 'entry_id' => absint( $entry_id ) ) );
    }




    /*
    |--------------------------------------------------------------------------
    | Fulfill implemented interface contracts
    |--------------------------------------------------------------------------
    */

    /**
     * Execute codes that needs to run plugin activation.
     *
     * @since 1.0
     * @access public
     * @implements LPFW\Interfaces\Initializable_Interface
     */
    public function initialize() {
    }

    /**
     * Execute Entries class.
     *
     * @since 1.0
     * @access public
     * @inherit LPFW\Interfaces\Model_Interface
     */
    public function run() {
    }

}
