<?php
namespace LPFW\Models;

use LPFW\Abstracts\Abstract_Main_Plugin_Class;
use LPFW\Helpers\Helper_Functions;
use LPFW\Helpers\Plugin_Constants;
use LPFW\Interfaces\Model_Interface;

if (!defined('ABSPATH')) {
    exit;
}
// Exit if accessed directly

/**
 * Model that houses the logic of extending the coupon system of woocommerce.
 * It houses the logic of handling coupon url.
 * Public Model.
 *
 * @since 1.0
 */
class Calculate implements Model_Interface
{

    /*
    |--------------------------------------------------------------------------
    | Class Properties
    |--------------------------------------------------------------------------
     */

    /**
     * Property that holds the single main instance of URL_Coupon.
     *
     * @since 1.0
     * @access private
     * @var Calculate
     */
    private static $_instance;

    /**
     * Model that houses all the plugin constants.
     *
     * @since 1.0
     * @access private
     * @var Plugin_Constants
     */
    private $_constants;

    /**
     * Property that houses all the helper functions of the plugin.
     *
     * @since 1.0
     * @access private
     * @var Helper_Functions
     */
    private $_helper_functions;

    /**
     * Property that houses the datetime object of the user's last active date.
     *
     * @since 1.0
     * @access private
     * @var DateTime
     */
    private $_last_active;

    /*
    |--------------------------------------------------------------------------
    | Class Methods
    |--------------------------------------------------------------------------
     */

    /**
     * Class constructor.
     *
     * @since 1.0
     * @access public
     *
     * @param Abstract_Main_Plugin_Class $main_plugin      Main plugin object.
     * @param Plugin_Constants           $constants        Plugin constants object.
     * @param Helper_Functions           $helper_functions Helper functions object.
     */
    public function __construct(Abstract_Main_Plugin_Class $main_plugin, Plugin_Constants $constants, Helper_Functions $helper_functions)
    {
        $this->_constants        = $constants;
        $this->_helper_functions = $helper_functions;

        $main_plugin->add_to_all_plugin_models($this);
        $main_plugin->add_to_public_models($this);

    }

    /**
     * Ensure that only one instance of this class is loaded or can be loaded ( Singleton Pattern ).
     *
     * @since 1.0
     * @access public
     *
     * @param Abstract_Main_Plugin_Class $main_plugin      Main plugin object.
     * @param Plugin_Constants           $constants        Plugin constants object.
     * @param Helper_Functions           $helper_functions Helper functions object.
     * @return Calculate
     */
    public static function get_instance(Abstract_Main_Plugin_Class $main_plugin, Plugin_Constants $constants, Helper_Functions $helper_functions)
    {
        if (!self::$_instance instanceof self) {
            self::$_instance = new self($main_plugin, $constants, $helper_functions);
        }

        return self::$_instance;

    }

    /*
    |--------------------------------------------------------------------------
    | Calculate methods
    |--------------------------------------------------------------------------
     */

    /**
     * Calculate the total points earned with a given cart/order subtotal.
     *
     * @since 1.9
     * @access public
     *
     * @param double $calc_total Cart/order calculated total.
     * @return int Points equivalent.
     */
    public function calculate_points_earn($calc_total)
    {
        $multiplier = abs($this->_helper_functions->sanitize_price(get_option($this->_constants->COST_POINTS_RATIO, '1')));
        return intval($calc_total * $multiplier);
    }

    /**
     * Calculate points worth.
     *
     * @since 1.9
     * @access public
     *
     * @param int $points User points.
     * @return float Amount points worth.
     */
    public function calculate_redeem_points_worth($points, $is_filter = true)
    {
        $ratio = (int) get_option($this->_constants->REDEEM_POINTS_RATIO, '10');
        $value = max(0, $points / $ratio);

        return $is_filter ? apply_filters('lpfw_filter_amount', $value) : $value;
    }

    /**
     * Get minimum threshold value.
     *
     * @since 1.0
     * @access public
     *
     * @return float Minimum threshold value.
     */
    public function get_minimum_threshold()
    {
        $raw = get_option($this->_constants->MINIMUM_POINTS_THRESHOLD, '0');
        return abs($this->_helper_functions->sanitize_price($raw));
    }

    /**
     * Get total based on points calculated options.
     *
     * @since 1.0
     * @access public
     *
     * @param WC_Order | null $order Order object.
     * @return float Total calculated amount.
     */
    public function get_total_based_on_points_calculate_options($order = null)
    {
        $options    = $this->_helper_functions->get_enabled_points_calc_options();
        $total      = is_object($order) ? $order->get_total('edit') : WC()->cart->get_total('edit');
        $subtotal   = is_object($order) ? $order->get_subtotal() : WC()->cart->get_subtotal();
        $shipping   = is_object($order) ? $order->get_shipping_total('edit') : WC()->cart->get_shipping_total();
        $calculated = (float) $total;

        // get total fees
        if (is_object($order)) {

            $fees_total = 0;
            foreach ($order->get_fees() as $item) {

                $fee_total = $item->get_total();

                if (0 > $fee_total) {
                    $max_discount = round($subtotal + $fees_total + $shipping, wc_get_price_decimals()) * -1;

                    if ($fee_total < $max_discount) {
                        $item->set_total($max_discount);
                    }
                }

                $fees_total += $item->get_total();
            }

        } else {
            $fees_total = WC()->cart->get_fee_total();
        }

        // get shipping overrides discount.
        $shipping_discount = 0;
        if (is_object($order)) {
            $shipping_discount = array_reduce($order->get_fees(), function ($c, $f) {
                return strpos($f->get_name(), '[shipping_discount]') !== false ? $c + $f->get_total() : $c;
            }, 0);
        } else {
            $shipping_discount = array_reduce(WC()->cart->get_fees(), function ($c, $f) {
                return strpos($f->name, '[shipping_discount]') !== false ? $c + $f->total : $c;
            }, 0);
        }

        // deduct shipping overrides discount from fees total.
        $fees_total = max(0, $fees_total - $shipping_discount);

        // discounts
        if (!in_array('discounts', $options)) {
            $discounts = is_object($order) ? $order->get_total_discount() : WC()->cart->get_discount_total();
            $calculated += $discounts;
        }

        // tax
        if (!in_array('tax', $options)) {
            $tax_total = is_object($order) ? $order->get_total_tax('edit') : WC()->cart->get_total_tax();
            $calculated -= $tax_total;
        }

        // shipping
        if (!in_array('shipping', $options)) {
            $calculated -= $shipping;
        }

        // shipping overrides discount
        if (!in_array('discounts', $options) || !in_array('shipping', $options)) {
            $calculated += abs($shipping_discount);
        }

        // fees
        if (!in_array('fees', $options)) {
            $calculated -= $fees_total;
        }

        return apply_filters('lpfw_calculate_totals', max(0, $calculated), $order);
    }

    /**
     * Calculate high spend points based on subtotal value excluding discounts.
     *
     * @since 1.0
     * @access public
     *
     * @param float $order_total Order/cart total amount.
     * @return int High spend calculated points.
     */
    public function calculate_high_spend_points($order_total)
    {
        $breakpoints = get_option($this->_constants->EARN_POINTS_BREAKPOINTS, array());
        $breakpoints = is_array($breakpoints) ? $breakpoints : json_decode($breakpoints, true);
        $order_total = (float) $order_total;
        $points      = 0;

        if (!is_array($breakpoints) || empty($breakpoints)) {
            return $points;
        }

        // sort breakpoints from highest to lowest to get concise value.
        usort($breakpoints, function ($a, $b) {
            if ($a['sanitized'] == $b['sanitized']) {
                return 0;
            }

            return ($a['sanitized'] > $b['sanitized']) ? -1 : 1;
        });

        foreach ($breakpoints as $breakpoint) {

            $amount = (float) $breakpoint['sanitized'];

            if ($order_total >= $amount) {
                $points = absint($breakpoint['points']);
                break;
            }
        }

        return $points;
    }

    /**
     * Get matching period points based on order.
     *
     * @since 1.11
     * @access public
     *
     * @param WC_Order $order Order object.
     * @return int Earned points.
     */
    public function get_matching_period_points($order = null)
    {
        $points = 0;
        $data   = get_option($this->_constants->EARN_POINTS_ORDER_PERIOD, array());
        $data   = is_array($data) ? $data : json_decode($data, true);

        if (!is_array($data) || empty($data)) {
            return $points;
        }

        $order_date = $order ? $order->get_date_created() : null;
        $timezone   = $order_date ? $order_date->getTimezone() : new \DateTimeZone($this->_helper_functions->get_site_current_timezone());

        if (!$order_date) {
            $order_date = new \WC_DateTime("now", $timezone);
        }

        foreach ($data as $row) {

            $start_date = \WC_DateTime::createFromFormat('m/d/Y g:i A', $row['sdate'] . ' ' . $row['stime'], $timezone);
            $end_date   = \WC_DateTime::createFromFormat('m/d/Y g:i A', $row['edate'] . ' ' . $row['etime'], $timezone);

            if ($order_date >= $start_date && $order_date <= $end_date) {
                return absint($row['points']);
            }

        }

        return $points;
    }

    /*
    |--------------------------------------------------------------------------
    | User related calculations
    |--------------------------------------------------------------------------
     */

    /**
     * Get total points of user.
     *
     * @since 1.0
     * @access private
     *
     * @global wpdb $wpdb Object that contains a set of functions used to interact with a database.
     *
     * @param int $user_id User ID.
     * @return int User total points.
     */
    public function get_user_total_points($user_id)
    {
        global $wpdb;

        $lp_entries_db = $wpdb->prefix . $this->_constants->DB_TABLE_NAME;
        $last_active   = $wpdb->get_var($wpdb->prepare("SELECT entry_date FROM `$lp_entries_db` WHERE user_id = %d ORDER BY entry_date DESC", $user_id));

        $query = $wpdb->prepare("(SELECT SUM(entry_amount) FROM `$lp_entries_db` WHERE user_id = %d AND entry_type != 'redeem') UNION (SELECT SUM(entry_amount) FROM `$lp_entries_db` WHERE user_id = %d AND entry_type = 'redeem')", $user_id, $user_id);
        $data  = $wpdb->get_results($query, ARRAY_N);

        // subtract reedeemed from earned. if value is negative, then return zero.
        $earned = isset($data[0]) ? intval($data[0][0]) : 0;
        $redeem = isset($data[1]) ? intval($data[1][0]) : $earned; // if second result is not set, it means that the total redeemed is equal to the total earned.
        $points = max(0, $earned - $redeem);

        // if user last active is not valid anymore, then set it as expired.
        if (!$this->_validate_user_last_active($last_active) && $points > 0) {
            \LPFW()->Entries->insert_entry($user_id, 'redeem', 'expire', $points);
            $points = 0;
        }

        // update cached value in user meta.
        update_user_meta($user_id, $this->_constants->USER_TOTAL_POINTS, $points);

        return $points;
    }

    /**
     * Validate user last active date.
     *
     * @since 1.11
     * @access private
     *
     * @param string $last_active Date user was last active in mysql date format.
     * @return bool True if still active, false otherwise.
     */
    private function _validate_user_last_active($last_active)
    {
        $expire_period = (int) get_option($this->_constants->INACTIVE_DAYS_POINTS_EXPIRE, 365);

        if (!$last_active || !$expire_period) {
            return true;
        }

        $utc              = new \DateTimeZone('UTC');
        $timezone         = new \DateTimeZone($this->_helper_functions->get_site_current_timezone());
        $datetime         = new \DateTime("now", $timezone);
        $expire_timestamp = $datetime->getTimestamp() - ($expire_period * DAY_IN_SECONDS);

        $this->_last_active = \DateTime::createFromFormat('Y-m-d H:i:s', $last_active, $utc);
        $this->_last_active->setTimezone($timezone);

        return $this->_last_active->getTimestamp() > $expire_timestamp;
    }

    /**
     * Get user last active date cached value.
     *
     * @since 1.0
     * @access public
     *
     * @return DateTIme
     */
    public function get_last_active()
    {
        return $this->_last_active;
    }

    /*
    |--------------------------------------------------------------------------
    | Fulfill implemented interface contracts
    |--------------------------------------------------------------------------
     */

    /**
     * Execute Calculate class.
     *
     * @since 1.0
     * @access public
     * @inherit LPFW\Interfaces\Model_Interface
     */
    public function run()
    {
    }

}
