<?php
namespace LPFW\Models;

use LPFW\Abstracts\Abstract_Main_Plugin_Class;
use LPFW\Helpers\Helper_Functions;
use LPFW\Helpers\Plugin_Constants;
use LPFW\Interfaces\Initiable_Interface;
use LPFW\Interfaces\Model_Interface;

if (!defined('ABSPATH')) {
    exit;
}
// Exit if accessed directly

/**
 * Model that houses the logic of extending the coupon system of woocommerce.
 * It houses the logic of handling coupon url.
 * Public Model.
 *
 * @since 1.0
 */
class Admin implements Model_Interface, Initiable_Interface
{

    /*
    |--------------------------------------------------------------------------
    | Class Properties
    |--------------------------------------------------------------------------
     */

    /**
     * Property that holds the single main instance of URL_Coupon.
     *
     * @since 1.0
     * @access private
     * @var Admin
     */
    private static $_instance;

    /**
     * Model that houses all the plugin constants.
     *
     * @since 1.0
     * @access private
     * @var Plugin_Constants
     */
    private $_constants;

    /**
     * Property that houses all the helper functions of the plugin.
     *
     * @since 1.0
     * @access private
     * @var Helper_Functions
     */
    private $_helper_functions;

    /**
     * Property that holds user cache in metabox.
     *
     * @since 1.0
     * @access private
     * @var object
     */
    private $_user;

    /*
    |--------------------------------------------------------------------------
    | Class Methods
    |--------------------------------------------------------------------------
     */

    /**
     * Class constructor.
     *
     * @since 1.0
     * @access public
     *
     * @param Abstract_Main_Plugin_Class $main_plugin      Main plugin object.
     * @param Plugin_Constants           $constants        Plugin constants object.
     * @param Helper_Functions           $helper_functions Helper functions object.
     */
    public function __construct(Abstract_Main_Plugin_Class $main_plugin, Plugin_Constants $constants, Helper_Functions $helper_functions)
    {

        $this->_constants        = $constants;
        $this->_helper_functions = $helper_functions;

        $main_plugin->add_to_all_plugin_models($this);
        $main_plugin->add_to_public_models($this);

    }

    /**
     * Ensure that only one instance of this class is loaded or can be loaded ( Singleton Pattern ).
     *
     * @since 1.0
     * @access public
     *
     * @param Abstract_Main_Plugin_Class $main_plugin      Main plugin object.
     * @param Plugin_Constants           $constants        Plugin constants object.
     * @param Helper_Functions           $helper_functions Helper functions object.
     * @return Admin
     */
    public static function get_instance(Abstract_Main_Plugin_Class $main_plugin, Plugin_Constants $constants, Helper_Functions $helper_functions)
    {

        if (!self::$_instance instanceof self) {
            self::$_instance = new self($main_plugin, $constants, $helper_functions);
        }

        return self::$_instance;

    }

    /*
    |--------------------------------------------------------------------------
    | Admin app
    |--------------------------------------------------------------------------
     */

    /**
     * Register Loyalty Program settings page.
     *
     * @since 1.0
     * @access public
     *
     * @param array $app_pages List of app pages.
     * @return array Filtered list of app pages.
     */
    public function register_loyalty_program_menu($app_pages)
    {

        $merged = array_merge(array(
            'acfw-loyalty-program' => array(
                'slug'  => 'acfw-loyalty-program',
                'label' => __('Loyalty Program', 'loyalty-program-for-woocommerce'),
                'page'  => 'loyalty_program',
            ),
        ), $app_pages);

        return $merged;
    }

    /**
     * Display admin app.
     *
     * @since 1.0
     * @access public
     */
    public function display_admin_app()
    {
        echo '<div id="lpfw_admin_app"></div>';
    }

    /*
    |--------------------------------------------------------------------------
    | Metabox
    |--------------------------------------------------------------------------
     */

    /**
     * Register Loyalty Program metabox.
     *
     * @since 1.0
     * @access public
     *
     * @param string  $post_type Post type.
     * @param WP_Post $post      Post object.
     */
    public function register_loyalty_program_metabox($post_type, $post)
    {

        if ($post_type !== 'shop_coupon') {
            return;
        }

        $user_id     = get_post_meta($post->ID, $this->_constants->COUPON_USER, true);
        $this->_user = $user_id ? get_userdata($user_id) : false;

        if (!$this->_user) {
            return;
        }

        add_meta_box(
            'acfw-loyalty-program-metabox',
            __('Loyalty Programs', 'loyalty-program-for-woocommerce'),
            array($this, 'loyalty_program_metabox_view'),
            'shop_coupon',
            'side',
            'high'
        );

    }

    /**
     * Loyalty programs metabox view.
     *
     * @since 1.0
     * @access public
     *
     * @param WP_Post $post Post object.
     */
    public function loyalty_program_metabox_view($post)
    {

        $usage  = (int) get_post_meta($post->ID, 'usage_count', true);
        $status = $usage > 0 ? __('Used', 'loyalty-program-for-woocommerce') : __('Not yet used', 'loyalty-program-for-woocommerce');
        $points = (int) get_post_meta($post->ID, $this->_constants->COUPON_POINTS, true);

        include $this->_constants->VIEWS_ROOT_PATH . 'admin/view-loyalty-programs-metabox.php';

    }

    /*
    |--------------------------------------------------------------------------
    | Fulfill implemented interface contracts
    |--------------------------------------------------------------------------
     */

    /**
     * Execute codes that needs to run plugin activation.
     *
     * @since 1.0
     * @access public
     * @implements LPFW\Interfaces\Initializable_Interface
     */
    public function initialize()
    {
    }

    /**
     * Execute Admin class.
     *
     * @since 1.0
     * @access public
     * @inherit LPFW\Interfaces\Model_Interface
     */
    public function run()
    {

        add_action('acfw_admin_app_pages', array($this, 'register_loyalty_program_menu'));
        add_action('add_meta_boxes', array($this, 'register_loyalty_program_metabox'), 10, 2);
        add_action('acfw_admin_app', array($this, 'display_admin_app'));
    }

}
