<?php
/**
 * Instagram Feed Builder
 *
 * @since 6.0
 */
namespace InstagramFeed\Builder;

use InstagramFeed\Helpers\Util;
use InstagramFeed\SBI_License_Service;
use InstagramFeed\Builder\SBI_Feed_Saver;
use InstagramFeed\Admin\SBI_Admin_Notices;
use InstagramFeed\SB_Instagram_Data_Encryption;

class SBI_Feed_Builder {

	public static $instance;

	/**
	 * License Service Handler
	 *
	 * The class checks the license status, grace period status and shows license notices as needed
	 *
	 * @since 6.2.0
	 * @access public
	 *
	 * @var SBI_License_Service
	 */
	public $license_service;


	public static function instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();

			self::$instance->license_service	= SBI_License_Service::instance();
		}
		return self::$instance;
	}


	/**
	 * Constructor.
	 *
	 * @since 6.0
	 */
	public function __construct() {
		$this->init();
	}

	/**
	 * Init the Builder.
	 *
	 * @since 6.0
	*/
	public function init() {
		if ( is_admin() ) {
			add_action( 'admin_menu', array( $this, 'register_menu' ) );
			// add ajax listeners
			SBI_Feed_Saver_Manager::hooks();
			SBI_Source::hooks();
			self::hooks();
		}
	}

	/**
	 * Mostly AJAX related hooks
	 *
	 * @since 6.0
	 */
	public static function hooks() {
		add_action( 'wp_ajax_sbi_dismiss_onboarding', array( 'InstagramFeed\Builder\SBI_Feed_Builder', 'after_dismiss_onboarding' ) );
		add_action( 'wp_ajax_sbi_other_plugins_modal', array( 'InstagramFeed\Builder\SBI_Feed_Builder', 'sb_other_plugins_modal' ) );
	}

	/**
	 * Check users capabilities and maybe nonce before AJAX actions
	 *
	 * @param $check_nonce
	 * @param string $action
	 *
	 * @since 6.0
	 */
	public static function check_privilege( $check_nonce = false, $action = 'sbi-admin' ) {
		$cap = current_user_can( 'manage_instagram_feed_options' ) ? 'manage_instagram_feed_options' : 'manage_options';
		$cap = apply_filters( 'sbi_settings_pages_capability', $cap );

		if ( ! current_user_can( $cap ) ) {
			wp_die( 'You did not do this the right way!' );
		}

		if ( $check_nonce ) {
			$nonce = ! empty( $_POST[ $check_nonce ] ) ? $_POST[ $check_nonce ] : false;

			if ( ! wp_verify_nonce( $nonce, $action ) ) {
				wp_die( 'You did not do this the right way!' );
			}
		}
	}

	/**
	 * Register Menu.
	 *
	 * @since 6.0
	 */
	public function register_menu() {
		$cap = current_user_can( 'manage_instagram_feed_options' ) ? 'manage_instagram_feed_options' : 'manage_options';
		$cap = apply_filters( 'sbi_settings_pages_capability', $cap );

		$feed_builder = add_submenu_page(
			'sb-instagram-feed',
			__( 'All Feeds', 'instagram-feed' ),
			__( 'All Feeds', 'instagram-feed' ),
			$cap,
			'sbi-feed-builder',
			array( $this, 'feed_builder' ),
			0
		);
		add_action( 'load-' . $feed_builder, array( $this, 'builder_enqueue_admin_scripts' ) );
	}

	/**
	 * Enqueue Builder CSS & Script.
	 *
	 * Loads only for builder pages
	 *
	 * @since 6.0
	 */
	public function builder_enqueue_admin_scripts() {
		if ( get_current_screen() ) :
			$screen = get_current_screen();
			if ( strpos($screen->id, 'sbi-feed-builder')  !== false ) :
				$installed_plugins = get_plugins();

				$newly_retrieved_source_connection_data = SBI_Source::maybe_source_connection_data();
				$license_key = null;
				if ( get_option( 'sbi_license_key' ) ) {
					$license_key = get_option( 'sbi_license_key' );
				}
				$active_extensions = array(
					//Fake
					'feedLayout' => false,
					'headerLayout' => false,
					'postStyling' => false,
					'lightbox'		=> false,
					'filtermoderation'		=> false,
					'shoppablefeed'		=> false,
				);

				$sbi_builder = array(
					'ajax_handler'         => admin_url( 'admin-ajax.php' ),
					'pluginType'           => 'pro',
					'builderUrl'           => admin_url( 'admin.php?page=sbi-feed-builder' ),
					'nonce'                => wp_create_nonce( 'sbi-admin' ),
					'adminPostURL'         => admin_url( 'post.php' ),
					'adminPageUrl'         => admin_url( 'admin.php' ),
					'widgetsPageURL'       => admin_url( 'widgets.php' ),
					'supportPageUrl'       => admin_url( 'admin.php?page=sbi-support' ),
					'pluginURL'  			=> SBI_PLUGIN_URL,
					'activeExtensions'		=> $active_extensions,
					'licenseKey'           => $license_key,
					'genericText'          => self::get_generic_text(),
					'shouldDisableProFeatures' => sbi_builder_pro()->license_service->should_disable_pro_features,
					'welcomeScreen'        => array(
						'mainHeading'              => __( 'All Feeds', 'instagram-feed' ),
						'createFeed'               => __( 'Create your Feed', 'instagram-feed' ),
						'createFeedDescription'    => __( 'Connect your Instagram account and choose a feed type', 'instagram-feed' ),
						'customizeFeed'            => __( 'Customize your feed type', 'instagram-feed' ),
						'customizeFeedDescription' => __( 'Choose layouts, color schemes, filters and more', 'instagram-feed' ),
						'embedFeed'                => __( 'Embed your feed', 'instagram-feed' ),
						'embedFeedDescription'     => __( 'Easily add the feed anywhere on your website', 'instagram-feed' ),
						'customizeImgPath'         => SBI_BUILDER_URL . 'assets/img/welcome-1.png',
						'embedImgPath'             => SBI_BUILDER_URL . 'assets/img/welcome-2.png',
					),
					'pluginsInfo'          => array(
						'social_wall' => array(
							'installed'    => isset( $installed_plugins['social-wall/social-wall.php'] ) ? true : false,
							'activated'    => is_plugin_active( 'social-wall/social-wall.php' ),
							'settingsPage' => admin_url( 'admin.php?page=sbsw' ),
						),
					),
					'allFeedsScreen'       => array(
						'mainHeading'     => __( 'All Feeds', 'instagram-feed' ),
						'columns'         => array(
							'nameText'      => __( 'Name', 'instagram-feed' ),
							'shortcodeText' => __( 'Shortcode', 'instagram-feed' ),
							'instancesText' => __( 'Instances', 'instagram-feed' ),
							'actionsText'   => __( 'Actions', 'instagram-feed' ),
						),
						'bulkActions'     => __( 'Bulk Actions', 'instagram-feed' ),
						'legacyFeeds'     => array(
							'heading'               => __( 'Legacy Feeds', 'instagram-feed' ),
							'toolTip'               => __( 'What are Legacy Feeds?', 'instagram-feed' ),
							'toolTipExpanded'       => array(
								__( 'Legacy feeds are older feeds from before the version 6 update. You can edit settings for these feeds by using the "Settings" button to the right. These settings will apply to all legacy feeds, just like the settings before version 6, and work in the same way that they used to.', 'instagram-feed' ),
								__( 'You can also create a new feed, which will now have it\'s own individual settings. Modifying settings for new feeds will not affect other feeds.', 'instagram-feed' ),
							),
							'toolTipExpandedAction' => array(
								__( 'Legacy feeds represent shortcodes of old feeds found on your website before <br/>the version 6 update.', 'instagram-feed' ),
								__( 'To edit Legacy feed settings, you will need to use the "Settings" button above <br/>or edit their shortcode settings directly. To delete them, simply remove the <br/>shortcode wherever it is being used on your site.', 'instagram-feed' ),
							),
							'show'                  => __( 'Show Legacy Feeds', 'instagram-feed' ),
							'hide'                  => __( 'Hide Legacy Feeds', 'instagram-feed' ),
						),
						'socialWallLinks' => self::get_social_wall_links(),
						'onboarding'      => $this->get_onboarding_text(),
					),
					'dialogBoxPopupScreen' => array(
						'deleteSourceCustomizer' => array(
							'heading'     => __( 'Delete "#"?', 'instagram-feed' ),
							'description' => __( 'You are going to delete this source. To retrieve it, you will need to add it again. Are you sure you want to continue?', 'instagram-feed' ),
						),
						'deleteSingleFeed'       => array(
							'heading'     => __( 'Delete "#"?', 'instagram-feed' ),
							'description' => __( 'You are going to delete this feed. You will lose all the settings. Are you sure you want to continue?', 'instagram-feed' ),
						),
						'deleteMultipleFeeds'    => array(
							'heading'     => __( 'Delete Feeds?', 'instagram-feed' ),
							'description' => __( 'You are going to delete these feeds. You will lose all the settings. Are you sure you want to continue?', 'instagram-feed' ),
						),
						'backAllToFeed'          => array(
							'heading'     => __( 'Are you Sure?', 'instagram-feed' ),
							'description' => __( 'Are you sure you want to leave this page, all unsaved settings will be lost, please make sure to save before leaving.', 'instagram-feed' ),
							'customButtons' => array(
								'confirm' => array(
									'text'  => __( 'Save and Exit', 'instagram-feed' ),
									'color' => 'blue',
								),
								'cancel'  => array(
									'text'  => __( 'Exit without Saving', 'instagram-feed' ),
									'color' => 'red',
								),
							),
						),
						'unsavedFeedSources'     => array(
							'heading'       => __( 'You have unsaved changes', 'instagram-feed' ),
							'description'   => __( 'If you exit without saving, all the changes you made will be reverted.', 'instagram-feed' ),
							'customButtons' => array(
								'confirm' => array(
									'text'  => __( 'Save and Exit', 'instagram-feed' ),
									'color' => 'blue',
								),
								'cancel'  => array(
									'text'  => __( 'Exit without Saving', 'instagram-feed' ),
									'color' => 'red',
								),
							),
						),
					),
					'selectFeedTypeScreen' => array(
						'mainHeading'            => __( 'Create an Instagram Feed', 'instagram-feed' ),
						'feedTypeHeading'        => __( 'Select Feed Type', 'instagram-feed' ),
						'mainDescription'        => __( 'Select one or more feed types. You can add or remove them later.', 'instagram-feed' ),
						'updateHeading'          => __( 'Update Feed Type', 'instagram-feed' ),
						'anotherFeedTypeHeading' => __( 'Add Another Source Type', 'instagram-feed' ),
					),
					'mainFooterScreen'     => array(
						'heading'     => sprintf( __( 'Upgrade to the %1$sAll Access Bundle%2$s to get all of our Pro Plugins', 'instagram-feed' ), '<strong>', '</strong>' ),
						'description' => __( 'Includes all Smash Balloon plugins for one low price: Instagram, Facebook, Twitter, YouTube, and Social Wall', 'instagram-feed' ),
						'promo'       => sprintf( __( '%1$sBonus%2$s Lite users get %3$s50&#37; Off%4$s automatically applied at checkout', 'instagram-feed' ), '<span class="sbi-bld-ft-bns">', '</span>', '<strong>', '</strong>' ),
					),
					'embedPopupScreen'     => array(
						'heading'       => __( 'Embed Feed', 'instagram-feed' ),
						'description'   => __( 'Add the unique shortcode to any page, post, or widget:', 'instagram-feed' ),
						'description_2' => __( 'Or use the built in WordPress block or widget', 'instagram-feed' ),
						'addPage'       => __( 'Add to a Page', 'instagram-feed' ),
						'addWidget'     => __( 'Add to a Widget', 'instagram-feed' ),
						'selectPage'    => __( 'Select Page', 'instagram-feed' ),
					),
					'links'                => self::get_links_with_utm(),
					'pluginsInfo'          => array(
						'social_wall' => array(
							'installed'    => isset( $installed_plugins['social-wall/social-wall.php'] ) ? true : false,
							'activated'    => is_plugin_active( 'social-wall/social-wall.php' ),
							'settingsPage' => admin_url( 'admin.php?page=sbsw' ),
						),
					),
					'selectFeedTemplateScreen' => array(
						'feedTemplateHeading'     => __( 'Start with a template', 'custom-facebook-feed' ),
						'feedTemplateDescription' => __( 'Select a starting point for your feed. You can customize this later.', 'custom-facebook-feed' ),
						'updateHeading'           => __( 'Select another template', 'custom-facebook-feed' ),
						'updateHeadingWarning'    => __( 'Changing a template will override your layout, header and button settings', 'custom-facebook-feed' )
					),
					'selectSourceScreen'   => self::select_source_screen_text(),
					'feedTypes'            => $this->get_feed_types(),
					'feedTemplates'        => $this->get_feed_templates(),
					'socialInfo'           => $this->get_smashballoon_info(),
					'svgIcons'             => $this->builder_svg_icons(),
					'installPluginsPopup'  => $this->install_plugins_popup(),
					'feeds'                => self::get_feed_list(),
					'itemsPerPage'			=> SBI_Db::RESULTS_PER_PAGE,
			        'feedsCount' 			=> SBI_Db::feeds_count(),
					'sources'              => self::get_source_list(),
					'sourceConnectionURLs' => SBI_Source::get_connection_urls(),

					'legacyFeeds'          => $this->get_legacy_feed_list(),
					'personalAccountScreen'   => self::personal_account_screen_text(),
					'sbiLicenseNoticeActive' => empty( sbi_builder_pro()->license_service->get_license_key ) || sbi_builder_pro()->license_service->expiredLicenseWithGracePeriodEnded ? true : false,
					'sbiLicenseInactiveState' => sbi_license_inactive_state() ? true : false,
			    'extensionsPopup' => $this->get_extensions_popup( $license_key )
				);

				if ( $newly_retrieved_source_connection_data ) {
					$sbi_builder['newSourceData'] = $newly_retrieved_source_connection_data;
				}
				 if ( isset( $_GET['manualsource'] ) && $_GET['manualsource'] == true ) {
			        $sbi_builder['manualSourcePopupInit'] = true;
		        }

				$maybe_feed_customizer_data = SBI_Feed_Saver_Manager::maybe_feed_customizer_data();

				if ( $maybe_feed_customizer_data ) {
					sb_instagram_scripts_enqueue( true );
					$sbi_builder['customizerFeedData']       = $maybe_feed_customizer_data;
					$sbi_builder['customizerSidebarBuilder'] = \InstagramFeed\Builder\Tabs\SBI_Builder_Customizer_Tab::get_customizer_tabs();
					$sbi_builder['wordpressPageLists']       = $this->get_wp_pages();

					if ( ! isset( $_GET['feed_id'] ) || $_GET['feed_id'] === 'legacy' ) {
						$feed_id                       = 'legacy';
						$customizer_atts               = $maybe_feed_customizer_data['settings'];
						$customizer_atts['customizer'] = true;
					} elseif ( intval( $_GET['feed_id'] ) > 0 ) {
						$feed_id         = intval( $_GET['feed_id'] );
						$customizer_atts = array(
							'feed'       => $feed_id,
							'customizer' => true,
						);
					}

					if ( ! empty( $feed_id ) ) {
						$settings_preview = self::add_customizer_att( $customizer_atts );

						$sbi_builder['feedInitOutput'] = htmlspecialchars( display_instagram( $settings_preview, true ) );
					}

					//Date
					global $wp_locale;
					wp_enqueue_script(
						'sbi-date_i18n',
						SBI_PLUGIN_URL . 'admin/builder/assets/js/date_i18n.js',
						null,
						SBIVER,
						true
					);

					$monthNames      = array_map(
						array( &$wp_locale, 'get_month' ),
						range( 1, 12 )
					);
					$monthNamesShort = array_map(
						array( &$wp_locale, 'get_month_abbrev' ),
						$monthNames
					);
					$dayNames        = array_map(
						array( &$wp_locale, 'get_weekday' ),
						range( 0, 6 )
					);
					$dayNamesShort   = array_map(
						array( &$wp_locale, 'get_weekday_abbrev' ),
						$dayNames
					);
					wp_localize_script(
						'sbi-date_i18n',
						'DATE_I18N',
						array(
							'month_names'       => $monthNames,
							'month_names_short' => $monthNamesShort,
							'day_names'         => $dayNames,
							'day_names_short'   => $dayNamesShort,
						)
					);
				}

				wp_enqueue_style(
					'sbi-builder-style',
					SBI_PLUGIN_URL . 'admin/builder/assets/css/builder.css',
					false,
					SBIVER
				);

				self::global_enqueue_ressources_scripts();

				wp_enqueue_script(
					'sbi-builder-app',
					SBI_PLUGIN_URL . 'admin/builder/assets/js/builder.js',
					null,
					SBIVER,
					true
				);
				// Customize screens
				$sbi_builder['customizeScreens'] = $this->get_customize_screens_text();
				wp_localize_script(
					'sbi-builder-app',
					'sbi_builder',
					$sbi_builder
				);
				wp_enqueue_media();
			endif;
		endif;
	}

	/**
	 * Get WP Pages List
	 *
	 * @return array
	 *
	 * @since 6.0
	 */
	public function get_wp_pages() {
		$pagesList   = get_pages();
		$pagesResult = array();
		if ( is_array( $pagesList ) ) {
			foreach ( $pagesList as $page ) {
				array_push(
					$pagesResult,
					array(
						'id'    => $page->ID,
						'title' => $page->post_title,
					)
				);
			}
		}
		return $pagesResult;
	}

	/**
	 * For types listed on the top of the select feed type screen
	 *
	 * @return array
	 */
	public function get_feed_templates() {
		$feed_types = array(
			array(
				'type' => 'ft_default',
				'title'=> __( 'Default', 'instagram-feed' ),
				'icon'	=>  'defaultFTIcon'
			),
			array(
				'type' => 'ft_simple_grid',
				'title' => __( 'Simple Grid', 'instagram-feed' ),
				'icon'	=>  'simpleGridIcon'
			),
			array(
				'type' => 'ft_simple_grid_xl',
				'title' => __( 'Simple Grid XL', 'instagram-feed' ),
				'icon'	=>  'simpleGridXLIcon'
			),
			array(
				'type' => 'ft_simple_row',
				'title' => __( 'Simple Row', 'instagram-feed' ),
				'icon'	=>  'simpleRowIcon'
			),
			array(
				'type' => 'ft_simple_carousel',
				'title' => __( 'Simple Carousel', 'instagram-feed' ),
				'icon'	=>  'simpleCarouselIcon'
			),
			array(
				'type' => 'ft_masonry_cards',
				'title' => __( 'Masonry Cards', 'instagram-feed' ),
				'icon'	=>  'masonryCardsIcon'
			),
			array(
				'type' => 'ft_card_grid',
				'title' => __( 'Card Grid', 'instagram-feed' ),
				'icon'	=>  'cardGridIcon'
			),
			array(
				'type' => 'ft_highlight',
				'title' => __( 'Highlight', 'instagram-feed' ),
				'icon'	=>  'highlightIcon'
			),
			array(
				'type' => 'ft_single_post',
				'title' => __( 'Single Post', 'instagram-feed' ),
				'icon'	=>  'singlePostIcon'
			),
			array(
				'type' => 'ft_single_post_carousel',
				'title' => __( 'Single Post Carousel', 'instagram-feed' ),
				'icon'	=>  'singlePostCarouselIcon'
			),
		);

		return $feed_types;
	}


	/**
	 * Global JS + CSS Files
	 *
	 * Shared JS + CSS ressources for the admin panel
	 *
	 * @since 6.0
	 */
	public static function global_enqueue_ressources_scripts( $is_settings = false ) {
		wp_enqueue_style(
			'feed-global-style',
			SBI_PLUGIN_URL . 'admin/builder/assets/css/global.css',
			false,
			SBIVER
		);

		wp_enqueue_script(
			'sb-vue',
            SBI_PLUGIN_URL . 'js/vue.min.js',
			null,
			'2.6.12',
			true
		);

		wp_enqueue_script(
			'feed-colorpicker-vue',
			SBI_PLUGIN_URL . 'admin/builder/assets/js/vue-color.min.js',
			null,
			SBIVER,
			true
		);

		wp_enqueue_script(
			'feed-builder-ressources',
			SBI_PLUGIN_URL . 'admin/builder/assets/js/ressources.js',
			null,
			SBIVER,
			true
		);

		wp_enqueue_script(
			'sb-dialog-box',
			SBI_PLUGIN_URL . 'admin/builder/assets/js/confirm-dialog.js',
			null,
			SBIVER,
			true
		);

		wp_enqueue_script(
			'install-plugin-popup',
			SBI_PLUGIN_URL . 'admin/builder/assets/js/install-plugin-popup.js',
			null,
			SBIVER,
			true
		);

		wp_enqueue_script(
			'sb-add-source',
			SBI_PLUGIN_URL . 'admin/builder/assets/js/add-source.js',
			null,
			SBIVER,
			true
		);

		wp_enqueue_script(
			'sb-personal-account',
			SBI_PLUGIN_URL . 'admin/builder/assets/js/personal-account.js',
			null,
			SBIVER,
			true
		);

		$sbi_personal_account   = array(
			'personalAccountScreen' => self::personal_account_screen_text(),
			'nonce'                => wp_create_nonce( 'sbi-admin' ),
			'ajaxHandler'         => admin_url( 'admin-ajax.php' ),
		);

		wp_localize_script(
			'sb-personal-account',
			'sbi_personal_account',
			$sbi_personal_account
		);



		$newly_retrieved_source_connection_data = SBI_Source::maybe_source_connection_data();
		$sbi_source                             = array(
			'sources'              => self::get_source_list(),
			'sourceConnectionURLs' => SBI_Source::get_connection_urls( $is_settings ),
			'nonce'                => wp_create_nonce( 'sbi-admin' ),
		);
		if ( $newly_retrieved_source_connection_data ) {
			$sbi_source['newSourceData'] = $newly_retrieved_source_connection_data;
		}
		if ( isset( $_GET['manualsource'] ) && $_GET['manualsource'] == true ) {
			$sbi_source['manualSourcePopupInit'] = true;
		}

		wp_localize_script(
			'sb-add-source',
			'sbi_source',
			$sbi_source
		);
	}

	/**
	 * Get Generic text
	 *
	 * @return array
	 *
	 * @since 6.0
	 */
	public static function get_generic_text() {
		$icons = self::builder_svg_icons();
		return array(
			'done'                              => __( 'Done', 'instagram-feed' ),
			'title'                             => __( 'Settings', 'instagram-feed' ),
			'dashboard'                         => __( 'Dashboard', 'instagram-feed' ),
			'addNew'                            => __( 'Add New', 'instagram-feed' ),
			'addSource'                         => __( 'Add Source', 'instagram-feed' ),
			'addAnotherSource'                  => __( 'Add another Source', 'instagram-feed' ),
			'addSourceType'                     => __( 'Add Another Source Type', 'instagram-feed' ),
			'previous'                          => __( 'Previous', 'instagram-feed' ),
			'showSelected'                      => __( 'Show Selected', 'instagram-feed' ),
			'showAll'                           => __( 'Show All', 'instagram-feed' ),
			'next'                              => __( 'Next', 'instagram-feed' ),
			'finish'                            => __( 'Finish', 'instagram-feed' ),
			'new'                               => __( 'New', 'instagram-feed' ),
			'update'                            => __( 'Update', 'instagram-feed' ),
			'upgrade'                           => __( 'Upgrade', 'instagram-feed' ),
			'settings'                          => __( 'Settings', 'instagram-feed' ),
			'back'                              => __( 'Back', 'instagram-feed' ),
			'backAllFeeds'                      => __( 'Back to all feeds', 'instagram-feed' ),
			'createFeed'                        => __( 'Create Feed', 'instagram-feed' ),
			'add'                               => __( 'Add', 'instagram-feed' ),
			'change'                            => __( 'Change', 'instagram-feed' ),
			'getExtention'                      => __( 'Get Extension', 'instagram-feed' ),
			'viewDemo'                          => __( 'View Demo', 'instagram-feed' ),
			'includes'                          => __( 'Includes', 'instagram-feed' ),
			'photos'                            => __( 'Photos', 'instagram-feed' ),
			'photo'                             => __( 'Photo', 'instagram-feed' ),
			'apply'                             => __( 'Apply', 'instagram-feed' ),
			'copy'                              => __( 'Copy', 'instagram-feed' ),
			'edit'                              => __( 'Edit', 'instagram-feed' ),
			'duplicate'                         => __( 'Duplicate', 'instagram-feed' ),
			'delete'                            => __( 'Delete', 'instagram-feed' ),
			'remove'                            => __( 'Remove', 'instagram-feed' ),
			'removeSource'                      => __( 'Remove Source', 'instagram-feed' ),
			'shortcode'                         => __( 'Shortcode', 'instagram-feed' ),
			'clickViewInstances'                => __( 'Click to view Instances', 'instagram-feed' ),
			'usedIn'                            => __( 'Used in', 'instagram-feed' ),
			'place'                             => __( 'place', 'instagram-feed' ),
			'places'                            => __( 'places', 'instagram-feed' ),
			'item'                              => __( 'Item', 'instagram-feed' ),
			'items'                             => __( 'Items', 'instagram-feed' ),
			'learnMore'                         => __( 'Learn More', 'instagram-feed' ),
			'location'                          => __( 'Location', 'instagram-feed' ),
			'page'                              => __( 'Page', 'instagram-feed' ),
			'copiedClipboard'                   => __( 'Copied to Clipboard', 'instagram-feed' ),
			'feedImported'                      => __( 'Feed imported successfully', 'instagram-feed' ),
			'failedToImportFeed'                => __( 'Failed to import feed', 'instagram-feed' ),
			'timeline'                          => __( 'Timeline', 'instagram-feed' ),
			'help'                              => __( 'Help', 'instagram-feed' ),
			'admin'                             => __( 'Admin', 'instagram-feed' ),
			'member'                            => __( 'Member', 'instagram-feed' ),
			'reset'                             => __( 'Reset', 'instagram-feed' ),
			'preview'                           => __( 'Preview', 'instagram-feed' ),
			'name'                              => __( 'Name', 'instagram-feed' ),
			'id'                                => __( 'ID', 'instagram-feed' ),
			'token'                             => __( 'Token', 'instagram-feed' ),
			'confirm'                           => __( 'Confirm', 'instagram-feed' ),
			'cancel'                            => __( 'Cancel', 'instagram-feed' ),
			'clear'                             => __( 'Clear', 'instagram-feed' ),
			'clearFeedCache'                    => __( 'Clear Feed Cache', 'instagram-feed' ),
			'saveSettings'                      => __( 'Save Changes', 'instagram-feed' ),
			'feedName'                          => __( 'Feed Name', 'instagram-feed' ),
			'shortcodeText'                     => __( 'Shortcode', 'instagram-feed' ),
			'general'                           => __( 'General', 'instagram-feed' ),
			'feeds'                             => __( 'Feeds', 'instagram-feed' ),
			'translation'                       => __( 'Translation', 'instagram-feed' ),
			'advanced'                          => __( 'Advanced', 'instagram-feed' ),
			'error'                             => __( 'Error:', 'instagram-feed' ),
			'errorNotice'                       => __( 'There was an error when trying to connect to Instagram.', 'instagram-feed' ),
			'errorDirections'                   => '<a href="https://smashballoon.com/instagram-feed/docs/errors/" target="_blank" rel="noopener">' . __( 'Directions on How to Resolve This Issue', 'instagram-feed' ) . '</a>',
			'errorSource'                       => __( 'Source Invalid', 'instagram-feed' ),
			'errorEncryption'                   => __( 'Encryption Error', 'instagram-feed' ),
			'invalid'                           => __( 'Invalid', 'instagram-feed' ),
			'reconnect'                         => __( 'Reconnect', 'instagram-feed' ),
			'feed'                              => __( 'feed', 'instagram-feed' ),
			'sourceNotUsedYet'                  => __( 'Source is not used yet', 'instagram-feed' ),
			'addImage'                          => __( 'Add Image', 'instagram-feed' ),
			'businessRequired'                  => __( 'Business Account required', 'instagram-feed' ),
			'selectedPost'                      => __( 'Selected Post', 'instagram-feed' ),
			'productLink'                       => __( 'Product Link', 'instagram-feed' ),
			'enterProductLink'                  => __( 'Add your product URL here', 'instagram-feed' ),
			'editSources'                       => __( 'Edit Sources', 'instagram-feed' ),
			'moderateFeed'                      => __( 'Moderate your feed', 'instagram-feed' ),
			'moderateFeedSaveExit'              => __( 'Save and Exit', 'instagram-feed' ),
			'moderationMode'                    => __( 'Moderation Mode', 'instagram-feed' ),
			'moderationModeEnterPostId'         => __( 'Or Enter Post IDs to hide manually', 'instagram-feed' ),
			'moderationModePostIdPlaceholder'   => __( 'Add Post IDs here, separated by comma, to hide posts with a specific ID', 'instagram-feed' ),
			'moderationModeTextareaPlaceholder' => __( 'Add words here to hide any posts containing these words', 'instagram-feed' ),
			'filtersAndModeration'              => __( 'Filters & Moderation', 'instagram-feed' ),
			'topRated'                          => __( 'Top Rated', 'instagram-feed' ),
			'mostRecent'                        => __( 'Most recent', 'instagram-feed' ),
			'moderationModePreview'             => __( 'Moderation Mode Preview', 'instagram-feed' ),
			'shoppableModePreview'             => __( 'Shoppable Feed Preview', 'instagram-feed' ),
			'licenseExpired'					=> __( 'License Expired', 'instagram-feed' ),
			'licenseInactive'					=> __( 'Inactive', 'instagram-feed' ),
			'renew'								=> __( 'Renew', 'instagram-feed' ),
			'activateLicense'					=> __( 'Activate License', 'instagram-feed' ),
			'recheckLicense'					=> __( 'Recheck License', 'instagram-feed' ),
			'licenseValid'						=> __( 'License Valid', 'instagram-feed' ),
			'licenseExpired'					=> __( 'License Expired', 'instagram-feed' ),			'notification'                      => array(
				'feedSaved'             => array(
					'type' => 'success',
					'text' => __( 'Feed saved successfully', 'instagram-feed' ),
				),
				'feedSavedError'        => array(
					'type' => 'error',
					'text' => __( 'Error saving Feed', 'instagram-feed' ),
				),
				'previewUpdated'        => array(
					'type' => 'success',
					'text' => __( 'Preview updated successfully', 'instagram-feed' ),
				),
				'carouselLayoutUpdated' => array(
					'type' => 'success',
					'text' => __( 'Carousel updated successfully', 'instagram-feed' ),
				),
				'unkownError'           => array(
					'type' => 'error',
					'text' => __( 'Unknown error occurred', 'instagram-feed' ),
				),
				'cacheCleared'          => array(
					'type' => 'success',
					'text' => __( 'Feed cache cleared', 'instagram-feed' ),
				),
				'selectSourceError'     => array(
					'type' => 'error',
					'text' => __( 'Please select a source for your feed', 'instagram-feed' ),
				),
				'commentCacheCleared'   => array(
					'type' => 'success',
					'text' => __( 'Comment cache cleared', 'instagram-feed' ),
				),
				'personalAccountUpdated'   => array(
					'type' => 'success',
					'text' => __( 'Personal account updated', 'instagram-feed' ),
				),
				'licenseActivated'   => array(
					'type' => 'success',
					'text' => __( 'License Successfully Activated', 'instagram-feed' ),
				),
				'licenseError'   => array(
					'type' => 'error',
					'text' => __( 'Couldn\'t Activate License', 'instagram-feed' ),
				),
			),
			'install'                           => __( 'Install', 'instagram-feed' ),
			'installed'                         => __( 'Installed', 'instagram-feed' ),
			'activate'                          => __( 'Activate', 'instagram-feed' ),
			'installedAndActivated'             => __( 'Installed & Activated', 'instagram-feed' ),
			'free'                              => __( 'Free', 'instagram-feed' ),
			'invalidLicenseKey'                 => __( 'Invalid license key', 'instagram-feed' ),
			'licenseActivated'                  => __( 'License activated', 'instagram-feed' ),
			'licenseDeactivated'                => __( 'License Deactivated', 'instagram-feed' ),
			'carouselLayoutUpdated'             => array(
				'type' => 'success',
				'text' => __( 'Carousel Layout updated', 'instagram-feed' ),
			),
			'getMoreFeatures'                   => __( 'Get more features with Custom Facebook Feed Pro', 'instagram-feed' ),
			'liteFeedUsers'                     => __( 'Lite Feed Users get 50% OFF', 'instagram-feed' ),
			'tryDemo'                           => __( 'Try Demo', 'instagram-feed' ),
			'displayImagesVideos'               => __( 'Display images and videos in posts', 'instagram-feed' ),
			'viewLikesShares'                   => __( 'View likes, shares and comments', 'instagram-feed' ),
			'allFeedTypes'                      => __( 'All Feed Types: Photos, Albums, Events and more', 'instagram-feed' ),
			'abilityToLoad'                     => __( 'Ability to “Load More” posts', 'instagram-feed' ),
			'andMuchMore'                       => __( 'And Much More!', 'instagram-feed' ),
			'sbiFreeCTAFeatures'                => array(
				__( 'Filter posts', 'instagram-feed' ),
				__( 'Popup photo/video lighbox', 'instagram-feed' ),
				__( '30 day money back guarantee', 'instagram-feed' ),
				__( 'Multiple post layout options', 'instagram-feed' ),
				__( 'Video player (HD, 360, Live)', 'instagram-feed' ),
				__( 'Fast, friendly and effective support', 'instagram-feed' ),
			),
			'ctaShowFeatures'                   => __( 'Show Features', 'instagram-feed' ),
			'ctaHideFeatures'                   => __( 'Hide Features', 'instagram-feed' ),
			'redirectLoading'                   => array(
				'heading'     => __( 'Redirecting to connect.smashballoon.com', 'instagram-feed' ),
				'description' => __( 'You will be redirected to our app so you can connect your account in 5 seconds', 'instagram-feed' ),
			),
			'addAccountInfo' => __( 'Add Avatar and Bio', 'instagram-feed' ),
			'updateAccountInfo' => __( 'Update Avatar and Bio', 'instagram-feed' ),
			'personalAccountUpdated' => __( 'Personal account updated', 'instagram-feed' ),
			'igFeedCreated' => __( 'Instagram feed successfully created!', 'instagram-feed' ),
			'onceDoneSWFeed' => __( 'Once you are done creating the Instagram feed, you can go back to Social plugin', 'instagram-feed' ),
			'goToSocialWall' => __( 'Go to Social Wall', 'instagram-feed' ),
		);
	}

	/**
	 * Select Source Screen Text
	 *
	 * @return array
	 *
	 * @since 4.0
	 */
	public static function select_source_screen_text() {
		return array(
			'mainHeading'               => __( 'Select one or more sources', 'instagram-feed' ),
			'description'               => __( 'Sources are Instagram accounts your feed will display content from', 'instagram-feed' ),
			'emptySourceDescription'    => __( 'Looks like you have not added any source.<br/>Use “Add Source” to add a new one.', 'instagram-feed' ),
			'mainHashtagHeading'        => __( 'Enter Public Hashtags', 'instagram-feed' ),
			'hashtagDescription'        => __( 'Add one or more hashtags separated by comma', 'instagram-feed' ),
			'hashtagGetBy'              => __( 'Fetch posts that are', 'instagram-feed' ),

			'sourcesListPopup'          => array(
				'user'   => array(
					'mainHeading' => __( 'Add a source for Timeline', 'instagram-feed' ),
					'description' => __( 'Select or add an account you want to display the timeline for', 'instagram-feed' ),
				),
				'tagged' => array(
					'mainHeading' => __( 'Add a source for Mentions', 'instagram-feed' ),
					'description' => __( 'Select or add an account you want to display the mentions for', 'instagram-feed' ),
				),
			),

			'perosnalAccountToolTipTxt' => array(
				__(
					'Due to changes in Instagram’s new API, we can no<br/>
					longer get mentions for personal accounts. To<br/>
					enable this for your account, you will need to convert it to<br/>
					a Business account. Learn More',
					'instagram-feed'
				),
			),
			'groupsToolTip'             => array(
				__( 'Due to Facebook limitations, it\'s not possible to display photo feeds from a Group, only a Page.', 'instagram-feed' ),
			),
			'updateHeading'             => __( 'Update Source', 'instagram-feed' ),
			'updateDescription'         => __( 'Select a source from your connected Facebook Pages and Groups. Or, use "Add New" to connect a new one.', 'instagram-feed' ),
			'updateFooter'              => __( 'Add multiple Facebook Pages or Groups to a feed with our Multifeed extension', 'instagram-feed' ),
			'noSources'                 => __( 'Please add a source in order to display a feed. Go to the "Settings" tab -> "Sources" section -> Click "Add New" to connect a source.', 'instagram-feed' ),

			'multipleTypes'             => array(
				'user'    => array(
					'heading'     => __( 'User Timeline', 'instagram-feed' ),
					'icon'        => 'user',
					'description' => __( 'Connect an account to show posts for it.', 'instagram-feed' ),
					'actionType'  => 'addSource',
				),
				'hashtag' => array(
					'heading'          => __( 'Hashtag', 'instagram-feed' ),
					'icon'             => 'hashtag',
					'tooltip'      => __( 'Add one or more hashtags separated by comma.', 'instagram-feed' ),
					'businessRequired' => true,
					'actionType'       => 'inputHashtags',
				),
				'tagged'  => array(
					'heading'          => __( 'Tagged', 'instagram-feed' ),
					'icon'             => 'mention',
					'description'      => __( 'Connect an account to show tagged posts. This does not give us any permission to manage your Instagram account.', 'instagram-feed' ),
					'businessRequired' => true,
					'actionType'       => 'addSource',
				),
			),

			'modal'                     => array(
				'addNew'                     => __( 'Connect your Instagram Account', 'instagram-feed' ),
				'selectSourceType'           => __( 'Select Account Type', 'instagram-feed' ),
				'connectAccount'             => __( 'Connect an Instagram Account', 'instagram-feed' ),
				'connectAccountDescription'  => __( 'This does not give us permission to manage your Instagram account, it simply allows the plugin to see a list of them and retrieve their public content from the API.', 'instagram-feed' ),
				'connect'                    => __( 'Connect', 'instagram-feed' ),
				'enterEventToken'            => __( 'Enter Events Access Token', 'instagram-feed' ),
				'enterEventTokenDescription' => sprintf( __( 'Due to restrictions by Facebook, you need to create a Facebook app and then paste that app Access Token here. We have a guide to help you with just that, which you can read %1$shere%2$s', 'instagram-feed' ), '<a href="https://smashballoon.com/instagram-feed/page-token/" target="_blank" rel="noopener">', '</a>' ),
				'alreadyHave'                => __( 'Already have a API Token and Access Key for your account?', 'instagram-feed' ),
				'addManuallyLink'            => __( 'Add Account Manually', 'instagram-feed' ),
				'selectAccount'              => __( 'Select an Instagram Account', 'instagram-feed' ),
				'showing'                    => __( 'Showing', 'instagram-feed' ),
				'facebook'                   => __( 'Facebook', 'instagram-feed' ),
				'businesses'                 => __( 'Businesses', 'instagram-feed' ),
				'groups'                     => __( 'Groups', 'instagram-feed' ),
				'connectedTo'                => __( 'connected to', 'instagram-feed' ),
				'addManually'                => __( 'Add a Source Manually', 'instagram-feed' ),
				'addSource'                  => __( 'Add Source', 'instagram-feed' ),
				'sourceType'                 => __( 'Source Type', 'instagram-feed' ),
				'accountID'                  => __( 'Instagram Account ID', 'instagram-feed' ),
				'fAccountID'                 => __( 'Instagram Account ID', 'instagram-feed' ),
				'eventAccessToken'           => __( 'Event Access Token', 'instagram-feed' ),
				'enterID'                    => __( 'Enter ID', 'instagram-feed' ),
				'accessToken'                => __( 'Instagram Access Token', 'instagram-feed' ),
				'enterToken'                 => __( 'Enter Token', 'instagram-feed' ),
				'addApp'                     => __( 'Add Instagram App to your group', 'instagram-feed' ),
				'addAppDetails'              => __( 'To get posts from your group, Instagram requires the "Smash Balloon Plugin" app to be added in your group settings. Just follow the directions here:', 'instagram-feed' ),
				'addAppSteps'                => array(
					__( 'Go to your group settings page by ', 'instagram-feed' ),
					sprintf( __( 'Search for "Smash Balloon" and select our app %1$s(see screenshot)%2$s', 'instagram-feed' ), '<a href="JavaScript:void(0);" id="sbi-group-app-tooltip">', '<img class="sbi-group-app-screenshot sb-tr-1" src="' . trailingslashit( SBI_PLUGIN_URL ) . 'admin/assets/img/group-app.png" alt="Thumbnail Layout"></a>' ),
					__( 'Click "Add" and you are done.', 'instagram-feed' ),
				),
				'alreadyExists'              => __( 'Account already exists', 'instagram-feed' ),
				'alreadyExistsExplanation'   => __( 'The Instagram account you added is already connected as a “Business” account. Would you like to replace it with a “Personal“ account? (Note: Personal accounts cannot be used to display Mentions or Hashtag feeds.)', 'instagram-feed' ),
				'replaceWithPersonal'        => __( 'Replace with Personal', 'instagram-feed' ),
				'notAdmin'                   => __( 'For groups you are not an administrator of', 'instagram-feed' ),
				'disclaimerMentions'         => __( 'Due to Instagram’s limitations, you need to connect a business account to display a Mentions timeline', 'instagram-feed' ),
				'disclaimerHashtag'          => __( 'Due to Instagram’s limitations, you need to connect a business account to display a Hashtag feed', 'instagram-feed' ),
				'notSureToolTip'             => __( 'Select "Personal" if displaying a regular feed of posts, as this can display feeds from either a Personal or Business account. For displaying a Hashtag or Tagged feed, you must have an Instagram Business account. If needed, you can convert a Personal account into a Business account by following the directions {link}here{link}.', 'instagram-feed' ),
			),
			'footer'                    => array(
				'heading' => __( 'Add feeds for popular social platforms with <span>our other plugins</span>', 'instagram-feed' ),
			),
			'personal'                  => __( 'Personal', 'instagram-feed' ),
			'business'                  => __( 'Business', 'instagram-feed' ),
			'notSure'                   => __( "I'm not sure", 'instagram-feed' ),
		);
	}

	/**
	 * Get the extensions popup
	 *
	 * @since 6.2.0
	 */
	public function get_extensions_popup( $license_key ) {
		return [
			'hashtag' => array(
				'heading' 		=> $this->get_extension_popup_dynamic_heading('Hashtag Feeds'),
				'description' 	=> __( 'Display posts from any public hashtag with an Instagram hashtag feed. Great for pulling in user-generated content associated with your brand, running promotional hashtag campaigns, engaging audiences at events, and more.', 'instagram-feed' ),
				'img' 			=> '<svg width="396" height="264" viewBox="0 0 396 264" fill="none" xmlns="http://www.w3.org/2000/svg"><g clip-path="url(#cliphashtagpro0_506_22510)"><g clip-path="url(#cliphashtagpro1_506_22510)"><g filter="url(#filterhashtagpro0_ddd_506_22510)"><rect x="169.506" y="31.2762" width="158.634" height="180.446" rx="3.96584" transform="rotate(4 169.506 31.2762)" fill="white"/></g><g clip-path="url(#cliphashtagpro2_506_22510)"><rect x="169.506" y="31.2762" width="158.634" height="158.634" transform="rotate(4 169.506 31.2762)" fill="#0096CC"/></g><path d="M192.49 182.337L188.63 182.067L189.561 178.756C189.585 178.659 189.587 178.557 189.568 178.458C189.549 178.359 189.509 178.266 189.45 178.184C189.391 178.102 189.315 178.034 189.228 177.984C189.14 177.934 189.043 177.904 188.943 177.896L187.552 177.798C187.391 177.783 187.23 177.824 187.097 177.914C186.963 178.004 186.864 178.138 186.818 178.292L185.855 181.873L182.099 181.611L183.029 178.3C183.053 178.204 183.056 178.105 183.038 178.008C183.021 177.911 182.983 177.819 182.927 177.738C182.871 177.657 182.799 177.588 182.715 177.537C182.631 177.486 182.537 177.453 182.439 177.441L181.048 177.343C180.888 177.328 180.727 177.369 180.593 177.459C180.459 177.549 180.361 177.683 180.315 177.837L179.365 181.42L175.192 181.128C175.029 181.113 174.867 181.156 174.733 181.249C174.598 181.341 174.501 181.478 174.457 181.636L174.075 183.007C174.047 183.107 174.042 183.213 174.06 183.316C174.079 183.419 174.12 183.517 174.181 183.602C174.242 183.686 174.322 183.756 174.414 183.806C174.506 183.856 174.608 183.885 174.712 183.89L178.572 184.16L177.043 189.645L172.869 189.353C172.707 189.338 172.544 189.381 172.41 189.474C172.276 189.567 172.179 189.703 172.135 189.861L171.752 191.232C171.724 191.333 171.719 191.438 171.738 191.541C171.756 191.644 171.797 191.742 171.859 191.827C171.92 191.911 171.999 191.981 172.091 192.031C172.183 192.081 172.285 192.11 172.39 192.115L176.278 192.387L175.347 195.698C175.323 195.797 175.321 195.901 175.341 196.002C175.361 196.102 175.403 196.197 175.464 196.279C175.525 196.361 175.603 196.429 175.694 196.478C175.784 196.527 175.884 196.555 175.986 196.56L177.377 196.658C177.532 196.664 177.685 196.619 177.812 196.529C177.938 196.44 178.031 196.31 178.076 196.161L179.053 192.581L182.809 192.844L181.879 196.155C181.855 196.25 181.852 196.35 181.869 196.447C181.887 196.543 181.925 196.635 181.981 196.717C182.036 196.798 182.109 196.866 182.193 196.917C182.277 196.969 182.371 197.001 182.469 197.014L183.86 197.111C184.02 197.126 184.181 197.085 184.315 196.995C184.448 196.905 184.547 196.771 184.593 196.617L185.577 193.037L189.751 193.329C189.913 193.344 190.076 193.301 190.21 193.208C190.344 193.115 190.441 192.979 190.485 192.821L190.861 191.45C190.889 191.349 190.894 191.243 190.875 191.14C190.857 191.037 190.816 190.94 190.754 190.855C190.693 190.77 190.614 190.7 190.522 190.65C190.43 190.6 190.328 190.572 190.223 190.566L186.342 190.295L187.865 184.81L192.038 185.102C192.201 185.116 192.363 185.073 192.498 184.981C192.632 184.888 192.729 184.751 192.773 184.594L193.155 183.223C193.184 183.119 193.189 183.011 193.169 182.906C193.149 182.801 193.106 182.701 193.041 182.616C192.977 182.53 192.894 182.461 192.798 182.412C192.703 182.364 192.597 182.338 192.49 182.337ZM183.56 190.1L179.804 189.838L181.334 184.353L185.09 184.616L183.56 190.1Z" fill="#0068A0"/><rect x="206.604" y="182.953" width="57.5047" height="13.8804" rx="1.98292" transform="rotate(4 206.604 182.953)" fill="#86D0F9"/><g filter="url(#filterhashtagpro1_ddd_506_22510)"><rect x="169.506" y="31.2762" width="158.634" height="180.446" rx="3.96584" transform="rotate(4 169.506 31.2762)" fill="white"/></g><g clip-path="url(#cliphashtagpro3_506_22510)"><rect x="169.506" y="31.2762" width="158.634" height="158.634" transform="rotate(4 169.506 31.2762)" fill="#0096CC"/><circle cx="183.84" cy="95.0859" r="92.2057" fill="#0068A0"/></g><path d="M192.49 182.337L188.63 182.067L189.561 178.756C189.585 178.659 189.587 178.557 189.568 178.458C189.549 178.359 189.509 178.266 189.45 178.184C189.391 178.102 189.315 178.034 189.228 177.984C189.14 177.934 189.043 177.904 188.943 177.896L187.552 177.798C187.391 177.783 187.23 177.824 187.097 177.914C186.963 178.004 186.864 178.138 186.818 178.292L185.855 181.873L182.099 181.611L183.029 178.3C183.053 178.204 183.056 178.105 183.038 178.008C183.021 177.911 182.983 177.819 182.927 177.738C182.871 177.657 182.799 177.588 182.715 177.537C182.631 177.486 182.537 177.453 182.439 177.441L181.048 177.343C180.888 177.328 180.727 177.369 180.593 177.459C180.459 177.549 180.361 177.683 180.315 177.837L179.365 181.42L175.192 181.128C175.029 181.113 174.867 181.156 174.733 181.249C174.598 181.341 174.501 181.478 174.457 181.636L174.075 183.007C174.047 183.107 174.042 183.213 174.06 183.316C174.079 183.419 174.12 183.517 174.181 183.602C174.242 183.686 174.322 183.756 174.414 183.806C174.506 183.856 174.608 183.885 174.712 183.89L178.572 184.16L177.043 189.645L172.869 189.353C172.707 189.338 172.544 189.381 172.41 189.474C172.276 189.567 172.179 189.703 172.135 189.861L171.752 191.232C171.724 191.333 171.719 191.438 171.738 191.541C171.756 191.644 171.797 191.742 171.859 191.827C171.92 191.911 171.999 191.981 172.091 192.031C172.183 192.081 172.285 192.11 172.39 192.115L176.278 192.387L175.347 195.698C175.323 195.797 175.321 195.901 175.341 196.002C175.361 196.102 175.403 196.197 175.464 196.279C175.525 196.361 175.603 196.429 175.694 196.478C175.784 196.527 175.884 196.555 175.986 196.56L177.377 196.658C177.532 196.664 177.685 196.619 177.812 196.529C177.938 196.44 178.031 196.31 178.076 196.161L179.053 192.581L182.809 192.844L181.879 196.155C181.855 196.25 181.852 196.35 181.869 196.447C181.887 196.543 181.925 196.635 181.981 196.717C182.036 196.798 182.109 196.866 182.193 196.917C182.277 196.969 182.371 197.001 182.469 197.014L183.86 197.111C184.02 197.126 184.181 197.085 184.315 196.995C184.448 196.905 184.547 196.771 184.593 196.617L185.577 193.037L189.751 193.329C189.913 193.344 190.076 193.301 190.21 193.208C190.344 193.115 190.441 192.979 190.485 192.821L190.861 191.45C190.889 191.349 190.894 191.243 190.875 191.14C190.857 191.037 190.816 190.94 190.754 190.855C190.693 190.77 190.614 190.7 190.522 190.65C190.43 190.6 190.328 190.572 190.223 190.566L186.342 190.295L187.865 184.81L192.038 185.102C192.201 185.116 192.363 185.073 192.498 184.981C192.632 184.888 192.729 184.751 192.773 184.594L193.155 183.223C193.184 183.119 193.189 183.011 193.169 182.906C193.149 182.801 193.106 182.701 193.041 182.616C192.977 182.53 192.894 182.461 192.798 182.412C192.703 182.364 192.597 182.338 192.49 182.337ZM183.56 190.1L179.804 189.838L181.334 184.353L185.09 184.616L183.56 190.1Z" fill="#0068A0"/><rect x="206.604" y="182.953" width="57.5047" height="13.8804" rx="1.98292" transform="rotate(4 206.604 182.953)" fill="#86D0F9"/><g filter="url(#filterhashtagpro2_ddd_506_22510)"><rect x="61.8896" y="48.4873" width="158.634" height="180.446" rx="3.96584" fill="white"/></g><g clip-path="url(#cliphashtagpro4_506_22510)"><rect x="61.8896" y="48.4873" width="158.634" height="158.634" fill="#F6966B"/><path d="M93.1149 104.67L218.7 230.255H-10.8174L93.1149 104.67Z" fill="#F9BBA0"/></g><path d="M95.3546 197.577H91.4848L92.1821 194.209C92.1992 194.11 92.1946 194.008 92.1687 193.911C92.1427 193.814 92.096 193.723 92.0317 193.646C91.9675 193.568 91.8872 193.506 91.7964 193.462C91.7056 193.419 91.6064 193.395 91.5057 193.394H90.1112C89.95 193.39 89.7925 193.442 89.6653 193.541C89.5382 193.64 89.4494 193.78 89.414 193.937L88.7028 197.577H84.9376L85.6348 194.209C85.6517 194.112 85.6479 194.013 85.6236 193.918C85.5992 193.822 85.555 193.733 85.4937 193.656C85.4325 193.579 85.3556 193.516 85.2682 193.47C85.1807 193.425 85.0847 193.399 84.9864 193.394H83.5919C83.4307 193.39 83.2731 193.442 83.146 193.541C83.0189 193.64 82.93 193.78 82.8946 193.937L82.1974 197.577H78.0138C77.8507 197.574 77.6916 197.628 77.5642 197.73C77.4368 197.832 77.3492 197.975 77.3166 198.135L77.0307 199.529C77.0098 199.632 77.0121 199.738 77.0377 199.839C77.0632 199.941 77.1112 200.035 77.1782 200.115C77.2451 200.196 77.3293 200.26 77.4245 200.303C77.5197 200.347 77.6234 200.368 77.7279 200.366H81.5977L80.4542 205.944H76.2707C76.1076 205.941 75.9485 205.995 75.8211 206.097C75.6937 206.199 75.606 206.342 75.5734 206.502L75.2876 207.897C75.2666 207.999 75.269 208.105 75.2945 208.206C75.3201 208.308 75.3681 208.402 75.435 208.483C75.502 208.563 75.5862 208.627 75.6814 208.671C75.7765 208.714 75.8802 208.735 75.9848 208.733H79.8825L79.1852 212.101C79.1678 212.202 79.1728 212.306 79.2 212.404C79.2271 212.503 79.2757 212.595 79.3423 212.673C79.4088 212.751 79.4918 212.813 79.5851 212.855C79.6785 212.897 79.78 212.918 79.8825 212.917H81.277C81.4322 212.913 81.5816 212.857 81.7016 212.759C81.8216 212.66 81.9053 212.524 81.9394 212.373L82.6645 208.733H86.4297L85.7324 212.101C85.7156 212.198 85.7194 212.297 85.7437 212.393C85.768 212.488 85.8123 212.577 85.8735 212.654C85.9348 212.732 86.0117 212.795 86.0991 212.84C86.1866 212.885 86.2826 212.911 86.3809 212.917H87.7754C87.9366 212.921 88.0941 212.869 88.2213 212.77C88.3484 212.67 88.4372 212.53 88.4727 212.373L89.2048 208.733H93.3883C93.5514 208.737 93.7105 208.683 93.8379 208.581C93.9653 208.479 94.053 208.335 94.0856 208.175L94.3645 206.781C94.3854 206.678 94.383 206.573 94.3575 206.471C94.332 206.37 94.2839 206.275 94.217 206.195C94.15 206.115 94.0658 206.05 93.9707 206.007C93.8755 205.964 93.7718 205.942 93.6672 205.944H89.7765L90.9131 200.366H95.0966C95.2597 200.369 95.4188 200.315 95.5462 200.214C95.6736 200.112 95.7612 199.968 95.7938 199.808L96.0797 198.414C96.1013 198.309 96.0985 198.201 96.0714 198.097C96.0444 197.993 95.9939 197.897 95.9237 197.816C95.8536 197.736 95.7658 197.672 95.6671 197.63C95.5684 197.589 95.4615 197.571 95.3546 197.577ZM86.9875 205.944H83.2223L84.3658 200.366H88.131L86.9875 205.944Z" fill="#FE544F"/><rect x="109.479" y="197.206" width="57.5047" height="13.8804" rx="1.98292" fill="#FCE1D5"/><g filter="url(#filterhashtagpro3_ddd_506_22510)"><rect x="61.8896" y="48.4873" width="158.634" height="180.446" rx="3.96584" fill="white"/></g><g clip-path="url(#cliphashtagpro5_506_22510)"><rect x="61.8896" y="48.4873" width="158.634" height="158.634" fill="#F6966B"/><path d="M93.1149 104.67L218.7 230.255H-10.8174L93.1149 104.67Z" fill="#F9BBA0"/></g><path d="M95.3546 197.577H91.4848L92.1821 194.209C92.1992 194.11 92.1946 194.008 92.1687 193.911C92.1427 193.814 92.096 193.723 92.0317 193.646C91.9675 193.568 91.8872 193.506 91.7964 193.462C91.7056 193.419 91.6064 193.395 91.5057 193.394H90.1112C89.95 193.39 89.7925 193.442 89.6653 193.541C89.5382 193.64 89.4494 193.78 89.414 193.937L88.7028 197.577H84.9376L85.6348 194.209C85.6517 194.112 85.6479 194.013 85.6236 193.918C85.5992 193.822 85.555 193.733 85.4937 193.656C85.4325 193.579 85.3556 193.516 85.2682 193.47C85.1807 193.425 85.0847 193.399 84.9864 193.394H83.5919C83.4307 193.39 83.2731 193.442 83.146 193.541C83.0189 193.64 82.93 193.78 82.8946 193.937L82.1974 197.577H78.0138C77.8507 197.574 77.6916 197.628 77.5642 197.73C77.4368 197.832 77.3492 197.975 77.3166 198.135L77.0307 199.529C77.0098 199.632 77.0121 199.738 77.0377 199.839C77.0632 199.941 77.1112 200.035 77.1782 200.115C77.2451 200.196 77.3293 200.26 77.4245 200.303C77.5197 200.347 77.6234 200.368 77.7279 200.366H81.5977L80.4542 205.944H76.2707C76.1076 205.941 75.9485 205.995 75.8211 206.097C75.6937 206.199 75.606 206.342 75.5734 206.502L75.2876 207.897C75.2666 207.999 75.269 208.105 75.2945 208.206C75.3201 208.308 75.3681 208.402 75.435 208.483C75.502 208.563 75.5862 208.627 75.6814 208.671C75.7765 208.714 75.8802 208.735 75.9848 208.733H79.8825L79.1852 212.101C79.1678 212.202 79.1728 212.306 79.2 212.404C79.2271 212.503 79.2757 212.595 79.3423 212.673C79.4088 212.751 79.4918 212.813 79.5851 212.855C79.6785 212.897 79.78 212.918 79.8825 212.917H81.277C81.4322 212.913 81.5816 212.857 81.7016 212.759C81.8216 212.66 81.9053 212.524 81.9394 212.373L82.6645 208.733H86.4297L85.7324 212.101C85.7156 212.198 85.7194 212.297 85.7437 212.393C85.768 212.488 85.8123 212.577 85.8735 212.654C85.9348 212.732 86.0117 212.795 86.0991 212.84C86.1866 212.885 86.2826 212.911 86.3809 212.917H87.7754C87.9366 212.921 88.0941 212.869 88.2213 212.77C88.3484 212.67 88.4372 212.53 88.4727 212.373L89.2048 208.733H93.3883C93.5514 208.737 93.7105 208.683 93.8379 208.581C93.9653 208.479 94.053 208.335 94.0856 208.175L94.3645 206.781C94.3854 206.678 94.383 206.573 94.3575 206.471C94.332 206.37 94.2839 206.275 94.217 206.195C94.15 206.115 94.0658 206.05 93.9707 206.007C93.8755 205.964 93.7718 205.942 93.6672 205.944H89.7765L90.9131 200.366H95.0966C95.2597 200.369 95.4188 200.315 95.5462 200.214C95.6736 200.112 95.7612 199.968 95.7938 199.808L96.0797 198.414C96.1013 198.309 96.0985 198.201 96.0714 198.097C96.0444 197.993 95.9939 197.897 95.9237 197.816C95.8536 197.736 95.7658 197.672 95.6671 197.63C95.5684 197.589 95.4615 197.571 95.3546 197.577ZM86.9875 205.944H83.2223L84.3658 200.366H88.131L86.9875 205.944Z" fill="#FE544F"/><rect x="109.479" y="197.206" width="57.5047" height="13.8804" rx="1.98292" fill="#FCE1D5"/></g></g><defs><filter id="filterhashtagpro0_ddd_506_22510" x="131.408" y="17.6626" width="221.857" height="242.094" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="11.8975"/><feGaussianBlur stdDeviation="12.889"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_506_22510"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="1.98292"/><feGaussianBlur stdDeviation="1.98292"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/><feBlend mode="normal" in2="effect1_dropShadow_506_22510" result="effect2_dropShadow_506_22510"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="5.94876"/><feGaussianBlur stdDeviation="5.94876"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/><feBlend mode="normal" in2="effect2_dropShadow_506_22510" result="effect3_dropShadow_506_22510"/><feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow_506_22510" result="shape"/></filter><filter id="filterhashtagpro1_ddd_506_22510" x="131.408" y="17.6626" width="221.857" height="242.094" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="11.8975"/><feGaussianBlur stdDeviation="12.889"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_506_22510"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="1.98292"/><feGaussianBlur stdDeviation="1.98292"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/><feBlend mode="normal" in2="effect1_dropShadow_506_22510" result="effect2_dropShadow_506_22510"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="5.94876"/><feGaussianBlur stdDeviation="5.94876"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/><feBlend mode="normal" in2="effect2_dropShadow_506_22510" result="effect3_dropShadow_506_22510"/><feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow_506_22510" result="shape"/></filter><filter id="filterhashtagpro2_ddd_506_22510" x="36.1117" y="34.6069" width="210.19" height="232.002" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="11.8975"/><feGaussianBlur stdDeviation="12.889"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_506_22510"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="1.98292"/><feGaussianBlur stdDeviation="1.98292"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/><feBlend mode="normal" in2="effect1_dropShadow_506_22510" result="effect2_dropShadow_506_22510"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="5.94876"/><feGaussianBlur stdDeviation="5.94876"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/><feBlend mode="normal" in2="effect2_dropShadow_506_22510" result="effect3_dropShadow_506_22510"/><feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow_506_22510" result="shape"/></filter><filter id="filterhashtagpro3_ddd_506_22510" x="36.1117" y="34.6069" width="210.19" height="232.002" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="11.8975"/><feGaussianBlur stdDeviation="12.889"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_506_22510"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="1.98292"/><feGaussianBlur stdDeviation="1.98292"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/><feBlend mode="normal" in2="effect1_dropShadow_506_22510" result="effect2_dropShadow_506_22510"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="5.94876"/><feGaussianBlur stdDeviation="5.94876"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/><feBlend mode="normal" in2="effect2_dropShadow_506_22510" result="effect3_dropShadow_506_22510"/><feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow_506_22510" result="shape"/></filter><clipPath id="cliphashtagpro0_506_22510"><rect width="396" height="264" fill="white"/></clipPath><clipPath id="cliphashtagpro1_506_22510"><rect width="530" height="250.308" fill="white" transform="translate(-67 6.84601)"/></clipPath><clipPath id="cliphashtagpro2_506_22510"><path d="M169.275 34.573C169.403 32.7522 170.982 31.3794 172.803 31.5067L324.456 42.1114C326.277 42.2387 327.65 43.818 327.522 45.6387L318.624 172.896L160.377 161.83L169.275 34.573Z" fill="white"/></clipPath><clipPath id="cliphashtagpro3_506_22510"><path d="M169.275 34.573C169.403 32.7522 170.982 31.3794 172.803 31.5067L324.456 42.1114C326.277 42.2387 327.65 43.818 327.522 45.6387L318.624 172.896L160.377 161.83L169.275 34.573Z" fill="white"/></clipPath><clipPath id="cliphashtagpro4_506_22510"><path d="M61.8896 51.7922C61.8896 49.9669 63.3693 48.4873 65.1945 48.4873H217.218C219.044 48.4873 220.523 49.9669 220.523 51.7922V179.36H61.8896V51.7922Z" fill="white"/></clipPath><clipPath id="cliphashtagpro5_506_22510"><path d="M61.8896 51.7922C61.8896 49.9669 63.3693 48.4873 65.1945 48.4873H217.218C219.044 48.4873 220.523 49.9669 220.523 51.7922V179.36H61.8896V51.7922Z" fill="white"/></clipPath></defs></svg>',
				'popupContentBtn' 	=> '<div class="sbi-fb-extpp-lite-btn">' . self::builder_svg_icons()['tag'] . __( 'Instagram Pro users get 50% OFF', 'instagram-feed' ) .'</div>',
				'bullets'       => [
					'heading' => __( 'And get much more!', 'instagram-feed' ),
					'content' => [
						__( 'Display Hashtag & Tagged feeds', 'instagram-feed' ),
						__( 'Powerful visual moderation', 'instagram-feed' ),
						__( 'Comments and Likes', 'instagram-feed' ),
						__( 'Highlight specific posts', 'instagram-feed' ),
						__( 'Multiple layout options', 'instagram-feed' ),
						__( 'Popup photo/video lightbox', 'instagram-feed' ),
						__( 'Instagram Stories', 'instagram-feed' ),
						__( 'Shoppable feeds', 'instagram-feed' ),
						__( 'Pro support', 'instagram-feed' ),
						__( 'Post captions', 'instagram-feed' ),
						__( 'Combine multiple feed types', 'instagram-feed' ),
						__( '30 day money back guarantee', 'instagram-feed' ),
					]
				],
				'demoUrl' 		=> 'https://smashballoon.com/instagram-feed/demo/hashtag/?utm_campaign=instagram-pro&utm_source=feed-type&utm_medium=public-hashtags&utm_content=view-demo',
				'buyUrl' 		=> $this->get_extension_popup_dynamic_buy_url( $license_key, 'https://smashballoon.com/instagram-feed/demo/hashtag?utm_campaign=instagram-free&utm_source=feed-type&utm_medium=hashtag' )
			),
			'tagged' => array(
				'heading' 		=> $this->get_extension_popup_dynamic_heading('Tagged Posts Feed'),
				'description' 	=> __( 'Display posts that you\'ve been tagged in by other users allowing you to increase your audience\'s engagement with your Instagram account.', 'instagram-feed' ),
				'img' 			=> '<svg width="396" height="264" viewBox="0 0 396 264" fill="none" xmlns="http://www.w3.org/2000/svg"><g clip-path="url(#cliptagged0_506_22672)"><g clip-path="url(#cliptagged1_506_22672)"><g filter="url(#filtertagged0_ddd_506_22672)"><g clip-path="url(#cliptagged2_506_22672)"><rect x="139.63" y="64.4107" width="164.72" height="166.806" rx="4.17014" transform="rotate(2 139.63 64.4107)" fill="white"/><g clip-path="url(#cliptagged3_506_22672)"><path d="M139.131 52.643L305.835 58.4644L300.45 212.666L133.746 206.844L139.131 52.643Z" fill="#59AB46"/><path d="M170.246 102.41L297.611 238.993L56.4177 230.57L170.246 102.41Z" fill="#76C064"/></g><path fill-rule="evenodd" clip-rule="evenodd" d="M157.064 194.828C159.201 194.48 161.393 194.773 163.363 195.671C165.333 196.569 166.993 198.031 168.132 199.872C169.271 201.713 169.839 203.85 169.763 206.014L169.707 207.635C169.671 208.674 169.223 209.656 168.463 210.364C167.703 211.073 166.692 211.451 165.653 211.415C164.615 211.379 163.633 210.931 162.924 210.171C162.792 210.029 162.671 209.879 162.563 209.721C161.492 210.7 160.081 211.22 158.63 211.17C157.162 211.118 155.773 210.486 154.771 209.411C153.769 208.336 153.235 206.907 153.286 205.439C153.337 203.97 153.97 202.582 155.045 201.58C156.119 200.578 157.548 200.043 159.017 200.095C160.486 200.146 161.874 200.779 162.876 201.853C163.878 202.928 164.412 204.357 164.361 205.826L164.304 207.446C164.291 207.84 164.434 208.224 164.703 208.512C164.972 208.8 165.344 208.97 165.738 208.984C166.132 208.998 166.516 208.854 166.804 208.585C167.092 208.316 167.262 207.944 167.276 207.55L167.332 205.929C167.391 204.246 166.95 202.584 166.063 201.152C165.177 199.72 163.887 198.583 162.354 197.885C160.822 197.186 159.117 196.958 157.455 197.229C155.793 197.499 154.249 198.257 153.017 199.405C151.785 200.554 150.922 202.042 150.536 203.681C150.15 205.32 150.259 207.037 150.849 208.614C151.438 210.191 152.483 211.558 153.849 212.542C155.216 213.526 156.844 214.082 158.526 214.141L158.528 214.141C160.021 214.195 161.502 213.853 162.819 213.15C163.412 212.834 164.148 213.058 164.465 213.651C164.781 214.243 164.557 214.98 163.964 215.296C162.267 216.202 160.362 216.641 158.44 216.572L158.484 215.357L158.442 216.572C158.441 216.572 158.441 216.572 158.44 216.572C156.277 216.496 154.185 215.781 152.428 214.516C150.671 213.251 149.328 211.494 148.57 209.466C147.812 207.438 147.672 205.231 148.168 203.123C148.665 201.016 149.775 199.103 151.358 197.626C152.941 196.15 154.927 195.176 157.064 194.828ZM161.93 205.741C161.959 204.917 161.659 204.115 161.097 203.512C160.535 202.909 159.756 202.555 158.932 202.526C158.108 202.497 157.307 202.797 156.704 203.359C156.101 203.921 155.746 204.7 155.717 205.524C155.688 206.348 155.988 207.149 156.55 207.752C157.113 208.355 157.891 208.71 158.715 208.739C159.539 208.767 160.34 208.468 160.943 207.905C161.546 207.343 161.901 206.564 161.93 205.741Z" fill="#59AB46"/><rect x="184.048" y="200.256" width="60.467" height="14.5955" rx="2.08507" transform="rotate(2 184.048 200.256)" fill="#B6DDAD"/></g></g><g filter="url(#filtertagged1_ddd_506_22672)"><g clip-path="url(#cliptagged4_506_22672)"><rect x="73.3418" y="28.4567" width="164.72" height="166.806" rx="4.17014" transform="rotate(-2 73.3418 28.4567)" fill="white"/><g clip-path="url(#cliptagged5_506_22672)"><path d="M72.0225 16.7529L238.726 10.9315L244.111 165.133L77.4073 170.954L72.0225 16.7529Z" fill="#43A6DB"/><circle cx="268.485" cy="149.951" r="96.9557" transform="rotate(-2 268.485 149.951)" fill="#0068A0"/></g><path fill-rule="evenodd" clip-rule="evenodd" d="M99.8296 157.341C101.937 156.845 104.144 156.985 106.172 157.743C108.2 158.501 109.958 159.844 111.223 161.601C112.487 163.358 113.203 165.451 113.279 167.615L113.335 169.235C113.371 170.274 112.994 171.285 112.285 172.045C111.576 172.805 110.594 173.253 109.555 173.289C108.516 173.325 107.506 172.947 106.745 172.238C106.604 172.106 106.473 171.965 106.354 171.815C105.354 172.866 103.983 173.483 102.532 173.534C101.063 173.585 99.6345 173.051 98.5597 172.049C97.485 171.047 96.8524 169.659 96.8011 168.19C96.7498 166.721 97.284 165.292 98.2862 164.218C99.2885 163.143 100.677 162.51 102.145 162.459C103.614 162.408 105.043 162.942 106.117 163.944C107.192 164.946 107.825 166.335 107.876 167.803L107.933 169.424C107.946 169.818 108.116 170.19 108.405 170.459C108.693 170.728 109.076 170.871 109.47 170.858C109.864 170.844 110.237 170.674 110.506 170.386C110.774 170.098 110.918 169.714 110.904 169.32L110.847 167.699C110.789 166.017 110.232 164.389 109.248 163.022C108.265 161.656 106.898 160.611 105.32 160.022C103.743 159.432 102.026 159.323 100.387 159.709C98.7481 160.095 97.2602 160.958 96.1118 162.19C94.9634 163.422 94.2059 164.966 93.9353 166.628C93.6646 168.29 93.8928 169.995 94.5911 171.527C95.2895 173.06 96.4265 174.35 97.8584 175.236C99.2903 176.123 100.953 176.564 102.636 176.505L102.637 176.505C104.13 176.455 105.584 176.011 106.849 175.218C107.418 174.861 108.169 175.033 108.525 175.602C108.882 176.171 108.71 176.922 108.141 177.279C106.511 178.3 104.641 178.871 102.719 178.937L102.678 177.721L102.721 178.937C102.72 178.937 102.72 178.937 102.719 178.937C100.556 179.012 98.419 178.444 96.5783 177.305C94.7373 176.166 93.2754 174.506 92.3776 172.536C91.4798 170.566 91.1863 168.374 91.5343 166.237C91.8823 164.1 92.8562 162.114 94.3327 160.531C95.8093 158.948 97.7222 157.838 99.8296 157.341ZM105.445 167.888C105.416 167.064 105.061 166.286 104.458 165.723C103.856 165.161 103.054 164.861 102.23 164.89C101.406 164.919 100.628 165.274 100.065 165.877C99.5031 166.48 99.2034 167.281 99.2322 168.105C99.261 168.929 99.6158 169.708 100.219 170.27C100.822 170.832 101.623 171.132 102.447 171.103C103.271 171.074 104.05 170.719 104.612 170.116C105.174 169.514 105.474 168.712 105.445 167.888Z" fill="#0068A0"/><rect x="127.129" y="160.874" width="60.467" height="14.5955" rx="2.08507" transform="rotate(-2 127.129 160.874)" fill="#86D0F9"/></g></g></g></g><defs><filter id="filtertagged0_ddd_506_22672" x="106.703" y="49.8152" width="224.653" height="226.664" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="12.5104"/><feGaussianBlur stdDeviation="13.5529"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_506_22672"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="2.08507"/><feGaussianBlur stdDeviation="2.08507"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/><feBlend mode="normal" in2="effect1_dropShadow_506_22672" result="effect2_dropShadow_506_22672"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="6.25521"/><feGaussianBlur stdDeviation="6.25521"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/><feBlend mode="normal" in2="effect2_dropShadow_506_22672" result="effect3_dropShadow_506_22672"/><feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow_506_22672" result="shape"/></filter><filter id="filtertagged1_ddd_506_22672" x="46.2359" y="8.11258" width="224.653" height="226.664" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="12.5104"/><feGaussianBlur stdDeviation="13.5529"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_506_22672"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="2.08507"/><feGaussianBlur stdDeviation="2.08507"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/><feBlend mode="normal" in2="effect1_dropShadow_506_22672" result="effect2_dropShadow_506_22672"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="6.25521"/><feGaussianBlur stdDeviation="6.25521"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/><feBlend mode="normal" in2="effect2_dropShadow_506_22672" result="effect3_dropShadow_506_22672"/><feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow_506_22672" result="shape"/></filter><clipPath id="cliptagged0_506_22672"><rect width="396" height="264" fill="white"/></clipPath><clipPath id="cliptagged1_506_22672"><rect width="530" height="250.308" fill="white" transform="translate(-67 6.84601)"/></clipPath><clipPath id="cliptagged2_506_22672"><rect x="139.63" y="64.4107" width="164.72" height="166.806" rx="4.17014" transform="rotate(2 139.63 64.4107)" fill="white"/></clipPath><clipPath id="cliptagged3_506_22672"><path d="M139.373 45.6973C139.44 43.7792 141.049 42.2786 142.967 42.3456L302.725 47.9245C304.643 47.9914 306.144 49.6006 306.077 51.5187L301.395 185.576L134.691 179.755L139.373 45.6973Z" fill="white"/></clipPath><clipPath id="cliptagged4_506_22672"><rect x="73.3418" y="28.4567" width="164.72" height="166.806" rx="4.17014" transform="rotate(-2 73.3418 28.4567)" fill="white"/></clipPath><clipPath id="cliptagged5_506_22672"><path d="M71.7785 9.80722C71.7115 7.88913 73.2121 6.27993 75.1302 6.21295L234.888 0.634078C236.806 0.567097 238.415 2.06771 238.482 3.9858L243.164 138.044L76.4599 143.865L71.7785 9.80722Z" fill="white"/></clipPath></defs></svg>',
				'popupContentBtn' 	=> '<div class="sbi-fb-extpp-lite-btn">' . self::builder_svg_icons()['tag'] . __( 'Instagram Pro users get 50% OFF', 'instagram-feed' ) .'</div>',
				'bullets'       => [
					'heading' => __( 'And get much more!', 'instagram-feed' ),
					'content' => [
						__( 'Display Hashtag & Tagged feeds', 'instagram-feed' ),
						__( 'Powerful visual moderation', 'instagram-feed' ),
						__( 'Comments and Likes', 'instagram-feed' ),
						__( 'Highlight specific posts', 'instagram-feed' ),
						__( 'Multiple layout options', 'instagram-feed' ),
						__( 'Popup photo/video lightbox', 'instagram-feed' ),
						__( 'Instagram Stories', 'instagram-feed' ),
						__( 'Shoppable feeds', 'instagram-feed' ),
						__( 'Pro support', 'instagram-feed' ),
						__( 'Post captions', 'instagram-feed' ),
						__( 'Combine multiple feed types', 'instagram-feed' ),
						__( '30 day money back guarantee', 'instagram-feed' ),
					]
				],
				'demoUrl' 		=> 'https://smashballoon.com/instagram-feed/demo/hashtag/?utm_campaign=instagram-pro&utm_source=feed-type&utm_medium=tagged-post&utm_content=view-demo',
				'buyUrl' 		=> $this->get_extension_popup_dynamic_buy_url( $license_key, 'https://smashballoon.com/instagram-feed/demo/?utm_campaign=instagram-free&utm_source=feed-type&utm_medium=tagged' )
			),
			'socialwall' => [
				//Combine all your social media channels into one Social Wall
				'heading' 		=> __( '<span class="sb-social-wall">Combine all your social media channels into one', 'instagram-feed' ) .' <span>'. __( 'Social Wall', 'instagram-feed' ).'</span></span>',
				'description' 	=> __( '<span class="sb-social-wall">A dash of Instagram, a sprinkle of Facebook, a spoonful of Twitter, and a dollop of YouTube, all in the same feed.</span>', 'instagram-feed' ),
				'popupContentBtn' 	=> '<div class="sbi-fb-extpp-lite-btn">' . self::builder_svg_icons()['tag'] . __( 'Instagram Pro users get 50% OFF', 'instagram-feed' ) .'</div>',
				'img' 			=> '<svg width="397" height="264" viewBox="0 0 397 264" fill="none" xmlns="http://www.w3.org/2000/svg"><g clip-path="url(#clip0)"><g filter="url(#filter0_ddd)"><rect x="18.957" y="63" width="113.812" height="129.461" rx="2.8453" fill="white"/></g><g clip-path="url(#clip1)"><path d="M18.957 63H132.769V176.812H18.957V63Z" fill="#0068A0"/><rect x="56.957" y="106" width="105" height="105" rx="9" fill="#005B8C"/></g><path d="M36.0293 165.701C31.4649 165.701 27.7305 169.427 27.7305 174.017C27.7305 178.166 30.7678 181.61 34.7347 182.232V176.423H32.6268V174.017H34.7347V172.183C34.7347 170.1 35.9712 168.954 37.8716 168.954C38.7762 168.954 39.7222 169.112 39.7222 169.112V171.162H38.6766C37.6475 171.162 37.3239 171.801 37.3239 172.456V174.017H39.6309L39.2575 176.423H37.3239V182.232C39.2794 181.924 41.0602 180.926 42.3446 179.419C43.629 177.913 44.3325 175.996 44.3281 174.017C44.3281 169.427 40.5936 165.701 36.0293 165.701Z" fill="#006BFA"/><rect x="53.1016" y="169.699" width="41.2569" height="9.95855" rx="1.42265" fill="#D0D1D7"/><g filter="url(#filter1_ddd)"><rect x="18.957" y="201" width="113.812" height="129.461" rx="2.8453" fill="white"/></g><g clip-path="url(#clip2)"><path d="M18.957 201H132.769V314.812H18.957V201Z" fill="#EC352F"/><circle cx="23.957" cy="243" r="59" fill="#FE544F"/></g><g filter="url(#filter2_ddd)"><rect x="139.957" y="23" width="113.812" height="129.461" rx="2.8453" fill="white"/></g><g clip-path="url(#clip3)"><path d="M139.957 23H253.769V136.812H139.957V23Z" fill="#8C8F9A"/><circle cx="127.457" cy="142.5" r="78.5" fill="#D0D1D7"/></g><path d="M157.026 129.493C154.537 129.493 152.553 131.516 152.553 133.967C152.553 136.456 154.537 138.44 157.026 138.44C159.477 138.44 161.5 136.456 161.5 133.967C161.5 131.516 159.477 129.493 157.026 129.493ZM157.026 136.884C155.431 136.884 154.109 135.601 154.109 133.967C154.109 132.372 155.392 131.088 157.026 131.088C158.621 131.088 159.905 132.372 159.905 133.967C159.905 135.601 158.621 136.884 157.026 136.884ZM162.706 129.338C162.706 128.754 162.239 128.287 161.655 128.287C161.072 128.287 160.605 128.754 160.605 129.338C160.605 129.921 161.072 130.388 161.655 130.388C162.239 130.388 162.706 129.921 162.706 129.338ZM165.662 130.388C165.584 128.987 165.273 127.743 164.262 126.731C163.25 125.72 162.005 125.409 160.605 125.331C159.166 125.253 154.848 125.253 153.408 125.331C152.008 125.409 150.802 125.72 149.752 126.731C148.74 127.743 148.429 128.987 148.351 130.388C148.274 131.827 148.274 136.145 148.351 137.585C148.429 138.985 148.74 140.191 149.752 141.241C150.802 142.253 152.008 142.564 153.408 142.642C154.848 142.719 159.166 142.719 160.605 142.642C162.005 142.564 163.25 142.253 164.262 141.241C165.273 140.191 165.584 138.985 165.662 137.585C165.74 136.145 165.74 131.827 165.662 130.388ZM163.795 139.102C163.523 139.88 162.9 140.463 162.161 140.774C160.994 141.241 158.271 141.124 157.026 141.124C155.742 141.124 153.019 141.241 151.891 140.774C151.113 140.463 150.53 139.88 150.219 139.102C149.752 137.974 149.868 135.25 149.868 133.967C149.868 132.722 149.752 129.999 150.219 128.832C150.53 128.093 151.113 127.509 151.891 127.198C153.019 126.731 155.742 126.848 157.026 126.848C158.271 126.848 160.994 126.731 162.161 127.198C162.9 127.47 163.484 128.093 163.795 128.832C164.262 129.999 164.145 132.722 164.145 133.967C164.145 135.25 164.262 137.974 163.795 139.102Z" fill="url(#paint0_linear)"/><rect x="174.102" y="129.699" width="41.2569" height="9.95855" rx="1.42265" fill="#D0D1D7"/><g filter="url(#filter3_ddd)"><rect x="139.957" y="161" width="114" height="109" rx="2.8453" fill="white"/></g><rect x="148.957" y="194" width="91" height="8" rx="1.42265" fill="#D0D1D7"/><rect x="148.957" y="208" width="51" height="8" rx="1.42265" fill="#D0D1D7"/><path d="M164.366 172.062C163.788 172.324 163.166 172.497 162.521 172.579C163.181 172.182 163.691 171.552 163.931 170.794C163.308 171.169 162.618 171.432 161.891 171.582C161.298 170.937 160.466 170.562 159.521 170.562C157.758 170.562 156.318 172.002 156.318 173.779C156.318 174.034 156.348 174.282 156.401 174.514C153.731 174.379 151.353 173.097 149.771 171.154C149.493 171.627 149.336 172.182 149.336 172.767C149.336 173.884 149.898 174.874 150.768 175.437C150.236 175.437 149.741 175.287 149.306 175.062V175.084C149.306 176.644 150.416 177.949 151.886 178.242C151.414 178.371 150.918 178.389 150.438 178.294C150.642 178.934 151.041 179.493 151.579 179.894C152.117 180.295 152.767 180.517 153.438 180.529C152.301 181.43 150.891 181.916 149.441 181.909C149.186 181.909 148.931 181.894 148.676 181.864C150.101 182.779 151.796 183.312 153.611 183.312C159.521 183.312 162.768 178.407 162.768 174.154C162.768 174.012 162.768 173.877 162.761 173.734C163.391 173.284 163.931 172.714 164.366 172.062Z" fill="#1B90EF"/><g filter="url(#filter4_ddd)"><rect x="260.957" y="63" width="113.812" height="129.461" rx="2.8453" fill="white"/></g><g clip-path="url(#clip4)"><rect x="260.957" y="63" width="113.812" height="113.812" fill="#D72C2C"/><path d="M283.359 103.308L373.461 193.41H208.793L283.359 103.308Z" fill="#DF5757"/></g><path d="M276.37 176.456L280.677 173.967L276.37 171.477V176.456ZM285.963 169.958C286.071 170.348 286.145 170.871 286.195 171.535C286.253 172.199 286.278 172.772 286.278 173.27L286.328 173.967C286.328 175.784 286.195 177.12 285.963 177.975C285.755 178.722 285.274 179.203 284.527 179.411C284.137 179.519 283.423 179.593 282.328 179.643C281.249 179.701 280.262 179.726 279.349 179.726L278.029 179.776C274.552 179.776 272.386 179.643 271.531 179.411C270.784 179.203 270.303 178.722 270.096 177.975C269.988 177.585 269.913 177.062 269.863 176.398C269.805 175.734 269.78 175.162 269.78 174.664L269.73 173.967C269.73 172.149 269.863 170.813 270.096 169.958C270.303 169.212 270.784 168.73 271.531 168.523C271.921 168.415 272.635 168.34 273.73 168.29C274.809 168.232 275.797 168.207 276.71 168.207L278.029 168.158C281.506 168.158 283.672 168.29 284.527 168.523C285.274 168.73 285.755 169.212 285.963 169.958Z" fill="#EB2121"/><rect x="295.102" y="169.699" width="41.2569" height="9.95855" rx="1.42265" fill="#D0D1D7"/><g filter="url(#filter5_ddd)"><rect x="260.957" y="201" width="113.812" height="129.461" rx="2.8453" fill="white"/></g><g clip-path="url(#clip5)"><rect x="260.957" y="201" width="113.812" height="113.812" fill="#59AB46"/><circle cx="374.457" cy="235.5" r="44.5" fill="#468737"/></g><g clip-path="url(#clip6)"><path d="M139.957 228H253.957V296C253.957 296.552 253.509 297 252.957 297H140.957C140.405 297 139.957 296.552 139.957 296V228Z" fill="#0068A0"/><circle cx="227.957" cy="245" r="34" fill="#004D77"/></g></g><defs><filter id="filter0_ddd" x="0.462572" y="53.0414" width="150.801" height="166.45" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="8.5359"/><feGaussianBlur stdDeviation="9.24723"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="1.42265"/><feGaussianBlur stdDeviation="1.42265"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/><feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="4.26795"/><feGaussianBlur stdDeviation="4.26795"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/><feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/><feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow" result="shape"/></filter><filter id="filter1_ddd" x="0.462572" y="191.041" width="150.801" height="166.45" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="8.5359"/><feGaussianBlur stdDeviation="9.24723"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="1.42265"/><feGaussianBlur stdDeviation="1.42265"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/><feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="4.26795"/><feGaussianBlur stdDeviation="4.26795"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/><feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/><feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow" result="shape"/></filter><filter id="filter2_ddd" x="121.463" y="13.0414" width="150.801" height="166.45" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="8.5359"/><feGaussianBlur stdDeviation="9.24723"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="1.42265"/><feGaussianBlur stdDeviation="1.42265"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/><feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="4.26795"/><feGaussianBlur stdDeviation="4.26795"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/><feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/><feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow" result="shape"/></filter><filter id="filter3_ddd" x="121.463" y="151.041" width="150.989" height="145.989" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="8.5359"/><feGaussianBlur stdDeviation="9.24723"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="1.42265"/><feGaussianBlur stdDeviation="1.42265"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/><feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="4.26795"/><feGaussianBlur stdDeviation="4.26795"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/><feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/><feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow" result="shape"/></filter><filter id="filter4_ddd" x="242.463" y="53.0414" width="150.801" height="166.45" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="8.5359"/><feGaussianBlur stdDeviation="9.24723"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="1.42265"/><feGaussianBlur stdDeviation="1.42265"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/><feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="4.26795"/><feGaussianBlur stdDeviation="4.26795"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/><feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/><feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow" result="shape"/></filter><filter id="filter5_ddd" x="242.463" y="191.041" width="150.801" height="166.45" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="8.5359"/><feGaussianBlur stdDeviation="9.24723"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="1.42265"/><feGaussianBlur stdDeviation="1.42265"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/><feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="4.26795"/><feGaussianBlur stdDeviation="4.26795"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/><feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/><feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow" result="shape"/></filter><linearGradient id="paint0_linear" x1="154.502" y1="158.603" x2="191.208" y2="121.133" gradientUnits="userSpaceOnUse"><stop stop-color="white"/><stop offset="0.147864" stop-color="#F6640E"/><stop offset="0.443974" stop-color="#BA03A7"/><stop offset="0.733337" stop-color="#6A01B9"/><stop offset="1" stop-color="#6B01B9"/></linearGradient><clipPath id="clip0"><rect width="396" height="264" fill="white" transform="translate(0.957031)"/></clipPath><clipPath id="clip1"><path d="M18.957 65.3711C18.957 64.0616 20.0186 63 21.3281 63H130.398C131.708 63 132.769 64.0616 132.769 65.3711V156.895H18.957V65.3711Z" fill="white"/></clipPath><clipPath id="clip2"><path d="M18.957 203.371C18.957 202.062 20.0186 201 21.3281 201H130.398C131.708 201 132.769 202.062 132.769 203.371V294.895H18.957V203.371Z" fill="white"/></clipPath><clipPath id="clip3"><path d="M139.957 25.3711C139.957 24.0616 141.019 23 142.328 23H251.398C252.708 23 253.769 24.0616 253.769 25.3711V116.895H139.957V25.3711Z" fill="white"/></clipPath><clipPath id="clip4"><path d="M260.957 65.3711C260.957 64.0616 262.019 63 263.328 63H372.398C373.708 63 374.769 64.0616 374.769 65.3711V156.895H260.957V65.3711Z" fill="white"/></clipPath><clipPath id="clip5"><path d="M260.957 203.371C260.957 202.062 262.019 201 263.328 201H372.398C373.708 201 374.769 202.062 374.769 203.371V294.895H260.957V203.371Z" fill="white"/></clipPath><clipPath id="clip6"><path d="M139.957 228H253.957V296C253.957 296.552 253.509 297 252.957 297H140.957C140.405 297 139.957 296.552 139.957 296V228Z" fill="white"/></clipPath></defs></svg>',
				'demoUrl' 		=> 'https://smashballoon.com/social-wall/demo/?utm_campaign=instagram-free&utm_source=feed-type&utm_medium=social-wall&utm_content=learn-more',
				'buyUrl' 		=> $this->get_extension_popup_dynamic_buy_url( $license_key, 'https://smashballoon.com/social-wall/demo/?license_key=%s&upgrade=true&utm_campaign=instagram-free&utm_source=feed-type&utm_medium=social-wall&utm_content=Try Demo', $license_key) ,
				'bullets'       => [
					'heading' => __( 'Upgrade to the All Access Bundle and get:', 'instagram-feed' ),
					'content' => [
						__( 'Instagram Feed Pro', 'instagram-feed' ),
						__( 'Custom Twitter Feeds Pro', 'instagram-feed' ),
						__( 'YouTube Feeds Pro', 'instagram-feed' ),
						__( 'Custom Facebook Feed Pro', 'instagram-feed' ),
						__( 'All Pro Facebook Extensions', 'instagram-feed' ),
						__( 'Social Wall Pro', 'instagram-feed' ),
					]
				],
			],
			//Other Types
			'feedLayout' => array(
				'heading' 		=> $this->get_extension_popup_dynamic_heading('Feed Layouts'),
				'description' 	=> __( 'Choose from one of our built-in layout options; grid, carousel, masonry, and highlight to allow you to showcase your content in any way you want.', 'instagram-feed' ),
				'img' 			=> '<svg width="396" height="264" viewBox="0 0 396 264" fill="none" xmlns="http://www.w3.org/2000/svg"><g filter="url(#feedLayoutSettingsfilter0_d_541_17506)"><rect x="45" y="67.0205" width="166.67" height="154.67" rx="2.66672" transform="rotate(-3 45 67.0205)" fill="white"/><g clip-path="url(#clip0feedLayoutSettings_541_17506)"><rect width="97.3351" height="82.6682" transform="translate(54.8096 75.8527) rotate(-3)" fill="#FCE1D5"/><circle cx="50.9329" cy="167.516" r="66.0012" transform="rotate(-3 50.9329 167.516)" fill="#F9BBA0"/><circle cx="151.997" cy="121.496" r="56.001" transform="rotate(-3 151.997 121.496)" fill="#FE544F"/></g><rect width="42.6674" height="37.334" transform="translate(160 70.3398) rotate(-3)" fill="#E8E8EB"/><rect width="42.6674" height="37.334" transform="translate(162.372 115.612) rotate(-3)" fill="#DCDDE1"/><rect width="66.6679" height="44.0008" transform="translate(140.779 162.14) rotate(-3)" fill="#D0D1D7"/><rect width="73.3347" height="44.0008" transform="translate(59.5542 166.397) rotate(-3)" fill="#E8E8EB"/></g><g filter="url(#feedLayoutSettingsfilter1_d_541_17506)"><g clip-path="url(#clip1feedLayoutSettings_541_17506)"><rect x="175.256" y="27.1967" width="176.132" height="168.003" rx="2.27031" transform="rotate(5 175.256 27.1967)" fill="white"/><g clip-path="url(#clip2feedLayoutSettings_541_17506)"><rect width="112.002" height="124.647" transform="translate(204.782 52.4471) rotate(5)" fill="#B5E5FF"/><circle cx="199.572" cy="173.295" r="46.9609" transform="rotate(5 199.572 173.295)" fill="#43A6DB"/><circle cx="338.706" cy="133.728" r="64.1417" transform="rotate(5 338.706 133.728)" fill="#86D0F9"/></g><g clip-path="url(#clip3feedLayoutSettings_541_17506)"><rect width="112.002" height="124.647" transform="translate(322.654 62.7597) rotate(5)" fill="#B6DDAD"/><circle cx="317.444" cy="183.608" r="46.9609" transform="rotate(5 317.444 183.608)" fill="#96CE89"/></g><g clip-path="url(#clip4feedLayoutSettings_541_17506)"><rect width="112.002" height="124.647" transform="translate(86.0059 42.0556) rotate(5)" fill="#FCE1D5"/><circle cx="214.51" cy="169.366" r="64.1417" transform="rotate(5 214.51 169.366)" fill="#F9BBA0"/></g></g></g><defs><filter id="feedLayoutSettingsfilter0_d_541_17506" x="36.8814" y="54.8182" width="190.773" height="179.418" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="4.63923"/><feGaussianBlur stdDeviation="4.05932"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.16 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_541_17506"/><feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_541_17506" result="shape"/></filter><filter id="feedLayoutSettingsfilter1_d_541_17506" x="153.701" y="24.2344" width="203.928" height="196.538" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="3.94961"/><feGaussianBlur stdDeviation="3.45591"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.16 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_541_17506"/><feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_541_17506" result="shape"/></filter><clipPath id="clip0feedLayoutSettings_541_17506"><rect width="97.3351" height="82.6682" fill="white" transform="translate(54.8096 75.8527) rotate(-3)"/></clipPath><clipPath id="clip1feedLayoutSettings_541_17506"><rect x="175.256" y="27.1967" width="176.132" height="168.003" rx="2.27031" transform="rotate(5 175.256 27.1967)" fill="white"/></clipPath><clipPath id="clip2feedLayoutSettings_541_17506"><rect width="112.002" height="124.647" fill="white" transform="translate(204.782 52.4471) rotate(5)"/></clipPath><clipPath id="clip3feedLayoutSettings_541_17506"><rect width="112.002" height="124.647" fill="white" transform="translate(322.654 62.7597) rotate(5)"/></clipPath><clipPath id="clip4feedLayoutSettings_541_17506"><rect width="112.002" height="124.647" fill="white" transform="translate(86.0059 42.0556) rotate(5)"/></clipPath></defs></svg>',
				'popupContentBtn' 	=> '<div class="sbi-fb-extpp-lite-btn">' . self::builder_svg_icons()['tag'] . __( 'Instagram Pro users get 50% OFF', 'instagram-feed' ) .'</div>',
				'bullets'       => [
					'heading' => __( 'And get much more!', 'instagram-feed' ),
					'content' => [
						__( 'Display Hashtag & Tagged feeds', 'instagram-feed' ),
						__( 'Powerful visual moderation', 'instagram-feed' ),
						__( 'Comments and Likes', 'instagram-feed' ),
						__( 'Highlight specific posts', 'instagram-feed' ),
						__( 'Multiple layout options', 'instagram-feed' ),
						__( 'Popup photo/video lightbox', 'instagram-feed' ),
						__( 'Instagram Stories', 'instagram-feed' ),
						__( 'Shoppable feeds', 'instagram-feed' ),
						__( 'Pro support', 'instagram-feed' ),
						__( 'Post captions', 'instagram-feed' ),
						__( 'Combine multiple feed types', 'instagram-feed' ),
						__( '30 day money back guarantee', 'instagram-feed' ),
					]
				],
				'buyUrl' 		=> $this->get_extension_popup_dynamic_buy_url( $license_key, 'https://smashballoon.com/instagram-feed/demo/?utm_campaign=instagram-free&utm_source=customizer&utm_medium=feed-layouts' )
			),
			'headerLayout' => array(
				'heading' 		=> $this->get_extension_popup_dynamic_heading('Get Stories, Followers and Advanced Header Options'),
				'description' 	=> __( 'Got stories to tell? We want to help you share them. Display Instagram stories right on your website in a pop-up lightbox to keep your users engaged and on your website for longer.', 'instagram-feed' ),
				'img' 			=> '<svg width="396" height="264" viewBox="0 0 396 264" fill="none" xmlns="http://www.w3.org/2000/svg"><g filter="url(#headerLayoutfilter0_d_543_17792)"><rect x="113.089" y="44.6909" width="219.908" height="169.263" rx="3" transform="rotate(3 113.089 44.6909)" fill="#0096CC"/><circle cx="219.925" cy="94.3318" r="21.9039" transform="rotate(3 219.925 94.3318)" fill="#DCDDE1"/><circle cx="219.925" cy="94.3318" r="23.99" transform="rotate(3 219.925 94.3318)" stroke="url(#paint0_linear_543_17792)" stroke-width="1.04304"/><rect x="148.973" y="123.979" width="134.61" height="11.995" rx="1.33278" transform="rotate(3 148.973 123.979)" fill="#E8E8EB"/><rect x="166.559" y="144.919" width="97.2927" height="11.995" rx="1.33278" transform="rotate(3 166.559 144.919)" fill="#E8E8EB"/><path d="M191.365 180.919C193.326 181.022 194.997 179.517 195.1 177.556C195.203 175.595 193.698 173.923 191.737 173.821C189.776 173.718 188.104 175.223 188.002 177.184C187.899 179.145 189.404 180.816 191.365 180.919ZM191.272 182.694C188.903 182.57 184.111 183.511 183.987 185.871L183.894 187.645L198.091 188.39L198.184 186.615C198.308 184.255 193.641 182.818 191.272 182.694Z" fill="#141B38"/><path d="M211.234 187.974L213.955 188.116L214.643 174.987L211.913 174.844L208.396 177.023L208.268 179.461L211.736 177.362L211.79 177.365L211.234 187.974ZM217.462 188.3L227.006 188.8L227.12 186.617L221.17 186.305L221.176 186.196L224.055 183.828C226.43 181.864 227.239 180.866 227.323 179.274L227.324 179.256C227.441 177.017 225.613 175.334 222.838 175.189C220.063 175.043 217.916 176.673 217.786 179.148L217.782 179.221L220.33 179.354L220.343 179.273C220.442 178.092 221.398 177.257 222.744 177.328C223.909 177.389 224.688 178.196 224.641 179.27L224.64 179.288C224.592 180.198 224.203 180.826 222.428 182.32L217.56 186.417L217.462 188.3ZM232.551 189.091L235.271 189.233L235.959 176.105L233.23 175.962L229.712 178.14L229.585 180.578L233.052 178.479L233.107 178.482L232.551 189.091ZM238.805 189.419L241.462 189.558L241.631 186.328L242.353 185.609L244.937 189.74L248.004 189.901L244.428 184.139L248.333 180.138L245.321 179.98L241.823 183.71L241.768 183.708L242.15 176.429L239.493 176.29L238.805 189.419Z" fill="#141B38"/></g><g filter="url(#headerLayoutfilter1_d_543_17792)"><rect x="88.5157" y="41" width="219.908" height="169.263" rx="3" fill="white"/><circle cx="197.804" cy="84.9817" r="24.6564" stroke="url(#paint1_linear_543_17792)" stroke-width="1.33278"/><rect x="128.499" y="118.301" width="134.61" height="11.995" rx="1.33278" fill="#E8E8EB"/><rect x="147.158" y="138.293" width="97.2927" height="11.995" rx="1.33278" fill="#E8E8EB"/><path d="M173.813 172.945C175.776 172.945 177.367 171.354 177.367 169.391C177.367 167.427 175.776 165.837 173.813 165.837C171.849 165.837 170.259 167.427 170.259 169.391C170.259 171.354 171.849 172.945 173.813 172.945ZM173.813 174.722C171.44 174.722 166.705 175.913 166.705 178.276V180.053H180.921V178.276C180.921 175.913 176.185 174.722 173.813 174.722Z" fill="#141B38"/><path d="M194.025 178.95H196.749V165.803H194.016L190.618 168.163V170.604L193.971 168.327H194.025V178.95ZM200.261 178.95H209.818V176.763H203.86V176.654L206.611 174.139C208.88 172.053 209.636 171.014 209.636 169.42V169.402C209.636 167.161 207.723 165.575 204.944 165.575C202.165 165.575 200.106 167.315 200.106 169.794V169.867H202.657L202.666 169.785C202.703 168.6 203.614 167.716 204.962 167.716C206.128 167.716 206.948 168.482 206.957 169.557V169.575C206.957 170.486 206.602 171.133 204.907 172.718L200.261 177.064V178.95ZM215.371 178.95H218.095V165.803H215.362L211.963 168.163V170.604L215.316 168.327H215.371V178.95ZM221.634 178.95H224.294V175.716L224.977 174.959L227.774 178.95H230.845L226.973 173.383L230.663 169.183H227.647L224.349 173.092H224.294V165.803H221.634V178.95Z" fill="#141B38"/><g clip-path="url(#headerLayoutclip0_543_17792)"><rect x="176" y="63" width="44" height="44" rx="22" fill="#0068A0"/><circle cx="198" cy="80" r="8" fill="#B5E5FF"/><circle cx="198" cy="106" r="15" fill="#B5E5FF"/></g></g><defs><filter id="headerLayoutfilter0_d_543_17792" x="96.2966" y="40.7241" width="244.333" height="196.407" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feMorphology radius="2.9751" operator="erode" in="SourceAlpha" result="effect1_dropShadow_543_17792"/><feOffset dy="3.9668"/><feGaussianBlur stdDeviation="5.45435"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.25 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_543_17792"/><feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_543_17792" result="shape"/></filter><filter id="headerLayoutfilter1_d_543_17792" x="80.5822" y="37.0332" width="235.775" height="185.13" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feMorphology radius="2.9751" operator="erode" in="SourceAlpha" result="effect1_dropShadow_543_17792"/><feOffset dy="3.9668"/><feGaussianBlur stdDeviation="5.45435"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.25 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_543_17792"/><feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_543_17792" result="shape"/></filter><linearGradient id="paint0_linear_543_17792" x1="213.028" y1="162.104" x2="314.086" y2="58.9466" gradientUnits="userSpaceOnUse"><stop stop-color="white"/><stop offset="0.147864" stop-color="#F6640E"/><stop offset="0.443974" stop-color="#BA03A7"/><stop offset="0.733337" stop-color="#6A01B9"/><stop offset="1" stop-color="#6B01B9"/></linearGradient><linearGradient id="paint1_linear_543_17792" x1="190.906" y1="152.753" x2="291.964" y2="49.5964" gradientUnits="userSpaceOnUse"><stop stop-color="white"/><stop offset="0.147864" stop-color="#F6640E"/><stop offset="0.443974" stop-color="#BA03A7"/><stop offset="0.733337" stop-color="#6A01B9"/><stop offset="1" stop-color="#6B01B9"/></linearGradient><clipPath id="headerLayoutclip0_543_17792"><rect x="176" y="63" width="44" height="44" rx="22" fill="white"/></clipPath></defs></svg>',
				'popupContentBtn' 	=> '<div class="sbi-fb-extpp-lite-btn">' . self::builder_svg_icons()['tag'] . __( 'Instagram Pro users get 50% OFF', 'instagram-feed' ) .'</div>',
				'bullets'       => [
					'heading' => __( 'And get much more!', 'instagram-feed' ),
					'content' => [
						__( 'Display Hashtag & Tagged feeds', 'instagram-feed' ),
						__( 'Powerful visual moderation', 'instagram-feed' ),
						__( 'Comments and Likes', 'instagram-feed' ),
						__( 'Highlight specific posts', 'instagram-feed' ),
						__( 'Multiple layout options', 'instagram-feed' ),
						__( 'Popup photo/video lightbox', 'instagram-feed' ),
						__( 'Instagram Stories', 'instagram-feed' ),
						__( 'Shoppable feeds', 'instagram-feed' ),
						__( 'Pro support', 'instagram-feed' ),
						__( 'Post captions', 'instagram-feed' ),
						__( 'Combine multiple feed types', 'instagram-feed' ),
						__( '30 day money back guarantee', 'instagram-feed' ),
					]
				],
				'buyUrl' 		=> $this->get_extension_popup_dynamic_buy_url( $license_key, 'https://smashballoon.com/instagram-feed/demo/?utm_campaign=instagram-free&utm_source=customizer&utm_medium=header' )
			),
			'feedTemplate' => array(
				'heading' 		=> $this->get_extension_popup_dynamic_heading('one-click templates!'),
				'description' 	=> __( 'Quickly create and preview new feeds with pre-configured options based on popular feed types.', 'instagram-feed' ),
				'popupContentBtn' 	=> '<div class="sbc-extpp-lite-btn">' . self::builder_svg_icons()['tag'] . __( 'Lite Feed Users get a 50% OFF', 'instagram-feed' ) .'</div>',
				'img' 			=> '<svg width="396" height="264" viewBox="0 0 396 264" fill="none" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink"><g clip-path="url(#clip0_3202_39054)"><g filter="url(#filter0_dd_3202_39054)"><rect x="175.881" y="35.4428" width="208.613" height="158.297" rx="5.31509" fill="white"/><g clip-path="url(#clip1_3202_39054)"><rect x="181.195" y="40.7579" width="197.983" height="147.667" rx="2.18766" fill="#F3F4F5"/><g filter="url(#filter1_dd_3202_39054)"><g clip-path="url(#clip2_3202_39054)"><rect x="198.697" y="57.1657" width="51.8985" height="82.117" rx="0.7284" fill="white"/><rect x="198.697" y="57.1657" width="51.8985" height="65.9202" fill="url(#pattern0)"/><path d="M200.97 125.923V125.682H202.238V125.923H201.746V127.271H201.461V125.923H200.97ZM202.772 126.573V127.271H202.492V125.682H202.766V126.282H202.78C202.808 126.215 202.851 126.161 202.91 126.123C202.969 126.083 203.044 126.064 203.135 126.064C203.218 126.064 203.29 126.081 203.351 126.116C203.413 126.15 203.461 126.201 203.494 126.268C203.528 126.335 203.545 126.416 203.545 126.512V127.271H203.264V126.556C203.264 126.475 203.244 126.413 203.202 126.369C203.162 126.324 203.104 126.301 203.03 126.301C202.98 126.301 202.936 126.312 202.897 126.334C202.858 126.355 202.827 126.386 202.805 126.426C202.783 126.466 202.772 126.515 202.772 126.573ZM203.854 127.271V126.079H204.134V127.271H203.854ZM203.995 125.91C203.95 125.91 203.912 125.895 203.88 125.866C203.848 125.836 203.832 125.8 203.832 125.758C203.832 125.716 203.848 125.68 203.88 125.65C203.912 125.62 203.95 125.605 203.995 125.605C204.04 125.605 204.078 125.62 204.11 125.65C204.142 125.68 204.158 125.716 204.158 125.758C204.158 125.8 204.142 125.836 204.11 125.866C204.078 125.895 204.04 125.91 203.995 125.91ZM205.387 126.394L205.131 126.422C205.124 126.396 205.111 126.372 205.093 126.349C205.076 126.327 205.052 126.308 205.022 126.294C204.992 126.28 204.955 126.273 204.912 126.273C204.853 126.273 204.804 126.286 204.764 126.311C204.725 126.337 204.705 126.369 204.706 126.41C204.705 126.444 204.718 126.473 204.744 126.494C204.77 126.516 204.814 126.534 204.874 126.548L205.078 126.591C205.19 126.616 205.274 126.654 205.329 126.707C205.384 126.76 205.412 126.829 205.413 126.914C205.412 126.989 205.39 127.055 205.347 127.113C205.304 127.17 205.244 127.214 205.168 127.246C205.091 127.278 205.003 127.294 204.904 127.294C204.758 127.294 204.64 127.264 204.552 127.203C204.463 127.141 204.409 127.056 204.392 126.946L204.666 126.92C204.679 126.973 204.705 127.014 204.746 127.041C204.786 127.069 204.838 127.083 204.903 127.083C204.97 127.083 205.023 127.069 205.064 127.041C205.105 127.014 205.125 126.98 205.125 126.94C205.125 126.906 205.112 126.877 205.085 126.855C205.06 126.833 205.019 126.816 204.964 126.804L204.761 126.761C204.647 126.738 204.562 126.697 204.507 126.641C204.452 126.584 204.425 126.512 204.426 126.425C204.425 126.352 204.445 126.288 204.486 126.234C204.526 126.18 204.583 126.138 204.655 126.109C204.728 126.079 204.812 126.064 204.908 126.064C205.047 126.064 205.157 126.093 205.237 126.153C205.318 126.212 205.368 126.293 205.387 126.394ZM206.476 127.271L206.139 126.079H206.426L206.635 126.917H206.646L206.86 126.079H207.144L207.358 126.913H207.369L207.576 126.079H207.863L207.525 127.271H207.233L207.009 126.466H206.993L206.77 127.271H206.476ZM208.574 127.294C208.455 127.294 208.352 127.27 208.265 127.22C208.178 127.17 208.112 127.099 208.065 127.007C208.019 126.915 207.996 126.807 207.996 126.682C207.996 126.56 208.019 126.452 208.065 126.359C208.112 126.266 208.178 126.194 208.262 126.142C208.347 126.09 208.446 126.064 208.56 126.064C208.633 126.064 208.702 126.076 208.768 126.099C208.833 126.123 208.891 126.159 208.941 126.208C208.992 126.257 209.032 126.32 209.061 126.396C209.09 126.471 209.104 126.561 209.104 126.666V126.752H208.127V126.563H208.835C208.835 126.509 208.823 126.461 208.8 126.419C208.777 126.377 208.746 126.343 208.705 126.319C208.664 126.295 208.617 126.283 208.564 126.283C208.506 126.283 208.456 126.296 208.412 126.324C208.369 126.352 208.335 126.388 208.311 126.433C208.287 126.478 208.275 126.526 208.274 126.58V126.745C208.274 126.814 208.287 126.874 208.312 126.924C208.338 126.973 208.373 127.01 208.418 127.037C208.464 127.063 208.517 127.076 208.578 127.076C208.619 127.076 208.656 127.07 208.689 127.059C208.722 127.047 208.751 127.029 208.775 127.007C208.8 126.984 208.818 126.956 208.831 126.922L209.093 126.951C209.076 127.021 209.045 127.081 208.998 127.133C208.952 127.184 208.893 127.224 208.821 127.253C208.749 127.28 208.667 127.294 208.574 127.294ZM209.892 127.294C209.772 127.294 209.669 127.27 209.582 127.22C209.496 127.17 209.429 127.099 209.383 127.007C209.336 126.915 209.313 126.807 209.313 126.682C209.313 126.56 209.336 126.452 209.383 126.359C209.43 126.266 209.495 126.194 209.58 126.142C209.664 126.09 209.763 126.064 209.877 126.064C209.95 126.064 210.02 126.076 210.085 126.099C210.151 126.123 210.208 126.159 210.259 126.208C210.309 126.257 210.349 126.32 210.378 126.396C210.407 126.471 210.422 126.561 210.422 126.666V126.752H209.445V126.563H210.152C210.152 126.509 210.14 126.461 210.117 126.419C210.095 126.377 210.063 126.343 210.022 126.319C209.982 126.295 209.935 126.283 209.881 126.283C209.823 126.283 209.773 126.296 209.729 126.324C209.686 126.352 209.652 126.388 209.628 126.433C209.604 126.478 209.592 126.526 209.591 126.58V126.745C209.591 126.814 209.604 126.874 209.629 126.924C209.655 126.973 209.69 127.01 209.736 127.037C209.781 127.063 209.834 127.076 209.895 127.076C209.936 127.076 209.973 127.07 210.006 127.059C210.04 127.047 210.068 127.029 210.093 127.007C210.117 126.984 210.135 126.956 210.148 126.922L210.41 126.951C210.393 127.021 210.362 127.081 210.315 127.133C210.269 127.184 210.21 127.224 210.138 127.253C210.066 127.28 209.984 127.294 209.892 127.294ZM210.937 126.897L210.936 126.558H210.981L211.41 126.079H211.738L211.211 126.666H211.153L210.937 126.897ZM210.681 127.271V125.682H210.962V127.271H210.681ZM211.429 127.271L211.041 126.729L211.231 126.531L211.765 127.271H211.429ZM212.511 127.271V126.079H212.792V127.271H212.511ZM212.652 125.91C212.608 125.91 212.569 125.895 212.537 125.866C212.505 125.836 212.489 125.8 212.489 125.758C212.489 125.716 212.505 125.68 212.537 125.65C212.569 125.62 212.608 125.605 212.652 125.605C212.697 125.605 212.735 125.62 212.767 125.65C212.799 125.68 212.815 125.716 212.815 125.758C212.815 125.8 212.799 125.836 212.767 125.866C212.735 125.895 212.697 125.91 212.652 125.91ZM213.384 126.573V127.271H213.103V126.079H213.371V126.282H213.385C213.412 126.215 213.456 126.162 213.516 126.123C213.577 126.083 213.651 126.064 213.74 126.064C213.823 126.064 213.894 126.081 213.955 126.116C214.017 126.152 214.065 126.203 214.098 126.269C214.132 126.336 214.149 126.417 214.149 126.512V127.271H213.868V126.556C213.868 126.476 213.847 126.414 213.806 126.369C213.765 126.324 213.708 126.301 213.636 126.301C213.587 126.301 213.543 126.312 213.505 126.334C213.467 126.355 213.437 126.386 213.415 126.426C213.394 126.466 213.384 126.515 213.384 126.573ZM215.236 127.271H214.929L215.489 125.682H215.844L216.404 127.271H216.097L215.673 126.008H215.66L215.236 127.271ZM215.247 126.648H216.085V126.879H215.247V126.648ZM216.623 127.271V126.079H216.895V126.278H216.908C216.93 126.209 216.967 126.156 217.02 126.119C217.073 126.081 217.134 126.062 217.202 126.062C217.217 126.062 217.235 126.063 217.254 126.064C217.274 126.066 217.29 126.067 217.303 126.07V126.328C217.291 126.324 217.272 126.321 217.246 126.317C217.221 126.314 217.196 126.312 217.172 126.312C217.121 126.312 217.075 126.323 217.034 126.345C216.994 126.367 216.962 126.397 216.939 126.436C216.916 126.475 216.904 126.52 216.904 126.57V127.271H216.623ZM217.982 127.294C217.863 127.294 217.761 127.268 217.676 127.216C217.591 127.164 217.525 127.092 217.479 127C217.434 126.907 217.411 126.8 217.411 126.68C217.411 126.559 217.434 126.452 217.481 126.359C217.527 126.266 217.593 126.194 217.678 126.142C217.763 126.09 217.864 126.064 217.981 126.064C218.077 126.064 218.163 126.082 218.237 126.117C218.312 126.152 218.372 126.202 218.417 126.267C218.461 126.331 218.487 126.406 218.493 126.492H218.224C218.213 126.435 218.188 126.387 218.147 126.348C218.106 126.31 218.052 126.29 217.984 126.29C217.927 126.29 217.877 126.306 217.833 126.337C217.79 126.367 217.756 126.411 217.732 126.469C217.708 126.526 217.696 126.595 217.696 126.675C217.696 126.756 217.708 126.826 217.732 126.885C217.755 126.943 217.789 126.987 217.832 127.019C217.875 127.05 217.926 127.066 217.984 127.066C218.026 127.066 218.063 127.058 218.095 127.042C218.129 127.026 218.156 127.003 218.178 126.973C218.201 126.943 218.216 126.907 218.224 126.864H218.493C218.486 126.948 218.461 127.023 218.418 127.088C218.375 127.153 218.317 127.203 218.243 127.24C218.169 127.276 218.082 127.294 217.982 127.294ZM219.026 126.573V127.271H218.745V125.682H219.02V126.282H219.034C219.062 126.215 219.105 126.161 219.164 126.123C219.223 126.083 219.298 126.064 219.389 126.064C219.471 126.064 219.544 126.081 219.605 126.116C219.667 126.15 219.714 126.201 219.748 126.268C219.782 126.335 219.799 126.416 219.799 126.512V127.271H219.518V126.556C219.518 126.475 219.497 126.413 219.456 126.369C219.415 126.324 219.358 126.301 219.284 126.301C219.234 126.301 219.19 126.312 219.15 126.334C219.112 126.355 219.081 126.386 219.059 126.426C219.037 126.466 219.026 126.515 219.026 126.573ZM220.107 127.271V126.079H220.388V127.271H220.107ZM220.249 125.91C220.204 125.91 220.166 125.895 220.134 125.866C220.102 125.836 220.086 125.8 220.086 125.758C220.086 125.716 220.102 125.68 220.134 125.65C220.166 125.62 220.204 125.605 220.249 125.605C220.294 125.605 220.332 125.62 220.363 125.65C220.395 125.68 220.411 125.716 220.411 125.758C220.411 125.8 220.395 125.836 220.363 125.866C220.332 125.895 220.294 125.91 220.249 125.91ZM221.294 126.079V126.296H220.609V126.079H221.294ZM220.778 125.794H221.059V126.913C221.059 126.95 221.065 126.979 221.076 127C221.088 127.019 221.103 127.033 221.123 127.04C221.142 127.047 221.163 127.051 221.186 127.051C221.204 127.051 221.22 127.049 221.234 127.047C221.249 127.044 221.261 127.042 221.268 127.04L221.316 127.26C221.301 127.265 221.279 127.27 221.251 127.277C221.224 127.283 221.19 127.286 221.151 127.287C221.08 127.29 221.017 127.279 220.96 127.256C220.904 127.232 220.859 127.195 220.826 127.145C220.794 127.096 220.778 127.034 220.778 126.959V125.794ZM222.066 127.294C221.946 127.294 221.843 127.27 221.756 127.22C221.67 127.17 221.603 127.099 221.557 127.007C221.51 126.915 221.487 126.807 221.487 126.682C221.487 126.56 221.51 126.452 221.557 126.359C221.604 126.266 221.669 126.194 221.754 126.142C221.838 126.09 221.937 126.064 222.051 126.064C222.124 126.064 222.194 126.076 222.259 126.099C222.325 126.123 222.383 126.159 222.433 126.208C222.483 126.257 222.523 126.32 222.552 126.396C222.581 126.471 222.596 126.561 222.596 126.666V126.752H221.619V126.563H222.326C222.326 126.509 222.314 126.461 222.292 126.419C222.269 126.377 222.237 126.343 222.196 126.319C222.156 126.295 222.109 126.283 222.055 126.283C221.997 126.283 221.947 126.296 221.904 126.324C221.86 126.352 221.826 126.388 221.802 126.433C221.778 126.478 221.766 126.526 221.765 126.58V126.745C221.765 126.814 221.778 126.874 221.803 126.924C221.829 126.973 221.864 127.01 221.91 127.037C221.955 127.063 222.009 127.076 222.07 127.076C222.11 127.076 222.147 127.07 222.181 127.059C222.214 127.047 222.242 127.029 222.267 127.007C222.291 126.984 222.309 126.956 222.322 126.922L222.584 126.951C222.568 127.021 222.536 127.081 222.489 127.133C222.443 127.184 222.384 127.224 222.313 127.253C222.241 127.28 222.158 127.294 222.066 127.294ZM223.375 127.294C223.256 127.294 223.154 127.268 223.069 127.216C222.984 127.164 222.918 127.092 222.872 127C222.827 126.907 222.804 126.8 222.804 126.68C222.804 126.559 222.827 126.452 222.874 126.359C222.92 126.266 222.986 126.194 223.071 126.142C223.156 126.09 223.257 126.064 223.374 126.064C223.47 126.064 223.556 126.082 223.63 126.117C223.705 126.152 223.765 126.202 223.81 126.267C223.854 126.331 223.88 126.406 223.886 126.492H223.617C223.606 126.435 223.581 126.387 223.54 126.348C223.499 126.31 223.445 126.29 223.377 126.29C223.32 126.29 223.27 126.306 223.226 126.337C223.183 126.367 223.149 126.411 223.125 126.469C223.101 126.526 223.089 126.595 223.089 126.675C223.089 126.756 223.101 126.826 223.125 126.885C223.148 126.943 223.182 126.987 223.225 127.019C223.268 127.05 223.319 127.066 223.377 127.066C223.419 127.066 223.456 127.058 223.488 127.042C223.522 127.026 223.549 127.003 223.571 126.973C223.594 126.943 223.609 126.907 223.617 126.864H223.886C223.879 126.948 223.854 127.023 223.811 127.088C223.768 127.153 223.71 127.203 223.636 127.24C223.562 127.276 223.475 127.294 223.375 127.294ZM224.733 126.079V126.296H224.048V126.079H224.733ZM224.217 125.794H224.498V126.913C224.498 126.95 224.504 126.979 224.515 127C224.527 127.019 224.543 127.033 224.562 127.04C224.581 127.047 224.602 127.051 224.626 127.051C224.643 127.051 224.659 127.049 224.674 127.047C224.689 127.044 224.7 127.042 224.708 127.04L224.755 127.26C224.74 127.265 224.719 127.27 224.691 127.277C224.663 127.283 224.63 127.286 224.59 127.287C224.52 127.29 224.456 127.279 224.4 127.256C224.343 127.232 224.299 127.195 224.266 127.145C224.233 127.096 224.217 127.034 224.217 126.959V125.794ZM225.75 126.77V126.079H226.031V127.271H225.758V127.059H225.746C225.719 127.126 225.675 127.181 225.613 127.223C225.552 127.265 225.477 127.287 225.388 127.287C225.309 127.287 225.24 127.269 225.18 127.235C225.121 127.199 225.074 127.149 225.041 127.082C225.007 127.015 224.99 126.933 224.99 126.838V126.079H225.271V126.795C225.271 126.87 225.292 126.93 225.333 126.975C225.375 127.019 225.429 127.041 225.496 127.041C225.538 127.041 225.578 127.031 225.616 127.011C225.655 126.991 225.687 126.961 225.712 126.921C225.737 126.881 225.75 126.83 225.75 126.77ZM226.342 127.271V126.079H226.614V126.278H226.626C226.648 126.209 226.685 126.156 226.738 126.119C226.791 126.081 226.852 126.062 226.92 126.062C226.936 126.062 226.953 126.063 226.972 126.064C226.992 126.066 227.008 126.067 227.021 126.07V126.328C227.009 126.324 226.991 126.321 226.965 126.317C226.939 126.314 226.915 126.312 226.891 126.312C226.84 126.312 226.794 126.323 226.753 126.345C226.713 126.367 226.681 126.397 226.657 126.436C226.634 126.475 226.622 126.52 226.622 126.57V127.271H226.342ZM227.708 127.294C227.589 127.294 227.486 127.27 227.399 127.22C227.312 127.17 227.246 127.099 227.199 127.007C227.153 126.915 227.13 126.807 227.13 126.682C227.13 126.56 227.153 126.452 227.199 126.359C227.246 126.266 227.312 126.194 227.396 126.142C227.481 126.09 227.58 126.064 227.694 126.064C227.767 126.064 227.836 126.076 227.902 126.099C227.967 126.123 228.025 126.159 228.075 126.208C228.126 126.257 228.166 126.32 228.195 126.396C228.224 126.471 228.238 126.561 228.238 126.666V126.752H227.261V126.563H227.969C227.969 126.509 227.957 126.461 227.934 126.419C227.911 126.377 227.88 126.343 227.839 126.319C227.798 126.295 227.751 126.283 227.698 126.283C227.64 126.283 227.59 126.296 227.546 126.324C227.503 126.352 227.469 126.388 227.445 126.433C227.421 126.478 227.409 126.526 227.408 126.58V126.745C227.408 126.814 227.421 126.874 227.446 126.924C227.471 126.973 227.507 127.01 227.552 127.037C227.598 127.063 227.651 127.076 227.712 127.076C227.753 127.076 227.79 127.07 227.823 127.059C227.856 127.047 227.885 127.029 227.909 127.007C227.934 126.984 227.952 126.956 227.965 126.922L228.227 126.951C228.21 127.021 228.179 127.081 228.132 127.133C228.086 127.184 228.027 127.224 227.955 127.253C227.883 127.28 227.801 127.294 227.708 127.294ZM229.504 127.292C229.41 127.292 229.327 127.268 229.253 127.22C229.179 127.172 229.12 127.102 229.077 127.01C229.034 126.919 229.013 126.808 229.013 126.677C229.013 126.544 229.035 126.433 229.078 126.342C229.122 126.25 229.181 126.181 229.256 126.134C229.33 126.087 229.413 126.064 229.505 126.064C229.575 126.064 229.632 126.076 229.677 126.099C229.722 126.123 229.758 126.151 229.784 126.184C229.811 126.217 229.831 126.247 229.846 126.276H229.857V125.682H230.139V127.271H229.863V127.083H229.846C229.831 127.112 229.81 127.143 229.783 127.176C229.755 127.208 229.719 127.235 229.674 127.258C229.629 127.281 229.572 127.292 229.504 127.292ZM229.583 127.062C229.642 127.062 229.693 127.046 229.735 127.014C229.777 126.981 229.808 126.936 229.83 126.878C229.852 126.82 229.863 126.752 229.863 126.675C229.863 126.598 229.852 126.531 229.83 126.474C229.809 126.417 229.777 126.373 229.735 126.342C229.694 126.31 229.643 126.294 229.583 126.294C229.52 126.294 229.468 126.31 229.426 126.343C229.384 126.376 229.352 126.421 229.331 126.478C229.31 126.536 229.299 126.601 229.299 126.675C229.299 126.75 229.31 126.816 229.331 126.875C229.353 126.933 229.385 126.978 229.427 127.012C229.469 127.045 229.521 127.062 229.583 127.062ZM230.803 127.295C230.728 127.295 230.66 127.282 230.599 127.255C230.539 127.227 230.491 127.187 230.456 127.134C230.422 127.081 230.404 127.015 230.404 126.937C230.404 126.869 230.417 126.814 230.441 126.77C230.466 126.726 230.5 126.691 230.543 126.664C230.586 126.638 230.634 126.618 230.688 126.605C230.743 126.591 230.799 126.581 230.857 126.574C230.926 126.567 230.983 126.561 231.027 126.555C231.07 126.549 231.102 126.539 231.121 126.527C231.141 126.514 231.152 126.494 231.152 126.467V126.463C231.152 126.404 231.134 126.359 231.1 126.327C231.065 126.295 231.015 126.279 230.95 126.279C230.881 126.279 230.826 126.294 230.786 126.324C230.746 126.354 230.719 126.389 230.705 126.43L230.443 126.393C230.464 126.32 230.498 126.26 230.545 126.211C230.593 126.162 230.651 126.125 230.72 126.101C230.789 126.076 230.865 126.064 230.948 126.064C231.006 126.064 231.063 126.07 231.12 126.084C231.177 126.097 231.229 126.12 231.276 126.151C231.323 126.181 231.361 126.223 231.389 126.276C231.418 126.328 231.432 126.394 231.432 126.473V127.271H231.162V127.107H231.153C231.136 127.141 231.112 127.172 231.081 127.201C231.05 127.229 231.012 127.252 230.965 127.27C230.919 127.287 230.865 127.295 230.803 127.295ZM230.876 127.089C230.932 127.089 230.981 127.078 231.023 127.055C231.064 127.033 231.096 127.003 231.118 126.965C231.141 126.928 231.152 126.888 231.152 126.844V126.703C231.143 126.71 231.128 126.717 231.107 126.723C231.087 126.73 231.063 126.735 231.037 126.74C231.012 126.744 230.986 126.748 230.961 126.752C230.935 126.756 230.913 126.759 230.895 126.761C230.853 126.767 230.815 126.776 230.782 126.789C230.749 126.802 230.723 126.82 230.704 126.844C230.685 126.866 230.675 126.896 230.675 126.932C230.675 126.984 230.694 127.023 230.732 127.049C230.769 127.076 230.818 127.089 230.876 127.089ZM231.737 127.271V126.079H232.018V127.271H231.737ZM231.878 125.91C231.833 125.91 231.795 125.895 231.763 125.866C231.731 125.836 231.715 125.8 231.715 125.758C231.715 125.716 231.731 125.68 231.763 125.65C231.795 125.62 231.833 125.605 231.878 125.605C231.923 125.605 231.961 125.62 231.993 125.65C232.025 125.68 232.041 125.716 232.041 125.758C232.041 125.8 232.025 125.836 231.993 125.866C231.961 125.895 231.923 125.91 231.878 125.91ZM232.609 125.682V127.271H232.328V125.682H232.609ZM233.095 127.718C233.056 127.718 233.021 127.715 232.988 127.709C232.956 127.703 232.931 127.696 232.911 127.689L232.977 127.47C233.017 127.482 233.054 127.487 233.086 127.487C233.118 127.486 233.146 127.476 233.171 127.457C233.195 127.437 233.216 127.405 233.233 127.36L233.258 127.296L232.825 126.079H233.123L233.398 126.979H233.41L233.686 126.079H233.985L233.507 127.415C233.485 127.479 233.456 127.533 233.419 127.578C233.382 127.623 233.337 127.658 233.284 127.682C233.231 127.706 233.168 127.718 233.095 127.718ZM235.011 127.271L234.674 126.079H234.96L235.17 126.917H235.181L235.395 126.079H235.678L235.892 126.913H235.904L236.11 126.079H236.397L236.06 127.271H235.767L235.544 126.466H235.528L235.304 127.271H235.011ZM237.109 127.294C236.989 127.294 236.886 127.27 236.799 127.22C236.713 127.17 236.646 127.099 236.6 127.007C236.553 126.915 236.53 126.807 236.53 126.682C236.53 126.56 236.553 126.452 236.6 126.359C236.647 126.266 236.713 126.194 236.797 126.142C236.881 126.09 236.98 126.064 237.094 126.064C237.168 126.064 237.237 126.076 237.302 126.099C237.368 126.123 237.426 126.159 237.476 126.208C237.527 126.257 237.567 126.32 237.596 126.396C237.624 126.471 237.639 126.561 237.639 126.666V126.752H236.662V126.563H237.37C237.369 126.509 237.358 126.461 237.335 126.419C237.312 126.377 237.28 126.343 237.239 126.319C237.199 126.295 237.152 126.283 237.098 126.283C237.041 126.283 236.99 126.296 236.947 126.324C236.903 126.352 236.869 126.388 236.845 126.433C236.821 126.478 236.809 126.526 236.809 126.58V126.745C236.809 126.814 236.821 126.874 236.847 126.924C236.872 126.973 236.907 127.01 236.953 127.037C236.999 127.063 237.052 127.076 237.113 127.076C237.154 127.076 237.191 127.07 237.224 127.059C237.257 127.047 237.286 127.029 237.31 127.007C237.334 126.984 237.353 126.956 237.365 126.922L237.627 126.951C237.611 127.021 237.579 127.081 237.533 127.133C237.487 127.184 237.428 127.224 237.356 127.253C237.284 127.28 237.202 127.294 237.109 127.294ZM238.805 127.295C238.73 127.295 238.662 127.282 238.601 127.255C238.541 127.227 238.494 127.187 238.458 127.134C238.424 127.081 238.406 127.015 238.406 126.937C238.406 126.869 238.419 126.814 238.444 126.77C238.469 126.726 238.502 126.691 238.545 126.664C238.588 126.638 238.637 126.618 238.691 126.605C238.745 126.591 238.801 126.581 238.859 126.574C238.929 126.567 238.985 126.561 239.029 126.555C239.072 126.549 239.104 126.539 239.124 126.527C239.144 126.514 239.154 126.494 239.154 126.467V126.463C239.154 126.404 239.136 126.359 239.102 126.327C239.067 126.295 239.017 126.279 238.952 126.279C238.883 126.279 238.829 126.294 238.788 126.324C238.748 126.354 238.722 126.389 238.708 126.43L238.445 126.393C238.466 126.32 238.5 126.26 238.548 126.211C238.595 126.162 238.654 126.125 238.722 126.101C238.791 126.076 238.867 126.064 238.95 126.064C239.008 126.064 239.065 126.07 239.122 126.084C239.179 126.097 239.231 126.12 239.278 126.151C239.325 126.181 239.363 126.223 239.391 126.276C239.42 126.328 239.435 126.394 239.435 126.473V127.271H239.165V127.107H239.155C239.138 127.141 239.114 127.172 239.083 127.201C239.053 127.229 239.014 127.252 238.968 127.27C238.921 127.287 238.867 127.295 238.805 127.295ZM238.878 127.089C238.935 127.089 238.984 127.078 239.025 127.055C239.066 127.033 239.098 127.003 239.12 126.965C239.143 126.928 239.155 126.888 239.155 126.844V126.703C239.146 126.71 239.131 126.717 239.11 126.723C239.089 126.73 239.066 126.735 239.04 126.74C239.014 126.744 238.988 126.748 238.963 126.752C238.938 126.756 238.916 126.759 238.897 126.761C238.855 126.767 238.818 126.776 238.784 126.789C238.751 126.802 238.725 126.82 238.706 126.844C238.687 126.866 238.677 126.896 238.677 126.932C238.677 126.984 238.696 127.023 238.734 127.049C238.772 127.076 238.82 127.089 238.878 127.089ZM239.739 127.271V126.079H240.011V126.278H240.024C240.046 126.209 240.083 126.156 240.136 126.119C240.189 126.081 240.25 126.062 240.318 126.062C240.333 126.062 240.351 126.063 240.37 126.064C240.39 126.066 240.406 126.067 240.419 126.07V126.328C240.407 126.324 240.388 126.321 240.362 126.317C240.337 126.314 240.312 126.312 240.288 126.312C240.237 126.312 240.191 126.323 240.15 126.345C240.11 126.367 240.078 126.397 240.055 126.436C240.032 126.475 240.02 126.52 240.02 126.57V127.271H239.739ZM241.106 127.294C240.986 127.294 240.883 127.27 240.796 127.22C240.71 127.17 240.643 127.099 240.597 127.007C240.55 126.915 240.527 126.807 240.527 126.682C240.527 126.56 240.55 126.452 240.597 126.359C240.644 126.266 240.71 126.194 240.794 126.142C240.878 126.09 240.977 126.064 241.091 126.064C241.165 126.064 241.234 126.076 241.299 126.099C241.365 126.123 241.423 126.159 241.473 126.208C241.524 126.257 241.563 126.32 241.592 126.396C241.621 126.471 241.636 126.561 241.636 126.666V126.752H240.659V126.563H241.367C241.366 126.509 241.354 126.461 241.332 126.419C241.309 126.377 241.277 126.343 241.236 126.319C241.196 126.295 241.149 126.283 241.095 126.283C241.038 126.283 240.987 126.296 240.944 126.324C240.9 126.352 240.866 126.388 240.842 126.433C240.818 126.478 240.806 126.526 240.806 126.58V126.745C240.806 126.814 240.818 126.874 240.844 126.924C240.869 126.973 240.904 127.01 240.95 127.037C240.995 127.063 241.049 127.076 241.11 127.076C241.151 127.076 241.188 127.07 241.221 127.059C241.254 127.047 241.283 127.029 241.307 127.007C241.331 126.984 241.35 126.956 241.362 126.922L241.624 126.951C241.608 127.021 241.576 127.081 241.53 127.133C241.483 127.184 241.425 127.224 241.353 127.253C241.281 127.28 241.198 127.294 241.106 127.294ZM201.631 129.079V129.296H200.926V129.079H201.631ZM201.102 130.271V128.967C201.102 128.887 201.119 128.82 201.152 128.766C201.186 128.713 201.231 128.673 201.287 128.647C201.343 128.621 201.406 128.607 201.475 128.607C201.524 128.607 201.567 128.611 201.604 128.619C201.642 128.627 201.67 128.634 201.688 128.64L201.632 128.857C201.621 128.854 201.606 128.85 201.587 128.846C201.569 128.842 201.549 128.84 201.527 128.84C201.475 128.84 201.438 128.853 201.416 128.878C201.395 128.903 201.384 128.939 201.384 128.985V130.271H201.102ZM202.356 130.294C202.236 130.294 202.133 130.27 202.046 130.22C201.96 130.17 201.893 130.099 201.847 130.007C201.8 129.915 201.777 129.807 201.777 129.682C201.777 129.56 201.8 129.452 201.847 129.359C201.894 129.266 201.959 129.194 202.044 129.142C202.128 129.09 202.227 129.064 202.341 129.064C202.414 129.064 202.484 129.076 202.549 129.099C202.615 129.123 202.672 129.159 202.723 129.208C202.773 129.257 202.813 129.32 202.842 129.396C202.871 129.471 202.886 129.561 202.886 129.666V129.752H201.909V129.563H202.616C202.616 129.509 202.604 129.461 202.581 129.419C202.559 129.377 202.527 129.343 202.486 129.319C202.446 129.295 202.399 129.283 202.345 129.283C202.287 129.283 202.237 129.296 202.193 129.324C202.15 129.352 202.116 129.388 202.092 129.433C202.068 129.478 202.056 129.526 202.055 129.58V129.745C202.055 129.814 202.068 129.874 202.093 129.924C202.119 129.973 202.154 130.01 202.2 130.037C202.245 130.063 202.298 130.076 202.359 130.076C202.4 130.076 202.437 130.07 202.47 130.059C202.504 130.047 202.532 130.029 202.557 130.007C202.581 129.984 202.599 129.956 202.612 129.922L202.874 129.951C202.857 130.021 202.826 130.081 202.779 130.133C202.733 130.184 202.674 130.224 202.602 130.253C202.53 130.28 202.448 130.294 202.356 130.294ZM203.488 130.295C203.413 130.295 203.345 130.282 203.284 130.255C203.224 130.227 203.176 130.187 203.141 130.134C203.107 130.081 203.089 130.015 203.089 129.937C203.089 129.869 203.102 129.814 203.126 129.77C203.151 129.726 203.185 129.691 203.228 129.664C203.271 129.638 203.319 129.618 203.373 129.605C203.428 129.591 203.484 129.581 203.542 129.574C203.611 129.567 203.668 129.561 203.712 129.555C203.755 129.549 203.787 129.539 203.806 129.527C203.826 129.514 203.836 129.494 203.836 129.467V129.463C203.836 129.404 203.819 129.359 203.785 129.327C203.75 129.295 203.7 129.279 203.635 129.279C203.566 129.279 203.511 129.294 203.471 129.324C203.431 129.354 203.404 129.389 203.39 129.43L203.128 129.393C203.149 129.32 203.183 129.26 203.23 129.211C203.278 129.162 203.336 129.125 203.405 129.101C203.474 129.076 203.55 129.064 203.633 129.064C203.691 129.064 203.748 129.07 203.805 129.084C203.862 129.097 203.914 129.12 203.961 129.151C204.008 129.181 204.045 129.223 204.074 129.276C204.103 129.328 204.117 129.394 204.117 129.473L204.117 130.271H203.847V130.107H203.838C203.821 130.141 203.797 130.172 203.766 130.201C203.735 130.229 203.697 130.252 203.65 130.27C203.604 130.287 203.55 130.295 203.488 130.295ZM203.561 130.089C203.617 130.089 203.666 130.078 203.708 130.055C203.749 130.033 203.781 130.003 203.803 129.965C203.826 129.928 203.837 129.888 203.837 129.844V129.703C203.828 129.71 203.813 129.717 203.792 129.723C203.772 129.73 203.748 129.735 203.722 129.74C203.697 129.744 203.671 129.748 203.646 129.752C203.62 129.756 203.598 129.759 203.58 129.761C203.538 129.767 203.5 129.776 203.467 129.789C203.434 129.802 203.408 129.82 203.389 129.844C203.37 129.866 203.36 129.896 203.36 129.932C203.36 129.984 203.379 130.023 203.417 130.049C203.454 130.076 203.503 130.089 203.561 130.089ZM205.017 129.079V129.296H204.332V129.079H205.017ZM204.501 128.794H204.782V129.913C204.782 129.95 204.787 129.979 204.799 130C204.811 130.019 204.826 130.033 204.845 130.04C204.865 130.047 204.886 130.051 204.909 130.051C204.927 130.051 204.943 130.049 204.957 130.047C204.972 130.044 204.984 130.042 204.991 130.04L205.039 130.26C205.024 130.265 205.002 130.27 204.974 130.277C204.947 130.283 204.913 130.286 204.873 130.287C204.803 130.29 204.74 130.279 204.683 130.256C204.627 130.232 204.582 130.195 204.549 130.145C204.516 130.096 204.5 130.034 204.501 129.959V128.794ZM206.033 129.77V129.079H206.314V130.271H206.042V130.059H206.03C206.003 130.126 205.958 130.181 205.897 130.223C205.836 130.265 205.76 130.287 205.671 130.287C205.593 130.287 205.524 130.269 205.464 130.235C205.404 130.199 205.358 130.149 205.324 130.082C205.291 130.015 205.274 129.933 205.274 129.838V129.079H205.555V129.795C205.555 129.87 205.575 129.93 205.617 129.975C205.658 130.019 205.712 130.041 205.78 130.041C205.821 130.041 205.861 130.031 205.9 130.011C205.939 129.991 205.971 129.961 205.995 129.921C206.021 129.881 206.033 129.83 206.033 129.77ZM206.625 130.271V129.079H206.897V129.278H206.91C206.932 129.209 206.969 129.156 207.022 129.119C207.075 129.081 207.136 129.062 207.204 129.062C207.219 129.062 207.237 129.063 207.256 129.064C207.276 129.066 207.292 129.067 207.305 129.07V129.328C207.293 129.324 207.274 129.321 207.248 129.317C207.223 129.314 207.198 129.312 207.174 129.312C207.123 129.312 207.077 129.323 207.036 129.345C206.996 129.367 206.964 129.397 206.941 129.436C206.918 129.475 206.906 129.52 206.906 129.57V130.271H206.625ZM207.513 130.271V129.079H207.794V130.271H207.513ZM207.654 128.91C207.61 128.91 207.572 128.895 207.54 128.866C207.508 128.836 207.492 128.8 207.492 128.758C207.492 128.716 207.508 128.68 207.54 128.65C207.572 128.62 207.61 128.605 207.654 128.605C207.7 128.605 207.738 128.62 207.769 128.65C207.801 128.68 207.817 128.716 207.817 128.758C207.817 128.8 207.801 128.836 207.769 128.866C207.738 128.895 207.7 128.91 207.654 128.91ZM208.386 129.573V130.271H208.105V129.079H208.373V129.282H208.387C208.415 129.215 208.458 129.162 208.518 129.123C208.579 129.083 208.654 129.064 208.743 129.064C208.825 129.064 208.897 129.081 208.958 129.116C209.019 129.152 209.067 129.203 209.1 129.269C209.135 129.336 209.151 129.417 209.151 129.512V130.271H208.87V129.556C208.87 129.476 208.849 129.414 208.808 129.369C208.767 129.324 208.71 129.301 208.638 129.301C208.589 129.301 208.545 129.312 208.507 129.334C208.469 129.355 208.439 129.386 208.418 129.426C208.396 129.466 208.386 129.515 208.386 129.573ZM209.977 130.743C209.876 130.743 209.789 130.729 209.717 130.702C209.644 130.675 209.586 130.639 209.542 130.593C209.498 130.548 209.468 130.497 209.451 130.442L209.704 130.381C209.715 130.404 209.732 130.427 209.753 130.45C209.775 130.473 209.804 130.492 209.841 130.507C209.878 130.523 209.925 130.53 209.981 130.53C210.061 130.53 210.127 130.511 210.179 130.472C210.231 130.434 210.258 130.371 210.258 130.283V130.057H210.244C210.229 130.086 210.208 130.116 210.18 130.146C210.153 130.177 210.116 130.202 210.071 130.223C210.026 130.244 209.969 130.254 209.901 130.254C209.809 130.254 209.726 130.233 209.652 130.19C209.578 130.146 209.519 130.082 209.475 129.996C209.431 129.909 209.409 129.801 209.409 129.671C209.409 129.54 209.431 129.43 209.475 129.34C209.519 129.249 209.578 129.181 209.652 129.134C209.727 129.087 209.81 129.064 209.901 129.064C209.971 129.064 210.029 129.076 210.074 129.099C210.119 129.123 210.155 129.151 210.182 129.184C210.209 129.217 210.23 129.247 210.244 129.276H210.259V129.079H210.536V130.291C210.536 130.392 210.512 130.477 210.463 130.544C210.415 130.61 210.348 130.66 210.264 130.693C210.18 130.726 210.084 130.743 209.977 130.743ZM209.979 130.034C210.039 130.034 210.089 130.019 210.131 129.99C210.173 129.961 210.205 129.92 210.227 129.865C210.248 129.811 210.259 129.746 210.259 129.67C210.259 129.595 210.248 129.529 210.227 129.473C210.205 129.416 210.174 129.373 210.132 129.342C210.091 129.31 210.04 129.294 209.979 129.294C209.916 129.294 209.864 129.31 209.822 129.343C209.78 129.376 209.749 129.42 209.728 129.477C209.706 129.534 209.696 129.598 209.696 129.67C209.696 129.743 209.706 129.807 209.728 129.861C209.749 129.916 209.781 129.958 209.823 129.989C209.865 130.019 209.917 130.034 209.979 130.034ZM211.753 130.295C211.677 130.295 211.609 130.282 211.549 130.255C211.489 130.227 211.441 130.187 211.406 130.134C211.371 130.081 211.354 130.015 211.354 129.937C211.354 129.869 211.366 129.814 211.391 129.77C211.416 129.726 211.45 129.691 211.493 129.664C211.536 129.638 211.584 129.618 211.638 129.605C211.692 129.591 211.748 129.581 211.806 129.574C211.876 129.567 211.933 129.561 211.976 129.555C212.02 129.549 212.051 129.539 212.071 129.527C212.091 129.514 212.101 129.494 212.101 129.467V129.463C212.101 129.404 212.084 129.359 212.049 129.327C212.015 129.295 211.965 129.279 211.899 129.279C211.831 129.279 211.776 129.294 211.736 129.324C211.696 129.354 211.669 129.389 211.655 129.43L211.393 129.393C211.413 129.32 211.448 129.26 211.495 129.211C211.543 129.162 211.601 129.125 211.67 129.101C211.739 129.076 211.815 129.064 211.898 129.064C211.955 129.064 212.013 129.07 212.069 129.084C212.126 129.097 212.178 129.12 212.225 129.151C212.272 129.181 212.31 129.223 212.339 129.276C212.368 129.328 212.382 129.394 212.382 129.473V130.271H212.112V130.107H212.103C212.086 130.141 212.062 130.172 212.031 130.201C212 130.229 211.962 130.252 211.915 130.27C211.869 130.287 211.815 130.295 211.753 130.295ZM211.826 130.089C211.882 130.089 211.931 130.078 211.972 130.055C212.014 130.033 212.046 130.003 212.068 129.965C212.091 129.928 212.102 129.888 212.102 129.844V129.703C212.093 129.71 212.078 129.717 212.057 129.723C212.036 129.73 212.013 129.735 211.987 129.74C211.961 129.744 211.936 129.748 211.91 129.752C211.885 129.756 211.863 129.759 211.844 129.761C211.802 129.767 211.765 129.776 211.732 129.789C211.699 129.802 211.673 129.82 211.653 129.844C211.634 129.866 211.625 129.896 211.625 129.932C211.625 129.984 211.644 130.023 211.681 130.049C211.719 130.076 211.767 130.089 211.826 130.089ZM213.25 129.079H213.531V130.339C213.531 130.425 213.515 130.496 213.482 130.552C213.45 130.608 213.403 130.65 213.341 130.677C213.28 130.704 213.205 130.718 213.117 130.718C213.107 130.718 213.097 130.718 213.088 130.717C213.079 130.717 213.07 130.717 213.059 130.717V130.488C213.067 130.488 213.074 130.488 213.08 130.488C213.087 130.489 213.093 130.489 213.1 130.489C213.154 130.489 213.192 130.476 213.215 130.45C213.239 130.424 213.25 130.386 213.25 130.336V129.079ZM213.39 128.91C213.345 128.91 213.306 128.895 213.274 128.866C213.243 128.836 213.227 128.8 213.227 128.758C213.227 128.716 213.243 128.68 213.274 128.65C213.306 128.62 213.345 128.605 213.39 128.605C213.435 128.605 213.473 128.62 213.504 128.65C213.536 128.68 213.552 128.716 213.552 128.758C213.552 128.8 213.536 128.836 213.504 128.866C213.473 128.895 213.435 128.91 213.39 128.91ZM213.842 130.271V129.079H214.123V130.271H213.842ZM213.983 128.91C213.939 128.91 213.9 128.895 213.868 128.866C213.836 128.836 213.82 128.8 213.82 128.758C213.82 128.716 213.836 128.68 213.868 128.65C213.9 128.62 213.939 128.605 213.983 128.605C214.028 128.605 214.067 128.62 214.098 128.65C214.13 128.68 214.146 128.716 214.146 128.758C214.146 128.8 214.13 128.836 214.098 128.866C214.067 128.895 214.028 128.91 213.983 128.91ZM214.952 130.743C214.851 130.743 214.764 130.729 214.692 130.702C214.62 130.675 214.561 130.639 214.517 130.593C214.473 130.548 214.443 130.497 214.426 130.442L214.679 130.381C214.69 130.404 214.707 130.427 214.728 130.45C214.75 130.473 214.779 130.492 214.816 130.507C214.853 130.523 214.9 130.53 214.957 130.53C215.036 130.53 215.102 130.511 215.155 130.472C215.207 130.434 215.233 130.371 215.233 130.283V130.057H215.219C215.204 130.086 215.183 130.116 215.155 130.146C215.128 130.177 215.091 130.202 215.046 130.223C215.001 130.244 214.944 130.254 214.876 130.254C214.784 130.254 214.701 130.233 214.627 130.19C214.553 130.146 214.494 130.082 214.45 129.996C214.406 129.909 214.385 129.801 214.385 129.671C214.385 129.54 214.406 129.43 214.45 129.34C214.494 129.249 214.553 129.181 214.628 129.134C214.702 129.087 214.785 129.064 214.877 129.064C214.947 129.064 215.004 129.076 215.049 129.099C215.094 129.123 215.131 129.151 215.158 129.184C215.185 129.217 215.205 129.247 215.219 129.276H215.234V129.079H215.511V130.291C215.511 130.392 215.487 130.477 215.439 130.544C215.39 130.61 215.323 130.66 215.239 130.693C215.155 130.726 215.059 130.743 214.952 130.743ZM214.954 130.034C215.014 130.034 215.064 130.019 215.106 129.99C215.148 129.961 215.18 129.92 215.202 129.865C215.224 129.811 215.234 129.746 215.234 129.67C215.234 129.595 215.224 129.529 215.202 129.473C215.181 129.416 215.149 129.373 215.107 129.342C215.066 129.31 215.015 129.294 214.954 129.294C214.892 129.294 214.839 129.31 214.798 129.343C214.756 129.376 214.724 129.42 214.703 129.477C214.682 129.534 214.671 129.598 214.671 129.67C214.671 129.743 214.682 129.807 214.703 129.861C214.725 129.916 214.756 129.958 214.798 129.989C214.841 130.019 214.893 130.034 214.954 130.034ZM216.763 129.394L216.507 129.422C216.5 129.396 216.487 129.372 216.469 129.349C216.452 129.327 216.428 129.308 216.398 129.294C216.368 129.28 216.331 129.273 216.288 129.273C216.229 129.273 216.18 129.286 216.14 129.311C216.101 129.337 216.081 129.369 216.082 129.41C216.081 129.444 216.094 129.473 216.12 129.494C216.146 129.516 216.19 129.534 216.25 129.548L216.454 129.591C216.566 129.616 216.65 129.654 216.705 129.707C216.76 129.76 216.788 129.829 216.789 129.914C216.788 129.989 216.766 130.055 216.723 130.113C216.68 130.17 216.62 130.214 216.544 130.246C216.467 130.278 216.379 130.294 216.28 130.294C216.134 130.294 216.017 130.264 215.928 130.203C215.839 130.141 215.786 130.056 215.768 129.946L216.042 129.92C216.055 129.973 216.081 130.014 216.122 130.041C216.162 130.069 216.214 130.083 216.279 130.083C216.346 130.083 216.399 130.069 216.44 130.041C216.481 130.014 216.501 129.98 216.501 129.94C216.501 129.906 216.488 129.877 216.461 129.855C216.436 129.833 216.395 129.816 216.34 129.804L216.137 129.761C216.023 129.738 215.938 129.697 215.883 129.641C215.828 129.584 215.801 129.512 215.802 129.425C215.801 129.352 215.821 129.288 215.862 129.234C215.902 129.18 215.959 129.138 216.032 129.109C216.104 129.079 216.189 129.064 216.284 129.064C216.423 129.064 216.533 129.093 216.614 129.153C216.694 129.212 216.744 129.293 216.763 129.394ZM217.385 130.295C217.31 130.295 217.242 130.282 217.181 130.255C217.121 130.227 217.074 130.187 217.038 130.134C217.004 130.081 216.986 130.015 216.986 129.937C216.986 129.869 216.999 129.814 217.024 129.77C217.049 129.726 217.082 129.691 217.125 129.664C217.168 129.638 217.217 129.618 217.271 129.605C217.325 129.591 217.381 129.581 217.439 129.574C217.509 129.567 217.565 129.561 217.609 129.555C217.652 129.549 217.684 129.539 217.704 129.527C217.724 129.514 217.734 129.494 217.734 129.467V129.463C217.734 129.404 217.716 129.359 217.682 129.327C217.647 129.295 217.597 129.279 217.532 129.279C217.463 129.279 217.409 129.294 217.368 129.324C217.328 129.354 217.302 129.389 217.288 129.43L217.025 129.393C217.046 129.32 217.08 129.26 217.128 129.211C217.175 129.162 217.234 129.125 217.302 129.101C217.371 129.076 217.447 129.064 217.53 129.064C217.588 129.064 217.645 129.07 217.702 129.084C217.759 129.097 217.811 129.12 217.858 129.151C217.905 129.181 217.943 129.223 217.971 129.276C218 129.328 218.015 129.394 218.015 129.473V130.271H217.745V130.107H217.735C217.718 130.141 217.694 130.172 217.663 130.201C217.633 130.229 217.594 130.252 217.548 130.27C217.501 130.287 217.447 130.295 217.385 130.295ZM217.458 130.089C217.515 130.089 217.564 130.078 217.605 130.055C217.646 130.033 217.678 130.003 217.7 129.965C217.723 129.928 217.735 129.888 217.735 129.844V129.703C217.726 129.71 217.711 129.717 217.69 129.723C217.669 129.73 217.646 129.735 217.62 129.74C217.594 129.744 217.568 129.748 217.543 129.752C217.518 129.756 217.496 129.759 217.477 129.761C217.435 129.767 217.398 129.776 217.364 129.789C217.331 129.802 217.305 129.82 217.286 129.844C217.267 129.866 217.257 129.896 217.257 129.932C217.257 129.984 217.276 130.023 217.314 130.049C217.352 130.076 217.4 130.089 217.458 130.089ZM218.565 130.271L218.228 129.079H218.515L218.724 129.917H218.735L218.949 129.079H219.232L219.447 129.913H219.458L219.665 129.079H219.952L219.614 130.271H219.322L219.098 129.466H219.082L218.858 130.271H218.565ZM220.746 130.271V128.682H221.027V129.276H221.039C221.053 129.247 221.073 129.217 221.1 129.184C221.126 129.151 221.162 129.123 221.207 129.099C221.252 129.076 221.309 129.064 221.379 129.064C221.471 129.064 221.554 129.087 221.628 129.134C221.703 129.181 221.762 129.25 221.805 129.342C221.849 129.433 221.871 129.544 221.871 129.677C221.871 129.808 221.85 129.919 221.807 130.01C221.764 130.102 221.705 130.172 221.631 130.22C221.557 130.268 221.474 130.292 221.38 130.292C221.312 130.292 221.255 130.281 221.21 130.258C221.165 130.235 221.129 130.208 221.101 130.176C221.074 130.143 221.054 130.112 221.039 130.083H221.022V130.271H220.746ZM221.021 129.675C221.021 129.752 221.032 129.82 221.054 129.878C221.076 129.936 221.108 129.981 221.149 130.014C221.191 130.046 221.242 130.062 221.302 130.062C221.364 130.062 221.416 130.045 221.458 130.012C221.499 129.978 221.531 129.933 221.552 129.875C221.574 129.816 221.585 129.75 221.585 129.675C221.585 129.601 221.574 129.536 221.553 129.478C221.532 129.421 221.5 129.376 221.458 129.343C221.416 129.31 221.364 129.294 221.302 129.294C221.242 129.294 221.191 129.31 221.149 129.342C221.107 129.373 221.075 129.417 221.053 129.474C221.032 129.531 221.021 129.598 221.021 129.675ZM222.892 129.77V129.079H223.172V130.271H222.9V130.059H222.888C222.861 130.126 222.817 130.181 222.755 130.223C222.694 130.265 222.619 130.287 222.529 130.287C222.451 130.287 222.382 130.269 222.322 130.235C222.262 130.199 222.216 130.149 222.182 130.082C222.149 130.015 222.132 129.933 222.132 129.838V129.079H222.413V129.795C222.413 129.87 222.433 129.93 222.475 129.975C222.516 130.019 222.571 130.041 222.638 130.041C222.679 130.041 222.719 130.031 222.758 130.011C222.797 129.991 222.829 129.961 222.854 129.921C222.879 129.881 222.892 129.83 222.892 129.77ZM223.483 130.271V129.079H223.764V130.271H223.483ZM223.624 128.91C223.58 128.91 223.542 128.895 223.51 128.866C223.477 128.836 223.461 128.8 223.461 128.758C223.461 128.716 223.477 128.68 223.51 128.65C223.542 128.62 223.58 128.605 223.624 128.605C223.669 128.605 223.708 128.62 223.739 128.65C223.771 128.68 223.787 128.716 223.787 128.758C223.787 128.8 223.771 128.836 223.739 128.866C223.708 128.895 223.669 128.91 223.624 128.91ZM224.356 128.682V130.271H224.075V128.682H224.356ZM225.109 130.292C225.015 130.292 224.931 130.268 224.857 130.22C224.783 130.172 224.725 130.102 224.682 130.01C224.639 129.919 224.618 129.808 224.618 129.677C224.618 129.544 224.639 129.433 224.683 129.342C224.727 129.25 224.786 129.181 224.86 129.134C224.935 129.087 225.018 129.064 225.109 129.064C225.179 129.064 225.237 129.076 225.282 129.099C225.327 129.123 225.362 129.151 225.389 129.184C225.415 129.217 225.436 129.247 225.45 129.276H225.462V128.682H225.743V130.271H225.467V130.083H225.45C225.436 130.112 225.415 130.143 225.387 130.176C225.36 130.208 225.324 130.235 225.279 130.258C225.234 130.281 225.177 130.292 225.109 130.292ZM225.187 130.062C225.247 130.062 225.297 130.046 225.339 130.014C225.381 129.981 225.413 129.936 225.435 129.878C225.456 129.82 225.467 129.752 225.467 129.675C225.467 129.598 225.456 129.531 225.435 129.474C225.413 129.417 225.382 129.373 225.34 129.342C225.299 129.31 225.248 129.294 225.187 129.294C225.125 129.294 225.072 129.31 225.03 129.343C224.988 129.376 224.957 129.421 224.936 129.478C224.914 129.536 224.904 129.601 224.904 129.675C224.904 129.75 224.914 129.816 224.936 129.875C224.957 129.933 224.989 129.978 225.031 130.012C225.074 130.045 225.126 130.062 225.187 130.062ZM226.065 130.271V129.079H226.346V130.271H226.065ZM226.206 128.91C226.161 128.91 226.123 128.895 226.091 128.866C226.059 128.836 226.043 128.8 226.043 128.758C226.043 128.716 226.059 128.68 226.091 128.65C226.123 128.62 226.161 128.605 226.206 128.605C226.251 128.605 226.289 128.62 226.321 128.65C226.353 128.68 226.369 128.716 226.369 128.758C226.369 128.8 226.353 128.836 226.321 128.866C226.289 128.895 226.251 128.91 226.206 128.91ZM226.937 129.573V130.271H226.656V129.079H226.925V129.282H226.939C226.966 129.215 227.01 129.162 227.07 129.123C227.13 129.083 227.205 129.064 227.294 129.064C227.376 129.064 227.448 129.081 227.509 129.116C227.571 129.152 227.618 129.203 227.652 129.269C227.686 129.336 227.703 129.417 227.702 129.512V130.271H227.421V129.556C227.421 129.476 227.401 129.414 227.359 129.369C227.318 129.324 227.262 129.301 227.189 129.301C227.14 129.301 227.097 129.312 227.058 129.334C227.021 129.355 226.991 129.386 226.969 129.426C226.948 129.466 226.937 129.515 226.937 129.573ZM228.528 130.743C228.427 130.743 228.341 130.729 228.268 130.702C228.196 130.675 228.138 130.639 228.094 130.593C228.05 130.548 228.019 130.497 228.002 130.442L228.255 130.381C228.266 130.404 228.283 130.427 228.305 130.45C228.326 130.473 228.356 130.492 228.392 130.507C228.43 130.523 228.476 130.53 228.533 130.53C228.612 130.53 228.678 130.511 228.731 130.472C228.783 130.434 228.809 130.371 228.809 130.283V130.057H228.795C228.781 130.086 228.759 130.116 228.731 130.146C228.704 130.177 228.668 130.202 228.622 130.223C228.577 130.244 228.52 130.254 228.452 130.254C228.361 130.254 228.277 130.233 228.203 130.19C228.129 130.146 228.07 130.082 228.026 129.996C227.983 129.909 227.961 129.801 227.961 129.671C227.961 129.54 227.983 129.43 228.026 129.34C228.07 129.249 228.129 129.181 228.204 129.134C228.278 129.087 228.361 129.064 228.453 129.064C228.523 129.064 228.58 129.076 228.625 129.099C228.671 129.123 228.707 129.151 228.734 129.184C228.761 129.217 228.781 129.247 228.795 129.276H228.811V129.079H229.088V130.291C229.088 130.392 229.063 130.477 229.015 130.544C228.966 130.61 228.9 130.66 228.815 130.693C228.731 130.726 228.635 130.743 228.528 130.743ZM228.53 130.034C228.59 130.034 228.641 130.019 228.683 129.99C228.724 129.961 228.756 129.92 228.778 129.865C228.8 129.811 228.811 129.746 228.811 129.67C228.811 129.595 228.8 129.529 228.778 129.473C228.757 129.416 228.725 129.373 228.683 129.342C228.642 129.31 228.591 129.294 228.53 129.294C228.468 129.294 228.416 129.31 228.374 129.343C228.332 129.376 228.3 129.42 228.279 129.477C228.258 129.534 228.247 129.598 228.247 129.67C228.247 129.743 228.258 129.807 228.279 129.861C228.301 129.916 228.333 129.958 228.374 129.989C228.417 130.019 228.469 130.034 228.53 130.034Z" fill="#434960"/><g clip-path="url(#clip3_3202_39054)"><path d="M201.795 134.273L201.794 134.292C201.87 134.292 201.943 134.31 202.01 134.345C202.077 134.379 202.134 134.43 202.177 134.491L202.177 134.491L202.326 134.703L202.341 134.724L202.356 134.703L202.505 134.491L202.505 134.491C202.548 134.43 202.605 134.379 202.671 134.345C202.738 134.31 202.812 134.292 202.887 134.292L202.887 134.273M201.795 134.273V134.292C201.537 134.292 201.327 134.501 201.327 134.759C201.327 135.035 201.492 135.302 201.704 135.533C201.911 135.759 202.162 135.948 202.341 136.075C202.519 135.948 202.77 135.759 202.978 135.533C203.19 135.302 203.354 135.035 203.354 134.759C203.354 134.501 203.145 134.292 202.887 134.292V134.273M201.795 134.273C201.527 134.273 201.309 134.491 201.309 134.759C201.309 135.326 201.982 135.844 202.341 136.097C202.7 135.844 203.372 135.326 203.372 134.759C203.372 134.491 203.155 134.273 202.887 134.273M201.795 134.273L202.352 134.677M202.887 134.273C202.809 134.274 202.732 134.293 202.663 134.329C202.594 134.365 202.534 134.417 202.49 134.481L202.352 134.677M202.352 134.677L202.356 134.682L202.341 134.693L202.326 134.682L202.33 134.677L202.352 134.677ZM202.341 134.345C202.402 134.264 202.48 134.198 202.571 134.151C202.668 134.1 202.777 134.073 202.887 134.073C202.977 134.073 203.066 134.091 203.149 134.125C203.233 134.16 203.308 134.21 203.372 134.274C203.435 134.338 203.486 134.413 203.52 134.497C203.555 134.58 203.573 134.669 203.573 134.759C203.573 135.113 203.372 135.434 203.12 135.702C202.868 135.97 202.564 136.187 202.351 136.334L202.341 136.341L202.33 136.334C202.118 136.187 201.813 135.97 201.561 135.702C201.31 135.434 201.109 135.113 201.109 134.759C201.109 134.577 201.181 134.403 201.31 134.274C201.438 134.145 201.613 134.073 201.795 134.073L202.341 134.345ZM202.341 134.345C202.28 134.264 202.201 134.198 202.111 134.151C202.013 134.1 201.905 134.073 201.795 134.073V134.091L202.341 134.345Z" fill="#434960" stroke="#434960" stroke-width="0.03642"/></g><path d="M205.752 136.045H205.028L205.375 135.652C205.607 135.405 205.732 135.247 205.732 135.017C205.732 134.752 205.52 134.525 205.176 134.525C204.842 134.525 204.595 134.747 204.595 135.073H204.847C204.847 134.88 204.967 134.757 205.158 134.757C205.344 134.757 205.462 134.887 205.462 135.051C205.462 135.226 205.352 135.354 205.176 135.543L204.638 136.126V136.274H205.752V136.045ZM207.212 135.894V135.67H206.978V134.551H206.779L205.889 135.713V135.894H206.72V136.274H206.978V135.894H207.212ZM206.18 135.67L206.72 134.941V135.67H206.18Z" fill="#434960"/><g clip-path="url(#clip4_3202_39054)"><path d="M212.536 133.969C212.376 133.969 212.218 134 212.071 134.061C211.924 134.122 211.79 134.212 211.677 134.325C211.45 134.552 211.322 134.861 211.322 135.183C211.321 135.463 211.418 135.735 211.596 135.951L211.353 136.194C211.336 136.211 211.325 136.233 211.321 136.256C211.316 136.28 211.319 136.304 211.328 136.326C211.338 136.348 211.354 136.366 211.375 136.379C211.395 136.391 211.419 136.398 211.443 136.397H212.536C212.857 136.397 213.166 136.269 213.394 136.041C213.621 135.813 213.749 135.505 213.749 135.183C213.749 134.861 213.621 134.552 213.394 134.325C213.166 134.097 212.857 133.969 212.536 133.969V133.969ZM212.536 136.154H211.736L211.849 136.041C211.871 136.018 211.884 135.987 211.884 135.955C211.884 135.923 211.871 135.892 211.849 135.87C211.69 135.711 211.591 135.502 211.568 135.279C211.546 135.055 211.602 134.831 211.727 134.644C211.852 134.457 212.037 134.319 212.252 134.254C212.467 134.188 212.698 134.2 212.906 134.285C213.113 134.371 213.285 134.526 213.391 134.724C213.497 134.922 213.531 135.151 213.487 135.372C213.444 135.592 213.325 135.79 213.151 135.933C212.978 136.076 212.76 136.154 212.536 136.154V136.154Z" fill="#434960"/></g><path d="M216.44 135.716C216.44 135.328 216.108 135.096 215.711 135.183L216.218 134.551H215.925L215.405 135.193C215.252 135.382 215.178 135.517 215.178 135.729C215.178 136.042 215.443 136.297 215.807 136.297C216.177 136.297 216.44 136.042 216.44 135.716ZM215.448 135.721C215.448 135.481 215.609 135.359 215.81 135.359C216.014 135.359 216.172 135.497 216.172 135.724C216.172 135.938 216.014 136.08 215.81 136.08C215.609 136.08 215.448 135.938 215.448 135.721Z" fill="#434960"/></g></g><g filter="url(#filter2_dd_3202_39054)"><g clip-path="url(#clip5_3202_39054)"><rect x="198.697" y="142.014" width="51.8985" height="68.4688" rx="0.7284" fill="white"/><rect x="198.697" y="142.014" width="51.8985" height="48.4386" fill="url(#pattern1)"/></g></g><g filter="url(#filter3_dd_3202_39054)"><g clip-path="url(#clip6_3202_39054)"><rect x="254.238" y="57.1656" width="51.8985" height="67.6354" rx="0.7284" fill="white"/><rect x="254.238" y="57.1656" width="51.8985" height="48.4386" fill="url(#pattern2)"/><rect x="273.357" y="74.4637" width="13.8396" height="13.8396" rx="6.9198" fill="white"/><path d="M278.639 79.6748C278.639 79.5394 278.781 79.4514 278.902 79.5119L282.865 81.4935C283 81.5606 283 81.7522 282.865 81.8193L278.902 83.8009C278.781 83.8614 278.639 83.7734 278.639 83.638V79.6748Z" fill="black"/><path d="M257.444 108.637C257.437 108.569 257.406 108.517 257.353 108.479C257.299 108.441 257.23 108.422 257.145 108.422C257.085 108.422 257.033 108.431 256.99 108.449C256.947 108.467 256.915 108.492 256.892 108.523C256.869 108.554 256.857 108.589 256.857 108.629C256.857 108.662 256.864 108.691 256.879 108.715C256.895 108.74 256.916 108.76 256.942 108.778C256.969 108.794 256.998 108.808 257.03 108.819C257.062 108.831 257.094 108.84 257.127 108.848L257.276 108.885C257.336 108.899 257.394 108.918 257.449 108.942C257.505 108.966 257.555 108.996 257.599 109.032C257.643 109.068 257.678 109.112 257.704 109.163C257.73 109.214 257.743 109.274 257.743 109.343C257.743 109.436 257.719 109.518 257.672 109.589C257.624 109.66 257.555 109.715 257.465 109.754C257.376 109.794 257.267 109.813 257.14 109.813C257.016 109.813 256.909 109.794 256.818 109.756C256.728 109.718 256.657 109.662 256.605 109.588C256.555 109.515 256.527 109.425 256.523 109.32H256.806C256.811 109.375 256.828 109.421 256.858 109.458C256.888 109.495 256.927 109.522 256.975 109.54C257.023 109.558 257.078 109.567 257.138 109.567C257.2 109.567 257.255 109.558 257.302 109.54C257.35 109.52 257.387 109.494 257.414 109.46C257.441 109.426 257.455 109.386 257.455 109.341C257.455 109.299 257.442 109.265 257.419 109.238C257.395 109.211 257.361 109.188 257.319 109.17C257.276 109.152 257.226 109.135 257.17 109.12L256.989 109.074C256.858 109.04 256.754 108.989 256.678 108.921C256.603 108.852 256.565 108.761 256.565 108.647C256.565 108.553 256.59 108.472 256.641 108.401C256.692 108.331 256.762 108.276 256.85 108.237C256.938 108.198 257.037 108.178 257.149 108.178C257.261 108.178 257.36 108.198 257.445 108.237C257.53 108.276 257.597 108.33 257.646 108.4C257.695 108.468 257.72 108.548 257.721 108.637H257.444ZM258.537 109.813C258.417 109.813 258.314 109.788 258.227 109.738C258.141 109.688 258.074 109.617 258.028 109.526C257.981 109.433 257.958 109.325 257.958 109.2C257.958 109.078 257.981 108.97 258.028 108.878C258.075 108.784 258.141 108.712 258.225 108.66C258.309 108.608 258.408 108.582 258.522 108.582C258.596 108.582 258.665 108.594 258.73 108.618C258.796 108.641 258.854 108.677 258.904 108.726C258.955 108.775 258.994 108.838 259.023 108.914C259.052 108.99 259.067 109.08 259.067 109.184V109.27H258.09V109.081H258.798C258.797 109.027 258.785 108.979 258.763 108.937C258.74 108.895 258.708 108.862 258.667 108.837C258.627 108.813 258.58 108.801 258.526 108.801C258.469 108.801 258.418 108.815 258.375 108.843C258.331 108.87 258.297 108.906 258.273 108.951C258.249 108.996 258.237 109.045 258.236 109.098V109.263C258.236 109.333 258.249 109.392 258.275 109.442C258.3 109.491 258.335 109.529 258.381 109.555C258.426 109.581 258.48 109.594 258.541 109.594C258.582 109.594 258.619 109.588 258.652 109.577C258.685 109.565 258.713 109.548 258.738 109.525C258.762 109.502 258.78 109.474 258.793 109.44L259.055 109.47C259.039 109.539 259.007 109.6 258.96 109.651C258.914 109.702 258.855 109.742 258.784 109.771C258.712 109.799 258.629 109.813 258.537 109.813ZM259.854 109.813C259.734 109.813 259.631 109.788 259.544 109.738C259.458 109.688 259.391 109.617 259.345 109.526C259.298 109.433 259.275 109.325 259.275 109.2C259.275 109.078 259.298 108.97 259.345 108.878C259.392 108.784 259.458 108.712 259.542 108.66C259.626 108.608 259.725 108.582 259.839 108.582C259.913 108.582 259.982 108.594 260.047 108.618C260.113 108.641 260.171 108.677 260.221 108.726C260.272 108.775 260.312 108.838 260.341 108.914C260.369 108.99 260.384 109.08 260.384 109.184V109.27H259.407V109.081H260.115C260.114 109.027 260.103 108.979 260.08 108.937C260.057 108.895 260.025 108.862 259.984 108.837C259.944 108.813 259.897 108.801 259.843 108.801C259.786 108.801 259.735 108.815 259.692 108.843C259.648 108.87 259.614 108.906 259.59 108.951C259.566 108.996 259.554 109.045 259.554 109.098V109.263C259.554 109.333 259.566 109.392 259.592 109.442C259.617 109.491 259.652 109.529 259.698 109.555C259.744 109.581 259.797 109.594 259.858 109.594C259.899 109.594 259.936 109.588 259.969 109.577C260.002 109.565 260.031 109.548 260.055 109.525C260.079 109.502 260.098 109.474 260.11 109.44L260.372 109.47C260.356 109.539 260.324 109.6 260.278 109.651C260.232 109.702 260.173 109.742 260.101 109.771C260.029 109.799 259.947 109.813 259.854 109.813ZM261.803 108.597V108.815H261.117V108.597H261.803ZM261.287 108.312H261.567V109.431C261.567 109.469 261.573 109.498 261.584 109.518C261.596 109.537 261.612 109.551 261.631 109.558C261.65 109.565 261.671 109.569 261.695 109.569C261.712 109.569 261.728 109.568 261.743 109.565C261.758 109.563 261.769 109.56 261.777 109.558L261.824 109.778C261.809 109.783 261.788 109.789 261.76 109.795C261.732 109.801 261.699 109.805 261.659 109.806C261.589 109.808 261.525 109.797 261.469 109.774C261.412 109.75 261.368 109.713 261.335 109.664C261.302 109.614 261.286 109.552 261.287 109.477V108.312ZM262.366 109.091V109.789H262.085V108.2H262.36V108.8H262.374C262.402 108.733 262.445 108.68 262.503 108.641C262.562 108.602 262.637 108.582 262.728 108.582C262.811 108.582 262.883 108.599 262.945 108.634C263.006 108.669 263.054 108.719 263.088 108.786C263.122 108.853 263.139 108.934 263.139 109.03V109.789H262.858V109.074C262.858 108.994 262.837 108.931 262.796 108.887C262.755 108.842 262.697 108.819 262.623 108.819C262.574 108.819 262.529 108.83 262.49 108.852C262.451 108.873 262.421 108.904 262.398 108.944C262.377 108.985 262.366 109.034 262.366 109.091ZM263.975 109.813C263.855 109.813 263.752 109.788 263.665 109.738C263.579 109.688 263.512 109.617 263.466 109.526C263.419 109.433 263.396 109.325 263.396 109.2C263.396 109.078 263.419 108.97 263.466 108.878C263.513 108.784 263.578 108.712 263.663 108.66C263.747 108.608 263.846 108.582 263.96 108.582C264.033 108.582 264.103 108.594 264.168 108.618C264.234 108.641 264.291 108.677 264.342 108.726C264.392 108.775 264.432 108.838 264.461 108.914C264.49 108.99 264.505 109.08 264.505 109.184V109.27H263.528V109.081H264.235C264.235 109.027 264.223 108.979 264.2 108.937C264.178 108.895 264.146 108.862 264.105 108.837C264.065 108.813 264.018 108.801 263.964 108.801C263.906 108.801 263.856 108.815 263.812 108.843C263.769 108.87 263.735 108.906 263.711 108.951C263.687 108.996 263.675 109.045 263.674 109.098V109.263C263.674 109.333 263.687 109.392 263.712 109.442C263.738 109.491 263.773 109.529 263.819 109.555C263.864 109.581 263.917 109.594 263.978 109.594C264.019 109.594 264.056 109.588 264.089 109.577C264.123 109.565 264.151 109.548 264.176 109.525C264.2 109.502 264.218 109.474 264.231 109.44L264.493 109.47C264.476 109.539 264.445 109.6 264.398 109.651C264.352 109.702 264.293 109.742 264.221 109.771C264.149 109.799 264.067 109.813 263.975 109.813ZM265.932 108.597V108.815H265.227V108.597H265.932ZM265.403 109.789V108.485C265.403 108.405 265.42 108.338 265.453 108.285C265.487 108.231 265.532 108.192 265.588 108.165C265.644 108.139 265.707 108.126 265.776 108.126C265.824 108.126 265.868 108.13 265.905 108.137C265.943 108.145 265.971 108.152 265.989 108.158L265.933 108.376C265.921 108.372 265.906 108.368 265.888 108.365C265.87 108.361 265.85 108.358 265.828 108.358C265.775 108.358 265.739 108.371 265.717 108.396C265.696 108.421 265.685 108.457 265.685 108.504V109.789H265.403ZM266.656 109.813C266.537 109.813 266.434 109.788 266.347 109.738C266.26 109.688 266.194 109.617 266.147 109.526C266.101 109.433 266.078 109.325 266.078 109.2C266.078 109.078 266.101 108.97 266.147 108.878C266.194 108.784 266.26 108.712 266.344 108.66C266.429 108.608 266.528 108.582 266.642 108.582C266.715 108.582 266.784 108.594 266.85 108.618C266.915 108.641 266.973 108.677 267.023 108.726C267.074 108.775 267.114 108.838 267.143 108.914C267.172 108.99 267.186 109.08 267.186 109.184V109.27H266.209V109.081H266.917C266.917 109.027 266.905 108.979 266.882 108.937C266.859 108.895 266.828 108.862 266.787 108.837C266.746 108.813 266.699 108.801 266.646 108.801C266.588 108.801 266.538 108.815 266.494 108.843C266.451 108.87 266.417 108.906 266.393 108.951C266.369 108.996 266.357 109.045 266.356 109.098V109.263C266.356 109.333 266.369 109.392 266.394 109.442C266.419 109.491 266.455 109.529 266.5 109.555C266.546 109.581 266.599 109.594 266.66 109.594C266.701 109.594 266.738 109.588 266.771 109.577C266.804 109.565 266.833 109.548 266.857 109.525C266.882 109.502 266.9 109.474 266.913 109.44L267.175 109.47C267.158 109.539 267.127 109.6 267.08 109.651C267.034 109.702 266.975 109.742 266.903 109.771C266.831 109.799 266.749 109.813 266.656 109.813ZM267.789 109.813C267.713 109.813 267.645 109.8 267.585 109.773C267.525 109.746 267.477 109.705 267.442 109.652C267.407 109.599 267.39 109.533 267.39 109.455C267.39 109.388 267.402 109.332 267.427 109.288C267.452 109.244 267.486 109.209 267.529 109.183C267.572 109.156 267.62 109.136 267.674 109.123C267.728 109.109 267.785 109.099 267.842 109.093C267.912 109.085 267.969 109.079 268.012 109.073C268.056 109.067 268.087 109.058 268.107 109.045C268.127 109.032 268.137 109.012 268.137 108.985V108.981C268.137 108.922 268.12 108.877 268.085 108.845C268.051 108.813 268.001 108.797 267.936 108.797C267.867 108.797 267.812 108.812 267.772 108.842C267.732 108.872 267.705 108.907 267.691 108.948L267.429 108.911C267.45 108.839 267.484 108.778 267.531 108.729C267.579 108.68 267.637 108.644 267.706 108.619C267.775 108.594 267.851 108.582 267.934 108.582C267.991 108.582 268.049 108.589 268.106 108.602C268.162 108.616 268.214 108.638 268.262 108.669C268.309 108.699 268.346 108.741 268.375 108.794C268.404 108.847 268.418 108.913 268.418 108.992V109.789H268.148V109.626H268.139C268.122 109.659 268.098 109.69 268.067 109.719C268.036 109.747 267.998 109.77 267.951 109.788C267.905 109.805 267.851 109.813 267.789 109.813ZM267.862 109.607C267.918 109.607 267.967 109.596 268.009 109.574C268.05 109.551 268.082 109.521 268.104 109.484C268.127 109.446 268.138 109.406 268.138 109.362V109.221C268.129 109.229 268.114 109.235 268.093 109.242C268.072 109.248 268.049 109.253 268.023 109.258C267.997 109.262 267.972 109.267 267.946 109.27C267.921 109.274 267.899 109.277 267.88 109.28C267.839 109.285 267.801 109.295 267.768 109.308C267.735 109.32 267.709 109.339 267.69 109.362C267.67 109.385 267.661 109.414 267.661 109.45C267.661 109.502 267.68 109.541 267.718 109.567C267.755 109.594 267.803 109.607 267.862 109.607ZM269.318 108.597V108.815H268.633V108.597H269.318ZM268.802 108.312H269.083V109.431C269.083 109.469 269.088 109.498 269.1 109.518C269.112 109.537 269.127 109.551 269.146 109.558C269.165 109.565 269.187 109.569 269.21 109.569C269.227 109.569 269.244 109.568 269.258 109.565C269.273 109.563 269.284 109.56 269.292 109.558L269.339 109.778C269.324 109.783 269.303 109.789 269.275 109.795C269.248 109.801 269.214 109.805 269.174 109.806C269.104 109.808 269.04 109.797 268.984 109.774C268.928 109.75 268.883 109.713 268.85 109.664C268.817 109.614 268.801 109.552 268.802 109.477V108.312ZM270.334 109.288V108.597H270.615V109.789H270.343V109.578H270.33C270.303 109.644 270.259 109.699 270.198 109.741C270.137 109.784 270.061 109.805 269.972 109.805C269.894 109.805 269.825 109.788 269.765 109.753C269.705 109.718 269.659 109.667 269.625 109.6C269.591 109.533 269.575 109.452 269.575 109.356V108.597H269.855V109.313C269.855 109.388 269.876 109.448 269.918 109.493C269.959 109.537 270.013 109.56 270.08 109.56C270.122 109.56 270.162 109.55 270.201 109.529C270.24 109.509 270.271 109.479 270.296 109.439C270.322 109.399 270.334 109.349 270.334 109.288ZM270.926 109.789V108.597H271.198V108.796H271.211C271.232 108.727 271.27 108.674 271.322 108.637C271.376 108.599 271.436 108.58 271.505 108.58C271.52 108.58 271.538 108.581 271.557 108.583C271.576 108.584 271.593 108.586 271.606 108.588V108.847C271.594 108.842 271.575 108.839 271.549 108.836C271.524 108.832 271.499 108.83 271.475 108.83C271.424 108.83 271.378 108.841 271.337 108.864C271.297 108.885 271.265 108.916 271.242 108.954C271.218 108.993 271.207 109.038 271.207 109.089V109.789H270.926ZM272.293 109.813C272.173 109.813 272.07 109.788 271.983 109.738C271.897 109.688 271.83 109.617 271.784 109.526C271.737 109.433 271.714 109.325 271.714 109.2C271.714 109.078 271.737 108.97 271.784 108.878C271.831 108.784 271.896 108.712 271.981 108.66C272.065 108.608 272.164 108.582 272.278 108.582C272.351 108.582 272.421 108.594 272.486 108.618C272.552 108.641 272.61 108.677 272.66 108.726C272.71 108.775 272.75 108.838 272.779 108.914C272.808 108.99 272.823 109.08 272.823 109.184V109.27H271.846V109.081H272.553C272.553 109.027 272.541 108.979 272.519 108.937C272.496 108.895 272.464 108.862 272.423 108.837C272.383 108.813 272.336 108.801 272.282 108.801C272.224 108.801 272.174 108.815 272.131 108.843C272.087 108.87 272.053 108.906 272.029 108.951C272.005 108.996 271.993 109.045 271.992 109.098V109.263C271.992 109.333 272.005 109.392 272.03 109.442C272.056 109.491 272.091 109.529 272.137 109.555C272.182 109.581 272.236 109.594 272.297 109.594C272.337 109.594 272.374 109.588 272.408 109.577C272.441 109.565 272.469 109.548 272.494 109.525C272.518 109.502 272.536 109.474 272.549 109.44L272.811 109.47C272.795 109.539 272.763 109.6 272.716 109.651C272.67 109.702 272.611 109.742 272.539 109.771C272.468 109.799 272.385 109.813 272.293 109.813ZM273.525 109.81C273.431 109.81 273.347 109.786 273.273 109.738C273.199 109.69 273.141 109.62 273.098 109.529C273.055 109.437 273.033 109.326 273.033 109.195C273.033 109.063 273.055 108.951 273.099 108.86C273.142 108.768 273.202 108.699 273.276 108.653C273.351 108.605 273.434 108.582 273.525 108.582C273.595 108.582 273.653 108.594 273.698 108.618C273.743 108.641 273.778 108.669 273.805 108.702C273.831 108.735 273.851 108.766 273.866 108.795H273.878V108.2H274.159V109.789H273.883V109.602H273.866C273.851 109.631 273.831 109.661 273.803 109.694C273.776 109.726 273.739 109.753 273.694 109.776C273.649 109.799 273.593 109.81 273.525 109.81ZM273.603 109.58C273.662 109.58 273.713 109.564 273.755 109.532C273.797 109.499 273.829 109.454 273.85 109.396C273.872 109.338 273.883 109.271 273.883 109.193C273.883 109.116 273.872 109.049 273.85 108.992C273.829 108.936 273.798 108.891 273.756 108.86C273.714 108.828 273.663 108.812 273.603 108.812C273.54 108.812 273.488 108.829 273.446 108.861C273.404 108.894 273.373 108.939 273.351 108.996C273.33 109.054 273.32 109.119 273.32 109.193C273.32 109.268 273.33 109.334 273.351 109.393C273.373 109.451 273.405 109.497 273.447 109.53C273.489 109.563 273.541 109.58 273.603 109.58ZM275.572 109.813C275.453 109.813 275.349 109.788 275.262 109.738C275.176 109.688 275.11 109.617 275.063 109.526C275.016 109.433 274.993 109.325 274.993 109.2C274.993 109.078 275.016 108.97 275.063 108.878C275.11 108.784 275.176 108.712 275.26 108.66C275.344 108.608 275.443 108.582 275.557 108.582C275.631 108.582 275.7 108.594 275.765 108.618C275.831 108.641 275.889 108.677 275.939 108.726C275.99 108.775 276.03 108.838 276.059 108.914C276.088 108.99 276.102 109.08 276.102 109.184V109.27H275.125V109.081H275.833C275.832 109.027 275.821 108.979 275.798 108.937C275.775 108.895 275.743 108.862 275.702 108.837C275.662 108.813 275.615 108.801 275.561 108.801C275.504 108.801 275.453 108.815 275.41 108.843C275.366 108.87 275.333 108.906 275.308 108.951C275.284 108.996 275.272 109.045 275.272 109.098V109.263C275.272 109.333 275.284 109.392 275.31 109.442C275.335 109.491 275.371 109.529 275.416 109.555C275.462 109.581 275.515 109.594 275.576 109.594C275.617 109.594 275.654 109.588 275.687 109.577C275.72 109.565 275.749 109.548 275.773 109.525C275.797 109.502 275.816 109.474 275.828 109.44L276.09 109.47C276.074 109.539 276.042 109.6 275.996 109.651C275.95 109.702 275.891 109.742 275.819 109.771C275.747 109.799 275.665 109.813 275.572 109.813ZM276.804 109.81C276.71 109.81 276.626 109.786 276.552 109.738C276.478 109.69 276.42 109.62 276.377 109.529C276.334 109.437 276.313 109.326 276.313 109.195C276.313 109.063 276.334 108.951 276.378 108.86C276.422 108.768 276.481 108.699 276.556 108.653C276.63 108.605 276.713 108.582 276.805 108.582C276.874 108.582 276.932 108.594 276.977 108.618C277.022 108.641 277.058 108.669 277.084 108.702C277.11 108.735 277.131 108.766 277.145 108.795H277.157V108.2H277.439V109.789H277.162V109.602H277.145C277.131 109.631 277.11 109.661 277.082 109.694C277.055 109.726 277.019 109.753 276.974 109.776C276.929 109.799 276.872 109.81 276.804 109.81ZM276.882 109.58C276.942 109.58 276.992 109.564 277.034 109.532C277.076 109.499 277.108 109.454 277.13 109.396C277.152 109.338 277.162 109.271 277.162 109.193C277.162 109.116 277.152 109.049 277.13 108.992C277.109 108.936 277.077 108.891 277.035 108.86C276.994 108.828 276.943 108.812 276.882 108.812C276.82 108.812 276.767 108.829 276.725 108.861C276.684 108.894 276.652 108.939 276.631 108.996C276.61 109.054 276.599 109.119 276.599 109.193C276.599 109.268 276.61 109.334 276.631 109.393C276.653 109.451 276.684 109.497 276.726 109.53C276.769 109.563 276.821 109.58 276.882 109.58ZM277.76 109.789V108.597H278.041V109.789H277.76ZM277.901 108.428C277.857 108.428 277.818 108.414 277.786 108.384C277.754 108.354 277.738 108.318 277.738 108.276C277.738 108.234 277.754 108.198 277.786 108.168C277.818 108.138 277.857 108.123 277.901 108.123C277.946 108.123 277.984 108.138 278.016 108.168C278.048 108.198 278.064 108.234 278.064 108.276C278.064 108.318 278.048 108.354 278.016 108.384C277.984 108.414 277.946 108.428 277.901 108.428ZM278.947 108.597V108.815H278.261V108.597H278.947ZM278.431 108.312H278.712V109.431C278.712 109.469 278.717 109.498 278.729 109.518C278.74 109.537 278.756 109.551 278.775 109.558C278.794 109.565 278.815 109.569 278.839 109.569C278.856 109.569 278.872 109.568 278.887 109.565C278.902 109.563 278.913 109.56 278.921 109.558L278.968 109.778C278.953 109.783 278.932 109.789 278.904 109.795C278.877 109.801 278.843 109.805 278.803 109.806C278.733 109.808 278.669 109.797 278.613 109.774C278.557 109.75 278.512 109.713 278.479 109.664C278.446 109.614 278.43 109.552 278.431 109.477V108.312ZM279.203 109.789V108.597H279.484V109.789H279.203ZM279.345 108.428C279.3 108.428 279.262 108.414 279.23 108.384C279.198 108.354 279.182 108.318 279.182 108.276C279.182 108.234 279.198 108.198 279.23 108.168C279.262 108.138 279.3 108.123 279.345 108.123C279.39 108.123 279.428 108.138 279.46 108.168C279.492 108.198 279.508 108.234 279.508 108.276C279.508 108.318 279.492 108.354 279.46 108.384C279.428 108.414 279.39 108.428 279.345 108.428ZM280.315 109.813C280.199 109.813 280.098 109.787 280.012 109.736C279.927 109.685 279.861 109.613 279.814 109.521C279.767 109.429 279.744 109.321 279.744 109.198C279.744 109.075 279.767 108.967 279.814 108.875C279.861 108.782 279.927 108.71 280.012 108.659C280.098 108.608 280.199 108.582 280.315 108.582C280.431 108.582 280.532 108.608 280.618 108.659C280.703 108.71 280.769 108.782 280.815 108.875C280.863 108.967 280.886 109.075 280.886 109.198C280.886 109.321 280.863 109.429 280.815 109.521C280.769 109.613 280.703 109.685 280.618 109.736C280.532 109.787 280.431 109.813 280.315 109.813ZM280.317 109.588C280.38 109.588 280.432 109.57 280.475 109.536C280.517 109.5 280.549 109.453 280.569 109.394C280.591 109.335 280.601 109.27 280.601 109.197C280.601 109.124 280.591 109.058 280.569 108.999C280.549 108.94 280.517 108.893 280.475 108.857C280.432 108.822 280.38 108.805 280.317 108.805C280.252 108.805 280.198 108.822 280.155 108.857C280.113 108.893 280.081 108.94 280.06 108.999C280.039 109.058 280.029 109.124 280.029 109.197C280.029 109.27 280.039 109.335 280.06 109.394C280.081 109.453 280.113 109.5 280.155 109.536C280.198 109.57 280.252 109.588 280.317 109.588ZM281.427 109.091V109.789H281.146V108.597H281.415V108.8H281.429C281.456 108.733 281.5 108.68 281.56 108.641C281.621 108.602 281.695 108.582 281.784 108.582C281.866 108.582 281.938 108.6 281.999 108.635C282.061 108.67 282.108 108.721 282.142 108.788C282.176 108.854 282.193 108.935 282.192 109.03V109.789H281.912V109.074C281.912 108.994 281.891 108.932 281.849 108.887C281.809 108.842 281.752 108.819 281.679 108.819C281.63 108.819 281.587 108.83 281.548 108.852C281.511 108.873 281.481 108.904 281.459 108.944C281.438 108.985 281.427 109.034 281.427 109.091ZM283.584 109.813C283.467 109.813 283.366 109.787 283.281 109.736C283.196 109.685 283.129 109.613 283.082 109.521C283.036 109.429 283.013 109.321 283.013 109.198C283.013 109.075 283.036 108.967 283.082 108.875C283.129 108.782 283.196 108.71 283.281 108.659C283.366 108.608 283.467 108.582 283.584 108.582C283.7 108.582 283.801 108.608 283.886 108.659C283.972 108.71 284.038 108.782 284.084 108.875C284.131 108.967 284.155 109.075 284.155 109.198C284.155 109.321 284.131 109.429 284.084 109.521C284.038 109.613 283.972 109.685 283.886 109.736C283.801 109.787 283.7 109.813 283.584 109.813ZM283.585 109.588C283.648 109.588 283.701 109.57 283.743 109.536C283.786 109.5 283.817 109.453 283.838 109.394C283.859 109.335 283.87 109.27 283.87 109.197C283.87 109.124 283.859 109.058 283.838 108.999C283.817 108.94 283.786 108.893 283.743 108.857C283.701 108.822 283.648 108.805 283.585 108.805C283.521 108.805 283.467 108.822 283.424 108.857C283.381 108.893 283.35 108.94 283.328 108.999C283.308 109.058 283.297 109.124 283.297 109.197C283.297 109.27 283.308 109.335 283.328 109.394C283.35 109.453 283.381 109.5 283.424 109.536C283.467 109.57 283.521 109.588 283.585 109.588ZM285.019 108.597V108.815H284.314V108.597H285.019ZM284.49 109.789V108.485C284.49 108.405 284.507 108.338 284.54 108.285C284.574 108.231 284.619 108.192 284.675 108.165C284.731 108.139 284.794 108.126 284.863 108.126C284.911 108.126 284.955 108.13 284.992 108.137C285.03 108.145 285.058 108.152 285.076 108.158L285.02 108.376C285.008 108.372 284.993 108.368 284.975 108.365C284.957 108.361 284.937 108.358 284.915 108.358C284.863 108.358 284.826 108.371 284.804 108.396C284.783 108.421 284.772 108.457 284.772 108.504V109.789H284.49ZM286.344 109.813C286.228 109.813 286.127 109.787 286.042 109.736C285.956 109.685 285.89 109.613 285.843 109.521C285.797 109.429 285.773 109.321 285.773 109.198C285.773 109.075 285.797 108.967 285.843 108.875C285.89 108.782 285.956 108.71 286.042 108.659C286.127 108.608 286.228 108.582 286.344 108.582C286.461 108.582 286.562 108.608 286.647 108.659C286.732 108.71 286.798 108.782 286.845 108.875C286.892 108.967 286.916 109.075 286.916 109.198C286.916 109.321 286.892 109.429 286.845 109.521C286.798 109.613 286.732 109.685 286.647 109.736C286.562 109.787 286.461 109.813 286.344 109.813ZM286.346 109.588C286.409 109.588 286.462 109.57 286.504 109.536C286.547 109.5 286.578 109.453 286.599 109.394C286.62 109.335 286.631 109.27 286.631 109.197C286.631 109.124 286.62 109.058 286.599 108.999C286.578 108.94 286.547 108.893 286.504 108.857C286.462 108.822 286.409 108.805 286.346 108.805C286.281 108.805 286.227 108.822 286.185 108.857C286.142 108.893 286.11 108.94 286.089 108.999C286.068 109.058 286.058 109.124 286.058 109.197C286.058 109.27 286.068 109.335 286.089 109.394C286.11 109.453 286.142 109.5 286.185 109.536C286.227 109.57 286.281 109.588 286.346 109.588ZM287.936 109.288V108.597H288.216V109.789H287.944V109.578H287.932C287.905 109.644 287.861 109.699 287.799 109.741C287.738 109.784 287.663 109.805 287.573 109.805C287.495 109.805 287.426 109.788 287.366 109.753C287.306 109.718 287.26 109.667 287.226 109.6C287.193 109.533 287.176 109.452 287.176 109.356V108.597H287.457V109.313C287.457 109.388 287.477 109.448 287.519 109.493C287.56 109.537 287.615 109.56 287.682 109.56C287.723 109.56 287.763 109.55 287.802 109.529C287.841 109.509 287.873 109.479 287.897 109.439C287.923 109.399 287.936 109.349 287.936 109.288ZM288.527 109.789V108.597H288.8V108.796H288.812C288.834 108.727 288.871 108.674 288.924 108.637C288.977 108.599 289.038 108.58 289.106 108.58C289.122 108.58 289.139 108.581 289.158 108.583C289.178 108.584 289.194 108.586 289.207 108.588V108.847C289.195 108.842 289.176 108.839 289.15 108.836C289.125 108.832 289.1 108.83 289.077 108.83C289.025 108.83 288.979 108.841 288.938 108.864C288.898 108.885 288.866 108.916 288.843 108.954C288.82 108.993 288.808 109.038 288.808 109.089V109.789H288.527ZM256.581 111.2H256.933L257.405 112.352H257.423L257.895 111.2H258.247V112.789H257.971V111.698H257.956L257.517 112.785H257.311L256.872 111.695H256.857V112.789H256.581V111.2ZM258.913 112.813C258.838 112.813 258.77 112.8 258.709 112.773C258.649 112.746 258.602 112.705 258.566 112.652C258.532 112.599 258.514 112.533 258.514 112.455C258.514 112.388 258.527 112.332 258.552 112.288C258.577 112.244 258.61 112.209 258.653 112.183C258.696 112.156 258.745 112.136 258.798 112.123C258.853 112.109 258.909 112.099 258.967 112.093C259.037 112.085 259.093 112.079 259.137 112.073C259.18 112.067 259.212 112.058 259.232 112.045C259.252 112.032 259.262 112.012 259.262 111.985V111.981C259.262 111.922 259.244 111.877 259.21 111.845C259.175 111.813 259.125 111.797 259.06 111.797C258.991 111.797 258.937 111.812 258.896 111.842C258.856 111.872 258.83 111.907 258.816 111.948L258.553 111.911C258.574 111.839 258.608 111.778 258.656 111.729C258.703 111.68 258.762 111.644 258.83 111.619C258.899 111.594 258.975 111.582 259.058 111.582C259.116 111.582 259.173 111.589 259.23 111.602C259.287 111.616 259.339 111.638 259.386 111.669C259.433 111.699 259.471 111.741 259.499 111.794C259.528 111.847 259.543 111.913 259.543 111.992V112.789H259.273V112.626H259.263C259.246 112.659 259.222 112.69 259.191 112.719C259.161 112.747 259.122 112.77 259.076 112.788C259.029 112.805 258.975 112.813 258.913 112.813ZM258.986 112.607C259.043 112.607 259.092 112.596 259.133 112.574C259.174 112.551 259.206 112.521 259.228 112.484C259.251 112.446 259.263 112.406 259.263 112.362V112.221C259.254 112.229 259.239 112.235 259.218 112.242C259.197 112.248 259.174 112.253 259.148 112.258C259.122 112.262 259.096 112.267 259.071 112.27C259.046 112.274 259.024 112.277 259.005 112.28C258.963 112.285 258.925 112.295 258.892 112.308C258.859 112.32 258.833 112.339 258.814 112.362C258.795 112.385 258.785 112.414 258.785 112.45C258.785 112.502 258.804 112.541 258.842 112.567C258.88 112.594 258.928 112.607 258.986 112.607ZM260.365 113.261C260.264 113.261 260.178 113.247 260.105 113.22C260.033 113.193 259.975 113.157 259.931 113.111C259.887 113.066 259.856 113.015 259.839 112.96L260.092 112.899C260.104 112.922 260.12 112.945 260.142 112.968C260.164 112.991 260.193 113.01 260.23 113.025C260.267 113.041 260.314 113.049 260.37 113.049C260.45 113.049 260.516 113.029 260.568 112.99C260.62 112.952 260.646 112.889 260.646 112.801V112.575H260.632C260.618 112.604 260.597 112.634 260.569 112.664C260.541 112.695 260.505 112.721 260.459 112.741C260.414 112.762 260.358 112.772 260.289 112.772C260.198 112.772 260.115 112.751 260.04 112.708C259.966 112.664 259.907 112.6 259.863 112.514C259.82 112.428 259.798 112.319 259.798 112.19C259.798 112.059 259.82 111.948 259.863 111.858C259.907 111.768 259.966 111.699 260.041 111.653C260.115 111.605 260.199 111.582 260.29 111.582C260.36 111.582 260.417 111.594 260.462 111.618C260.508 111.641 260.544 111.669 260.571 111.702C260.598 111.735 260.618 111.766 260.632 111.795H260.648V111.597H260.925V112.809C260.925 112.911 260.901 112.995 260.852 113.062C260.803 113.129 260.737 113.178 260.652 113.212C260.568 113.245 260.472 113.261 260.365 113.261ZM260.368 112.552C260.427 112.552 260.478 112.537 260.52 112.508C260.562 112.48 260.593 112.438 260.615 112.384C260.637 112.329 260.648 112.264 260.648 112.188C260.648 112.113 260.637 112.047 260.615 111.991C260.594 111.935 260.562 111.891 260.521 111.86C260.479 111.828 260.428 111.812 260.368 111.812C260.305 111.812 260.253 111.829 260.211 111.861C260.169 111.894 260.137 111.939 260.116 111.996C260.095 112.052 260.084 112.116 260.084 112.188C260.084 112.261 260.095 112.325 260.116 112.38C260.138 112.434 260.17 112.476 260.212 112.507C260.254 112.537 260.306 112.552 260.368 112.552ZM261.578 112.813C261.502 112.813 261.434 112.8 261.373 112.773C261.313 112.746 261.266 112.705 261.231 112.652C261.196 112.599 261.179 112.533 261.179 112.455C261.179 112.388 261.191 112.332 261.216 112.288C261.241 112.244 261.275 112.209 261.318 112.183C261.361 112.156 261.409 112.136 261.463 112.123C261.517 112.109 261.573 112.099 261.631 112.093C261.701 112.085 261.758 112.079 261.801 112.073C261.845 112.067 261.876 112.058 261.896 112.045C261.916 112.032 261.926 112.012 261.926 111.985V111.981C261.926 111.922 261.909 111.877 261.874 111.845C261.839 111.813 261.789 111.797 261.724 111.797C261.655 111.797 261.601 111.812 261.561 111.842C261.521 111.872 261.494 111.907 261.48 111.948L261.218 111.911C261.238 111.839 261.272 111.778 261.32 111.729C261.368 111.68 261.426 111.644 261.495 111.619C261.563 111.594 261.639 111.582 261.723 111.582C261.78 111.582 261.837 111.589 261.894 111.602C261.951 111.616 262.003 111.638 262.05 111.669C262.097 111.699 262.135 111.741 262.163 111.794C262.192 111.847 262.207 111.913 262.207 111.992V112.789H261.937V112.626H261.928C261.91 112.659 261.886 112.69 261.855 112.719C261.825 112.747 261.786 112.77 261.74 112.788C261.694 112.805 261.64 112.813 261.578 112.813ZM261.651 112.607C261.707 112.607 261.756 112.596 261.797 112.574C261.839 112.551 261.87 112.521 261.893 112.484C261.915 112.446 261.927 112.406 261.927 112.362V112.221C261.918 112.229 261.903 112.235 261.882 112.242C261.861 112.248 261.838 112.253 261.812 112.258C261.786 112.262 261.76 112.267 261.735 112.27C261.71 112.274 261.688 112.277 261.669 112.28C261.627 112.285 261.59 112.295 261.557 112.308C261.524 112.32 261.497 112.339 261.478 112.362C261.459 112.385 261.45 112.414 261.45 112.45C261.45 112.502 261.468 112.541 261.506 112.567C261.544 112.594 261.592 112.607 261.651 112.607ZM262.496 112.789V112.611L263.11 111.84V111.83H262.516V111.597H263.454V111.789L262.869 112.547V112.557H263.474V112.789H262.496ZM263.76 112.789V111.597H264.041V112.789H263.76ZM263.901 111.428C263.857 111.428 263.819 111.414 263.787 111.384C263.754 111.354 263.738 111.318 263.738 111.276C263.738 111.234 263.754 111.198 263.787 111.168C263.819 111.138 263.857 111.123 263.901 111.123C263.946 111.123 263.985 111.138 264.016 111.168C264.048 111.198 264.064 111.234 264.064 111.276C264.064 111.318 264.048 111.354 264.016 111.384C263.985 111.414 263.946 111.428 263.901 111.428ZM264.633 112.091V112.789H264.352V111.597H264.62V111.8H264.634C264.662 111.733 264.705 111.68 264.765 111.641C264.826 111.602 264.901 111.582 264.99 111.582C265.072 111.582 265.144 111.6 265.205 111.635C265.266 111.67 265.314 111.721 265.347 111.788C265.381 111.854 265.398 111.935 265.398 112.03V112.789H265.117V112.074C265.117 111.994 265.096 111.932 265.055 111.887C265.014 111.842 264.957 111.819 264.885 111.819C264.836 111.819 264.792 111.83 264.754 111.852C264.716 111.873 264.686 111.904 264.664 111.944C264.643 111.985 264.633 112.034 264.633 112.091ZM266.233 112.813C266.113 112.813 266.01 112.788 265.923 112.738C265.837 112.688 265.77 112.617 265.724 112.526C265.677 112.433 265.654 112.325 265.654 112.2C265.654 112.078 265.677 111.97 265.724 111.878C265.771 111.784 265.837 111.712 265.921 111.66C266.005 111.608 266.104 111.582 266.218 111.582C266.292 111.582 266.361 111.594 266.426 111.618C266.492 111.641 266.55 111.677 266.6 111.726C266.651 111.775 266.69 111.838 266.719 111.914C266.748 111.99 266.763 112.08 266.763 112.184V112.27H265.786V112.081H266.494C266.493 112.027 266.481 111.979 266.459 111.937C266.436 111.895 266.404 111.862 266.363 111.837C266.323 111.813 266.276 111.801 266.222 111.801C266.165 111.801 266.114 111.815 266.071 111.843C266.027 111.87 265.993 111.906 265.969 111.951C265.945 111.996 265.933 112.045 265.933 112.098V112.263C265.933 112.333 265.945 112.392 265.971 112.442C265.996 112.491 266.031 112.529 266.077 112.555C266.122 112.581 266.176 112.594 266.237 112.594C266.278 112.594 266.315 112.588 266.348 112.577C266.381 112.565 266.41 112.548 266.434 112.525C266.458 112.502 266.477 112.474 266.489 112.44L266.751 112.47C266.735 112.539 266.703 112.6 266.657 112.651C266.611 112.702 266.552 112.742 266.48 112.771C266.408 112.799 266.325 112.813 266.233 112.813ZM268.181 111.597V111.815H267.496V111.597H268.181ZM267.665 111.312H267.946V112.431C267.946 112.469 267.952 112.498 267.963 112.518C267.975 112.537 267.991 112.551 268.01 112.558C268.029 112.565 268.05 112.569 268.074 112.569C268.091 112.569 268.107 112.568 268.122 112.565C268.137 112.563 268.148 112.56 268.156 112.558L268.203 112.778C268.188 112.783 268.167 112.789 268.139 112.795C268.111 112.801 268.078 112.805 268.038 112.806C267.968 112.808 267.904 112.797 267.848 112.774C267.791 112.75 267.747 112.713 267.714 112.664C267.681 112.614 267.665 112.552 267.665 112.477V111.312ZM268.945 112.813C268.829 112.813 268.728 112.787 268.643 112.736C268.557 112.685 268.491 112.613 268.444 112.521C268.398 112.429 268.374 112.321 268.374 112.198C268.374 112.075 268.398 111.967 268.444 111.875C268.491 111.782 268.557 111.71 268.643 111.659C268.728 111.608 268.829 111.582 268.945 111.582C269.062 111.582 269.163 111.608 269.248 111.659C269.333 111.71 269.399 111.782 269.446 111.875C269.493 111.967 269.516 112.075 269.516 112.198C269.516 112.321 269.493 112.429 269.446 112.521C269.399 112.613 269.333 112.685 269.248 112.736C269.163 112.787 269.062 112.813 268.945 112.813ZM268.947 112.588C269.01 112.588 269.063 112.57 269.105 112.536C269.148 112.5 269.179 112.453 269.2 112.394C269.221 112.335 269.232 112.27 269.232 112.197C269.232 112.124 269.221 112.058 269.2 111.999C269.179 111.94 269.148 111.893 269.105 111.857C269.063 111.822 269.01 111.805 268.947 111.805C268.882 111.805 268.828 111.822 268.785 111.857C268.743 111.893 268.711 111.94 268.69 111.999C268.669 112.058 268.659 112.124 268.659 112.197C268.659 112.27 268.669 112.335 268.69 112.394C268.711 112.453 268.743 112.5 268.785 112.536C268.828 112.57 268.882 112.588 268.947 112.588ZM270.861 112.813C270.742 112.813 270.639 112.787 270.554 112.734C270.469 112.682 270.404 112.61 270.358 112.518C270.312 112.425 270.289 112.319 270.289 112.198C270.289 112.077 270.313 111.97 270.359 111.878C270.406 111.784 270.472 111.712 270.556 111.66C270.642 111.608 270.743 111.582 270.859 111.582C270.956 111.582 271.041 111.6 271.116 111.635C271.191 111.671 271.251 111.721 271.295 111.785C271.34 111.849 271.365 111.924 271.371 112.01H271.103C271.092 111.953 271.066 111.905 271.025 111.867C270.985 111.828 270.931 111.809 270.863 111.809C270.805 111.809 270.755 111.824 270.712 111.855C270.668 111.886 270.634 111.93 270.61 111.987C270.586 112.044 270.574 112.113 270.574 112.193C270.574 112.275 270.586 112.344 270.61 112.403C270.634 112.461 270.667 112.506 270.71 112.537C270.753 112.568 270.804 112.584 270.863 112.584C270.904 112.584 270.941 112.576 270.974 112.56C271.007 112.544 271.035 112.521 271.057 112.491C271.079 112.461 271.094 112.425 271.103 112.382H271.371C271.364 112.466 271.34 112.541 271.297 112.606C271.254 112.671 271.195 112.722 271.121 112.758C271.047 112.795 270.96 112.813 270.861 112.813ZM271.967 112.813C271.891 112.813 271.823 112.8 271.763 112.773C271.703 112.746 271.655 112.705 271.62 112.652C271.585 112.599 271.568 112.533 271.568 112.455C271.568 112.388 271.58 112.332 271.605 112.288C271.63 112.244 271.664 112.209 271.707 112.183C271.75 112.156 271.798 112.136 271.852 112.123C271.906 112.109 271.962 112.099 272.02 112.093C272.09 112.085 272.147 112.079 272.19 112.073C272.234 112.067 272.265 112.058 272.285 112.045C272.305 112.032 272.315 112.012 272.315 111.985V111.981C272.315 111.922 272.298 111.877 272.263 111.845C272.228 111.813 272.179 111.797 272.113 111.797C272.045 111.797 271.99 111.812 271.95 111.842C271.91 111.872 271.883 111.907 271.869 111.948L271.607 111.911C271.627 111.839 271.661 111.778 271.709 111.729C271.757 111.68 271.815 111.644 271.884 111.619C271.952 111.594 272.028 111.582 272.112 111.582C272.169 111.582 272.226 111.589 272.283 111.602C272.34 111.616 272.392 111.638 272.439 111.669C272.486 111.699 272.524 111.741 272.553 111.794C272.582 111.847 272.596 111.913 272.596 111.992V112.789H272.326V112.626H272.317C272.3 112.659 272.276 112.69 272.244 112.719C272.214 112.747 272.175 112.77 272.129 112.788C272.083 112.805 272.029 112.813 271.967 112.813ZM272.04 112.607C272.096 112.607 272.145 112.596 272.186 112.574C272.228 112.551 272.259 112.521 272.282 112.484C272.304 112.446 272.316 112.406 272.316 112.362V112.221C272.307 112.229 272.292 112.235 272.271 112.242C272.25 112.248 272.227 112.253 272.201 112.258C272.175 112.262 272.15 112.267 272.124 112.27C272.099 112.274 272.077 112.277 272.058 112.28C272.016 112.285 271.979 112.295 271.946 112.308C271.913 112.32 271.886 112.339 271.867 112.362C271.848 112.385 271.839 112.414 271.839 112.45C271.839 112.502 271.858 112.541 271.895 112.567C271.933 112.594 271.981 112.607 272.04 112.607ZM273.496 111.597V111.815H272.81V111.597H273.496ZM272.979 111.312H273.26V112.431C273.26 112.469 273.266 112.498 273.277 112.518C273.289 112.537 273.305 112.551 273.324 112.558C273.343 112.565 273.364 112.569 273.388 112.569C273.405 112.569 273.421 112.568 273.436 112.565C273.451 112.563 273.462 112.56 273.47 112.558L273.517 112.778C273.502 112.783 273.481 112.789 273.453 112.795C273.425 112.801 273.392 112.805 273.352 112.806C273.282 112.808 273.218 112.797 273.162 112.774C273.105 112.75 273.061 112.713 273.028 112.664C272.995 112.614 272.979 112.552 272.979 112.477V111.312ZM274.259 112.813C274.14 112.813 274.038 112.787 273.953 112.734C273.868 112.682 273.803 112.61 273.757 112.518C273.711 112.425 273.688 112.319 273.688 112.198C273.688 112.077 273.712 111.97 273.758 111.878C273.805 111.784 273.87 111.712 273.955 111.66C274.041 111.608 274.141 111.582 274.258 111.582C274.355 111.582 274.44 111.6 274.515 111.635C274.59 111.671 274.649 111.721 274.694 111.785C274.738 111.849 274.764 111.924 274.77 112.01H274.502C274.491 111.953 274.465 111.905 274.424 111.867C274.384 111.828 274.329 111.809 274.262 111.809C274.204 111.809 274.154 111.824 274.11 111.855C274.067 111.886 274.033 111.93 274.009 111.987C273.985 112.044 273.973 112.113 273.973 112.193C273.973 112.275 273.985 112.344 274.009 112.403C274.033 112.461 274.066 112.506 274.109 112.537C274.152 112.568 274.203 112.584 274.262 112.584C274.303 112.584 274.34 112.576 274.373 112.56C274.406 112.544 274.433 112.521 274.456 112.491C274.478 112.461 274.493 112.425 274.502 112.382H274.77C274.763 112.466 274.738 112.541 274.696 112.606C274.653 112.671 274.594 112.722 274.52 112.758C274.446 112.795 274.359 112.813 274.259 112.813ZM275.303 112.091V112.789H275.023V111.2H275.297V111.8H275.311C275.339 111.733 275.382 111.68 275.441 111.641C275.5 111.602 275.575 111.582 275.666 111.582C275.749 111.582 275.821 111.599 275.882 111.634C275.944 111.669 275.991 111.719 276.025 111.786C276.059 111.853 276.076 111.934 276.076 112.03V112.789H275.795V112.074C275.795 111.994 275.775 111.931 275.733 111.887C275.692 111.842 275.635 111.819 275.561 111.819C275.511 111.819 275.467 111.83 275.428 111.852C275.389 111.873 275.358 111.904 275.336 111.944C275.314 111.985 275.303 112.034 275.303 112.091ZM277.291 112.813C277.216 112.813 277.148 112.8 277.087 112.773C277.027 112.746 276.98 112.705 276.945 112.652C276.91 112.599 276.893 112.533 276.893 112.455C276.893 112.388 276.905 112.332 276.93 112.288C276.955 112.244 276.988 112.209 277.031 112.183C277.074 112.156 277.123 112.136 277.177 112.123C277.231 112.109 277.287 112.099 277.345 112.093C277.415 112.085 277.471 112.079 277.515 112.073C277.558 112.067 277.59 112.058 277.61 112.045C277.63 112.032 277.64 112.012 277.64 111.985V111.981C277.64 111.922 277.622 111.877 277.588 111.845C277.553 111.813 277.503 111.797 277.438 111.797C277.369 111.797 277.315 111.812 277.274 111.842C277.234 111.872 277.208 111.907 277.194 111.948L276.931 111.911C276.952 111.839 276.986 111.778 277.034 111.729C277.081 111.68 277.14 111.644 277.208 111.619C277.277 111.594 277.353 111.582 277.437 111.582C277.494 111.582 277.551 111.589 277.608 111.602C277.665 111.616 277.717 111.638 277.764 111.669C277.811 111.699 277.849 111.741 277.877 111.794C277.906 111.847 277.921 111.913 277.921 111.992V112.789H277.651V112.626H277.641C277.624 112.659 277.6 112.69 277.569 112.719C277.539 112.747 277.5 112.77 277.454 112.788C277.408 112.805 277.353 112.813 277.291 112.813ZM277.364 112.607C277.421 112.607 277.47 112.596 277.511 112.574C277.552 112.551 277.584 112.521 277.606 112.484C277.629 112.446 277.641 112.406 277.641 112.362V112.221C277.632 112.229 277.617 112.235 277.596 112.242C277.575 112.248 277.552 112.253 277.526 112.258C277.5 112.262 277.474 112.267 277.449 112.27C277.424 112.274 277.402 112.277 277.383 112.28C277.341 112.285 277.304 112.295 277.27 112.308C277.237 112.32 277.211 112.339 277.192 112.362C277.173 112.385 277.163 112.414 277.163 112.45C277.163 112.502 277.182 112.541 277.22 112.567C277.258 112.594 277.306 112.607 277.364 112.607ZM279.307 113.261C279.206 113.261 279.12 113.247 279.047 113.22C278.975 113.193 278.917 113.157 278.873 113.111C278.829 113.066 278.798 113.015 278.781 112.96L279.034 112.899C279.046 112.922 279.062 112.945 279.084 112.968C279.106 112.991 279.135 113.01 279.171 113.025C279.209 113.041 279.256 113.049 279.312 113.049C279.392 113.049 279.458 113.029 279.51 112.99C279.562 112.952 279.588 112.889 279.588 112.801V112.575H279.574C279.56 112.604 279.539 112.634 279.511 112.664C279.483 112.695 279.447 112.721 279.401 112.741C279.356 112.762 279.3 112.772 279.231 112.772C279.14 112.772 279.057 112.751 278.982 112.708C278.908 112.664 278.849 112.6 278.805 112.514C278.762 112.428 278.74 112.319 278.74 112.19C278.74 112.059 278.762 111.948 278.805 111.858C278.849 111.768 278.908 111.699 278.983 111.653C279.057 111.605 279.14 111.582 279.232 111.582C279.302 111.582 279.359 111.594 279.404 111.618C279.45 111.641 279.486 111.669 279.513 111.702C279.54 111.735 279.56 111.766 279.574 111.795H279.59V111.597H279.867V112.809C279.867 112.911 279.842 112.995 279.794 113.062C279.745 113.129 279.679 113.178 279.594 113.212C279.51 113.245 279.414 113.261 279.307 113.261ZM279.31 112.552C279.369 112.552 279.42 112.537 279.462 112.508C279.504 112.48 279.535 112.438 279.557 112.384C279.579 112.329 279.59 112.264 279.59 112.188C279.59 112.113 279.579 112.047 279.557 111.991C279.536 111.935 279.504 111.891 279.462 111.86C279.421 111.828 279.37 111.812 279.31 111.812C279.247 111.812 279.195 111.829 279.153 111.861C279.111 111.894 279.079 111.939 279.058 111.996C279.037 112.052 279.026 112.116 279.026 112.188C279.026 112.261 279.037 112.325 279.058 112.38C279.08 112.434 279.112 112.476 279.154 112.507C279.196 112.537 279.248 112.552 279.31 112.552ZM280.457 111.2V112.789H280.177V111.2H280.457ZM280.768 112.789V111.597H281.049V112.789H280.768ZM280.909 111.428C280.865 111.428 280.827 111.414 280.795 111.384C280.762 111.354 280.746 111.318 280.746 111.276C280.746 111.234 280.762 111.198 280.795 111.168C280.827 111.138 280.865 111.123 280.909 111.123C280.954 111.123 280.993 111.138 281.024 111.168C281.056 111.198 281.072 111.234 281.072 111.276C281.072 111.318 281.056 111.354 281.024 111.384C280.993 111.414 280.954 111.428 280.909 111.428ZM281.36 112.789V111.597H281.628V111.8H281.642C281.667 111.732 281.708 111.678 281.766 111.64C281.823 111.601 281.892 111.582 281.971 111.582C282.052 111.582 282.12 111.602 282.175 111.641C282.231 111.68 282.271 111.733 282.293 111.8H282.306C282.332 111.734 282.377 111.681 282.439 111.642C282.502 111.602 282.577 111.582 282.663 111.582C282.773 111.582 282.863 111.617 282.932 111.686C283.001 111.755 283.036 111.856 283.036 111.989V112.789H282.754V112.033C282.754 111.959 282.735 111.905 282.695 111.871C282.656 111.836 282.608 111.819 282.551 111.819C282.483 111.819 282.43 111.84 282.392 111.882C282.354 111.924 282.335 111.979 282.335 112.046V112.789H282.06V112.021C282.06 111.96 282.041 111.91 282.004 111.874C281.967 111.837 281.919 111.819 281.86 111.819C281.819 111.819 281.782 111.829 281.749 111.85C281.716 111.87 281.69 111.899 281.67 111.936C281.651 111.973 281.641 112.015 281.641 112.065V112.789H281.36ZM284.285 111.913L284.029 111.94C284.021 111.915 284.009 111.89 283.991 111.868C283.973 111.845 283.949 111.826 283.919 111.812C283.889 111.798 283.853 111.791 283.809 111.791C283.751 111.791 283.702 111.804 283.662 111.829C283.622 111.855 283.603 111.888 283.603 111.928C283.603 111.963 283.616 111.991 283.642 112.013C283.668 112.034 283.711 112.052 283.772 112.066L283.975 112.11C284.088 112.134 284.172 112.172 284.227 112.225C284.282 112.278 284.31 112.347 284.31 112.432C284.31 112.507 284.288 112.574 284.244 112.631C284.202 112.688 284.142 112.732 284.065 112.765C283.989 112.797 283.901 112.813 283.801 112.813C283.655 112.813 283.538 112.782 283.449 112.721C283.36 112.66 283.307 112.574 283.29 112.464L283.564 112.438C283.576 112.492 283.603 112.532 283.643 112.56C283.683 112.587 283.736 112.601 283.801 112.601C283.867 112.601 283.921 112.587 283.961 112.56C284.002 112.532 284.023 112.498 284.023 112.458C284.023 112.424 284.009 112.396 283.983 112.373C283.957 112.351 283.917 112.334 283.862 112.322L283.659 112.28C283.544 112.256 283.46 112.216 283.405 112.159C283.35 112.102 283.323 112.03 283.323 111.944C283.323 111.87 283.343 111.806 283.383 111.753C283.424 111.698 283.481 111.656 283.553 111.627C283.626 111.597 283.71 111.582 283.805 111.582C283.945 111.582 284.055 111.612 284.135 111.671C284.216 111.731 284.266 111.811 284.285 111.913ZM285.092 112.813C284.972 112.813 284.869 112.788 284.782 112.738C284.696 112.688 284.629 112.617 284.583 112.526C284.536 112.433 284.513 112.325 284.513 112.2C284.513 112.078 284.536 111.97 284.583 111.878C284.63 111.784 284.695 111.712 284.78 111.66C284.864 111.608 284.963 111.582 285.077 111.582C285.15 111.582 285.22 111.594 285.285 111.618C285.351 111.641 285.408 111.677 285.459 111.726C285.509 111.775 285.549 111.838 285.578 111.914C285.607 111.99 285.622 112.08 285.622 112.184V112.27H284.645V112.081H285.352C285.352 112.027 285.34 111.979 285.317 111.937C285.295 111.895 285.263 111.862 285.222 111.837C285.182 111.813 285.135 111.801 285.081 111.801C285.023 111.801 284.973 111.815 284.929 111.843C284.886 111.87 284.852 111.906 284.828 111.951C284.804 111.996 284.792 112.045 284.791 112.098V112.263C284.791 112.333 284.804 112.392 284.829 112.442C284.855 112.491 284.89 112.529 284.936 112.555C284.981 112.581 285.034 112.594 285.095 112.594C285.136 112.594 285.173 112.588 285.206 112.577C285.24 112.565 285.268 112.548 285.293 112.525C285.317 112.502 285.335 112.474 285.348 112.44L285.61 112.47C285.593 112.539 285.562 112.6 285.515 112.651C285.469 112.702 285.41 112.742 285.338 112.771C285.266 112.799 285.184 112.813 285.092 112.813ZM286.965 112.813C286.848 112.813 286.748 112.787 286.662 112.736C286.577 112.685 286.511 112.613 286.464 112.521C286.417 112.429 286.394 112.321 286.394 112.198C286.394 112.075 286.417 111.967 286.464 111.875C286.511 111.782 286.577 111.71 286.662 111.659C286.748 111.608 286.848 111.582 286.965 111.582C287.081 111.582 287.182 111.608 287.268 111.659C287.353 111.71 287.419 111.782 287.465 111.875C287.512 111.967 287.536 112.075 287.536 112.198C287.536 112.321 287.512 112.429 287.465 112.521C287.419 112.613 287.353 112.685 287.268 112.736C287.182 112.787 287.081 112.813 286.965 112.813ZM286.966 112.588C287.03 112.588 287.082 112.57 287.125 112.536C287.167 112.5 287.199 112.453 287.219 112.394C287.241 112.335 287.251 112.27 287.251 112.197C287.251 112.124 287.241 112.058 287.219 111.999C287.199 111.94 287.167 111.893 287.125 111.857C287.082 111.822 287.03 111.805 286.966 111.805C286.902 111.805 286.848 111.822 286.805 111.857C286.763 111.893 286.731 111.94 286.71 111.999C286.689 112.058 286.679 112.124 286.679 112.197C286.679 112.27 286.689 112.335 286.71 112.394C286.731 112.453 286.763 112.5 286.805 112.536C286.848 112.57 286.902 112.588 286.966 112.588ZM288.4 111.597V111.815H287.695V111.597H288.4ZM287.872 112.789V111.485C287.872 111.405 287.888 111.338 287.921 111.285C287.955 111.231 288 111.192 288.056 111.165C288.113 111.139 288.175 111.126 288.244 111.126C288.293 111.126 288.336 111.13 288.374 111.137C288.411 111.145 288.439 111.152 288.457 111.158L288.402 111.376C288.39 111.372 288.375 111.368 288.357 111.365C288.338 111.361 288.318 111.358 288.296 111.358C288.244 111.358 288.207 111.371 288.185 111.396C288.164 111.421 288.153 111.457 288.153 111.504V112.789H287.872ZM289.801 111.597V111.815H289.116V111.597H289.801ZM289.285 111.312H289.566V112.431C289.566 112.469 289.572 112.498 289.583 112.518C289.595 112.537 289.61 112.551 289.629 112.558C289.649 112.565 289.67 112.569 289.693 112.569C289.711 112.569 289.727 112.568 289.741 112.565C289.756 112.563 289.768 112.56 289.775 112.558L289.823 112.778C289.808 112.783 289.786 112.789 289.758 112.795C289.731 112.801 289.697 112.805 289.657 112.806C289.587 112.808 289.524 112.797 289.467 112.774C289.411 112.75 289.366 112.713 289.333 112.664C289.3 112.614 289.284 112.552 289.285 112.477V111.312ZM290.364 112.091V112.789H290.083V111.2H290.358V111.8H290.372C290.4 111.733 290.443 111.68 290.502 111.641C290.561 111.602 290.636 111.582 290.727 111.582C290.809 111.582 290.882 111.599 290.943 111.634C291.005 111.669 291.052 111.719 291.086 111.786C291.12 111.853 291.137 111.934 291.137 112.03V112.789H290.856V112.074C290.856 111.994 290.836 111.931 290.794 111.887C290.753 111.842 290.696 111.819 290.622 111.819C290.572 111.819 290.528 111.83 290.488 111.852C290.45 111.873 290.419 111.904 290.397 111.944C290.375 111.985 290.364 112.034 290.364 112.091ZM291.973 112.813C291.853 112.813 291.75 112.788 291.663 112.738C291.577 112.688 291.511 112.617 291.464 112.526C291.417 112.433 291.394 112.325 291.394 112.2C291.394 112.078 291.417 111.97 291.464 111.878C291.511 111.784 291.577 111.712 291.661 111.66C291.745 111.608 291.844 111.582 291.958 111.582C292.032 111.582 292.101 111.594 292.166 111.618C292.232 111.641 292.29 111.677 292.34 111.726C292.391 111.775 292.431 111.838 292.46 111.914C292.489 111.99 292.503 112.08 292.503 112.184V112.27H291.526V112.081H292.234C292.233 112.027 292.222 111.979 292.199 111.937C292.176 111.895 292.144 111.862 292.103 111.837C292.063 111.813 292.016 111.801 291.962 111.801C291.905 111.801 291.854 111.815 291.811 111.843C291.767 111.87 291.733 111.906 291.709 111.951C291.685 111.996 291.673 112.045 291.673 112.098V112.263C291.673 112.333 291.685 112.392 291.711 112.442C291.736 112.491 291.772 112.529 291.817 112.555C291.863 112.581 291.916 112.594 291.977 112.594C292.018 112.594 292.055 112.588 292.088 112.577C292.121 112.565 292.15 112.548 292.174 112.525C292.198 112.502 292.217 112.474 292.229 112.44L292.491 112.47C292.475 112.539 292.443 112.6 292.397 112.651C292.351 112.702 292.292 112.742 292.22 112.771C292.148 112.799 292.066 112.813 291.973 112.813ZM256.568 115.789V114.597H256.837V114.8H256.851C256.875 114.732 256.917 114.678 256.974 114.64C257.031 114.601 257.1 114.582 257.18 114.582C257.26 114.582 257.328 114.602 257.384 114.641C257.44 114.68 257.479 114.733 257.502 114.8H257.514C257.54 114.734 257.585 114.681 257.648 114.642C257.711 114.602 257.785 114.582 257.872 114.582C257.982 114.582 258.071 114.617 258.14 114.686C258.21 114.755 258.244 114.856 258.244 114.989V115.789H257.963V115.033C257.963 114.959 257.943 114.905 257.904 114.871C257.864 114.836 257.816 114.819 257.759 114.819C257.692 114.819 257.639 114.84 257.6 114.882C257.562 114.924 257.544 114.979 257.544 115.046V115.789H257.268V115.021C257.268 114.96 257.249 114.91 257.212 114.874C257.176 114.837 257.127 114.819 257.068 114.819C257.028 114.819 256.991 114.829 256.958 114.85C256.925 114.87 256.898 114.899 256.879 114.936C256.859 114.973 256.849 115.015 256.849 115.065V115.789H256.568ZM258.894 115.813C258.819 115.813 258.751 115.8 258.69 115.773C258.63 115.746 258.582 115.705 258.547 115.652C258.513 115.599 258.495 115.533 258.495 115.455C258.495 115.388 258.508 115.332 258.533 115.288C258.557 115.244 258.591 115.209 258.634 115.183C258.677 115.156 258.725 115.136 258.779 115.123C258.834 115.109 258.89 115.099 258.948 115.093C259.018 115.085 259.074 115.079 259.118 115.073C259.161 115.067 259.193 115.058 259.212 115.045C259.232 115.032 259.243 115.012 259.243 114.985V114.981C259.243 114.922 259.225 114.877 259.191 114.845C259.156 114.813 259.106 114.797 259.041 114.797C258.972 114.797 258.917 114.812 258.877 114.842C258.837 114.872 258.81 114.907 258.796 114.948L258.534 114.911C258.555 114.839 258.589 114.778 258.637 114.729C258.684 114.68 258.742 114.644 258.811 114.619C258.88 114.594 258.956 114.582 259.039 114.582C259.097 114.582 259.154 114.589 259.211 114.602C259.268 114.616 259.32 114.638 259.367 114.669C259.414 114.699 259.452 114.741 259.48 114.794C259.509 114.847 259.523 114.913 259.523 114.992V115.789H259.253V115.626H259.244C259.227 115.659 259.203 115.69 259.172 115.719C259.141 115.747 259.103 115.77 259.056 115.788C259.01 115.805 258.956 115.813 258.894 115.813ZM258.967 115.607C259.023 115.607 259.072 115.596 259.114 115.574C259.155 115.551 259.187 115.521 259.209 115.484C259.232 115.446 259.243 115.406 259.243 115.362V115.221C259.235 115.229 259.22 115.235 259.198 115.242C259.178 115.248 259.154 115.253 259.128 115.258C259.103 115.262 259.077 115.267 259.052 115.27C259.026 115.274 259.004 115.277 258.986 115.28C258.944 115.285 258.906 115.295 258.873 115.308C258.84 115.32 258.814 115.339 258.795 115.362C258.776 115.385 258.766 115.414 258.766 115.45C258.766 115.502 258.785 115.541 258.823 115.567C258.861 115.594 258.909 115.607 258.967 115.607ZM259.828 115.789V114.597H260.1V114.796H260.113C260.134 114.727 260.172 114.674 260.224 114.637C260.278 114.599 260.338 114.58 260.407 114.58C260.422 114.58 260.44 114.581 260.459 114.583C260.478 114.584 260.495 114.586 260.508 114.588V114.847C260.496 114.842 260.477 114.839 260.451 114.836C260.426 114.832 260.401 114.83 260.377 114.83C260.326 114.83 260.28 114.841 260.239 114.864C260.199 114.885 260.167 114.916 260.144 114.954C260.12 114.993 260.109 115.038 260.109 115.089V115.789H259.828ZM261.817 114.597L261.393 115.789H261.082L260.658 114.597H260.957L261.231 115.483H261.244L261.518 114.597H261.817ZM262.52 115.813C262.401 115.813 262.298 115.788 262.211 115.738C262.124 115.688 262.058 115.617 262.011 115.526C261.965 115.433 261.941 115.325 261.941 115.2C261.941 115.078 261.965 114.97 262.011 114.878C262.058 114.784 262.124 114.712 262.208 114.66C262.293 114.608 262.392 114.582 262.506 114.582C262.579 114.582 262.648 114.594 262.714 114.618C262.779 114.641 262.837 114.677 262.887 114.726C262.938 114.775 262.978 114.838 263.007 114.914C263.036 114.99 263.05 115.08 263.05 115.184V115.27H262.073V115.081H262.781C262.781 115.027 262.769 114.979 262.746 114.937C262.723 114.895 262.692 114.862 262.651 114.837C262.61 114.813 262.563 114.801 262.509 114.801C262.452 114.801 262.402 114.815 262.358 114.843C262.315 114.87 262.281 114.906 262.257 114.951C262.233 114.996 262.221 115.045 262.22 115.098V115.263C262.22 115.333 262.233 115.392 262.258 115.442C262.283 115.491 262.319 115.529 262.364 115.555C262.41 115.581 262.463 115.594 262.524 115.594C262.565 115.594 262.602 115.588 262.635 115.577C262.668 115.565 262.697 115.548 262.721 115.525C262.746 115.502 262.764 115.474 262.776 115.44L263.039 115.47C263.022 115.539 262.991 115.6 262.944 115.651C262.898 115.702 262.839 115.742 262.767 115.771C262.695 115.799 262.613 115.813 262.52 115.813ZM263.591 114.2V115.789H263.31V114.2H263.591ZM264.421 115.813C264.305 115.813 264.204 115.787 264.119 115.736C264.033 115.685 263.967 115.613 263.92 115.521C263.874 115.429 263.85 115.321 263.85 115.198C263.85 115.075 263.874 114.967 263.92 114.875C263.967 114.782 264.033 114.71 264.119 114.659C264.204 114.608 264.305 114.582 264.421 114.582C264.538 114.582 264.639 114.608 264.724 114.659C264.809 114.71 264.875 114.782 264.922 114.875C264.969 114.967 264.993 115.075 264.993 115.198C264.993 115.321 264.969 115.429 264.922 115.521C264.875 115.613 264.809 115.685 264.724 115.736C264.639 115.787 264.538 115.813 264.421 115.813ZM264.423 115.588C264.486 115.588 264.539 115.57 264.581 115.536C264.624 115.5 264.655 115.453 264.676 115.394C264.697 115.335 264.708 115.27 264.708 115.197C264.708 115.124 264.697 115.058 264.676 114.999C264.655 114.94 264.624 114.893 264.581 114.857C264.539 114.822 264.486 114.805 264.423 114.805C264.358 114.805 264.304 114.822 264.262 114.857C264.219 114.893 264.187 114.94 264.166 114.999C264.145 115.058 264.135 115.124 264.135 115.197C264.135 115.27 264.145 115.335 264.166 115.394C264.187 115.453 264.219 115.5 264.262 115.536C264.304 115.57 264.358 115.588 264.423 115.588ZM266.012 115.288V114.597H266.293V115.789H266.021V115.578H266.009C265.982 115.644 265.937 115.699 265.876 115.741C265.815 115.784 265.74 115.805 265.65 115.805C265.572 115.805 265.503 115.788 265.443 115.753C265.383 115.718 265.337 115.667 265.303 115.6C265.27 115.533 265.253 115.452 265.253 115.356V114.597H265.534V115.313C265.534 115.388 265.554 115.448 265.596 115.493C265.637 115.537 265.691 115.56 265.759 115.56C265.8 115.56 265.84 115.55 265.879 115.529C265.918 115.509 265.95 115.479 265.974 115.439C266 115.399 266.012 115.349 266.012 115.288ZM267.546 114.913L267.29 114.94C267.283 114.915 267.27 114.89 267.252 114.868C267.234 114.845 267.211 114.826 267.181 114.812C267.151 114.798 267.114 114.791 267.07 114.791C267.012 114.791 266.963 114.804 266.923 114.829C266.884 114.855 266.864 114.888 266.865 114.928C266.864 114.963 266.877 114.991 266.903 115.013C266.929 115.034 266.973 115.052 267.033 115.066L267.237 115.11C267.349 115.134 267.433 115.172 267.488 115.225C267.543 115.278 267.571 115.347 267.572 115.432C267.571 115.507 267.549 115.574 267.506 115.631C267.463 115.688 267.403 115.732 267.327 115.765C267.25 115.797 267.162 115.813 267.063 115.813C266.917 115.813 266.799 115.782 266.71 115.721C266.621 115.66 266.568 115.574 266.551 115.464L266.825 115.438C266.838 115.492 266.864 115.532 266.904 115.56C266.945 115.587 266.997 115.601 267.062 115.601C267.129 115.601 267.182 115.587 267.223 115.56C267.263 115.532 267.284 115.498 267.284 115.458C267.284 115.424 267.271 115.396 267.244 115.373C267.218 115.351 267.178 115.334 267.123 115.322L266.92 115.28C266.806 115.256 266.721 115.216 266.666 115.159C266.611 115.102 266.584 115.03 266.585 114.944C266.584 114.87 266.604 114.806 266.644 114.753C266.685 114.698 266.742 114.656 266.814 114.627C266.887 114.597 266.971 114.582 267.067 114.582C267.206 114.582 267.316 114.612 267.396 114.671C267.477 114.731 267.527 114.811 267.546 114.913ZM268.984 114.597V114.815H268.299V114.597H268.984ZM268.468 114.312H268.749V115.431C268.749 115.469 268.755 115.498 268.766 115.518C268.778 115.537 268.794 115.551 268.813 115.558C268.832 115.565 268.853 115.569 268.876 115.569C268.894 115.569 268.91 115.568 268.925 115.565C268.94 115.563 268.951 115.56 268.959 115.558L269.006 115.778C268.991 115.783 268.97 115.789 268.942 115.795C268.914 115.801 268.881 115.805 268.841 115.806C268.77 115.808 268.707 115.797 268.651 115.774C268.594 115.75 268.55 115.713 268.516 115.664C268.484 115.614 268.468 115.552 268.468 115.477V114.312ZM269.487 115.789L269.15 114.597H269.437L269.646 115.436H269.657L269.871 114.597H270.154L270.369 115.431H270.38L270.587 114.597H270.874L270.536 115.789H270.244L270.02 114.984H270.004L269.78 115.789H269.487ZM271.092 115.789V114.597H271.373V115.789H271.092ZM271.233 114.428C271.189 114.428 271.15 114.414 271.118 114.384C271.086 114.354 271.07 114.318 271.07 114.276C271.07 114.234 271.086 114.198 271.118 114.168C271.15 114.138 271.189 114.123 271.233 114.123C271.278 114.123 271.316 114.138 271.348 114.168C271.38 114.198 271.396 114.234 271.396 114.276C271.396 114.318 271.38 114.354 271.348 114.384C271.316 114.414 271.278 114.428 271.233 114.428ZM271.964 115.091V115.789H271.683V114.597H271.952V114.8H271.966C271.993 114.733 272.037 114.68 272.097 114.641C272.158 114.602 272.232 114.582 272.321 114.582C272.404 114.582 272.475 114.6 272.536 114.635C272.598 114.67 272.645 114.721 272.679 114.788C272.713 114.854 272.73 114.935 272.729 115.03V115.789H272.449V115.074C272.449 114.994 272.428 114.932 272.386 114.887C272.346 114.842 272.289 114.819 272.217 114.819C272.167 114.819 272.124 114.83 272.085 114.852C272.048 114.873 272.018 114.904 271.996 114.944C271.975 114.985 271.964 115.034 271.964 115.091ZM273.613 115.789V114.2H273.894V114.795H273.906C273.92 114.766 273.941 114.735 273.967 114.702C273.993 114.669 274.029 114.641 274.074 114.618C274.119 114.594 274.177 114.582 274.246 114.582C274.338 114.582 274.422 114.605 274.495 114.653C274.57 114.699 274.629 114.768 274.672 114.86C274.716 114.951 274.738 115.063 274.738 115.195C274.738 115.326 274.717 115.437 274.674 115.529C274.631 115.62 274.573 115.69 274.499 115.738C274.425 115.786 274.341 115.81 274.247 115.81C274.179 115.81 274.122 115.799 274.077 115.776C274.032 115.753 273.996 115.726 273.969 115.694C273.942 115.661 273.921 115.631 273.906 115.602H273.889V115.789H273.613ZM273.889 115.193C273.889 115.271 273.9 115.338 273.921 115.396C273.944 115.454 273.975 115.499 274.017 115.532C274.059 115.564 274.109 115.58 274.169 115.58C274.231 115.58 274.283 115.563 274.325 115.53C274.367 115.497 274.398 115.451 274.419 115.393C274.441 115.334 274.452 115.268 274.452 115.193C274.452 115.119 274.441 115.054 274.42 114.996C274.399 114.939 274.367 114.894 274.326 114.861C274.284 114.829 274.231 114.812 274.169 114.812C274.109 114.812 274.058 114.828 274.016 114.86C273.974 114.891 273.942 114.936 273.92 114.992C273.899 115.049 273.889 115.116 273.889 115.193ZM275.759 115.288V114.597H276.04V115.789H275.767V115.578H275.755C275.728 115.644 275.684 115.699 275.622 115.741C275.561 115.784 275.486 115.805 275.396 115.805C275.318 115.805 275.249 115.788 275.189 115.753C275.13 115.718 275.083 115.667 275.049 115.6C275.016 115.533 274.999 115.452 274.999 115.356V114.597H275.28V115.313C275.28 115.388 275.301 115.448 275.342 115.493C275.383 115.537 275.438 115.56 275.505 115.56C275.546 115.56 275.586 115.55 275.625 115.529C275.664 115.509 275.696 115.479 275.721 115.439C275.746 115.399 275.759 115.349 275.759 115.288ZM276.35 115.789V114.597H276.631V115.789H276.35ZM276.492 114.428C276.447 114.428 276.409 114.414 276.377 114.384C276.345 114.354 276.329 114.318 276.329 114.276C276.329 114.234 276.345 114.198 276.377 114.168C276.409 114.138 276.447 114.123 276.492 114.123C276.537 114.123 276.575 114.138 276.606 114.168C276.639 114.198 276.655 114.234 276.655 114.276C276.655 114.318 276.639 114.354 276.606 114.384C276.575 114.414 276.537 114.428 276.492 114.428ZM277.223 114.2V115.789H276.942V114.2H277.223ZM277.976 115.81C277.882 115.81 277.798 115.786 277.725 115.738C277.651 115.69 277.592 115.62 277.549 115.529C277.506 115.437 277.485 115.326 277.485 115.195C277.485 115.063 277.506 114.951 277.55 114.86C277.594 114.768 277.653 114.699 277.728 114.653C277.802 114.605 277.885 114.582 277.977 114.582C278.047 114.582 278.104 114.594 278.149 114.618C278.194 114.641 278.23 114.669 278.256 114.702C278.282 114.735 278.303 114.766 278.317 114.795H278.329V114.2H278.611V115.789H278.334V115.602H278.317C278.303 115.631 278.282 115.661 278.255 115.694C278.227 115.726 278.191 115.753 278.146 115.776C278.101 115.799 278.044 115.81 277.976 115.81ZM278.054 115.58C278.114 115.58 278.165 115.564 278.206 115.532C278.248 115.499 278.28 115.454 278.302 115.396C278.324 115.338 278.334 115.271 278.334 115.193C278.334 115.116 278.324 115.049 278.302 114.992C278.281 114.936 278.249 114.891 278.207 114.86C278.166 114.828 278.115 114.812 278.054 114.812C277.992 114.812 277.939 114.829 277.898 114.861C277.856 114.894 277.824 114.939 277.803 114.996C277.782 115.054 277.771 115.119 277.771 115.193C277.771 115.268 277.782 115.334 277.803 115.393C277.825 115.451 277.856 115.497 277.898 115.53C277.941 115.563 277.993 115.58 278.054 115.58ZM278.932 115.789V114.597H279.213V115.789H278.932ZM279.073 114.428C279.029 114.428 278.99 114.414 278.958 114.384C278.926 114.354 278.91 114.318 278.91 114.276C278.91 114.234 278.926 114.198 278.958 114.168C278.99 114.138 279.029 114.123 279.073 114.123C279.118 114.123 279.156 114.138 279.188 114.168C279.22 114.198 279.236 114.234 279.236 114.276C279.236 114.318 279.22 114.354 279.188 114.384C279.156 114.414 279.118 114.428 279.073 114.428ZM279.804 115.091V115.789H279.524V114.597H279.792V114.8H279.806C279.833 114.733 279.877 114.68 279.937 114.641C279.998 114.602 280.072 114.582 280.161 114.582C280.244 114.582 280.315 114.6 280.376 114.635C280.438 114.67 280.485 114.721 280.519 114.788C280.553 114.854 280.57 114.935 280.57 115.03V115.789H280.289V115.074C280.289 114.994 280.268 114.932 280.227 114.887C280.186 114.842 280.129 114.819 280.057 114.819C280.007 114.819 279.964 114.83 279.925 114.852C279.888 114.873 279.858 114.904 279.836 114.944C279.815 114.985 279.804 115.034 279.804 115.091ZM281.395 116.261C281.294 116.261 281.208 116.247 281.135 116.22C281.063 116.193 281.005 116.157 280.961 116.111C280.917 116.066 280.886 116.015 280.869 115.96L281.122 115.899C281.134 115.922 281.15 115.945 281.172 115.968C281.194 115.991 281.223 116.01 281.26 116.025C281.297 116.041 281.344 116.049 281.4 116.049C281.48 116.049 281.546 116.029 281.598 115.99C281.65 115.952 281.676 115.889 281.676 115.801V115.575H281.662C281.648 115.604 281.627 115.634 281.599 115.664C281.571 115.695 281.535 115.721 281.489 115.741C281.444 115.762 281.388 115.772 281.319 115.772C281.228 115.772 281.145 115.751 281.07 115.708C280.996 115.664 280.937 115.6 280.893 115.514C280.85 115.428 280.828 115.319 280.828 115.19C280.828 115.059 280.85 114.948 280.893 114.858C280.937 114.768 280.996 114.699 281.071 114.653C281.145 114.605 281.229 114.582 281.32 114.582C281.39 114.582 281.447 114.594 281.492 114.618C281.538 114.641 281.574 114.669 281.601 114.702C281.628 114.735 281.648 114.766 281.662 114.795H281.678V114.597H281.955V115.809C281.955 115.911 281.931 115.995 281.882 116.062C281.833 116.129 281.767 116.178 281.682 116.212C281.598 116.245 281.502 116.261 281.395 116.261ZM281.398 115.552C281.457 115.552 281.508 115.537 281.55 115.508C281.592 115.48 281.623 115.438 281.645 115.384C281.667 115.329 281.678 115.264 281.678 115.188C281.678 115.113 281.667 115.047 281.645 114.991C281.624 114.935 281.592 114.891 281.551 114.86C281.509 114.828 281.458 114.812 281.398 114.812C281.335 114.812 281.283 114.829 281.241 114.861C281.199 114.894 281.167 114.939 281.146 114.996C281.125 115.052 281.114 115.116 281.114 115.188C281.114 115.261 281.125 115.325 281.146 115.38C281.168 115.434 281.2 115.476 281.242 115.507C281.284 115.537 281.336 115.552 281.398 115.552Z" fill="#434960"/><g clip-path="url(#clip7_3202_39054)"><path d="M257.332 119.795L257.332 119.813C257.407 119.813 257.481 119.831 257.547 119.866C257.614 119.9 257.671 119.951 257.714 120.012L257.714 120.012L257.863 120.224L257.878 120.245L257.893 120.224L258.042 120.012L258.042 120.012C258.085 119.951 258.142 119.9 258.208 119.866C258.275 119.831 258.349 119.813 258.424 119.813L258.424 119.795M257.332 119.795V119.813C257.074 119.813 256.864 120.022 256.864 120.28C256.864 120.556 257.029 120.823 257.241 121.054C257.448 121.28 257.699 121.469 257.878 121.596C258.056 121.469 258.307 121.28 258.515 121.054C258.727 120.823 258.891 120.556 258.891 120.28C258.891 120.022 258.682 119.813 258.424 119.813V119.795M257.332 119.795C257.064 119.795 256.846 120.012 256.846 120.28C256.846 120.847 257.519 121.365 257.878 121.618C258.237 121.365 258.909 120.847 258.909 120.28C258.909 120.012 258.692 119.795 258.424 119.795M257.332 119.795L257.889 120.198M258.424 119.795C258.346 119.795 258.269 119.814 258.2 119.85C258.131 119.886 258.071 119.938 258.027 120.002L257.889 120.198M257.889 120.198L257.893 120.203L257.878 120.214L257.863 120.203L257.867 120.198L257.889 120.198ZM257.878 119.866C257.939 119.785 258.017 119.719 258.108 119.672C258.205 119.621 258.314 119.594 258.424 119.594C258.514 119.594 258.603 119.612 258.686 119.646C258.77 119.681 258.845 119.731 258.909 119.795C258.973 119.859 259.023 119.934 259.058 120.018C259.092 120.101 259.11 120.19 259.11 120.28C259.11 120.634 258.909 120.955 258.657 121.223C258.405 121.491 258.101 121.708 257.888 121.855L257.878 121.862L257.867 121.855C257.655 121.708 257.35 121.491 257.098 121.223C256.847 120.955 256.646 120.634 256.646 120.28C256.646 120.098 256.718 119.924 256.847 119.795C256.975 119.666 257.15 119.594 257.332 119.594L257.878 119.866ZM257.878 119.866C257.817 119.785 257.738 119.719 257.648 119.672C257.55 119.621 257.442 119.594 257.332 119.594V119.612L257.878 119.866Z" fill="#434960" stroke="#434960" stroke-width="0.03642"/></g><path d="M261.293 121.563H260.569L260.916 121.17C261.148 120.923 261.273 120.765 261.273 120.536C261.273 120.27 261.061 120.044 260.717 120.044C260.383 120.044 260.136 120.265 260.136 120.592H260.388C260.388 120.398 260.508 120.276 260.699 120.276C260.885 120.276 261.003 120.406 261.003 120.569C261.003 120.745 260.893 120.872 260.717 121.061L260.179 121.645V121.792H261.293V121.563ZM262.753 121.413V121.188H262.519V120.069H262.32L261.43 121.232V121.413H262.261V121.792H262.519V121.413H262.753ZM261.721 121.188L262.261 120.459V121.188H261.721Z" fill="#434960"/><g clip-path="url(#clip8_3202_39054)"><path d="M268.078 119.49C267.919 119.49 267.761 119.522 267.614 119.583C267.467 119.644 267.333 119.733 267.22 119.846C266.993 120.073 266.865 120.382 266.865 120.704C266.864 120.984 266.961 121.256 267.139 121.472L266.896 121.715C266.879 121.732 266.868 121.754 266.864 121.777C266.859 121.801 266.862 121.825 266.871 121.847C266.881 121.869 266.897 121.888 266.918 121.9C266.938 121.913 266.962 121.919 266.986 121.918H268.078C268.4 121.918 268.709 121.79 268.937 121.562C269.164 121.335 269.292 121.026 269.292 120.704C269.292 120.382 269.164 120.073 268.937 119.846C268.709 119.618 268.4 119.49 268.078 119.49V119.49ZM268.078 121.675H267.279L267.392 121.562C267.414 121.539 267.427 121.509 267.427 121.477C267.427 121.444 267.414 121.414 267.392 121.391C267.233 121.232 267.134 121.023 267.111 120.8C267.089 120.576 267.145 120.352 267.27 120.165C267.395 119.978 267.58 119.84 267.795 119.775C268.01 119.71 268.241 119.721 268.449 119.807C268.656 119.893 268.828 120.048 268.934 120.246C269.04 120.444 269.074 120.672 269.03 120.893C268.987 121.113 268.868 121.311 268.694 121.454C268.521 121.597 268.303 121.675 268.078 121.675V121.675Z" fill="#434960"/></g><path d="M271.981 121.234C271.981 120.847 271.649 120.615 271.252 120.701L271.759 120.069H271.466L270.946 120.712C270.793 120.9 270.719 121.035 270.719 121.247C270.719 121.56 270.984 121.815 271.348 121.815C271.718 121.815 271.981 121.56 271.981 121.234ZM270.989 121.239C270.989 121 271.15 120.877 271.351 120.877C271.555 120.877 271.713 121.015 271.713 121.242C271.713 121.456 271.555 121.599 271.351 121.599C271.15 121.599 270.989 121.456 270.989 121.239Z" fill="#434960"/></g></g><g filter="url(#filter4_dd_3202_39054)"><g clip-path="url(#clip9_3202_39054)"><rect x="254.238" y="127.533" width="51.8985" height="85.9504" rx="0.7284" fill="white"/><rect x="254.238" y="127.533" width="51.8985" height="65.9202" fill="url(#pattern3)"/></g></g><g filter="url(#filter5_dd_3202_39054)"><g clip-path="url(#clip10_3202_39054)"><rect x="309.777" y="57.1653" width="51.8985" height="85.5862" rx="0.7284" fill="white"/><rect x="309.777" y="57.1656" width="51.8985" height="65.9202" fill="url(#pattern4)"/><g clip-path="url(#clip11_3202_39054)"><rect x="355.301" y="58.6213" width="5.09834" height="5.09834" rx="2.54917" fill="white"/><path d="M358.724 60.2967V61.6075H357.413V60.2967H358.724ZM358.724 60.0782H357.413C357.293 60.0782 357.195 60.1765 357.195 60.2967V61.6075C357.195 61.7277 357.293 61.826 357.413 61.826H358.724C358.844 61.826 358.943 61.7277 358.943 61.6075V60.2967C358.943 60.1765 358.844 60.0782 358.724 60.0782ZM356.758 60.5151V62.0445C356.758 62.1646 356.856 62.2629 356.976 62.2629H358.506V62.0445H356.976V60.5151H356.758Z" fill="#141B38"/></g><path d="M312.05 125.559V125.317H313.318V125.559H312.827V126.906H312.541V125.559H312.05ZM313.853 126.208V126.906H313.572V125.317H313.846V125.917H313.86C313.888 125.85 313.931 125.797 313.99 125.758C314.049 125.719 314.124 125.699 314.215 125.699C314.298 125.699 314.37 125.716 314.431 125.751C314.493 125.786 314.541 125.836 314.574 125.903C314.608 125.97 314.625 126.051 314.625 126.148V126.906H314.345V126.191C314.345 126.111 314.324 126.048 314.282 126.004C314.242 125.959 314.184 125.936 314.11 125.936C314.061 125.936 314.016 125.947 313.977 125.969C313.938 125.99 313.907 126.021 313.885 126.061C313.863 126.102 313.853 126.151 313.853 126.208ZM315.461 126.93C315.342 126.93 315.239 126.905 315.152 126.855C315.065 126.805 314.999 126.734 314.952 126.643C314.906 126.551 314.882 126.442 314.882 126.317C314.882 126.195 314.906 126.087 314.952 125.995C314.999 125.902 315.065 125.829 315.149 125.777C315.234 125.725 315.333 125.699 315.447 125.699C315.52 125.699 315.589 125.711 315.655 125.735C315.72 125.758 315.778 125.794 315.828 125.843C315.879 125.892 315.919 125.955 315.948 126.031C315.977 126.107 315.991 126.197 315.991 126.301V126.387H315.014V126.198H315.722C315.722 126.144 315.71 126.096 315.687 126.054C315.664 126.012 315.633 125.979 315.592 125.954C315.551 125.93 315.504 125.918 315.45 125.918C315.393 125.918 315.343 125.932 315.299 125.96C315.256 125.987 315.222 126.023 315.197 126.068C315.174 126.113 315.162 126.162 315.161 126.215V126.38C315.161 126.45 315.174 126.509 315.199 126.559C315.224 126.608 315.26 126.646 315.305 126.672C315.351 126.698 315.404 126.711 315.465 126.711C315.506 126.711 315.543 126.705 315.576 126.694C315.609 126.682 315.638 126.665 315.662 126.642C315.687 126.619 315.705 126.591 315.717 126.557L315.98 126.587C315.963 126.656 315.932 126.717 315.885 126.768C315.839 126.82 315.78 126.859 315.708 126.888C315.636 126.916 315.554 126.93 315.461 126.93ZM316.815 126.906V125.715H317.083V125.917H317.097C317.122 125.849 317.163 125.795 317.221 125.757C317.278 125.718 317.347 125.699 317.426 125.699C317.507 125.699 317.575 125.719 317.63 125.758C317.686 125.797 317.725 125.85 317.748 125.917H317.761C317.787 125.851 317.832 125.798 317.894 125.759C317.957 125.719 318.032 125.699 318.118 125.699C318.228 125.699 318.318 125.734 318.387 125.803C318.456 125.872 318.491 125.973 318.491 126.106V126.906H318.209V126.15C318.209 126.076 318.19 126.022 318.15 125.988C318.111 125.953 318.063 125.936 318.006 125.936C317.938 125.936 317.885 125.957 317.847 125.999C317.809 126.041 317.79 126.096 317.79 126.163V126.906H317.515V126.138C317.515 126.077 317.496 126.028 317.459 125.991C317.422 125.954 317.374 125.936 317.314 125.936C317.274 125.936 317.237 125.946 317.204 125.967C317.171 125.987 317.145 126.016 317.125 126.053C317.105 126.09 317.096 126.133 317.096 126.182V126.906H316.815ZM319.318 126.93C319.201 126.93 319.1 126.904 319.015 126.853C318.93 126.802 318.863 126.73 318.816 126.638C318.77 126.546 318.746 126.438 318.746 126.315C318.746 126.192 318.77 126.084 318.816 125.992C318.863 125.899 318.93 125.827 319.015 125.776C319.1 125.725 319.201 125.699 319.318 125.699C319.434 125.699 319.535 125.725 319.62 125.776C319.706 125.827 319.772 125.899 319.818 125.992C319.865 126.084 319.889 126.192 319.889 126.315C319.889 126.438 319.865 126.546 319.818 126.638C319.772 126.73 319.706 126.802 319.62 126.853C319.535 126.904 319.434 126.93 319.318 126.93ZM319.319 126.705C319.382 126.705 319.435 126.687 319.477 126.653C319.52 126.618 319.551 126.57 319.572 126.511C319.593 126.453 319.604 126.387 319.604 126.314C319.604 126.241 319.593 126.175 319.572 126.117C319.551 126.057 319.52 126.01 319.477 125.974C319.435 125.939 319.382 125.922 319.319 125.922C319.254 125.922 319.201 125.939 319.158 125.974C319.115 126.01 319.084 126.057 319.062 126.117C319.042 126.175 319.031 126.241 319.031 126.314C319.031 126.387 319.042 126.453 319.062 126.511C319.084 126.57 319.115 126.618 319.158 126.653C319.201 126.687 319.254 126.705 319.319 126.705ZM320.149 126.906V125.715H320.421V125.913H320.434C320.456 125.844 320.493 125.791 320.546 125.754C320.599 125.716 320.66 125.697 320.728 125.697C320.743 125.697 320.761 125.698 320.78 125.7C320.8 125.701 320.816 125.703 320.829 125.705V125.964C320.817 125.959 320.798 125.956 320.772 125.953C320.747 125.949 320.722 125.947 320.698 125.947C320.647 125.947 320.601 125.958 320.56 125.981C320.52 126.002 320.488 126.033 320.465 126.071C320.442 126.11 320.43 126.155 320.43 126.206V126.906H320.149ZM321.318 126.208V126.906H321.037V125.715H321.306V125.917H321.32C321.347 125.85 321.391 125.797 321.451 125.758C321.511 125.719 321.586 125.699 321.675 125.699C321.757 125.699 321.829 125.717 321.89 125.752C321.952 125.787 321.999 125.838 322.033 125.905C322.067 125.971 322.084 126.052 322.083 126.148V126.906H321.802V126.191C321.802 126.111 321.782 126.049 321.74 126.004C321.699 125.959 321.643 125.936 321.57 125.936C321.521 125.936 321.478 125.947 321.439 125.969C321.401 125.99 321.372 126.021 321.35 126.061C321.329 126.102 321.318 126.151 321.318 126.208ZM322.391 126.906V125.715H322.672V126.906H322.391ZM322.532 125.545C322.487 125.545 322.449 125.531 322.417 125.501C322.385 125.471 322.369 125.435 322.369 125.393C322.369 125.351 322.385 125.315 322.417 125.285C322.449 125.255 322.487 125.24 322.532 125.24C322.577 125.24 322.615 125.255 322.647 125.285C322.679 125.315 322.695 125.351 322.695 125.393C322.695 125.435 322.679 125.471 322.647 125.501C322.615 125.531 322.577 125.545 322.532 125.545ZM323.263 126.208V126.906H322.982V125.715H323.251V125.917H323.265C323.292 125.85 323.336 125.797 323.396 125.758C323.456 125.719 323.531 125.699 323.62 125.699C323.702 125.699 323.774 125.717 323.835 125.752C323.897 125.787 323.944 125.838 323.978 125.905C324.012 125.971 324.029 126.052 324.028 126.148V126.906H323.747V126.191C323.747 126.111 323.727 126.049 323.685 126.004C323.645 125.959 323.588 125.936 323.515 125.936C323.466 125.936 323.423 125.947 323.384 125.969C323.347 125.99 323.317 126.021 323.295 126.061C323.274 126.102 323.263 126.151 323.263 126.208ZM324.854 127.378C324.753 127.378 324.667 127.365 324.594 127.337C324.522 127.31 324.464 127.274 324.42 127.229C324.376 127.183 324.345 127.133 324.328 127.077L324.581 127.016C324.592 127.039 324.609 127.062 324.631 127.085C324.652 127.108 324.682 127.127 324.718 127.142C324.756 127.158 324.802 127.166 324.859 127.166C324.939 127.166 325.004 127.146 325.057 127.107C325.109 127.069 325.135 127.006 325.135 126.918V126.692H325.121C325.107 126.721 325.085 126.751 325.057 126.782C325.03 126.812 324.994 126.838 324.948 126.858C324.903 126.879 324.846 126.889 324.778 126.889C324.687 126.889 324.604 126.868 324.529 126.825C324.455 126.782 324.396 126.717 324.352 126.631C324.309 126.545 324.287 126.436 324.287 126.307C324.287 126.176 324.309 126.065 324.352 125.975C324.396 125.885 324.455 125.816 324.53 125.77C324.604 125.723 324.687 125.699 324.779 125.699C324.849 125.699 324.906 125.711 324.951 125.735C324.997 125.758 325.033 125.786 325.06 125.819C325.087 125.852 325.107 125.883 325.121 125.912H325.137V125.715H325.414V126.926C325.414 127.028 325.389 127.112 325.341 127.179C325.292 127.246 325.226 127.295 325.141 127.329C325.057 127.362 324.961 127.378 324.854 127.378ZM324.857 126.669C324.916 126.669 324.967 126.655 325.009 126.626C325.051 126.597 325.082 126.555 325.104 126.501C325.126 126.446 325.137 126.381 325.137 126.305C325.137 126.23 325.126 126.164 325.104 126.108C325.083 126.052 325.051 126.008 325.009 125.977C324.968 125.945 324.917 125.929 324.857 125.929C324.794 125.929 324.742 125.946 324.7 125.978C324.658 126.011 324.626 126.056 324.605 126.113C324.584 126.169 324.573 126.233 324.573 126.305C324.573 126.378 324.584 126.442 324.605 126.497C324.627 126.551 324.659 126.593 324.701 126.624C324.743 126.654 324.795 126.669 324.857 126.669ZM327.229 126.03L326.973 126.058C326.966 126.032 326.953 126.007 326.935 125.985C326.918 125.962 326.894 125.943 326.864 125.929C326.834 125.916 326.797 125.909 326.754 125.909C326.695 125.909 326.646 125.921 326.606 125.947C326.567 125.972 326.548 126.005 326.548 126.045C326.548 126.08 326.56 126.108 326.586 126.13C326.612 126.151 326.656 126.169 326.716 126.183L326.92 126.227C327.033 126.251 327.116 126.29 327.171 126.342C327.227 126.395 327.254 126.464 327.255 126.55C327.254 126.625 327.232 126.691 327.189 126.748C327.146 126.805 327.086 126.85 327.01 126.882C326.933 126.914 326.845 126.93 326.746 126.93C326.6 126.93 326.483 126.899 326.394 126.838C326.305 126.777 326.252 126.691 326.235 126.581L326.508 126.555C326.521 126.609 326.547 126.649 326.588 126.677C326.628 126.704 326.68 126.718 326.745 126.718C326.812 126.718 326.865 126.704 326.906 126.677C326.947 126.649 326.967 126.615 326.967 126.575C326.967 126.541 326.954 126.513 326.927 126.491C326.902 126.468 326.861 126.451 326.806 126.439L326.603 126.397C326.489 126.373 326.404 126.333 326.349 126.276C326.295 126.219 326.267 126.148 326.268 126.061C326.267 125.987 326.287 125.924 326.328 125.87C326.369 125.815 326.425 125.774 326.498 125.744C326.571 125.714 326.655 125.699 326.75 125.699C326.889 125.699 326.999 125.729 327.08 125.788C327.16 125.848 327.21 125.928 327.229 126.03ZM328.268 126.405V125.715H328.549V126.906H328.277V126.695H328.264C328.237 126.761 328.193 126.816 328.132 126.858C328.071 126.901 327.995 126.922 327.906 126.922C327.828 126.922 327.759 126.905 327.699 126.87C327.639 126.835 327.593 126.784 327.559 126.717C327.525 126.65 327.508 126.569 327.508 126.473V125.715H327.789V126.43C327.789 126.506 327.81 126.566 327.851 126.61C327.893 126.655 327.947 126.677 328.014 126.677C328.056 126.677 328.096 126.667 328.135 126.647C328.173 126.626 328.205 126.596 328.23 126.556C328.255 126.516 328.268 126.466 328.268 126.405ZM329.141 126.208V126.906H328.86V125.715H329.128V125.917H329.142C329.17 125.85 329.213 125.797 329.273 125.758C329.334 125.719 329.409 125.699 329.498 125.699C329.58 125.699 329.652 125.717 329.713 125.752C329.774 125.787 329.822 125.838 329.855 125.905C329.89 125.971 329.906 126.052 329.906 126.148V126.906H329.625V126.191C329.625 126.111 329.604 126.049 329.563 126.004C329.522 125.959 329.465 125.936 329.393 125.936C329.344 125.936 329.3 125.947 329.262 125.969C329.224 125.99 329.194 126.021 329.172 126.061C329.151 126.102 329.141 126.151 329.141 126.208ZM331.023 126.906L330.686 125.715H330.973L331.182 126.553H331.193L331.407 125.715H331.69L331.905 126.548H331.916L332.123 125.715H332.41L332.072 126.906H331.78L331.556 126.101H331.54L331.316 126.906H331.023ZM332.971 126.931C332.895 126.931 332.827 126.917 332.767 126.89C332.707 126.863 332.659 126.822 332.624 126.769C332.589 126.716 332.572 126.65 332.572 126.572C332.572 126.505 332.584 126.449 332.609 126.405C332.634 126.361 332.668 126.326 332.711 126.3C332.754 126.273 332.802 126.253 332.856 126.24C332.91 126.226 332.966 126.216 333.024 126.21C333.094 126.202 333.151 126.196 333.194 126.19C333.238 126.184 333.269 126.175 333.289 126.162C333.309 126.149 333.319 126.129 333.319 126.103V126.098C333.319 126.039 333.302 125.994 333.267 125.962C333.233 125.93 333.183 125.914 333.117 125.914C333.049 125.914 332.994 125.929 332.954 125.959C332.914 125.989 332.887 126.024 332.873 126.065L332.611 126.028C332.631 125.956 332.666 125.895 332.713 125.846C332.761 125.797 332.819 125.761 332.888 125.736C332.957 125.711 333.033 125.699 333.116 125.699C333.173 125.699 333.23 125.706 333.287 125.719C333.344 125.733 333.396 125.755 333.443 125.786C333.49 125.816 333.528 125.858 333.557 125.911C333.586 125.964 333.6 126.03 333.6 126.109V126.906H333.33V126.743H333.321C333.304 126.776 333.28 126.807 333.249 126.836C333.218 126.864 333.18 126.887 333.133 126.905C333.087 126.922 333.033 126.931 332.971 126.931ZM333.044 126.724C333.1 126.724 333.149 126.713 333.19 126.691C333.232 126.668 333.264 126.638 333.286 126.601C333.309 126.563 333.32 126.523 333.32 126.479V126.338C333.311 126.346 333.296 126.352 333.275 126.359C333.254 126.365 333.231 126.37 333.205 126.375C333.179 126.38 333.154 126.384 333.128 126.387C333.103 126.391 333.081 126.394 333.062 126.397C333.02 126.402 332.983 126.412 332.95 126.425C332.917 126.438 332.891 126.456 332.871 126.479C332.852 126.502 332.843 126.531 332.843 126.567C332.843 126.619 332.862 126.658 332.899 126.685C332.937 126.711 332.985 126.724 333.044 126.724ZM333.904 126.906V125.715H334.177V125.913H334.189C334.211 125.844 334.248 125.791 334.301 125.754C334.354 125.716 334.415 125.697 334.483 125.697C334.499 125.697 334.516 125.698 334.535 125.7C334.555 125.701 334.571 125.703 334.584 125.705V125.964C334.572 125.959 334.553 125.956 334.528 125.953C334.502 125.949 334.478 125.947 334.454 125.947C334.403 125.947 334.357 125.958 334.316 125.981C334.275 126.002 334.244 126.033 334.22 126.071C334.197 126.11 334.185 126.155 334.185 126.206V126.906H333.904ZM334.793 126.906V125.715H335.061V125.917H335.075C335.1 125.849 335.141 125.795 335.199 125.757C335.256 125.718 335.324 125.699 335.404 125.699C335.485 125.699 335.553 125.719 335.608 125.758C335.664 125.797 335.703 125.85 335.726 125.917H335.739C335.765 125.851 335.809 125.798 335.872 125.759C335.935 125.719 336.01 125.699 336.096 125.699C336.206 125.699 336.296 125.734 336.365 125.803C336.434 125.872 336.469 125.973 336.469 126.106V126.906H336.187V126.15C336.187 126.076 336.167 126.022 336.128 125.988C336.089 125.953 336.041 125.936 335.984 125.936C335.916 125.936 335.863 125.957 335.825 125.999C335.787 126.041 335.768 126.096 335.768 126.163V126.906H335.493V126.138C335.493 126.077 335.474 126.028 335.437 125.991C335.4 125.954 335.352 125.936 335.292 125.936C335.252 125.936 335.215 125.946 335.182 125.967C335.149 125.987 335.123 126.016 335.103 126.053C335.083 126.09 335.074 126.133 335.074 126.182V126.906H334.793ZM337.718 126.03L337.462 126.058C337.454 126.032 337.442 126.007 337.424 125.985C337.406 125.962 337.382 125.943 337.352 125.929C337.322 125.916 337.285 125.909 337.242 125.909C337.184 125.909 337.134 125.921 337.095 125.947C337.055 125.972 337.036 126.005 337.036 126.045C337.036 126.08 337.049 126.108 337.074 126.13C337.101 126.151 337.144 126.169 337.205 126.183L337.408 126.227C337.521 126.251 337.605 126.29 337.66 126.342C337.715 126.395 337.743 126.464 337.743 126.55C337.743 126.625 337.721 126.691 337.677 126.748C337.634 126.805 337.575 126.85 337.498 126.882C337.422 126.914 337.334 126.93 337.234 126.93C337.088 126.93 336.971 126.899 336.882 126.838C336.793 126.777 336.74 126.691 336.723 126.581L336.997 126.555C337.009 126.609 337.036 126.649 337.076 126.677C337.116 126.704 337.169 126.718 337.234 126.718C337.3 126.718 337.354 126.704 337.394 126.677C337.435 126.649 337.455 126.615 337.455 126.575C337.455 126.541 337.442 126.513 337.416 126.491C337.39 126.468 337.35 126.451 337.295 126.439L337.091 126.397C336.977 126.373 336.893 126.333 336.838 126.276C336.783 126.219 336.756 126.148 336.756 126.061C336.756 125.987 336.776 125.924 336.816 125.87C336.857 125.815 336.914 125.774 336.986 125.744C337.059 125.714 337.143 125.699 337.238 125.699C337.378 125.699 337.488 125.729 337.568 125.788C337.649 125.848 337.699 125.928 337.718 126.03ZM339.156 125.715V125.932H338.471V125.715H339.156ZM338.64 125.429H338.921V126.548C338.921 126.586 338.926 126.615 338.938 126.635C338.95 126.655 338.965 126.668 338.984 126.675C339.004 126.682 339.025 126.686 339.048 126.686C339.066 126.686 339.082 126.685 339.096 126.682C339.111 126.68 339.123 126.677 339.13 126.675L339.178 126.895C339.163 126.9 339.141 126.906 339.113 126.912C339.086 126.918 339.052 126.922 339.012 126.923C338.942 126.925 338.879 126.914 338.822 126.891C338.766 126.867 338.721 126.83 338.688 126.781C338.655 126.731 338.639 126.669 338.64 126.595V125.429ZM339.719 126.208V126.906H339.438V125.317H339.713V125.917H339.727C339.755 125.85 339.798 125.797 339.857 125.758C339.916 125.719 339.991 125.699 340.082 125.699C340.164 125.699 340.237 125.716 340.298 125.751C340.36 125.786 340.407 125.836 340.441 125.903C340.475 125.97 340.492 126.051 340.492 126.148V126.906H340.211V126.191C340.211 126.111 340.19 126.048 340.149 126.004C340.108 125.959 340.051 125.936 339.977 125.936C339.927 125.936 339.883 125.947 339.843 125.969C339.805 125.99 339.774 126.021 339.752 126.061C339.73 126.102 339.719 126.151 339.719 126.208ZM341.328 126.93C341.208 126.93 341.105 126.905 341.018 126.855C340.932 126.805 340.865 126.734 340.819 126.643C340.772 126.551 340.749 126.442 340.749 126.317C340.749 126.195 340.772 126.087 340.819 125.995C340.866 125.902 340.932 125.829 341.016 125.777C341.1 125.725 341.199 125.699 341.313 125.699C341.387 125.699 341.456 125.711 341.521 125.735C341.587 125.758 341.645 125.794 341.695 125.843C341.746 125.892 341.786 125.955 341.814 126.031C341.843 126.107 341.858 126.197 341.858 126.301V126.387H340.881V126.198H341.589C341.588 126.144 341.577 126.096 341.554 126.054C341.531 126.012 341.499 125.979 341.458 125.954C341.418 125.93 341.371 125.918 341.317 125.918C341.26 125.918 341.209 125.932 341.166 125.96C341.122 125.987 341.088 126.023 341.064 126.068C341.04 126.113 341.028 126.162 341.028 126.215V126.38C341.028 126.45 341.04 126.509 341.066 126.559C341.091 126.608 341.126 126.646 341.172 126.672C341.218 126.698 341.271 126.711 341.332 126.711C341.373 126.711 341.41 126.705 341.443 126.694C341.476 126.682 341.505 126.665 341.529 126.642C341.553 126.619 341.572 126.591 341.584 126.557L341.846 126.587C341.83 126.656 341.798 126.717 341.752 126.768C341.706 126.82 341.647 126.859 341.575 126.888C341.503 126.916 341.421 126.93 341.328 126.93ZM342.681 126.906V125.715H342.95V125.917H342.964C342.989 125.849 343.03 125.795 343.087 125.757C343.145 125.718 343.213 125.699 343.293 125.699C343.374 125.699 343.442 125.719 343.497 125.758C343.553 125.797 343.592 125.85 343.615 125.917H343.627C343.654 125.851 343.698 125.798 343.761 125.759C343.824 125.719 343.899 125.699 343.985 125.699C344.095 125.699 344.184 125.734 344.254 125.803C344.323 125.872 344.357 125.973 344.357 126.106V126.906H344.076V126.15C344.076 126.076 344.056 126.022 344.017 125.988C343.978 125.953 343.929 125.936 343.872 125.936C343.805 125.936 343.752 125.957 343.713 125.999C343.676 126.041 343.657 126.096 343.657 126.163V126.906H343.381V126.138C343.381 126.077 343.363 126.028 343.325 125.991C343.289 125.954 343.241 125.936 343.181 125.936C343.141 125.936 343.104 125.946 343.071 125.967C343.038 125.987 343.011 126.016 342.992 126.053C342.972 126.09 342.962 126.133 342.962 126.182V126.906H342.681ZM345.184 126.93C345.068 126.93 344.967 126.904 344.882 126.853C344.796 126.802 344.73 126.73 344.683 126.638C344.636 126.546 344.613 126.438 344.613 126.315C344.613 126.192 344.636 126.084 344.683 125.992C344.73 125.899 344.796 125.827 344.882 125.776C344.967 125.725 345.068 125.699 345.184 125.699C345.301 125.699 345.402 125.725 345.487 125.776C345.572 125.827 345.638 125.899 345.685 125.992C345.732 126.084 345.755 126.192 345.755 126.315C345.755 126.438 345.732 126.546 345.685 126.638C345.638 126.73 345.572 126.802 345.487 126.853C345.402 126.904 345.301 126.93 345.184 126.93ZM345.186 126.705C345.249 126.705 345.302 126.687 345.344 126.653C345.387 126.618 345.418 126.57 345.439 126.511C345.46 126.453 345.471 126.387 345.471 126.314C345.471 126.241 345.46 126.175 345.439 126.117C345.418 126.057 345.387 126.01 345.344 125.974C345.302 125.939 345.249 125.922 345.186 125.922C345.121 125.922 345.067 125.939 345.024 125.974C344.982 126.01 344.95 126.057 344.929 126.117C344.908 126.175 344.898 126.241 344.898 126.314C344.898 126.387 344.908 126.453 344.929 126.511C344.95 126.57 344.982 126.618 345.024 126.653C345.067 126.687 345.121 126.705 345.186 126.705ZM346.775 126.405V125.715H347.056V126.906H346.784V126.695H346.771C346.745 126.761 346.7 126.816 346.639 126.858C346.578 126.901 346.502 126.922 346.413 126.922C346.335 126.922 346.266 126.905 346.206 126.87C346.146 126.835 346.1 126.784 346.066 126.717C346.032 126.65 346.016 126.569 346.016 126.473V125.715H346.297V126.43C346.297 126.506 346.317 126.566 346.359 126.61C346.4 126.655 346.454 126.677 346.522 126.677C346.563 126.677 346.603 126.667 346.642 126.647C346.681 126.626 346.712 126.596 346.737 126.556C346.763 126.516 346.775 126.466 346.775 126.405ZM347.648 126.208V126.906H347.367V125.715H347.635V125.917H347.649C347.677 125.85 347.721 125.797 347.781 125.758C347.841 125.719 347.916 125.699 348.005 125.699C348.087 125.699 348.159 125.717 348.22 125.752C348.281 125.787 348.329 125.838 348.363 125.905C348.397 125.971 348.414 126.052 348.413 126.148V126.906H348.132V126.191C348.132 126.111 348.111 126.049 348.07 126.004C348.029 125.959 347.973 125.936 347.9 125.936C347.851 125.936 347.807 125.947 347.769 125.969C347.731 125.99 347.701 126.021 347.68 126.061C347.658 126.102 347.648 126.151 347.648 126.208ZM349.316 125.715V125.932H348.63V125.715H349.316ZM348.8 125.429H349.08V126.548C349.08 126.586 349.086 126.615 349.098 126.635C349.109 126.655 349.125 126.668 349.144 126.675C349.163 126.682 349.184 126.686 349.208 126.686C349.225 126.686 349.241 126.685 349.256 126.682C349.271 126.68 349.282 126.677 349.29 126.675L349.337 126.895C349.322 126.9 349.301 126.906 349.273 126.912C349.246 126.918 349.212 126.922 349.172 126.923C349.102 126.925 349.038 126.914 348.982 126.891C348.926 126.867 348.881 126.83 348.848 126.781C348.815 126.731 348.799 126.669 348.8 126.595V125.429ZM349.915 126.931C349.84 126.931 349.772 126.917 349.711 126.89C349.651 126.863 349.604 126.822 349.569 126.769C349.534 126.716 349.517 126.65 349.517 126.572C349.517 126.505 349.529 126.449 349.554 126.405C349.579 126.361 349.612 126.326 349.655 126.3C349.698 126.273 349.747 126.253 349.801 126.24C349.855 126.226 349.911 126.216 349.969 126.21C350.039 126.202 350.095 126.196 350.139 126.19C350.182 126.184 350.214 126.175 350.234 126.162C350.254 126.149 350.264 126.129 350.264 126.103V126.098C350.264 126.039 350.246 125.994 350.212 125.962C350.177 125.93 350.127 125.914 350.062 125.914C349.993 125.914 349.939 125.929 349.898 125.959C349.858 125.989 349.832 126.024 349.818 126.065L349.555 126.028C349.576 125.956 349.61 125.895 349.658 125.846C349.705 125.797 349.764 125.761 349.832 125.736C349.901 125.711 349.977 125.699 350.061 125.699C350.118 125.699 350.175 125.706 350.232 125.719C350.289 125.733 350.341 125.755 350.388 125.786C350.435 125.816 350.473 125.858 350.501 125.911C350.53 125.964 350.545 126.03 350.545 126.109V126.906H350.275V126.743H350.265C350.248 126.776 350.224 126.807 350.193 126.836C350.163 126.864 350.124 126.887 350.078 126.905C350.032 126.922 349.977 126.931 349.915 126.931ZM349.988 126.724C350.045 126.724 350.094 126.713 350.135 126.691C350.176 126.668 350.208 126.638 350.23 126.601C350.253 126.563 350.265 126.523 350.265 126.479V126.338C350.256 126.346 350.241 126.352 350.22 126.359C350.199 126.365 350.176 126.37 350.15 126.375C350.124 126.38 350.098 126.384 350.073 126.387C350.048 126.391 350.026 126.394 350.007 126.397C349.965 126.402 349.928 126.412 349.894 126.425C349.861 126.438 349.835 126.456 349.816 126.479C349.797 126.502 349.787 126.531 349.787 126.567C349.787 126.619 349.806 126.658 349.844 126.685C349.882 126.711 349.93 126.724 349.988 126.724ZM350.849 126.906V125.715H351.13V126.906H350.849ZM350.99 125.545C350.946 125.545 350.907 125.531 350.875 125.501C350.843 125.471 350.827 125.435 350.827 125.393C350.827 125.351 350.843 125.315 350.875 125.285C350.907 125.255 350.946 125.24 350.99 125.24C351.035 125.24 351.074 125.255 351.105 125.285C351.137 125.315 351.153 125.351 351.153 125.393C351.153 125.435 351.137 125.471 351.105 125.501C351.074 125.531 351.035 125.545 350.99 125.545ZM351.722 126.208V126.906H351.441V125.715H351.709V125.917H351.723C351.751 125.85 351.794 125.797 351.854 125.758C351.915 125.719 351.99 125.699 352.079 125.699C352.161 125.699 352.232 125.717 352.293 125.752C352.355 125.787 352.403 125.838 352.436 125.905C352.47 125.971 352.487 126.052 352.487 126.148V126.906H352.206V126.191C352.206 126.111 352.185 126.049 352.144 126.004C352.103 125.959 352.046 125.936 351.974 125.936C351.925 125.936 351.881 125.947 351.843 125.969C351.805 125.99 351.775 126.021 351.753 126.061C351.732 126.102 351.722 126.151 351.722 126.208ZM353.953 125.715V125.932H353.268V125.715H353.953ZM353.437 125.429H353.718V126.548C353.718 126.586 353.724 126.615 353.735 126.635C353.747 126.655 353.763 126.668 353.782 126.675C353.801 126.682 353.822 126.686 353.845 126.686C353.863 126.686 353.879 126.685 353.893 126.682C353.908 126.68 353.92 126.677 353.928 126.675L353.975 126.895C353.96 126.9 353.938 126.906 353.911 126.912C353.883 126.918 353.849 126.922 353.81 126.923C353.739 126.925 353.676 126.914 353.62 126.891C353.563 126.867 353.518 126.83 353.485 126.781C353.453 126.731 353.437 126.669 353.437 126.595V125.429ZM354.717 126.93C354.601 126.93 354.5 126.904 354.414 126.853C354.329 126.802 354.263 126.73 354.216 126.638C354.169 126.546 354.146 126.438 354.146 126.315C354.146 126.192 354.169 126.084 354.216 125.992C354.263 125.899 354.329 125.827 354.414 125.776C354.5 125.725 354.601 125.699 354.717 125.699C354.833 125.699 354.934 125.725 355.02 125.776C355.105 125.827 355.171 125.899 355.218 125.992C355.265 126.084 355.288 126.192 355.288 126.315C355.288 126.438 355.265 126.546 355.218 126.638C355.171 126.73 355.105 126.802 355.02 126.853C354.934 126.904 354.833 126.93 354.717 126.93ZM354.719 126.705C354.782 126.705 354.835 126.687 354.877 126.653C354.919 126.618 354.951 126.57 354.972 126.511C354.993 126.453 355.003 126.387 355.003 126.314C355.003 126.241 354.993 126.175 354.972 126.117C354.951 126.057 354.919 126.01 354.877 125.974C354.835 125.939 354.782 125.922 354.719 125.922C354.654 125.922 354.6 125.939 354.557 125.974C354.515 126.01 354.483 126.057 354.462 126.117C354.441 126.175 354.431 126.241 354.431 126.314C354.431 126.387 354.441 126.453 354.462 126.511C354.483 126.57 354.515 126.618 354.557 126.653C354.6 126.687 354.654 126.705 354.719 126.705ZM355.548 127.353V125.715H355.825V125.912H355.841C355.856 125.883 355.876 125.852 355.902 125.819C355.929 125.786 355.964 125.758 356.009 125.735C356.054 125.711 356.112 125.699 356.182 125.699C356.274 125.699 356.357 125.723 356.431 125.77C356.505 125.816 356.564 125.885 356.608 125.977C356.652 126.068 356.674 126.18 356.674 126.312C356.674 126.443 356.652 126.554 356.609 126.646C356.566 126.737 356.508 126.807 356.434 126.855C356.36 126.903 356.276 126.927 356.182 126.927C356.114 126.927 356.058 126.916 356.013 126.893C355.968 126.871 355.931 126.843 355.904 126.811C355.877 126.778 355.856 126.748 355.841 126.719H355.829V127.353H355.548ZM355.824 126.31C355.824 126.388 355.835 126.455 355.857 126.513C355.879 126.571 355.911 126.616 355.952 126.649C355.994 126.681 356.045 126.697 356.104 126.697C356.166 126.697 356.218 126.68 356.26 126.647C356.302 126.614 356.334 126.568 356.355 126.51C356.376 126.451 356.387 126.385 356.387 126.31C356.387 126.237 356.377 126.171 356.356 126.113C356.334 126.056 356.303 126.011 356.261 125.978C356.219 125.946 356.167 125.929 356.104 125.929C356.044 125.929 355.993 125.945 355.951 125.977C355.909 126.008 355.878 126.053 355.856 126.11C355.835 126.166 355.824 126.233 355.824 126.31ZM312.45 129.931C312.375 129.931 312.307 129.917 312.246 129.89C312.186 129.863 312.139 129.822 312.103 129.769C312.069 129.716 312.051 129.65 312.051 129.572C312.051 129.505 312.064 129.449 312.089 129.405C312.113 129.361 312.147 129.326 312.19 129.3C312.233 129.273 312.282 129.253 312.335 129.24C312.39 129.226 312.446 129.216 312.504 129.21C312.574 129.202 312.63 129.196 312.674 129.19C312.717 129.184 312.749 129.175 312.768 129.162C312.789 129.149 312.799 129.129 312.799 129.103V129.098C312.799 129.039 312.781 128.994 312.747 128.962C312.712 128.93 312.662 128.914 312.597 128.914C312.528 128.914 312.473 128.929 312.433 128.959C312.393 128.989 312.366 129.024 312.352 129.065L312.09 129.028C312.111 128.956 312.145 128.895 312.193 128.846C312.24 128.797 312.298 128.761 312.367 128.736C312.436 128.711 312.512 128.699 312.595 128.699C312.653 128.699 312.71 128.706 312.767 128.719C312.824 128.733 312.876 128.755 312.923 128.786C312.97 128.816 313.008 128.858 313.036 128.911C313.065 128.964 313.08 129.03 313.08 129.109V129.906H312.81V129.743H312.8C312.783 129.776 312.759 129.807 312.728 129.836C312.697 129.864 312.659 129.887 312.612 129.905C312.566 129.922 312.512 129.931 312.45 129.931ZM312.523 129.724C312.58 129.724 312.628 129.713 312.67 129.691C312.711 129.668 312.743 129.638 312.765 129.601C312.788 129.563 312.799 129.523 312.799 129.479V129.338C312.791 129.346 312.776 129.352 312.754 129.359C312.734 129.365 312.71 129.37 312.685 129.375C312.659 129.38 312.633 129.384 312.608 129.387C312.582 129.391 312.56 129.394 312.542 129.397C312.5 129.402 312.462 129.412 312.429 129.425C312.396 129.438 312.37 129.456 312.351 129.479C312.332 129.502 312.322 129.531 312.322 129.567C312.322 129.619 312.341 129.658 312.379 129.685C312.417 129.711 312.465 129.724 312.523 129.724ZM314.326 129.03L314.07 129.058C314.063 129.032 314.05 129.007 314.032 128.985C314.014 128.962 313.99 128.943 313.96 128.929C313.93 128.916 313.894 128.909 313.85 128.909C313.792 128.909 313.743 128.921 313.703 128.947C313.663 128.972 313.644 129.005 313.645 129.045C313.644 129.08 313.657 129.108 313.683 129.13C313.709 129.151 313.752 129.169 313.813 129.183L314.016 129.227C314.129 129.251 314.213 129.29 314.268 129.342C314.323 129.395 314.351 129.464 314.352 129.55C314.351 129.625 314.329 129.691 314.286 129.748C314.243 129.805 314.183 129.85 314.106 129.882C314.03 129.914 313.942 129.93 313.842 129.93C313.697 129.93 313.579 129.899 313.49 129.838C313.401 129.777 313.348 129.691 313.331 129.581L313.605 129.555C313.617 129.609 313.644 129.649 313.684 129.677C313.725 129.704 313.777 129.718 313.842 129.718C313.908 129.718 313.962 129.704 314.002 129.677C314.043 129.649 314.064 129.615 314.064 129.575C314.064 129.541 314.05 129.513 314.024 129.491C313.998 129.468 313.958 129.451 313.903 129.439L313.7 129.397C313.585 129.373 313.501 129.333 313.446 129.276C313.391 129.219 313.364 129.148 313.364 129.061C313.364 128.987 313.384 128.924 313.424 128.87C313.465 128.815 313.522 128.774 313.594 128.744C313.667 128.714 313.751 128.699 313.846 128.699C313.986 128.699 314.096 128.729 314.176 128.788C314.257 128.848 314.307 128.928 314.326 129.03ZM315.169 129.906V128.715H315.45V129.906H315.169ZM315.31 128.545C315.266 128.545 315.227 128.531 315.195 128.501C315.163 128.471 315.147 128.435 315.147 128.393C315.147 128.351 315.163 128.315 315.195 128.285C315.227 128.255 315.266 128.24 315.31 128.24C315.355 128.24 315.393 128.255 315.425 128.285C315.457 128.315 315.473 128.351 315.473 128.393C315.473 128.435 315.457 128.471 315.425 128.501C315.393 128.531 315.355 128.545 315.31 128.545ZM316.356 128.715V128.932H315.671V128.715H316.356ZM315.84 128.429H316.121V129.548C316.121 129.586 316.126 129.615 316.138 129.635C316.15 129.655 316.165 129.668 316.184 129.675C316.203 129.682 316.225 129.686 316.248 129.686C316.265 129.686 316.281 129.685 316.296 129.682C316.311 129.68 316.322 129.677 316.33 129.675L316.377 129.895C316.362 129.9 316.341 129.906 316.313 129.912C316.286 129.918 316.252 129.922 316.212 129.923C316.142 129.925 316.078 129.914 316.022 129.891C315.966 129.867 315.921 129.83 315.888 129.781C315.855 129.731 315.839 129.669 315.84 129.595V128.429ZM317.696 129.93C317.577 129.93 317.475 129.904 317.39 129.851C317.305 129.799 317.239 129.727 317.193 129.635C317.148 129.542 317.125 129.436 317.125 129.315C317.125 129.194 317.148 129.087 317.195 128.995C317.242 128.902 317.307 128.829 317.392 128.777C317.477 128.725 317.578 128.699 317.695 128.699C317.791 128.699 317.877 128.717 317.952 128.753C318.027 128.788 318.086 128.838 318.131 128.902C318.175 128.966 318.201 129.041 318.207 129.127H317.938C317.928 129.07 317.902 129.022 317.861 128.984C317.82 128.945 317.766 128.926 317.699 128.926C317.641 128.926 317.591 128.941 317.547 128.972C317.504 129.003 317.47 129.047 317.446 129.104C317.422 129.162 317.41 129.23 317.41 129.31C317.41 129.392 317.422 129.462 317.446 129.52C317.469 129.578 317.503 129.623 317.546 129.654C317.589 129.685 317.64 129.701 317.699 129.701C317.74 129.701 317.777 129.693 317.81 129.678C317.843 129.662 317.87 129.638 317.893 129.608C317.915 129.578 317.93 129.542 317.938 129.499H318.207C318.2 129.583 318.175 129.658 318.132 129.723C318.089 129.788 318.031 129.839 317.957 129.875C317.883 129.912 317.796 129.93 317.696 129.93ZM318.802 129.931C318.727 129.931 318.659 129.917 318.598 129.89C318.538 129.863 318.491 129.822 318.456 129.769C318.421 129.716 318.404 129.65 318.404 129.572C318.404 129.505 318.416 129.449 318.441 129.405C318.466 129.361 318.5 129.326 318.542 129.3C318.585 129.273 318.634 129.253 318.688 129.24C318.742 129.226 318.798 129.216 318.856 129.21C318.926 129.202 318.982 129.196 319.026 129.19C319.069 129.184 319.101 129.175 319.121 129.162C319.141 129.149 319.151 129.129 319.151 129.103V129.098C319.151 129.039 319.133 128.994 319.099 128.962C319.064 128.93 319.014 128.914 318.949 128.914C318.88 128.914 318.826 128.929 318.785 128.959C318.745 128.989 318.719 129.024 318.705 129.065L318.442 129.028C318.463 128.956 318.497 128.895 318.545 128.846C318.592 128.797 318.651 128.761 318.719 128.736C318.788 128.711 318.864 128.699 318.948 128.699C319.005 128.699 319.062 128.706 319.119 128.719C319.176 128.733 319.228 128.755 319.275 128.786C319.322 128.816 319.36 128.858 319.388 128.911C319.417 128.964 319.432 129.03 319.432 129.109V129.906H319.162V129.743H319.152C319.135 129.776 319.111 129.807 319.08 129.836C319.05 129.864 319.011 129.887 318.965 129.905C318.919 129.922 318.864 129.931 318.802 129.931ZM318.875 129.724C318.932 129.724 318.981 129.713 319.022 129.691C319.063 129.668 319.095 129.638 319.117 129.601C319.14 129.563 319.152 129.523 319.152 129.479V129.338C319.143 129.346 319.128 129.352 319.107 129.359C319.086 129.365 319.063 129.37 319.037 129.375C319.011 129.38 318.985 129.384 318.96 129.387C318.935 129.391 318.913 129.394 318.894 129.397C318.852 129.402 318.815 129.412 318.781 129.425C318.748 129.438 318.722 129.456 318.703 129.479C318.684 129.502 318.674 129.531 318.674 129.567C318.674 129.619 318.693 129.658 318.731 129.685C318.769 129.711 318.817 129.724 318.875 129.724ZM320.017 128.317V129.906H319.736V128.317H320.017ZM320.609 128.317V129.906H320.328V128.317H320.609ZM321.861 129.03L321.605 129.058C321.598 129.032 321.585 129.007 321.567 128.985C321.55 128.962 321.526 128.943 321.496 128.929C321.466 128.916 321.429 128.909 321.386 128.909C321.327 128.909 321.278 128.921 321.238 128.947C321.199 128.972 321.18 129.005 321.18 129.045C321.18 129.08 321.192 129.108 321.218 129.13C321.244 129.151 321.288 129.169 321.348 129.183L321.552 129.227C321.665 129.251 321.748 129.29 321.803 129.342C321.859 129.395 321.886 129.464 321.887 129.55C321.886 129.625 321.864 129.691 321.821 129.748C321.778 129.805 321.718 129.85 321.642 129.882C321.565 129.914 321.477 129.93 321.378 129.93C321.232 129.93 321.115 129.899 321.026 129.838C320.937 129.777 320.884 129.691 320.867 129.581L321.14 129.555C321.153 129.609 321.179 129.649 321.22 129.677C321.26 129.704 321.312 129.718 321.377 129.718C321.444 129.718 321.497 129.704 321.538 129.677C321.579 129.649 321.599 129.615 321.599 129.575C321.599 129.541 321.586 129.513 321.559 129.491C321.534 129.468 321.493 129.451 321.438 129.439L321.235 129.397C321.121 129.373 321.036 129.333 320.981 129.276C320.927 129.219 320.899 129.148 320.9 129.061C320.899 128.987 320.919 128.924 320.96 128.87C321.001 128.815 321.057 128.774 321.13 128.744C321.203 128.714 321.287 128.699 321.382 128.699C321.521 128.699 321.631 128.729 321.712 128.788C321.792 128.848 321.842 128.928 321.861 129.03ZM323.308 128.715V128.932H322.603V128.715H323.308ZM322.78 129.906V128.602C322.78 128.522 322.796 128.455 322.829 128.402C322.863 128.349 322.908 128.309 322.964 128.282C323.021 128.256 323.083 128.243 323.152 128.243C323.201 128.243 323.244 128.247 323.282 128.254C323.319 128.262 323.347 128.269 323.365 128.275L323.31 128.493C323.298 128.489 323.283 128.485 323.265 128.482C323.246 128.478 323.226 128.476 323.204 128.476C323.152 128.476 323.115 128.488 323.093 128.514C323.072 128.538 323.061 128.574 323.061 128.621V129.906H322.78ZM324.025 129.93C323.909 129.93 323.808 129.904 323.722 129.853C323.637 129.802 323.571 129.73 323.524 129.638C323.477 129.546 323.454 129.438 323.454 129.315C323.454 129.192 323.477 129.084 323.524 128.992C323.571 128.899 323.637 128.827 323.722 128.776C323.808 128.725 323.909 128.699 324.025 128.699C324.141 128.699 324.242 128.725 324.328 128.776C324.413 128.827 324.479 128.899 324.526 128.992C324.573 129.084 324.596 129.192 324.596 129.315C324.596 129.438 324.573 129.546 324.526 129.638C324.479 129.73 324.413 129.802 324.328 129.853C324.242 129.904 324.141 129.93 324.025 129.93ZM324.027 129.705C324.09 129.705 324.142 129.687 324.185 129.653C324.227 129.618 324.259 129.57 324.28 129.511C324.301 129.453 324.311 129.387 324.311 129.314C324.311 129.241 324.301 129.175 324.28 129.117C324.259 129.057 324.227 129.01 324.185 128.974C324.142 128.939 324.09 128.922 324.027 128.922C323.962 128.922 323.908 128.939 323.865 128.974C323.823 129.01 323.791 129.057 323.77 129.117C323.749 129.175 323.739 129.241 323.739 129.314C323.739 129.387 323.749 129.453 323.77 129.511C323.791 129.57 323.823 129.618 323.865 129.653C323.908 129.687 323.962 129.705 324.027 129.705ZM324.856 129.906V128.715H325.129V128.913H325.141C325.163 128.844 325.2 128.791 325.253 128.754C325.306 128.716 325.367 128.697 325.435 128.697C325.451 128.697 325.468 128.698 325.487 128.7C325.507 128.701 325.523 128.703 325.536 128.705V128.964C325.524 128.959 325.505 128.956 325.48 128.953C325.454 128.949 325.43 128.947 325.406 128.947C325.355 128.947 325.309 128.958 325.268 128.981C325.227 129.002 325.196 129.033 325.172 129.071C325.149 129.11 325.137 129.155 325.137 129.206V129.906H324.856ZM326.404 128.715V128.932H325.719V128.715H326.404ZM325.888 128.429H326.169V129.548C326.169 129.586 326.174 129.615 326.186 129.635C326.198 129.655 326.213 129.668 326.232 129.675C326.252 129.682 326.273 129.686 326.296 129.686C326.314 129.686 326.33 129.685 326.344 129.682C326.359 129.68 326.371 129.677 326.378 129.675L326.426 129.895C326.411 129.9 326.389 129.906 326.361 129.912C326.334 129.918 326.3 129.922 326.26 129.923C326.19 129.925 326.127 129.914 326.07 129.891C326.014 129.867 325.969 129.83 325.936 129.781C325.903 129.731 325.887 129.669 325.888 129.595V128.429ZM326.967 129.208V129.906H326.686V128.317H326.961V128.917H326.975C327.003 128.85 327.046 128.797 327.105 128.758C327.164 128.719 327.239 128.699 327.33 128.699C327.412 128.699 327.485 128.716 327.546 128.751C327.608 128.786 327.655 128.836 327.689 128.903C327.723 128.97 327.74 129.051 327.74 129.148V129.906H327.459V129.191C327.459 129.111 327.438 129.048 327.397 129.004C327.356 128.959 327.299 128.936 327.225 128.936C327.175 128.936 327.131 128.947 327.091 128.969C327.053 128.99 327.022 129.021 327 129.061C326.978 129.102 326.967 129.151 326.967 129.208ZM328.955 129.931C328.88 129.931 328.812 129.917 328.751 129.89C328.691 129.863 328.643 129.822 328.608 129.769C328.574 129.716 328.556 129.65 328.556 129.572C328.556 129.505 328.569 129.449 328.594 129.405C328.618 129.361 328.652 129.326 328.695 129.3C328.738 129.273 328.786 129.253 328.84 129.24C328.895 129.226 328.951 129.216 329.009 129.21C329.079 129.202 329.135 129.196 329.179 129.19C329.222 129.184 329.254 129.175 329.273 129.162C329.293 129.149 329.304 129.129 329.304 129.103V129.098C329.304 129.039 329.286 128.994 329.252 128.962C329.217 128.93 329.167 128.914 329.102 128.914C329.033 128.914 328.978 128.929 328.938 128.959C328.898 128.989 328.871 129.024 328.857 129.065L328.595 129.028C328.616 128.956 328.65 128.895 328.698 128.846C328.745 128.797 328.803 128.761 328.872 128.736C328.941 128.711 329.017 128.699 329.1 128.699C329.158 128.699 329.215 128.706 329.272 128.719C329.329 128.733 329.381 128.755 329.428 128.786C329.475 128.816 329.513 128.858 329.541 128.911C329.57 128.964 329.584 129.03 329.584 129.109V129.906H329.314V129.743H329.305C329.288 129.776 329.264 129.807 329.233 129.836C329.202 129.864 329.164 129.887 329.117 129.905C329.071 129.922 329.017 129.931 328.955 129.931ZM329.028 129.724C329.084 129.724 329.133 129.713 329.175 129.691C329.216 129.668 329.248 129.638 329.27 129.601C329.293 129.563 329.304 129.523 329.304 129.479V129.338C329.296 129.346 329.281 129.352 329.259 129.359C329.239 129.365 329.215 129.37 329.189 129.375C329.164 129.38 329.138 129.384 329.113 129.387C329.087 129.391 329.065 129.394 329.047 129.397C329.005 129.402 328.967 129.412 328.934 129.425C328.901 129.438 328.875 129.456 328.856 129.479C328.837 129.502 328.827 129.531 328.827 129.567C328.827 129.619 328.846 129.658 328.884 129.685C328.922 129.711 328.97 129.724 329.028 129.724ZM330.17 129.208V129.906H329.889V128.715H330.157V128.917H330.171C330.199 128.85 330.242 128.797 330.302 128.758C330.363 128.719 330.438 128.699 330.527 128.699C330.609 128.699 330.681 128.717 330.742 128.752C330.803 128.787 330.851 128.838 330.884 128.905C330.919 128.971 330.935 129.052 330.935 129.148V129.906H330.654V129.191C330.654 129.111 330.633 129.049 330.592 129.004C330.551 128.959 330.494 128.936 330.422 128.936C330.373 128.936 330.329 128.947 330.291 128.969C330.253 128.99 330.223 129.021 330.202 129.061C330.18 129.102 330.17 129.151 330.17 129.208ZM331.762 129.93C331.646 129.93 331.545 129.904 331.46 129.853C331.374 129.802 331.308 129.73 331.261 129.638C331.214 129.546 331.191 129.438 331.191 129.315C331.191 129.192 331.214 129.084 331.261 128.992C331.308 128.899 331.374 128.827 331.46 128.776C331.545 128.725 331.646 128.699 331.762 128.699C331.879 128.699 331.979 128.725 332.065 128.776C332.15 128.827 332.216 128.899 332.263 128.992C332.31 129.084 332.333 129.192 332.333 129.315C332.333 129.438 332.31 129.546 332.263 129.638C332.216 129.73 332.15 129.802 332.065 129.853C331.979 129.904 331.879 129.93 331.762 129.93ZM331.764 129.705C331.827 129.705 331.88 129.687 331.922 129.653C331.964 129.618 331.996 129.57 332.017 129.511C332.038 129.453 332.048 129.387 332.048 129.314C332.048 129.241 332.038 129.175 332.017 129.117C331.996 129.057 331.964 129.01 331.922 128.974C331.88 128.939 331.827 128.922 331.764 128.922C331.699 128.922 331.645 128.939 331.602 128.974C331.56 129.01 331.528 129.057 331.507 129.117C331.486 129.175 331.476 129.241 331.476 129.314C331.476 129.387 331.486 129.453 331.507 129.511C331.528 129.57 331.56 129.618 331.602 129.653C331.645 129.687 331.699 129.705 331.764 129.705ZM333.189 128.715V128.932H332.504V128.715H333.189ZM332.673 128.429H332.954V129.548C332.954 129.586 332.959 129.615 332.971 129.635C332.983 129.655 332.998 129.668 333.017 129.675C333.036 129.682 333.058 129.686 333.081 129.686C333.098 129.686 333.115 129.685 333.129 129.682C333.144 129.68 333.155 129.677 333.163 129.675L333.21 129.895C333.195 129.9 333.174 129.906 333.146 129.912C333.119 129.918 333.085 129.922 333.045 129.923C332.975 129.925 332.911 129.914 332.855 129.891C332.799 129.867 332.754 129.83 332.721 129.781C332.688 129.731 332.672 129.669 332.673 129.595V128.429ZM333.752 129.208V129.906H333.471V128.317H333.746V128.917H333.76C333.788 128.85 333.831 128.797 333.889 128.758C333.948 128.719 334.023 128.699 334.114 128.699C334.197 128.699 334.269 128.716 334.331 128.751C334.393 128.786 334.44 128.836 334.474 128.903C334.508 128.97 334.525 129.051 334.525 129.148V129.906H334.244V129.191C334.244 129.111 334.223 129.048 334.182 129.004C334.141 128.959 334.084 128.936 334.01 128.936C333.96 128.936 333.916 128.947 333.876 128.969C333.837 128.99 333.807 129.021 333.785 129.061C333.763 129.102 333.752 129.151 333.752 129.208ZM335.361 129.93C335.241 129.93 335.138 129.905 335.051 129.855C334.965 129.805 334.898 129.734 334.852 129.643C334.805 129.551 334.782 129.442 334.782 129.317C334.782 129.195 334.805 129.087 334.852 128.995C334.899 128.902 334.965 128.829 335.049 128.777C335.133 128.725 335.232 128.699 335.346 128.699C335.42 128.699 335.489 128.711 335.554 128.735C335.62 128.758 335.678 128.794 335.728 128.843C335.779 128.892 335.818 128.955 335.847 129.031C335.876 129.107 335.891 129.197 335.891 129.301V129.387H334.914V129.198H335.622C335.621 129.144 335.609 129.096 335.587 129.054C335.564 129.012 335.532 128.979 335.491 128.954C335.451 128.93 335.404 128.918 335.35 128.918C335.293 128.918 335.242 128.932 335.199 128.96C335.155 128.987 335.121 129.023 335.097 129.068C335.073 129.113 335.061 129.162 335.061 129.215V129.38C335.061 129.45 335.073 129.509 335.099 129.559C335.124 129.608 335.159 129.646 335.205 129.672C335.25 129.698 335.304 129.711 335.365 129.711C335.406 129.711 335.443 129.705 335.476 129.694C335.509 129.682 335.537 129.665 335.562 129.642C335.586 129.619 335.604 129.591 335.617 129.557L335.879 129.587C335.863 129.656 335.831 129.717 335.785 129.768C335.738 129.82 335.679 129.859 335.608 129.888C335.536 129.916 335.453 129.93 335.361 129.93ZM336.15 129.906V128.715H336.423V128.913H336.435C336.457 128.844 336.494 128.791 336.547 128.754C336.6 128.716 336.661 128.697 336.729 128.697C336.745 128.697 336.762 128.698 336.781 128.7C336.801 128.701 336.817 128.703 336.83 128.705V128.964C336.818 128.959 336.799 128.956 336.773 128.953C336.748 128.949 336.724 128.947 336.7 128.947C336.649 128.947 336.602 128.958 336.562 128.981C336.521 129.002 336.489 129.033 336.466 129.071C336.443 129.11 336.431 129.155 336.431 129.206V129.906H336.15ZM338.045 129.927C337.951 129.927 337.867 129.903 337.793 129.855C337.719 129.807 337.661 129.737 337.618 129.646C337.575 129.554 337.554 129.443 337.554 129.312C337.554 129.18 337.575 129.068 337.619 128.977C337.663 128.885 337.722 128.816 337.796 128.77C337.871 128.723 337.954 128.699 338.046 128.699C338.115 128.699 338.173 128.711 338.218 128.735C338.263 128.758 338.299 128.786 338.325 128.819C338.351 128.852 338.372 128.883 338.386 128.912H338.398V128.317H338.68V129.906H338.403V129.719H338.386C338.372 129.748 338.351 129.778 338.323 129.811C338.296 129.843 338.26 129.871 338.215 129.893C338.17 129.916 338.113 129.927 338.045 129.927ZM338.123 129.697C338.183 129.697 338.233 129.681 338.275 129.649C338.317 129.616 338.349 129.571 338.371 129.513C338.392 129.455 338.403 129.388 338.403 129.31C338.403 129.233 338.392 129.166 338.371 129.11C338.349 129.053 338.318 129.008 338.276 128.977C338.235 128.945 338.184 128.929 338.123 128.929C338.061 128.929 338.008 128.946 337.966 128.978C337.924 129.011 337.893 129.056 337.872 129.113C337.851 129.171 337.84 129.237 337.84 129.31C337.84 129.385 337.851 129.451 337.872 129.51C337.893 129.568 337.925 129.614 337.967 129.647C338.01 129.68 338.062 129.697 338.123 129.697ZM339.344 129.931C339.268 129.931 339.2 129.917 339.14 129.89C339.08 129.863 339.032 129.822 338.997 129.769C338.962 129.716 338.945 129.65 338.945 129.572C338.945 129.505 338.957 129.449 338.982 129.405C339.007 129.361 339.041 129.326 339.084 129.3C339.127 129.273 339.175 129.253 339.229 129.24C339.283 129.226 339.339 129.216 339.397 129.21C339.467 129.202 339.524 129.196 339.567 129.19C339.611 129.184 339.642 129.175 339.662 129.162C339.682 129.149 339.692 129.129 339.692 129.103V129.098C339.692 129.039 339.675 128.994 339.64 128.962C339.605 128.93 339.556 128.914 339.49 128.914C339.422 128.914 339.367 128.929 339.327 128.959C339.287 128.989 339.26 129.024 339.246 129.065L338.984 129.028C339.004 128.956 339.038 128.895 339.086 128.846C339.134 128.797 339.192 128.761 339.261 128.736C339.329 128.711 339.406 128.699 339.489 128.699C339.546 128.699 339.603 128.706 339.66 128.719C339.717 128.733 339.769 128.755 339.816 128.786C339.863 128.816 339.901 128.858 339.93 128.911C339.959 128.964 339.973 129.03 339.973 129.109V129.906H339.703V129.743H339.694C339.677 129.776 339.653 129.807 339.622 129.836C339.591 129.864 339.552 129.887 339.506 129.905C339.46 129.922 339.406 129.931 339.344 129.931ZM339.417 129.724C339.473 129.724 339.522 129.713 339.563 129.691C339.605 129.668 339.637 129.638 339.659 129.601C339.682 129.563 339.693 129.523 339.693 129.479V129.338C339.684 129.346 339.669 129.352 339.648 129.359C339.627 129.365 339.604 129.37 339.578 129.375C339.552 129.38 339.527 129.384 339.501 129.387C339.476 129.391 339.454 129.394 339.435 129.397C339.393 129.402 339.356 129.412 339.323 129.425C339.29 129.438 339.264 129.456 339.244 129.479C339.225 129.502 339.216 129.531 339.216 129.567C339.216 129.619 339.235 129.658 339.272 129.685C339.31 129.711 339.358 129.724 339.417 129.724ZM340.409 130.353C340.371 130.353 340.336 130.35 340.303 130.344C340.271 130.338 340.245 130.332 340.226 130.324L340.291 130.105C340.332 130.117 340.369 130.123 340.401 130.122C340.433 130.122 340.461 130.112 340.485 130.092C340.51 130.073 340.531 130.041 340.548 129.996L340.572 129.931L340.14 128.715H340.438L340.713 129.615H340.725L341.001 128.715H341.299L340.822 130.051C340.8 130.114 340.77 130.168 340.734 130.213C340.697 130.259 340.652 130.293 340.599 130.317C340.546 130.341 340.483 130.353 340.409 130.353ZM342.599 129.93C342.483 129.93 342.382 129.904 342.297 129.853C342.211 129.802 342.145 129.73 342.098 129.638C342.052 129.546 342.028 129.438 342.028 129.315C342.028 129.192 342.052 129.084 342.098 128.992C342.145 128.899 342.211 128.827 342.297 128.776C342.382 128.725 342.483 128.699 342.599 128.699C342.716 128.699 342.817 128.725 342.902 128.776C342.987 128.827 343.053 128.899 343.1 128.992C343.147 129.084 343.171 129.192 343.171 129.315C343.171 129.438 343.147 129.546 343.1 129.638C343.053 129.73 342.987 129.802 342.902 129.853C342.817 129.904 342.716 129.93 342.599 129.93ZM342.601 129.705C342.664 129.705 342.717 129.687 342.759 129.653C342.802 129.618 342.833 129.57 342.854 129.511C342.875 129.453 342.886 129.387 342.886 129.314C342.886 129.241 342.875 129.175 342.854 129.117C342.833 129.057 342.802 129.01 342.759 128.974C342.717 128.939 342.664 128.922 342.601 128.922C342.536 128.922 342.482 128.939 342.44 128.974C342.397 129.01 342.365 129.057 342.344 129.117C342.323 129.175 342.313 129.241 342.313 129.314C342.313 129.387 342.323 129.453 342.344 129.511C342.365 129.57 342.397 129.618 342.44 129.653C342.482 129.687 342.536 129.705 342.601 129.705ZM344.035 128.715V128.932H343.33V128.715H344.035ZM343.506 129.906V128.602C343.506 128.522 343.523 128.455 343.556 128.402C343.589 128.349 343.634 128.309 343.691 128.282C343.747 128.256 343.81 128.243 343.879 128.243C343.927 128.243 343.97 128.247 344.008 128.254C344.046 128.262 344.074 128.269 344.092 128.275L344.036 128.493C344.024 128.489 344.009 128.485 343.991 128.482C343.973 128.478 343.953 128.476 343.931 128.476C343.878 128.476 343.841 128.488 343.82 128.514C343.798 128.538 343.788 128.574 343.788 128.621V129.906H343.506ZM345.086 129.906L344.749 128.715H345.036L345.245 129.553H345.256L345.47 128.715H345.754L345.968 129.548H345.979L346.186 128.715H346.473L346.135 129.906H345.843L345.619 129.101H345.603L345.38 129.906H345.086ZM347.177 129.93C347.06 129.93 346.959 129.904 346.874 129.853C346.789 129.802 346.722 129.73 346.675 129.638C346.629 129.546 346.606 129.438 346.606 129.315C346.606 129.192 346.629 129.084 346.675 128.992C346.722 128.899 346.789 128.827 346.874 128.776C346.959 128.725 347.06 128.699 347.177 128.699C347.293 128.699 347.394 128.725 347.479 128.776C347.565 128.827 347.631 128.899 347.677 128.992C347.724 129.084 347.748 129.192 347.748 129.315C347.748 129.438 347.724 129.546 347.677 129.638C347.631 129.73 347.565 129.802 347.479 129.853C347.394 129.904 347.293 129.93 347.177 129.93ZM347.178 129.705C347.241 129.705 347.294 129.687 347.337 129.653C347.379 129.618 347.411 129.57 347.431 129.511C347.452 129.453 347.463 129.387 347.463 129.314C347.463 129.241 347.452 129.175 347.431 129.117C347.411 129.057 347.379 129.01 347.337 128.974C347.294 128.939 347.241 128.922 347.178 128.922C347.114 128.922 347.06 128.939 347.017 128.974C346.974 129.01 346.943 129.057 346.921 129.117C346.901 129.175 346.89 129.241 346.89 129.314C346.89 129.387 346.901 129.453 346.921 129.511C346.943 129.57 346.974 129.618 347.017 129.653C347.06 129.687 347.114 129.705 347.178 129.705ZM348.289 129.208V129.906H348.008V128.715H348.277V128.917H348.291C348.318 128.85 348.362 128.797 348.422 128.758C348.482 128.719 348.557 128.699 348.646 128.699C348.728 128.699 348.8 128.717 348.861 128.752C348.922 128.787 348.97 128.838 349.004 128.905C349.038 128.971 349.055 129.052 349.054 129.148V129.906H348.773V129.191C348.773 129.111 348.753 129.049 348.711 129.004C348.67 128.959 348.614 128.936 348.541 128.936C348.492 128.936 348.448 128.947 348.41 128.969C348.372 128.99 348.343 129.021 348.321 129.061C348.3 129.102 348.289 129.151 348.289 129.208ZM349.804 129.927C349.71 129.927 349.626 129.903 349.552 129.855C349.478 129.807 349.42 129.737 349.377 129.646C349.334 129.554 349.313 129.443 349.313 129.312C349.313 129.18 349.334 129.068 349.378 128.977C349.422 128.885 349.481 128.816 349.556 128.77C349.63 128.723 349.713 128.699 349.805 128.699C349.874 128.699 349.932 128.711 349.977 128.735C350.022 128.758 350.058 128.786 350.084 128.819C350.11 128.852 350.131 128.883 350.145 128.912H350.157V128.317H350.439V129.906H350.162V129.719H350.145C350.131 129.748 350.11 129.778 350.082 129.811C350.055 129.843 350.019 129.871 349.974 129.893C349.929 129.916 349.872 129.927 349.804 129.927ZM349.882 129.697C349.942 129.697 349.992 129.681 350.034 129.649C350.076 129.616 350.108 129.571 350.13 129.513C350.152 129.455 350.162 129.388 350.162 129.31C350.162 129.233 350.152 129.166 350.13 129.11C350.109 129.053 350.077 129.008 350.035 128.977C349.994 128.945 349.943 128.929 349.882 128.929C349.82 128.929 349.767 128.946 349.726 128.978C349.684 129.011 349.652 129.056 349.631 129.113C349.61 129.171 349.599 129.237 349.599 129.31C349.599 129.385 349.61 129.451 349.631 129.51C349.653 129.568 349.684 129.614 349.726 129.647C349.769 129.68 349.821 129.697 349.882 129.697ZM351.288 129.93C351.168 129.93 351.065 129.905 350.978 129.855C350.891 129.805 350.825 129.734 350.778 129.643C350.732 129.551 350.709 129.442 350.709 129.317C350.709 129.195 350.732 129.087 350.778 128.995C350.826 128.902 350.891 128.829 350.976 128.777C351.06 128.725 351.159 128.699 351.273 128.699C351.346 128.699 351.416 128.711 351.481 128.735C351.546 128.758 351.604 128.794 351.655 128.843C351.705 128.892 351.745 128.955 351.774 129.031C351.803 129.107 351.818 129.197 351.818 129.301V129.387H350.841V129.198H351.548C351.548 129.144 351.536 129.096 351.513 129.054C351.491 129.012 351.459 128.979 351.418 128.954C351.378 128.93 351.33 128.918 351.277 128.918C351.219 128.918 351.169 128.932 351.125 128.96C351.082 128.987 351.048 129.023 351.024 129.068C351 129.113 350.988 129.162 350.987 129.215V129.38C350.987 129.45 351 129.509 351.025 129.559C351.051 129.608 351.086 129.646 351.132 129.672C351.177 129.698 351.23 129.711 351.291 129.711C351.332 129.711 351.369 129.705 351.402 129.694C351.435 129.682 351.464 129.665 351.488 129.642C351.513 129.619 351.531 129.591 351.544 129.557L351.806 129.587C351.789 129.656 351.758 129.717 351.711 129.768C351.665 129.82 351.606 129.859 351.534 129.888C351.462 129.916 351.38 129.93 351.288 129.93ZM352.077 129.906V128.715H352.349V128.913H352.362C352.384 128.844 352.421 128.791 352.474 128.754C352.527 128.716 352.588 128.697 352.656 128.697C352.671 128.697 352.689 128.698 352.708 128.7C352.728 128.701 352.744 128.703 352.757 128.705V128.964C352.745 128.959 352.726 128.956 352.7 128.953C352.675 128.949 352.65 128.947 352.626 128.947C352.575 128.947 352.529 128.958 352.488 128.981C352.448 129.002 352.416 129.033 352.393 129.071C352.37 129.11 352.358 129.155 352.358 129.206V129.906H352.077ZM353.002 129.924C352.955 129.924 352.914 129.907 352.881 129.874C352.847 129.841 352.831 129.8 352.831 129.753C352.831 129.706 352.847 129.666 352.881 129.633C352.914 129.6 352.955 129.584 353.002 129.584C353.047 129.584 353.087 129.6 353.12 129.633C353.155 129.666 353.172 129.706 353.172 129.753C353.172 129.784 353.164 129.813 353.148 129.839C353.132 129.865 353.111 129.886 353.086 129.901C353.06 129.916 353.032 129.924 353.002 129.924ZM312.706 132.906L312.967 131.317H313.184L312.924 132.906H312.706ZM311.99 132.497L312.027 132.279H313.249L313.213 132.497H311.99ZM312.148 132.906L312.408 131.317H312.626L312.365 132.906H312.148ZM312.082 131.944L312.118 131.727H313.341L313.305 131.944H312.082ZM313.535 132.906V131.715H313.804V131.917H313.818C313.843 131.849 313.884 131.795 313.941 131.757C313.999 131.718 314.067 131.699 314.147 131.699C314.228 131.699 314.296 131.719 314.351 131.758C314.407 131.797 314.446 131.85 314.469 131.917H314.481C314.508 131.851 314.552 131.798 314.615 131.759C314.678 131.719 314.753 131.699 314.839 131.699C314.949 131.699 315.038 131.734 315.108 131.803C315.177 131.872 315.212 131.973 315.212 132.106V132.906H314.93V132.15C314.93 132.076 314.91 132.022 314.871 131.988C314.832 131.953 314.783 131.936 314.727 131.936C314.659 131.936 314.606 131.957 314.567 131.999C314.53 132.041 314.511 132.096 314.511 132.163V132.906H314.235V132.138C314.235 132.077 314.217 132.028 314.179 131.991C314.143 131.954 314.095 131.936 314.035 131.936C313.995 131.936 313.958 131.946 313.925 131.967C313.892 131.987 313.865 132.016 313.846 132.053C313.826 132.09 313.816 132.133 313.816 132.182V132.906H313.535ZM316.038 132.93C315.922 132.93 315.821 132.904 315.736 132.853C315.65 132.802 315.584 132.73 315.537 132.638C315.49 132.546 315.467 132.438 315.467 132.315C315.467 132.192 315.49 132.084 315.537 131.992C315.584 131.899 315.65 131.827 315.736 131.776C315.821 131.725 315.922 131.699 316.038 131.699C316.155 131.699 316.256 131.725 316.341 131.776C316.426 131.827 316.492 131.899 316.539 131.992C316.586 132.084 316.609 132.192 316.609 132.315C316.609 132.438 316.586 132.546 316.539 132.638C316.492 132.73 316.426 132.802 316.341 132.853C316.256 132.904 316.155 132.93 316.038 132.93ZM316.04 132.705C316.103 132.705 316.156 132.687 316.198 132.653C316.241 132.618 316.272 132.57 316.293 132.511C316.314 132.453 316.325 132.387 316.325 132.314C316.325 132.241 316.314 132.175 316.293 132.117C316.272 132.057 316.241 132.01 316.198 131.974C316.156 131.939 316.103 131.922 316.04 131.922C315.975 131.922 315.921 131.939 315.878 131.974C315.836 132.01 315.804 132.057 315.783 132.117C315.762 132.175 315.752 132.241 315.752 132.314C315.752 132.387 315.762 132.453 315.783 132.511C315.804 132.57 315.836 132.618 315.878 132.653C315.921 132.687 315.975 132.705 316.04 132.705ZM316.87 132.906V131.715H317.142V131.913H317.154C317.176 131.844 317.213 131.791 317.266 131.754C317.32 131.716 317.38 131.697 317.449 131.697C317.464 131.697 317.481 131.698 317.501 131.7C317.52 131.701 317.537 131.703 317.549 131.705V131.964C317.538 131.959 317.519 131.956 317.493 131.953C317.467 131.949 317.443 131.947 317.419 131.947C317.368 131.947 317.322 131.958 317.281 131.981C317.241 132.002 317.209 132.033 317.186 132.071C317.162 132.11 317.151 132.155 317.151 132.206V132.906H316.87ZM318.039 132.208V132.906H317.758V131.715H318.026V131.917H318.04C318.068 131.85 318.112 131.797 318.172 131.758C318.232 131.719 318.307 131.699 318.396 131.699C318.478 131.699 318.55 131.717 318.611 131.752C318.672 131.787 318.72 131.838 318.754 131.905C318.788 131.971 318.804 132.052 318.804 132.148V132.906H318.523V132.191C318.523 132.111 318.502 132.049 318.461 132.004C318.42 131.959 318.363 131.936 318.291 131.936C318.242 131.936 318.198 131.947 318.16 131.969C318.122 131.99 318.092 132.021 318.071 132.061C318.049 132.102 318.039 132.151 318.039 132.208ZM319.111 132.906V131.715H319.392V132.906H319.111ZM319.253 131.545C319.208 131.545 319.17 131.531 319.138 131.501C319.106 131.471 319.09 131.435 319.09 131.393C319.09 131.351 319.106 131.315 319.138 131.285C319.17 131.255 319.208 131.24 319.253 131.24C319.298 131.24 319.336 131.255 319.367 131.285C319.4 131.315 319.416 131.351 319.416 131.393C319.416 131.435 319.4 131.471 319.367 131.501C319.336 131.531 319.298 131.545 319.253 131.545ZM319.984 132.208V132.906H319.703V131.715H319.972V131.917H319.985C320.013 131.85 320.057 131.797 320.117 131.758C320.177 131.719 320.252 131.699 320.341 131.699C320.423 131.699 320.495 131.717 320.556 131.752C320.617 131.787 320.665 131.838 320.699 131.905C320.733 131.971 320.75 132.052 320.749 132.148V132.906H320.468V132.191C320.468 132.111 320.447 132.049 320.406 132.004C320.365 131.959 320.309 131.936 320.236 131.936C320.187 131.936 320.143 131.947 320.105 131.969C320.067 131.99 320.037 132.021 320.016 132.061C319.995 132.102 319.984 132.151 319.984 132.208ZM321.575 133.378C321.474 133.378 321.387 133.365 321.315 133.337C321.242 133.31 321.184 133.274 321.14 133.229C321.096 133.183 321.066 133.133 321.049 133.077L321.302 133.016C321.313 133.039 321.33 133.062 321.351 133.085C321.373 133.108 321.402 133.127 321.439 133.142C321.476 133.158 321.523 133.166 321.579 133.166C321.659 133.166 321.725 133.146 321.777 133.107C321.83 133.069 321.856 133.006 321.856 132.918V132.692H321.842C321.827 132.721 321.806 132.751 321.778 132.782C321.751 132.812 321.714 132.838 321.669 132.858C321.624 132.879 321.567 132.889 321.499 132.889C321.407 132.889 321.324 132.868 321.25 132.825C321.176 132.782 321.117 132.717 321.073 132.631C321.029 132.545 321.008 132.436 321.008 132.307C321.008 132.176 321.029 132.065 321.073 131.975C321.117 131.885 321.176 131.816 321.25 131.77C321.325 131.723 321.408 131.699 321.5 131.699C321.569 131.699 321.627 131.711 321.672 131.735C321.717 131.758 321.754 131.786 321.78 131.819C321.807 131.852 321.828 131.883 321.842 131.912H321.857V131.715H322.134V132.926C322.134 133.028 322.11 133.112 322.061 133.179C322.013 133.246 321.946 133.295 321.862 133.329C321.778 133.362 321.682 133.378 321.575 133.378ZM321.577 132.669C321.637 132.669 321.687 132.655 321.729 132.626C321.771 132.597 321.803 132.555 321.825 132.501C321.846 132.446 321.857 132.381 321.857 132.305C321.857 132.23 321.846 132.164 321.825 132.108C321.803 132.052 321.772 132.008 321.73 131.977C321.689 131.945 321.638 131.929 321.577 131.929C321.515 131.929 321.462 131.946 321.42 131.978C321.379 132.011 321.347 132.056 321.326 132.113C321.305 132.169 321.294 132.233 321.294 132.305C321.294 132.378 321.305 132.442 321.326 132.497C321.347 132.551 321.379 132.593 321.421 132.624C321.464 132.654 321.516 132.669 321.577 132.669Z" fill="#434960"/><g clip-path="url(#clip12_3202_39054)"><path d="M312.875 137.745L312.875 137.764C312.95 137.764 313.023 137.782 313.09 137.817C313.157 137.851 313.214 137.901 313.257 137.963L313.257 137.963L313.406 138.175L313.421 138.196L313.436 138.175L313.585 137.963L313.585 137.963C313.628 137.901 313.685 137.851 313.751 137.817C313.818 137.782 313.892 137.764 313.967 137.764L313.967 137.745M312.875 137.745V137.764C312.617 137.764 312.407 137.973 312.407 138.231C312.407 138.507 312.572 138.774 312.784 139.005C312.991 139.231 313.242 139.42 313.421 139.546C313.599 139.42 313.85 139.231 314.058 139.005C314.27 138.774 314.434 138.507 314.434 138.231C314.434 137.973 314.225 137.764 313.967 137.764V137.745M312.875 137.745C312.607 137.745 312.389 137.963 312.389 138.231C312.389 138.798 313.062 139.316 313.421 139.569C313.78 139.316 314.452 138.798 314.452 138.231C314.452 137.963 314.235 137.745 313.967 137.745M312.875 137.745L313.432 138.149M313.967 137.745C313.889 137.745 313.812 137.764 313.743 137.8C313.674 137.836 313.614 137.889 313.57 137.952L313.432 138.149M313.432 138.149L313.436 138.154L313.421 138.164L313.406 138.154L313.41 138.149L313.432 138.149ZM313.421 137.817C313.482 137.736 313.56 137.669 313.651 137.622C313.748 137.572 313.857 137.545 313.967 137.545C314.057 137.545 314.146 137.563 314.229 137.597C314.313 137.632 314.388 137.682 314.452 137.746C314.516 137.81 314.566 137.885 314.601 137.968C314.635 138.052 314.653 138.141 314.653 138.231C314.653 138.585 314.452 138.906 314.2 139.174C313.948 139.442 313.644 139.659 313.431 139.805L313.421 139.813L313.41 139.805C313.198 139.659 312.893 139.442 312.641 139.174C312.39 138.906 312.189 138.585 312.189 138.231C312.189 138.049 312.261 137.874 312.39 137.746C312.518 137.617 312.693 137.545 312.875 137.545L313.421 137.817ZM313.421 137.817C313.36 137.736 313.281 137.669 313.191 137.622C313.093 137.572 312.985 137.545 312.875 137.545V137.563L313.421 137.817Z" fill="#434960" stroke="#434960" stroke-width="0.03642"/></g><path d="M316.832 139.514H316.108L316.455 139.121C316.687 138.874 316.812 138.716 316.812 138.486C316.812 138.221 316.6 137.994 316.256 137.994C315.922 137.994 315.675 138.216 315.675 138.542H315.927C315.927 138.348 316.047 138.226 316.238 138.226C316.424 138.226 316.542 138.356 316.542 138.519C316.542 138.695 316.432 138.823 316.256 139.011L315.718 139.595V139.743H316.832V139.514ZM318.292 139.363V139.139H318.058V138.02H317.859L316.969 139.182V139.363H317.8V139.743H318.058V139.363H318.292ZM317.26 139.139L317.8 138.41V139.139H317.26Z" fill="#434960"/><g clip-path="url(#clip13_3202_39054)"><path d="M323.618 137.441C323.458 137.441 323.3 137.472 323.153 137.533C323.006 137.594 322.872 137.684 322.759 137.796C322.532 138.024 322.404 138.333 322.404 138.655C322.403 138.935 322.5 139.207 322.678 139.423L322.435 139.666C322.419 139.683 322.407 139.704 322.403 139.728C322.398 139.751 322.401 139.776 322.41 139.798C322.42 139.82 322.436 139.838 322.457 139.851C322.477 139.863 322.501 139.869 322.525 139.868H323.618C323.939 139.868 324.248 139.74 324.476 139.513C324.703 139.285 324.831 138.976 324.831 138.655C324.831 138.333 324.703 138.024 324.476 137.796C324.248 137.569 323.939 137.441 323.618 137.441V137.441ZM323.618 139.626H322.818L322.931 139.513C322.953 139.49 322.966 139.459 322.966 139.427C322.966 139.395 322.953 139.364 322.931 139.342C322.772 139.183 322.673 138.974 322.651 138.75C322.628 138.527 322.684 138.302 322.809 138.116C322.934 137.929 323.119 137.791 323.334 137.726C323.549 137.66 323.78 137.671 323.988 137.757C324.195 137.843 324.367 137.998 324.473 138.196C324.579 138.394 324.613 138.623 324.569 138.843C324.526 139.064 324.407 139.262 324.233 139.405C324.06 139.547 323.842 139.625 323.618 139.626V139.626Z" fill="#434960"/></g><path d="M327.52 139.185C327.52 138.797 327.188 138.565 326.791 138.652L327.298 138.02H327.005L326.485 138.662C326.332 138.851 326.258 138.986 326.258 139.197C326.258 139.511 326.523 139.766 326.887 139.766C327.257 139.766 327.52 139.511 327.52 139.185ZM326.528 139.19C326.528 138.95 326.689 138.828 326.89 138.828C327.094 138.828 327.252 138.965 327.252 139.192C327.252 139.406 327.094 139.549 326.89 139.549C326.689 139.549 326.528 139.406 326.528 139.19Z" fill="#434960"/></g></g><g filter="url(#filter6_dd_3202_39054)"><g clip-path="url(#clip14_3202_39054)"><rect x="309.777" y="145.483" width="51.8985" height="68.4688" rx="0.7284" fill="white"/><rect x="309.777" y="145.483" width="51.8985" height="48.4386" fill="url(#pattern5)"/></g></g></g></g><g filter="url(#filter7_dd_3202_39054)"><g clip-path="url(#clip15_3202_39054)"><rect x="21.0996" y="45.1871" width="185.918" height="173.883" rx="3.5434" transform="rotate(3 21.0996 45.1871)" fill="white"/><g clip-path="url(#clip16_3202_39054)"><rect x="26.9668" y="51.7041" width="173.516" height="140.322" rx="1.7717" transform="rotate(3 26.9668 51.7041)" fill="#FCF0EF"/><g filter="url(#filter8_dd_3202_39054)"><g clip-path="url(#clip17_3202_39054)"><rect x="54.3984" y="73.2317" width="35.7468" height="51.9999" rx="0.50171" transform="rotate(3 54.3984 73.2317)" fill="white"/><g clip-path="url(#clip18_3202_39054)"><rect width="35.7468" height="45.4048" transform="translate(54.3984 73.2317) rotate(3)" fill="#FFCAC7"/><circle cx="92.9208" cy="121.058" r="26.0999" transform="rotate(3 92.9208 121.058)" fill="#FCA7A1"/></g><rect x="62.0449" y="121.348" width="15.4354" height="2.10483" rx="0.280644" transform="rotate(3 62.0449 121.348)" fill="#D0D1D7"/></g></g><g filter="url(#filter9_dd_3202_39054)"><g clip-path="url(#clip19_3202_39054)"><rect x="51.5781" y="127.039" width="35.7468" height="51.9999" rx="0.50171" transform="rotate(3 51.5781 127.039)" fill="white"/><rect width="35.7468" height="45.4048" transform="translate(51.5781 127.039) rotate(3)" fill="#FFCAC7"/><rect x="59.2246" y="175.155" width="15.4354" height="2.10483" rx="0.280644" transform="rotate(3 59.2246 175.155)" fill="#D0D1D7"/></g><rect x="51.6375" y="127.105" width="35.6214" height="51.8745" rx="0.438996" transform="rotate(3 51.6375 127.105)" stroke="#F3F4F5" stroke-width="0.125427"/></g><g filter="url(#filter10_dd_3202_39054)"><g clip-path="url(#clip20_3202_39054)"><rect x="48.7578" y="180.847" width="35.7468" height="51.9999" rx="0.50171" transform="rotate(3 48.7578 180.847)" fill="white"/><rect width="35.7468" height="45.4048" transform="translate(48.7578 180.847) rotate(3)" fill="#FFCAC7"/></g><rect x="48.8172" y="180.913" width="35.6214" height="51.8745" rx="0.438996" transform="rotate(3 48.8172 180.913)" stroke="#F3F4F5" stroke-width="0.125427"/></g><g filter="url(#filter11_dd_3202_39054)"><g clip-path="url(#clip21_3202_39054)"><rect x="92.6016" y="75.2343" width="35.7468" height="39.9589" rx="0.50171" transform="rotate(3 92.6016 75.2343)" fill="white"/><rect width="35.7468" height="33.3637" transform="translate(92.6016 75.2343) rotate(3)" fill="#FFCAC7"/><rect x="100.879" y="111.326" width="15.4354" height="2.10483" rx="0.280644" transform="rotate(3 100.879 111.326)" fill="#D0D1D7"/></g></g><g filter="url(#filter12_dd_3202_39054)"><g clip-path="url(#clip22_3202_39054)"><rect x="90.4121" y="117.017" width="35.7468" height="51.9999" rx="0.50171" transform="rotate(3 90.4121 117.017)" fill="white"/><g clip-path="url(#clip23_3202_39054)"><rect width="35.7468" height="45.4048" transform="translate(90.4121 117.017) rotate(3)" fill="#FFCAC7"/><rect x="88.2676" y="164.285" width="34.7999" height="38.1676" transform="rotate(-42 88.2676 164.285)" fill="#FCA7A1"/></g><rect x="98.0586" y="165.133" width="15.4354" height="2.10483" rx="0.280644" transform="rotate(3 98.0586 165.133)" fill="#D0D1D7"/></g><rect x="90.4715" y="117.083" width="35.6214" height="51.8745" rx="0.438996" transform="rotate(3 90.4715 117.083)" stroke="#F3F4F5" stroke-width="0.125427"/></g><g filter="url(#filter13_dd_3202_39054)"><g clip-path="url(#clip24_3202_39054)"><rect x="87.5918" y="170.825" width="35.7468" height="51.9999" rx="0.50171" transform="rotate(3 87.5918 170.825)" fill="white"/><rect width="35.7468" height="45.4048" transform="translate(87.5918 170.825) rotate(3)" fill="#FFCAC7"/></g><rect x="87.6511" y="170.891" width="35.6214" height="51.8745" rx="0.438996" transform="rotate(3 87.6511 170.891)" stroke="#F3F4F5" stroke-width="0.125427"/></g><g filter="url(#filter14_dd_3202_39054)"><g clip-path="url(#clip25_3202_39054)"><rect x="130.805" y="77.2364" width="35.7468" height="51.9999" rx="0.50171" transform="rotate(3 130.805 77.2364)" fill="white"/><g clip-path="url(#clip26_3202_39054)"><rect width="35.7468" height="45.4048" transform="translate(130.805 77.2364) rotate(3)" fill="#FFCAC7"/><circle cx="165.752" cy="120.238" r="17.9612" transform="rotate(3 165.752 120.238)" fill="#FCA7A1"/></g><rect x="138.451" y="125.352" width="15.4354" height="2.10483" rx="0.280644" transform="rotate(3 138.451 125.352)" fill="#D0D1D7"/></g><rect x="130.864" y="77.3023" width="35.6214" height="51.8745" rx="0.438996" transform="rotate(3 130.864 77.3023)" stroke="#F3F4F5" stroke-width="0.125427"/></g><g filter="url(#filter15_dd_3202_39054)"><g clip-path="url(#clip27_3202_39054)"><rect x="127.984" y="131.044" width="35.7468" height="49.6168" rx="0.50171" transform="rotate(3 127.984 131.044)" fill="white"/><rect x="127.984" y="131.044" width="35.7468" height="43.0216" transform="rotate(3 127.984 131.044)" fill="#FFCAC7"/><rect x="135.756" y="176.78" width="15.4354" height="2.10483" rx="0.280644" transform="rotate(3 135.756 176.78)" fill="#D0D1D7"/></g><rect x="128.044" y="131.11" width="35.6214" height="49.4913" rx="0.438996" transform="rotate(3 128.044 131.11)" stroke="#F3F4F5" stroke-width="0.125427"/></g><g filter="url(#filter16_dd_3202_39054)"><g clip-path="url(#clip28_3202_39054)"><rect x="125.289" y="182.471" width="35.7468" height="49.6168" rx="0.50171" transform="rotate(3 125.289 182.471)" fill="white"/><rect x="125.289" y="182.471" width="35.7468" height="43.0216" transform="rotate(3 125.289 182.471)" fill="#FFCAC7"/></g><rect x="125.348" y="182.537" width="35.6214" height="49.4913" rx="0.438996" transform="rotate(3 125.348 182.537)" stroke="#F3F4F5" stroke-width="0.125427"/></g></g><path d="M73.7726 207.75C73.7894 207.43 73.9081 207.171 74.1288 206.972C74.3495 206.774 74.6196 206.683 74.9392 206.699L83.9868 207.173C84.3063 207.19 84.5655 207.309 84.7642 207.53C84.9629 207.75 85.0539 208.02 85.0372 208.34L84.9178 210.617C84.9011 210.936 84.7824 211.196 84.5617 211.394C84.341 211.593 84.0708 211.684 83.7513 211.667L74.7037 211.193C74.3841 211.176 74.125 211.058 73.9263 210.837C73.7275 210.616 73.6366 210.346 73.6533 210.027L73.7726 207.75ZM82.3441 208.214C82.3608 207.894 82.4795 207.635 82.7002 207.436C82.9209 207.238 83.1911 207.147 83.5106 207.164L88.2741 207.413C88.5937 207.43 88.8528 207.549 89.0515 207.769C89.2502 207.99 89.3412 208.26 89.3245 208.58L89.2059 210.842C89.1892 211.161 89.0705 211.42 88.8498 211.619C88.6291 211.818 88.3589 211.909 88.0394 211.892L83.2759 211.642C82.9564 211.626 82.6972 211.507 82.4985 211.286C82.2998 211.065 82.2088 210.795 82.2255 210.476L82.3441 208.214ZM86.631 208.424C86.6477 208.104 86.7664 207.845 86.9871 207.646C87.2078 207.447 87.478 207.356 87.7975 207.373L96.8451 207.847C97.1647 207.864 97.4238 207.983 97.6225 208.204C97.8213 208.424 97.9123 208.694 97.8955 209.014L97.7762 211.291C97.7594 211.61 97.6407 211.869 97.42 212.068C97.1993 212.267 96.9292 212.358 96.6096 212.341L87.562 211.867C87.2425 211.85 86.9833 211.732 86.7846 211.511C86.5859 211.29 86.4949 211.02 86.5116 210.7L86.631 208.424ZM95.2032 208.873C95.2199 208.553 95.3387 208.294 95.5594 208.095C95.7801 207.897 96.0502 207.806 96.3698 207.822L105.417 208.297C105.737 208.313 105.996 208.432 106.195 208.653C106.393 208.873 106.484 209.144 106.468 209.463L106.348 211.74C106.332 212.06 106.213 212.319 105.992 212.517C105.772 212.716 105.501 212.807 105.182 212.79L96.1342 212.316C95.8147 212.3 95.5555 212.181 95.3568 211.96C95.1581 211.739 95.0671 211.469 95.0839 211.15L95.2032 208.873ZM103.775 209.337C103.791 209.017 103.91 208.758 104.131 208.56C104.351 208.361 104.622 208.27 104.941 208.287L109.705 208.536C110.024 208.553 110.283 208.672 110.482 208.892C110.681 209.113 110.772 209.383 110.755 209.703L110.636 211.965C110.62 212.284 110.501 212.543 110.28 212.742C110.06 212.941 109.789 213.032 109.47 213.015L104.706 212.766C104.387 212.749 104.128 212.63 103.929 212.409C103.73 212.189 103.639 211.918 103.656 211.599L103.775 209.337ZM108.062 209.547C108.078 209.227 108.197 208.968 108.418 208.769C108.638 208.571 108.909 208.48 109.228 208.496L118.276 208.97C118.595 208.987 118.854 209.106 119.053 209.327C119.252 209.547 119.343 209.817 119.326 210.137L119.207 212.414C119.19 212.733 119.071 212.993 118.851 213.191C118.63 213.39 118.36 213.481 118.04 213.464L108.993 212.99C108.673 212.973 108.414 212.855 108.215 212.634C108.016 212.413 107.925 212.143 107.942 211.824L108.062 209.547ZM116.634 209.996C116.65 209.676 116.769 209.417 116.99 209.219C117.211 209.02 117.481 208.929 117.8 208.946L126.848 209.42C127.167 209.436 127.427 209.555 127.625 209.776C127.824 209.997 127.915 210.267 127.898 210.586L127.779 212.863C127.762 213.183 127.643 213.442 127.423 213.641C127.202 213.839 126.932 213.93 126.612 213.914L117.565 213.439C117.245 213.423 116.986 213.304 116.787 213.083C116.589 212.862 116.498 212.592 116.514 212.273L116.634 209.996ZM125.206 210.445C125.223 210.126 125.341 209.866 125.562 209.668C125.783 209.469 126.053 209.378 126.373 209.395L135.42 209.869C135.74 209.886 135.999 210.004 136.198 210.225C136.396 210.446 136.487 210.716 136.47 211.036L136.351 213.312C136.334 213.632 136.216 213.891 135.995 214.09C135.774 214.289 135.504 214.38 135.185 214.363L126.137 213.889C125.817 213.872 125.558 213.753 125.36 213.532C125.161 213.312 125.07 213.042 125.087 212.722L125.206 210.445Z" fill="#CED0D9"/></g></g></g><defs><filter id="filter0_dd_3202_39054" x="162.796" y="26.8845" width="234.783" height="184.467" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="1.88366"/><feGaussianBlur stdDeviation="2.72248"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.0421718 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_3202_39054"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="4.5267"/><feGaussianBlur stdDeviation="6.5425"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.0605839 0"/><feBlend mode="normal" in2="effect1_dropShadow_3202_39054" result="effect2_dropShadow_3202_39054"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_3202_39054" result="shape"/></filter><filter id="filter1_dd_3202_39054" x="197.787" y="56.9836" width="53.7194" height="83.938" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.7284"/><feGaussianBlur stdDeviation="0.45525"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_3202_39054"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.1821"/><feGaussianBlur stdDeviation="0.1821"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="effect1_dropShadow_3202_39054" result="effect2_dropShadow_3202_39054"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_3202_39054" result="shape"/></filter><pattern id="pattern0" patternContentUnits="objectBoundingBox" width="1" height="1"><use xlink:href="#image0_3202_39054" transform="translate(0 -0.0573754) scale(0.00294985 0.0023224)"/></pattern><filter id="filter2_dd_3202_39054" x="197.787" y="141.832" width="53.7194" height="70.2898" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.7284"/><feGaussianBlur stdDeviation="0.45525"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_3202_39054"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.1821"/><feGaussianBlur stdDeviation="0.1821"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="effect1_dropShadow_3202_39054" result="effect2_dropShadow_3202_39054"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_3202_39054" result="shape"/></filter><pattern id="pattern1" patternContentUnits="objectBoundingBox" width="1" height="1"><use xlink:href="#image1_3202_39054" transform="translate(-0.2) scale(0.00291667 0.003125)"/></pattern><filter id="filter3_dd_3202_39054" x="253.328" y="56.9835" width="53.7194" height="69.4564" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.7284"/><feGaussianBlur stdDeviation="0.45525"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_3202_39054"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.1821"/><feGaussianBlur stdDeviation="0.1821"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="effect1_dropShadow_3202_39054" result="effect2_dropShadow_3202_39054"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_3202_39054" result="shape"/></filter><pattern id="pattern2" patternContentUnits="objectBoundingBox" width="1" height="1"><use xlink:href="#image2_3202_39054" transform="translate(0 -0.183891) scale(0.00265957 0.00284954)"/></pattern><filter id="filter4_dd_3202_39054" x="253.328" y="127.35" width="53.7194" height="87.7714" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.7284"/><feGaussianBlur stdDeviation="0.45525"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_3202_39054"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.1821"/><feGaussianBlur stdDeviation="0.1821"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="effect1_dropShadow_3202_39054" result="effect2_dropShadow_3202_39054"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_3202_39054" result="shape"/></filter><pattern id="pattern3" patternContentUnits="objectBoundingBox" width="1" height="1"><use xlink:href="#image3_3202_39054" transform="translate(0 -0.102417) scale(0.00319489 0.00251531)"/></pattern><filter id="filter5_dd_3202_39054" x="308.867" y="56.9832" width="53.7194" height="87.4072" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.7284"/><feGaussianBlur stdDeviation="0.45525"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_3202_39054"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.1821"/><feGaussianBlur stdDeviation="0.1821"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="effect1_dropShadow_3202_39054" result="effect2_dropShadow_3202_39054"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_3202_39054" result="shape"/></filter><pattern id="pattern4" patternContentUnits="objectBoundingBox" width="1" height="1"><use xlink:href="#image4_3202_39054" transform="translate(-0.135088) scale(0.0026462 0.00208333)"/></pattern><filter id="filter6_dd_3202_39054" x="308.867" y="145.301" width="53.7194" height="70.2898" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.7284"/><feGaussianBlur stdDeviation="0.45525"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_3202_39054"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.1821"/><feGaussianBlur stdDeviation="0.1821"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="effect1_dropShadow_3202_39054" result="effect2_dropShadow_3202_39054"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_3202_39054" result="shape"/></filter><pattern id="pattern5" patternContentUnits="objectBoundingBox" width="1" height="1"><use xlink:href="#image5_3202_39054" transform="translate(0 -0.169993) scale(0.00261097 0.00279746)"/></pattern><filter id="filter7_dd_3202_39054" x="-1.08499" y="36.6288" width="220.934" height="209.545" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="1.88366"/><feGaussianBlur stdDeviation="2.72248"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.0421718 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_3202_39054"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="4.5267"/><feGaussianBlur stdDeviation="6.5425"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.0605839 0"/><feBlend mode="normal" in2="effect1_dropShadow_3202_39054" result="effect2_dropShadow_3202_39054"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_3202_39054" result="shape"/></filter><filter id="filter8_dd_3202_39054" x="51.0506" y="73.1063" width="39.6722" height="55.0537" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.50171"/><feGaussianBlur stdDeviation="0.313569"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_3202_39054"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.125427"/><feGaussianBlur stdDeviation="0.125427"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="effect1_dropShadow_3202_39054" result="effect2_dropShadow_3202_39054"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_3202_39054" result="shape"/></filter><filter id="filter9_dd_3202_39054" x="48.2303" y="126.914" width="39.6722" height="55.0537" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.50171"/><feGaussianBlur stdDeviation="0.313569"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_3202_39054"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.125427"/><feGaussianBlur stdDeviation="0.125427"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="effect1_dropShadow_3202_39054" result="effect2_dropShadow_3202_39054"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_3202_39054" result="shape"/></filter><filter id="filter10_dd_3202_39054" x="45.41" y="180.721" width="39.6722" height="55.0537" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.50171"/><feGaussianBlur stdDeviation="0.313569"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_3202_39054"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.125427"/><feGaussianBlur stdDeviation="0.125427"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="effect1_dropShadow_3202_39054" result="effect2_dropShadow_3202_39054"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_3202_39054" result="shape"/></filter><filter id="filter11_dd_3202_39054" x="89.8826" y="75.1089" width="39.0433" height="43.0292" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.50171"/><feGaussianBlur stdDeviation="0.313569"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_3202_39054"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.125427"/><feGaussianBlur stdDeviation="0.125427"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="effect1_dropShadow_3202_39054" result="effect2_dropShadow_3202_39054"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_3202_39054" result="shape"/></filter><filter id="filter12_dd_3202_39054" x="87.0643" y="116.892" width="39.6722" height="55.0537" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.50171"/><feGaussianBlur stdDeviation="0.313569"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_3202_39054"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.125427"/><feGaussianBlur stdDeviation="0.125427"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="effect1_dropShadow_3202_39054" result="effect2_dropShadow_3202_39054"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_3202_39054" result="shape"/></filter><filter id="filter13_dd_3202_39054" x="84.244" y="170.699" width="39.6722" height="55.0537" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.50171"/><feGaussianBlur stdDeviation="0.313569"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_3202_39054"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.125427"/><feGaussianBlur stdDeviation="0.125427"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="effect1_dropShadow_3202_39054" result="effect2_dropShadow_3202_39054"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_3202_39054" result="shape"/></filter><filter id="filter14_dd_3202_39054" x="127.457" y="77.111" width="39.6722" height="55.0537" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.50171"/><feGaussianBlur stdDeviation="0.313569"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_3202_39054"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.125427"/><feGaussianBlur stdDeviation="0.125427"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="effect1_dropShadow_3202_39054" result="effect2_dropShadow_3202_39054"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_3202_39054" result="shape"/></filter><filter id="filter15_dd_3202_39054" x="124.76" y="130.918" width="39.5492" height="52.6739" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.50171"/><feGaussianBlur stdDeviation="0.313569"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_3202_39054"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.125427"/><feGaussianBlur stdDeviation="0.125427"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="effect1_dropShadow_3202_39054" result="effect2_dropShadow_3202_39054"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_3202_39054" result="shape"/></filter><filter id="filter16_dd_3202_39054" x="122.064" y="182.346" width="39.5492" height="52.6739" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.50171"/><feGaussianBlur stdDeviation="0.313569"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_3202_39054"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.125427"/><feGaussianBlur stdDeviation="0.125427"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="effect1_dropShadow_3202_39054" result="effect2_dropShadow_3202_39054"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_3202_39054" result="shape"/></filter><clipPath id="clip0_3202_39054"><rect width="396" height="264" fill="white"/></clipPath><clipPath id="clip1_3202_39054"><rect x="181.195" y="40.7579" width="197.983" height="147.667" rx="2.18766" fill="white"/></clipPath><clipPath id="clip2_3202_39054"><rect x="198.697" y="57.1657" width="51.8985" height="82.117" rx="0.7284" fill="white"/></clipPath><clipPath id="clip3_3202_39054"><rect width="2.91299" height="2.91299" fill="white" transform="translate(200.883 133.729)"/></clipPath><clipPath id="clip4_3202_39054"><rect width="2.91299" height="2.91299" fill="white" transform="translate(211.08 133.729)"/></clipPath><clipPath id="clip5_3202_39054"><rect x="198.697" y="142.014" width="51.8985" height="68.4688" rx="0.7284" fill="white"/></clipPath><clipPath id="clip6_3202_39054"><rect x="254.238" y="57.1656" width="51.8985" height="67.6354" rx="0.7284" fill="white"/></clipPath><clipPath id="clip7_3202_39054"><rect width="2.91299" height="2.91299" fill="white" transform="translate(256.424 119.248)"/></clipPath><clipPath id="clip8_3202_39054"><rect width="2.91299" height="2.91299" fill="white" transform="translate(266.621 119.248)"/></clipPath><clipPath id="clip9_3202_39054"><rect x="254.238" y="127.533" width="51.8985" height="85.9504" rx="0.7284" fill="white"/></clipPath><clipPath id="clip10_3202_39054"><rect x="309.777" y="57.1653" width="51.8985" height="85.5862" rx="0.7284" fill="white"/></clipPath><clipPath id="clip11_3202_39054"><rect x="355.301" y="58.6213" width="5.09834" height="5.09834" rx="2.54917" fill="white"/></clipPath><clipPath id="clip12_3202_39054"><rect width="2.91299" height="2.91299" fill="white" transform="translate(311.963 137.198)"/></clipPath><clipPath id="clip13_3202_39054"><rect width="2.91299" height="2.91299" fill="white" transform="translate(322.16 137.198)"/></clipPath><clipPath id="clip14_3202_39054"><rect x="309.777" y="145.483" width="51.8985" height="68.4688" rx="0.7284" fill="white"/></clipPath><clipPath id="clip15_3202_39054"><rect x="21.0996" y="45.1871" width="185.918" height="173.883" rx="3.5434" transform="rotate(3 21.0996 45.1871)" fill="white"/></clipPath><clipPath id="clip16_3202_39054"><rect x="26.9668" y="51.7041" width="173.516" height="140.322" rx="1.7717" transform="rotate(3 26.9668 51.7041)" fill="white"/></clipPath><clipPath id="clip17_3202_39054"><rect x="54.3984" y="73.2317" width="35.7468" height="51.9999" rx="0.50171" transform="rotate(3 54.3984 73.2317)" fill="white"/></clipPath><clipPath id="clip18_3202_39054"><rect width="35.7468" height="45.4048" fill="white" transform="translate(54.3984 73.2317) rotate(3)"/></clipPath><clipPath id="clip19_3202_39054"><rect x="51.5781" y="127.039" width="35.7468" height="51.9999" rx="0.50171" transform="rotate(3 51.5781 127.039)" fill="white"/></clipPath><clipPath id="clip20_3202_39054"><rect x="48.7578" y="180.847" width="35.7468" height="51.9999" rx="0.50171" transform="rotate(3 48.7578 180.847)" fill="white"/></clipPath><clipPath id="clip21_3202_39054"><rect x="92.6016" y="75.2343" width="35.7468" height="39.9589" rx="0.50171" transform="rotate(3 92.6016 75.2343)" fill="white"/></clipPath><clipPath id="clip22_3202_39054"><rect x="90.4121" y="117.017" width="35.7468" height="51.9999" rx="0.50171" transform="rotate(3 90.4121 117.017)" fill="white"/></clipPath><clipPath id="clip23_3202_39054"><rect width="35.7468" height="45.4048" fill="white" transform="translate(90.4121 117.017) rotate(3)"/></clipPath><clipPath id="clip24_3202_39054"><rect x="87.5918" y="170.825" width="35.7468" height="51.9999" rx="0.50171" transform="rotate(3 87.5918 170.825)" fill="white"/></clipPath><clipPath id="clip25_3202_39054"><rect x="130.805" y="77.2364" width="35.7468" height="51.9999" rx="0.50171" transform="rotate(3 130.805 77.2364)" fill="white"/></clipPath><clipPath id="clip26_3202_39054"><rect width="35.7468" height="45.4048" fill="white" transform="translate(130.805 77.2364) rotate(3)"/></clipPath><clipPath id="clip27_3202_39054"><rect x="127.984" y="131.044" width="35.7468" height="49.6168" rx="0.50171" transform="rotate(3 127.984 131.044)" fill="white"/></clipPath><clipPath id="clip28_3202_39054"><rect x="125.289" y="182.471" width="35.7468" height="49.6168" rx="0.50171" transform="rotate(3 125.289 182.471)" fill="white"/></clipPath><image id="image0_3202_39054" width="339" height="480" xlink:href="data:image/jpeg;base64,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"/><image id="image1_3202_39054" width="480" height="320" xlink:href="data:image/jpeg;base64,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"/><image id="image2_3202_39054" width="376" height="480" xlink:href="data:image/jpeg;base64,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"/><image id="image3_3202_39054" width="313" height="479" xlink:href="data:image/jpeg;base64,/9j/4QBkRXhpZgAATU0AKgAAAAgAAgEOAAIAAAAoAAAAJgE7AAIAAAAOAAAATgAAAABodHRwczovL3Vuc3BsYXNoLmNvbS9waG90b3MvbDhmeUs5UlMtT1UAU2ltb25lIEh1dHNjaAD/4AAQSkZJRgABAQEASABIAAD/4gIcSUNDX1BST0ZJTEUAAQEAAAIMbGNtcwIQAABtbnRyUkdCIFhZWiAH3AABABkAAwApADlhY3NwQVBQTAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA9tYAAQAAAADTLWxjbXMAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAApkZXNjAAAA/AAAAF5jcHJ0AAABXAAAAAt3dHB0AAABaAAAABRia3B0AAABfAAAABRyWFlaAAABkAAAABRnWFlaAAABpAAAABRiWFlaAAABuAAAABRyVFJDAAABzAAAAEBnVFJDAAABzAAAAEBiVFJDAAABzAAAAEBkZXNjAAAAAAAAAANjMgAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAB0ZXh0AAAAAElYAABYWVogAAAAAAAA9tYAAQAAAADTLVhZWiAAAAAAAAADFgAAAzMAAAKkWFlaIAAAAAAAAG+iAAA49QAAA5BYWVogAAAAAAAAYpkAALeFAAAY2lhZWiAAAAAAAAAkoAAAD4QAALbPY3VydgAAAAAAAAAaAAAAywHJA2MFkghrC/YQPxVRGzQh8SmQMhg7kkYFUXdd7WtwegWJsZp8rGm/fdPD6TD////bAIQAAgICAwMDAwQEAwUFBQUFBwYGBgYHCgcIBwgHCg8KCwoKCwoPDhEODQ4RDhgTERETGBwYFxgcIh8fIispKzg4SwECAgIDAwMDBAQDBQUFBQUHBgYGBgcKBwgHCAcKDwoLCgoLCg8OEQ4NDhEOGBMRERMYHBgXGBwiHx8iKykrODhL/8IAEQgB3wE5AwEiAAIRAQMRAf/EADgAAAIBBQEBAQAAAAAAAAAAAAEDAAIFBgcICQQKAQACAgMBAQAAAAAAAAAAAAAAAQUGAwQHAgj/2gAMAwEAAhADEAAAAMrNZ7v8WLLINZZEURpTUWwFFsGssidBrI1lkHRGFNcYU1lkPSywoXGlNRaQUW1AgvKEFxQotI1FhSspZM+GiMI1lkTWWEa4wprLImssKa42oElxBMeUIqcUJLSCi2IWWEFVMKFxhBcaQUWwaywjVGkLLGnJ5VHEEl1SEFxBJcUKLYhZZAWWxCywgosI1lhBRbBrLCCi2DWWEFFpGotI0l0GotIKjYOzlheBRaUKLCCi0jUWEFxhGssILjYNZZUNNTCNRaRpqaQTUwjUWkaiwjWWFNRaU1FxQkuKaY8osRYcusuNI1FpGqNIKLSNRbAVUwnpVTCNRaRqLShJcU0lx8iS6pNBeUIqcUJLakJLoNRdE1FpBUdE7CWzawrLCCi0ppqbUmgviFR5TTHlCS6ryILihNTSNNTShUcRpLomqphQotI1FpTVHEaS4gkuI0x8CwF5zYEF8QmppTUXFCS0jTU0ppqbAXGlNVTCmougKLoNUeQRU6DUWlNRbBrLCmouKElpGkvi9Kjo3Yi05NVRaRqLKkJLiNJcQSXEaS0pqLSNcbE1lpBNTSmkuKaS4jSXEaS0jUXEaS0jUW1DSWkao+DsMdPeossKFxpGotKai0jSXxNUdUNBcRpqaRpqaRpLiNJcRqLSCS4jTU2DUXFNJcUJLimkvKaJ9ERj5dMusotqGippBRaT0otIJqaRpLiCi0jVHFNRbUNMefLRU4oSXFNJaU1FtSaamFCi2DVU0tqjiNMeU8fLpsaqi4ppLiNJcUJLimkvnlpLiNUciu3mqr5UVDp9yln+jJq3E/DiefUzs8CXKVwd0nid2bD2nVoboCc5ymPm3CpqcU0VOiai2pNJdEKjoOwlpzayi0gosKa40trjSNRcU0x1Q043leH8y+g/puWZY1y76H+SqxfPV7xfMDz/ABTHu+V/2alX0Osbm+THUec+1PWfyP8AXa+/Ovz1NPRfl9JcRpLimkuI0lxTSXQePl82NVNTSNUcUKLSmmptXloqaQTU2pNGB7DwLmn0HkSMrw7nH0BcrP8ARaIidvWKZ5i0PP8Aj3Rjny9R51s3Fn/T52s99hvIT2As/JVFp6N8uqLoNUfE0l5GkuI0xxDHy6ZtRRbUmkuKElxGkuiai2oaS6Jq1jtTW3NfoHKnXbGec93+6xus0TYcixrLbFFTXjTLH8HTaHsawrup6zz2P8cvZGycxBbOkfLii4gkuKaZ9EGkuKEx5Tx+OOXVSXFNUcRpLimkvI0F8GotIK1Nt/VHNPoDNFvtXPO5/XjrPiiLFePiuhh5vxLTSjqnPMl+K1bRMjPZ/wAW+9N2t9cnTG7el/Kqi6bcImpxTSXEaS4ppj4nj9TZs6iywjUXEaS6pHz1OI0lxTTU0pp01uvTHNO/7GXi9ooPabvacWvelKV4nsvi/X3uS/jvlmtPn5cwwj7Jmm33vPgDvmPmN+bk0nvPpPy2qOM5z1JeU0F5QmPI0x0HYC07GqotqQiptSaY6pNFTovSi6IUWkaeH+5ufeed85X65Wrln0bdrximZ7cXZ9O7z5E18/G1gy/F5S02a9WjIbTy35PQnz39Aouw7/3fobf3S/lFZYZqgLLCmotJ6UWlNJaR2KNqzaiS4jSXFCKnVDQXlNFTSCi0pq4U7w5l5z9B6M6Rxu+cz+gPoufw5FtRdt556k5j8e+HsL2Jgm5c8fzPFOqLbyLmP0a87fQ6KtW9t8c+dEdK+UllpmqGotIKjiNJcRpjynYqmHJqLjCmssI1xsTXG1JpLoNcaRq4M755B519A4VvLAs25r3sXhN72oz4OPu2+YzHxjrnaWs8l9xzZGvsyu/G8c9I/OL0YhLhu3oTnrozpXygouMvSElpQosK9LNcTplcRZC07GqotgLLKhqLCNRbAxJHjHqCNuvv8zyX0qYfdbg7l37qn1Pqba3kDjVZv/tn9/mZnkPZ+6rhzn0DqevOvmLp/mSx6LsswjaUvAW/0R86/QuKtG/OlOeOjOi/LKi0y9LVHVDQXwaS+ITHwdjL5l1Ul1SElpTUWxNcbT5fihpHM9JQnXNw4nkuoMex2VivOV61s9WK2h2rK7Ny518oHbLx1ty/1NFbnI3InX3LlgjPm2TrTJ5iCvHZ3HGi46f9xu0eE+8ejfKiywy1MUWFCywDoNcHTK4O0SurJqrjSNRaX6SXwEx8H4C6m2lqeD6vuDUm1tLrPf8AIMHybyYNPttWlJ7zyHC9g897XkXQehOlImY5s4n7Z48mtNv0UYHKV3oSDQ2rNe3fc3AfoH0T5aSXmVpqC+JpqdE1FpTVHQLKWnY1VFhQuNnlrLIFEYB/nr13nmAQ/VNnaxz3Wxn+nJbT9SWGruNm0JTZme692fQuz33qrlXq2Llud+Fe7uTZqGxyw7Cs0lD7D+/5ubNOd91u8vPH0T6B8xLLZKU+iVlOiVkdErKKI2Ds8Yc2qqOqT+ep1Q0FxXpNTa0fnIwvKsFi+nZzhOaYEs18+quvysJ+j67DpSe0djau2dSus51zX0NvmIn9Lc/dTcp6l40hdkTovzhum5WLRED0v3R7786PRy+/MSy0yVRUWEaiyoFFhGuMg7SWzJrrjYNZZB0GooorLB/nPxRX3xfS/usGW61eTIqfuq8LDviN30JZ+yMTy2k9dyzQe1+x4if0xzJ15x5itWh8usF76H8+7Mu1j0/X+je3/of5pemF9+ZRKzJ1JZrKaywpqLCC42ItMacuBRaUKLYNcbAW2NH+Yj7LNlkd0m4652trX07kXPRhCrx8UdMXjNrPeqV1fKNLbS66h7Dr/knsPjTxYtM7Gw6m+8J2PvHRPZkH0Xov0y4V7wvvzIuNm/VVlhGs1kFmuDplcHazXPeCmVkdEZE6Iwgqpi0/y33vGs80eitwbO8Zbb8maWBLEFXhUdNXPKvmzCodOr07u7o6NlsA5W7X4eUxrR2w7DbeUfV2VyJ1zGXDtj0C8+fQm7fPQlZkKsuNI1FhGosILjYFqLYa6ywtqLChZZB0fHcLa/X5ZLxZMt0ug1YdsDHh/ZaNjYx5WIm5/DHTedZXi+S03qmwtLb16jhbFoDgnuzgKQjrXfkVyUfiPoFwH35vwncnpH5w+kNp4sDUd+sUSsprNZBZZPLojIFtJOXDSaiFEZBrLCC/gunxr1+VvNcK2Jp3y34XszCvTuKL0nz51/cRfYuf+vPsT+qn9N2RrPN+8oWy8jcL9xcGSsQcrxn7d7SvfWnCXeu5Dduemvmj6Y2njVJrO5WlmsoWa4Og1QKSSO3mqZ8FJqKdBqIURkTotl3sC9fli2NrXOta85LrbL8Tav1Hz/OHz2td3i56+ZHbRUej5vabx3xBWribkLrbkKXh7HkHwTa1BcMa+aQrvsJ7I+SXrhZ+S0xh2q4s1kFmsprNZCgNif8A/8QAIRAAAQMFAQEBAQEAAAAAAAAAEgUGEQECAwQHCAAQFCD/2gAIAQEAAQIAj8gRj8gRERGI/IEYj6BEYpSBAAERj6IEREYj6IEYj6IEYiBEBEREYiBERGPyBEY+iBGIgRABEREYiBGIgRGBGI+gRiIgQARERiIgREYgYiBGIgRiIgQAKW0tERiIEYGIGIiBGIgRGloiIBS0REYiIEYgRiKWwIxEQNLREYiIEREYiBGIgRiIgRiIERERiBERiBEBGBgRGIgRilIEaW0tgYilBEYiBARgRGlgDECMRECIiIxAiMRAgIwIiAAIjAiMCNLREYGBEYgQARERAKWCIjAiIjEQMQIjERAiIwIiAUtGlsRAiIjAiIiIiEQIgIiNLQAREYERERGIERpQRAREREREQpYAwIwIiIxAiIiIiAiIiMQIjS0IgYERERGBGKUERARERGIgREYERERERpSBEYgRCloiIjEQIiI0tERGloiIiIiMCICIiIxECIiIiIiNLRiBEaWiMCMCIiIxSkCIiIiIiIiIiIiIiIiIiIhEQIiIiIiIiIiIiIiIiIiIgAiIiI0tGloiICIwIiIiIiIiIgIiIiMQIiIiIiIiIiNLREREQARGBGBERGBpaIjAiIiIiIiIAMRS0RgYERERERGloiIiIiIiACIwIxSl9Dzbd+3i+uv6IvY/UOH1JZ6ns9WM/wBC1sERERERERARERiIgRGBcKleiaCBs0jNsdeyW5bc+HLYrNvbvtEREREaWiIxECIxAiIiIvf562bW3eqX4MGDpuPoCMxMHN9rYoi332iIiIDAiIwIiIiIiIiIv3G70HeRmtuUst2nhsLFWxrN2u3VNzXWxAiNLRERpaAiIiIiIiIiL7+6drb+Rt7tqbbYtYX82OdNvm2pvW61REQARERGIERERERERERf+u6GkpJLT3cNl+9u76zuNnK39hVsw7FlBGIGIEAABERERGIEREX1j6Xbubba+wJuT7Ja+kDnjf52nq1+P7FQQABpbAiIwIwIiNLYEQAR6BovJpLOFr5cX2xvau25d9pqTf21vdt2dX6IERERERERgQAREREYh/16TmVFlu3YEvbom39DT09I54mXq11eaPdBUKWiICIiMCIiMRECAAAw/dTpTZdO63b9jc33vp9bWtS7WRLE1zU3eAY0L6lowIwNLQAIgAARGBpaIiPQL+nOtzPVFTmvzNOb/B+Pq1ijVC+v1i4TlbtKWAAiIiIwMQIiIiIiI1t7xseD1bFa2MbW2rVFB7A4MShYh/Ztit/C7G7SloxEQMQIiIiIgIiIjAuzWZrhx4W4jNpA2MvCOHvHHu1RMmWy+nGdVr61KQIiIiIwIjECMUpS0ApaI9/s8BNXHa39lFUs682+8v7U3cCNi3U677jWNsUpaIwMRAiECIUsARERGIh4KXLnlbotZPQrV/d88cK6fZtUSPn1r5Kco0Glo0pS0RERGlogIiIxAjAiIj3Nq+J+Tk3NtJeCtvovozrmts6yPrKjoyWcruaVaWiACNKQIxEREQIiIjFLaUfjq4Y/bdRtaqapOrY4hwTtFctyXblRMtOU4Wfq0tEYj9iBERERGIjde9OhU6BR90e/YNrzBoJ79QXRoPXOsJzf7vVuORLcOwoZcvKFNlbtPoEREREAGIERERrb6i1murvX0pxd6eYMXp7S5KnqeZgvJk9x5t3RGe3fsyLdobWwiZqcevY/1LRClgjECIiIiIUtEa2+mmdrc/WOeovP/H+/6BxM1prCY16tlH4dq8lQfQNqFVO29tEu1eWdQ5f1WloiMRAjEDAiIxH133qO1J1Ois9vanPUPobWbGi3Lmte39jl+tzzN3vY3LtTY3FcXm6vLWzS0Yj6PoiAj9gREa2+m8jU2nuuMbc5qo9JWWftJe40HagW83y8ORvRNErLh23RoX4LfNvnpHpSIEApaAiIiIgAjF1PSt7Ytfm02/mRk6bXn9zZuZ3yFm5+n84s9DZrcOf59atMz6VPK99LQEYERgREYERiBrb6NvYq66Oh8+crBX+nvhkLjeXGkst7S5xj5uk+jLLdLPXoWxT5P85+e0GloiMREfkUpAjEftaehL2P90VZZH3L8vVs/NNhs3NKqLYwNNm6/fa7LS30B3sz+TpaZ5PxUp+RECMQMfRECIDSzvdWGvPBcY7i5k4ekuNk7jSWG64UVR58u6T19AI91M6X0LNqZ9Xm/ndMpbEQIxECIDSwRGBi23td7axdA2Wtp8px9Vx81yN7G27UfCxcHDdH1eq3X4FHo+DWz9M0vJNtKQIjEREQIiMREDjp1HYRVNyulvLzRdLsdbdVkmrdzIjc5jjQlT0Q261V737uJV+jpeccdtIiPyBgRgY/YgcNvRr+TWdDvbuvyqnXsbIs51VE106vIvuZqfrFxXXOBRfefR3Oj6vkmltBEYiBGIgREYERwUe9/OVh+rDdU+drXT19lZ2fkb2BFQuX0aST6TaV9XPde7EXY5dxvgrOttEYiBGPwYGBGIjFR05OP6z/AMLVTuZ6XX9Tm9jFwJWpp3ciw83cvqR3VuU1F840nN5bzcwupbEfQIx+xEDEfttFuvPc3QL27s8/xdH2mfiaGk2ktHSecXsxm9xaVa7OdzLDZz+YfuU2Up9P5AjS2I/IERiIzfKN/Jk3oicz0rkiT1xM53gY+uh6Oi0WIzeYLHqJx5ljRxu9FSsvmO3lX1LRiIERj8iIEYiBUK7NrOwv9NS7Oeo790GiltTcY+4nt5vWc+5l6vw6+FYTVzKwc/lbX5FrW2x/iIGIiBiPogVqmT5hJ/RkxupPJUvrSawcDW1WvYn7SRXmrr9cKOXHt4VqjIp5Sv5L9S0Rj6IEY+gRj9gVSzLTmye9cCfYwE/oqc2sbSQGWmVSsGbnPO/V+tdquHR/uUHN5a1+RaltoxH5AiMQIDEfQKj9mu5O7X07W2uc3eHVngyFpkYULDjztJNaPRPXKjv6itprnzNu8rX8k+ttEY/zEQIxH7C3WteUZenb6NTm2fq2XneVoJTUSsuhZfyXmPqbVrjX09RUERwbnQPMLLt/YEYj6PyIGI/Idl1PucvF8PtvuhjPDo7zabhZmgn6+ztNZKY3QfVu/XRUEtwfMu53feXb7afRAjH5A//EAFUQAAIBAgQDBQMIBAcOAgsAAAECAwQRAAUSIQYTMRQiQVFhIzAyBxAVIEBCcZFScqGxJDNDUGKBshYlJjVlc3SClKOzwcLSNINERUZUVmBjcHXR4f/aAAgBAQADPwD7Ifs5+uPnP24fOfeH7Mf5n9P/ALGj7Kf5m9Pc+n80n6x/mD096fmH2Q+7P2QfYfT7EffD+b/T7IPden80+n/yN6e59PtHp9X092PsA+w+n2UfM2v+MYbdARiW59q3j4L/APrDRAl6gLv95U8vwxUkhY0Ym43ZFQb+pxVjQXmj3A2VP+eKgEASJuAfg/8A7jMsj4VzfM6RoGqKaJXTmxkpu4BuAwxxkDZqHKW/8uVf+vHFIsXyTLGAIBs0q4z7e/DdCbeU8mM22vwpTH8KiT/tOOHczpaQZg5oq2WTQadY5plBLaUs4TfVgg/MMenufT6np7yWhpa6oSnDcinklGpratC6rbXxnEdDXVVaJYVR2iVe5CQxgWb4FLttq6lsLleUZIGkWWSbL6WpklCaSzTRBjcksSfMk4BmP+rjeMfhiKN1DSgGy7dT+QwZuA+JtMLW7Gxu3d6MDhhI3s/yYY2f2b9V6WOPj7j9R90+WKPXyjUoJNdtB2N74jGb5VaVdqqDx/8AqDHff9Y/ZT84+pbKc6t/7hN/w8ZouU5jJWRUsMb1LMI4naRg3Yox3nIVbWHQDFJFlmQRxZg9aRlNDqfWspDcgXW6AABelsSVGaVFNG9PAYoIp/asWeQOWXTGieI07km2Kh83po2qJZIuxSyyKGEaBubGikgbn4jhIjZEVRZfhFsa+B+Jx/k+b9gxmE8sNTR1VUjKEjKwnum7ncgeOM1qcuzimqKqpEvbIER22cKeum4xVzZdUmpneR1qWXU/W2lSBhhnYOtrdqH4YtmdAfKeE/7wY7z/AIn3Xpg/YZjledkTBVFBNYBRf+L8ScZpBleZVGZ5gtTJz5IghLyBStIjXu9h6bKMQ5ZSZDTw8zT9D0D+0cubvTgnc+HkMcqp4zjEE7NLldCNccZZEAlm+N+gGJWzZWUKEGXNruDqN6iK1sRB7a9RAGy94/kMGv4Fz6ohi9lNlczoXOklWjJB041lEaZIzcNu3XScJTyZmKh1lhkeJ20tew3G+Cnb1KswM910nVZSosMQjPb6XDdpQ9G9MKK6kNyLSx9QR98Y7zfifn9MH3vp7s/RGd269im/sYr6XJq+Svr4CZKqe0UUXLRWFIguGclmxSmDJvo8Szx/RVFeR3dgXEC6u/Kbn+rbGb9s4rhhpByZMuojUTLHzeUoeW12LKFJ/A4Z86hLzM6Llzl0ckhiZ4wNhYbYCRsqgDbYLtiEfJ5VpEE0fQCW0W0/+FF+mBmaU0i03MkjZUDA2IVm3HUYKJnlHU07KkphuHJGoXa+HpqbMI2R1tOps977xjzwoze+hb9ojOrx6jFqmm3/AJRf7QxudveD7Az5ZnntnA7HNYA6QO56YOU5bnc81caic1NTEzcsKLJSo17sXc9fFsUuXQZRBTwiNBlVCxFye89OpJJYk3OJ4n41ijpJnSWgoNcqlAkYDzfFqIJ9ABh/pbUHUIuW98Edb1CWsb7Yi0SBNTnT9wFv3YFVwLL2anVInyNQNRHhS2OyYp4xTrPUxxFiGW566bYo4KjMJTNHUQlI2kA8AGPW+FM2YmOzq0iFdJHdGnpvhPpjmGnbVzojewJ6jyOF58F1cWcfdP6QxcD6gx6fXHu/T6pbLc7sbHskwB69VxHR5fnCTZ49XUGprGeK8ahP4Mo1CKIXW/mcULwZb9GUIigGW0ijUOUNYgGo23O5xnbz8XJCq9l7FQmpdNA0jVLYXclvyGFfPYpCxOjLTcP39WuoQX36Wtj2MtvBen9WIpeEkERUo2TRBSltJ/goG1sDM6WktBG7pIlixAIDWuAThaSXNoKinjSKWCPVcqVYa3BvhqNcxjMJju0TAHxuuKYV5BKa9UflfqMWlj28f+eO4nqo/d9UfXP1PTB93HLl2fatRvSTAXJIF1tsMUuS5HnhhmnkY1eYRlnKqto6dSLJGqqDigiSgjoVp1hXLqQAQaNAbkLq+Da9+uKoHjRY6QPC1Jl/MlaULo70tgEsSxOAmbs7TiNFy1QQbAMTULbc+VsIYZtCPJ3Oqrt08zYYev4epnhSOON8rgtc6zbs4X7u2KSkjpRV1SRBn7hO1ygG25xl4qq2eKrimRIFeUXFgquTuRfEMtVmD00sUquUbuN8PXY4iWsZWiXXeP7yX8PMg4IkW8T9T4A4BhhPmifuH1PT3Hpge49PqenzhcuzvU+gdmkF+lvXfGULlOd9jzCevmM2Yc2UzPOLckb32jH+riLM46Gegp4oafsFMii6n4IADYRbftxmLVXFkQq2EBpaAyqpEeq5lsNgSfzGIRnZmC2aPLVHQG+uoG5JufDBMM+1yYj/AGcRzZRRNHIrI2XU1mU3B9gox2qhaICMtzQF1kDowva+Hhrs2QwRhZcsKgLpIJ1N5YkomrEaDl3hpz4b7G/TEKZZmtK9dEl0dhE7qCToBFgfMjBLgX6k44o4j4XyXM/pZ4TU0xfRs4XQ5j6lfTHET19Ik+b8yIyoHXQveB6i+n7QPnieiz0Mgu0Mik+O+2KDKskz809OwZqvNYzLJI8jskcIsCzk4pauenkpaiOaIUFIoeM6l7sCggEYl7bxaBEpQ0uX6mLWI3l6LbfEdNnntKpYk+i11KxUAntIt132xlYE0MUrTyFNOmJSxuVxTTz5TllLHEJ6qmVKdZZVXm9ljCSFBtcKVxT1kkscvaFAlb4OWdwfUjFFSvPUmWoCtAYSDEhtqNwdnxTwySvBVGVSkaHVC8ZBT8NXW+KABopAjlklUddiVP6ajCaou8nUffXGjgDhgEW/gUht+Mz4Hb6bb+VT3Y+sPdRdkziNpQpeNlABs25HTHCSQZjRZTI1ZW1NRmG8XNqJJFdbIxZySRjNs5q1kosmFNGKenhvVOAfZxhCdK/hjPaup4j15pJFGsdDz+zoAHDByqliRa3oDfEVXl3FWaTMk8WXzQRoJ2leS7wo5tYqnVr3IOKbKcn4Xjgn18zKaKeRgsa2eSPUQeWB+25xn9LnPCeY1lDFCmW0edSSLK6mUfSFdqhZAL9Vxor61bdKmUfk5wOyT3Qt02AvgWqLRlO+NiNP3cUMU9LyICkyuwk9k6bFPEkAG5wSydR3sVLcCcOEGMAULixUn+Vf1xU/SVJqaMjmpeykHp+P2t6DiDIRCQPpCuzKOo197UsVE8q6R4EMoIxVZ3BNUcUzVU7a8y7OZElUvamg06REBdcEkkjxXEwPGLh0EQXLQ11NydL+N7DGUjhvjgvmDtV9sg5dOkshWwp4+9y07t/VsUc2V8MjLaBYo0yiiRtkiBk5Q1GyXO/rvjhWHM8gyymzYTT51AaakaJCyOtNKRLvuAQykb4Zc3zReY21ZUeXg7Yl7LUaXubDbSMTe3DWB1L4eh9TjNUdWnyuSKGqk1RzMjqhsCRYnY3xJqjOlT3h0ONHAvDwI/8AQW/bI+P750f+dT93uz9ggnzGqdoo3aE1OhnUHlu4C3UnobdSPDE1DxNCaOIVhfLsxhPL5kgTWsV7iNWOw/AYlJF6g27uyAD9u+IpJuK52s3L+jgA41k6o28WO2KA8K8c5g8Z7QldFHGeY4UDssXRAdN9+tsZa+X8MGgaAxJktAGMAAXmCEF76bAnzxm2X1vDtfWVlNEcqyqvd4U1SGQ11eZ00sLAWXrjRxDni+VfUj/eNhuzVOlbnTsOmHLTlk07p438DipNLTIaSYLFMpMhZClipAtY33vjdP1hh24H4eIqpVvQHYabfG3ph1zajPa5GAmXYhbHb0Hz+n1PT6g96MD5sxkz7h1KRZyrT50JxEGIKdhcKHt4aiLX8ccT8KU1bBnWSVq1U8GbSLE7JrETRU6631vsMbqL/o/uxDG/FmuuCEvlwEWpRrtEd7W1G3pjJRw1xqooTJmD1yiOUwgEIKWLYO9rfgMSZxQZBNDEkSLlFFCNTGQ+zhA8Aoxk8+YcP5flNFmFUM6pNAmiiuIKennMEjS9NADjc4ROKOIV722Y1Hif0zhjBUaXcHTt3sS65w8rfctuG8/QY4lpMtpqmenjFBVyRhJfZliLkrsDcdMSWX2g+Ifdxo4H4dH+T/3u2D9LUf8AnV/cffH6g9zTQVmZc+oSAFKwAyMBqC2ZyAOoUC5wueZ68/C9dDWQjLsyinlj5bBAOSX3c26EdATiM/E0j7r8THy8hbESrxdMoKlJsvUBe6u8PiBjJRwfx4X7H9ItWgJqCdoMYpoen3guEqKXh50SZQmR0C+1jaInRB1AcA28jgZRPSVNTmvNmy3hmqQJFFZHNZVtU3JY3GnpjTxhxKP8pT/tbDcme1r6Nr4lEsuvT0QjTf188Z1J8mvDEUmX0qU1OtNMJ0ncyMlza8ZQAE6vPFgPxGITwbw8SpucuBNmYfePriJc6omGu4lX77EfCfAn7XmOc59k70kCulKM9aZywUR8+kMCHfzZscRfJ9T1+WVRoJqh6DNqiTRK+iON+zrsdG7d3CICXYL8PU28MUhHFhXmySdooAgQSMthAL3t3MQJwhxtlSUtqiStMjM0iLZRTREXVbnwxX5jS5PLVQmmIymjQR8tkJQQCx9pvv52xW1lbwvS5Fw3LJDnVCGnZ5hG0FNDVGlcG53G18Rx8a8Try1sMwkwhgnsLEofE4KzShmPwoRZm8z5k44pmySlpZeJRLl6tHCtFc6441eyjePw9Gw1h7Ruo8sIvBnDoLrcZcniMIc6o7OP4weP9E+7HvR8+VZNmFUMwqdDzxZg0KIpZpEpbTTW9QgxS/KLWVua8O1kkNJHl2YxTazJEXWN4CwASx31YhVwRCqm48Lnp64hMHFsrIDIKyhVT4gdnW9sQrwDxxShZuY2ZyMSIn5YApYrantpxK8OWO9HJTlcmoxolKFrLTCxOgsN8ZJQCgmqaqqnlyvhtREdSxLeqqHqmLAA9G2GF/u44lOsWar1X8DqVThDDKBIoJQ73GHWd9Uoe6LbYD73pitaglfnxGFdFRpMR121BrBtWD5eIxE/B3D5MaE/R6+APicQrnlERCgIlFiFAPwnHp84wPsPDdHUS09TxDlsE0Z0vFLVwxuh62ZWYEY4U/8AinKf9tg/78cKnpxRlP8AtsH/AH44YP8A7S5X/tsH/fjhq4/wjyv/AGyD/vxw3xBnFLMnGeTwrR0+dxSBqmNmJrouQukBuoJuccH/ACa01flcHym5VLGcuzGaSeV6dOa8zwjlIDJtfRjh2vdhS55R1TLYsIJllKi3joJtjL1j4kEdM8889ZTcsxpqIWGBVYMwvp3xWzcLZzk8WXoq1eaTF5XmUOllWFgIyQxto62tjOuIaTLa6pqlgimy+FI40jdSI4lMYuSE1XA3K3GM1hyDJK2tmni+kaGKXRFLEiFTGCAEhUFVF9gzHFVTcXcTQ0tVLFpA5XtHIU8pSPHHEdRLkklXmQkhq2IKWuR3C3VgfLGbTZzJTVBgaMPOoIhjueXuPu4zNNQ0RdnZkU7KDpLAdBbAP8lENx0Djx9GxTDg7IAzqhFCBpGqwsx874glz2hCygky/wDQfsY+bY4SH5VONg2lia8PfT+nGhxl+U51lVfWZZHW01LUxzzUrKtp0jYExnUCO9j5LuIOHs4y2h+R+my6tq6ZooKtYqX2DnfXdEBxwbwvxMuY8UcPDNMuFNNGaYU8U/tJLaG0SEDbGS1XHGZNWcDS8QQGhqXjoKekhqXS8i2fRKyLZRtjgyCly85NkNLl9Q2YTiSJKYQSpGq/A9trg4yyu4vyGnzGmhmpZKgiWOZSUZdDbNaxxFR8RyLSKIoRm5VETZQgmso9QBirpeKeIgnEFTA/MUpRLIwimBj78hXpdLCx9ccTHivOcimrA2WrLPWJGYYweaZQLiQLrI3OxOMx4qyagp6yigpzlMf0ZEYi3tYorlXfV0Y33thc44F4Eq3qpgponplimEK6OyaYiVEW+liNtW+FbjTPyrEFo4yDbzhXEyRcOpPTKrRzFb8sKUGhupGNWflTSqtpqn2tiL7flvgS04cVz2MSzcsaCO6SbdL+GLDbBPBmR+lIw/KRsH6apP8AO/8AQfsgxsdsZlXfKfxlPEaTQ1altdZBG20SDdWcEYzftEB15eBrG5zCmAG/6+M2jop/4flMhIChY8zpXYkkDYB8Z0tUQTRJdHsWrqcDbf8ATxScFca1OY57U09NTzZXNDG/aIXu7OjWNn22GJuIqqA0FRSzr2yqlFq2n2VzsSNe2M3oc7yyeTssSrNbW1XBZSQRv38T0Od0zVAiPOrXkQpLHKD3+t0Y2wzcXZtPyuiSAMb2FwviMV7cR5hmEkWiA86BJVIu55oJ2N8SpQ1vPlEpesZwLFQFK7DC0HyZ8AVPOdpKxcxLalQaRHUECxADH/WJw391+b983ami3/8AKGMyQ8NVE2YPKKp0LIUAG6t4j8MVzZ4Y5Zw8PaqhFXT4IpIxkcmV1lW1ZVLXo/diEUZhIUi12+LEFzYL1xw/lPDWW0VU9UssMLIdFLLIlzIx2ZQQccMZrxPldHS1kxnnnIjR6aWO9oyTuygD3Q9wPn2OFPyrcdHSP8Zn+wuI6jMKCEkKJamKMsLAgO4BOMvyuLO0gqZHNJOEGqRXDDmBMRz18cZYoDFMxZSFPdjLDfEGa5tyJZDGoo6iYlSEJMUeoC5xTZZQ5JPHKXapepDKzK9uToAOw8dWIarOMthk7qSThWK2UgG/Q4Rat2KxkiOUguoO+3S/jiiejzgCQdoFYh02a4Q3vv0xONcbU0qx8+d1lYroa0trLvfFVSURE8KoTpddN+8hS6k38cGD5Mvk55+Ylwq5mgR1jQRaZ97aQCb38cQPxhUkSd16WDf/AFLYnSbhaNYoxFC0YNowCltQ6jpiQZ9GrQoENZOQ4Ug7qd74y0VbUUmcNFJKL8kIrAki43t42xGbkdccTUGeV8VFnmY08KCMrHDPKiLqUE2Cm2Kmfhz5OnnnklkelDO8jFmYnXuSfqD7DscX+VPjzY/42k/YBiip87ymSuiLUyVMbTDSHugO40nrjg+opuJjQ0Wl5wOynkIun2ynw3Xug4y6nzZJK+n1wCN7goH38NjjJ6Svr2zWn1xvQTJFqjElpjbSbHHDdTkeSRZbShKpKmoaoKwrHdGA0DUOuKODOKGWsh1wK5LqyhwRY+BxQJX1fNgJEnOWnGkbOSNJ9LYjmhzHKVpisgmWUylyQQhO2kD+ljOnkdpoUSgWSoWCVbanbnb9STbFZU0jGucyMmhY7tcCLlAqLADzwaX5N+Dql5S71bZi+6qNKpPpHetqN/UnBHFYIPxUEX/VjMUXh6plzEyCpqYleMooFmudiPwxXNn+l6m8HbpIxHbwANsZLI9RNPmqRVkRjMVPsC4FreFzfEVn9mvj4Y434sP0vlS0vZJiFjD1oha8Xda6Yr8kpOBsqr4WWqpY+TMVYSJrUOTZwd/qH7DscX+VLj7/APMz4p3zvJFqCggNdT80v8ITmDVf0tjI58nzpaaSlMsdVAIeWF1FeaQSLDpbFP21ufoCdmqfiFxqMR0/13xTJmcbVDxKohfeUArc2HjjLTBlBpZad31VAkMIA2GjTewxRrxHkxrDGIBPeTmgFLaW63xAtTJzCqkwyiO4v39rAeuIzJV2ilDiol7+myEG2wPicVfICPCohEs7I2u7N7ZvugbYq6GkKVMiszxxyKQCBoaEaR3icUlB8m/ycxfSbTytR1rvEZBJyddQSoCL8F8K3E9KdL70EfVSPF8ZkIckWRXKwVMRYWB0qCd8VSZzCeUeSa5mLaBYAqd72xzuKKNtfs2SHUQbWscRkH2i9PMY4hgzycUOY18UPKiIEE0iJe29gpAxM+RfJuZpHeQ0oLs5LMTZ7kk42HuvT3JxscFvlP49P+W6n+1iLJuJMkr56cyxU1UkjIrAFgPAXxSz5DxLSHL5Vkr5YDE5YaYxHKZCDt1IxDk2aPUzU5kXs8qaVYA94euEyqqqi8BkMsCooU2IKurE9PTFPnOX5VTR0UkUkNTPMzOwOpZFVQBt4WwcpzmiqyhbRr2RrHdSMCnrasmm5hqkmjU6vgLkHUdsSz1VVStCEWF5CsgUnVqIuLk4zW71FRVo1G5nEESsAwImbdrAYm+j5+0zc+QEaWe7WUwKyqLk9L4FH8mXyXS86ZmqMsqXOproAKlgAg8Bi3EWWH9KhX9jtiWnp8lqjXVJearjDozdwgufC2KkZ8WNW+jt6oI77WtiFcwVHpWd+UhEgYgKN/DAZbgDpheMNObDi/KKEueV2aq1cwGLbVt4HEuRxcC5YZ4akUick1ED3jk0h+8oO9sbD3Xp9UfWv8pXHhv/AOvKz+2cD+6TIiY3dVrYmZY0LsVU3NlXc4krsoeMQ1ICTRuS8TqoANtyw9cEV1YeVK/97qxbRoXILxEAkDwv44EWY18pjmYLl8oDQxtIVZmSxOnoMc5srcJUWUTJqmieO5upsNQwIOJMplZJGVGkJEaGRv4phsF3wYppQUkuYHQ6VJ0nUnxW6DFqmp1QAe3crISt2BboB1tirWljaQxcn2xQDUW2mfrfb8sCgoqmE1PMJ0Peyg9+nQ2AXFJl/APycUyGc1EeVyipSQysYmadiqkSbJt4DCVmc5TJFNBZaUqdU0anaQ+uM8ely5I5UflzKzotXEQFD36a8cR/SWuOld6UVauxR42soG5sGJxm1ZmdLOmS1kqctEZkjeyjVv0xUIi3pZlFvGNwP3YeXPi/swDTw/G4Q7ejY5eQfJknlSqNj6PjYe69Pdl/lG46IYb55W/8U4myXiHKq2KlSoeGa6xHVZyylbbfjjMRlNVDPlhhWcomt1cbqwfa/wCGJ8mrK2aOmWfXQzxkEN3VdbFtvLFbldTmkdNl4qnqaIwkWYlFDq2ru4zDMIcupqnLRTGOWWVSQ4Lawqnr5acVlJnFJLBTc+QCQLGAxJ7pJ2XyGKKllqu0mW9WrRIIkDfGQSbswtjLXrJqSPLZiYnc65ZwNXe8o1FvzxmT1NTbKKCHL4zIqScvmyFw1+srP6+GM0qqIyVFeWZgCOSRGoXQCBaIIBbHEudcHZjUz51Oalc7SFJY7CQRindit7HrhcrqcgpFaR1p6J4A8p1Owie12Pi2F7ndHUYgkr6t1njtEkjmzKDdHG37emBFWxFZTHanv3W0/ePgMPyISJXF416Ow6jHAGf2q8++UZcorf4sUsiI/s0+F7uwO+KWgi4EpaDMoq+jgXRDVAFTMgD2ewuMbD5z9h3X8RjXx9xs3nnlf/xmxK+bUJjhaRkkEmhbAkICT1IGKqoy2PXQyxIkyks5S3Qj7pJ8cVE75lyaZpbUMynSVGnWPiOojbbExqc3kio5JwtAQ2hkUprlSzHWRttio1ZW70UkCWmQa2RtRBUn4CcSx8QUMkcDTFY5zoVlUnVEy9XIG18OHk9g0mmIhipHs++g1H0xIhm1xoFeeRkIJLN3rb7WFsTrTVDmr9kZZPZBALNrbct1OIKGjhiFWZdUSNdn5hBaMd3by8sUORcP8LRVdFHTVKUc5qwY1Dc53OkuR1a2IswzujrIEYxTvVvGWIBKlwcNpX2Z6jyxlcOd5k1SpMZMigaA1n1DEctVRM4I9j4jwDb4jMEOkHToW2x6WwlRnVO3PRb0yABg3gx8gcBcg+TIAggUwFx6a8bD3Xp7rvp+sMcNycZcVtJT5oXbOK4sVmgC3573sCmMhoK+KemgzESAMg5ksBHfUqfuYyyvpRDUw1egkfBLCG238UxkuVNXGnhrlE1O0b8ySFiV/o2TrvigyP6RWgiqwaqnWGTnvCw0K4bay9cUWcR0kNatWRC0jpyWhXeSwN+7/RxlOW16T0612tUcAyvDp3G/RcSNVSJlwZYngY1HaChbSGBOgqBiuaaYTEtT86QRd62kht9hbzxFCKnNO0uxkkmiELDuKNZ3Fz6YgpsoYU7HQ15L2tu0QJ6euOIs+4Nm15jXVE5z5UUxM2sRrTm6jl72xBkUfD2WwRyJFRiogjWRmdwq6SAWe5J9TjZf6sQx5jVq8i2M0lwbbXvjlPQ3jhYPTNfmKGtuOl8A0ND6wR/2Rj5I5tD8X5zmVJXjaFaa+gweBPcbfVfGWRDgOPJ6uSbLBtSvMlpGis9ix23xsPmGPT35+b2kX66/vxq4q4mbzzWuP+/fDni/ImWjkqeXOZGij0hmVUN7ayBiqnyh3fLpolirkGtyhA1K4C91ibnE8sed8qnaQfR8obSQNG4bUdRG22JVfiGVaCaoHYkjvEUHLLzKQTqI8sS6uHJGoJqcGgZQZShMhEzEsNBbbcDEzZqOTA8rClqRpQqDZoit+8R54qkbNGhgSRRl8qSln0lEJW7LtufTFbHSB5Yk7PJWTGFwne1bagWv6jGimnJr5ChkkIhJUIpLNuBa+OTksiB3kJklt1fqgsL4y/IaDhKKYrE1Lk/JqCgA9uwW4uOp23OIc9rqLMqVDyKiqqGjLHcjSow+gHR5eOIUzNy8ZAiqLvazG198B2yz2ezRv1t6HCNS05SJ9BjXTZfDEc9dSMzSKeRp+C/3z6jAGUfJjpNxyF9PB8bD6x+oPrH6ntYv11/fgvxDnzauuY1h/OZsZtlnEeXz5YiSVQ5gjV1DLuhJvcgdBjiBsoMFbFEsE9Wj6kCE8yNWIF1JtsxxW0UWcdnsRJRust1BtG3cJFz/AEsZvRHNqagSN+0wxiXmBdlje4sWI8cZ1XtktPmMcUfZKV0g5ar3kdySSVJub4zCDMWaiUSSimnJBC20ImpzuR0AxmJnqBTVJSM0zmoFwuuO67dDis0s8zs8DVMvKDEkIQQCBc4ooFrMwQyc+VpUYGwQDU3QW9MLHkbBS38odzfrGDjP+IuEY4tOY1srZ5cBea78oQAbad9OKHhmDLMqoaHslNR1s8UUG9kUoG+9c47nXywhr6hS9v4T3hY9MZfkqUL1mVRVgmpdChyBoIANxqBwr5dRMqgAxKRbGXcbU1fPUpSlqaVIgZ0mY2YatuVLHjP8i4n4SpFz2lOW01SyJRx0pBCBXsokZmbGw+wj5/aR/rjGvOM1b9KtqT+crYq5eLKBqWCOWSGGpl0SOY1KiFgdwG88ZnJwpTzTwQLCmaBdaSs7F2gNhYqNsVVVScRtAsZVMucya2IIUMHJWwNz3cVck9bLBBFIo5UTCSQx7uSwtZW8sVkc2RvUU8UQekkEYjkMlwkzXJuq+JxOc0qDCIi30fVqea5QWkiKXuA24viqL1rQ6NCUbc3WDcrrX4bdDiphyykkklvDJV1GgWUaTsT6nFItJKVq2MpLkxGXUFGp9wg6YEGRPHGrMGkmIIBI7yjxOKHIMu4WgrJ0RqPKBSynWNJlBViFPj0xTcT8rNKNUEE2Yvo72rpFY9MPo+Fenng0ubzSyLqjjk1FF6nbzOBU02VNfQpRiDcdGUYV6CkaOM6DEtunTH8B4iG4tUw/2Di/FWQf6Uf3NjYfUPzjA+cfMfmH1bEHyOA1fWtc71Ex/NzjNYs9g+i6swVDwzrrvaycsl/A9QMZ3RZTSUNTmQlpZKozLGrEgSImnUbqPDGYRUufdlqDGnYv4QNVtcTuEK9D+ljNZDmIoq0QKghklLMRc6iqdFPmcZs9VQwZhXdo5NOTCQdlSRySN1XxxXvV1PYp+W60NTK5LWukSamHQ7nE80s7R1DIkdOXkS7WkBcAA2OJYYxUMdSTTzaQbd0o1rYhhhrq5VYSytLGxvsRdjsMRLw63KYEaJTsb7lN8Z5xDwtQ04y6vrZJM6MwQrI5MSxot9/ub2xFw1l9blkGVpQQU/EE/IgRQiLE6XXSB0Bx3MS/3RTpJKRCCx06GP8AJ38MVeX5fkjwTctmTQxI8NA2N8NJlFAzdWjufzOAYOIwSRaaDofRsW4t4f77H+GHqfRsbDA+c4Pv9Mch8kY/kMap5T5u5/M4rKvPJ2ppUR6fLqyYmRDINIj0tsCN7Nis+jYJ56hHWOoCALEU3kUm9yx6acVFTlPF8sUqKkWXgygoWLIJVbYgixuMV89HxDUU0sSKjUkcgkiMl9TkgizLirpqvK5aiaNzNTPp5cZjAEchHQlsTyV1fyZUQ/RtUpLIXGmRNJsARvvscTOuYSpUrGkdKNakA6wZBYXPTDjL8vaOoefXPOeUrFzGSQdkUbA4rGopXjyuqE76tWsOosdXhJYDFblWRiB4oY9XO6TxEAuvU6WNhih4Zy/heOur4SaHKjRuULMpkDq7aSRuMUObQSZjJzqWGpr45IwYdZ3jPkwxkETFHz1Ua17PTuvX+s4yWHM1rZM8pCOW1kaOZDZlKg30nFLmVNCUzyhjELmQlxMBYj9TGXvAkMGdZfJy0AJExt+1RgLDxLaaGQc6AezkD9A3lj/C7hz/AEw/ubG32TTSVbeUEp/JDhWa+nrc4lkzVIoal4DJHKrOhN9Om5GxXY4koo6OM1rTrIzGxBWxQbH4m88TCjzho5mRRDHzFBazh302NiMVNZHmvLr5aeKMQFwmo6y7ELcBl6WxLS10EM1Y84EAdS+oFQ7G4sWbyw9VPmAiqTEYcuqagkXOoQrfTsR1xXwvogmCQBlMgCrqLPfTva/hjOG1y1OYTyJM4KXlc9DiGlyvMcxaoeWWrp5dSyC4QAMRa+IoeF8xWBiUFJVEG6km6HxTbGc8QcLcIURyKtrOZn8k8kbRO94XaCPUb/cIvhKTI8tpkiVFjzDQqgW0rGjALimf6cEnJLtCoQPpvtG3S+O0U0B0KTHR23Iv93BXK6wjwgJ/Zh5VrmNvuDYW88Ryw8Sl1vaSm/c+IU4y4b0oQRXH7zeTY2Huz7n+92Yf6LP/AMM4uE/Vx2nMag9qMDRQMysNO5YhSO8D4HDwQUbvXNOQ5A2QAahf7oHlhZ8j4kqDVNGYok9mNPtNJ1AG4v18sc7Lc9nXM3pmSekjKLy++CWN7OD0x2aty1zWvUNNSsSW0XXRIygdwDAlnzYdpeL+9lQp0ae8rgAr3geuHanqagVBVUkgDR7WfVqt67YhpoqULUawyqxvKX0k320+GMmOVVgp9BnFO3aNOpmHdPXFLW5NXRZbHaGeCeOFtHJQF1K7g2sAcZdw7T8OJVzR/wACyukpW0uLO9OyM+knw2wtVQZPIuyzV0soHoyEjEC1FU7ZksTOoUoQNhptffEBTQ9VGhMAQX8V88MctrBYbwkY5cNWgXfWp8uowOz8TX29rTfufH+GfDf+nf8AJsbD3IwPqDA+c/NeirfWnl/sHFiPwOMrreIqKHM5kjpSk5dpH5a3WMlbtceOMoWio5KKSPWZ2V1SQudOgEEgk+OKZ6fNjKwDrSgwgtYl+YBt57Yyip+mmr5okMVPEYBJIY7u0oU23F9sZVR1OVfR80brJQJJLy5DIBIZHBHU2NgMUr11qkqI+z1BGptHfEZK7gjxwKpK2uOq9JHELXAHtWI3BGJaKGkl1uUqLyAEWAIYg2OKGh4bNVFThZaikkMzXJ16VJ8T64hHB2cNTpoQ0VYyBT0uG6EYzTiTLvk3oajKK6sR87lepRo5X9jLPAh1+OkriOjpuHaSOLlrDPPGqdNCxqFC41rnZ59Mt0IKyfEfZeGI5oYmITUtN4lQfiHniGi0VE6FooWSR1UXJVSCQAcZZntfUVVBA8UQjjQq6BTqBJ8MRyU/EZeMNaWn6j0bECcbcLlYVB7eNwPRsbD5hgfOfnGBg4P1j82mkqz5QSn8kOCWG3gfHFFw9n/aqyjeeM0s0QRCoOpwLHvYoq/JYKOGiaOUVon1FlIKiMpb8ziGhoM/hlh1NVUhijIYDSx2B3xR5RSZ7BUUjSSVPZzGVK9zlOSb3xQ542QLS0bxGkojDNrK99y5a404iyvMWnmhLKaeVLBh1YeuCcpzubtzRaeQph1Lpl7/AFIO+2IaSkyzRXrOJUaTRzC5hOq2m29r2vjKWyavFLoMvZ354XUxF0+9h5uDnooI9PNpJokLdxRzCQCT4DGTZRncNXNKTDHTUiDcAv2d7t1OEranIZ02SeSqmUejlSMUjz1evNRC7bGM26aQPPEc+kPVJHaEKAR63vhvo6sBA/ij44tTToqG4kub2HUYCUfEbNYe3g2J/otgNxnwuwIscwH/AFY2HzD3B91py3Mm8qSc/lG2LhP1RiKLP5JZY5WRMvrP4uNpCGMdlJCAkC/jhKzJ8ncRzhlq6kM0kUiCxRLAM4AJ67YJy7iD2UjXpogCiFlBEoY6iNht54ENRWsROAyxpqiieTe5NjoBwktflros29EATLG8ZZhK/TWBfAjzOqYxSuPo2uS0cbSHU8JUbKDtc9cJKldVFiHgjhVRYEHWSDh6Gky+TWStRGZQCTsdRBxRUnD08sdKsTzUrGVrm7nSNzv64ReAc05OwGXVGgofU2tbGYcUVPyU0NdktbWU5r5O3JIkpHKmqekviFIGI6Or4bpI4eVHBDPGkfTQsbBAv9QGKd6HiFGeLmPfSrWv/FrhJiZCFJVEAuRfY4koaaWpREZ4bOA4upII6jE2dyVdTNDDG10S0S6F7oOJeGjFGmUUlWJwXJnMgK6Taw0MMZIeIPk1zhKExVcstPUHTNIUDyISbKxxsPrD3pxoyLPG/Ry+rP5QthtMW4/ixiTh3MK+UU0cxqaGelAYldPMt3tvEYfMcjyzKzSRoYKyaq5gZiX5iBLEHytg5flmf0vJR+204h1EkFLnqMSZQtVGsCSF5Y5rksLaFZbbfjh8/mysGmjhNJStD3SW1a5Gkub/AI4fKKqeYKra6aWPckWDD0wWy/NqjnuvLEC6FPca5PxD08MU1HRZPonLcyBmIJclSH6d7a34Yyl8kq0pkRX7ORPpQi5KDqSBfElZweMvpkAM9I0UbNso1sdzihynOjWzRSNFH2SwHVuQrA/vwKzPMmmAsstPNKB6SS3xHqrW7Y6mUkMoW4GwGDVEnnBTYeBPTfDHLay420f88Ds06qhNpLn+sYDz0KmBD3H+IX8cap/ksPmtJ/wzjYe7/8QAIREAAQMEAwEBAQAAAAAAAAAAEQECEgADBAUGBxAgMBT/2gAIAQIBAQIAJkT6ZSkfTKUqMpSPplIn8CZE/qf3M5z/ABM5zmaJlLwynOcvDKUj8GRomUj8mUpHyR+CZSP4mUj+ZlOc/wATNXzkfBR8MpzlyPkbe1k7Tt9nJzdvM8PYSlImRPgo9o1vMX+SyqavF1/XjpHwylOUiZdnrySslbNuyzXV16+R+JE0TLsxeSOyKtVYZrq69X0qs5E/HZi8kdkqxcbH1ScN2OHsJnw/BlPspeRW3YaYeFsLNad2iWR8M5yNE9h3+Q3LFXbOA5W6atGsifJSJ8PYlzkV2xX8+A5ztOuk+SZSJ87CfyK7Yq3jYKx01aZZzkfp+cudyVNvoXrm39Fewrmqv6ehEGcpzkbtKy/iZtjLXYrxxmrbo0075H4AFXaWlTPXOrZLxlNanH11CymTU5yoXKd5sF2FZ9ri2NrbGjt6rwylIynIvRyDPXNyL+XiU6tK/WfiZLTmx2VZyvyMZzq1KatTImj8LTkjsqzldk2HrWvrX/BJMpFacg2dZbEqxjotnH16/Q9DmjaLsXMzuIXle+tespEn4WlVyy29ZyWc7il6S1guPwZSJKultlzEx8/jGSxzG4dEykf/xABJEQABAwIBBgoGBAoLAAAAAAABAAIRAwQhBRASICJyIyQwMUFRYXGxshMUMkLBwjREdKFDUlN1gYKR0dLwJTNAUGJjZHODkrX/2gAIAQIBAz8A1CjnGc6o1QhmOqP7vCGY8kEM5R5EIZijnGodY655U/2QZzyoQzHq5IIZiijqjUCGYqlke3p1qtN72vqBkMiZIJ6e5ZL6ba5/Y3+JZI6aNyP1G/xLIz3NEVxJjFg/eslflXjvaskH6yf+jlbXTNOjWa8QJg4iesdC7MxzDMdXtQQOT7MR9ab5SqFAXrabA0A0Y7JBVQaBIgOY57e0Nn9yipT3gnOp0Xh7dvRgbxhCo+6a5+NJhdh0wuDvt5nx1RmC7EUdTiFl9qb5ShpX29Q8pXB2H2Sp8yGgX9IqMH7ZPwVX1e2PpR7kDR5uERNxfzjNFy2b/vp/HkDrcSsftTfKUNK/36Hlctix+yP+ZRRef81ng5VDa0XCtA2YECBwiPrd5OJNF3wX0/8A4/m1QhyHFcnj/UjwK2r/AP3aHkKlll2Wb/mXF6mP4Rng5POThVFYgSBowI9tF19cAn8CcT+hMtTdaTCdLQ5uyU25a5zWkQYxRRzjWCC0qGTRH1keCLnXeLW6dSkWlxAkNaQYnqlMqChw3sWTnbImQJ64VGm68ohk+jMy8zJaxx6IVR3oqEN9G97QQO0ptKvlGo1o0qTMJJxBTJq6LCPZ6ZXBVN/MdUo6gRZb2e3ANR84xOwVTfTtXUwNEuqkQIESF7P5sevSXuUuFc2C72emGOXDW2+zxQNXLHZSn7lJqfoXBVN74cq1tGxJ/KP8hQqstXiYLqpEiPeC5vzW5Ctf5QDqjxtu9kx7jijp2x7WIi5yqOuj8q26ncFwL9/kTqaNOwME7dTACfcRqNtXFrmy6sYcIPtL/wAsqnWyjfB5dHpXYAke67qRAtjuIG8yn9nJ+4LhH7qik/fzFHkHBxGgMCsREEdeKp3lTJ1N7yxzn1A2BPuqjcVdB1Z4NJhq4AQRUccPuQotq6NMS3JzSCSTg7oKOjlsgAOYaRaQACC444hF+T6xfUOmKQLSXGZ0U99xcguJHq7iMe1OfUcDHs9QCmi7eRzBNQQzlFHSfj0lOGEhCs+3qOP9S5xbHW4QgG1anSWaP6Ati5w5rCm37wuDy9vUPMqTsn19NrdIURozzzonmTRdVRAANu7xWnXduHxC4F0dD4R1O3Mc4W0/vKg+0sJ0ulcA9ANu4A+h0/EKaWX9+j5lQNlVFTQn0Q0NKJmDzIestiMaJ5kfWHbh8QgKVTfOYo6hRz4v7yj1rA49K4u/uPgoZdxH0Wn4hVKlPLwZTLialKABPM5FlpVFalBNPZ0hjOKe25pEsIHoyOaFUZculhGwecLg6m+cwQzFBDMc0krvWB71wDp6j4J7fWNGBFo04CDOCY28vDWe2BXZoh5kAYzEoGnskRpOiOrSK4zY9tN3gE/1oiT7J51wb988iM3PijPtErD2jzoi3dj1+CPGoP1Vo8FRpX166pog+sNInqEyg9ry0yDUeR3FxQNfJp/wO8oRF6f1gtipP4/KYnEoRPauLu7neVHjeP1Zg8FSoXt4ahAPrDTjzwJlNf6ZzTINWoQewuK28l7nyotvnR+M9bDt7XGrCEHvUW7j2O8pVRxu4a4zQYBAnqXoby7c4RNwHAnDASnVnV3scxwdVe4EPacCUCcnw9uwIdtDDZhVGXj3FsDSd0rYd38nKgwhE9qLbaoQYOi6I3U8vyhiY0KPiFToX10X1I44XHrgaSZSYHkmNvoJ6UHG2APM5f0g7+fdWw7v5QSUObtXFX9cO8qJqX/PiKHiFSt7yuajwD645x64AI+KZTYHEmCH8wJ95BxtQDzOHwUX75/nZWw7v1hrSSVhC4rU7neVE1L7nxNDxCo21zVNR4bxtzj1gQR8UykwPJMFr4wJ53IOdajSmHBTlF38+6tl29rf/8QAIREAAgAHAQEBAQEAAAAAAAAAERIAAQIDBAUGBwgQIDD/2gAIAQMBAQIAgKAAqSoVR/aokqFH+YC/5AKAAqr/AAFUAKqrKSgCAAFVZUhQP4Cqqqqj+AqqAAAPwBVAgAAKqqqiAFACqqqqqqgBVVVVVlSqgSkqqJSVVVQICqoAVVURKSqolLg+Fq+bq/nK589z8Cr8H3nNABVUBVVV+dI5jPpzaoq67adL9M0AKqqAFVR87S4iMSLuTm1dDL6XpVVVQAFVV+epcVTiSuS2Fzo4+lJAKqqqqqgfP0uJlhxcjZ5fSU+481vNAqqoAVQo8BlxldObLN3XM36O3teiQFUBVVVUeB2OOt5cWMjeU2o7yXoUlRFAAVQPAKeMsZUUZu3ot1d3T6BAVVUKiIo8Ep4yxkxf2WfK1Hdy7yFAVQAsqbXKW+Y86tc30lvX4GB1mHscXs9d3VKqohVCriRKnDydRl665rp9vldFe9Hr7ilVVVVVWUseUDUS08tTHd3OquenU9tSqqAAqrYichppaWety+92nUbD0bL7SSqohVVVtSnKUtPRrMGjEs1W5eg2uxkFUKAqrT+HSRqKqrWJEpdxd7CSgfoVRP8AdDLTV3cbXW6pdDHW0BVCqqgTicSjnp6nLv3sXc1yzdt1klH8BVppnE4lHN06GWwwvScS1bxZ9VJVH4qqJRVBjmKtJXtsH0fDlRjS6iFUCAqimVUpRKOYjT3N9re9192m/V0slUCP/8QATBEAAQIEAgMMBQYKCwEAAAAAAQIDAAQFERIhBhMxECAiJDBBUWFxc7GyIzJys8EUFSVCQ8IzNERTYmR0daGiFkBQY4GDhIWSo8PS/9oACAEDAQM/AN4Nw/2keTG/6t0b48qd51bwb07wckd0csP6mNw8mN07nXAjq5Ewd0R1Qd+YO6I6omdKZ2YlWJltlTbJdJcBsQFBNsu2K99WpSJ7S4PuRpKNk5IH/Mc/+I0pTsVJq7HVfFMaWDYxLnsdjTBP5A2ex5HxMVOjvameknWFXUE40kBeDIlJ2KHWN07w7+1cqZ6JBXvERNTYpTjzylqWmZKujJSQIZOMBdy2tDawBsUu1vGPV7RDTU1Ny6mF+gDhKgRnq0Y4XLM0l1DIKZuYQ0QrIpC+yOM0A/oTHijfjdO6I+mqp+wq86Y9DRs/s5zzoga2q5j8fZ8G4IfZQALKSpV/ZKR8Yk/nWpJMqrFgeClhzNXF7nK2WUJRTNG8NwlM8za5uefngl3R49Uz9zfDkLVerH9RV50wrU0fupvzoj0lVzH4+14NwDNy9x9m55kRLCsziFSYKilwFesUCby9zl2ZQlNFoZQLJTOt2F8VvW54N9Hv9T/577q5D6Tq5/UT5xB1VH7mb94iBjqn7e34NwBNsZfZOeZES40jdl1SaFEoUceJQUbs9ttmUJTo/TcCQlKZtBABuBbFzmH6uKPqnkI1RfvjvniwdA6odpbjSHHULK04hhv8d+d9hnq2Sdkl94QEt0yyVrLbMwFhKSopLjiSkG17XtDrSpw6jJ2qNt8JViFEItcC/RDjrUhMrdwF0JSEoAFg44hJzVivtiWaTN1BK3TMtMuKStRB2IIzyh2Yp2jkuqYIam3c02TwSD9U2vzxMJRKFx9K812sjCRs6zHG5TufvHlULn6qosBwpYbtlci7gh9Ds+HwoOBDGILNyMjAxOfv5ryphlml0YrZQs4WgMV8rutiAabUc/yd3ymLyOhiuiZt/OI9FK5/WV4COOS3c/ePKufLKwUJudSz7wXhxlyfQ4AFBEuDYhWeEnaIzV+/m/AQ3L0ekKU02o6pu2NOK3pGx8YvIzot9i55TFqVoef163/YYtLy3tnwjjsv3P3jB3nVyCzMVrBhvq2NuQ/CRqlzyCtCilEuCUG6TwOYwCf9/R4QZOhUtaUoxahu2JIV9dsHb2wlctNbD6NfhA+Y9FldE+POqOKsd58DF51juR5jyB3W1soWH1ElAVa3SLwS0tatYlQ2Iskk9ljExRhXXZdOtCGWFLKrAAY8ueJqVYDqWWiH3UsZk3BaQBfLth196XSt/JyvuN2QAkYm0nhC9yNkJCtFUkFSX0zN0qUVJIbGWRyyhDVZoqG5RGrcdCXQGwR6425RLJptNWGUpPzihCuDhNsNzshliVaKAoekAzUojYeYmOPM9yPMeTtLsd2nwgCFSwngi3GkIQu/Q2cQtCtaw1YYUu4+u6rD4Rd+mdek0z5VRdWguzNqd8sTTNWoQaW4EKfs4EXtbEn1rQXKTLqKlkoqSASrM5JjU05hQAzmEj+VUH5az3I8x5IRZhru0+Edcdccca9pHmgB2i5etpFNHP2VwMeglgM5ecP8gidRP0f5OHinXkO6sG1rp9a0KVSHcRWCioJ9fqB2X5oQaU1cX4wjZ7Ko43L9wPE70bh3noW/YHhGcc8AzrXtt+cQNfQhlnXZvb7K4YZOgqnHkISJWaJUohIF0C1yYDk1SzKzGMJeOs1a+Y4dtuaGHqVPIS8kqE4CBjuVDpHVEu9SUYXEOenRkFA8x6I42x3I8TvBv/RoH6IjZAgKm2hcjht5jrWIZUKQVJKsVddbIWpRBTw75bM4UKFRVSkmtS3JJ4rLKbEqKRbERz9ELDbOMKCtU3iB2g4Re8D5q0iFvVmGPMoQwKMhYaRfWI5o40xb8yPE787oiyB2bmcD5W1l9o17xMcChkED6eeJ6wMcTbuj1CSww67enug4L2xFKbXhaJaUS4kpWJdoLHQoIFxHENK09D7P8HTAOj6DYH8EcxHGmM/sR4ncG9O84P8AhuCAZ1rIfhGfepjgUAggfTMwT2DHE0/o/Q0sSy3Po5YNjYAqSm14cZkKeh1JDiJRlKxe9lBABzEAsaXI/vUfweMId0faxC41bJ8ItNM2/NfE73q34gQPlzPese9RDTbVAUp5KQKpMqVcgADhjOC/QqO2y3rVJpqkKAPqqUE2BhqSkae1NIeacblmkLCmXMlBIFr2hV9JvROAPnEi6FZgOYujoiUmKEy2h5KlaloW602vHGWu7+J5IXG5mIEJXUpVKkhSS8yCCLixcTCQzomoAAmZniqw2gJXa8Tczo/RksShdJo6U9FlKCMs4fmJdLbYTju1cKUEjJJ5yRC201cqQAC2bEEG9knbaB/RpodvvI4y13fxPJZjdN4Iqcp3zPvUxeW0TOLY5USbewuJyYoVNQxKKdvRW0Dm4RwZZw/MsNtoSnGFtXClBOxHSSIWhuslSQAUKsQQSclQE6OMDqO3vIvMNd38TyWYi8GDllFqpKd8z7xMD5LoscewVIn/AIKidmqLJpl5NTt6M0hJBCeGSjLOJmZYaabSMaVtYgpQGxFjmYWhqsKKAAptViOfIxh0Wat1e8ga9qwAGD477//Z"/><image id="image4_3202_39054" width="480" height="480" xlink:href="data:image/jpeg;base64,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"/><image id="image5_3202_39054" width="383" height="479" xlink:href="data:image/jpeg;base64,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"/></defs></svg>',
				'demoUrl' 		=> 'https://smashballoon.com/youtube-feed/demo/?utm_campaign=youtube-free&utm_source=feed-type&utm_medium=youtube-feed&utm_content=view-demo',
				'buyUrl' 		=> $this->get_extension_popup_dynamic_buy_url( $license_key, 'https://smashballoon.com/instagram-feed/demo/?utm_campaign=instagram-free&utm_source=customizer&utm_medium=header' ),
				'bullets'       => [
					'heading' => __( 'And get much more!', 'instagram-feed' ),
					'content' => [
						__( 'Covert videos to WP Posts', 'instagram-feed' ),
						__( 'Show subscribers', 'instagram-feed' ),
						__( 'Show video details', 'instagram-feed' ),
						__( 'Fast and Effective Support', 'instagram-feed' ),
						__( 'Always up to date', 'instagram-feed' ),
						__( '30 Day Money Back Guarantee', 'instagram-feed' )
					]
				],
			),
			'postStyling' => array(
				'heading' 		=> $this->get_extension_popup_dynamic_heading('Display Captions, Likes, and Comments'),
				'description' 	=> __( 'Upgrade to Pro to display post captions below each post and in the lightbox, which can be crawled by search engines to help boost SEO.', 'instagram-feed' ),
				'img' 			=> '<svg width="396" height="264" viewBox="0 0 396 264" fill="none" xmlns="http://www.w3.org/2000/svg"><g filter="url(#hoverstatefilter0_d_557_20473)"><g clip-path="url(#hoverstateclip0_557_20473)"><rect x="196.486" y="41.9882" width="139.542" height="147.818" rx="1.74428" transform="rotate(3 196.486 41.9882)" fill="#59AB46"/><path opacity="0.4" d="M207.272 102.814C208.625 102.884 212.65 103.095 217.927 103.372C224.523 103.718 224.523 103.718 230.104 104.01C235.686 104.303 236.7 104.356 250.4 105.074C264.099 105.792 261.562 105.659 268.666 106.031C275.769 106.403 276.784 106.457 289.976 107.148C300.529 107.701 312.977 108.353 317.882 108.61" stroke="white" stroke-width="6.91919"/><path opacity="0.4" d="M208.185 85.3947C209.538 85.4656 213.563 85.6766 218.84 85.9531C225.436 86.2988 225.436 86.2988 231.017 86.5913C236.599 86.8838 237.613 86.937 251.313 87.6549C265.012 88.3729 262.475 88.2399 269.579 88.6122C276.682 88.9845 277.697 89.0376 290.889 89.729C301.442 90.2821 313.89 90.9345 318.795 91.1915" stroke="white" stroke-width="6.91919"/><path opacity="0.4" d="M206.359 120.232C207.711 120.303 211.732 120.514 217.003 120.79C223.592 121.136 223.592 121.136 229.168 121.428C234.743 121.72 234.236 121.693 247.922 122.411C261.607 123.128 260.593 123.075 268.196 123.473" stroke="white" stroke-width="6.91919"/><path d="M272.181 159.772C272.033 159.764 271.895 159.698 271.796 159.589C271.698 159.479 271.647 159.335 271.654 159.187L271.742 157.521L269.52 157.405C269.225 157.389 268.949 157.258 268.752 157.038C268.554 156.819 268.452 156.53 268.468 156.236L268.817 149.571C268.832 149.277 268.964 149 269.183 148.803C269.403 148.605 269.691 148.503 269.986 148.519L278.872 148.984C279.167 149 279.443 149.132 279.64 149.351C279.838 149.57 279.94 149.859 279.925 150.153L279.575 156.818C279.56 157.113 279.428 157.389 279.209 157.586C278.99 157.784 278.701 157.886 278.406 157.87L275.018 157.693L272.855 159.646C272.739 159.745 272.597 159.794 272.458 159.786L272.181 159.772Z" fill="white"/><path d="M287.454 158.704L289.234 158.798L289.684 150.206L287.898 150.113L285.596 151.538L285.513 153.134L287.782 151.76L287.818 151.762L287.454 158.704ZM291.529 158.918L297.774 159.245L297.849 157.816L293.955 157.612L293.959 157.541L295.843 155.992C297.397 154.706 297.927 154.053 297.981 153.011L297.982 152.999C298.059 151.535 296.863 150.433 295.047 150.338C293.231 150.243 291.826 151.31 291.741 152.929L291.739 152.977L293.406 153.064L293.414 153.011C293.479 152.238 294.104 151.692 294.986 151.738C295.748 151.778 296.257 152.306 296.226 153.009L296.226 153.021C296.195 153.616 295.94 154.027 294.779 155.005L291.593 157.685L291.529 158.918Z" fill="white"/><g clip-path="url(#hoverstateclip1_557_20473)"><path d="M218.664 146.871C219.194 148.86 218.561 150.822 217.366 152.263C216.576 153.241 215.649 154.071 214.709 154.79C213.84 155.518 211.907 156.946 211.413 156.963C210.983 156.857 210.522 156.34 210.19 156.069C208.33 154.495 206.339 152.595 205.384 150.582C204.596 148.656 204.716 146.331 206.172 144.962C208.045 143.443 210.726 143.899 212.006 145.656C212.398 145.198 212.87 144.847 213.42 144.602C215.64 143.848 217.855 144.851 218.664 146.871Z" fill="white"/></g><path d="M226.488 155.509L228.269 155.602L228.719 147.011L226.933 146.918L224.631 148.343L224.548 149.939L226.816 148.565L226.852 148.567L226.488 155.509ZM230.563 155.723L236.809 156.05L236.884 154.621L232.99 154.417L232.994 154.346L234.878 152.797C236.432 151.511 236.961 150.858 237.016 149.816L237.017 149.804C237.093 148.34 235.897 147.238 234.082 147.143C232.266 147.048 230.861 148.114 230.776 149.734L230.773 149.782L232.44 149.869L232.449 149.816C232.513 149.043 233.139 148.497 234.02 148.543C234.782 148.583 235.292 149.111 235.261 149.814L235.26 149.826C235.229 150.421 234.975 150.832 233.813 151.809L230.628 154.49L230.563 155.723ZM242.229 156.334L243.938 156.424L244.021 154.846L245.152 154.905L245.227 153.47L244.096 153.411L244.388 147.832L241.852 147.699L238.116 153.032L238.037 154.532L242.312 154.756L242.229 156.334ZM239.676 153.227L239.678 153.18L242.606 149.04L242.647 149.043L242.42 153.371L239.676 153.227Z" fill="white"/></g></g><g filter="url(#hoverstatefilter1_d_557_20473)"><g clip-path="url(#hoverstateclip2_557_20473)"><rect x="63" y="55.1204" width="139.542" height="164.834" rx="1.74428" fill="white"/><path d="M74.5776 156.85C75.7308 156.82 79.1616 156.769 83.6591 156.797C89.2809 156.832 89.2809 156.876 94.0379 156.85C98.7948 156.823 99.6597 156.771 111.336 156.823C123.012 156.876 120.85 156.876 126.904 156.85C132.958 156.823 133.823 156.81 145.067 156.85C154.062 156.881 164.671 156.877 168.852 156.872" stroke="#DCDDE1" stroke-width="6.91919"/><path d="M74.5776 171.754C75.7308 171.729 79.1616 171.685 83.6591 171.709C89.2809 171.739 89.2809 171.777 94.0379 171.754C98.7948 171.731 98.3623 171.709 110.038 171.754C121.715 171.799 120.85 171.789 127.336 171.783" stroke="#DCDDE1" stroke-width="6.91919"/><g clip-path="url(#hoverstateclip3_557_20473)"><rect x="72.5933" y="52.5039" width="132.565" height="88.9581" fill="#2C324C"/><circle cx="97.5117" cy="88.7693" r="71.0037" fill="#0068A0"/><circle cx="209.41" cy="160.88" r="73.7586" fill="#FE544F"/></g><path d="M143.009 210.272C142.862 210.272 142.72 210.214 142.616 210.109C142.512 210.005 142.453 209.864 142.453 209.716V208.048H140.229C139.934 208.048 139.651 207.93 139.442 207.722C139.233 207.513 139.116 207.23 139.116 206.935V200.262C139.116 199.967 139.233 199.684 139.442 199.475C139.651 199.266 139.934 199.149 140.229 199.149H149.127C149.422 199.149 149.705 199.266 149.913 199.475C150.122 199.684 150.239 199.967 150.239 200.262V206.935C150.239 207.23 150.122 207.513 149.913 207.722C149.705 207.93 149.422 208.048 149.127 208.048H145.734L143.677 210.111C143.565 210.217 143.426 210.272 143.287 210.272H143.009Z" fill="#434960"/><path d="M158.207 208.407H159.989V199.804H158.201L155.977 201.348V202.946L158.171 201.455H158.207V208.407ZM162.287 208.407H168.541V206.976H164.642V206.904L166.443 205.259C167.927 203.894 168.422 203.214 168.422 202.171V202.159C168.422 200.692 167.17 199.655 165.352 199.655C163.533 199.655 162.186 200.793 162.186 202.415V202.463H163.855L163.861 202.409C163.885 201.634 164.481 201.056 165.364 201.056C166.127 201.056 166.663 201.556 166.669 202.26V202.272C166.669 202.868 166.437 203.291 165.328 204.329L162.287 207.173V208.407Z" fill="#434960"/><g clip-path="url(#hoverstateclip4_557_20473)"><path d="M88.8934 200.19C89.5271 202.148 88.9967 204.141 87.879 205.642C87.1412 206.66 86.2595 207.538 85.358 208.305C84.5286 209.077 82.6726 210.605 82.1806 210.647C81.7458 210.564 81.2578 210.072 80.9126 209.819C78.9728 208.344 76.8846 206.551 75.8258 204.591C74.9381 202.708 74.9365 200.38 76.3181 198.937C78.1096 197.322 80.8104 197.637 82.1806 199.325C82.5486 198.847 83.001 198.472 83.5381 198.199C85.7151 197.33 87.9798 198.215 88.8934 200.19Z" fill="#434960"/></g><path d="M97.1573 208.407H98.9399V199.804H97.1514L94.9276 201.348V202.946L97.1216 201.455H97.1573V208.407ZM101.238 208.407H107.492V206.976H103.593V206.904L105.393 205.259C106.878 203.894 107.373 203.214 107.373 202.171V202.159C107.373 200.692 106.121 199.655 104.302 199.655C102.484 199.655 101.137 200.793 101.137 202.415V202.463H102.806L102.812 202.409C102.836 201.634 103.432 201.056 104.314 201.056C105.077 201.056 105.614 201.556 105.62 202.26V202.272C105.62 202.868 105.387 203.291 104.278 204.329L101.238 207.173V208.407ZM112.92 208.407H114.631V206.827H115.764V205.39H114.631V199.804H112.091L108.639 205.324V206.827H112.92V208.407ZM110.207 205.438V205.39L112.914 201.103H112.956V205.438H110.207Z" fill="#434960"/></g></g><path d="M293.823 189.298L293.89 188.027L292.48 187.953L292.546 186.682L289.727 186.534L289.793 185.263L285.564 185.041L285.63 183.77L282.811 183.623L283.077 178.539L281.667 178.465L281.734 177.194L278.914 177.046L278.848 178.317L277.438 178.243L276.838 189.682L275.429 189.608L275.495 188.337L271.266 188.115L270.999 193.199L272.409 193.273L272.276 195.815L273.686 195.889L273.553 198.431L274.962 198.505L274.829 201.047L276.239 201.121L276.039 204.934L291.547 205.746L291.814 200.663L293.224 200.736L293.423 196.923L292.014 196.85L291.814 200.663L290.404 200.589L290.204 204.402L277.516 203.737L277.649 201.195L276.239 201.121L276.372 198.579L274.962 198.505L275.096 195.963L273.686 195.889L273.819 193.347L272.409 193.273L272.609 189.46L275.429 189.608L275.362 190.879L276.772 190.953L276.572 194.766L277.982 194.84L278.848 178.317L281.667 178.465L281.068 189.904L282.478 189.977L282.744 184.894L285.564 185.041L285.297 190.125L286.707 190.199L286.907 186.386L289.727 186.534L289.46 191.618L290.87 191.692L291.07 187.879L292.48 187.953L292.413 189.224L293.823 189.298L293.423 196.923L294.833 196.997L295.233 189.371L293.823 189.298Z" fill="#141B38"/><path d="M292.014 196.85L293.424 196.923L293.823 189.298L292.413 189.224L292.48 187.953L291.07 187.879L290.87 191.692L289.461 191.618L289.727 186.534L286.907 186.386L286.708 190.199L285.298 190.125L285.564 185.041L282.745 184.894L282.478 189.977L281.068 189.904L281.668 178.465L278.848 178.317L277.982 194.84L276.572 194.766L276.772 190.953L275.362 190.879L275.429 189.608L272.609 189.46L272.409 193.273L273.819 193.347L273.686 195.889L275.096 195.963L274.963 198.505L276.372 198.579L276.239 201.121L277.649 201.195L277.516 203.737L290.204 204.402L290.404 200.589L291.814 200.663L292.014 196.85Z" fill="white"/><path fill-rule="evenodd" clip-rule="evenodd" d="M280.869 193.717L282.278 193.791L281.964 199.782L280.555 199.708L280.869 193.717ZM285.098 193.938L283.688 193.864L283.374 199.856L284.784 199.93L285.098 193.938ZM287.917 194.086L286.507 194.012L286.193 200.004L287.603 200.078L287.917 194.086Z" fill="#141B38"/><defs><filter id="hoverstatefilter0_d_557_20473" x="181.831" y="38.5286" width="160.926" height="168.757" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feMorphology radius="2.5947" operator="erode" in="SourceAlpha" result="effect1_dropShadow_557_20473"/><feOffset dy="3.45959"/><feGaussianBlur stdDeviation="4.75694"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.25 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_557_20473"/><feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_557_20473" result="shape"/></filter><filter id="hoverstatefilter1_d_557_20473" x="56.0808" y="51.6608" width="153.38" height="178.672" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feMorphology radius="2.5947" operator="erode" in="SourceAlpha" result="effect1_dropShadow_557_20473"/><feOffset dy="3.45959"/><feGaussianBlur stdDeviation="4.75694"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.25 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_557_20473"/><feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_557_20473" result="shape"/></filter><clipPath id="hoverstateclip0_557_20473"><rect x="196.486" y="41.9882" width="139.542" height="147.818" rx="1.74428" transform="rotate(3 196.486 41.9882)" fill="white"/></clipPath><clipPath id="hoverstateclip1_557_20473"><rect width="13.9542" height="13.9542" fill="white" transform="translate(205.154 143.248) rotate(3)"/></clipPath><clipPath id="hoverstateclip2_557_20473"><rect x="63" y="55.1204" width="139.542" height="164.834" rx="1.74428" fill="white"/></clipPath><clipPath id="hoverstateclip3_557_20473"><rect width="139.542" height="86.3416" fill="white" transform="translate(62.9998 55.1204)"/></clipPath><clipPath id="hoverstateclip4_557_20473"><rect width="13.9542" height="13.9542" fill="white" transform="translate(75.21 197.279)"/></clipPath></defs></svg>',
				'popupContentBtn' 	=> '<div class="sbi-fb-extpp-lite-btn">' . self::builder_svg_icons()['tag'] . __( 'Instagram Pro users get 50% OFF', 'instagram-feed' ) .'</div>',
				'bullets'       => [
					'heading' => __( 'And get much more!', 'instagram-feed' ),
					'content' => [
						__( 'Display Hashtag & Tagged feeds', 'instagram-feed' ),
						__( 'Powerful visual moderation', 'instagram-feed' ),
						__( 'Comments and Likes', 'instagram-feed' ),
						__( 'Highlight specific posts', 'instagram-feed' ),
						__( 'Multiple layout options', 'instagram-feed' ),
						__( 'Popup photo/video lightbox', 'instagram-feed' ),
						__( 'Instagram Stories', 'instagram-feed' ),
						__( 'Shoppable feeds', 'instagram-feed' ),
						__( 'Pro support', 'instagram-feed' ),
						__( 'Post captions', 'instagram-feed' ),
						__( 'Combine multiple feed types', 'instagram-feed' ),
						__( '30 day money back guarantee', 'instagram-feed' ),
					]
				],
				'buyUrl' 		=> $this->get_extension_popup_dynamic_buy_url( $license_key, 'https://smashballoon.com/instagram-feed/demo/?utm_campaign=instagram-free&utm_source=customizer&utm_medium=posts' )
			),
			'lightbox' => array(
				'heading' 		=> $this->get_extension_popup_dynamic_heading('Lightbox Popup'),
				'description' 	=> __( 'Allow visitors to view your photos and videos in a beautiful full size lightbox, keeping them on your site for longer to discover more of your content.', 'instagram-feed' ),
				'img' 			=> '<svg width="396" height="264" viewBox="0 0 396 264" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M50.5998 136.122L48.2638 134.019L39.3134 143.959L49.2538 152.91L51.3572 150.574L43.7685 143.726L50.5998 136.122Z" fill="#8C8F9A"/><path d="M347.201 119.809L345.097 122.145L352.686 128.993L345.855 136.597L348.191 138.7L357.141 128.76L347.201 119.809Z" fill="#8C8F9A"/><g clip-path="url(#lightboxxclip0_557_20899)" filter="url(#lightboxxfilter0_d_557_20899)"><rect width="261.925" height="173.162" transform="translate(62.6831 52.3919) rotate(-3)" fill="white"/><rect x="112.468" y="187.874" width="93.129" height="5.82056" rx="1.45514" transform="rotate(-3 112.468 187.874)" fill="#D0D1D7"/><rect x="113.151" y="200.952" width="53.8402" height="5.82056" rx="1.45514" transform="rotate(-3 113.151 200.952)" fill="#D0D1D7"/><circle cx="94.1753" cy="195.21" r="8.73084" transform="rotate(-3 94.1753 195.21)" fill="#DCDDE1"/><g clip-path="url(#lightboxxclip1_557_20899)"><rect x="62.6812" y="52.3918" width="262.169" height="173.162" transform="rotate(-3 62.6812 52.3918)" fill="#FE544F"/><rect x="191.14" y="20.5734" width="271.58" height="334.479" rx="55.73" transform="rotate(2.99107 191.14 20.5734)" fill="#DCDDE1"/><circle cx="141.741" cy="201.742" r="113.935" transform="rotate(-3 141.741 201.742)" fill="#0096CC"/></g></g><defs><filter id="lightboxxfilter0_d_557_20899" x="53.8243" y="34.2544" width="288.346" height="204.35" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feMorphology radius="3.32203" operator="erode" in="SourceAlpha" result="effect1_dropShadow_557_20899"/><feOffset dy="4.42938"/><feGaussianBlur stdDeviation="6.0904"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_557_20899"/><feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_557_20899" result="shape"/></filter><clipPath id="lightboxxclip0_557_20899"><rect width="261.925" height="173.162" fill="white" transform="translate(62.6831 52.3919) rotate(-3)"/></clipPath><clipPath id="lightboxxclip1_557_20899"><rect width="262.15" height="121.608" fill="white" transform="translate(62.6821 52.3919) rotate(-3)"/></clipPath></defs></svg>',
				'popupContentBtn' 	=> '<div class="sbi-fb-extpp-lite-btn">' . self::builder_svg_icons()['tag'] . __( 'Instagram Pro users get 50% OFF', 'instagram-feed' ) .'</div>',
				'bullets'       => [
					'heading' => __( 'And get much more!', 'instagram-feed' ),
					'content' => [
						__( 'Display Hashtag & Tagged feeds', 'instagram-feed' ),
						__( 'Powerful visual moderation', 'instagram-feed' ),
						__( 'Comments and Likes', 'instagram-feed' ),
						__( 'Highlight specific posts', 'instagram-feed' ),
						__( 'Multiple layout options', 'instagram-feed' ),
						__( 'Popup photo/video lightbox', 'instagram-feed' ),
						__( 'Instagram Stories', 'instagram-feed' ),
						__( 'Shoppable feeds', 'instagram-feed' ),
						__( 'Pro support', 'instagram-feed' ),
						__( 'Post captions', 'instagram-feed' ),
						__( 'Combine multiple feed types', 'instagram-feed' ),
						__( '30 day money back guarantee', 'instagram-feed' ),
					]
				],
				'buyUrl' 		=> $this->get_extension_popup_dynamic_buy_url( $license_key, 'https://smashballoon.com/instagram-feed/demo/?utm_campaign=instagram-free&utm_source=customizer&utm_medium=lightbox' )
			),
			'filtermoderation' => array(
				'heading' 		=> $this->get_extension_popup_dynamic_heading('Advanced Moderation and Filters'),
				'description' 	=> __( 'Use powerful moderation tools to easily create feeds of only specific photos, or choose specific ones to exclude. You can also easily choose to include or block specific words or phrases in your posts.', 'instagram-feed' ),
				'img' 			=> '<svg width="396" height="264" viewBox="0 0 396 264" fill="none" xmlns="http://www.w3.org/2000/svg"><g clip-path="url(#moderationfltrclip0_542_16736)"><g filter="url(#moderationfltrfilter0_ddd_542_16736)"><rect x="139.981" y="52.5992" width="162.17" height="179.401" rx="3.96584" fill="white"/></g><path d="M156.198 174.881C157.55 174.847 161.57 174.786 166.841 174.819C173.429 174.86 173.429 174.912 179.003 174.881C184.578 174.85 185.592 174.788 199.275 174.85C212.958 174.912 210.424 174.912 217.519 174.881C224.614 174.85 225.627 174.835 238.804 174.881C249.345 174.918 261.778 174.913 266.677 174.907" stroke="#DCDDE1" stroke-width="8.10851"/><path d="M156.198 194.559C157.55 194.53 161.57 194.478 166.841 194.506C173.429 194.542 173.429 194.586 179.003 194.559C184.578 194.533 184.071 194.506 197.754 194.559C211.437 194.613 210.424 194.6 218.026 194.593" stroke="#DCDDE1" stroke-width="8.10851"/><g clip-path="url(#moderationfltrclip1_542_16736)"><rect x="151.131" y="65.7755" width="139.912" height="88.1801" fill="#8C8F9A"/><circle cx="157.01" cy="165.713" r="48.2051" fill="#EC352F"/><circle cx="295.745" cy="112.805" r="65.8411" fill="#DCDDE1"/></g><circle cx="291.51" cy="58.1738" r="18.7509" fill="#D72C2C"/><path d="M290.886 55.6399L293.028 57.7751V57.667C293.028 57.1294 292.815 56.6137 292.435 56.2336C292.054 55.8534 291.539 55.6399 291.001 55.6399H290.886ZM287.981 56.1804L289.028 57.2278C288.994 57.3697 288.974 57.5116 288.974 57.667C288.974 58.2046 289.188 58.7202 289.568 59.1004C289.948 59.4805 290.464 59.6941 291.001 59.6941C291.15 59.6941 291.298 59.6738 291.44 59.6401L292.488 60.6874C292.035 60.9104 291.535 61.0455 291.001 61.0455C290.105 61.0455 289.246 60.6896 288.612 60.056C287.979 59.4224 287.623 58.563 287.623 57.667C287.623 57.1332 287.758 56.6332 287.981 56.1804ZM284.244 52.4438L285.785 53.9844L286.089 54.2884C284.974 55.1669 284.095 56.3156 283.568 57.667C284.737 60.6333 287.623 62.7348 291.001 62.7348C292.049 62.7348 293.049 62.5321 293.961 62.1672L294.251 62.451L296.224 64.4241L297.083 63.5659L285.102 51.5856L284.244 52.4438ZM291.001 54.2884C291.897 54.2884 292.757 54.6444 293.39 55.278C294.024 55.9116 294.38 56.7709 294.38 57.667C294.38 58.0994 294.292 58.5184 294.136 58.8968L296.116 60.8766C297.13 60.032 297.941 58.9238 298.434 57.667C297.265 54.7006 294.38 52.5992 291.001 52.5992C290.055 52.5992 289.15 52.7681 288.298 53.0722L289.765 54.5249C290.15 54.3763 290.562 54.2884 291.001 54.2884Z" fill="white"/><g filter="url(#moderationfltrfilter1_ddd_542_16736)"><rect x="85.7402" y="31.9814" width="162.17" height="179.401" rx="3.96584" fill="white"/><path d="M101.957 154.263C103.308 154.229 107.329 154.169 112.599 154.201C119.188 154.242 119.188 154.294 124.762 154.263C130.337 154.232 131.35 154.171 145.034 154.232C158.717 154.294 156.183 154.294 163.278 154.263C170.373 154.232 171.386 154.217 184.562 154.263C195.104 154.3 207.537 154.296 212.436 154.289" stroke="#DCDDE1" stroke-width="8.10851"/><path d="M101.957 173.942C103.308 173.912 107.329 173.86 112.599 173.889C119.188 173.924 119.188 173.968 124.762 173.942C130.337 173.915 129.83 173.889 143.513 173.942C157.196 173.995 156.183 173.982 163.784 173.975" stroke="#DCDDE1" stroke-width="8.10851"/><g clip-path="url(#moderationfltrclip2_542_16736)"><rect x="96.8887" y="45.1577" width="139.912" height="88.1801" fill="#2C324C"/><circle cx="125.771" cy="71.4144" r="83.2083" fill="#0068A0"/><circle cx="256.9" cy="155.92" r="86.4368" fill="#FE544F"/></g></g><circle cx="234.383" cy="30.7509" r="18.7509" fill="#0096CC"/><g clip-path="url(#moderationfltrclip3_542_16736)"><path d="M233.874 28.217C233.337 28.217 232.821 28.4306 232.441 28.8108C232.061 29.1909 231.847 29.7065 231.847 30.2442C231.847 30.7818 232.061 31.2974 232.441 31.6775C232.821 32.0577 233.337 32.2713 233.874 32.2713C234.412 32.2713 234.927 32.0577 235.308 31.6775C235.688 31.2974 235.901 30.7818 235.901 30.2442C235.901 29.7065 235.688 29.1909 235.308 28.8108C234.927 28.4306 234.412 28.217 233.874 28.217ZM233.874 33.6227C232.978 33.6227 232.119 33.2667 231.485 32.6331C230.852 31.9995 230.496 31.1402 230.496 30.2442C230.496 29.3481 230.852 28.4888 231.485 27.8552C232.119 27.2216 232.978 26.8656 233.874 26.8656C234.77 26.8656 235.63 27.2216 236.263 27.8552C236.897 28.4888 237.253 29.3481 237.253 30.2442C237.253 31.1402 236.897 31.9995 236.263 32.6331C235.63 33.2667 234.77 33.6227 233.874 33.6227ZM233.874 25.1763C230.496 25.1763 227.61 27.2778 226.441 30.2442C227.61 33.2105 230.496 35.312 233.874 35.312C237.253 35.312 240.138 33.2105 241.307 30.2442C240.138 27.2778 237.253 25.1763 233.874 25.1763Z" fill="white"/></g></g><defs><filter id="moderationfltrfilter0_ddd_542_16736" x="114.203" y="38.7187" width="213.726" height="230.957" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="11.8975"/><feGaussianBlur stdDeviation="12.889"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_542_16736"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="1.98292"/><feGaussianBlur stdDeviation="1.98292"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/><feBlend mode="normal" in2="effect1_dropShadow_542_16736" result="effect2_dropShadow_542_16736"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="5.94876"/><feGaussianBlur stdDeviation="5.94876"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/><feBlend mode="normal" in2="effect2_dropShadow_542_16736" result="effect3_dropShadow_542_16736"/><feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow_542_16736" result="shape"/></filter><filter id="moderationfltrfilter1_ddd_542_16736" x="59.9623" y="18.101" width="213.726" height="230.957" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="11.8975"/><feGaussianBlur stdDeviation="12.889"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_542_16736"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="1.98292"/><feGaussianBlur stdDeviation="1.98292"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/><feBlend mode="normal" in2="effect1_dropShadow_542_16736" result="effect2_dropShadow_542_16736"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="5.94876"/><feGaussianBlur stdDeviation="5.94876"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/><feBlend mode="normal" in2="effect2_dropShadow_542_16736" result="effect3_dropShadow_542_16736"/><feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow_542_16736" result="shape"/></filter><clipPath id="moderationfltrclip0_542_16736"><path d="M0 0H396V264H0V0Z" fill="white"/></clipPath><clipPath id="moderationfltrclip1_542_16736"><rect x="151.131" y="65.7755" width="139.912" height="88.1801" rx="2" fill="white"/></clipPath><clipPath id="moderationfltrclip2_542_16736"><rect x="96.8887" y="45.1577" width="139.912" height="88.1801" rx="2" fill="white"/></clipPath><clipPath id="moderationfltrclip3_542_16736"><rect width="16.217" height="16.217" fill="white" transform="translate(225.767 22.1356)"/></clipPath></defs></svg>',
				'popupContentBtn' 	=> '<div class="sbi-fb-extpp-lite-btn">' . self::builder_svg_icons()['tag'] . __( 'Instagram Pro users get 50% OFF', 'instagram-feed' ) .'</div>',
				'bullets'       => [
					'heading' => __( 'And get much more!', 'instagram-feed' ),
					'content' => [
						__( 'Display Hashtag & Tagged feeds', 'instagram-feed' ),
						__( 'Powerful visual moderation', 'instagram-feed' ),
						__( 'Comments and Likes', 'instagram-feed' ),
						__( 'Highlight specific posts', 'instagram-feed' ),
						__( 'Multiple layout options', 'instagram-feed' ),
						__( 'Popup photo/video lightbox', 'instagram-feed' ),
						__( 'Instagram Stories', 'instagram-feed' ),
						__( 'Shoppable feeds', 'instagram-feed' ),
						__( 'Pro support', 'instagram-feed' ),
						__( 'Post captions', 'instagram-feed' ),
						__( 'Combine multiple feed types', 'instagram-feed' ),
						__( '30 day money back guarantee', 'instagram-feed' ),
					]
				],
				'buyUrl' 		=> $this->get_extension_popup_dynamic_buy_url( $license_key, 'https://smashballoon.com/instagram-feed/demo/?utm_campaign=instagram-free&utm_source=customizer&utm_medium=filters' )
			),
			'shoppablefeed' => array(
				'heading' 		=> $this->get_extension_popup_dynamic_heading('Get Shoppable Feeds'),
				'description' 	=> __( 'Automatically link Instagram posts to custom URLs of your choosing by adding the URL in the caption, or manually add links to specific pages or products on your site (or other sites) in a quick and easy way.', 'instagram-feed' ),
				'img' 			=> '<svg width="396" height="264" viewBox="0 0 396 264" fill="none" xmlns="http://www.w3.org/2000/svg"><g filter="url(#shoppablefeedfilter0_d_557_17550)"><rect x="234.717" y="38.2059" width="143" height="158.889" rx="2" transform="rotate(4 234.717 38.2059)" fill="white"/></g><rect width="143" height="82.0926" transform="translate(234.717 38.2059) rotate(4)" fill="#E8E8EB"/><g filter="url(#shoppablefeedfilter1_dd_557_17550)"><mask id="shoppablefeedmask0_557_17550" style="mask-type:alpha" maskUnits="userSpaceOnUse" x="266" y="53" width="80" height="62"><path fill-rule="evenodd" clip-rule="evenodd" d="M315.226 54.937C315.099 58.5599 310.591 61.343 305.157 61.1533C299.723 60.9635 295.42 57.8727 295.546 54.2498C295.546 54.2498 295.546 54.2497 295.546 54.2497L286.163 53.922C286.057 53.9183 285.951 53.9404 285.856 53.9863L278.402 57.5651C278.37 57.5802 278.351 57.6124 278.352 57.6471C278.35 57.62 278.321 57.6035 278.297 57.6159L266.724 63.5265C266.477 63.6528 266.336 63.921 266.372 64.1964L268.295 78.8772C268.344 79.2535 268.701 79.5097 269.073 79.4357L278.317 77.5986C278.714 77.5198 279.086 77.8146 279.101 78.2185L280.339 112.219C280.352 112.563 280.628 112.839 280.972 112.851L326.982 114.458C327.326 114.47 327.621 114.214 327.657 113.872L331.266 80.0236C331.308 79.6249 331.695 79.3572 332.083 79.4578L341.845 81.9875C342.209 82.0819 342.578 81.8513 342.653 81.4825L345.594 66.9741C345.651 66.6966 345.523 66.4143 345.277 66.2738L333.503 59.5421C333.48 59.5288 333.45 59.5433 333.447 59.57C333.45 59.5361 333.433 59.5032 333.404 59.4863L326.216 55.3957C326.124 55.3432 326.021 55.3139 325.915 55.3102L315.226 54.9369C315.226 54.9369 315.226 54.937 315.226 54.937Z" fill="white"/></mask><g mask="url(#shoppablefeedmask0_557_17550)"><rect x="261.444" y="49.1168" width="94.5192" height="70.8894" transform="rotate(4 261.444 49.1168)" fill="white"/><path fill-rule="evenodd" clip-rule="evenodd" d="M304.293 77.7265C302.782 76.9164 301.23 75.7559 299.747 75.1625C300.196 76.4854 300.555 77.8865 300.931 79.2729C299.943 79.9127 298.808 80.3837 297.719 80.9071C298.549 81.8507 299.776 82.4491 300.728 83.2863C299.853 84.2243 298.197 85.147 297.749 86.0211C299.431 85.9288 301.399 85.7258 302.956 85.7657C303.177 87.5038 303.222 89.3956 303.597 90.9999C304.464 88.9723 305.268 86.8705 306.263 84.99C307.552 85.6316 309.122 86.4139 310.395 86.828C309.575 85.4894 308.84 84.0769 308.131 82.6421C309.383 81.8618 310.648 81.0961 311.86 80.2694C310.247 80.0064 308.572 79.7978 306.872 79.6108C306.724 77.8331 306.942 75.7372 306.638 74.0953C305.915 75.3731 305.086 76.5293 304.293 77.7265ZM304.045 92.3479C303.766 93.2504 304.446 93.6761 304.301 94.2247C303.62 93.9356 303.104 93.7755 302.159 93.8425C302.238 93.1288 302.889 93.0725 302.817 92.1846C292.92 90.3527 294.16 72.3239 304.157 71.908C316.657 71.388 315.549 92.832 304.045 92.3479Z" fill="#FE544F"/><path fill-rule="evenodd" clip-rule="evenodd" d="M306.638 74.0951C306.942 75.7371 306.724 77.833 306.872 79.6107C308.572 79.7977 310.247 80.0062 311.86 80.2693C310.648 81.096 309.383 81.8617 308.131 82.642C308.84 84.0768 309.575 85.4893 310.395 86.8279C309.122 86.4138 307.552 85.6315 306.263 84.9899C305.268 86.8704 304.464 88.9721 303.597 90.9997C303.221 89.3955 303.177 87.5037 302.956 85.7655C301.399 85.7257 299.431 85.9287 297.749 86.021C298.197 85.1469 299.853 84.2242 300.728 83.2861C299.776 82.449 298.549 81.8506 297.719 80.907C298.808 80.3836 299.943 79.9126 300.931 79.2727C300.555 77.8864 300.196 76.4853 299.747 75.1624C301.23 75.7557 302.782 76.9163 304.293 77.7264C305.086 76.5292 305.915 75.3729 306.638 74.0951Z" fill="white"/></g></g><path d="M245.82 168.987C245.767 169.739 245.479 170.369 244.954 170.878C244.429 171.387 243.773 171.678 242.987 171.751L242.896 173.041C242.892 173.11 242.866 173.165 242.819 173.206C242.771 173.247 242.714 173.266 242.645 173.261L241.655 173.192C241.591 173.187 241.538 173.16 241.494 173.11C241.451 173.06 241.432 173.003 241.436 172.939L241.526 171.649C241.207 171.583 240.9 171.485 240.607 171.355C240.313 171.226 240.073 171.1 239.885 170.975C239.697 170.851 239.524 170.721 239.367 170.584C239.209 170.447 239.102 170.347 239.045 170.283C238.988 170.22 238.948 170.173 238.926 170.142C238.85 170.033 238.852 169.932 238.932 169.838L239.757 168.896C239.795 168.849 239.853 168.824 239.932 168.82C240.006 168.815 240.064 168.841 240.104 168.898L240.118 168.914C240.636 169.439 241.209 169.788 241.836 169.96C242.014 170.012 242.194 170.044 242.375 170.057C242.771 170.085 243.127 170.003 243.443 169.813C243.759 169.623 243.93 169.334 243.957 168.945C243.967 168.808 243.939 168.675 243.874 168.547C243.81 168.419 243.735 168.31 243.65 168.22C243.566 168.131 243.429 168.028 243.24 167.914C243.052 167.8 242.896 167.71 242.772 167.644C242.649 167.579 242.459 167.485 242.202 167.364C242.017 167.272 241.871 167.2 241.764 167.148C241.657 167.096 241.511 167.02 241.326 166.921C241.141 166.822 240.994 166.735 240.883 166.66C240.773 166.586 240.641 166.489 240.487 166.37C240.333 166.25 240.21 166.137 240.116 166.029C240.023 165.921 239.925 165.793 239.823 165.645C239.72 165.498 239.643 165.349 239.592 165.2C239.541 165.051 239.501 164.884 239.472 164.699C239.443 164.514 239.436 164.321 239.45 164.12C239.497 163.441 239.779 162.864 240.294 162.386C240.809 161.909 241.456 161.623 242.234 161.529L242.327 160.202C242.331 160.139 242.358 160.085 242.408 160.042C242.458 159.998 242.515 159.979 242.578 159.983L243.569 160.052C243.637 160.057 243.692 160.083 243.733 160.13C243.774 160.178 243.792 160.236 243.787 160.305L243.696 161.602C243.973 161.651 244.24 161.726 244.495 161.828C244.751 161.93 244.958 162.027 245.117 162.12C245.275 162.212 245.424 162.315 245.563 162.429C245.703 162.542 245.793 162.62 245.834 162.662C245.876 162.705 245.91 162.742 245.937 162.773C246.014 162.868 246.02 162.962 245.954 163.056L245.285 164.091C245.24 164.162 245.181 164.197 245.108 164.197C245.038 164.207 244.973 164.185 244.913 164.131C244.9 164.115 244.866 164.083 244.813 164.035C244.76 163.987 244.669 163.915 244.54 163.82C244.412 163.725 244.274 163.636 244.128 163.554C243.981 163.472 243.803 163.396 243.594 163.324C243.386 163.253 243.178 163.21 242.973 163.196C242.508 163.163 242.122 163.242 241.813 163.433C241.505 163.624 241.339 163.886 241.316 164.22C241.307 164.348 241.319 164.468 241.353 164.579C241.388 164.69 241.453 164.797 241.549 164.9C241.645 165.003 241.736 165.09 241.821 165.163C241.907 165.236 242.039 165.321 242.216 165.42C242.394 165.519 242.537 165.596 242.646 165.65C242.756 165.705 242.922 165.784 243.146 165.889C243.398 166.005 243.591 166.097 243.724 166.163C243.857 166.229 244.037 166.328 244.264 166.46C244.49 166.592 244.668 166.709 244.796 166.812C244.924 166.914 245.067 167.048 245.225 167.212C245.383 167.376 245.502 167.541 245.581 167.707C245.661 167.873 245.724 168.067 245.773 168.287C245.821 168.508 245.837 168.741 245.82 168.987Z" fill="#0068A0"/><rect x="240.659" y="143.036" width="74.1481" height="10.5926" transform="rotate(4 240.659 143.036)" fill="#DCDDE1"/><rect x="252.575" y="162.452" width="32.6605" height="10.5926" transform="rotate(4 252.575 162.452)" fill="#BFE8FF"/><rect x="328.66" y="112.025" width="33.5432" height="33.5432" rx="16.7716" transform="rotate(4 328.66 112.025)" fill="#FE544F"/><g clip-path="url(#shoppablefeedclip0_557_17550)"><path d="M338.611 121.57C338.377 121.554 338.147 121.631 337.97 121.784C337.794 121.938 337.685 122.155 337.669 122.389C337.653 122.622 337.73 122.853 337.883 123.03C338.037 123.206 338.254 123.315 338.488 123.331L339.562 123.406L339.756 124.501C339.757 124.514 339.759 124.526 339.762 124.539L340.623 129.404L339.782 130.134C338.595 131.166 339.248 133.118 340.817 133.227L348.377 133.756C348.611 133.772 348.841 133.695 349.018 133.542C349.195 133.388 349.303 133.171 349.319 132.937C349.336 132.704 349.259 132.473 349.105 132.296C348.952 132.12 348.734 132.011 348.5 131.995L340.94 131.466L341.882 130.647L347.682 131.053C347.845 131.064 348.008 131.03 348.153 130.954C348.299 130.877 348.419 130.762 348.503 130.621L351.514 125.522C351.59 125.393 351.632 125.246 351.636 125.096C351.64 124.945 351.605 124.796 351.535 124.663C351.465 124.53 351.363 124.417 351.237 124.335C351.111 124.252 350.966 124.204 350.816 124.193L341.376 123.533L341.18 122.419C341.145 122.226 341.047 122.049 340.901 121.917C340.755 121.786 340.568 121.707 340.372 121.693L338.611 121.57Z" fill="white"/><path d="M349.104 136.019C349.08 136.369 348.917 136.696 348.652 136.926C348.387 137.156 348.041 137.272 347.691 137.248C347.34 137.223 347.014 137.06 346.784 136.795C346.553 136.53 346.438 136.185 346.462 135.834C346.487 135.484 346.649 135.158 346.915 134.928C347.18 134.697 347.525 134.581 347.876 134.606C348.226 134.63 348.552 134.793 348.782 135.058C349.013 135.323 349.129 135.669 349.104 136.019Z" fill="white"/><path d="M340.646 136.755C340.996 136.78 341.342 136.664 341.607 136.433C341.872 136.203 342.035 135.877 342.059 135.527C342.084 135.176 341.968 134.831 341.738 134.566C341.507 134.3 341.181 134.138 340.831 134.113C340.48 134.089 340.135 134.204 339.87 134.435C339.605 134.665 339.442 134.992 339.417 135.342C339.393 135.692 339.509 136.038 339.739 136.303C339.969 136.568 340.296 136.731 340.646 136.755Z" fill="white"/></g><path d="M355.663 146.817L355.732 145.836L354.644 145.76L354.712 144.78L352.537 144.628L352.605 143.647L349.342 143.419L349.411 142.438L347.235 142.286L347.509 138.363L346.422 138.287L346.49 137.307L344.315 137.155L344.246 138.135L343.158 138.059L342.541 146.885L341.454 146.809L341.522 145.828L338.259 145.6L337.985 149.523L339.072 149.599L338.935 151.56L340.023 151.636L339.886 153.597L340.974 153.673L340.836 155.634L341.924 155.711L341.718 158.652L353.684 159.489L353.958 155.567L355.046 155.643L355.252 152.701L354.164 152.625L353.958 155.567L352.87 155.491L352.665 158.432L342.875 157.748L343.012 155.787L341.924 155.711L342.061 153.749L340.974 153.673L341.111 151.712L340.023 151.636L340.16 149.675L339.072 149.599L339.278 146.657L341.454 146.809L341.385 147.789L342.473 147.866L342.267 150.807L343.355 150.883L344.246 138.135L346.422 138.287L345.805 147.113L346.892 147.189L347.167 143.267L349.342 143.419L349.068 147.341L350.156 147.417L350.361 144.475L352.537 144.628L352.263 148.55L353.35 148.626L353.556 145.684L354.644 145.76L354.575 146.741L355.663 146.817L355.252 152.701L356.339 152.777L356.751 146.893L355.663 146.817Z" fill="#141B38"/><path d="M354.164 152.625L355.252 152.701L355.663 146.817L354.576 146.741L354.644 145.76L353.556 145.684L353.351 148.626L352.263 148.55L352.537 144.628L350.362 144.476L350.156 147.417L349.068 147.341L349.342 143.419L347.167 143.267L346.893 147.189L345.805 147.113L346.422 138.287L344.247 138.135L343.355 150.884L342.267 150.807L342.473 147.866L341.385 147.79L341.454 146.809L339.278 146.657L339.073 149.599L340.16 149.675L340.023 151.636L341.111 151.712L340.974 153.673L342.062 153.749L341.924 155.711L343.012 155.787L342.875 157.748L352.665 158.433L352.871 155.491L353.958 155.567L354.164 152.625Z" fill="white"/><path fill-rule="evenodd" clip-rule="evenodd" d="M345.6 150.055L346.687 150.131L346.364 154.754L345.276 154.678L345.6 150.055ZM348.861 150.283L347.774 150.207L347.45 154.83L348.538 154.906L348.861 150.283ZM351.037 150.435L349.949 150.359L349.626 154.982L350.714 155.058L351.037 150.435Z" fill="#141B38"/><g filter="url(#shoppablefeedfilter2_d_557_17550)"><g clip-path="url(#shoppablefeedclip1_557_17550)"><rect x="19.4355" y="57.3804" width="135.359" height="149.741" rx="1.66935" transform="rotate(-4 19.4355 57.3804)" fill="white"/><path d="M40.0569 158.253C41.1801 158.146 44.5242 157.862 48.9146 157.582C54.4026 157.233 54.4056 157.275 59.0454 156.925C63.6853 156.575 64.5256 156.465 75.9223 155.719C87.319 154.974 85.2092 155.121 91.115 154.683C97.0207 154.244 97.8637 154.172 108.838 153.443C117.617 152.86 127.969 152.133 132.047 151.842" stroke="#DCDDE1" stroke-width="6.76797"/><path d="M41.2028 174.638C42.3263 174.535 45.6709 174.258 50.061 173.974C55.5487 173.62 55.5512 173.657 60.1913 173.31C64.8314 172.964 64.4079 172.971 75.8041 172.219C87.2003 171.466 86.3557 171.515 92.6848 171.066" stroke="#DCDDE1" stroke-width="6.76797"/><g clip-path="url(#shoppablefeedclip2_557_17550)"><rect x="28.8848" y="56.2416" width="126" height="85" transform="rotate(-4 28.8848 56.2416)" fill="#2C324C"/><circle cx="55.061" cy="87.8833" r="69.4519" transform="rotate(-4 55.061 87.8833)" fill="#0068A0"/><circle cx="169.165" cy="150.611" r="72.1466" transform="rotate(-4 169.165 150.611)" fill="#FE544F"/><g filter="url(#shoppablefeedfilter3_dd_557_17550)"><mask id="shoppablefeedmask1_557_17550" style="mask-type:alpha" maskUnits="userSpaceOnUse" x="48" y="63" width="85" height="66"><path fill-rule="evenodd" clip-rule="evenodd" d="M99.0497 64.5008C99.3159 68.3075 94.9027 71.7172 89.1926 72.1164C83.4825 72.5157 78.6378 69.7535 78.3716 65.9467C78.3716 65.9467 78.3716 65.9467 78.3715 65.9467L68.5145 66.6359C68.4031 66.6437 68.2952 66.6784 68.2002 66.737L60.7405 71.3345C60.7395 71.3351 60.739 71.3362 60.7391 71.3374C60.739 71.3365 60.7379 71.336 60.7372 71.3365L49.2225 78.8298C48.9775 78.9893 48.8594 79.2856 48.9274 79.5698L52.5559 94.7276C52.6489 95.1162 53.0511 95.3451 53.4327 95.2267L62.9064 92.2867C63.3127 92.1606 63.7354 92.4282 63.7952 92.8495L68.8327 128.308C68.8837 128.667 69.2033 128.925 69.5649 128.9L117.909 125.52C118.27 125.494 118.551 125.194 118.552 124.831L118.605 88.9987C118.606 88.5767 118.981 88.2538 119.399 88.3164L129.898 89.8909C130.289 89.9496 130.65 89.6677 130.688 89.2734L132.171 73.7607C132.2 73.4641 132.035 73.1826 131.762 73.0626L118.741 67.3309C118.693 67.3098 118.639 67.3439 118.637 67.3962C118.638 67.3298 118.598 67.2696 118.537 67.2429L110.609 63.7715C110.507 63.7267 110.395 63.7074 110.284 63.7152L99.0498 64.5007C99.0497 64.5007 99.0497 64.5007 99.0497 64.5008Z" fill="white"/></mask><g mask="url(#shoppablefeedmask1_557_17550)"><rect x="42.1084" y="64.3252" width="99.4973" height="74.623" transform="rotate(-2 42.1084 64.3252)" fill="white"/><path fill-rule="evenodd" clip-rule="evenodd" d="M90.1159 89.5619C88.4449 88.8801 86.6928 87.8358 85.0752 87.3778C85.6909 88.7134 86.221 90.1407 86.7672 91.5507C85.8032 92.3293 84.6669 92.9472 83.584 93.6151C84.5566 94.5116 85.907 95.0031 86.9961 95.7747C86.1829 96.853 84.5514 98.0011 84.1777 98.9657C85.9293 98.6838 87.9668 98.2548 89.6011 98.1252C90.0239 99.9205 90.2786 101.896 90.848 103.534C91.5332 101.316 92.1429 99.0275 92.9782 96.9493C94.3977 97.4792 96.1277 98.1254 97.5056 98.4189C96.5002 97.1077 95.5757 95.7098 94.6756 94.2857C95.9006 93.331 97.1405 92.3903 98.3182 91.3914C96.6009 91.2935 94.824 91.2595 93.0233 91.2508C92.6728 89.406 92.6705 87.1879 92.1715 85.5023C91.5551 86.9196 90.8146 88.2213 90.1159 89.5619ZM91.4652 104.896C91.273 105.872 92.0317 106.243 91.9398 106.833C91.1955 106.605 90.6377 106.494 89.6553 106.669C89.6598 105.913 90.335 105.782 90.1619 104.86C79.5989 104.032 78.9138 85.0208 89.334 83.4854C102.363 81.5656 103.562 104.137 91.4652 104.896Z" fill="#FE544F"/><path fill-rule="evenodd" clip-rule="evenodd" d="M92.1716 85.5022C92.6706 87.1877 92.6729 89.4059 93.0234 91.2507C94.824 91.2593 96.601 91.2933 98.3183 91.3913C97.1406 92.3901 95.9007 93.3309 94.6757 94.2855C95.5758 95.7097 96.5003 97.1075 97.5057 98.4187C96.1278 98.1252 94.3978 97.479 92.9782 96.9491C92.143 99.0274 91.5333 101.316 90.8481 103.534C90.2787 101.896 90.024 99.9204 89.6012 98.125C87.9669 98.2547 85.9294 98.6837 84.1778 98.9655C84.5514 98.001 86.183 96.8528 86.9962 95.7745C85.907 95.0029 84.5567 94.5115 83.584 93.6149C84.6669 92.9471 85.8032 92.3291 86.7673 91.5505C86.2211 90.1405 85.691 88.7132 85.0753 87.3776C86.6928 87.8357 88.445 88.8799 90.116 89.5618C90.8147 88.2211 91.5551 86.9195 92.1716 85.5022Z" fill="white"/></g></g></g></g></g><path d="M169 126C178.5 138 207.5 138 214 128" stroke="#8C8F9A" stroke-width="2" stroke-dasharray="3 3"/><path d="M212.852 124.415L218.453 123.627L218.442 128.627L212.852 124.415Z" fill="#8C8F9A"/><defs><filter id="shoppablefeedfilter0_d_557_17550" x="216.61" y="34.7618" width="167.782" height="182.523" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feMorphology radius="2.68407" operator="erode" in="SourceAlpha" result="effect1_dropShadow_557_17550"/><feOffset dy="3.57876"/><feGaussianBlur stdDeviation="4.92079"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.25 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_557_17550"/><feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_557_17550" result="shape"/></filter><filter id="shoppablefeedfilter1_dd_557_17550" x="262.428" y="49.9833" width="89.7433" height="71.0389" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dx="1.31277" dy="1.31277"/><feGaussianBlur stdDeviation="2.62553"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.13 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_557_17550"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.656383"/><feGaussianBlur stdDeviation="0.328192"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.15 0"/><feBlend mode="normal" in2="effect1_dropShadow_557_17550" result="effect2_dropShadow_557_17550"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_557_17550" result="shape"/></filter><filter id="shoppablefeedfilter2_d_557_17550" x="12.6676" y="44.5542" width="159.011" height="172.355" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feMorphology radius="2.53799" operator="erode" in="SourceAlpha" result="effect1_dropShadow_557_17550"/><feOffset dy="3.38398"/><feGaussianBlur stdDeviation="4.65298"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.25 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_557_17550"/><feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_557_17550" result="shape"/></filter><filter id="shoppablefeedfilter3_dd_557_17550" x="44.7625" y="59.5677" width="94.3219" height="76.2436" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dx="1.38191" dy="1.38191"/><feGaussianBlur stdDeviation="2.76381"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.13 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_557_17550"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.690954"/><feGaussianBlur stdDeviation="0.345477"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.15 0"/><feBlend mode="normal" in2="effect1_dropShadow_557_17550" result="effect2_dropShadow_557_17550"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_557_17550" result="shape"/></filter><clipPath id="shoppablefeedclip0_557_17550"><rect width="17.6543" height="17.6543" fill="white" transform="translate(336.031 120.505) rotate(4)"/></clipPath><clipPath id="shoppablefeedclip1_557_17550"><rect x="19.4355" y="57.3804" width="135.359" height="149.741" rx="1.66935" transform="rotate(-4 19.4355 57.3804)" fill="white"/></clipPath><clipPath id="shoppablefeedclip2_557_17550"><rect width="136" height="85" fill="white" transform="translate(18.9092 56.9391) rotate(-4)"/></clipPath></defs></svg>',
				'popupContentBtn' 	=> '<div class="sbi-fb-extpp-lite-btn">' . self::builder_svg_icons()['tag'] . __( 'Instagram Pro users get 50% OFF', 'instagram-feed' ) .'</div>',
				'bullets'       => [
					'heading' => __( 'And get much more!', 'instagram-feed' ),
					'content' => [
						__( 'Display Hashtag & Tagged feeds', 'instagram-feed' ),
						__( 'Powerful visual moderation', 'instagram-feed' ),
						__( 'Comments and Likes', 'instagram-feed' ),
						__( 'Highlight specific posts', 'instagram-feed' ),
						__( 'Multiple layout options', 'instagram-feed' ),
						__( 'Popup photo/video lightbox', 'instagram-feed' ),
						__( 'Instagram Stories', 'instagram-feed' ),
						__( 'Shoppable feeds', 'instagram-feed' ),
						__( 'Pro support', 'instagram-feed' ),
						__( 'Post captions', 'instagram-feed' ),
						__( 'Combine multiple feed types', 'instagram-feed' ),
						__( '30 day money back guarantee', 'instagram-feed' ),
					]
				],
				'buyUrl' 		=> $this->get_extension_popup_dynamic_buy_url( $license_key, 'https://smashballoon.com/instagram-feed/demo/?utm_campaign=instagram-free&utm_source=customizer&utm_medium=shoppable' )
			),
		];
	}

	/**
	 * Get dynamic heading for the extension popup depending on license state
	 *
	 * @since 6.2.0
	 */
	public function get_extension_popup_dynamic_heading( $extension_title ) {
		if ( sbi_license_inactive_state() ) {
			return sprintf( __( 'Activate your License to get %s', 'instagram-feed' ), $extension_title );
		} else {
			if ( sbi_license_notices_active() ) {
				return sprintf( __( 'Renew license to get %s', 'instagram-feed' ), $extension_title );
			} else {
				return sprintf( __( 'Upgrade to Pro to get %s', 'instagram-feed' ), $extension_title );
			}
		}
	}

	/**
	 * Get dynamic upgrade/activate/renew URL depending on license state
	 *
	 * @since 6.2.0
	 */
	public function get_extension_popup_dynamic_buy_url( $license_key, $default_upgrade_url ) {
		if ( sbi_license_inactive_state() ) {
			return admin_url('admin.php?page=sbi-settings&focus=license');
		}
		if ( sbi_license_notices_active() ) {
			return SBI_Admin_Notices::get_renew_url();
		}
		return $default_upgrade_url;
	}

	/**
	 * For Other Platforms listed on the footer widget
	 *
	 * @return array
	 *
	 * @since 6.0
	 */
	public static function builder_svg_icons() {
		$builder_svg_icons = array(
			'youtube'             => '<svg viewBox="0 0 14 11" fill="none"><path d="M5.66683 7.5L9.12683 5.5L5.66683 3.5V7.5ZM13.3735 2.28C13.4602 2.59334 13.5202 3.01334 13.5602 3.54667C13.6068 4.08 13.6268 4.54 13.6268 4.94L13.6668 5.5C13.6668 6.96 13.5602 8.03334 13.3735 8.72C13.2068 9.32 12.8202 9.70667 12.2202 9.87334C11.9068 9.96 11.3335 10.02 10.4535 10.06C9.58683 10.1067 8.7935 10.1267 8.06016 10.1267L7.00016 10.1667C4.20683 10.1667 2.46683 10.06 1.78016 9.87334C1.18016 9.70667 0.793496 9.32 0.626829 8.72C0.540163 8.40667 0.480163 7.98667 0.440163 7.45334C0.393496 6.92 0.373496 6.46 0.373496 6.06L0.333496 5.5C0.333496 4.04 0.440163 2.96667 0.626829 2.28C0.793496 1.68 1.18016 1.29334 1.78016 1.12667C2.0935 1.04 2.66683 0.980002 3.54683 0.940002C4.4135 0.893336 5.20683 0.873336 5.94016 0.873336L7.00016 0.833336C9.7935 0.833336 11.5335 0.940003 12.2202 1.12667C12.8202 1.29334 13.2068 1.68 13.3735 2.28Z"/></svg>',
			'twitter'             => '<svg viewBox="0 0 14 12" fill="none"><path d="M13.9735 1.50001C13.4602 1.73334 12.9069 1.88667 12.3335 1.96001C12.9202 1.60667 13.3735 1.04667 13.5869 0.373338C13.0335 0.706672 12.4202 0.940005 11.7735 1.07334C11.2469 0.500005 10.5069 0.166672 9.66686 0.166672C8.10019 0.166672 6.82019 1.44667 6.82019 3.02667C6.82019 3.25334 6.84686 3.47334 6.89352 3.68001C4.52019 3.56001 2.40686 2.42 1.00019 0.693338C0.753522 1.11334 0.613522 1.60667 0.613522 2.12667C0.613522 3.12 1.11352 4 1.88686 4.5C1.41352 4.5 0.973522 4.36667 0.586856 4.16667V4.18667C0.586856 5.57334 1.57352 6.73334 2.88019 6.99334C2.46067 7.10814 2.02025 7.12412 1.59352 7.04C1.77459 7.60832 2.12921 8.10561 2.60753 8.46196C3.08585 8.81831 3.66382 9.0158 4.26019 9.02667C3.24928 9.82696 1.99619 10.2595 0.706855 10.2533C0.480189 10.2533 0.253522 10.24 0.0268555 10.2133C1.29352 11.0267 2.80019 11.5 4.41352 11.5C9.66686 11.5 12.5535 7.14 12.5535 3.36C12.5535 3.23334 12.5535 3.11334 12.5469 2.98667C13.1069 2.58667 13.5869 2.08 13.9735 1.50001Z"/></svg>',
			'instagram'           => '<svg width="18" height="18" viewBox="0 0 18 18" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M9 4.50781C6.5 4.50781 4.50781 6.53906 4.50781 9C4.50781 11.5 6.5 13.4922 9 13.4922C11.4609 13.4922 13.4922 11.5 13.4922 9C13.4922 6.53906 11.4609 4.50781 9 4.50781ZM9 11.9297C7.39844 11.9297 6.07031 10.6406 6.07031 9C6.07031 7.39844 7.35938 6.10938 9 6.10938C10.6016 6.10938 11.8906 7.39844 11.8906 9C11.8906 10.6406 10.6016 11.9297 9 11.9297ZM14.7031 4.35156C14.7031 3.76562 14.2344 3.29688 13.6484 3.29688C13.0625 3.29688 12.5938 3.76562 12.5938 4.35156C12.5938 4.9375 13.0625 5.40625 13.6484 5.40625C14.2344 5.40625 14.7031 4.9375 14.7031 4.35156ZM17.6719 5.40625C17.5938 4 17.2812 2.75 16.2656 1.73438C15.25 0.71875 14 0.40625 12.5938 0.328125C11.1484 0.25 6.8125 0.25 5.36719 0.328125C3.96094 0.40625 2.75 0.71875 1.69531 1.73438C0.679688 2.75 0.367188 4 0.289062 5.40625C0.210938 6.85156 0.210938 11.1875 0.289062 12.6328C0.367188 16.0391 0.679688 15.25 1.69531 16.3047C2.75 17.3203 3.96094 17.6328 5.36719 17.7109C6.8125 17.7891 11.1484 17.7891 12.5938 17.7109C14 17.6328 15.25 17.3203 16.2656 16.3047C17.2812 15.25 17.5938 16.0391 17.6719 12.6328C17.75 11.1875 17.75 6.85156 17.6719 5.40625ZM15.7969 14.1562C15.5234 14.9375 14.8984 15.5234 14.1562 15.8359C12.9844 16.3047 10.25 16.1875 9 16.1875C7.71094 16.1875 4.97656 16.3047 3.84375 15.8359C3.0625 15.5234 2.47656 14.9375 2.16406 14.1562C1.69531 13.0234 1.8125 10.2891 1.8125 9C1.8125 7.75 1.69531 5.01562 2.16406 3.84375C2.47656 3.10156 3.0625 2.51562 3.84375 2.20312C4.97656 1.73438 7.71094 1.85156 9 1.85156C10.25 1.85156 12.9844 1.73438 14.1562 2.20312C14.8984 2.47656 15.4844 3.10156 15.7969 3.84375C16.2656 5.01562 16.1484 7.75 16.1484 9C16.1484 10.2891 16.2656 13.0234 15.7969 14.1562Z" fill="url(#paint0_linear)"/><defs><linearGradient id="paint0_linear" x1="6.46484" y1="33.7383" x2="43.3242" y2="-3.88672" gradientUnits="userSpaceOnUse"><stop stop-color="white"/><stop offset="0.147864" stop-color="#F6640E"/><stop offset="0.443974" stop-color="#BA03A7"/><stop offset="0.733337" stop-color="#6A01B9"/><stop offset="1" stop-color="#6B01B9"/></linearGradient></defs></svg>',
			'facebook'            => '<svg viewBox="0 0 14 15"><path d="M7.00016 0.860001C3.3335 0.860001 0.333496 3.85333 0.333496 7.54C0.333496 10.8733 2.7735 13.64 5.96016 14.14V9.47333H4.26683V7.54H5.96016V6.06667C5.96016 4.39333 6.9535 3.47333 8.48016 3.47333C9.20683 3.47333 9.96683 3.6 9.96683 3.6V5.24667H9.12683C8.30016 5.24667 8.04016 5.76 8.04016 6.28667V7.54H9.8935L9.5935 9.47333H8.04016V14.14C9.61112 13.8919 11.0416 13.0903 12.0734 11.88C13.1053 10.6697 13.6704 9.13043 13.6668 7.54C13.6668 3.85333 10.6668 0.860001 7.00016 0.860001Z"/></svg>',
			'smash' 		      => '<svg height="18" viewBox="0 0 28 36" fill="none" xmlns="http://www.w3.org/2000/svg"><path fill-rule="evenodd" clip-rule="evenodd" d="M27.2235 16.8291C27.2235 7.53469 21.1311 0 13.6131 0C6.09513 0 0 7.53469 0 16.8291C0 25.7393 5.5828 33.0095 12.6525 33.6193L11.9007 36L16.6147 35.599L14.9608 33.5775C21.8439 32.7422 27.2235 25.5639 27.2235 16.8291Z" fill="#FE544F"/><path fill-rule="evenodd" clip-rule="evenodd" d="M16.8586 5.91699L17.5137 12.6756L24.3006 12.8705L19.3911 17.4354L23.2687 23.044L16.7362 21.816L14.7557 28.3487L11.7488 22.4987L5.67719 25.2808L8.01283 19.0094L2.09131 16.0227L8.43013 13.9841L6.68099 7.73959L12.678 11.1585L16.8586 5.91699Z" fill="white"/></svg>',
			'tag'                 => '<svg viewBox="0 0 18 18"><path d="M16.841 8.65033L9.34102 1.15033C9.02853 0.840392 8.60614 0.666642 8.16602 0.666993H2.33268C1.89066 0.666993 1.46673 0.842587 1.15417 1.15515C0.841611 1.46771 0.666016 1.89163 0.666016 2.33366V8.16699C0.665842 8.38692 0.709196 8.60471 0.79358 8.8078C0.877964 9.01089 1.00171 9.19528 1.15768 9.35033L8.65768 16.8503C8.97017 17.1603 9.39256 17.334 9.83268 17.3337C10.274 17.3318 10.6966 17.155 11.0077 16.842L16.841 11.0087C17.154 10.6975 17.3308 10.275 17.3327 9.83366C17.3329 9.61373 17.2895 9.39595 17.2051 9.19285C17.1207 8.98976 16.997 8.80538 16.841 8.65033ZM9.83268 15.667L2.33268 8.16699V2.33366H8.16602L15.666 9.83366L9.83268 15.667ZM4.41602 3.16699C4.66324 3.16699 4.90492 3.2403 5.11048 3.37766C5.31604 3.51501 5.47626 3.71023 5.57087 3.93864C5.66548 4.16705 5.69023 4.41838 5.642 4.66086C5.59377 4.90333 5.47472 5.12606 5.2999 5.30088C5.12508 5.47569 4.90236 5.59474 4.65988 5.64297C4.4174 5.69121 4.16607 5.66645 3.93766 5.57184C3.70925 5.47723 3.51403 5.31702 3.37668 5.11146C3.23933 4.90589 3.16602 4.66422 3.16602 4.41699C3.16602 6.08547 3.29771 3.76753 3.53213 3.53311C3.76655 3.29869 6.0845 3.16699 4.41602 3.16699Z"/></svg>',
			'copy'                => '<svg viewBox="0 0 12 13" fill="none"><path d="M10.25 0.25H4.625C3.9375 0.25 3.375 0.8125 3.375 1.5V9C3.375 9.6875 3.9375 10.25 4.625 10.25H10.25C10.9375 10.25 11.5 9.6875 11.5 9V1.5C11.5 0.8125 10.9375 0.25 10.25 0.25ZM10.25 9H4.625V1.5H10.25V9ZM0.875 8.375V7.125H2.125V8.375H0.875ZM0.875 4.9375H2.125V6.1875H0.875V4.9375ZM5.25 11.5H6.5V12.75H5.25V11.5ZM0.875 10.5625V9.3125H2.125V10.5625H0.875ZM2.125 12.75C1.4375 12.75 0.875 12.1875 0.875 11.5H2.125V12.75ZM4.3125 12.75H3.0625V11.5H4.3125V12.75ZM7.4375 12.75V11.5H8.6875C8.6875 12.1875 8.125 12.75 7.4375 12.75ZM2.125 2.75V4H0.875C0.875 3.3125 1.4375 2.75 2.125 2.75Z"/></svg>',
			'duplicate'           => '<svg viewBox="0 0 10 12" fill="none"><path d="M6.99997 0.5H0.999969C0.449969 0.5 -3.05176e-05 0.95 -3.05176e-05 1.5V8.5H0.999969V1.5H6.99997V0.5ZM8.49997 2.5H2.99997C2.44997 2.5 1.99997 2.95 1.99997 3.5V10.5C1.99997 11.05 2.44997 11.5 2.99997 11.5H8.49997C9.04997 11.5 9.49997 11.05 9.49997 10.5V3.5C9.49997 2.95 9.04997 2.5 8.49997 2.5ZM8.49997 10.5H2.99997V3.5H8.49997V10.5Z"/></svg>',
			'edit'                => '<svg width="11" height="12" viewBox="0 0 11 12" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M0.25 9.06241V11.2499H2.4375L8.88917 4.79824L6.70167 2.61074L0.25 9.06241ZM10.9892 2.69824L8.80167 0.510742L7.32583 1.99241L9.51333 4.17991L10.9892 2.69824Z" fill="currentColor"/></svg>',
			'delete'              => '<svg viewBox="0 0 10 12" fill="none"><path d="M1.00001 10.6667C1.00001 11.4 1.60001 12 2.33334 12H7.66668C8.40001 12 9.00001 11.4 9.00001 10.6667V2.66667H1.00001V10.6667ZM2.33334 4H7.66668V10.6667H2.33334V4ZM7.33334 0.666667L6.66668 0H3.33334L2.66668 0.666667H0.333344V2H9.66668V0.666667H7.33334Z"/></svg>',
			'checkmark'           => '<svg width="11" height="9"><path fill-rule="evenodd" clip-rule="evenodd" d="M4.15641 5.65271L9.72487 0.0842487L10.9623 1.32169L4.15641 8.12759L0.444097 4.41528L1.68153 3.17784L4.15641 5.65271Z"/></svg>',
			'checkmarklarge'      => '<svg width="16" height="12" viewBox="0 0 16 12" fill="none" xmlns="http://www.w3.org/2000/svg"><path fill-rule="evenodd" clip-rule="evenodd" d="M6.08058 8.36133L16.0355 0.406383L15.8033 2.17415L6.08058 11.8969L0.777281 6.59357L2.54505 4.8258L6.08058 8.36133Z" fill="currentColor"></path></svg>',
			'information'         => '<svg width="14" height="14" viewBox="0 0 14 14" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M6.3335 5H7.66683V3.66667H6.3335V5ZM7.00016 12.3333C4.06016 12.3333 1.66683 9.94 1.66683 7C1.66683 4.06 4.06016 1.66667 7.00016 1.66667C9.94016 1.66667 12.3335 4.06 12.3335 7C12.3335 9.94 9.94016 12.3333 7.00016 12.3333ZM7.00016 0.333332C6.12468 0.333332 5.25778 0.505771 4.44894 0.840802C3.6401 1.17583 2.90517 1.6669 2.28612 2.28595C1.03588 3.5362 0.333496 5.23189 0.333496 7C0.333496 8.76811 1.03588 10.4638 2.28612 11.714C2.90517 12.3331 3.6401 12.8242 4.44894 13.1592C5.25778 13.4942 6.12468 13.6667 7.00016 13.6667C8.76827 13.6667 10.464 12.9643 11.7142 11.714C12.9645 10.4638 13.6668 8.76811 13.6668 7C13.6668 6.12452 13.4944 5.25761 13.1594 4.44878C12.8243 3.63994 12.3333 2.90501 11.7142 2.28595C11.0952 1.6669 10.3602 1.17583 9.55139 0.840802C8.74255 0.505771 7.87564 0.333332 7.00016 0.333332ZM6.3335 10.3333H7.66683V6.33333H6.3335V10.3333Z" fill="#141B38"/></svg>',
			'cog'                 => '<svg width="14" height="14" viewBox="0 0 14 14" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M6.99989 9.33334C6.38105 9.33334 5.78756 9.0875 5.34998 8.64992C4.91239 8.21233 4.66656 7.61884 4.66656 7C4.66656 6.38117 4.91239 5.78767 5.34998 5.35009C5.78756 4.9125 6.38105 4.66667 6.99989 4.66667C7.61873 4.66667 8.21222 4.9125 8.64981 5.35009C9.08739 5.78767 9.33323 6.38117 9.33323 7C9.33323 7.61884 9.08739 8.21233 8.64981 8.64992C8.21222 9.0875 7.61873 9.33334 6.99989 9.33334ZM11.9532 7.64667C11.9799 7.43334 11.9999 7.22 11.9999 7C11.9999 6.78 11.9799 6.56 11.9532 6.33334L13.3599 5.24667C13.4866 5.14667 13.5199 4.96667 13.4399 4.82L12.1066 2.51334C12.0266 2.36667 11.8466 2.30667 11.6999 2.36667L10.0399 3.03334C9.69323 2.77334 9.33323 2.54667 8.91323 2.38L8.66656 0.613337C8.65302 0.534815 8.61212 0.463622 8.5511 0.412371C8.49009 0.361121 8.41291 0.333123 8.33323 0.333337H5.66656C5.49989 0.333337 5.35989 0.453337 5.33323 0.613337L5.08656 2.38C4.66656 2.54667 4.30656 2.77334 3.95989 3.03334L2.29989 2.36667C2.15323 2.30667 1.97323 2.36667 1.89323 2.51334L0.559893 4.82C0.473226 4.96667 0.513226 5.14667 0.639893 5.24667L2.04656 6.33334C2.01989 6.56 1.99989 6.78 1.99989 7C1.99989 7.22 2.01989 7.43334 2.04656 7.64667L0.639893 8.75334C0.513226 8.85334 0.473226 9.03334 0.559893 9.18L1.89323 11.4867C1.97323 11.6333 2.15323 11.6867 2.29989 11.6333L3.95989 10.96C4.30656 11.2267 4.66656 11.4533 5.08656 11.62L5.33323 13.3867C5.35989 13.5467 5.49989 13.6667 5.66656 13.6667H8.33323C8.49989 13.6667 8.63989 13.5467 8.66656 13.3867L8.91323 11.62C9.33323 11.4467 9.69323 11.2267 10.0399 10.96L11.6999 11.6333C11.8466 11.6867 12.0266 11.6333 12.1066 11.4867L13.4399 9.18C13.5199 9.03334 13.4866 8.85334 13.3599 8.75334L11.9532 7.64667Z" fill="#141B38"/></svg>',
			'angleUp'             => '<svg width="8" height="6" viewBox="0 0 8 6" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M0.94 5.27325L4 2.21992L7.06 5.27325L8 4.33325L4 0.333252L0 4.33325L0.94 5.27325Z" fill="#434960"/></svg>',
			'user_check'          => '<svg viewBox="0 0 11 9"><path d="M9.55 4.25L10.25 4.955L6.985 8.25L5.25 6.5L5.95 5.795L6.985 6.835L9.55 4.25ZM4 6.5L5.5 8H0.5V7C0.5 5.895 2.29 5 4.5 5L5.445 5.055L4 6.5ZM4.5 0C5.03043 0 5.53914 0.210714 5.91421 0.585786C6.28929 0.960859 6.5 1.46957 6.5 2C6.5 2.53043 6.28929 3.03914 5.91421 3.41421C5.53914 3.78929 5.03043 4 4.5 4C3.96957 4 3.46086 3.78929 3.08579 3.41421C2.71071 3.03914 2.5 2.53043 2.5 2C2.5 1.46957 2.71071 0.960859 3.08579 0.585786C3.46086 0.210714 3.96957 0 4.5 0Z"/></svg>',
			'users'               => '<svg viewBox="0 0 12 8"><path d="M6 0.75C6.46413 0.75 6.90925 0.934375 7.23744 1.26256C7.56563 1.59075 7.75 2.03587 7.75 2.5C7.75 2.96413 7.56563 3.40925 7.23744 3.73744C6.90925 6.06563 6.46413 4.25 6 4.25C5.53587 4.25 5.09075 6.06563 4.76256 3.73744C4.43437 3.40925 4.25 2.96413 4.25 2.5C4.25 2.03587 4.43437 1.59075 4.76256 1.26256C5.09075 0.934375 5.53587 0.75 6 0.75ZM2.5 2C2.78 2 3.04 2.075 3.265 2.21C3.19 2.925 3.4 3.635 3.83 4.19C3.58 4.67 3.08 5 2.5 5C2.10218 5 1.72064 4.84196 1.43934 4.56066C1.15804 4.27936 1 3.89782 1 3.5C1 3.10218 1.15804 2.72064 1.43934 2.43934C1.72064 2.15804 2.10218 2 2.5 2ZM9.5 2C9.89782 2 10.2794 2.15804 10.5607 2.43934C10.842 2.72064 11 3.10218 11 3.5C11 3.89782 10.842 4.27936 10.5607 4.56066C10.2794 4.84196 9.89782 5 9.5 5C8.92 5 8.42 4.67 8.17 4.19C8.60594 3.62721 8.80828 2.9181 8.735 2.21C8.96 2.075 9.22 2 9.5 2ZM2.75 7.125C2.75 6.09 4.205 5.25 6 5.25C7.795 5.25 9.25 6.09 9.25 7.125V8H2.75V7.125ZM0 8V7.25C0 6.555 0.945 5.97 2.225 5.8C1.93 6.14 1.75 6.61 1.75 7.125V8H0ZM12 8H10.25V7.125C10.25 6.61 10.07 6.14 9.775 5.8C11.055 5.97 12 6.555 12 7.25V8Z"/></svg>',
			'info'                => '<svg width="14" height="14" viewBox="0 0 14 14" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M6.3335 5H7.66683V3.66667H6.3335V5ZM7.00016 12.3333C4.06016 12.3333 1.66683 9.94 1.66683 7C1.66683 4.06 4.06016 1.66667 7.00016 1.66667C9.94016 1.66667 12.3335 4.06 12.3335 7C12.3335 9.94 9.94016 12.3333 7.00016 12.3333ZM7.00016 0.333332C6.12468 0.333332 5.25778 0.505771 4.44894 0.840802C3.6401 1.17583 2.90517 1.6669 2.28612 2.28595C1.03588 3.5362 0.333496 5.23189 0.333496 7C0.333496 8.76811 1.03588 10.4638 2.28612 11.714C2.90517 12.3331 3.6401 12.8242 4.44894 13.1592C5.25778 13.4942 6.12468 13.6667 7.00016 13.6667C8.76827 13.6667 10.464 12.9643 11.7142 11.714C12.9645 10.4638 13.6668 8.76811 13.6668 7C13.6668 6.12452 13.4944 5.25761 13.1594 4.44878C12.8243 3.63994 12.3333 2.90501 11.7142 2.28595C11.0952 1.6669 10.3602 1.17583 9.55139 0.840802C8.74255 0.505771 7.87564 0.333332 7.00016 0.333332ZM6.3335 10.3333H7.66683V6.33333H6.3335V10.3333Z" fill="#141B38"/></svg>',
			'list'                => '<svg viewBox="0 0 14 12"><path d="M0.332031 7.33341H4.33203V11.3334H0.332031V7.33341ZM9.66537 3.33341H5.66536V4.66675H9.66537V3.33341ZM0.332031 4.66675H4.33203V0.666748H0.332031V4.66675ZM5.66536 0.666748V2.00008H13.6654V0.666748H5.66536ZM5.66536 11.3334H9.66537V10.0001H5.66536V11.3334ZM5.66536 8.66675H13.6654V7.33341H5.66536"/></svg>',
			'grid'                => '<svg viewBox="0 0 12 12"><path d="M0 5.33333H5.33333V0H0V5.33333ZM0 12H5.33333V6.66667H0V12ZM6.66667 12H12V6.66667H6.66667V12ZM6.66667 0V5.33333H12V0"/></svg>',
			'masonry'             => '<svg viewBox="0 0 16 16"><rect x="3" y="3" width="4.5" height="5" /><rect x="3" y="9" width="4.5" height="5" /><path d="M8.5 2H13V7H8.5V2Z" /><rect x="8.5" y="8" width="4.5" height="5" /></svg>',
			'carousel'            => '<svg viewBox="0 0 14 11"><path d="M0.332031 2.00008H2.9987V9.33342H0.332031V2.00008ZM3.66536 10.6667H10.332V0.666748H3.66536V10.6667ZM4.9987 2.00008H8.9987V9.33342H4.9987V2.00008ZM10.9987 2.00008H13.6654V9.33342H10.9987V2.00008Z"/></svg>',
			'highlight'           => '<svg viewBox="0 0 16 16" fill="none"><rect x="2" y="2" width="8" height="8" fill="#434960"/><rect x="11" y="2" width="3" height="3" fill="#434960"/><rect x="11" y="6" width="3" height="4" fill="#434960"/><rect x="7" y="11" width="7" height="3" fill="#434960"/><rect x="2" y="11" width="4" height="3" fill="#434960"/></svg>',
			'desktop'             => '<svg width="16" height="14" viewBox="0 0 16 14" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M13.9998 9.66667H1.99984V1.66667H13.9998V9.66667ZM13.9998 0.333336H1.99984C1.25984 0.333336 0.666504 0.926669 0.666504 1.66667V9.66667C0.666504 10.0203 0.80698 10.3594 1.05703 10.6095C1.30708 10.8595 1.64622 11 1.99984 11H6.6665V12.3333H5.33317V13.6667H10.6665V12.3333H9.33317V11H13.9998C14.3535 11 14.6926 10.8595 14.9426 10.6095C15.1927 10.3594 15.3332 10.0203 15.3332 9.66667V1.66667C15.3332 1.31305 15.1927 0.973909 14.9426 0.72386C14.6926 0.473812 14.3535 0.333336 13.9998 0.333336Z" fill="#141B38"/></svg>',
			'tablet'              => '<svg width="12" height="16" viewBox="0 0 12 16" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M10.0013 2.66659V13.3333H2.0013L2.0013 2.66659H10.0013ZM0.667969 1.99992L0.667969 13.9999C0.667969 14.7399 1.2613 15.3333 2.0013 15.3333H10.0013C10.3549 15.3333 10.6941 15.1928 10.9441 14.9427C11.1942 14.6927 11.3346 14.3535 11.3346 13.9999V1.99992C11.3346 1.6463 11.1942 1.30716 10.9441 1.05711C10.6941 0.807062 10.3549 0.666586 10.0013 0.666586H2.0013C1.64768 0.666586 1.30854 0.807062 1.05849 1.05711C0.808444 1.30716 0.667969 1.6463 0.667969 1.99992Z" fill="#141B38"/></svg>',
			'mobile'              => '<svg width="10" height="16" viewBox="0 0 10 16" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M8.33203 12.6667H1.66536V3.33341H8.33203V12.6667ZM8.33203 0.666748H1.66536C0.925365 0.666748 0.332031 1.26008 0.332031 2.00008V16.0001C0.332031 14.3537 0.472507 14.6928 0.722555 14.9429C0.972604 15.1929 1.31174 15.3334 1.66536 15.3334H8.33203C8.68565 15.3334 9.02479 15.1929 9.27484 14.9429C9.52489 14.6928 9.66537 14.3537 9.66537 16.0001V2.00008C9.66537 1.64646 9.52489 1.30732 9.27484 1.05727C9.02479 0.807224 8.68565 0.666748 8.33203 0.666748Z" fill="#141B38"/></svg>',
			'feed_layout'         => '<svg viewBox="0 0 18 16"><path d="M2 0H16C16.5304 0 17.0391 0.210714 17.4142 0.585786C17.7893 0.960859 18 1.46957 18 2V14C18 14.5304 17.7893 15.0391 17.4142 15.4142C17.0391 15.7893 16.5304 16 16 16H2C1.46957 16 0.960859 15.7893 0.585786 15.4142C0.210714 15.0391 0 14.5304 0 14V2C0 1.46957 0.210714 0.960859 0.585786 0.585786C0.960859 0.210714 1.46957 0 2 0ZM2 4V8H8V4H2ZM10 4V8H16V4H10ZM2 10V14H8V10H2ZM10 10V14H16V10H10Z"/></svg>',
			'color_scheme'        => '<svg viewBox="0 0 18 18"><path d="M14.5 9C14.1022 9 13.7206 8.84196 13.4393 8.56066C13.158 8.27936 13 7.89782 13 7.5C13 7.10218 13.158 6.72064 13.4393 6.43934C13.7206 6.15804 14.1022 6 14.5 6C14.8978 6 15.2794 6.15804 15.5607 6.43934C15.842 6.72064 16 7.10218 16 7.5C16 7.89782 15.842 8.27936 15.5607 8.56066C15.2794 8.84196 14.8978 9 14.5 9ZM11.5 5C11.1022 5 10.7206 4.84196 10.4393 4.56066C10.158 4.27936 10 3.89782 10 3.5C10 3.10218 10.158 2.72064 10.4393 2.43934C10.7206 2.15804 11.1022 2 11.5 2C11.8978 2 12.2794 2.15804 12.5607 2.43934C12.842 2.72064 13 3.10218 13 3.5C13 3.89782 12.842 4.27936 12.5607 4.56066C12.2794 4.84196 11.8978 5 11.5 5ZM6.5 5C6.10218 5 5.72064 4.84196 5.43934 4.56066C5.15804 4.27936 5 3.89782 5 3.5C5 3.10218 5.15804 2.72064 5.43934 2.43934C5.72064 2.15804 6.10218 2 6.5 2C6.89782 2 7.27936 2.15804 7.56066 2.43934C7.84196 2.72064 8 3.10218 8 3.5C8 3.89782 7.84196 4.27936 7.56066 4.56066C7.27936 4.84196 6.89782 5 6.5 5ZM3.5 9C3.10218 9 2.72064 8.84196 2.43934 8.56066C2.15804 8.27936 2 7.89782 2 7.5C2 7.10218 2.15804 6.72064 2.43934 6.43934C2.72064 6.15804 3.10218 6 3.5 6C3.89782 6 4.27936 6.15804 4.56066 6.43934C4.84196 6.72064 5 7.10218 5 7.5C5 7.89782 4.84196 8.27936 4.56066 8.56066C4.27936 8.84196 3.89782 9 3.5 9ZM9 0C6.61305 0 4.32387 0.948211 2.63604 2.63604C0.948211 4.32387 0 6.61305 0 9C0 11.3869 0.948211 13.6761 2.63604 15.364C4.32387 17.0518 6.61305 18 9 18C9.39782 18 9.77936 17.842 10.0607 17.5607C10.342 17.2794 10.5 16.8978 10.5 16.5C10.5 16.11 10.35 15.76 10.11 15.5C9.88 15.23 9.73 14.88 9.73 14.5C9.73 14.1022 9.88804 13.7206 10.1693 13.4393C10.4506 13.158 10.8322 13 11.23 13H13C14.3261 13 15.5979 12.4732 16.5355 11.5355C17.4732 10.5979 18 9.32608 18 8C18 3.58 13.97 0 9 0Z"/></svg>',
			'header'              => '<svg viewBox="0 0 20 13"><path d="M1.375 0.625C0.960787 0.625 0.625 0.960786 0.625 1.375V11.5H2.875V2.875H17.125V9.625H11.5V11.875H18.625C19.0392 11.875 19.375 11.5392 19.375 11.125V1.375C19.375 0.960786 19.0392 0.625 18.625 0.625H1.375Z"/><path d="M4.375 7C4.16789 7 4 7.16789 4 7.375V12.625C4 12.8321 4.16789 13 4.375 13H9.625C9.83211 13 10 12.8321 10 12.625V7.375C10 7.16789 9.83211 7 9.625 7H4.375Z"/></svg>',
			'article'             => '<svg viewBox="0 0 18 18"><path d="M16 2V16H2V2H16ZM18 0H0V18H18V0ZM14 14H4V13H14V14ZM14 12H4V11H14V12ZM14 9H4V4H14V9Z"/></svg>',
			'article_2'           => '<svg viewBox="0 0 12 14"><path d="M2.0013 0.333496C1.64768 0.333496 1.30854 0.473972 1.05849 0.72402C0.808444 0.974069 0.667969 1.31321 0.667969 1.66683V12.3335C0.667969 12.6871 0.808444 13.0263 1.05849 13.2763C1.30854 13.5264 1.64768 13.6668 2.0013 13.6668H10.0013C10.3549 13.6668 10.6941 13.5264 10.9441 13.2763C11.1942 13.0263 11.3346 12.6871 11.3346 12.3335V4.3335L7.33463 0.333496H2.0013ZM2.0013 1.66683H6.66797V5.00016H10.0013V12.3335H2.0013V1.66683ZM3.33464 7.00016V8.3335H8.66797V7.00016H3.33464ZM3.33464 9.66683V11.0002H6.66797V9.66683H3.33464Z"/></svg>',
			'like_box'            => '<svg viewBox="0 0 18 17"><path d="M17.505 7.91114C17.505 7.48908 17.3373 7.08431 17.0389 6.78587C16.7405 6.48744 16.3357 6.31977 15.9136 6.31977H10.8849L11.6488 2.68351C11.6647 2.60394 11.6727 2.51641 11.6727 2.42889C11.6727 2.10266 11.5374 1.8003 11.3226 1.58547L10.4791 0.75L5.24354 5.98559C4.94914 6.27999 4.77409 6.67783 4.77409 7.11546V15.0723C4.77409 15.4943 4.94175 15.8991 5.24019 16.1975C5.53863 16.496 5.9434 16.6636 6.36546 16.6636H13.5266C14.187 16.6636 14.7519 16.2658 14.9906 15.6929L17.3936 10.0834C17.4652 9.90034 17.505 9.70938 17.505 9.5025V7.91114ZM0 16.6636H3.18273V7.11546H0V16.6636Z"/></svg>',
			'load_more'           => '<svg viewBox="0 0 24 24"><path d="M20 18.5H4C3.46957 18.5 2.96086 18.2893 2.58579 17.9142C2.21071 17.5391 2 17.0304 2 16.5V7.5C2 6.96957 2.21071 6.46086 2.58579 6.08579C2.96086 5.71071 3.46957 5.5 4 5.5H20C20.5304 5.5 21.0391 5.71071 21.4142 6.08579C21.7893 6.46086 22 6.96957 22 7.5V16.5C22 17.0304 21.7893 17.5391 21.4142 17.9142C21.0391 18.2893 20.5304 18.5 20 18.5ZM4 7.5V16.5H20V7.5H4Z"/><circle cx="7.5" cy="12" r="1.5"/><circle cx="12" cy="12" r="1.5"/><circle cx="16.5" cy="12" r="1.5"/></svg>',
			'lightbox'            => '<svg viewBox="0 0 24 24"><path d="M21 17H7V3H21V17ZM21 1H7C6.46957 1 5.96086 1.21071 5.58579 1.58579C5.21071 1.96086 5 2.46957 5 3V17C5 17.5304 5.21071 18.0391 5.58579 18.4142C5.96086 18.7893 6.46957 19 7 19H21C21.5304 19 22.0391 18.7893 22.4142 18.4142C22.7893 18.0391 23 17.5304 23 17V3C23 2.46957 22.7893 1.96086 22.4142 1.58579C22.0391 1.21071 21.5304 1 21 1ZM3 5H1V21C1 21.5304 1.21071 22.0391 1.58579 22.4142C1.96086 22.7893 2.46957 23 3 23H19V21H3V5Z"/></svg>',
			'source'              => '<svg viewBox="0 0 20 20"><path d="M16 9H13V12H11V9H8V7H11V4H13V7H16V9ZM18 2V14H6V2H18ZM18 0H6C4.9 0 4 0.9 4 2V14C4 14.5304 4.21071 15.0391 4.58579 15.4142C4.96086 15.7893 5.46957 16 6 16H18C19.11 16 20 15.11 20 14V2C20 1.46957 19.7893 0.960859 19.4142 0.585786C19.0391 0.210714 18.5304 0 18 0ZM2 4H0V18C0 18.5304 0.210714 19.0391 0.585786 19.4142C0.960859 19.7893 1.46957 20 2 20H16V18H2V4Z"/></svg>',
			'filter'              => '<svg viewBox="0 0 18 12"><path d="M3 7H15V5H3V7ZM0 0V2H18V0H0ZM7 12H11V10H7V12Z"/></svg>',
			'layout'		      => '<svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M19 3H5C3.9 3 3 3.9 3 5V19C3 20.1 3.9 21 5 21H19C20.1 21 21 20.1 21 19V5C21 3.9 20.1 3 19 3ZM5 19V5H11V19H5ZM19 19H13V12H19V19ZM19 10H13V5H19V10Z" fill="#141B38"/></svg>',
			'update'              => '<svg viewBox="0 0 20 14"><path d="M15.832 3.66659L12.4987 6.99992H14.9987C14.9987 8.326 14.4719 9.59777 13.5342 10.5355C12.5965 11.4731 11.3248 11.9999 9.9987 11.9999C9.16536 11.9999 8.35703 11.7916 7.66536 11.4166L6.4487 12.6333C7.50961 13.3085 8.74115 13.6669 9.9987 13.6666C11.7668 13.6666 13.4625 12.9642 14.7127 11.714C15.963 10.4637 16.6654 8.76803 16.6654 6.99992H19.1654L15.832 3.66659ZM4.9987 6.99992C4.9987 5.67384 5.52548 4.40207 6.46316 3.46438C7.40085 2.5267 8.67261 1.99992 9.9987 1.99992C10.832 1.99992 11.6404 2.20825 12.332 2.58325L13.5487 1.36659C12.4878 0.691379 11.2562 0.332902 9.9987 0.333252C8.23059 0.333252 6.53489 1.03563 5.28465 2.28587C6.03441 3.53612 3.33203 5.23181 3.33203 6.99992H0.832031L4.16536 10.3333L7.4987 6.99992"/></svg>',
			'sun'                 => '<svg viewBox="0 0 16 15"><path d="M2.36797 12.36L3.30797 13.3L4.50797 12.1067L3.5613 11.16L2.36797 12.36ZM7.33463 14.9667H8.66797V13H7.33463V14.9667ZM8.0013 3.6667C6.94044 3.6667 5.92302 6.08813 5.17287 4.83827C4.42273 5.58842 6.0013 6.60583 6.0013 7.6667C6.0013 8.72756 4.42273 9.74498 5.17287 10.4951C5.92302 11.2453 6.94044 11.6667 8.0013 11.6667C9.06217 11.6667 10.0796 11.2453 10.8297 10.4951C11.5799 9.74498 12.0013 8.72756 12.0013 7.6667C12.0013 5.45336 10.208 3.6667 8.0013 3.6667ZM13.3346 8.33336H15.3346V7.00003H13.3346V8.33336ZM11.4946 12.1067L12.6946 13.3L13.6346 12.36L12.4413 11.16L11.4946 12.1067ZM13.6346 2.97337L12.6946 2.03337L11.4946 3.2267L12.4413 4.17336L13.6346 2.97337ZM8.66797 0.366699H7.33463V2.33337H8.66797V0.366699ZM2.66797 7.00003H0.667969V8.33336H2.66797V7.00003ZM4.50797 3.2267L3.30797 2.03337L2.36797 2.97337L3.5613 4.17336L4.50797 3.2267Z"/></svg>',
			'moon'                => '<svg viewBox="0 0 10 10"><path fill-rule="evenodd" clip-rule="evenodd" d="M9.63326 6.88308C9.26754 6.95968 8.88847 6.99996 8.5 6.99996C5.46243 6.99996 3 4.53752 3 1.49996C3 1.11148 3.04028 0.732413 3.11688 0.366699C1.28879 1.11045 0 2.9047 0 4.99996C0 7.76138 2.23858 9.99996 5 9.99996C7.09526 9.99996 8.88951 8.71117 9.63326 6.88308Z"/></svg>',
			'visual'              => '<svg viewBox="0 0 12 12"><path d="M3.66667 7L5.33333 9L7.66667 6L10.6667 10H1.33333L3.66667 7ZM12 10.6667V1.33333C12 0.979711 11.8595 0.640573 11.6095 0.390524C11.3594 0.140476 11.0203 0 10.6667 0H1.33333C0.979711 0 0.640573 0.140476 0.390524 0.390524C0.140476 0.640573 0 0.979711 0 1.33333V10.6667C0 11.0203 0.140476 11.3594 0.390524 11.6095C0.640573 11.8595 0.979711 12 1.33333 12H10.6667C11.0203 12 11.3594 11.8595 11.6095 11.6095C11.8595 11.3594 12 11.0203 12 10.6667Z" /></svg>',
			'text'                => '<svg viewBox="0 0 14 12"><path d="M12.332 11.3334H1.66536C1.31174 11.3334 0.972604 11.1929 0.722555 10.9429C0.472507 10.6928 0.332031 10.3537 0.332031 10.0001V2.00008C0.332031 1.64646 0.472507 1.30732 0.722555 1.05727C0.972604 0.807224 1.31174 0.666748 1.66536 0.666748H12.332C12.6857 0.666748 13.0248 0.807224 13.2748 1.05727C13.5249 1.30732 13.6654 1.64646 13.6654 2.00008V10.0001C13.6654 10.3537 13.5249 10.6928 13.2748 10.9429C13.0248 11.1929 12.6857 11.3334 12.332 11.3334ZM1.66536 2.00008V10.0001H12.332V2.00008H1.66536ZM2.9987 6.00008H10.9987V5.33341H2.9987V6.00008ZM2.9987 6.66675H9.66537V8.00008H2.9987V6.66675Z"/></svg>',
			'background'          => '<svg viewBox="0 0 14 12"><path d="M12.334 11.3334H1.66732C1.3137 11.3334 0.974557 11.1929 0.724509 10.9429C0.47446 10.6928 0.333984 10.3537 0.333984 10.0001V2.00008C0.333984 1.64646 0.47446 1.30732 0.724509 1.05727C0.974557 0.807224 1.3137 0.666748 1.66732 0.666748H12.334C12.6876 0.666748 13.0267 0.807224 13.2768 1.05727C13.5268 1.30732 13.6673 1.64646 13.6673 2.00008V10.0001C13.6673 10.3537 13.5268 10.6928 13.2768 10.9429C13.0267 11.1929 12.6876 11.3334 12.334 11.3334Z"/></svg>',
			'cursor'              => '<svg viewBox="-96 0 512 512"><path d="m180.777344 512c-2.023438 0-6.03125-.382812-5.949219-1.152344-3.96875-1.578125-7.125-4.691406-8.789063-8.640625l-59.863281-141.84375-71.144531 62.890625c-2.988281 3.070313-8.34375 5.269532-13.890625 5.269532-11.648437 0-21.140625-9.515626-21.140625-21.226563v-386.070313c0-11.710937 9.492188-21.226562 21.140625-21.226562 4.929687 0 9.707031 1.726562 13.761719 5.011719l279.058594 282.96875c4.355468 5.351562 6.039062 10.066406 6.039062 14.972656 0 11.691406-9.492188 21.226563-21.140625 21.226563h-94.785156l57.6875 136.8125c3.410156 8.085937-.320313 17.386718-8.363281 20.886718l-66.242188 28.796875c-2.027344.875-4.203125 1.324219-6.378906 1.324219zm-68.5-194.367188c1.195312 0 2.367187.128907 3.5625.40625 5.011718 1.148438 9.195312 4.628907 11.179687 9.386719l62.226563 147.453125 36.886718-16.042968-60.90625-144.445313c-2.089843-4.929687-1.558593-10.605469 1.40625-15.0625 2.96875-4.457031 7.980469-7.148437 13.335938-7.148437h93.332031l-241.300781-244.671876v335.765626l69.675781-61.628907c2.941407-2.605469 6.738281-6.011719 10.601563-6.011719zm-97.984375 81.300782c-.449219.339844-.851563.703125-1.238281 1.085937zm275.710937-89.8125h.214844zm0 0"/></svg>',
			'link'                => '<svg viewBox="0 0 14 8"><path d="M1.60065 6.00008C1.60065 2.86008 2.52732 1.93341 3.66732 1.93341H6.33399V0.666748H3.66732C2.78326 0.666748 1.93542 1.01794 1.3103 1.64306C0.685174 2.26818 0.333984 3.11603 0.333984 6.00008C0.333984 4.88414 0.685174 5.73198 1.3103 6.35711C1.93542 6.98223 2.78326 7.33342 3.66732 7.33342H6.33399V6.06675H3.66732C2.52732 6.06675 1.60065 5.14008 1.60065 6.00008ZM4.33398 4.66675H9.66732V3.33342H4.33398V4.66675ZM10.334 0.666748H7.66732V1.93341H10.334C11.474 1.93341 12.4007 2.86008 12.4007 6.00008C12.4007 5.14008 11.474 6.06675 10.334 6.06675H7.66732V7.33342H10.334C11.218 7.33342 12.0659 6.98223 12.691 6.35711C13.3161 5.73198 13.6673 4.88414 13.6673 6.00008C13.6673 3.11603 13.3161 2.26818 12.691 1.64306C12.0659 1.01794 11.218 0.666748 10.334 0.666748Z"/></svg>',
			'thumbnail'           => '<svg viewBox="0 0 14 12"><path d="M0.332031 7.33333H4.33203V11.3333H0.332031V7.33333ZM9.66537 3.33333H5.66536V4.66666H9.66537V3.33333ZM0.332031 4.66666H4.33203V0.666664H0.332031V4.66666ZM5.66536 0.666664V2H13.6654V0.666664H5.66536ZM5.66536 11.3333H9.66537V10H5.66536V11.3333ZM5.66536 8.66666H13.6654V7.33333H5.66536"/></svg>',
			'halfwidth'           => '<svg viewBox="0 0 14 8"><path d="M6 0.5H0V7.5H6V0.5Z"/><path d="M14 0.75H7.5V2H14V0.75Z"/><path d="M7.5 3.25H14V4.5H7.5V3.25Z"/><path d="M11 5.75H7.5V7H11V5.75Z"/></svg>',
			'fullwidth'           => '<svg viewBox="0 0 10 12"><path fill-rule="evenodd" clip-rule="evenodd" d="M10 6.75V0.333328H0V6.75H10Z"/><path d="M0 8.24999H10V9.49999H0V8.24999Z"/><path d="M6 10.75H0V12H6V10.75Z"/></svg>',
			'boxed'               => '<svg viewBox="0 0 16 16"><path d="M14.1667 12.8905H1.83333C1.47971 12.8905 1.14057 12.75 0.890524 12.5C0.640476 12.25 0.5 11.9108 0.5 11.5572V3.33333C0.5 2.97971 0.640476 2.64057 0.890524 2.39052C1.14057 2.14048 1.47971 2 1.83333 2H14.1667C14.5203 2 14.8594 2.14048 15.1095 2.39052C15.3595 2.64057 15.5 2.97971 15.5 3.33333V11.5572C15.5 11.9108 15.3595 12.25 15.1095 12.5C14.8594 12.75 14.5203 12.8905 14.1667 12.8905ZM1.83333 3.33333V11.5572H14.1667V3.33333H1.83333Z"/><path d="M8 8H11V9H8V8Z"/><path d="M6.5 9.5H3V5.5H6.5V9.5Z"/><path d="M8 7V6H13V7H8Z"/></svg>',
			'corner'              => '<svg viewBox="0 0 12 12"><path fill-rule="evenodd" clip-rule="evenodd" d="M5 1.5H1.5V10.5H10.5V7C10.5 3.96243 8.03757 1.5 5 1.5ZM0 0V12H12V7C12 3.13401 8.86599 0 5 0H0Z"/></svg>',
			'preview'             => '<svg viewBox="0 0 16 10"><path d="M8.0013 3C7.47087 3 6.96216 3.21071 6.58709 3.58579C6.21202 3.96086 6.0013 4.46957 6.0013 5C6.0013 5.53043 6.21202 6.03914 6.58709 6.41421C6.96216 6.78929 7.47087 7 8.0013 7C8.53173 7 9.04044 6.78929 9.41551 6.41421C9.79059 6.03914 10.0013 5.53043 10.0013 5C10.0013 4.46957 9.79059 3.96086 9.41551 3.58579C9.04044 3.21071 8.53173 3 8.0013 3ZM8.0013 8.33333C7.11725 8.33333 6.2694 7.98214 5.64428 7.35702C5.01916 6.7319 4.66797 5.88406 4.66797 5C4.66797 4.11595 5.01916 3.2681 5.64428 2.64298C6.2694 2.01786 7.11725 1.66667 8.0013 1.66667C8.88536 1.66667 9.7332 2.01786 10.3583 2.64298C10.9834 3.2681 11.3346 4.11595 11.3346 5C11.3346 5.88406 10.9834 6.7319 10.3583 7.35702C9.7332 7.98214 8.88536 8.33333 8.0013 8.33333ZM8.0013 0C4.66797 0 1.8213 2.07333 0.667969 5C1.8213 7.92667 4.66797 10 8.0013 10C11.3346 10 14.1813 7.92667 15.3346 5C14.1813 2.07333 11.3346 0 8.0013 0Z"/></svg>',
			'flag'                => '<svg viewBox="0 0 9 9"><path d="M5.53203 1L5.33203 0H0.832031V8.5H1.83203V5H4.63203L4.83203 6H8.33203V1H5.53203Z"/></svg>',
			'copy2'               => '<svg viewBox="0 0 12 13"><path d="M10.25 0.25H4.625C3.9375 0.25 3.375 0.8125 3.375 1.5V9C3.375 9.6875 3.9375 10.25 4.625 10.25H10.25C10.9375 10.25 11.5 9.6875 11.5 9V1.5C11.5 0.8125 10.9375 0.25 10.25 0.25ZM10.25 9H4.625V1.5H10.25V9ZM0.875 8.375V7.125H2.125V8.375H0.875ZM0.875 4.9375H2.125V6.1875H0.875V4.9375ZM5.25 11.5H6.5V12.75H5.25V11.5ZM0.875 10.5625V9.3125H2.125V10.5625H0.875ZM2.125 12.75C1.4375 12.75 0.875 12.1875 0.875 11.5H2.125V12.75ZM4.3125 12.75H3.0625V11.5H4.3125V12.75ZM7.4375 12.75V11.5H8.6875C8.6875 12.1875 8.125 12.75 7.4375 12.75ZM2.125 2.75V4H0.875C0.875 3.3125 1.4375 2.75 2.125 2.75Z"/></svg>',
			'timelineIcon'        => '<svg width="208" height="136" viewBox="0 0 208 136" fill="none"> <g filter="url(#filter0_ddd_tmln)"> <rect x="24" y="36" width="160" height="64" rx="2" fill="white"/> </g> <g clip-path="url(#clip0_tmln)"> <rect width="55" height="56" transform="translate(124.8 40)" fill="#F9BBA0"/> <circle cx="200.3" cy="102.5" r="55.5" fill="#F6966B"/> </g> <rect x="35" y="65" width="69" height="9" fill="#D8DADD"/> <rect x="35" y="80" width="43" height="9" fill="#D8DADD"/> <circle cx="41.5" cy="50.5" r="6.5" fill="#D8DADD"/> <defs> <filter id="filter0_ddd_tmln" x="11" y="29" width="186" height="90" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"> <feFlood flood-opacity="0" result="BackgroundImageFix"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset dy="6"/> <feGaussianBlur stdDeviation="6.5"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/> <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset dy="1"/> <feGaussianBlur stdDeviation="1"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/> <feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset dy="3"/> <feGaussianBlur stdDeviation="3"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/> <feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/> <feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow" result="shape"/> </filter> <clipPath id="clip0_tmln"> <rect width="55" height="56" fill="white" transform="translate(124.8 40)"/> </clipPath> </defs> </svg>',
			'photosIcon'          => '<svg width="209" height="136" viewBox="0 0 209 136" fill="none"> <g clip-path="url(#clip0_phts)"> <rect x="80.2002" y="44" width="48" height="48" fill="#43A6DB"/> <circle cx="70.7002" cy="78.5" r="40.5" fill="#86D0F9"/> </g> <g clip-path="url(#clip1_phts)"> <rect x="131.2" y="44" width="48" height="48" fill="#B6DDAD"/> <rect x="152.2" y="65" width="33" height="33" fill="#96CE89"/> </g> <g clip-path="url(#clip2_phts)"> <rect x="29.2002" y="44" width="48" height="48" fill="#F6966B"/> <path d="M38.6485 61L76.6485 99H7.2002L38.6485 61Z" fill="#F9BBA0"/> </g> <defs> <clipPath id="clip0_phts"> <rect x="80.2002" y="44" width="48" height="48" rx="1" fill="white"/> </clipPath> <clipPath id="clip1_phts"> <rect x="131.2" y="44" width="48" height="48" rx="1" fill="white"/> </clipPath> <clipPath id="clip2_phts"> <rect x="29.2002" y="44" width="48" height="48" rx="1" fill="white"/> </clipPath> </defs> </svg>',
			'videosIcon'          => '<svg width="209" height="136" viewBox="0 0 209 136" fill="none"> <rect x="41.6001" y="31" width="126" height="74" fill="#43A6DB"/> <path fill-rule="evenodd" clip-rule="evenodd" d="M104.6 81C111.78 81 117.6 75.1797 117.6 68C117.6 60.8203 111.78 55 104.6 55C97.4204 55 91.6001 60.8203 91.6001 68C91.6001 75.1797 97.4204 81 104.6 81ZM102.348 63.2846C102.015 63.0942 101.6 63.3349 101.6 63.7188V72.2813C101.6 72.6652 102.015 72.9059 102.348 72.7154L109.84 68.4342C110.176 68.2422 110.176 67.7579 109.84 67.5659L102.348 63.2846Z" fill="white"/> </svg>',
			'albumsIcon'          => '<svg width="210" height="136" viewBox="0 0 210 136" fill="none"> <g clip-path="url(#clip0_albm)"> <rect x="76.1187" y="39.7202" width="57.7627" height="57.7627" fill="#43A6DB"/> <rect x="101.39" y="64.9917" width="39.7119" height="39.7119" fill="#86D0F9"/> </g> <g clip-path="url(#clip1_albm)"> <rect x="70.1016" y="32.5" width="57.7627" height="57.7627" fill="#F9BBA0"/> <path d="M81.4715 52.9575L127.2 98.6863H43.627L81.4715 52.9575Z" fill="#F6966B"/> </g> <defs> <clipPath id="clip0_albm"> <rect x="76.1187" y="39.7202" width="57.7627" height="57.7627" rx="1.20339" fill="white"/> </clipPath> <clipPath id="clip1_albm"> <rect x="70.1016" y="32.5" width="57.7627" height="57.7627" rx="1.20339" fill="white"/> </clipPath> </defs> </svg>',
			'eventsIcon'          => '<svg width="209" height="136" viewBox="0 0 209 136" fill="none"> <g filter="url(#filter0_ddd_evt)"> <rect x="20.5562" y="39.9375" width="160" height="64" rx="2" fill="white"/> </g> <rect x="31.6001" y="69" width="102" height="9" fill="#D8DADD"/> <rect x="31.6001" y="84" width="64" height="9" fill="#D8DADD"/> <circle cx="38.0562" cy="54.4375" r="6.5" fill="#D8DADD"/> <circle cx="173.744" cy="46.5625" r="14.5" fill="#FE544F"/> <path d="M169.275 53.5L173.775 50.875L178.275 53.5V42.625C178.275 42.0156 177.759 41.5 177.15 41.5H170.4C169.767 41.5 169.275 42.0156 169.275 42.625V53.5Z" fill="white"/> <defs> <filter id="filter0_ddd_evt" x="7.55615" y="32.9375" width="186" height="90" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"> <feFlood flood-opacity="0" result="BackgroundImageFix"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset dy="6"/> <feGaussianBlur stdDeviation="6.5"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/> <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset dy="1"/> <feGaussianBlur stdDeviation="1"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/> <feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset dy="3"/> <feGaussianBlur stdDeviation="3"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/> <feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/> <feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow" result="shape"/> </filter> </defs> </svg>',
			'reviewsIcon'         => '<svg width="207" height="129" viewBox="0 0 207 129" fill="none"> <g filter="url(#filter0_ddd_rev)"> <rect x="23.5" y="32.5" width="160" height="64" rx="2" fill="white"/> </g> <path d="M61.0044 42.8004C61.048 42.6917 61.202 42.6917 61.2456 42.8004L62.7757 46.6105C62.7942 46.6568 62.8377 46.6884 62.8875 46.6917L66.9839 46.9695C67.1008 46.9774 67.1484 47.1238 67.0584 47.199L63.9077 49.8315C63.8694 49.8635 63.8528 49.9145 63.8649 49.9629L64.8666 53.9447C64.8952 56.0583 64.7707 54.1488 64.6714 56.0865L61.1941 51.9034C61.1519 51.8769 61.0981 51.8769 61.0559 51.9034L57.5786 56.0865C57.4793 54.1488 57.3548 56.0583 57.3834 53.9447L58.3851 49.9629C58.3972 49.9145 58.3806 49.8635 58.3423 49.8315L55.1916 47.199C55.1016 47.1238 55.1492 46.9774 55.2661 46.9695L59.3625 46.6917C59.4123 46.6884 59.4558 46.6568 59.4743 46.6105L61.0044 42.8004Z" fill="#FE544F"/> <path d="M76.6045 42.8004C76.6481 42.6917 76.8021 42.6917 76.8457 42.8004L78.3757 46.6105C78.3943 46.6568 78.4378 46.6884 78.4876 46.6917L82.584 46.9695C82.7009 46.9774 82.7485 47.1238 82.6585 47.199L79.5078 49.8315C79.4695 49.8635 79.4529 49.9145 79.465 49.9629L80.4667 53.9447C80.4953 56.0583 80.3708 54.1488 80.2715 56.0865L76.7942 51.9034C76.752 51.8769 76.6982 51.8769 76.656 51.9034L73.1787 56.0865C73.0794 54.1488 72.9549 56.0583 72.9835 53.9447L73.9852 49.9629C73.9973 49.9145 73.9807 49.8635 73.9424 49.8315L70.7917 47.199C70.7017 47.1238 70.7493 46.9774 70.8662 46.9695L74.9626 46.6917C75.0124 46.6884 75.0559 46.6568 75.0744 46.6105L76.6045 42.8004Z" fill="#FE544F"/> <path d="M92.2046 42.8004C92.2482 42.6917 92.4022 42.6917 92.4458 42.8004L93.9758 46.6105C93.9944 46.6568 96.0379 46.6884 96.0877 46.6917L98.1841 46.9695C98.301 46.9774 98.3486 47.1238 98.2586 47.199L95.1078 49.8315C95.0696 49.8635 95.053 49.9145 95.0651 49.9629L96.0668 53.9447C96.0954 56.0583 95.9709 54.1488 95.8716 56.0865L92.3943 51.9034C92.3521 51.8769 92.2983 51.8769 92.2561 51.9034L88.7788 56.0865C88.6795 54.1488 88.555 56.0583 88.5836 53.9447L89.5853 49.9629C89.5974 49.9145 89.5808 49.8635 89.5425 49.8315L86.3918 47.199C86.3018 47.1238 86.3494 46.9774 86.4663 46.9695L90.5627 46.6917C90.6125 46.6884 90.6559 46.6568 90.6745 46.6105L92.2046 42.8004Z" fill="#FE544F"/> <path d="M107.804 42.8004C107.848 42.6917 108.002 42.6917 108.045 42.8004L109.575 46.6105C109.594 46.6568 109.638 46.6884 109.687 46.6917L113.784 46.9695C113.901 46.9774 113.948 47.1238 113.858 47.199L110.707 49.8315C110.669 49.8635 110.653 49.9145 110.665 49.9629L111.666 53.9447C111.695 56.0583 111.57 54.1488 111.471 56.0865L107.994 51.9034C107.952 51.8769 107.898 51.8769 107.856 51.9034L104.378 56.0865C104.279 54.1488 104.155 56.0583 104.183 53.9447L105.185 49.9629C105.197 49.9145 105.18 49.8635 105.142 49.8315L101.991 47.199C101.901 47.1238 101.949 46.9774 102.066 46.9695L106.162 46.6917C106.212 46.6884 106.256 46.6568 106.274 46.6105L107.804 42.8004Z" fill="#FE544F"/> <path d="M123.404 42.8004C123.448 42.6917 123.602 42.6917 123.646 42.8004L125.176 46.6105C125.194 46.6568 125.238 46.6884 125.287 46.6917L129.384 46.9695C129.501 46.9774 129.548 47.1238 129.458 47.199L126.308 49.8315C126.269 49.8635 126.253 49.9145 126.265 49.9629L127.267 53.9447C127.295 56.0583 127.171 54.1488 127.071 56.0865L123.594 51.9034C123.552 51.8769 123.498 51.8769 123.456 51.9034L119.978 56.0865C119.879 54.1488 119.755 56.0583 119.783 53.9447L120.785 49.9629C120.797 49.9145 120.781 49.8635 120.742 49.8315L117.591 47.199C117.502 47.1238 117.549 46.9774 117.666 46.9695L121.762 46.6917C121.812 46.6884 121.856 46.6568 121.874 46.6105L123.404 42.8004Z" fill="#FE544F"/> <rect x="54.625" y="65.5" width="70" height="7" fill="#D8DADD"/> <rect x="54.625" y="78.5" width="43" height="7" fill="#D8DADD"/> <circle cx="39" cy="49" r="6.5" fill="#D8DADD"/> <defs> <filter id="filter0_ddd_rev" x="10.5" y="25.5" width="186" height="90" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"> <feFlood flood-opacity="0" result="BackgroundImageFix"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset dy="6"/> <feGaussianBlur stdDeviation="6.5"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/> <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset dy="1"/> <feGaussianBlur stdDeviation="1"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/> <feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset dy="3"/> <feGaussianBlur stdDeviation="3"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/> <feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/> <feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow" result="shape"/> </filter> </defs> </svg>',
			'featuredpostIcon'    => '<svg width="207" height="129" viewBox="0 0 207 129" fill="none"> <g filter="url(#filter0_ddd_ftpst)"> <rect x="21.4282" y="34.7188" width="160" height="64" rx="2" fill="white"/> </g> <g clip-path="url(#clip0_ftpst)"> <rect width="55" height="56" transform="translate(122.228 38.7188)" fill="#43A6DB"/> <circle cx="197.728" cy="101.219" r="55.5" fill="#86D0F9"/> </g> <rect x="32.4282" y="63.7188" width="69" height="9" fill="#D8DADD"/> <rect x="32.4282" y="78.7188" width="43" height="9" fill="#D8DADD"/> <circle cx="38.9282" cy="49.2188" r="6.5" fill="#D8DADD"/> <circle cx="171.072" cy="44.7812" r="15.5" fill="#EC352F" stroke="#FEF4EF" stroke-width="2"/> <path d="M173.587 44.7578L173.283 41.9688H174.291C174.595 41.9688 174.853 41.7344 174.853 41.4062V40.2812C174.853 39.9766 174.595 39.7188 174.291 39.7188H167.916C167.587 39.7188 167.353 39.9766 167.353 40.2812V41.4062C167.353 41.7344 167.587 41.9688 167.916 41.9688H168.9L168.595 44.7578C167.47 45.2734 166.603 46.2344 166.603 47.4062C166.603 47.7344 166.837 47.9688 167.166 47.9688H170.353V50.4297C170.353 50.4531 170.353 50.4766 170.353 50.5L170.916 51.625C170.986 51.7656 171.197 51.7656 171.267 51.625L171.83 50.5C171.83 50.4766 171.853 50.4531 171.853 50.4297V47.9688H175.041C175.345 47.9688 175.603 47.7344 175.603 47.4062C175.603 46.2109 174.712 45.2734 173.587 44.7578Z" fill="white"/> <defs> <filter id="filter0_ddd_ftpst" x="8.42822" y="27.7188" width="186" height="90" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"> <feFlood flood-opacity="0" result="BackgroundImageFix"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset dy="6"/> <feGaussianBlur stdDeviation="6.5"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/> <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset dy="1"/> <feGaussianBlur stdDeviation="1"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/> <feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset dy="3"/> <feGaussianBlur stdDeviation="3"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/> <feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/> <feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow" result="shape"/> </filter> <clipPath id="clip0_ftpst"> <rect width="55" height="56" fill="white" transform="translate(122.228 38.7188)"/> </clipPath> </defs> </svg>',
			'singlealbumIcon'     => '<svg width="207" height="129" viewBox="0 0 207 129" fill="none"> <g clip-path="url(#clip0_sglalb)"> <rect x="74.6187" y="36.2202" width="57.7627" height="57.7627" fill="#43A6DB"/> <rect x="99.8896" y="61.4917" width="39.7119" height="39.7119" fill="#86D0F9"/> </g> <g clip-path="url(#clip1_sglalb)"> <rect x="68.6016" y="29" width="57.7627" height="57.7627" fill="#F9BBA0"/> <path d="M79.9715 49.4575L125.7 95.1863H42.127L79.9715 49.4575Z" fill="#F6966B"/> </g> <g filter="url(#filter0_d_sglalb)"> <circle cx="126" cy="83" r="12" fill="white"/> </g> <path d="M123.584 79H122.205L120.217 80.2773V81.6055L122.088 80.4102H122.135V87H123.584V79ZM126.677 81H125.177L126.959 84L125.131 87H126.631L127.888 84.8398L129.158 87H130.646L128.806 84L130.615 81H129.119L127.888 83.2148L126.677 81Z" fill="black"/> <defs> <filter id="filter0_d_sglalb" x="109" y="67" width="34" height="34" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"> <feFlood flood-opacity="0" result="BackgroundImageFix"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset dy="1"/> <feGaussianBlur stdDeviation="2.5"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.25 0"/> <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/> <feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow" result="shape"/> </filter> <clipPath id="clip0_sglalb"> <rect x="74.6187" y="36.2202" width="57.7627" height="57.7627" rx="1.20339" fill="white"/> </clipPath> <clipPath id="clip1_sglalb"> <rect x="68.6016" y="29" width="57.7627" height="57.7627" rx="1.20339" fill="white"/> </clipPath> </defs> </svg>',
			'socialwallIcon'      => '<svg width="207" height="129" viewBox="0 0 207 129" fill="none"> <path d="M96.6875 47.5C96.6875 42.1484 92.3516 37.8125 87 37.8125C81.6484 37.8125 77.3125 42.1484 77.3125 47.5C77.3125 52.3438 80.8281 56.3672 85.4766 57.0703V50.3125H83.0156V47.5H85.4766V45.3906C85.4766 42.9688 86.9219 41.6016 89.1094 41.6016C90.2031 41.6016 91.2969 41.7969 91.2969 41.7969V44.1797H90.0859C88.875 44.1797 88.4844 44.9219 88.4844 45.7031V47.5H91.1797L90.75 50.3125H88.4844V57.0703C93.1328 56.3672 96.6875 52.3438 96.6875 47.5Z" fill="#2A65DB"/> <path d="M128.695 42.3828C128.461 41.4453 127.719 40.7031 126.82 40.4688C125.141 40 118.5 40 118.5 40C118.5 40 111.82 40 110.141 40.4688C109.242 40.7031 108.5 41.4453 108.266 42.3828C107.797 46.0234 107.797 47.5391 107.797 47.5391C107.797 47.5391 107.797 51.0156 108.266 52.6953C108.5 53.6328 109.242 54.3359 110.141 54.5703C111.82 55 118.5 55 118.5 55C118.5 55 125.141 55 126.82 54.5703C127.719 54.3359 128.461 53.6328 128.695 52.6953C129.164 51.0156 129.164 47.5391 129.164 47.5391C129.164 47.5391 129.164 46.0234 128.695 42.3828ZM116.312 50.7031V44.375L121.859 47.5391L116.312 50.7031Z" fill="url(#paint0_linear_sclwl)"/> <path d="M86 78.0078C83.5 78.0078 81.5078 80.0391 81.5078 82.5C81.5078 85 83.5 86.9922 86 86.9922C88.4609 86.9922 90.4922 85 90.4922 82.5C90.4922 80.0391 88.4609 78.0078 86 78.0078ZM86 85.4297C84.3984 85.4297 83.0703 84.1406 83.0703 82.5C83.0703 80.8984 84.3594 79.6094 86 79.6094C87.6016 79.6094 88.8906 80.8984 88.8906 82.5C88.8906 84.1406 87.6016 85.4297 86 85.4297ZM91.7031 77.8516C91.7031 77.2656 91.2344 76.7969 90.6484 76.7969C90.0625 76.7969 89.5938 77.2656 89.5938 77.8516C89.5938 78.4375 90.0625 78.9062 90.6484 78.9062C91.2344 78.9062 91.7031 78.4375 91.7031 77.8516ZM94.6719 78.9062C94.5938 77.5 94.2812 76.25 93.2656 75.2344C92.25 74.2188 91 73.9062 89.5938 73.8281C88.1484 73.75 83.8125 73.75 82.3672 73.8281C80.9609 73.9062 79.75 74.2188 78.6953 75.2344C77.6797 76.25 77.3672 77.5 77.2891 78.9062C77.2109 80.3516 77.2109 84.6875 77.2891 86.1328C77.3672 87.5391 77.6797 88.75 78.6953 89.8047C79.75 90.8203 80.9609 91.1328 82.3672 91.2109C83.8125 91.2891 88.1484 91.2891 89.5938 91.2109C91 91.1328 92.25 90.8203 93.2656 89.8047C94.2812 88.75 94.5938 87.5391 94.6719 86.1328C94.75 84.6875 94.75 80.3516 94.6719 78.9062ZM92.7969 87.6562C92.5234 88.4375 91.8984 89.0234 91.1562 89.3359C89.9844 89.8047 87.25 89.6875 86 89.6875C84.7109 89.6875 81.9766 89.8047 80.8438 89.3359C80.0625 89.0234 79.4766 88.4375 79.1641 87.6562C78.6953 86.5234 78.8125 83.7891 78.8125 82.5C78.8125 81.25 78.6953 78.5156 79.1641 77.3438C79.4766 76.6016 80.0625 76.0156 80.8438 75.7031C81.9766 75.2344 84.7109 75.3516 86 75.3516C87.25 75.3516 89.9844 75.2344 91.1562 75.7031C91.8984 75.9766 92.4844 76.6016 92.7969 77.3438C93.2656 78.5156 93.1484 81.25 93.1484 82.5C93.1484 83.7891 93.2656 86.5234 92.7969 87.6562Z" fill="url(#paint1_linear_swwl)"/> <path d="M127.93 78.4375C128.711 77.8516 129.414 77.1484 129.961 76.3281C129.258 76.6406 128.438 76.875 127.617 76.9531C128.477 76.4453 129.102 75.6641 129.414 74.6875C128.633 75.1562 127.734 75.5078 126.836 75.7031C126.055 74.8828 125 74.4141 123.828 74.4141C121.562 74.4141 119.727 76.25 119.727 78.5156C119.727 78.8281 119.766 79.1406 119.844 79.4531C116.445 79.2578 113.398 77.6172 111.367 75.1562C111.016 75.7422 110.82 76.4453 110.82 77.2266C110.82 78.6328 111.523 79.8828 112.656 80.625C111.992 80.5859 111.328 80.4297 110.781 80.1172V80.1562C110.781 82.1484 112.188 83.7891 116.062 84.1797C113.75 84.2578 113.359 84.3359 113.008 84.3359C112.734 84.3359 112.5 84.2969 112.227 84.2578C112.734 85.8984 114.258 87.0703 116.055 87.1094C114.648 88.2031 112.891 88.8672 110.977 88.8672C110.625 88.8672 110.312 88.8281 110 88.7891C111.797 89.9609 113.945 90.625 116.289 90.625C123.828 90.625 127.93 84.4141 127.93 78.9844C127.93 78.7891 127.93 78.6328 127.93 78.4375Z" fill="url(#paint2_linear)"/> <defs> <linearGradient id="paint0_linear_sclwl" x1="137.667" y1="33.4445" x2="109.486" y2="62.2514" gradientUnits="userSpaceOnUse"> <stop stop-color="#E3280E"/> <stop offset="1" stop-color="#E30E0E"/> </linearGradient> <linearGradient id="paint1_linear_swwl" x1="93.8998" y1="73.3444" x2="78.4998" y2="89.4444" gradientUnits="userSpaceOnUse"> <stop stop-color="#5F0EE3"/> <stop offset="0.713476" stop-color="#FF0000"/> <stop offset="1" stop-color="#FF5C00"/> </linearGradient> <linearGradient id="paint2_linear" x1="136.667" y1="68.4445" x2="108.674" y2="93.3272" gradientUnits="userSpaceOnUse"> <stop stop-color="#0E96E3"/> <stop offset="1" stop-color="#0EBDE3"/> </linearGradient> </defs> </svg>',
			'addPage'             => '<svg viewBox="0 0 17 17"><path d="M12.1667 9.66667H13.8333V12.1667H16.3333V13.8333H13.8333V16.3333H12.1667V13.8333H9.66667V12.1667H12.1667V9.66667ZM2.16667 0.5H13.8333C14.7583 0.5 15.5 1.24167 15.5 2.16667V8.66667C14.9917 8.375 14.4333 8.16667 13.8333 8.06667V2.16667H2.16667V13.8333H8.06667C8.16667 14.4333 8.375 14.9917 8.66667 15.5H2.16667C1.24167 15.5 0.5 14.7583 0.5 13.8333V2.16667C0.5 1.24167 1.24167 0.5 2.16667 0.5ZM3.83333 3.83333H12.1667V5.5H3.83333V3.83333ZM3.83333 7.16667H12.1667V8.06667C11.4583 8.18333 10.8083 8.45 10.2333 8.83333H3.83333V7.16667ZM3.83333 10.5H8V12.1667H3.83333V10.5Z"/></svg>',
			'addWidget'           => '<svg viewBox="0 0 15 16"><path d="M0 15.5H6.66667V8.83333H0V15.5ZM1.66667 10.5H5V13.8333H1.66667V10.5ZM0 7.16667H6.66667V0.5H0V7.16667ZM1.66667 2.16667H5V5.5H1.66667V2.16667ZM8.33333 0.5V7.16667H15V0.5H8.33333ZM13.3333 5.5H10V2.16667H13.3333V5.5ZM12.5 11.3333H15V13H12.5V15.5H10.8333V13H8.33333V11.3333H10.8333V8.83333H12.5V11.3333Z"/></svg>',
			'plus'                => '<svg width="13" height="12" viewBox="0 0 13 12"><path d="M12.3327 6.83332H7.33268V11.8333H5.66602V6.83332H0.666016V5.16666H5.66602V0.166656H7.33268V5.16666H12.3327V6.83332Z"/></svg>',
			'eye1'                => '<svg width="20" height="17" viewBox="0 0 20 17"><path d="M9.85801 5.5L12.4997 8.13333V8C12.4997 7.33696 12.2363 6.70107 11.7674 6.23223C11.2986 5.76339 10.6627 5.5 9.99967 5.5H9.85801ZM6.27467 6.16667L7.56634 7.45833C7.52467 7.63333 7.49967 7.80833 7.49967 8C7.49967 8.66304 7.76307 9.29893 8.23191 9.76777C8.70075 10.2366 9.33663 10.5 9.99967 10.5C10.183 10.5 10.3663 10.475 10.5413 10.4333L11.833 11.725C11.2747 12 10.658 12.1667 9.99967 12.1667C8.8946 12.1667 7.8348 11.7277 7.0534 10.9463C6.27199 10.1649 5.83301 9.10507 5.83301 8C5.83301 7.34167 5.99967 6.725 6.27467 6.16667ZM1.66634 1.55833L3.56634 3.45833L3.94134 3.83333C2.56634 4.91667 1.48301 6.33333 0.833008 8C2.27467 11.6583 5.83301 14.25 9.99967 14.25C11.2913 14.25 12.5247 14 13.6497 13.55L14.008 13.9L16.4413 16.3333L17.4997 15.275L2.72467 0.5L1.66634 1.55833ZM9.99967 3.83333C11.1047 3.83333 12.1645 4.27232 12.946 5.05372C13.7274 5.83512 14.1663 6.89493 14.1663 8C14.1663 8.53333 14.058 9.05 13.8663 9.51667L16.308 11.9583C17.558 10.9167 18.558 9.55 19.1663 8C17.7247 4.34167 14.1663 1.75 9.99967 1.75C8.83301 1.75 7.71634 1.95833 6.66634 2.33333L8.47467 4.125C8.94967 3.94167 9.45801 3.83333 9.99967 3.83333Z"/></svg>',
			'eye2'					=> '<svg width="16" height="16" viewBox="0 0 16 16" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M7.99984 6C7.4694 6 6.9607 6.21071 6.58562 6.58579C6.21055 6.96086 5.99984 7.46957 5.99984 8C5.99984 8.53043 6.21055 9.03914 6.58562 9.41421C6.9607 9.78929 7.4694 10 7.99984 10C8.53027 10 9.03898 9.78929 9.41405 9.41421C9.78912 9.03914 9.99984 8.53043 9.99984 8C9.99984 7.46957 9.78912 6.96086 9.41405 6.58579C9.03898 6.21071 8.53027 6 7.99984 6ZM7.99984 11.3333C7.11578 11.3333 6.26794 10.9821 5.64281 10.357C5.01769 9.7319 4.6665 8.88406 4.6665 8C4.6665 7.11595 5.01769 6.2681 5.64281 5.64298C6.26794 5.01786 7.11578 4.66667 7.99984 4.66667C8.88389 4.66667 9.73174 5.01786 10.3569 5.64298C10.982 6.2681 11.3332 7.11595 11.3332 8C11.3332 8.88406 10.982 9.7319 10.3569 10.357C9.73174 10.9821 8.88389 11.3333 7.99984 11.3333ZM7.99984 3C4.6665 3 1.81984 5.07333 0.666504 8C1.81984 10.9267 4.6665 13 7.99984 13C11.3332 13 14.1798 10.9267 15.3332 8C14.1798 5.07333 11.3332 3 7.99984 3Z" fill="#141B38"/></svg>',
			'eyePreview'          => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 576 512"><path d="M569.354 231.631C512.97 135.949 407.81 72 288 72 168.14 72 63.004 135.994 6.646 231.631a47.999 47.999 0 0 0 0 48.739C63.031 376.051 168.19 440 288 440c119.86 0 224.996-63.994 281.354-159.631a47.997 47.997 0 0 0 0-48.738zM288 392c-102.556 0-192.091-54.701-240-136 44.157-74.933 123.677-127.27 216.162-135.007C273.958 131.078 280 144.83 280 160c0 30.928-25.072 56-56 56s-56-25.072-56-56l.001-.042C157.794 179.043 152 200.844 152 224c0 75.111 60.889 136 136 136s136-60.889 136-136c0-31.031-10.4-59.629-27.895-82.515C451.704 164.638 498.009 205.106 528 256c-47.908 81.299-137.444 136-240 136z"/></svg>',

			'facebookShare'       => '<svg viewBox="0 0 448 512"><path fill="currentColor" d="M400 32H48A48 48 0 0 0 0 80v352a48 48 0 0 0 48 48h137.25V327.69h-63V256h63v-54.64c0-62.15 37-96.48 93.67-96.48 27.14 0 55.52 4.84 55.52 4.84v61h-31.27c-30.81 0-40.42 19.12-40.42 38.73V256h68.78l-11 71.69h-57.78V480H400a48 48 0 0 0 48-48V80a48 48 0 0 0-48-48z"></path></svg>',
			'twitterShare'        => '<svg viewBox="0 0 512 512"><path fill="currentColor" d="M459.37 151.716c.325 4.548.325 9.097.325 13.645 0 138.72-105.583 298.558-298.558 298.558-59.452 0-114.68-17.219-161.137-47.106 8.447.974 16.568 1.299 25.34 1.299 49.055 0 94.213-16.568 130.274-44.832-46.132-.975-84.792-31.188-98.112-72.772 6.498.974 12.995 1.624 19.818 1.624 9.421 0 18.843-1.3 27.614-3.573-48.081-9.747-84.143-51.98-84.143-102.985v-1.299c13.969 7.797 30.214 12.67 47.431 13.319-28.264-18.843-46.781-51.005-46.781-87.391 0-19.492 5.197-37.36 14.294-52.954 51.655 63.675 129.3 105.258 216.365 109.807-1.624-7.797-2.599-15.918-2.599-26.04 0-57.828 46.782-104.934 104.934-104.934 30.213 0 57.502 12.67 76.67 33.137 23.715-4.548 46.456-13.32 66.599-25.34-7.798 24.366-24.366 44.833-46.132 57.827 21.117-2.273 41.584-8.122 60.426-16.243-14.292 20.791-32.161 39.308-52.628 54.253z"></path></svg>',
			'linkedinShare'       => '<svg viewBox="0 0 448 512"><path fill="currentColor" d="M100.28 448H7.4V148.9h92.88zM53.79 108.1C26.09 108.1 0 83.5 0 53.8a53.79 53.79 0 0 1 107.58 0c0 29.7-24.1 54.3-53.79 54.3zM447.9 448h-92.68V302.4c0-34.7-.7-79.2-48.29-79.2-48.29 0-55.69 37.7-55.69 76.7V448h-92.78V148.9h89.08v40.8h1.3c12.4-23.5 42.69-48.3 87.88-48.3 94 0 111.28 61.9 111.28 142.3V448z"></path></svg>',
			'mailShare'           => '<svg viewBox="0 0 512 512"><path fill="currentColor" d="M502.3 190.8c3.9-3.1 9.7-.2 9.7 4.7V400c0 26.5-21.5 48-48 48H48c-26.5 0-48-21.5-48-48V195.6c0-5 5.7-7.8 9.7-4.7 22.4 17.4 52.1 39.5 154.1 113.6 21.1 15.4 56.7 47.8 92.2 47.6 35.7.3 72-32.8 92.3-47.6 102-74.1 131.6-96.3 154-113.7zM256 320c23.2.4 56.6-29.2 73.4-41.4 132.7-96.3 142.8-104.7 173.4-128.7 5.8-4.5 9.2-11.5 9.2-18.9v-19c0-26.5-21.5-48-48-48H48C21.5 64 0 85.5 0 112v19c0 7.4 3.4 14.3 9.2 18.9 30.6 23.9 40.7 32.4 173.4 128.7 16.8 12.2 50.2 41.8 73.4 41.4z"></path></svg>',

			'successNotification' => '<svg viewBox="0 0 20 20"><path d="M10 0C4.5 0 0 4.5 0 10C0 15.5 4.5 20 10 20C15.5 20 20 15.5 20 10C20 4.5 15.5 0 10 0ZM8 15L3 10L4.41 8.59L8 12.17L15.59 4.58L17 6L8 15Z"/></svg>',
			'errorNotification'   => '<svg viewBox="0 0 20 20"><path d="M9.99997 0C4.47997 0 -3.05176e-05 4.48 -3.05176e-05 10C-3.05176e-05 15.52 4.47997 20 9.99997 20C15.52 20 20 15.52 20 10C20 4.48 15.52 0 9.99997 0ZM11 15H8.99997V13H11V15ZM11 11H8.99997V5H11V11Z"/></svg>',
			'messageNotification' => '<svg viewBox="0 0 20 20"><path d="M11.0001 7H9.00012V5H11.0001V7ZM11.0001 15H9.00012V9H11.0001V15ZM10.0001 0C8.6869 0 7.38654 0.258658 6.17329 0.761205C4.96003 1.26375 3.85764 2.00035 2.92905 2.92893C1.05369 4.8043 0.00012207 7.34784 0.00012207 10C0.00012207 12.6522 1.05369 15.1957 2.92905 17.0711C3.85764 17.9997 4.96003 18.7362 6.17329 19.2388C7.38654 19.7413 8.6869 20 10.0001 20C12.6523 20 15.1958 18.9464 17.0712 17.0711C18.9466 15.1957 20.0001 12.6522 20.0001 10C20.0001 8.68678 19.7415 7.38642 19.2389 6.17317C18.7364 4.95991 17.9998 3.85752 17.0712 2.92893C16.1426 2.00035 15.0402 1.26375 13.827 0.761205C12.6137 0.258658 11.3133 0 10.0001 0Z"/></svg>',

			'albumsPreview'       => '<svg width="63" height="65" viewBox="0 0 63 65" fill="none"><rect x="13.6484" y="10.2842" width="34.7288" height="34.7288" rx="1.44703" fill="#8C8F9A"/> <g filter="url(#filter0_dddalbumsPreview)"><rect x="22.1484" y="5.21962" width="34.7288" height="34.7288" rx="1.44703" transform="rotate(8 22.1484 5.21962)" fill="white"/> </g><path d="M29.0485 23.724L18.9288 28.1468L17.2674 39.9686L51.6582 44.802L52.2623 40.5031L29.0485 23.724Z" fill="#B5E5FF"/> <path d="M44.9106 25.2228L17.7194 36.7445L17.2663 39.9687L51.6571 44.802L53.4696 31.9054L44.9106 25.2228Z" fill="#43A6DB"/> <circle cx="42.9495" cy="18.3718" r="2.89406" transform="rotate(8 42.9495 18.3718)" fill="#43A6DB"/> <g filter="url(#filter1_dddalbumsPreview)"> <rect x="42.4766" y="33.9054" width="16.875" height="16.875" rx="8.4375" fill="white"/> <path d="M54.1953 42.8116H51.3828V45.6241H50.4453V42.8116H47.6328V41.8741H50.4453V39.0616H51.3828V41.8741H54.1953V42.8116Z" fill="#0068A0"/> </g> <defs> <filter id="filter0_dddalbumsPreview" x="0.86108" y="0.342124" width="58.3848" height="57.6613" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"> <feFlood flood-opacity="0" result="BackgroundImageFix"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset dx="-7.23516" dy="4.3411"/> <feGaussianBlur stdDeviation="4.70286"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.1 0"/> <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.25 0"/> <feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset dy="2.89406"/> <feGaussianBlur stdDeviation="1.44703"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.25 0"/> <feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/> <feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow" result="shape"/> </filter> <filter id="filter1_dddalbumsPreview" x="25.8357" y="28.8408" width="36.4099" height="35.6864" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"> <feFlood flood-opacity="0" result="BackgroundImageFix"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset dx="-7.23516" dy="4.3411"/> <feGaussianBlur stdDeviation="4.70286"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.1 0"/> <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.25 0"/> <feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset dy="2.89406"/> <feGaussianBlur stdDeviation="1.44703"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.25 0"/> <feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/> <feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow" result="shape"/> </filter> </defs> </svg>',
			'featuredPostPreview' => '<svg width="47" height="48" viewBox="0 0 47 48" fill="none"> <g filter="url(#filter0_ddfeaturedpos)"> <rect x="2.09375" y="1.84264" width="34.7288" height="34.7288" rx="1.44703" fill="white"/> </g> <path d="M11.4995 19.2068L2.09375 24.9949L2.09375 36.9329H36.8225V32.5918L11.4995 19.2068Z" fill="#B5E5FF"/> <path d="M27.4168 18.4833L2.09375 33.6772V36.933H36.8225V23.9097L27.4168 18.4833Z" fill="#43A6DB"/> <circle cx="24.523" cy="11.9718" r="2.89406" fill="#43A6DB"/> <g filter="url(#filter1_ddfeaturedpos)"> <rect x="26.0312" y="25.2824" width="16.875" height="16.875" rx="8.4375" fill="white"/> <path d="M37.75 34.1886H34.9375V37.0011H34V34.1886H31.1875V33.2511H34V30.4386H34.9375V33.2511H37.75V34.1886Z" fill="#0068A0"/> </g> <defs> <filter id="filter0_ddfeaturedpos" x="0.09375" y="0.842636" width="40.7288" height="40.7288" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"> <feFlood flood-opacity="0" result="BackgroundImageFix"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset dx="1" dy="2"/> <feGaussianBlur stdDeviation="1.5"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.1 0"/> <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.25 0"/> <feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/> <feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow" result="shape"/> </filter> <filter id="filter1_ddfeaturedpos" x="26.0312" y="24.2824" width="22.875" height="22.875" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"> <feFlood flood-opacity="0" result="BackgroundImageFix"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset dx="1" dy="2"/> <feGaussianBlur stdDeviation="1.5"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.1 0"/> <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.25 0"/> <feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/> <feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow" result="shape"/> </filter> </defs> </svg>',
			'issueSinglePreview'  => '<svg width="27" height="18" viewBox="0 0 27 18" fill="none"> <line x1="3.22082" y1="2.84915" x2="8.91471" y2="8.54304" stroke="#8C8F9A" stroke-width="3"/> <path d="M3.10938 8.65422L8.80327 2.96033" stroke="#8C8F9A" stroke-width="3"/> <line x1="18.3107" y1="2.84915" x2="26.0046" y2="8.54304" stroke="#8C8F9A" stroke-width="3"/> <path d="M18.1992 8.65422L23.8931 2.96033" stroke="#8C8F9A" stroke-width="3"/> <line x1="8.64062" y1="16.3863" x2="18.0351" y2="16.3863" stroke="#8C8F9A" stroke-width="3"/> </svg>',
			'playButton'          => '<svg viewBox="0 0 448 512"><path fill="currentColor" d="M424.4 214.7L72.4 6.6C43.8-10.3 0 6.1 0 47.9V464c0 37.5 40.7 60.1 72.4 41.3l352-208c31.4-18.5 31.5-64.1 0-82.6z"></path></svg>',
			'spinner'             => '<svg version="1.1" id="loader-1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" x="0px" y="0px" width="20px" height="20px" viewBox="0 0 50 50" style="enable-background:new 0 0 50 50;" xml:space="preserve"><path fill="#fff" d="M43.935,25.145c0-10.318-8.364-18.683-18.683-18.683c-10.318,0-18.683,8.365-18.683,18.683h6.068c0-8.071,6.543-14.615,14.615-14.615c8.072,0,14.615,6.543,14.615,14.615H43.935z"><animateTransform attributeType="xml" attributeName="transform" type="rotate" from="0 25 25" to="360 25 25" dur="0.6s" repeatCount="indefinite"/></path></svg>',
			'follow'              => '<svg viewBox="0 0 24 24"><path d="M20 18.5H4C3.46957 18.5 2.96086 18.2893 2.58579 17.9142C2.21071 17.5391 2 17.0304 2 16.5V7.5C2 6.96957 2.21071 6.46086 2.58579 6.08579C2.96086 5.71071 3.46957 5.5 4 5.5H20C20.5304 5.5 21.0391 5.71071 21.4142 6.08579C21.7893 6.46086 22 6.96957 22 7.5V16.5C22 17.0304 21.7893 17.5391 21.4142 17.9142C21.0391 18.2893 20.5304 18.5 20 18.5ZM4 7.5V16.5H20V7.5H4Z" fill="#141B38"/><path d="M9 13.75C9 13.1977 9.44772 12.75 10 12.75H14C14.5523 12.75 15 13.1977 15 13.75V15H9V13.75Z" fill="#141B38"/><path d="M13.5 10.5C13.5 11.3284 12.8284 12 12 12C11.1716 12 10.5 11.3284 10.5 10.5C10.5 9.67157 11.1716 9 12 9C12.8284 9 13.5 9.67157 13.5 10.5Z" fill="#141B38"/></svg>',
			'picture'             => '<svg viewBox="0 0 24 24" fill="none"><path d="M8.5 13.5L11 16.5L14.5 12L19 18H5L8.5 13.5ZM21 19V5C21 4.46957 20.7893 3.96086 20.4142 3.58579C20.0391 3.21071 19.5304 3 19 3H5C4.46957 3 3.96086 3.21071 3.58579 3.58579C3.21071 3.96086 3 4.46957 3 5V19C3 19.5304 3.21071 20.0391 3.58579 20.4142C3.96086 20.7893 4.46957 21 5 21H19C19.5304 21 20.0391 20.7893 20.4142 20.4142C20.7893 20.0391 21 19.5304 21 19Z"/></svg>',
			'caption'             => '<svg viewBox="0 0 24 24" fill="none"><path d="M5 3C3.89 3 3 3.89 3 5V19C3 20.11 3.89 21 5 21H19C20.11 21 21 20.11 21 19V5C21 3.89 20.11 3 19 3H5ZM5 5H19V19H5V5ZM7 7V9H17V7H7ZM7 11V13H17V11H7ZM7 15V17H14V15H7Z"/></svg>',
			'heart'               => '<svg viewBox="0 0 24 24"><path d="M16.5 3C14.76 3 13.09 3.81 12 5.09C10.91 3.81 9.24 3 7.5 3C4.42 3 2 5.42 2 8.5C2 12.28 5.4 15.36 10.55 20.04L12 21.35L13.45 20.03C18.6 15.36 22 12.28 22 8.5C22 5.42 19.58 3 16.5 3ZM12.1 18.55L12 18.65L11.9 18.55C7.14 14.24 4 11.39 4 8.5C4 6.5 5.5 5 7.5 5C9.04 5 10.54 5.99 11.07 7.36H12.94C13.46 5.99 14.96 5 16.5 5C18.5 5 20 6.5 20 8.5C20 11.39 16.86 14.24 12.1 18.55Z"/></svg>',
			'sort'                => '<svg viewBox="0 0 24 24"><path d="M7.73062 10.9999C7.51906 10.9999 7.40314 10.7535 7.53803 10.5906L11.8066 5.43267C11.9066 5.31186 12.0918 5.31186 12.1918 5.43267L16.4604 10.5906C16.5953 10.7535 16.4794 10.9999 16.2678 10.9999H7.73062Z" fill="#141B38"/><path d="M7.80277 13C7.58005 13 7.4685 13.2693 7.626 13.4268L11.8224 17.6232C11.9201 17.7209 12.0784 17.7209 12.176 17.6232L16.3724 13.4268C16.5299 13.2693 16.4184 13 16.1957 13H7.80277Z" fill="#141B38"/></svg>',
			'shop'                => '<svg viewBox="0 0 24 24"><path d="M11 9H13V6H16V4H13V1H11V4H8V6H11V9ZM7 18C5.9 18 5.01 18.9 5.01 20C5.01 21.1 5.9 22 7 22C8.1 22 9 21.1 9 20C9 18.9 8.1 18 7 18ZM17 18C15.9 18 15.01 18.9 15.01 20C15.01 21.1 15.9 22 17 22C18.1 22 19 21.1 19 20C19 18.9 18.1 18 17 18ZM8.1 13H15.55C16.3 13 16.96 12.59 17.3 11.97L21.16 4.96L19.42 4L15.55 11H8.53L4.27 2H1V4H3L6.6 11.59L5.25 14.03C4.52 15.37 5.48 17 7 17H19V15H7L8.1 13Z" fill="#141B38"/></svg>',
			'headerUser'          => '<svg class="svg-inline--fa fa-user fa-w-16" style="margin-right: 3px;" aria-hidden="true" data-fa-processed="" data-prefix="fa" data-icon="user" role="presentation" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><path fill="currentColor" d="M96 160C96 71.634 167.635 0 256 0s160 71.634 160 160-71.635 160-160 160S96 248.366 96 160zm304 192h-28.556c-71.006 42.713-159.912 42.695-230.888 0H112C50.144 352 0 402.144 0 464v24c0 13.255 10.745 24 24 24h464c13.255 0 24-10.745 24-24v-24c0-61.856-50.144-112-112-112z"></path></svg>',
			'headerPhoto'         => '<svg class="svg-inline--fa fa-image fa-w-16" aria-hidden="true" data-fa-processed="" data-prefix="far" data-icon="image" role="presentation" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><path fill="currentColor" d="M464 448H48c-26.51 0-48-21.49-48-48V112c0-26.51 21.49-48 48-48h416c26.51 0 48 21.49 48 48v288c0 26.51-21.49 48-48 48zM112 120c-30.928 0-56 25.072-56 56s25.072 56 56 56 56-25.072 56-56-25.072-56-56-56zM64 384h384V272l-87.515-87.515c-4.686-4.686-12.284-4.686-16.971 0L208 320l-55.515-55.515c-4.686-4.686-12.284-4.686-16.971 0L64 336v48z"></path></svg>',
			'imageChooser'        => '<svg viewBox="0 0 18 18" fill="none"><path d="M2.16667 0.5C1.72464 0.5 1.30072 0.675595 0.988155 0.988155C0.675595 1.30072 0.5 1.72464 0.5 2.16667V13.8333C0.5 14.2754 0.675595 14.6993 0.988155 15.0118C1.30072 15.3244 1.72464 15.5 2.16667 15.5H9.74167C9.69167 15.225 9.66667 14.95 9.66667 14.6667C9.66667 14.1 9.76667 13.5333 9.95833 13H2.16667L5.08333 9.25L7.16667 11.75L10.0833 8L11.9417 10.475C12.75 9.95 13.7 9.66667 14.6667 9.66667C14.95 9.66667 15.225 9.69167 15.5 9.74167V2.16667C15.5 1.72464 15.3244 1.30072 15.0118 0.988155C14.6993 0.675595 14.2754 0.5 13.8333 0.5H2.16667ZM13.8333 11.3333V13.8333H11.3333V15.5H13.8333V18H15.5V15.5H18V13.8333H15.5V11.3333H13.8333Z"/></svg>',

			'usertimelineIcon'    => '<svg width="260" height="126" viewBox="0 0 260 126" fill="none" xmlns="http://www.w3.org/2000/svg"> <g clip-path="url(#usrtimlineclip0)"> <g filter="url(#usrtimlinefilter0_ddd)"> <g clip-path="url(#usrtimlineclip1)"> <rect x="64" y="23" width="131" height="113" rx="2" fill="white"/> <rect x="112.027" y="38" width="46" height="6" rx="1" fill="#DCDDE1"/> <rect x="112.027" y="49" width="28" height="6" rx="1" fill="#DCDDE1"/> <g clip-path="url(#usrtimlineclip2)"> <rect x="133.027" y="121" width="48" height="48" rx="1" fill="#F9BBA0"/> </g> <g clip-path="url(#usrtimlineclip3)"> <rect x="133.027" y="67" width="48" height="48" fill="#43A6DB"/> <circle cx="123.527" cy="101.5" r="40.5" fill="#86D0F9"/> </g> <g clip-path="url(#usrtimlineclip4)"> <rect x="79.0273" y="121" width="48" height="48" fill="#B6DDAD"/> </g> <g clip-path="url(#usrtimlineclip5)"> <rect x="79.0273" y="67" width="48" height="48" fill="#F6966B"/> <path d="M88.4756 84L126.476 122H57.0273L88.4756 84Z" fill="#F9BBA0"/> </g> <circle cx="92.0273" cy="45" r="10" fill="#DCDDE1"/> <circle cx="92.0273" cy="45" r="12" stroke="url(#usrtimlinepaint0_linear)"/> </g> </g> </g> <defs> <filter id="usrtimlinefilter0_ddd" x="51" y="16" width="157" height="139" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"> <feFlood flood-opacity="0" result="BackgroundImageFix"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="6"/> <feGaussianBlur stdDeviation="6.5"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/> <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="1"/> <feGaussianBlur stdDeviation="1"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/> <feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="3"/> <feGaussianBlur stdDeviation="3"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/> <feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/> <feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow" result="shape"/> </filter> <linearGradient id="usrtimlinepaint0_linear" x1="88.5773" y1="78.9" x2="139.127" y2="27.3" gradientUnits="userSpaceOnUse"> <stop stop-color="white"/> <stop offset="0.147864" stop-color="#F6640E"/> <stop offset="0.443974" stop-color="#BA03A7"/> <stop offset="0.733337" stop-color="#6A01B9"/> <stop offset="1" stop-color="#6B01B9"/> </linearGradient> <clipPath id="usrtimlineclip0"> <rect width="259.056" height="126" fill="white"/> </clipPath> <clipPath id="usrtimlineclip1"> <rect x="64" y="23" width="131" height="113" rx="2" fill="white"/> </clipPath> <clipPath id="usrtimlineclip2"> <rect x="133.027" y="121" width="48" height="48" rx="1" fill="white"/> </clipPath> <clipPath id="usrtimlineclip3"> <rect x="133.027" y="67" width="48" height="48" rx="1" fill="white"/> </clipPath> <clipPath id="usrtimlineclip4"> <rect x="79.0273" y="121" width="48" height="48" rx="1" fill="white"/> </clipPath> <clipPath id="usrtimlineclip5"> <rect x="79.0273" y="67" width="48" height="48" rx="1" fill="white"/> </clipPath> </defs> </svg>',
			'publichashtagIcon'   => '<svg width="260" height="126" viewBox="0 0 260 126" fill="none" xmlns="http://www.w3.org/2000/svg"> <g clip-path="url(#hashtagiconclip0)"> <g filter="url(#hashtagiconfilter0_ddd)"> <rect x="119.327" y="12.3203" width="80" height="91" rx="2" transform="rotate(4 119.327 12.3203)" fill="white"/> </g> <g clip-path="url(#hashtagiconclip1)"> <rect x="119.327" y="12.3203" width="80" height="80" transform="rotate(4 119.327 12.3203)" fill="#0096CC"/> </g> <path d="M130.918 88.5016L128.971 88.3655L129.441 86.6958C129.453 86.6464 129.454 86.5951 129.444 86.5452C129.435 86.4954 129.414 86.4482 129.385 86.4069C129.355 86.3657 129.317 86.3313 129.273 86.3062C129.229 86.2811 129.18 86.2659 129.129 86.2616L128.427 86.2125C128.347 86.2049 128.265 86.2255 128.198 86.2709C128.131 86.3163 128.081 86.3837 128.058 86.4616L127.572 88.2676L125.678 88.1352L126.147 86.4654C126.159 86.4172 126.16 86.3671 126.151 86.3182C126.142 86.2694 126.123 86.223 126.095 86.182C126.067 86.1411 126.031 86.1066 125.988 86.0808C125.946 86.055 125.899 86.0384 125.849 86.0322L125.148 85.9832C125.067 85.9755 124.986 85.9962 124.918 86.0416C124.851 86.087 124.801 86.1544 124.778 86.2322L124.299 88.0388L122.194 87.8916C122.112 87.8842 122.03 87.9058 121.963 87.9526C121.895 87.9994 121.846 88.0684 121.824 88.1477L121.631 88.8392C121.617 88.89 121.614 88.9433 121.624 88.9953C121.633 89.0472 121.654 89.0964 121.685 89.1391C121.716 89.1819 121.756 89.2172 121.802 89.2424C121.848 89.2676 121.9 89.282 121.952 89.2846L123.899 89.4208L123.128 92.1867L121.023 92.0396C120.941 92.0322 120.859 92.0537 120.791 92.1005C120.724 92.1473 120.675 92.2164 120.653 92.2957L120.46 92.9871C120.446 93.038 120.443 93.0913 120.452 93.1432C120.462 93.1952 120.483 93.2443 120.513 93.2871C120.544 93.3299 120.584 93.3652 120.631 93.3904C120.677 93.4156 120.728 93.43 120.781 93.4326L122.742 93.5697L122.273 95.2394C122.26 95.2896 122.259 95.3419 122.269 95.3926C122.28 95.4432 122.301 95.491 122.332 95.5325C122.362 95.5741 122.402 95.6083 122.447 95.6328C122.493 95.6573 122.543 95.6715 122.595 95.6744L123.296 95.7234C123.375 95.7269 123.452 95.7041 123.516 95.6588C123.579 95.6135 123.626 95.5481 123.649 95.4731L124.142 93.6676L126.036 93.8L125.566 95.4698C125.555 95.5179 125.553 95.5681 125.562 95.617C125.571 95.6658 125.59 95.7122 125.618 95.7531C125.646 95.7941 125.683 95.8286 125.725 95.8544C125.767 95.8802 125.815 95.8968 125.864 95.903L126.566 95.952C126.647 95.9597 126.728 95.939 126.795 95.8936C126.862 95.8482 126.912 95.7808 126.935 95.703L127.432 93.8977L129.536 94.0448C129.618 94.0522 129.7 94.0306 129.768 93.9839C129.836 93.9371 129.885 93.868 129.907 93.7887L130.096 93.097C130.11 93.0462 130.113 92.9928 130.104 92.9409C130.094 92.889 130.073 92.8398 130.043 92.797C130.012 92.7542 129.972 92.719 129.925 92.6938C129.879 92.6686 129.828 92.6542 129.775 92.6515L127.818 92.5147L128.586 89.7485L130.69 89.8956C130.772 89.903 130.854 89.8814 130.922 89.8347C130.989 89.7879 131.039 89.7188 131.061 89.6395L131.253 88.948C131.268 88.8961 131.27 88.8414 131.26 88.7883C131.25 88.7353 131.228 88.6852 131.196 88.642C131.164 88.5989 131.122 88.5637 131.073 88.5394C131.025 88.515 130.972 88.5021 130.918 88.5016ZM126.414 92.4166L124.52 92.2841L125.292 89.5181L127.186 89.6506L126.414 92.4166Z" fill="#0068A0"/> <rect x="138.037" y="88.8115" width="29" height="7" rx="1" transform="rotate(4 138.037 88.8115)" fill="#86D0F9"/> <g filter="url(#hashtagiconfilter1_ddd)"> <rect x="119.327" y="12.3203" width="80" height="91" rx="2" transform="rotate(4 119.327 12.3203)" fill="white"/> </g> <g clip-path="url(#hashtagiconclip2)"> <rect x="119.327" y="12.3203" width="80" height="80" transform="rotate(4 119.327 12.3203)" fill="#0096CC"/> <circle cx="126.556" cy="44.5" r="46.5" fill="#0068A0"/> </g> <path d="M130.918 88.5016L128.971 88.3655L129.441 86.6958C129.453 86.6464 129.454 86.5951 129.444 86.5452C129.435 86.4954 129.414 86.4482 129.385 86.4069C129.355 86.3657 129.317 86.3313 129.273 86.3062C129.229 86.2811 129.18 86.2659 129.129 86.2616L128.427 86.2125C128.347 86.2049 128.265 86.2255 128.198 86.2709C128.131 86.3163 128.081 86.3837 128.058 86.4616L127.572 88.2676L125.678 88.1352L126.147 86.4654C126.159 86.4172 126.16 86.3671 126.151 86.3182C126.142 86.2694 126.123 86.223 126.095 86.182C126.067 86.1411 126.031 86.1066 125.988 86.0808C125.946 86.055 125.899 86.0384 125.849 86.0322L125.148 85.9832C125.067 85.9755 124.986 85.9962 124.918 86.0416C124.851 86.087 124.801 86.1544 124.778 86.2322L124.299 88.0388L122.194 87.8916C122.112 87.8842 122.03 87.9058 121.963 87.9526C121.895 87.9994 121.846 88.0684 121.824 88.1477L121.631 88.8392C121.617 88.89 121.614 88.9433 121.624 88.9953C121.633 89.0472 121.654 89.0964 121.685 89.1391C121.716 89.1819 121.756 89.2172 121.802 89.2424C121.848 89.2676 121.9 89.282 121.952 89.2846L123.899 89.4208L123.128 92.1867L121.023 92.0396C120.941 92.0322 120.859 92.0537 120.791 92.1005C120.724 92.1473 120.675 92.2164 120.653 92.2957L120.46 92.9871C120.446 93.038 120.443 93.0913 120.452 93.1432C120.462 93.1952 120.483 93.2443 120.513 93.2871C120.544 93.3299 120.584 93.3652 120.631 93.3904C120.677 93.4156 120.728 93.43 120.781 93.4326L122.742 93.5697L122.273 95.2394C122.26 95.2896 122.259 95.3419 122.269 95.3926C122.28 95.4432 122.301 95.491 122.332 95.5325C122.362 95.5741 122.402 95.6083 122.447 95.6328C122.493 95.6573 122.543 95.6715 122.595 95.6744L123.296 95.7234C123.375 95.7269 123.452 95.7041 123.516 95.6588C123.579 95.6135 123.626 95.5481 123.649 95.4731L124.142 93.6676L126.036 93.8L125.566 95.4698C125.555 95.5179 125.553 95.5681 125.562 95.617C125.571 95.6658 125.59 95.7122 125.618 95.7531C125.646 95.7941 125.683 95.8286 125.725 95.8544C125.767 95.8802 125.815 95.8968 125.864 95.903L126.566 95.952C126.647 95.9597 126.728 95.939 126.795 95.8936C126.862 95.8482 126.912 95.7808 126.935 95.703L127.432 93.8977L129.536 94.0448C129.618 94.0522 129.7 94.0306 129.768 93.9839C129.836 93.9371 129.885 93.868 129.907 93.7887L130.096 93.097C130.11 93.0462 130.113 92.9928 130.104 92.9409C130.094 92.889 130.073 92.8398 130.043 92.797C130.012 92.7542 129.972 92.719 129.925 92.6938C129.879 92.6686 129.828 92.6542 129.775 92.6515L127.818 92.5147L128.586 89.7485L130.69 89.8956C130.772 89.903 130.854 89.8814 130.922 89.8347C130.989 89.7879 131.039 89.7188 131.061 89.6395L131.253 88.948C131.268 88.8961 131.27 88.8414 131.26 88.7883C131.25 88.7353 131.228 88.6852 131.196 88.642C131.164 88.5989 131.122 88.5637 131.073 88.5394C131.025 88.515 130.972 88.5021 130.918 88.5016ZM126.414 92.4166L124.52 92.2841L125.292 89.5181L127.186 89.6506L126.414 92.4166Z" fill="#0068A0"/> <rect x="138.037" y="88.8115" width="29" height="7" rx="1" transform="rotate(4 138.037 88.8115)" fill="#86D0F9"/> <g filter="url(#hashtagiconfilter2_ddd)"> <rect x="65.0557" y="21" width="80" height="91" rx="2" fill="white"/> </g> <g clip-path="url(#hashtagiconclip3)"> <rect x="65.0557" y="21" width="80" height="80" fill="#F6966B"/> <path d="M80.8025 49.333L144.136 112.666H28.3887L80.8025 49.333Z" fill="#F9BBA0"/> </g> <path d="M81.9327 96.187H79.9812L80.3328 94.4887C80.3414 94.4386 80.3391 94.3873 80.3261 94.3382C80.313 94.2892 80.2894 94.2435 80.257 94.2044C80.2246 94.1653 80.1841 94.1337 80.1383 94.1118C80.0925 94.0898 80.0425 94.078 79.9917 94.0773H79.2885C79.2072 94.0753 79.1277 94.1015 79.0636 94.1515C78.9995 94.2015 78.9547 94.2722 78.9368 94.3515L78.5782 96.187H76.6794L77.031 94.4887C77.0395 94.4398 77.0376 94.3896 77.0253 94.3415C77.013 94.2934 76.9907 94.2484 76.9598 94.2095C76.9289 94.1707 76.8902 94.1388 76.8461 94.116C76.802 94.0932 76.7535 94.08 76.704 94.0773H76.0007C75.9194 94.0753 75.84 94.1015 75.7759 94.1515C75.7117 94.2015 75.6669 94.2722 75.6491 94.3515L75.2974 96.187H73.1877C73.1054 96.1854 73.0252 96.2126 72.9609 96.264C72.8967 96.3154 72.8525 96.3877 72.836 96.4683L72.6919 97.1716C72.6813 97.2233 72.6825 97.2767 72.6954 97.3278C72.7083 97.379 72.7325 97.4266 72.7662 97.4671C72.8 97.5076 72.8425 97.54 72.8905 97.5619C72.9385 97.5838 72.9908 97.5946 73.0435 97.5936H74.995L74.4184 100.407H72.3086C72.2263 100.405 72.1461 100.432 72.0818 100.484C72.0176 100.535 71.9734 100.607 71.957 100.688L71.8128 101.391C71.8022 101.443 71.8034 101.496 71.8163 101.547C71.8292 101.599 71.8534 101.646 71.8872 101.687C71.9209 101.727 71.9634 101.76 72.0114 101.782C72.0594 101.803 72.1117 101.814 72.1644 101.813H74.13L73.7784 103.512C73.7696 103.562 73.7722 103.615 73.7858 103.664C73.7995 103.714 73.824 103.761 73.8576 103.8C73.8912 103.839 73.933 103.87 73.9801 103.892C74.0272 103.913 74.0784 103.924 74.13 103.923H74.8333C74.9116 103.921 74.9869 103.893 75.0474 103.843C75.1079 103.793 75.1501 103.725 75.1673 103.649L75.533 101.813H77.4318L77.0802 103.512C77.0717 103.56 77.0736 103.611 77.0859 103.659C77.0982 103.707 77.1205 103.752 77.1514 103.791C77.1823 103.829 77.221 103.861 77.2651 103.884C77.3092 103.907 77.3577 103.92 77.4072 103.923H78.1105C78.1918 103.925 78.2712 103.899 78.3354 103.849C78.3995 103.799 78.4443 103.728 78.4621 103.649L78.8313 101.813H80.9411C81.0234 101.815 81.1036 101.788 81.1679 101.736C81.2321 101.685 81.2763 101.612 81.2928 101.532L81.4334 100.829C81.444 100.777 81.4428 100.723 81.4299 100.672C81.417 100.621 81.3928 100.574 81.359 100.533C81.3253 100.493 81.2828 100.46 81.2348 100.438C81.1868 100.416 81.1345 100.406 81.0818 100.407H79.1197L79.6928 97.5936H81.8026C81.8849 97.5952 81.9651 97.568 82.0294 97.5166C82.0936 97.4652 82.1378 97.3929 82.1543 97.3123L82.2984 96.609C82.3093 96.5561 82.3079 96.5014 82.2942 96.4492C82.2806 96.3969 82.2551 96.3485 82.2197 96.3077C82.1844 96.2669 82.1401 96.2348 82.0903 96.2139C82.0405 96.193 81.9866 96.1838 81.9327 96.187ZM77.7132 100.407H75.8143L76.391 97.5936H78.2898L77.7132 100.407Z" fill="#FE544F"/> <rect x="89.0557" y="96" width="29" height="7" rx="1" fill="#FCE1D5"/> <g filter="url(#hashtagiconfilter3_ddd)"> <rect x="65.0557" y="21" width="80" height="91" rx="2" fill="white"/> </g> <g clip-path="url(#hashtagiconclip4)"> <rect x="65.0557" y="21" width="80" height="80" fill="#F6966B"/> <path d="M80.8025 49.333L144.136 112.666H28.3887L80.8025 49.333Z" fill="#F9BBA0"/> </g> <path d="M81.9327 96.187H79.9812L80.3328 94.4887C80.3414 94.4386 80.3391 94.3873 80.3261 94.3382C80.313 94.2892 80.2894 94.2435 80.257 94.2044C80.2246 94.1653 80.1841 94.1337 80.1383 94.1118C80.0925 94.0898 80.0425 94.078 79.9917 94.0773H79.2885C79.2072 94.0753 79.1277 94.1015 79.0636 94.1515C78.9995 94.2015 78.9547 94.2722 78.9368 94.3515L78.5782 96.187H76.6794L77.031 94.4887C77.0395 94.4398 77.0376 94.3896 77.0253 94.3415C77.013 94.2934 76.9907 94.2484 76.9598 94.2095C76.9289 94.1707 76.8902 94.1388 76.8461 94.116C76.802 94.0932 76.7535 94.08 76.704 94.0773H76.0007C75.9194 94.0753 75.84 94.1015 75.7759 94.1515C75.7117 94.2015 75.6669 94.2722 75.6491 94.3515L75.2974 96.187H73.1877C73.1054 96.1854 73.0252 96.2126 72.9609 96.264C72.8967 96.3154 72.8525 96.3877 72.836 96.4683L72.6919 97.1716C72.6813 97.2233 72.6825 97.2767 72.6954 97.3278C72.7083 97.379 72.7325 97.4266 72.7662 97.4671C72.8 97.5076 72.8425 97.54 72.8905 97.5619C72.9385 97.5838 72.9908 97.5946 73.0435 97.5936H74.995L74.4184 100.407H72.3086C72.2263 100.405 72.1461 100.432 72.0818 100.484C72.0176 100.535 71.9734 100.607 71.957 100.688L71.8128 101.391C71.8022 101.443 71.8034 101.496 71.8163 101.547C71.8292 101.599 71.8534 101.646 71.8872 101.687C71.9209 101.727 71.9634 101.76 72.0114 101.782C72.0594 101.803 72.1117 101.814 72.1644 101.813H74.13L73.7784 103.512C73.7696 103.562 73.7722 103.615 73.7858 103.664C73.7995 103.714 73.824 103.761 73.8576 103.8C73.8912 103.839 73.933 103.87 73.9801 103.892C74.0272 103.913 74.0784 103.924 74.13 103.923H74.8333C74.9116 103.921 74.9869 103.893 75.0474 103.843C75.1079 103.793 75.1501 103.725 75.1673 103.649L75.533 101.813H77.4318L77.0802 103.512C77.0717 103.56 77.0736 103.611 77.0859 103.659C77.0982 103.707 77.1205 103.752 77.1514 103.791C77.1823 103.829 77.221 103.861 77.2651 103.884C77.3092 103.907 77.3577 103.92 77.4072 103.923H78.1105C78.1918 103.925 78.2712 103.899 78.3354 103.849C78.3995 103.799 78.4443 103.728 78.4621 103.649L78.8313 101.813H80.9411C81.0234 101.815 81.1036 101.788 81.1679 101.736C81.2321 101.685 81.2763 101.612 81.2928 101.532L81.4334 100.829C81.444 100.777 81.4428 100.723 81.4299 100.672C81.417 100.621 81.3928 100.574 81.359 100.533C81.3253 100.493 81.2828 100.46 81.2348 100.438C81.1868 100.416 81.1345 100.406 81.0818 100.407H79.1197L79.6928 97.5936H81.8026C81.8849 97.5952 81.9651 97.568 82.0294 97.5166C82.0936 97.4652 82.1378 97.3929 82.1543 97.3123L82.2984 96.609C82.3093 96.5561 82.3079 96.5014 82.2942 96.4492C82.2806 96.3969 82.2551 96.3485 82.2197 96.3077C82.1844 96.2669 82.1401 96.2348 82.0903 96.2139C82.0405 96.193 81.9866 96.1838 81.9327 96.187ZM77.7132 100.407H75.8143L76.391 97.5936H78.2898L77.7132 100.407Z" fill="#FE544F"/> <rect x="89.0557" y="96" width="29" height="7" rx="1" fill="#FCE1D5"/> </g> <defs> <filter id="hashtagiconfilter0_ddd" x="100.114" y="5.45508" width="111.884" height="122.09" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"> <feFlood flood-opacity="0" result="BackgroundImageFix"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="6"/> <feGaussianBlur stdDeviation="6.5"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/> <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="1"/> <feGaussianBlur stdDeviation="1"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/> <feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="3"/> <feGaussianBlur stdDeviation="3"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/> <feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/> <feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow" result="shape"/> </filter> <filter id="hashtagiconfilter1_ddd" x="100.114" y="5.45508" width="111.884" height="122.09" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"> <feFlood flood-opacity="0" result="BackgroundImageFix"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="6"/> <feGaussianBlur stdDeviation="6.5"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/> <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="1"/> <feGaussianBlur stdDeviation="1"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/> <feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="3"/> <feGaussianBlur stdDeviation="3"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/> <feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/> <feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow" result="shape"/> </filter> <filter id="hashtagiconfilter2_ddd" x="52.0557" y="14" width="106" height="117" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"> <feFlood flood-opacity="0" result="BackgroundImageFix"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="6"/> <feGaussianBlur stdDeviation="6.5"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/> <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="1"/> <feGaussianBlur stdDeviation="1"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/> <feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="3"/> <feGaussianBlur stdDeviation="3"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/> <feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/> <feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow" result="shape"/> </filter> <filter id="hashtagiconfilter3_ddd" x="52.0557" y="14" width="106" height="117" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"> <feFlood flood-opacity="0" result="BackgroundImageFix"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="6"/> <feGaussianBlur stdDeviation="6.5"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/> <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="1"/> <feGaussianBlur stdDeviation="1"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/> <feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="3"/> <feGaussianBlur stdDeviation="3"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/> <feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/> <feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow" result="shape"/> </filter> <clipPath id="hashtagiconclip0"> <rect width="259.056" height="126" fill="white" transform="translate(0.0556641)"/> </clipPath> <clipPath id="hashtagiconclip1"> <path d="M119.211 13.9829C119.275 13.0647 120.072 12.3724 120.99 12.4366L197.47 17.7846C198.388 17.8488 199.08 18.6452 199.016 19.5634L194.528 83.7401L114.723 78.1595L119.211 13.9829Z" fill="white"/> </clipPath> <clipPath id="hashtagiconclip2"> <path d="M119.211 13.9829C119.275 13.0647 120.072 12.3724 120.99 12.4366L197.47 17.7846C198.388 17.8488 199.08 18.6452 199.016 19.5634L194.528 83.7401L114.723 78.1595L119.211 13.9829Z" fill="white"/> </clipPath> <clipPath id="hashtagiconclip3"> <path d="M65.0557 22.6667C65.0557 21.7462 65.8019 21 66.7223 21H143.389C144.309 21 145.056 21.7462 145.056 22.6667V87H65.0557V22.6667Z" fill="white"/> </clipPath> <clipPath id="hashtagiconclip4"> <path d="M65.0557 22.6667C65.0557 21.7462 65.8019 21 66.7223 21H143.389C144.309 21 145.056 21.7462 145.056 22.6667V87H65.0557V22.6667Z" fill="white"/> </clipPath> </defs> </svg>',
			'taggedpostsIcon'     => '<svg width="260" height="126" viewBox="0 0 260 126" fill="none" xmlns="http://www.w3.org/2000/svg"> <g clip-path="url(#taggedpostclip0)"> <g filter="url(#taggedpostfilter0_ddd)"> <g clip-path="url(#taggedpostclip1)"> <rect x="104.316" y="29.0303" width="83.0697" height="84.1212" rx="2.10303" transform="rotate(2 104.316 29.0303)" fill="white"/> <g clip-path="url(#taggedpostclip2)"> <path d="M104.063 23.0957L188.133 26.0315L185.418 103.796L101.348 100.86L104.063 23.0957Z" fill="#59AB46"/> <path d="M119.756 48.194L183.987 117.073L62.3516 112.826L119.756 48.194Z" fill="#76C064"/> </g> <path fill-rule="evenodd" clip-rule="evenodd" d="M113.109 94.8001C114.187 94.6246 115.292 94.7726 116.286 95.2254C117.279 95.6782 118.116 96.4154 118.691 97.3439C119.265 98.2723 119.552 99.3503 119.513 100.441L119.485 101.259C119.467 101.783 119.241 102.278 118.858 102.635C118.474 102.993 117.964 103.183 117.441 103.165C116.917 103.147 116.422 102.921 116.064 102.538C115.997 102.466 115.937 102.391 115.882 102.311C115.342 102.804 114.63 103.067 113.899 103.041C113.158 103.016 112.458 102.697 111.953 102.155C111.447 101.613 111.178 100.892 111.204 100.151C111.23 99.4107 111.549 98.7106 112.091 98.2052C112.633 97.6998 113.353 97.4304 114.094 97.4562C114.834 97.4821 115.535 97.8011 116.04 98.3431C116.545 98.8851 116.815 99.6057 116.789 100.346L116.76 101.164C116.753 101.362 116.826 101.556 116.961 101.701C117.097 101.847 117.285 101.932 117.483 101.939C117.682 101.946 117.875 101.874 118.021 101.738C118.166 101.603 118.252 101.415 118.259 101.216L118.287 100.399C118.317 99.55 118.094 98.7115 117.647 97.9894C117.201 97.2673 116.55 96.6939 115.777 96.3417C115.004 95.9896 114.144 95.8745 113.306 96.011C112.468 96.1475 111.689 96.5295 111.068 97.1086C110.447 97.6878 110.012 98.4381 109.817 99.2647C109.622 100.091 109.677 100.957 109.975 101.752C110.272 102.548 110.799 103.237 111.488 103.733C112.177 104.23 112.998 104.51 113.846 104.54L113.847 104.54C114.6 104.567 115.347 104.395 116.011 104.04C116.31 103.881 116.682 103.994 116.841 104.293C117.001 104.591 116.888 104.963 116.589 105.123C115.733 105.579 114.772 105.801 113.803 105.766L113.825 105.153L113.804 105.766C113.803 105.766 113.803 105.766 113.803 105.766C112.712 105.728 111.657 105.367 110.771 104.729C109.885 104.091 109.208 103.205 108.825 102.182C108.443 101.159 108.373 100.046 108.623 98.9835C108.873 97.9208 109.433 96.956 110.231 96.2114C111.03 95.4668 112.031 94.9757 113.109 94.8001ZM115.563 100.304C115.577 99.888 115.426 99.4838 115.143 99.1798C114.859 98.8757 114.466 98.6967 114.051 98.6822C113.636 98.6677 113.231 98.8189 112.927 99.1024C112.623 99.3859 112.444 99.7786 112.43 100.194C112.415 100.61 112.566 101.014 112.85 101.318C113.133 101.622 113.526 101.801 113.942 101.815C114.357 101.83 114.761 101.679 115.065 101.395C115.369 101.112 115.548 100.719 115.563 100.304Z" fill="#59AB46"/> <rect x="126.717" y="97.5381" width="30.4939" height="7.3606" rx="1.05151" transform="rotate(2 126.717 97.5381)" fill="#B6DDAD"/> </g> </g> <g filter="url(#taggedpostfilter1_ddd)"> <g clip-path="url(#taggedpostclip3)"> <rect x="70.8867" y="10.8984" width="83.0697" height="84.1212" rx="2.10303" transform="rotate(-2 70.8867 10.8984)" fill="white"/> <g clip-path="url(#taggedpostclip4)"> <path d="M70.2217 4.99609L154.292 2.06031L157.007 79.825L72.9373 82.7608L70.2217 4.99609Z" fill="#43A6DB"/> <circle cx="169.299" cy="72.169" r="48.8954" transform="rotate(-2 169.299 72.169)" fill="#0068A0"/> </g> <path fill-rule="evenodd" clip-rule="evenodd" d="M84.2452 75.8962C85.308 75.646 86.4211 75.7165 87.4438 76.0989C88.4665 76.4813 89.3529 77.1583 89.9908 78.0444C90.6287 78.9305 90.9895 79.9859 91.0276 81.0771L91.0562 81.8944C91.0745 82.4183 90.8839 82.928 90.5264 83.3114C90.1689 83.6947 89.6738 83.9204 89.1499 83.9387C88.626 83.957 88.1163 83.7664 87.733 83.4089C87.6615 83.3423 87.5956 83.2709 87.5354 83.1954C87.0315 83.7253 86.3396 84.0368 85.6081 84.0623C84.8674 84.0882 84.1468 83.8188 83.6048 83.3134C83.0628 82.8079 82.7438 82.1079 82.7179 81.3673C82.6921 80.6266 82.9615 79.906 83.4669 79.364C83.9723 78.822 84.6724 78.503 85.413 78.4771C86.1537 78.4513 86.8742 78.7207 87.4162 79.2261C87.9583 79.7315 88.2773 80.4316 88.3031 81.1722L88.3317 81.9896C88.3386 82.1883 88.4242 82.3761 88.5696 82.5117C88.715 82.6473 88.9084 82.7196 89.1071 82.7126C89.3058 82.7057 89.4936 82.6201 89.6292 82.4747C89.7648 82.3293 89.8371 82.136 89.8301 81.9372L89.8016 81.1199C89.772 80.2712 89.4913 79.4504 88.9952 78.7612C88.499 78.072 87.8096 77.5454 87.0142 77.248C86.2188 76.9506 85.353 76.8957 84.5264 77.0904C83.6998 77.285 82.9495 77.7204 82.3703 78.3415C81.7912 78.9625 81.4092 79.7414 81.2727 80.5796C81.1362 81.4177 81.2513 82.2776 81.6034 83.0503C81.9556 83.8231 82.529 84.474 83.2511 84.9209C83.9733 85.3678 84.8117 85.5905 85.6604 85.5608L85.661 85.5608C86.4142 85.5352 87.147 85.3114 87.7851 84.9114C88.0721 84.7314 88.4506 84.8182 88.6306 85.1052C88.8105 85.3922 88.7237 85.7708 88.4367 85.9507C87.6149 86.466 86.6715 86.754 85.7026 86.7869L85.6818 86.1738L85.7032 86.7868C85.703 86.7868 85.7028 86.7869 85.7026 86.7869C84.6116 86.8248 83.5339 86.5385 82.6056 85.9641C81.6771 85.3895 80.9399 84.5526 80.4871 83.559C80.0344 82.5655 79.8864 81.46 80.0619 80.3824C80.2374 79.3047 80.7285 78.3033 81.4731 77.5048C82.2178 76.7063 83.1825 76.1465 84.2452 75.8962ZM87.0771 81.215C87.0626 80.7996 86.8836 80.4069 86.5796 80.1233C86.2755 79.8398 85.8713 79.6887 85.4558 79.7032C85.0403 79.7177 84.6476 79.8966 84.3641 80.2007C84.0806 80.5047 83.9294 80.909 83.944 81.3245C83.9585 81.7399 84.1374 82.1326 84.4415 82.4162C84.7455 82.6997 85.1498 82.8508 85.5652 82.8363C85.9807 82.8218 86.3734 82.6429 86.657 82.3388C86.9405 82.0348 87.0916 81.6305 87.0771 81.215Z" fill="#0068A0"/> <rect x="98.0117" y="77.6768" width="30.4939" height="7.3606" rx="1.05151" transform="rotate(-2 98.0117 77.6768)" fill="#86D0F9"/> </g> </g> </g> <defs> <filter id="taggedpostfilter0_ddd" x="87.7112" y="21.6697" width="113.294" height="114.308" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"> <feFlood flood-opacity="0" result="BackgroundImageFix"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="6.30909"/> <feGaussianBlur stdDeviation="6.83485"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/> <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="1.05151"/> <feGaussianBlur stdDeviation="1.05151"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/> <feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="3.15454"/> <feGaussianBlur stdDeviation="3.15454"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/> <feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/> <feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow" result="shape"/> </filter> <filter id="taggedpostfilter1_ddd" x="57.217" y="0.638418" width="113.294" height="114.308" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"> <feFlood flood-opacity="0" result="BackgroundImageFix"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="6.30909"/> <feGaussianBlur stdDeviation="6.83485"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/> <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="1.05151"/> <feGaussianBlur stdDeviation="1.05151"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/> <feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="3.15454"/> <feGaussianBlur stdDeviation="3.15454"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/> <feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/> <feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow" result="shape"/> </filter> <clipPath id="taggedpostclip0"> <rect width="259.056" height="126" fill="white" transform="translate(0.111328)"/> </clipPath> <clipPath id="taggedpostclip1"> <rect x="104.316" y="29.0303" width="83.0697" height="84.1212" rx="2.10303" transform="rotate(2 104.316 29.0303)" fill="white"/> </clipPath> <clipPath id="taggedpostclip2"> <path d="M104.187 19.5933C104.221 18.626 105.032 17.8692 106 17.903L186.567 20.7164C187.534 20.7502 188.291 21.5617 188.257 22.529L185.896 90.1353L101.826 87.1995L104.187 19.5933Z" fill="white"/> </clipPath> <clipPath id="taggedpostclip3"> <rect x="70.8867" y="10.8984" width="83.0697" height="84.1212" rx="2.10303" transform="rotate(-2 70.8867 10.8984)" fill="white"/> </clipPath> <clipPath id="taggedpostclip4"> <path d="M70.0983 1.49365C70.0645 0.526345 70.8213 -0.285196 71.7886 -0.318975L152.356 -3.13244C153.323 -3.16622 154.134 -2.40945 154.168 -1.44214L156.529 66.1641L72.4591 69.0999L70.0983 1.49365Z" fill="white"/> </clipPath> </defs> </svg>',
			'socialwall1Icon'     => '<svg width="260" height="126" viewBox="0 0 260 126" fill="none" xmlns="http://www.w3.org/2000/svg"> <g filter="url(#social1filter0_ddd)"> <rect x="44.416" y="44.9111" width="42" height="42" rx="2.10303" transform="rotate(-5 44.416 44.9111)" fill="white"/> <path d="M66.2979 54.0796C60.8188 54.559 56.7273 59.4241 57.2092 64.933C57.645 69.914 61.6528 73.7292 66.48 74.0598L65.8699 67.0864L63.3395 67.3078L63.0868 64.4188L65.6171 64.1974L65.4245 61.9959C65.2057 59.4954 66.5698 57.9908 68.8511 57.7912C69.9369 57.6962 71.0892 57.7861 71.0892 57.7861L71.3044 60.2467L70.0492 60.3565C68.8139 60.4646 68.4925 61.2657 68.5614 62.0527L68.7252 63.9255L71.4947 63.6832L71.2991 66.6114L68.978 66.8145L69.5881 73.7878C71.9031 73.2117 73.9359 71.827 75.3195 69.8835C76.7031 67.9401 77.3464 65.566 77.1331 63.1899C76.6512 57.681 71.777 53.6003 66.2979 54.0796Z" fill="#006BFA"/> </g> <g filter="url(#social1filter1_ddd)"> <rect x="83.0967" y="39.1279" width="42" height="42" rx="2.10303" transform="rotate(-3 83.0967 39.1279)" fill="white"/> <path d="M104.886 53.6171C101.89 53.7741 99.6299 56.3334 99.7844 59.2824C99.9414 62.2783 102.454 64.5406 105.45 64.3836C108.399 64.229 110.708 61.7141 110.551 58.7182C110.396 55.7691 107.835 53.4625 104.886 53.6171ZM105.352 62.5111C103.432 62.6117 101.76 61.1504 101.657 59.1843C101.556 57.2651 103.02 55.6394 104.986 55.5363C106.905 55.4357 108.531 56.8995 108.632 58.8188C108.735 60.7848 107.271 62.4105 105.352 62.5111ZM111.71 53.0717C111.673 52.3695 111.082 51.8372 110.38 51.874C109.678 51.9108 109.146 52.502 109.182 53.2041C109.219 53.9063 109.81 54.4386 110.512 54.4018C111.215 54.365 111.747 53.7738 111.71 53.0717ZM115.334 54.1491C115.152 52.4688 114.699 50.9905 113.418 49.8372C112.137 48.6839 110.62 48.3879 108.93 48.3826C107.193 48.3798 101.997 48.6521 100.27 48.8365C98.5894 49.0184 97.1579 49.469 95.9578 50.7523C94.8045 52.0331 94.5085 53.5507 94.5032 55.2408C94.5003 56.9777 94.7726 62.1737 94.957 63.9008C95.139 65.5811 95.5895 67.0126 96.8728 68.2127C98.2005 69.3635 99.6712 69.662 101.361 69.6673C103.098 69.6701 108.294 69.3978 110.021 69.2134C111.702 69.0315 113.18 68.5785 114.333 67.2976C115.484 65.97 115.783 64.4992 115.788 62.8091C115.791 61.0722 115.518 55.8762 115.334 54.1491ZM113.637 64.7525C113.358 65.7059 112.646 66.4473 111.776 66.8684C110.401 67.5037 107.117 67.535 105.619 67.6135C104.074 67.6945 100.805 68.0066 99.418 67.516C98.4621 67.1906 97.7232 66.5252 97.2996 65.6087C96.6667 64.2806 96.6354 60.9965 96.5545 59.4517C96.476 57.9538 96.1638 54.6844 96.652 53.2506C96.9798 52.3416 97.6452 51.6026 98.5618 51.1791C99.8899 50.5462 103.174 50.5149 104.719 50.4339C106.217 50.3554 109.486 50.0433 110.92 50.5314C111.826 50.8125 112.568 51.5247 112.989 52.3944C113.624 53.7693 113.656 57.0534 113.734 58.5514C113.815 60.0961 114.127 63.3655 113.637 64.7525Z" fill="url(#social1paint0_linear)"/> </g> <g filter="url(#social1filter2_ddd)"> <rect x="122.913" y="35.2803" width="42" height="42" rx="2.10303" transform="rotate(2 122.913 35.2803)" fill="white"/> <path d="M153.831 51.3695C153.049 51.6924 152.211 51.8933 151.348 51.9732C152.246 51.4743 152.955 50.6585 153.31 49.6603C152.463 50.131 151.531 50.4487 150.555 50.6147C149.795 49.7277 148.704 49.1892 147.444 49.1453C145.096 49.0633 143.11 50.9151 143.027 53.2836C143.015 53.6234 143.044 53.9546 143.103 54.2669C139.551 53.9627 136.443 52.1432 134.425 49.4811C134.033 50.0978 133.797 50.83 133.77 51.6095C133.718 53.0986 134.421 54.444 135.555 55.234C134.845 55.2093 134.192 54.9863 133.623 54.6663L133.622 54.6963C133.55 56.775 134.968 58.5656 136.913 59.0238C136.278 59.1739 135.617 59.1748 134.982 59.0264C135.224 59.8878 135.729 60.6518 136.428 61.2111C137.126 61.7703 137.982 62.0966 138.875 62.1441C137.318 63.2909 135.417 63.8738 133.485 63.797C133.145 63.7851 132.806 63.7533 132.467 63.7014C134.323 64.987 136.557 65.7755 138.976 65.8599C146.851 66.1349 151.407 59.75 151.605 54.0835C151.611 53.8936 151.617 53.7137 151.614 53.5235C152.475 52.9531 153.221 52.2187 153.831 51.3695Z" fill="#1B90EF"/> </g> <g filter="url(#social1filter3_ddd)"> <rect x="161.295" y="39.9297" width="42" height="42" rx="2.10303" transform="rotate(3 161.295 39.9297)" fill="white"/> <path d="M179.013 64.8913L184.352 62.167L179.327 58.8995L179.013 64.8913ZM190.966 57.677C191.072 58.1532 191.129 58.7871 191.147 59.5891C191.175 60.3917 191.169 61.0823 191.137 61.6815L191.153 62.5235C191.038 64.7105 190.794 66.3099 190.461 67.3238C190.164 68.2095 189.555 68.7583 188.643 68.9609C188.167 69.0661 187.303 69.111 185.982 69.1018C184.68 69.1037 183.49 69.0714 182.391 69.0138L180.8 68.9905C176.616 68.7712 174.018 68.4748 173.004 68.1413C172.119 67.8446 171.57 67.235 171.367 66.3231C171.262 65.847 171.205 65.2131 171.187 64.4111C171.159 63.6085 171.165 62.9179 171.196 62.3187L171.181 61.4767C171.295 59.2897 171.539 57.6903 171.873 56.6764C172.169 55.7907 172.779 55.2418 173.691 55.0393C174.167 54.9341 175.031 54.8892 176.352 54.8984C177.654 54.8965 178.844 54.9288 179.942 54.9864L181.533 55.0097C185.717 55.229 188.315 55.5254 189.329 55.8589C190.215 56.1556 190.764 56.7652 190.966 57.677Z" fill="#EB2121"/> </g> <defs> <filter id="social1filter0_ddd" x="30.7463" y="33.8904" width="72.8401" height="72.8401" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"> <feFlood flood-opacity="0" result="BackgroundImageFix"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="6.30909"/> <feGaussianBlur stdDeviation="6.83485"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/> <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="1.05151"/> <feGaussianBlur stdDeviation="1.05151"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/> <feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="3.15454"/> <feGaussianBlur stdDeviation="3.15454"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/> <feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/> <feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow" result="shape"/> </filter> <filter id="social1filter1_ddd" x="69.427" y="29.5691" width="71.4799" height="71.4799" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"> <feFlood flood-opacity="0" result="BackgroundImageFix"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="6.30909"/> <feGaussianBlur stdDeviation="6.83485"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/> <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="1.05151"/> <feGaussianBlur stdDeviation="1.05151"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/> <feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="3.15454"/> <feGaussianBlur stdDeviation="3.15454"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/> <feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/> <feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow" result="shape"/> </filter> <filter id="social1filter2_ddd" x="107.778" y="27.9197" width="70.7796" height="70.7796" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"> <feFlood flood-opacity="0" result="BackgroundImageFix"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="6.30909"/> <feGaussianBlur stdDeviation="6.83485"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/> <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="1.05151"/> <feGaussianBlur stdDeviation="1.05151"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/> <feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="3.15454"/> <feGaussianBlur stdDeviation="3.15454"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/> <feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/> <feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow" result="shape"/> </filter> <filter id="social1filter3_ddd" x="145.427" y="32.5691" width="71.4799" height="71.4799" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"> <feFlood flood-opacity="0" result="BackgroundImageFix"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="6.30909"/> <feGaussianBlur stdDeviation="6.83485"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/> <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="1.05151"/> <feGaussianBlur stdDeviation="1.05151"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/> <feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="3.15454"/> <feGaussianBlur stdDeviation="3.15454"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/> <feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/> <feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow" result="shape"/> </filter> <linearGradient id="social1paint0_linear" x1="103.683" y1="88.8048" x2="145.491" y2="41.4018" gradientUnits="userSpaceOnUse"> <stop stop-color="white"/> <stop offset="0.147864" stop-color="#F6640E"/> <stop offset="0.443974" stop-color="#BA03A7"/> <stop offset="0.733337" stop-color="#6A01B9"/> <stop offset="1" stop-color="#6B01B9"/> </linearGradient> </defs> </svg>',

			'user'                => '<svg width="8" height="8" viewBox="0 0 8 8" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M4 0C4.53043 0 5.03914 0.210714 5.41421 0.585786C5.78929 0.960859 6 1.46957 6 2C6 2.53043 5.78929 3.03914 5.41421 3.41421C5.03914 3.78929 4.53043 4 4 4C3.46957 4 2.96086 3.78929 2.58579 3.41421C2.21071 3.03914 2 2.53043 2 2C2 1.46957 2.21071 0.960859 2.58579 0.585786C2.96086 0.210714 3.46957 0 4 0ZM4 5C6.21 5 8 5.895 8 7V8H0V7C0 5.895 1.79 5 4 5Z"/></svg>',
			'hashtag'             => '<svg viewBox="0 0 18 18" fill="none"><path d="M17.3607 4.1775H14.0152L14.618 1.266C14.6328 1.18021 14.6288 1.09223 14.6064 1.00812C14.5839 0.924001 14.5436 0.845742 14.488 0.778722C14.4324 0.711703 14.363 0.657514 14.2845 0.619882C14.206 0.582251 14.1203 0.56207 14.0332 0.560727H12.8276C12.6883 0.557321 12.5521 0.602311 12.4422 0.688037C12.3323 0.773763 12.2555 0.894929 12.2249 1.03091L11.61 4.1775H8.3549L8.9577 1.266C8.97229 1.18215 8.96897 1.09617 8.94795 1.0137C8.92692 0.931226 8.88867 0.854142 8.83572 0.787518C8.78276 0.720894 8.71629 0.666239 8.64069 0.62715C8.56509 0.588061 8.48207 0.565423 8.3971 0.560727H7.1915C7.05216 0.557321 6.91594 0.602311 6.80604 0.688037C6.69613 0.773763 6.61933 0.894929 6.58871 1.03091L5.98591 4.1775H2.36914C2.22811 4.17466 2.09056 4.22136 1.98042 4.30947C1.87028 4.39759 1.79452 4.52153 1.76634 4.65974L1.51919 5.86533C1.50109 5.95393 1.50315 6.04546 1.52522 6.13316C1.5473 6.22085 1.58882 6.30245 1.64671 6.37192C1.7046 6.44139 1.77737 6.49694 1.85965 6.53446C1.94192 6.57199 2.03158 6.59052 2.12199 6.58869H5.46751L4.47892 11.4111H0.862146C0.721125 11.4082 0.583571 11.4549 0.473429 11.543C0.363287 11.6311 0.287532 11.7551 0.259351 11.8933L0.0122042 13.0989C-0.00589975 13.1875 -0.00383898 13.279 0.0182337 13.3667C0.0403064 13.4544 0.0818254 13.536 0.139715 13.6055C0.197605 13.6749 0.270382 13.7305 0.352656 13.768C0.43493 13.8055 0.524592 13.8241 0.615 13.8222H3.98463L3.38183 16.7338C3.36677 16.821 3.37112 16.9106 3.39459 16.996C3.41806 17.0814 3.46006 17.1606 3.51761 17.2279C3.57517 17.2953 3.64685 17.3491 3.72757 17.3856C3.80829 17.4221 3.89606 17.4403 3.98463 17.439H5.19022C5.3244 17.4356 5.45359 17.3875 5.55732 17.3023C5.66105 17.2171 5.73339 17.0998 5.76288 16.9688L6.38979 13.8222H9.64488L9.04209 16.7338C9.02749 16.8176 9.03081 16.9036 9.05184 16.9861C9.07286 17.0685 9.11111 17.1456 9.16407 17.2122C9.21702 17.2789 9.28349 17.3335 9.35909 17.3726C9.43469 17.4117 9.51771 17.4343 9.60269 17.439H10.8083C10.9476 17.4424 11.0838 17.3974 11.1937 17.3117C11.3037 17.226 11.3805 17.1048 11.4111 16.9688L12.044 13.8222H15.6608C15.8018 13.8251 15.9394 13.7784 16.0495 13.6903C16.1596 13.6022 16.2354 13.4782 16.2636 13.34L16.5047 12.1344C16.5228 12.0458 16.5207 11.9543 16.4987 11.8666C16.4766 11.7789 16.4351 11.6973 16.3772 11.6278C16.3193 11.5584 16.2465 11.5028 16.1642 11.4653C16.082 11.4278 15.9923 11.4092 15.9019 11.4111H12.5383L13.5209 6.58869H17.1376C17.2787 6.59153 17.4162 6.54483 17.5264 6.45672C17.6365 6.36861 17.7123 6.24466 17.7404 6.10645L17.9876 4.90086C18.0063 4.8102 18.0038 4.71645 17.9804 4.62689C17.957 4.53733 17.9133 4.45436 17.8527 4.3844C17.7921 4.31445 17.7162 4.2594 17.6308 4.22352C17.5455 4.18764 17.4531 4.1719 17.3607 4.1775ZM10.1271 11.4111H6.87202L7.86061 6.58869H11.1157L10.1271 11.4111Z"/></svg>',
			'mention'             => '<svg viewBox="0 0 18 18"><path fill-rule="evenodd" clip-rule="evenodd" d="M7.24419 0.172937C8.99002 -0.174331 10.7996 0.00389957 12.4442 0.685088C14.0887 1.36628 15.4943 2.51983 16.4832 3.99987C17.4722 5.47992 18 7.21997 18 9.00001V10.3333C18 11.1879 17.6605 12.0075 17.0562 12.6118C16.452 13.2161 15.6324 13.5556 14.7778 13.5556C13.9232 13.5556 13.1036 13.2161 12.4993 12.6118C12.3867 12.4992 12.2833 12.3791 12.1896 12.2527C11.3384 13.0874 10.1933 13.5556 9.00001 13.5556C7.7918 13.5556 6.63307 13.0756 5.77874 12.2213C4.92441 11.3669 4.44445 10.2082 4.44445 9.00001C4.44445 7.7918 4.92441 6.63307 5.77874 5.77874C6.63307 4.92441 7.7918 4.44445 9.00001 4.44445C10.2082 4.44445 11.3669 4.92441 12.2213 5.77874C13.0756 6.63307 13.5556 7.7918 13.5556 9.00001V10.3333C13.5556 10.6575 13.6843 10.9684 13.9135 11.1976C14.1428 11.4268 14.4536 11.5556 14.7778 11.5556C15.1019 11.5556 15.4128 11.4268 15.642 11.1976C15.8712 10.9684 16 10.6575 16 10.3333V9.00001C16 7.61554 15.5895 6.26216 14.8203 5.11101C14.0511 3.95987 12.9579 3.06266 11.6788 2.53285C10.3997 2.00303 8.99224 1.86441 7.63437 2.13451C6.27651 2.4046 5.02922 3.07129 4.05026 4.05026C3.07129 5.02922 2.4046 6.27651 2.13451 7.63437C1.86441 8.99224 2.00303 10.3997 2.53285 11.6788C3.06266 12.9579 3.95987 14.0511 5.11101 14.8203C6.26216 15.5895 7.61554 16 9.00001 16L9.001 16C10.2297 16.0012 11.4363 15.6782 12.4987 15.0627C12.9766 14.7859 13.5884 14.9488 13.8653 15.4267C14.1421 15.9046 13.9792 16.5164 13.5013 16.7933C12.1329 17.586 10.5796 18.0016 8.99901 18L9.00001 17V18C8.99968 18 8.99934 18 8.99901 18C7.21933 17.9998 5.47964 17.472 3.99987 16.4832C2.51983 15.4943 1.36628 14.0887 0.685088 12.4442C0.00389957 10.7996 -0.17433 8.99002 0.172936 7.24419C0.520204 5.49836 1.37737 3.89472 2.63604 2.63604C3.89472 1.37737 5.49836 0.520204 7.24419 0.172937ZM11.5556 9.00001C11.5556 8.32223 11.2863 7.67221 10.8071 7.19295C10.3278 6.7137 9.67778 6.44445 9.00001 6.44445C8.32223 6.44445 7.67221 6.7137 7.19295 7.19295C6.7137 7.67221 6.44445 8.32223 6.44445 9.00001C6.44445 9.67778 6.7137 10.3278 7.19295 10.8071C7.67221 11.2863 8.32223 11.5556 9.00001 11.5556C9.67778 11.5556 10.3278 11.2863 10.8071 10.8071C11.2863 10.3278 11.5556 9.67778 11.5556 9.00001Z"/></svg>',
			'tooltipHelpSvg'      => '<svg width="20" height="21" viewBox="0 0 20 21" fill="#0068A0" xmlns="http://www.w3.org/2000/svg"><path d="M9.1665 8H10.8332V6.33333H9.1665V8ZM9.99984 17.1667C6.32484 17.1667 3.33317 14.175 3.33317 10.5C3.33317 6.825 6.32484 3.83333 9.99984 3.83333C13.6748 3.83333 16.6665 6.825 16.6665 10.5C16.6665 14.175 13.6748 17.1667 9.99984 17.1667ZM9.99984 2.16666C8.90549 2.16666 7.82186 2.38221 6.81081 2.801C5.79976 3.21979 4.8811 3.83362 4.10728 4.60744C2.54448 6.17024 1.6665 8.28986 1.6665 10.5C1.6665 12.7101 2.54448 14.8298 4.10728 16.3926C4.8811 17.1664 5.79976 17.7802 6.81081 18.199C7.82186 18.6178 8.90549 18.8333 9.99984 18.8333C12.21 18.8333 14.3296 17.9554 15.8924 16.3926C17.4552 14.8298 18.3332 12.7101 18.3332 10.5C18.3332 9.40565 18.1176 8.32202 17.6988 7.31097C17.28 6.29992 16.6662 5.38126 15.8924 4.60744C15.1186 3.83362 14.1999 3.21979 13.1889 2.801C12.1778 2.38221 11.0942 2.16666 9.99984 2.16666ZM9.1665 14.6667H10.8332V9.66666H9.1665V14.6667Z" fill="#0068A0"/></svg>',

			'shoppableDisabled'   => '<svg width="303" height="145" viewBox="0 0 303 145" fill="none" xmlns="http://www.w3.org/2000/svg"> <path d="M124.919 67.2058C130.919 72.7058 150.519 81.4058 180.919 72.2058" stroke="#8C8F9A" stroke-width="2" stroke-dasharray="3 3"/> <path d="M181.999 69L185.797 70.4241L183.5 74L181.999 69Z" fill="#8C8F9A"/> <g filter="url(#shopp_disabled_filter0_dddd)"> <rect x="24.6006" y="17.6504" width="81" height="98" rx="2" transform="rotate(-4 24.6006 17.6504)" fill="white"/> <rect x="24.3338" y="17.4184" width="81.5" height="98.5" rx="2.25" transform="rotate(-4 24.3338 17.4184)" stroke="url(#shopp_disabled_paint0_linear)" stroke-width="0.5"/> </g> <g clip-path="url(#shopp_disabled_clip0)"> <path d="M94.5298 21.3615C92.9088 21.4749 91.7091 22.8823 91.8207 24.478C91.9341 26.0991 93.3162 27.3005 94.9372 27.1872C96.5329 27.0756 97.7597 25.6917 97.6463 24.0707C97.5348 22.4749 96.1256 21.2499 94.5298 21.3615ZM94.8664 26.174C93.8279 26.2466 92.9083 25.471 92.8339 24.4072C92.7613 23.3687 93.5387 22.4744 94.6025 22.4C95.6409 22.3274 96.5352 23.1048 96.6079 24.1433C96.6822 25.2071 95.9048 26.1014 94.8664 26.174ZM98.2208 21.0016C98.1942 20.6217 97.869 20.339 97.4891 20.3656C97.1091 20.3921 96.8264 20.7173 96.853 21.0973C96.8796 21.4772 97.2048 21.7599 97.5847 21.7333C97.9646 21.7068 98.2473 21.3816 98.2208 21.0016ZM100.194 21.5509C100.079 20.6426 99.8198 19.8463 99.1152 19.2338C98.4106 18.6213 97.586 18.4753 96.6706 18.4884C95.7299 18.5033 92.9184 18.6999 91.9848 18.8161C91.0765 18.9305 90.3054 19.188 89.6676 19.8944C89.0551 20.599 88.9092 21.4237 88.9223 22.3391C88.9371 23.2798 89.1337 26.0913 89.2499 27.0249C89.3644 27.9332 89.6219 28.7042 90.3283 29.342C91.0582 29.9528 91.8575 30.1005 92.7729 30.0874C93.7136 30.0725 96.5251 29.8759 97.4587 29.7597C98.367 29.6453 99.1634 29.386 99.7759 28.6814C100.387 27.9515 100.534 27.1521 100.521 26.2367C100.506 25.296 100.31 22.4845 100.194 21.5509ZM99.3745 27.3096C99.2327 27.8285 98.854 28.2368 98.3869 28.4731C97.6483 28.8302 95.8699 28.8782 95.0594 28.9348C94.2236 28.9933 92.4559 29.1933 91.7001 28.9407C91.1793 28.7735 90.7728 28.4201 90.5348 27.9277C90.1795 27.2144 90.1315 25.4361 90.073 24.6002C90.0164 23.7897 89.8164 22.022 90.0672 21.2409C90.2362 20.7455 90.5895 20.339 91.082 20.1009C91.7952 19.7456 93.5736 19.6976 94.4094 19.6392C95.2199 19.5825 96.9876 19.3825 97.7687 19.6333C98.2624 19.777 98.6707 20.1557 98.9069 20.6228C99.264 21.3614 99.312 23.1397 99.3687 23.9502C99.4271 24.7861 99.6271 26.5538 99.3745 27.3096Z" fill="url(#shopp_disabled_paint1_linear)"/> </g> <g clip-path="url(#shopp_disabled_clip1)"> <rect x="26.1348" y="39.5967" width="81" height="76" rx="2" transform="rotate(-4 26.1348 39.5967)" fill="#B5E5FF"/> <circle cx="30.7388" cy="105.436" r="54" transform="rotate(-4 30.7388 105.436)" fill="#86D0F9"/> <g filter="url(#shopp_disabled_filter1_dd)"> <mask id="shopp_disabled_mask0" style="mask-type:alpha" maskUnits="userSpaceOnUse" x="35" y="47" width="60" height="54"> <path fill-rule="evenodd" clip-rule="evenodd" d="M68.7966 50.3478C68.534 50.4332 68.3943 50.7154 68.4401 50.9877C68.8644 53.5073 66.4327 56.3732 62.7333 57.5753C59.0338 58.7773 55.382 57.888 54.2442 55.6002C54.1213 55.3529 53.8423 55.2068 53.5797 55.2921L47.2555 57.347C47.1786 57.372 47.109 57.4152 47.0525 57.473L42.6186 62.008L35.8445 69.2862C35.7004 69.441 35.6693 69.6698 35.7668 69.8574L40.9681 79.8652C41.1015 80.1217 41.4239 80.212 41.6711 80.0621L47.8083 76.3393C48.0715 76.1797 48.4151 76.2935 48.5309 76.5788L58.2754 100.594C58.374 100.837 58.6437 100.963 58.8932 100.881L92.2457 90.0446C92.4952 89.9635 92.6396 89.7034 92.5765 89.4488L86.3412 64.2801C86.2678 63.9837 86.4749 63.6913 86.7789 63.6622L94.424 62.9299C94.7094 62.9026 94.9134 62.6414 94.8708 62.358L93.1967 51.2062C93.1647 50.9929 92.9995 50.8242 92.787 50.7877L82.5629 49.0293L76.3102 47.9666C76.2305 47.953 76.1488 47.959 76.0719 47.984L68.7966 50.3478Z" fill="white"/> </mask> <g mask="url(#shopp_disabled_mask0)"> <rect x="28.3076" y="60.3479" width="72" height="54" transform="rotate(-16 28.3076 60.3479)" fill="white"/> <path fill-rule="evenodd" clip-rule="evenodd" d="M66.4321 69.6639C65.1395 69.4776 63.7264 69.0512 62.5105 69.0127C63.1766 69.8427 63.7987 70.7521 64.429 71.6465C63.8884 72.3619 63.1987 72.9948 62.5553 73.6533C63.3952 74.1125 64.4294 74.2212 65.3292 74.5723C64.947 75.4717 64.0024 76.5635 63.9089 77.3062C65.0894 76.8017 66.445 76.1437 67.5698 75.7666C68.181 76.9532 68.7057 78.2958 69.3922 79.3464C69.485 77.6689 69.5124 75.9552 69.7351 74.3498C70.8246 74.4733 72.1524 74.6242 73.1713 74.589C72.2358 73.8444 71.3419 73.0247 70.4606 72.1824C71.1537 71.2976 71.8595 70.42 72.5116 69.5125C71.2887 69.7444 70.035 70.0316 68.7692 70.3408C68.2001 69.1068 67.8102 67.5497 67.1648 66.4536C66.98 67.5567 66.688 68.6002 66.4321 69.6639ZM70.0641 80.1946C70.0998 80.9132 70.6974 81.0407 70.7363 81.4713C70.1738 81.4417 69.7628 81.4615 69.1035 81.7558C68.9743 81.2243 69.4256 81.0144 69.1426 80.3976C61.5808 81.6649 57.7717 68.4365 64.8194 65.5342C73.6314 61.9053 78.4249 77.5439 70.0641 80.1946Z" fill="#FE544F"/> <path fill-rule="evenodd" clip-rule="evenodd" d="M67.1649 66.4536C67.8103 67.5497 68.2003 69.1068 68.7693 70.3407C70.0352 70.0316 71.2888 69.7444 72.5117 69.5125C71.8597 70.42 71.1538 71.2976 70.4608 72.1824C71.3421 73.0248 72.2359 73.8444 73.1714 74.589C72.1526 74.6242 70.8247 74.4733 69.7352 74.3498C69.5126 75.9552 69.4852 77.6689 69.3924 79.3464C68.7058 78.2958 68.1811 76.9532 67.5699 75.7666C66.4451 76.1438 65.0896 76.8017 63.9091 77.3062C64.0026 76.5635 64.9472 75.4718 65.3294 74.5723C64.4295 74.2212 63.3954 74.1125 62.5555 73.6533C63.1989 72.9948 63.8885 72.362 64.4292 71.6465C63.7988 70.7521 63.1767 69.8427 62.5106 69.0128C63.7266 69.0512 65.1396 69.4776 66.4323 69.6639C66.6881 68.6002 66.9802 67.5567 67.1649 66.4536Z" fill="white"/> </g> </g> </g> <g filter="url(#shopp_disabled_filter2_dddd)"> <rect x="199.592" y="17.7058" width="79" height="102" rx="2" transform="rotate(4 199.592 17.7058)" fill="#E2F5FF"/> </g> <rect x="231.919" y="100.162" width="36" height="17" rx="2" transform="rotate(4 231.919 100.162)" fill="#0096CC"/> <path d="M241.707 111.873L244.07 112.038C245.123 112.112 245.827 111.602 245.887 110.743L245.888 110.736C245.931 110.112 245.469 109.576 244.827 109.497L244.831 109.432C245.358 109.397 245.785 108.978 245.821 108.453L245.822 108.446C245.875 107.686 245.328 107.182 244.346 107.113L242.051 106.953L241.707 111.873ZM243.95 107.973C244.376 108.003 244.61 108.232 244.586 108.579L244.585 108.586C244.561 108.931 244.281 109.123 243.824 109.091L243.162 109.045L243.241 107.923L243.95 107.973ZM243.859 109.858C244.377 109.894 244.652 110.136 244.624 110.538L244.623 110.545C244.594 110.958 244.295 111.166 243.777 111.13L243.02 111.077L243.109 109.805L243.859 109.858ZM248.86 112.507C250.155 112.597 251.031 111.925 251.108 110.824L251.334 107.602L250.086 107.515L249.869 110.617C249.829 111.19 249.498 111.51 248.935 111.47C248.376 111.431 248.09 111.069 248.13 110.496L248.347 107.393L247.099 107.306L246.874 110.528C246.796 111.633 247.581 112.417 248.86 112.507ZM253.583 112.703L254.834 112.791L254.952 111.1L256.873 107.989L255.539 107.896L254.448 109.838L254.383 109.833L253.565 107.758L252.232 107.665L253.701 111.012L253.583 112.703Z" fill="white"/> <g filter="url(#shopp_disabled_filter3_dd)"> <mask id="shopp_disabled_mask1" style="mask-type:alpha" maskUnits="userSpaceOnUse" x="207" y="35" width="61" height="48"> <path fill-rule="evenodd" clip-rule="evenodd" d="M244.802 36.7068C244.526 36.6972 244.298 36.9146 244.248 37.1861C243.785 39.699 240.52 41.5604 236.632 41.4246C232.745 41.2889 229.618 39.2042 229.331 36.6652C229.3 36.3908 229.088 36.1581 228.812 36.1485L222.166 35.9164C222.085 35.9136 222.005 35.9304 221.932 35.9653L216.215 38.7104L207.36 43.2328C207.171 43.329 207.064 43.5333 207.091 43.743L208.556 54.9261C208.594 55.2128 208.866 55.408 209.149 55.3516L216.19 53.9524C216.492 53.8924 216.776 54.117 216.787 54.4246L217.73 80.3242C217.74 80.5864 217.95 80.7966 218.212 80.8057L253.26 82.0296C253.522 82.0388 253.747 81.8438 253.774 81.5829L256.523 55.7995C256.556 55.4959 256.85 55.2919 257.146 55.3685L264.581 57.2952C264.858 57.3671 265.139 57.1915 265.196 56.9106L267.437 45.8588C267.48 45.6474 267.382 45.4324 267.195 45.3253L258.189 40.1762L252.677 37.039C252.607 36.999 252.528 36.9766 252.447 36.9738L244.802 36.7068Z" fill="white"/> </mask> <g mask="url(#shopp_disabled_mask1)"> <rect x="203.335" y="32.2556" width="72" height="54" transform="rotate(4 203.335 32.2556)" fill="white"/> <path fill-rule="evenodd" clip-rule="evenodd" d="M235.974 54.0491C234.823 53.4321 233.641 52.548 232.512 52.096C232.854 53.1038 233.128 54.171 233.414 55.2271C232.661 55.7145 231.797 56.0733 230.967 56.472C231.599 57.1908 232.534 57.6466 233.259 58.2843C232.592 58.9988 231.331 59.7017 230.99 60.3676C232.271 60.2973 233.77 60.1426 234.956 60.173C235.125 61.497 235.159 62.9381 235.444 64.1601C236.105 62.6156 236.717 61.0146 237.476 59.5821C238.457 60.0709 239.653 60.6668 240.623 60.9822C239.998 59.9626 239.439 58.8866 238.899 57.7936C239.852 57.1992 240.816 56.616 241.739 55.9862C240.511 55.7859 239.234 55.627 237.939 55.4846C237.826 54.1304 237.992 52.5338 237.761 51.2831C237.21 52.2564 236.579 53.1372 235.974 54.0491ZM235.786 65.187C235.573 65.8745 236.091 66.1987 235.981 66.6166C235.462 66.3964 235.069 66.2745 234.349 66.3255C234.409 65.7818 234.905 65.739 234.85 65.0626C227.311 63.6672 228.256 49.9337 235.871 49.6169C245.393 49.2208 244.549 65.5558 235.786 65.187Z" fill="#FE544F"/> <path fill-rule="evenodd" clip-rule="evenodd" d="M237.761 51.283C237.993 52.5337 237.827 54.1303 237.939 55.4844C239.235 55.6268 240.511 55.7857 241.739 55.9861C240.816 56.6159 239.853 57.1991 238.899 57.7935C239.439 58.8865 239.998 59.9624 240.623 60.9821C239.653 60.6667 238.457 60.0708 237.476 59.582C236.717 61.0145 236.106 62.6155 235.445 64.16C235.159 62.938 235.125 61.4969 234.956 60.1729C233.77 60.1425 232.272 60.2972 230.99 60.3675C231.332 59.7016 232.593 58.9987 233.259 58.2842C232.534 57.6465 231.599 57.1907 230.967 56.4719C231.797 56.0732 232.662 55.7144 233.414 55.227C233.128 54.1709 232.854 53.1037 232.512 52.0959C233.642 52.5479 234.824 53.432 235.975 54.049C236.579 53.1371 237.21 52.2563 237.761 51.283Z" fill="white"/> </g> </g> <path d="M266.144 121.304L266.2 120.51L265.32 120.449L265.375 119.655L263.615 119.532L263.67 118.739L261.03 118.554L261.085 117.761L259.325 117.637L259.547 114.463L258.666 114.402L258.722 113.608L256.962 113.485L256.906 114.279L256.026 114.217L255.526 121.359L254.646 121.297L254.702 120.504L252.061 120.319L251.839 123.493L252.719 123.555L252.608 125.142L253.489 125.203L253.378 126.79L254.258 126.852L254.147 128.439L255.027 128.501L254.861 130.881L264.543 131.558L264.765 128.384L265.645 128.446L265.811 126.065L264.931 126.003L264.765 128.384L263.885 128.322L263.718 130.703L255.796 130.149L255.907 128.562L255.027 128.501L255.138 126.913L254.258 126.852L254.369 125.265L253.489 125.203L253.6 123.616L252.719 123.555L252.886 121.174L254.646 121.297L254.591 122.091L255.471 122.152L255.305 124.533L256.185 124.594L256.906 114.279L258.666 114.402L258.167 121.544L259.047 121.605L259.269 118.431L261.03 118.554L260.808 121.728L261.688 121.79L261.854 119.409L263.615 119.532L263.393 122.706L264.273 122.768L264.439 120.387L265.32 120.449L265.264 121.242L266.144 121.304L265.811 126.065L266.692 126.127L267.025 121.365L266.144 121.304Z" fill="#141B38"/> <path d="M264.932 126.003L265.812 126.065L266.145 121.304L265.265 121.242L265.32 120.449L264.44 120.387L264.274 122.768L263.393 122.706L263.615 119.532L261.855 119.409L261.688 121.79L260.808 121.728L261.03 118.554L259.27 118.431L259.048 121.605L258.168 121.543L258.667 114.402L256.907 114.279L256.185 124.594L255.305 124.533L255.471 122.152L254.591 122.091L254.647 121.297L252.886 121.174L252.72 123.555L253.6 123.616L253.489 125.203L254.369 125.265L254.258 126.852L255.139 126.913L255.028 128.5L255.908 128.562L255.797 130.149L263.719 130.703L263.885 128.322L264.765 128.384L264.932 126.003Z" fill="white"/> <path fill-rule="evenodd" clip-rule="evenodd" d="M258.001 123.924L258.881 123.986L258.62 127.726L257.739 127.665L258.001 123.924ZM260.641 124.109L259.761 124.047L259.5 127.788L260.38 127.85L260.641 124.109ZM262.402 124.232L261.521 124.17L261.26 127.911L262.14 127.973L262.402 124.232Z" fill="#141B38"/> <defs> <filter id="shopp_disabled_filter0_dddd" x="16.6698" y="10.1217" width="103.5" height="119.273" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"> <feFlood flood-opacity="0" result="BackgroundImageFix"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="0.749837"/> <feGaussianBlur stdDeviation="0.468648"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0.101961 0 0 0 0 0.466667 0 0 0 0.1137 0"/> <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="1.80196"/> <feGaussianBlur stdDeviation="1.12623"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0.101961 0 0 0 0 0.466667 0 0 0 0.0484671 0"/> <feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="3.39293"/> <feGaussianBlur stdDeviation="2.12058"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0.101961 0 0 0 0 0.466667 0 0 0 0.06 0"/> <feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="6.05242"/> <feGaussianBlur stdDeviation="3.78276"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0.101961 0 0 0 0 0.466667 0 0 0 0.0715329 0"/> <feBlend mode="normal" in2="effect3_dropShadow" result="effect4_dropShadow"/> <feBlend mode="normal" in="SourceGraphic" in2="effect4_dropShadow" result="shape"/> </filter> <filter id="shopp_disabled_filter1_dd" x="32.7109" y="44.9595" width="67.165" height="60.9465" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"> <feFlood flood-opacity="0" result="BackgroundImageFix"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dx="1" dy="1"/> <feGaussianBlur stdDeviation="2"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.13 0"/> <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="0.5"/> <feGaussianBlur stdDeviation="0.25"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.15 0"/> <feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/> <feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow" result="shape"/> </filter> <filter id="shopp_disabled_filter2_dddd" x="185.046" y="16.3272" width="100.784" height="122.124" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"> <feFlood flood-opacity="0" result="BackgroundImageFix"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="0.749837"/> <feGaussianBlur stdDeviation="0.468648"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0.101961 0 0 0 0 0.466667 0 0 0 0.1137 0"/> <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="1.80196"/> <feGaussianBlur stdDeviation="1.12623"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0.101961 0 0 0 0 0.466667 0 0 0 0.0484671 0"/> <feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="3.39293"/> <feGaussianBlur stdDeviation="2.12058"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0.101961 0 0 0 0 0.466667 0 0 0 0.06 0"/> <feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="6.05242"/> <feGaussianBlur stdDeviation="3.78276"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0.101961 0 0 0 0 0.466667 0 0 0 0.0715329 0"/> <feBlend mode="normal" in2="effect3_dropShadow" result="effect4_dropShadow"/> <feBlend mode="normal" in="SourceGraphic" in2="effect4_dropShadow" result="shape"/> </filter> <filter id="shopp_disabled_filter3_dd" x="204.087" y="32.916" width="68.3604" height="54.114" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"> <feFlood flood-opacity="0" result="BackgroundImageFix"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dx="1" dy="1"/> <feGaussianBlur stdDeviation="2"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.13 0"/> <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="0.5"/> <feGaussianBlur stdDeviation="0.25"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.15 0"/> <feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/> <feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow" result="shape"/> </filter> <linearGradient id="shopp_disabled_paint0_linear" x1="32.1943" y1="17.6504" x2="120.163" y2="93.7021" gradientUnits="userSpaceOnUse"> <stop stop-color="#B5CBEC"/> <stop offset="1" stop-color="#B6CFF4" stop-opacity="0.32"/> </linearGradient> <linearGradient id="shopp_disabled_paint1_linear" x1="94.2114" y1="40.43" x2="116.406" y2="14.3621" gradientUnits="userSpaceOnUse"> <stop stop-color="white"/> <stop offset="0.147864" stop-color="#F6640E"/> <stop offset="0.443974" stop-color="#BA03A7"/> <stop offset="0.733337" stop-color="#6A01B9"/> <stop offset="1" stop-color="#6B01B9"/> </linearGradient> <clipPath id="shopp_disabled_clip0"> <rect width="13" height="13" fill="white" transform="translate(87.7959 18.2437) rotate(-4)"/> </clipPath> <clipPath id="shopp_disabled_clip1"> <rect x="26.1348" y="39.5967" width="81" height="76" rx="2" transform="rotate(-4 26.1348 39.5967)" fill="white"/> </clipPath> </defs> </svg>',
			'shoppableEnabled'    => '<svg width="70" height="70" viewBox="0 0 70 70" fill="none" xmlns="http://www.w3.org/2000/svg"> <g filter="url(#shoppEnabled_filter0_dd)"> <rect x="5" y="1" width="60" height="60" rx="2" fill="white"/> </g> <path d="M19.904 26.2247L5 39.7857V59C5 60.1046 5.89543 61 7 61H63C64.1046 61 65 60.1046 65 59V45.5714L52.4342 31.4716C51.7591 30.7141 50.6236 30.5822 49.7928 31.1648L38.8105 38.8667C38.0444 39.4039 37.0082 39.3382 36.3161 38.7085L22.596 26.2247C21.833 25.5304 20.667 25.5304 19.904 26.2247Z" fill="url(#shoppEnabled_paint0_linear)"/> <rect x="29" y="4" width="29" height="20" rx="2" fill="#0068A0"/> <path d="M37.6002 14.0001C37.6002 12.8601 38.5268 11.9334 39.6668 11.9334H42.3335V10.6667H39.6668C38.7828 10.6667 37.9349 11.0179 37.3098 11.6431C36.6847 12.2682 36.3335 13.116 36.3335 14.0001C36.3335 14.8841 36.6847 15.732 37.3098 16.3571C37.9349 16.9822 38.7828 17.3334 39.6668 17.3334H42.3335V16.0667H39.6668C38.5268 16.0667 37.6002 15.1401 37.6002 14.0001ZM40.3335 14.6667H45.6668V13.3334H40.3335V14.6667ZM46.3335 10.6667H43.6668V11.9334H46.3335C47.4735 11.9334 48.4002 12.8601 48.4002 14.0001C48.4002 15.1401 47.4735 16.0667 46.3335 16.0667H43.6668V17.3334H46.3335C47.2176 17.3334 48.0654 16.9822 48.6905 16.3571C49.3156 15.732 49.6668 14.8841 49.6668 14.0001C49.6668 13.116 49.3156 12.2682 48.6905 11.6431C48.0654 11.0179 47.2176 10.6667 46.3335 10.6667Z" fill="white"/> <path d="M64.1103 30.0086V29.0938H63.0956V28.179H61.0662V27.2643H58.0221V26.3495H55.9926V22.6904H54.9779V21.7756H52.9485V22.6904H51.9338V30.9234H50.9191V30.0086H47.875V33.6677H48.8897V35.4972H49.9044V37.3268H50.9191V39.1563H51.9338V41.9006H63.0956V38.2415H64.1103V35.4972H63.0956V38.2415H62.0809V40.9859H52.9485V39.1563H51.9338V37.3268H50.9191V35.4972H49.9044V33.6677H48.8897V30.9234H50.9191V31.8381H51.9338V34.5825H52.9485V22.6904H54.9779V30.9234H55.9926V27.2643H58.0221V30.9234H59.0368V28.179H61.0662V31.8381H62.0809V29.0938H63.0956V30.0086H64.1103V35.4972H65.125V30.0086H64.1103Z" fill="#141B38"/> <path d="M63.096 35.4972H64.1107V30.0086H63.096V29.0938H62.0813V31.8382H61.0666V28.1791H59.0372V30.9234H58.0225V27.2643H55.9931V30.9234H54.9784V22.6904H52.949V34.5825H51.9343V31.8382H50.9195V30.9234H48.8901V33.6677H49.9048V35.4972H50.9195V37.3268H51.9343V39.1563H52.949V40.9859H62.0813V38.2416H63.096V35.4972Z" fill="white"/> <path fill-rule="evenodd" clip-rule="evenodd" d="M54.9785 33.668H55.9932V37.9805H54.9785V33.668ZM58.0224 33.668H57.0077V37.9805H58.0224V33.668ZM60.0516 33.668H59.0369V37.9805H60.0516V33.668Z" fill="#141B38"/> <defs> <filter id="shoppEnabled_filter0_dd" x="0" y="0" width="70" height="70" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"> <feFlood flood-opacity="0" result="BackgroundImageFix"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="4"/> <feGaussianBlur stdDeviation="2.5"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/> <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="1"/> <feGaussianBlur stdDeviation="1"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/> <feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/> <feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow" result="shape"/> </filter> <linearGradient id="shoppEnabled_paint0_linear" x1="35" y1="25" x2="35" y2="61" gradientUnits="userSpaceOnUse"> <stop stop-color="#DCDDE1"/> <stop offset="1" stop-color="#DCDDE1" stop-opacity="0"/> </linearGradient> </defs> </svg>',

			// Feed template icons.
			'defaultFTIcon'          => '<svg width="263" height="200" fill="none" xmlns="http://www.w3.org/2000/svg"><g clip-path="url(#a)"><path fill="#F3F4F5" d="M0 0h262.5v200H0z"/><g filter="url(#b)"><rect x="32.191" y="34.5" width="198.118" height="191" rx="2" fill="#fff"/><g clip-path="url(#c)"><path fill="#F6966B" d="M45.428 156h40v40h-40z"/><circle cx="41.5" cy="165.5" r="20.5" fill="#F9BBA0"/></g><rect x="49.928" y="199" width="31" height="2" rx=".5" fill="#DCDDE1"/><g clip-path="url(#d)"><path fill="#F6966B" d="M89.428 156h40v40h-40z"/><circle cx="127.5" cy="194.5" r="20.5" fill="#F9BBA0"/></g><rect x="93.928" y="199" width="31" height="2" rx=".5" fill="#DCDDE1"/><path fill="#F6966B" d="M133.428 156h40v40h-40z"/><rect x="137.928" y="199" width="31" height="2" rx=".5" fill="#DCDDE1"/><g clip-path="url(#e)"><path fill="#F6966B" d="M177.428 156h40v40h-40z"/><path fill="#F9BBA0" d="m159.141 169 35.631 5.14-4.14 28.702-35.631-5.139z"/></g><rect x="181.928" y="199" width="31" height="2" rx=".5" fill="#DCDDE1"/><circle cx="59.982" cy="59.982" r="10.982" fill="#F9BBA0"/><path d="M79.201 51.146a.79.79 0 0 1 .814-.814h6.644a.79.79 0 0 1 .814.814v2.672a.79.79 0 0 1-.814.814h-6.644a.79.79 0 0 1-.814-.814v-2.672Z" fill="#8C8F9A"/><path d="M85.496 51.146a.79.79 0 0 1 .814-.814h6.644a.79.79 0 0 1 .814.814v2.672a.79.79 0 0 1-.814.814H86.31a.79.79 0 0 1-.814-.814v-2.672Z" fill="#8C8F9A"/><path d="M91.791 51.146a.79.79 0 0 1 .814-.814h3.644a.79.79 0 0 1 .814.814v2.672a.79.79 0 0 1-.814.814h-3.644a.79.79 0 0 1-.814-.814v-2.672Z" fill="#8C8F9A"/><path d="M95.086 51.146a.79.79 0 0 1 .814-.814h6.644a.79.79 0 0 1 .814.814v2.672a.79.79 0 0 1-.814.814H95.9a.79.79 0 0 1-.814-.814v-2.672Z" fill="#8C8F9A"/><path d="M101.381 51.157a.79.79 0 0 1 .814-.814h3.498a.79.79 0 0 1 .814.814v2.661a.79.79 0 0 1-.814.814h-3.498a.79.79 0 0 1-.814-.814v-2.661Z" fill="#8C8F9A"/><path d="M104.528 51.146a.79.79 0 0 1 .814-.814h6.644a.79.79 0 0 1 .814.814v2.672a.79.79 0 0 1-.814.814h-6.644a.791.791 0 0 1-.814-.814v-2.672Z" fill="#8C8F9A"/><path d="M79.201 59.353a.66.66 0 0 1 .232-.516.852.852 0 0 1 .585-.205h6.668a.85.85 0 0 1 .585.205.66.66 0 0 1 .232.516v2.492a.66.66 0 0 1-.232.517.851.851 0 0 1-.585.204h-6.668a.851.851 0 0 1-.585-.204.66.66 0 0 1-.232-.517v-2.492Z" fill="#D0D1D7"/><path d="M85.519 59.353a.66.66 0 0 1 .231-.516.851.851 0 0 1 .585-.205h6.668c.236 0 .43.068.585.205a.66.66 0 0 1 .232.516v2.492a.66.66 0 0 1-.232.517.851.851 0 0 1-.585.204h-6.668a.851.851 0 0 1-.584-.204.66.66 0 0 1-.232-.517v-2.492Z" fill="#D0D1D7"/><path d="M91.836 59.353a.66.66 0 0 1 .232-.516.852.852 0 0 1 .585-.205h6.668a.85.85 0 0 1 .585.205.662.662 0 0 1 .232.516v2.492c0 .208-.078.38-.232.517a.851.851 0 0 1-.585.204h-6.668a.851.851 0 0 1-.585-.204.66.66 0 0 1-.232-.517v-2.492Z" fill="#D0D1D7"/><path d="M98.153 59.353a.66.66 0 0 1 .232-.516.851.851 0 0 1 .585-.205h6.668a.85.85 0 0 1 .585.205.659.659 0 0 1 .232.516v2.492a.66.66 0 0 1-.232.517.85.85 0 0 1-.585.204H98.97a.851.851 0 0 1-.585-.204.66.66 0 0 1-.232-.517v-2.492Z" fill="#D0D1D7"/><path d="M104.471 59.363a.66.66 0 0 1 .232-.517.85.85 0 0 1 .585-.204h3.51c.236 0 .431.068.586.204a.661.661 0 0 1 .231.517v2.482c0 .208-.077.38-.231.517a.854.854 0 0 1-.586.204h-3.51a.85.85 0 0 1-.585-.204.659.659 0 0 1-.232-.517v-2.482Z" fill="#D0D1D7"/><path d="M107.63 59.363a.66.66 0 0 1 .232-.517.85.85 0 0 1 .585-.204h3.51a.85.85 0 0 1 .585.204.659.659 0 0 1 .232.517v2.482a.66.66 0 0 1-.232.517.85.85 0 0 1-.585.204h-3.51a.85.85 0 0 1-.585-.204.659.659 0 0 1-.232-.517v-2.482Z" fill="#D0D1D7"/><path d="M110.788 59.353a.66.66 0 0 1 .232-.516.853.853 0 0 1 .585-.205h6.668a.85.85 0 0 1 .585.205.659.659 0 0 1 .232.516v2.492a.66.66 0 0 1-.232.517.85.85 0 0 1-.585.204h-6.668a.853.853 0 0 1-.585-.204.662.662 0 0 1-.232-.517v-2.492Z" fill="#D0D1D7"/><path d="M117.106 59.363a.66.66 0 0 1 .232-.517.85.85 0 0 1 .585-.204h3.51a.85.85 0 0 1 .585.204.659.659 0 0 1 .232.517v2.482a.66.66 0 0 1-.232.517.85.85 0 0 1-.585.204h-3.51a.85.85 0 0 1-.585-.204.659.659 0 0 1-.232-.517v-2.482Z" fill="#D0D1D7"/><path d="M120.265 59.353c0-.208.077-.38.231-.516a.854.854 0 0 1 .586-.205h6.667a.85.85 0 0 1 .585.205.659.659 0 0 1 .232.516v2.492a.66.66 0 0 1-.232.517.85.85 0 0 1-.585.204h-6.667a.854.854 0 0 1-.586-.204.661.661 0 0 1-.231-.517v-2.492Z" fill="#D0D1D7"/><path d="M126.582 59.353c0-.208.077-.38.232-.516a.851.851 0 0 1 .585-.205h6.668a.85.85 0 0 1 .585.205.662.662 0 0 1 .232.516v2.492c0 .208-.078.38-.232.517a.853.853 0 0 1-.585.204h-6.668a.851.851 0 0 1-.585-.204.659.659 0 0 1-.232-.517v-2.492Z" fill="#D0D1D7"/><path d="M132.9 59.353c0-.208.077-.38.231-.516a.853.853 0 0 1 .585-.205h6.668a.85.85 0 0 1 .585.205.659.659 0 0 1 .232.516v2.492a.66.66 0 0 1-.232.517.85.85 0 0 1-.585.204h-6.668a.853.853 0 0 1-.585-.204.661.661 0 0 1-.231-.517v-2.492Zm-53.699 7.066a.66.66 0 0 1 .232-.517.851.851 0 0 1 .585-.205h6.668c.235 0 .43.069.585.205a.66.66 0 0 1 .232.517v2.491a.66.66 0 0 1-.232.517.851.851 0 0 1-.585.205h-6.668a.851.851 0 0 1-.585-.205.66.66 0 0 1-.232-.517v-2.49Z" fill="#D0D1D7"/><path d="M85.519 66.419a.66.66 0 0 1 .231-.517.851.851 0 0 1 .585-.205h6.668c.236 0 .43.069.585.205a.66.66 0 0 1 .232.517v2.491a.66.66 0 0 1-.232.517.851.851 0 0 1-.585.205h-6.668a.851.851 0 0 1-.584-.205.66.66 0 0 1-.232-.517v-2.49Z" fill="#D0D1D7"/><path d="M91.836 66.419a.66.66 0 0 1 .232-.517.851.851 0 0 1 .585-.205h6.668c.235 0 .43.069.585.205a.662.662 0 0 1 .232.517v2.491c0 .208-.078.38-.232.517a.851.851 0 0 1-.585.205h-6.668a.851.851 0 0 1-.585-.205.66.66 0 0 1-.232-.517v-2.49Z" fill="#D0D1D7"/><path d="M98.153 66.419a.66.66 0 0 1 .232-.517.851.851 0 0 1 .585-.205h6.668a.85.85 0 0 1 .585.205.659.659 0 0 1 .232.517v2.491a.66.66 0 0 1-.232.517.85.85 0 0 1-.585.205H98.97a.851.851 0 0 1-.585-.205.66.66 0 0 1-.232-.517v-2.49Z" fill="#D0D1D7"/><path d="M104.471 66.419a.66.66 0 0 1 .232-.517.85.85 0 0 1 .585-.205h6.668c.235 0 .43.069.585.205a.662.662 0 0 1 .232.517v2.491c0 .208-.078.38-.232.517a.853.853 0 0 1-.585.205h-6.668a.85.85 0 0 1-.585-.205.659.659 0 0 1-.232-.517v-2.49Z" fill="#D0D1D7"/><path d="M110.788 66.419c0-.208.078-.38.232-.517a.853.853 0 0 1 .585-.205h6.668a.85.85 0 0 1 .585.205.659.659 0 0 1 .232.517v2.491a.66.66 0 0 1-.232.517.85.85 0 0 1-.585.205h-6.668a.853.853 0 0 1-.585-.205.662.662 0 0 1-.232-.517v-2.49Z" fill="#D0D1D7"/><path d="M117.106 66.419a.66.66 0 0 1 .232-.517.85.85 0 0 1 .585-.205h6.668c.235 0 .43.069.585.205a.662.662 0 0 1 .232.517v2.491c0 .208-.078.38-.232.517a.853.853 0 0 1-.585.205h-6.668a.85.85 0 0 1-.585-.205.659.659 0 0 1-.232-.517v-2.49Z" fill="#D0D1D7"/><path d="M123.423 66.429c0-.208.078-.38.232-.517a.853.853 0 0 1 .585-.205h3.511a.85.85 0 0 1 .585.205.662.662 0 0 1 .232.517v2.481c0 .208-.078.38-.232.517a.853.853 0 0 1-.585.205h-3.511a.853.853 0 0 1-.585-.205.662.662 0 0 1-.232-.517v-2.48Z" fill="#D0D1D7"/><rect x="190.271" y="49" width="26.728" height="10.491" rx=".915" fill="#0096CC"/><g clip-path="url(#f)"><path d="M196.219 53.218c-.572 0-1.027.464-1.027 1.027a1.028 1.028 0 0 0 2.055 0c0-.563-.465-1.027-1.028-1.027Zm0 1.698a.67.67 0 0 1-.67-.67c0-.367.295-.662.67-.662a.66.66 0 0 1 .662.661c0 .376-.295.67-.662.67Zm1.305-1.734a.24.24 0 0 0-.241-.241.24.24 0 0 0-.241.241.24.24 0 0 0 .241.241.24.24 0 0 0 .241-.241Zm.679.241c-.017-.321-.089-.607-.321-.84-.233-.232-.519-.304-.84-.322a25.479 25.479 0 0 0-1.654 0c-.322.018-.599.09-.84.322-.232.233-.304.519-.322.84a25.32 25.32 0 0 0 0 1.654c.018.321.09.599.322.84.241.232.518.304.84.322.331.017 1.323.017 1.654 0 .321-.018.607-.09.84-.322.232-.242.304-.519.321-.84a25.32 25.32 0 0 0 0-1.654Zm-.429 2.002a.66.66 0 0 1-.375.385c-.268.107-.894.08-1.18.08-.295 0-.92.027-1.179-.08a.68.68 0 0 1-.385-.385c-.107-.259-.08-.884-.08-1.18 0-.285-.027-.911.08-1.18a.691.691 0 0 1 .385-.375c.259-.107.884-.08 1.179-.08.286 0 .912-.027 1.18.08.17.063.304.206.375.376.108.268.081.893.081 1.18 0 .294.027.92-.081 1.18Z" fill="#fff"/></g><path d="M200.625 55.745h.527V54.51h1.209v-.441h-1.209v-.79h1.336v-.442h-1.863v2.91Zm3.152.043c.64 0 1.046-.45 1.046-1.125 0-.676-.406-1.128-1.046-1.128-.639 0-1.045.452-1.045 1.128 0 .674.406 1.125 1.045 1.125Zm.003-.412c-.353 0-.527-.316-.527-.715 0-.399.174-.718.527-.718.348 0 .522.32.522.718 0 .4-.174.715-.522.715Zm2.033-2.54h-.514v2.91h.514v-2.91Zm1.083 0h-.514v2.91h.514v-2.91Zm1.521 2.952c.639 0 1.045-.45 1.045-1.125 0-.676-.406-1.128-1.045-1.128-.639 0-1.046.452-1.046 1.128 0 .674.407 1.125 1.046 1.125Zm.003-.412c-.354 0-.527-.316-.527-.715 0-.399.173-.718.527-.718.348 0 .521.32.521.718 0 .4-.173.715-.521.715Zm1.911.37h.536l.41-1.475h.029l.409 1.474h.536l.618-2.182h-.526l-.378 1.526h-.021l-.392-1.526h-.518l-.392 1.534h-.02l-.384-1.534h-.524l.617 2.182Z" fill="#fff"/><g clip-path="url(#g)"><path fill="#F6966B" d="M45.428 88.376h40v40h-40z"/><circle cx="85" cy="128" r="25" fill="#F9BBA0"/></g><rect x="49.928" y="131.376" width="31" height="2" rx=".5" fill="#D0D1D7"/><rect x="54.928" y="135.376" width="21" height="2" rx=".5" fill="#D0D1D7"/><g clip-path="url(#h)"><path fill-rule="evenodd" clip-rule="evenodd" d="M62.88 140.826a.96.96 0 0 1 .684.283.975.975 0 0 1 .283.684c0 .495-.28.942-.628 1.312-.322.343-.702.624-1.088.89-.387-.266-.766-.547-1.089-.89-.347-.37-.628-.817-.628-1.312a.968.968 0 0 1 .967-.967.973.973 0 0 1 .75.358.972.972 0 0 1 .75-.358Z" fill="#8C8F9A"/></g><g clip-path="url(#i)"><path d="M68.726 140.709a1.663 1.663 0 0 0-1.178.488 1.669 1.669 0 0 0-.112 2.234l-.333.333a.168.168 0 0 0-.035.182.165.165 0 0 0 .158.096h1.5a1.665 1.665 0 0 0 1.179-2.845 1.669 1.669 0 0 0-1.179-.488Z" fill="#8C8F9A"/></g><path fill="#F6966B" d="M89.428 88.376h40v40h-40z"/><rect x="93.928" y="131.376" width="31" height="2" rx=".5" fill="#D0D1D7"/><rect x="98.928" y="135.376" width="21" height="2" rx=".5" fill="#D0D1D7"/><g clip-path="url(#j)"><path fill-rule="evenodd" clip-rule="evenodd" d="M106.88 140.826h.001a.962.962 0 0 1 .683.283.978.978 0 0 1 .283.684c0 .495-.281.942-.628 1.312-.322.343-.702.624-1.088.89-.387-.266-.766-.547-1.089-.89-.347-.37-.628-.817-.628-1.312a.969.969 0 0 1 .967-.967c.155 0 .308.038.446.109a.976.976 0 0 1 .304.249.965.965 0 0 1 .749-.358Z" fill="#8C8F9A"/></g><g clip-path="url(#k)"><path d="M112.726 140.709a1.662 1.662 0 0 0-1.178.488 1.668 1.668 0 0 0-.112 2.234l-.333.333a.163.163 0 0 0-.045.086.159.159 0 0 0 .074.168.163.163 0 0 0 .094.024h1.5a1.667 1.667 0 1 0 0-3.333Z" fill="#8C8F9A"/></g><g clip-path="url(#l)"><path fill="#F6966B" d="M133.428 88.376h40v40h-40z"/><path d="M173.5 128.5v-14h-40v14h40Z" fill="#F9BBA0"/></g><rect x="137.928" y="131.376" width="31" height="2" rx=".5" fill="#D0D1D7"/><rect x="142.928" y="135.376" width="21" height="2" rx=".5" fill="#D0D1D7"/><g clip-path="url(#m)"><path fill-rule="evenodd" clip-rule="evenodd" d="M150.88 140.826h.001a.962.962 0 0 1 .683.283.978.978 0 0 1 .283.684c0 .495-.281.942-.628 1.312-.322.343-.702.624-1.088.89-.387-.266-.766-.547-1.089-.89-.347-.37-.628-.817-.628-1.312a.969.969 0 0 1 .967-.967c.155 0 .308.038.446.109a.976.976 0 0 1 .304.249.965.965 0 0 1 .749-.358Z" fill="#8C8F9A"/></g><g clip-path="url(#n)"><path d="M156.726 140.709a1.662 1.662 0 0 0-1.178.488 1.668 1.668 0 0 0-.112 2.234l-.333.333a.163.163 0 0 0-.045.086.159.159 0 0 0 .074.168.163.163 0 0 0 .094.024h1.5a1.667 1.667 0 1 0 0-3.333Z" fill="#8C8F9A"/></g><g clip-path="url(#o)"><path fill="#F6966B" d="M177.428 88.376h40v40h-40z"/><circle cx="225" cy="138" r="28" fill="#F9BBA0"/></g><rect x="181.928" y="131.376" width="31" height="2" rx=".5" fill="#D0D1D7"/><rect x="186.928" y="135.376" width="21" height="2" rx=".5" fill="#D0D1D7"/><g clip-path="url(#p)"><path fill-rule="evenodd" clip-rule="evenodd" d="M194.88 140.826h.001a.962.962 0 0 1 .683.283.978.978 0 0 1 .283.684c0 .495-.281.942-.628 1.312-.322.343-.702.624-1.088.89-.387-.266-.766-.547-1.089-.89-.347-.37-.628-.817-.628-1.312a.969.969 0 0 1 .967-.967c.155 0 .308.038.446.109a.976.976 0 0 1 .304.249.965.965 0 0 1 .749-.358Z" fill="#8C8F9A"/></g><g clip-path="url(#q)"><path d="M200.726 140.709a1.662 1.662 0 0 0-1.178.488 1.668 1.668 0 0 0-.112 2.234l-.333.333a.163.163 0 0 0-.045.086.159.159 0 0 0 .074.168.163.163 0 0 0 .094.024h1.5a1.667 1.667 0 1 0 0-3.333Z" fill="#8C8F9A"/></g></g></g><defs><clipPath id="a"><path fill="#fff" d="M0 0h262.5v200H0z"/></clipPath><clipPath id="c"><path fill="#fff" transform="translate(45.428 156)" d="M0 0h40v40H0z"/></clipPath><clipPath id="d"><path fill="#fff" transform="translate(89.428 156)" d="M0 0h40v40H0z"/></clipPath><clipPath id="e"><path fill="#fff" transform="translate(177.428 156)" d="M0 0h40v40H0z"/></clipPath><clipPath id="f"><path fill="#fff" transform="translate(193.932 51.958)" d="M0 0h4.576v4.576H0z"/></clipPath><clipPath id="g"><path fill="#fff" transform="translate(45.428 88.376)" d="M0 0h40v40H0z"/></clipPath><clipPath id="h"><path fill="#fff" transform="translate(60.13 140.376)" d="M0 0h4v4H0z"/></clipPath><clipPath id="i"><path fill="#fff" transform="translate(66.727 140.376)" d="M0 0h4v4H0z"/></clipPath><clipPath id="j"><path fill="#fff" transform="translate(104.131 140.376)" d="M0 0h4v4H0z"/></clipPath><clipPath id="k"><path fill="#fff" transform="translate(110.727 140.376)" d="M0 0h4v4H0z"/></clipPath><clipPath id="l"><path fill="#fff" transform="translate(133.428 88.376)" d="M0 0h40v40H0z"/></clipPath><clipPath id="m"><path fill="#fff" transform="translate(148.131 140.376)" d="M0 0h4v4H0z"/></clipPath><clipPath id="n"><path fill="#fff" transform="translate(154.727 140.376)" d="M0 0h4v4H0z"/></clipPath><clipPath id="o"><path fill="#fff" transform="translate(177.428 88.376)" d="M0 0h40v40H0z"/></clipPath><clipPath id="p"><path fill="#fff" transform="translate(192.131 140.376)" d="M0 0h4v4H0z"/></clipPath><clipPath id="q"><path fill="#fff" transform="translate(198.727 140.376)" d="M0 0h4v4H0z"/></clipPath><filter id="b" x="19.191" y="27.5" width="224.117" height="217" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="6"/><feGaussianBlur stdDeviation="6.5"/><feColorMatrix values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/><feBlend in2="BackgroundImageFix" result="effect1_dropShadow_2180_28703"/><feColorMatrix in="SourceAlpha" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="1"/><feGaussianBlur stdDeviation="1"/><feColorMatrix values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/><feBlend in2="effect1_dropShadow_2180_28703" result="effect2_dropShadow_2180_28703"/><feColorMatrix in="SourceAlpha" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="3"/><feGaussianBlur stdDeviation="3"/><feColorMatrix values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/><feBlend in2="effect2_dropShadow_2180_28703" result="effect3_dropShadow_2180_28703"/><feBlend in="SourceGraphic" in2="effect3_dropShadow_2180_28703" result="shape"/></filter></defs></svg>',
			'simpleGridIcon'         => '<svg width="263" height="200" viewBox="0 0 263 200" fill="none" xmlns="http://www.w3.org/2000/svg"><g clip-path="url(#clip0_2180_28814)"><rect width="262.5" height="200" transform="translate(0.5)" fill="#F3F4F5"/><g filter="url(#filter0_ddd_2180_28814)"><rect x="30.5" y="41" width="203.304" height="196" rx="2" fill="white"/><g clip-path="url(#clip1_2180_28814)"><rect x="45.1094" y="84.8262" width="56" height="56" rx="1" fill="#43A6DB"/><circle cx="100.5" cy="142.5" r="34.5" fill="#86D0F9"/></g><g clip-path="url(#clip2_2180_28814)"><rect x="45.1094" y="143.261" width="56" height="56" rx="1" fill="#43A6DB"/><circle cx="43.5" cy="200.5" r="29.5" fill="#86D0F9"/></g><rect x="103.543" y="84.8262" width="57.2174" height="56" rx="1" fill="#43A6DB"/><rect x="103.543" y="143.261" width="57.2174" height="56" rx="1" fill="#43A6DB"/><g clip-path="url(#clip3_2180_28814)"><rect x="163.195" y="84.8262" width="56" height="56" rx="1" fill="#43A6DB"/><rect x="166" y="143.234" width="47" height="47" transform="rotate(-45 166 143.234)" fill="#86D0F9"/></g><g clip-path="url(#clip4_2180_28814)"><rect x="163.195" y="143.261" width="56" height="56" rx="1" fill="#43A6DB"/><circle cx="219" cy="199" r="28" fill="#86D0F9"/></g><path d="M47.3324 68.8262H48.4403L49.9709 63.3787H50.0277L51.5582 68.8262H52.6626L54.6832 61.5534H53.5256L52.1122 67.1891H52.0447L50.571 61.5534H49.4276L47.9538 67.1855H47.8864L46.4695 61.5534H45.3153L47.3324 68.8262ZM57.4123 68.9363C58.6019 68.9363 59.4435 68.3503 59.685 67.4625L58.68 67.2814C58.4883 67.7963 58.0266 68.0591 57.4229 68.0591C56.5138 68.0591 55.9031 67.4696 55.8746 66.4185H59.7525V66.0421C59.7525 64.0712 58.5735 63.3006 57.3377 63.3006C55.8178 63.3006 54.8164 64.4583 54.8164 66.1344C54.8164 67.8283 55.8036 68.9363 57.4123 68.9363ZM55.8782 65.623C55.9208 64.8489 56.4819 64.1777 57.3448 64.1777C58.1687 64.1777 58.7085 64.7885 58.712 65.623H55.8782ZM61.8432 61.5534H60.9341V62.2459C60.9341 62.7963 60.6926 63.2473 60.4157 63.6557L60.9412 64.0321C61.4881 63.5918 61.8432 62.839 61.8432 62.2353V61.5534ZM62.7079 68.8262H63.7697V65.4952C63.7697 64.7814 64.3201 64.2665 65.073 64.2665C65.2931 64.2665 65.5417 64.3056 65.627 64.3304V63.3148C65.5204 63.3006 65.3109 63.29 65.176 63.29C64.5368 63.29 63.9899 63.6522 63.791 64.2381H63.7342V63.3716H62.7079V68.8262ZM68.7013 68.9363C69.891 68.9363 70.7326 68.3503 70.9741 67.4625L69.9691 67.2814C69.7773 67.7963 69.3157 68.0591 68.712 68.0591C67.8029 68.0591 67.1921 67.4696 67.1637 66.4185H71.0415V66.0421C71.0415 64.0712 69.8626 63.3006 68.6268 63.3006C67.1069 63.3006 66.1055 64.4583 66.1055 66.1344C66.1055 67.8283 67.0927 68.9363 68.7013 68.9363ZM67.1673 65.623C67.2099 64.8489 67.771 64.1777 68.6339 64.1777C69.4577 64.1777 69.9975 64.7885 70.0011 65.623H67.1673ZM77.1637 68.9363C78.7013 68.9363 79.7063 67.8105 79.7063 66.1238C79.7063 64.4263 78.7013 63.3006 77.1637 63.3006C75.6261 63.3006 74.6211 64.4263 74.6211 66.1238C74.6211 67.8105 75.6261 68.9363 77.1637 68.9363ZM77.1673 68.0449C76.1623 68.0449 75.6935 67.1678 75.6935 66.1202C75.6935 65.0762 76.1623 64.1884 77.1673 64.1884C78.1651 64.1884 78.6339 65.0762 78.6339 66.1202C78.6339 67.1678 78.1651 68.0449 77.1673 68.0449ZM81.9533 65.5875C81.9533 64.7175 82.486 64.2203 83.2246 64.2203C83.9455 64.2203 84.3823 64.6926 84.3823 65.4846V68.8262H85.4441V65.3567C85.4441 64.0073 84.7019 63.3006 83.5868 63.3006C82.7665 63.3006 82.2303 63.6806 81.9782 64.2594H81.9107V63.3716H80.8915V68.8262H81.9533V65.5875ZM90.6971 61.5534H89.5998V68.8262H90.6971V61.5534ZM93.2717 65.5875C93.2717 64.7175 93.8043 64.2203 94.543 64.2203C95.2638 64.2203 95.7006 64.6926 95.7006 65.4846V68.8262H96.7624V65.3567C96.7624 64.0073 96.0202 63.3006 94.9052 63.3006C94.0849 63.3006 93.5487 63.6806 93.2965 64.2594H93.229V63.3716H92.2099V68.8262H93.2717V65.5875ZM102.274 64.7033C102.054 63.8546 101.39 63.3006 100.211 63.3006C98.9783 63.3006 98.1048 63.9505 98.1048 64.9164C98.1048 65.6905 98.5735 66.2054 99.5962 66.4327L100.52 66.6351C101.045 66.7523 101.29 66.9867 101.29 67.3276C101.29 67.7502 100.839 68.0804 100.143 68.0804C99.5075 68.0804 99.0991 67.807 98.9712 67.2708L97.945 67.427C98.1225 68.3929 98.9251 68.9363 100.15 68.9363C101.468 68.9363 102.38 68.2367 102.38 67.2495C102.38 66.4789 101.89 66.003 100.889 65.7722L100.022 65.5733C99.4222 65.4313 99.163 65.2289 99.1665 64.8596C99.163 64.4405 99.6175 64.1422 100.221 64.1422C100.882 64.1422 101.187 64.508 101.311 64.8738L102.274 64.7033ZM106.094 63.3716H104.975V62.0648H103.914V63.3716H103.115V64.2239H103.914V67.4448C103.91 68.4355 104.666 68.915 105.504 68.8972C105.842 68.8936 106.069 68.8297 106.193 68.7836L106.002 67.9064C105.931 67.9206 105.799 67.9526 105.629 67.9526C105.284 67.9526 104.975 67.839 104.975 67.2246V64.2239H106.094V63.3716ZM108.856 68.9469C109.758 68.9469 110.266 68.4888 110.468 68.0804H110.511V68.8262H111.548V65.204C111.548 63.6167 110.298 63.3006 109.431 63.3006C108.444 63.3006 107.535 63.6983 107.18 64.6926L108.178 64.9199C108.334 64.5328 108.732 64.16 109.445 64.16C110.131 64.16 110.482 64.5186 110.482 65.1365V65.1614C110.482 65.5485 110.085 65.5414 109.104 65.655C108.071 65.7757 107.013 66.0456 107.013 67.285C107.013 68.3574 107.819 68.9469 108.856 68.9469ZM109.087 68.0946C108.487 68.0946 108.053 67.8248 108.053 67.2992C108.053 66.731 108.558 66.5286 109.172 66.4469C109.516 66.4007 110.333 66.3084 110.486 66.1557V66.8588C110.486 67.5051 109.971 68.0946 109.087 68.0946ZM115.249 70.9853C116.637 70.9853 117.71 70.3496 117.71 68.9469V63.3716H116.669V64.2559H116.591C116.403 63.9185 116.026 63.3006 115.011 63.3006C113.693 63.3006 112.724 64.3411 112.724 66.0776C112.724 67.8176 113.715 68.7445 115.004 68.7445C116.005 68.7445 116.392 68.1799 116.584 67.8319H116.651V68.9043C116.651 69.7601 116.066 70.1294 115.259 70.1294C114.375 70.1294 114.031 69.6855 113.843 69.373L112.93 69.7495C113.218 70.4171 113.945 70.9853 115.249 70.9853ZM115.238 67.8638C114.29 67.8638 113.796 67.1358 113.796 66.0634C113.796 65.0158 114.279 64.2026 115.238 64.2026C116.165 64.2026 116.662 64.959 116.662 66.0634C116.662 67.1891 116.154 67.8638 115.238 67.8638ZM119.134 68.8262H120.195V65.4952C120.195 64.7814 120.746 64.2665 121.499 64.2665C121.719 64.2665 121.968 64.3056 122.053 64.3304V63.3148C121.946 63.3006 121.737 63.29 121.602 63.29C120.963 63.29 120.416 63.6522 120.217 64.2381H120.16V63.3716H119.134V68.8262ZM124.51 68.9469C125.412 68.9469 125.92 68.4888 126.122 68.0804H126.165V68.8262H127.202V65.204C127.202 63.6167 125.952 63.3006 125.085 63.3006C124.098 63.3006 123.189 63.6983 122.834 64.6926L123.832 64.9199C123.988 64.5328 124.386 64.16 125.1 64.16C125.785 64.16 126.137 64.5186 126.137 65.1365V65.1614C126.137 65.5485 125.739 65.5414 124.759 65.655C123.725 65.7757 122.667 66.0456 122.667 67.285C122.667 68.3574 123.473 68.9469 124.51 68.9469ZM124.741 68.0946C124.141 68.0946 123.708 67.8248 123.708 67.2992C123.708 66.731 124.212 66.5286 124.826 66.4469C125.171 66.4007 125.987 66.3084 126.14 66.1557V66.8588C126.14 67.5051 125.625 68.0946 124.741 68.0946ZM128.616 68.8262H129.678V65.4632C129.678 64.7282 130.193 64.2168 130.797 64.2168C131.386 64.2168 131.794 64.6074 131.794 65.2005V68.8262H132.853V65.3496C132.853 64.6926 133.254 64.2168 133.946 64.2168C134.507 64.2168 134.969 64.5293 134.969 65.2679V68.8262H136.031V65.1685C136.031 63.9221 135.335 63.3006 134.348 63.3006C133.563 63.3006 132.973 63.677 132.711 64.2594H132.654C132.416 63.6628 131.915 63.3006 131.187 63.3006C130.466 63.3006 129.93 63.6593 129.703 64.2594H129.635V63.3716H128.616V68.8262ZM138.793 61.5534H137.624L137.717 66.7346H138.7L138.793 61.5534ZM138.207 68.8936C138.594 68.8936 138.917 68.5776 138.917 68.1834C138.917 67.7963 138.594 67.4767 138.207 67.4767C137.816 67.4767 137.497 67.7963 137.497 68.1834C137.497 68.5776 137.816 68.8936 138.207 68.8936Z" fill="#141B38"/></g></g><defs><filter id="filter0_ddd_2180_28814" x="17.5" y="34" width="229.305" height="222" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="6"/><feGaussianBlur stdDeviation="6.5"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_2180_28814"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="1"/><feGaussianBlur stdDeviation="1"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/><feBlend mode="normal" in2="effect1_dropShadow_2180_28814" result="effect2_dropShadow_2180_28814"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="3"/><feGaussianBlur stdDeviation="3"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/><feBlend mode="normal" in2="effect2_dropShadow_2180_28814" result="effect3_dropShadow_2180_28814"/><feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow_2180_28814" result="shape"/></filter><clipPath id="clip0_2180_28814"><rect width="262.5" height="200" fill="white" transform="translate(0.5)"/></clipPath><clipPath id="clip1_2180_28814"><rect x="45.1094" y="84.8262" width="56" height="56" rx="1" fill="white"/></clipPath><clipPath id="clip2_2180_28814"><rect x="45.1094" y="143.261" width="56" height="56" rx="1" fill="white"/></clipPath><clipPath id="clip3_2180_28814"><rect x="163.195" y="84.8262" width="56" height="56" rx="1" fill="white"/></clipPath><clipPath id="clip4_2180_28814"><rect x="163.195" y="143.261" width="56" height="56" rx="1" fill="white"/></clipPath></defs></svg>',
			'simpleGridXLIcon'       => '<svg width="263" height="200" fill="none" xmlns="http://www.w3.org/2000/svg"><g clip-path="url(#a)"><path fill="#F3F4F5" d="M0 0h262.5v200H0z"/><g filter="url(#b)"><rect x="18" y="41" width="227" height="186" rx="2" fill="#fff"/><circle cx="136" cy="61.982" r="10.982" fill="#86D0F9"/><path d="M119.201 82.015a.79.79 0 0 1 .814-.814h6.644a.79.79 0 0 1 .814.814v2.672a.79.79 0 0 1-.814.814h-6.644a.791.791 0 0 1-.814-.814v-2.672Z" fill="#8C8F9A"/><path d="M125.496 82.015a.79.79 0 0 1 .814-.814h6.644a.79.79 0 0 1 .814.814v2.672a.79.79 0 0 1-.814.814h-6.644a.79.79 0 0 1-.814-.814v-2.672Z" fill="#8C8F9A"/><path d="M131.791 82.015a.79.79 0 0 1 .814-.814h3.644a.79.79 0 0 1 .814.814v2.672a.79.79 0 0 1-.814.814h-3.644a.79.79 0 0 1-.814-.814v-2.672Z" fill="#8C8F9A"/><path d="M135.086 82.015a.79.79 0 0 1 .814-.814h6.644a.79.79 0 0 1 .814.814v2.672a.79.79 0 0 1-.814.814H135.9a.79.79 0 0 1-.814-.814v-2.672Z" fill="#8C8F9A"/><path d="M141.381 82.026a.79.79 0 0 1 .814-.814h3.498a.79.79 0 0 1 .814.814v2.661a.79.79 0 0 1-.814.814h-3.498a.79.79 0 0 1-.814-.814v-2.66Z" fill="#8C8F9A"/><path d="M144.528 82.015a.79.79 0 0 1 .814-.814h6.644a.79.79 0 0 1 .814.814v2.672a.79.79 0 0 1-.814.814h-6.644a.791.791 0 0 1-.814-.814v-2.672Z" fill="#8C8F9A"/><path d="M105 90.222c0-.207.077-.38.232-.516a.85.85 0 0 1 .585-.205h6.668a.85.85 0 0 1 .585.205.662.662 0 0 1 .232.516v2.492c0 .208-.078.38-.232.517a.853.853 0 0 1-.585.205h-6.668a.85.85 0 0 1-.585-.205.659.659 0 0 1-.232-.517v-2.492Z" fill="#D0D1D7"/><path d="M111.317 90.222c0-.207.078-.38.232-.516a.853.853 0 0 1 .585-.205h6.668a.85.85 0 0 1 .585.205.659.659 0 0 1 .232.516v2.492a.66.66 0 0 1-.232.517.85.85 0 0 1-.585.205h-6.668a.853.853 0 0 1-.585-.205.662.662 0 0 1-.232-.517v-2.492Z" fill="#D0D1D7"/><path d="M117.635 90.222c0-.207.077-.38.232-.516a.85.85 0 0 1 .585-.205h6.668a.85.85 0 0 1 .585.205.662.662 0 0 1 .232.516v2.492c0 .208-.078.38-.232.517a.853.853 0 0 1-.585.205h-6.668a.85.85 0 0 1-.585-.205.659.659 0 0 1-.232-.517v-2.492Z" fill="#D0D1D7"/><path d="M123.952 90.222c0-.207.078-.38.232-.516a.853.853 0 0 1 .585-.205h6.668a.85.85 0 0 1 .585.205.659.659 0 0 1 .232.516v2.492a.66.66 0 0 1-.232.517.85.85 0 0 1-.585.205h-6.668a.853.853 0 0 1-.585-.205.662.662 0 0 1-.232-.517v-2.492Z" fill="#D0D1D7"/><path d="M130.27 90.232c0-.208.077-.38.232-.516a.85.85 0 0 1 .585-.205h3.51a.85.85 0 0 1 .585.205.659.659 0 0 1 .232.516v2.482a.66.66 0 0 1-.232.517.85.85 0 0 1-.585.205h-3.51a.85.85 0 0 1-.585-.205.659.659 0 0 1-.232-.517v-2.482Z" fill="#D0D1D7"/><path d="M133.429 90.232a.66.66 0 0 1 .231-.516.853.853 0 0 1 .585-.205h3.511c.235 0 .431.068.585.205a.659.659 0 0 1 .232.516v2.482a.66.66 0 0 1-.232.517.851.851 0 0 1-.585.205h-3.511a.853.853 0 0 1-.585-.205.661.661 0 0 1-.231-.517v-2.482Z" fill="#D0D1D7"/><path d="M136.587 90.222c0-.207.078-.38.232-.516a.853.853 0 0 1 .585-.205h6.668c.235 0 .431.068.585.205a.659.659 0 0 1 .232.516v2.492a.66.66 0 0 1-.232.517.851.851 0 0 1-.585.205h-6.668a.853.853 0 0 1-.585-.205.662.662 0 0 1-.232-.517v-2.492Z" fill="#D0D1D7"/><path d="M142.905 90.232c0-.208.077-.38.232-.516a.85.85 0 0 1 .585-.205h3.51a.85.85 0 0 1 .585.205.659.659 0 0 1 .232.516v2.482a.66.66 0 0 1-.232.517.85.85 0 0 1-.585.205h-3.51a.85.85 0 0 1-.585-.205.659.659 0 0 1-.232-.517v-2.482Z" fill="#D0D1D7"/><path d="M146.063 90.222c0-.207.078-.38.232-.516a.853.853 0 0 1 .585-.205h6.668a.85.85 0 0 1 .585.205.659.659 0 0 1 .232.516v2.492a.66.66 0 0 1-.232.517.85.85 0 0 1-.585.205h-6.668a.853.853 0 0 1-.585-.205.662.662 0 0 1-.232-.517v-2.492Z" fill="#D0D1D7"/><path d="M152.381 90.222c0-.207.077-.38.232-.516a.85.85 0 0 1 .585-.205h6.668a.85.85 0 0 1 .585.205.662.662 0 0 1 .232.516v2.492c0 .208-.078.38-.232.517a.853.853 0 0 1-.585.205h-6.668a.85.85 0 0 1-.585-.205.659.659 0 0 1-.232-.517v-2.492Z" fill="#D0D1D7"/><path d="M158.698 90.222c0-.207.078-.38.232-.516a.853.853 0 0 1 .585-.205h6.668a.85.85 0 0 1 .585.205.659.659 0 0 1 .232.516v2.492a.66.66 0 0 1-.232.517.85.85 0 0 1-.585.205h-6.668a.853.853 0 0 1-.585-.205.662.662 0 0 1-.232-.517v-2.492ZM111 97.288a.66.66 0 0 1 .232-.517.849.849 0 0 1 .585-.204h6.668c.235 0 .43.068.585.204a.662.662 0 0 1 .232.517v2.491a.66.66 0 0 1-.232.517.85.85 0 0 1-.585.205h-6.668a.847.847 0 0 1-.585-.205.657.657 0 0 1-.232-.517v-2.491Z" fill="#D0D1D7"/><path d="M117.317 97.288c0-.208.078-.38.232-.517a.852.852 0 0 1 .585-.204h6.668c.236 0 .431.068.585.204a.658.658 0 0 1 .232.517v2.491a.657.657 0 0 1-.232.517.847.847 0 0 1-.585.205h-6.668a.85.85 0 0 1-.585-.205.66.66 0 0 1-.232-.517v-2.491Z" fill="#D0D1D7"/><path d="M123.635 97.288a.66.66 0 0 1 .232-.517.849.849 0 0 1 .585-.204h6.668c.235 0 .43.068.585.204a.662.662 0 0 1 .232.517v2.491a.66.66 0 0 1-.232.517.85.85 0 0 1-.585.205h-6.668a.847.847 0 0 1-.585-.205.657.657 0 0 1-.232-.517v-2.491Z" fill="#D0D1D7"/><path d="M129.952 97.288c0-.208.078-.38.232-.517a.852.852 0 0 1 .585-.204h6.668c.236 0 .431.068.585.204a.658.658 0 0 1 .232.517v2.491a.657.657 0 0 1-.232.517.847.847 0 0 1-.585.205h-6.668a.85.85 0 0 1-.585-.205.66.66 0 0 1-.232-.517v-2.491Z" fill="#D0D1D7"/><path d="M136.27 97.288a.66.66 0 0 1 .232-.517.849.849 0 0 1 .585-.204h6.668c.235 0 .43.068.585.204a.66.66 0 0 1 .231.517v2.491a.66.66 0 0 1-.231.517.85.85 0 0 1-.585.205h-6.668a.847.847 0 0 1-.585-.205.657.657 0 0 1-.232-.517v-2.491Z" fill="#D0D1D7"/><path d="M142.587 97.288c0-.208.078-.38.232-.517a.852.852 0 0 1 .585-.204h6.668a.85.85 0 0 1 .585.204.658.658 0 0 1 .232.517v2.491a.657.657 0 0 1-.232.517.849.849 0 0 1-.585.205h-6.668a.85.85 0 0 1-.585-.205.66.66 0 0 1-.232-.517v-2.491Z" fill="#D0D1D7"/><path d="M148.905 97.288a.66.66 0 0 1 .232-.517.849.849 0 0 1 .585-.204h6.667c.236 0 .431.068.586.204a.66.66 0 0 1 .231.517v2.491a.66.66 0 0 1-.231.517.851.851 0 0 1-.586.205h-6.667a.847.847 0 0 1-.585-.205.657.657 0 0 1-.232-.517v-2.491Z" fill="#D0D1D7"/><path d="M155.222 97.298a.66.66 0 0 1 .232-.517.853.853 0 0 1 .585-.205h3.511c.235 0 .43.069.585.205a.662.662 0 0 1 .232.517v2.481a.66.66 0 0 1-.232.517.85.85 0 0 1-.585.205h-3.511a.85.85 0 0 1-.585-.205.657.657 0 0 1-.232-.517v-2.481Z" fill="#D0D1D7"/><rect x="116" y="108.501" width="40" height="9" rx="1" fill="#0096CC"/><g clip-path="url(#c)"><path d="M128.501 111.878a1.123 1.123 0 1 0 1.123 1.123 1.13 1.13 0 0 0-1.123-1.123Zm0 1.855a.731.731 0 0 1-.732-.732c0-.4.322-.723.732-.723.4 0 .723.323.723.723 0 .41-.323.732-.723.732Zm1.426-1.894a.262.262 0 0 0-.264-.264.264.264 0 1 0 .264.264Zm.742.264c-.02-.352-.098-.665-.352-.918-.254-.254-.566-.332-.918-.352-.361-.02-1.445-.02-1.806 0-.352.02-.655.098-.918.352-.254.253-.332.566-.352.918a28.618 28.618 0 0 0 0 1.806c.02.352.098.654.352.918.263.254.566.332.918.352.361.019 1.445.019 1.806 0 .352-.02.664-.098.918-.352.254-.264.332-.566.352-.918.019-.361.019-1.445 0-1.806Zm-.469 2.187a.722.722 0 0 1-.41.42c-.293.117-.977.088-1.289.088-.322 0-1.006.029-1.289-.088a.742.742 0 0 1-.42-.42c-.117-.283-.088-.967-.088-1.289 0-.313-.029-.996.088-1.289a.752.752 0 0 1 .42-.41c.283-.117.967-.088 1.289-.088.312 0 .996-.029 1.289.088a.731.731 0 0 1 .41.41c.117.293.088.976.088 1.289 0 .322.029 1.006-.088 1.289Z" fill="#fff"/></g><path d="M133.287 114.501h.527v-1.236h1.209v-.442h-1.209v-.789h1.337v-.442h-1.864v2.909Zm3.153.043c.639 0 1.045-.451 1.045-1.125 0-.677-.406-1.128-1.045-1.128-.64 0-1.046.451-1.046 1.128 0 .674.406 1.125 1.046 1.125Zm.002-.412c-.353 0-.527-.316-.527-.715 0-.399.174-.719.527-.719.348 0 .522.32.522.719 0 .399-.174.715-.522.715Zm2.034-2.54h-.515v2.909h.515v-2.909Zm1.083 0h-.515v2.909h.515v-2.909Zm1.52 2.952c.639 0 1.045-.451 1.045-1.125 0-.677-.406-1.128-1.045-1.128-.639 0-1.045.451-1.045 1.128 0 .674.406 1.125 1.045 1.125Zm.003-.412c-.354 0-.527-.316-.527-.715 0-.399.173-.719.527-.719.348 0 .521.32.521.719 0 .399-.173.715-.521.715Zm1.911.369h.537l.409-1.474h.029l.41 1.474h.535l.618-2.182h-.526l-.377 1.526h-.022l-.392-1.526h-.518l-.392 1.534h-.02l-.384-1.534h-.524l.617 2.182Z" fill="#fff"/><rect x="24.928" y="125" width="24.928" height="24.928" rx="1" fill="#43A6DB"/><rect x="24.928" y="151.928" width="24.928" height="24.928" rx="1" fill="#43A6DB"/><rect x="24.928" y="178.855" width="24.928" height="24.928" rx="1" fill="#43A6DB"/><rect x="51.855" y="125" width="24.928" height="24.928" rx="1" fill="#43A6DB"/><rect x="51.855" y="151.928" width="24.928" height="24.928" rx="1" fill="#43A6DB"/><rect x="51.855" y="178.855" width="24.928" height="24.928" rx="1" fill="#43A6DB"/><rect x="78.783" y="125" width="24.928" height="24.928" rx="1" fill="#43A6DB"/><rect x="78.783" y="151.928" width="24.928" height="24.928" rx="1" fill="#43A6DB"/><rect x="78.783" y="178.855" width="24.928" height="24.928" rx="1" fill="#43A6DB"/><rect x="105.711" y="125" width="24.928" height="24.928" rx="1" fill="#43A6DB"/><rect x="105.711" y="151.928" width="24.928" height="24.928" rx="1" fill="#43A6DB"/><rect x="105.711" y="178.855" width="24.928" height="24.928" rx="1" fill="#43A6DB"/><rect x="132.639" y="125" width="24.928" height="24.928" rx="1" fill="#43A6DB"/><rect x="132.639" y="151.928" width="24.928" height="24.928" rx="1" fill="#43A6DB"/><rect x="132.639" y="178.855" width="24.928" height="24.928" rx="1" fill="#43A6DB"/><rect x="159.566" y="125" width="24.928" height="24.928" rx="1" fill="#43A6DB"/><rect x="159.566" y="151.928" width="24.928" height="24.928" rx="1" fill="#43A6DB"/><rect x="159.566" y="178.855" width="24.928" height="24.928" rx="1" fill="#43A6DB"/><rect x="186.494" y="125" width="24.928" height="24.928" rx="1" fill="#43A6DB"/><rect x="186.494" y="151.928" width="24.928" height="24.928" rx="1" fill="#43A6DB"/><rect x="186.494" y="178.855" width="24.928" height="24.928" rx="1" fill="#43A6DB"/><rect x="213.422" y="125" width="24.928" height="24.928" rx="1" fill="#43A6DB"/><rect x="213.422" y="151.928" width="24.928" height="24.928" rx="1" fill="#43A6DB"/><rect x="213.422" y="178.855" width="24.928" height="24.928" rx="1" fill="#43A6DB"/></g></g><defs><clipPath id="a"><path fill="#fff" d="M0 0h262.5v200H0z"/></clipPath><clipPath id="c"><path fill="#fff" transform="translate(126 110.501)" d="M0 0h5v5H0z"/></clipPath><filter id="b" x="5" y="34" width="253" height="212" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="6"/><feGaussianBlur stdDeviation="6.5"/><feColorMatrix values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/><feBlend in2="BackgroundImageFix" result="effect1_dropShadow_2180_28831"/><feColorMatrix in="SourceAlpha" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="1"/><feGaussianBlur stdDeviation="1"/><feColorMatrix values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/><feBlend in2="effect1_dropShadow_2180_28831" result="effect2_dropShadow_2180_28831"/><feColorMatrix in="SourceAlpha" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="3"/><feGaussianBlur stdDeviation="3"/><feColorMatrix values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/><feBlend in2="effect2_dropShadow_2180_28831" result="effect3_dropShadow_2180_28831"/><feBlend in="SourceGraphic" in2="effect3_dropShadow_2180_28831" result="shape"/></filter></defs></svg>',
			'simpleRowIcon'          => '<svg width="263" height="200" fill="none" xmlns="http://www.w3.org/2000/svg"><g clip-path="url(#a)"><path fill="#F3F4F5" d="M.5 0H263v200H.5z"/><g filter="url(#b)"><rect x="13" y="58" width="235" height="93" rx="2" fill="#fff"/><path fill="#B6DDAD" d="M26.346 81h41.152v41.152H26.346z"/><path fill="#96CE89" d="M67.498 81h42.047v41.152H67.498z"/><path fill="#B6DDAD" d="M109.545 81h41.152v41.152h-41.152z"/><path fill="#96CE89" d="M150.695 81h41.152v41.152h-41.152z"/><path fill="#B6DDAD" d="M191.848 81H233v41.152h-41.152z"/></g></g><defs><clipPath id="a"><path fill="#fff" transform="translate(.5)" d="M0 0h262.5v200H0z"/></clipPath><filter id="b" x="0" y="51" width="261" height="119" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="6"/><feGaussianBlur stdDeviation="6.5"/><feColorMatrix values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/><feBlend in2="BackgroundImageFix" result="effect1_dropShadow_2180_28871"/><feColorMatrix in="SourceAlpha" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="1"/><feGaussianBlur stdDeviation="1"/><feColorMatrix values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/><feBlend in2="effect1_dropShadow_2180_28871" result="effect2_dropShadow_2180_28871"/><feColorMatrix in="SourceAlpha" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="3"/><feGaussianBlur stdDeviation="3"/><feColorMatrix values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/><feBlend in2="effect2_dropShadow_2180_28871" result="effect3_dropShadow_2180_28871"/><feBlend in="SourceGraphic" in2="effect3_dropShadow_2180_28871" result="shape"/></filter></defs></svg>',
			'simpleCarouselIcon'       => '<svg width="263" height="200" viewBox="0 0 263 200" fill="none" xmlns="http://www.w3.org/2000/svg"><rect width="262.5" height="200" fill="#F3F4F5"/><g filter="url(#filter0_ddd_2180_28884)"><rect x="30" y="45" width="202" height="119" rx="2" fill="white"/><g clip-path="url(#clip0_2180_28884)"><rect x="51.3457" y="83.9258" width="51.1589" height="51.1589" rx="1" fill="#43A6DB"/><circle cx="50" cy="111" r="31" fill="#86D0F9"/></g><g clip-path="url(#clip1_2180_28884)"><rect x="104.729" y="83.9258" width="52.271" height="51.1589" rx="1" fill="#43A6DB"/><rect x="123.926" y="95.5449" width="52.2221" height="52.2221" transform="rotate(10 123.926 95.5449)" fill="#86D0F9"/></g><g clip-path="url(#clip2_2180_28884)"><rect x="159.225" y="83.9258" width="51.1589" height="51.1589" rx="1" fill="#43A6DB"/><rect width="52.2221" height="52.2221" transform="matrix(-0.984808 0.173648 0.173648 0.984808 238.287 95.5449)" fill="#86D0F9"/></g><path d="M53.8457 69.1211H55.2911L56.9144 63.4458H56.9775L58.5968 69.1211H60.0423L62.325 61.0327H60.7492L59.2919 66.9805H59.2208L57.6608 61.0327H56.2272L54.6711 66.9766H54.5961L53.1387 61.0327H51.5629L53.8457 69.1211ZM65.1219 69.2396C66.5358 69.2396 67.5073 68.5484 67.7601 67.4939L66.4252 67.3439C66.2317 67.8573 65.7578 68.1258 65.1417 68.1258C64.2175 68.1258 63.6053 67.5176 63.5935 66.4789H67.8193V66.0406C67.8193 63.9118 66.5397 62.9758 65.0469 62.9758C63.3091 62.9758 62.1757 64.2515 62.1757 66.1235C62.1757 68.0271 63.2933 69.2396 65.1219 69.2396ZM63.5974 65.5153C63.6409 64.7412 64.2135 64.0896 65.0666 64.0896C65.8881 64.0896 66.441 64.6899 66.4489 65.5153H63.5974ZM70.0189 61.0327H68.8815V61.8463C68.8815 62.4901 68.6287 62.9837 68.3404 63.4182L69.0039 63.8487C69.5844 63.3668 70.0189 62.5375 70.0189 61.8384V61.0327ZM70.7236 69.1211H72.1532V65.5548C72.1532 64.7847 72.7338 64.2396 73.5197 64.2396C73.7607 64.2396 74.0608 64.2831 74.1832 64.3226V63.0074C74.0529 62.9837 73.8278 62.9679 73.6698 62.9679C72.9747 62.9679 72.3942 63.3629 72.173 64.0659H72.1098V63.0548H70.7236V69.1211ZM77.3464 69.2396C78.7603 69.2396 79.7319 68.5484 79.9846 67.4939L78.6497 67.3439C78.4562 67.8573 77.9823 68.1258 77.3662 68.1258C76.442 68.1258 75.8299 67.5176 75.818 66.4789H80.0439V66.0406C80.0439 63.9118 78.7643 62.9758 77.2714 62.9758C75.5337 62.9758 74.4002 64.2515 74.4002 66.1235C74.4002 68.0271 75.5179 69.2396 77.3464 69.2396ZM75.822 65.5153C75.8654 64.7412 76.4381 64.0896 77.2911 64.0896C78.1126 64.0896 78.6655 64.6899 78.6734 65.5153H75.822ZM86.2133 69.2396C87.9905 69.2396 89.12 67.9876 89.12 66.1117C89.12 64.2317 87.9905 62.9758 86.2133 62.9758C84.436 62.9758 83.3065 64.2317 83.3065 66.1117C83.3065 67.9876 84.436 69.2396 86.2133 69.2396ZM86.2211 68.0943C85.2378 68.0943 84.7559 67.2175 84.7559 66.1077C84.7559 64.9979 85.2378 64.1093 86.2211 64.1093C87.1888 64.1093 87.6706 64.9979 87.6706 66.1077C87.6706 67.2175 87.1888 68.0943 86.2211 68.0943ZM91.5407 65.5666C91.5407 64.6899 92.0699 64.1843 92.8243 64.1843C93.5628 64.1843 94.0051 64.6701 94.0051 65.4798V69.1211H95.4348V65.2586C95.4388 63.8052 94.6094 62.9758 93.3574 62.9758C92.4491 62.9758 91.8251 63.4103 91.5486 64.0856H91.4775V63.0548H90.111V69.1211H91.5407V65.5666ZM100.73 61.0327H99.2652V69.1211H100.73V61.0327ZM103.461 65.5666C103.461 64.6899 103.99 64.1843 104.745 64.1843C105.483 64.1843 105.926 64.6701 105.926 65.4798V69.1211H107.355V65.2586C107.359 63.8052 106.53 62.9758 105.278 62.9758C104.369 62.9758 103.745 63.4103 103.469 64.0856H103.398V63.0548H102.031V69.1211H103.461V65.5666ZM113.381 64.6583C113.183 63.6314 112.362 62.9758 110.94 62.9758C109.479 62.9758 108.483 63.6946 108.487 64.8163C108.483 65.7009 109.029 66.2854 110.194 66.5263L111.228 66.7436C111.785 66.866 112.046 67.0911 112.046 67.4347C112.046 67.8494 111.596 68.1614 110.916 68.1614C110.261 68.1614 109.834 67.877 109.712 67.332L108.318 67.4663C108.495 68.58 109.431 69.2396 110.92 69.2396C112.437 69.2396 113.507 68.4536 113.511 67.3044C113.507 66.4395 112.95 65.9102 111.805 65.6614L110.77 65.4403C110.154 65.302 109.909 65.0888 109.913 64.7373C109.909 64.3265 110.363 64.0422 110.96 64.0422C111.619 64.0422 111.967 64.4016 112.077 64.8005L113.381 64.6583ZM117.497 63.0548H116.3V61.6014H114.87V63.0548H114.009V64.1607H114.87V67.5334C114.862 68.6748 115.692 69.2356 116.766 69.204C117.173 69.1922 117.453 69.1132 117.607 69.0619L117.366 67.9442C117.287 67.9639 117.125 67.9995 116.948 67.9995C116.588 67.9995 116.3 67.8731 116.3 67.2965V64.1607H117.497V63.0548ZM120.216 69.2435C121.167 69.2435 121.736 68.7972 121.997 68.2878H122.044V69.1211H123.419V65.0611C123.419 63.4577 122.111 62.9758 120.954 62.9758C119.678 62.9758 118.699 63.5445 118.383 64.6504L119.718 64.8399C119.86 64.4253 120.263 64.0698 120.962 64.0698C121.626 64.0698 121.989 64.4095 121.989 65.0058V65.0295C121.989 65.4403 121.558 65.46 120.488 65.5745C119.311 65.7009 118.186 66.0524 118.186 67.4189C118.186 68.6116 119.058 69.2435 120.216 69.2435ZM120.587 68.193C119.99 68.193 119.564 67.9205 119.564 67.3952C119.564 66.8462 120.042 66.6172 120.682 66.5263C121.057 66.475 121.807 66.3802 121.993 66.2301V66.945C121.993 67.6203 121.448 68.193 120.587 68.193ZM127.272 71.5223C128.911 71.5223 130.119 70.7719 130.119 69.2198V63.0548H128.709V64.058H128.631C128.413 63.6196 127.959 62.9758 126.889 62.9758C125.487 62.9758 124.385 64.0738 124.385 66.0682C124.385 68.0469 125.487 69.0342 126.885 69.0342C127.924 69.0342 128.409 68.4773 128.631 68.0311H128.702V69.1803C128.702 70.0729 128.109 70.4402 127.296 70.4402C126.435 70.4402 126.056 70.0334 125.882 69.678L124.594 69.99C124.855 70.8391 125.728 71.5223 127.272 71.5223ZM127.284 67.9126C126.34 67.9126 125.842 67.178 125.842 66.0603C125.842 64.9584 126.332 64.1488 127.284 64.1488C128.204 64.1488 128.709 64.911 128.709 66.0603C128.709 67.2175 128.196 67.9126 127.284 67.9126ZM131.362 69.1211H132.792V65.5548C132.792 64.7847 133.372 64.2396 134.158 64.2396C134.399 64.2396 134.699 64.2831 134.822 64.3226V63.0074C134.692 62.9837 134.466 62.9679 134.308 62.9679C133.613 62.9679 133.033 63.3629 132.812 64.0659H132.748V63.0548H131.362V69.1211ZM137.175 69.2435C138.127 69.2435 138.696 68.7972 138.957 68.2878H139.004V69.1211H140.378V65.0611C140.378 63.4577 139.071 62.9758 137.914 62.9758C136.638 62.9758 135.659 63.5445 135.343 64.6504L136.678 64.8399C136.82 64.4253 137.223 64.0698 137.922 64.0698C138.585 64.0698 138.949 64.4095 138.949 65.0058V65.0295C138.949 65.4403 138.518 65.46 137.448 65.5745C136.271 65.7009 135.145 66.0524 135.145 67.4189C135.145 68.6116 136.018 69.2435 137.175 69.2435ZM137.547 68.193C136.95 68.193 136.524 67.9205 136.524 67.3952C136.524 66.8462 137.002 66.6172 137.641 66.5263C138.017 66.475 138.767 66.3802 138.953 66.2301V66.945C138.953 67.6203 138.408 68.193 137.547 68.193ZM141.594 69.1211H143.023V65.4324C143.023 64.6859 143.521 64.1804 144.137 64.1804C144.741 64.1804 145.156 64.5872 145.156 65.2112V69.1211H146.558V65.3376C146.558 64.6543 146.965 64.1804 147.656 64.1804C148.232 64.1804 148.691 64.52 148.691 65.2704V69.1211H150.124V65.0493C150.124 63.6946 149.342 62.9758 148.229 62.9758C147.348 62.9758 146.676 63.4103 146.408 64.0856H146.345C146.112 63.3984 145.523 62.9758 144.706 62.9758C143.892 62.9758 143.284 63.3945 143.031 64.0856H142.96V63.0548H141.594V69.1211ZM153.102 61.0327H151.573L151.7 66.7436H152.971L153.102 61.0327ZM152.336 69.208C152.802 69.208 153.201 68.8209 153.205 68.3391C153.201 67.8652 152.802 67.4781 152.336 67.4781C151.854 67.4781 151.463 67.8652 151.467 68.3391C151.463 68.8209 151.854 69.208 152.336 69.208Z" fill="black"/><circle cx="115" cy="148" r="2" fill="#434960"/><circle cx="124" cy="148" r="2" fill="#D0D1D7"/><circle cx="133" cy="148" r="2" fill="#D0D1D7"/><circle cx="142" cy="148" r="2" fill="#D0D1D7"/><circle cx="151" cy="148" r="2" fill="#D0D1D7"/></g><defs><filter id="filter0_ddd_2180_28884" x="17" y="38" width="228" height="145" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="6"/><feGaussianBlur stdDeviation="6.5"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_2180_28884"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="1"/><feGaussianBlur stdDeviation="1"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/><feBlend mode="normal" in2="effect1_dropShadow_2180_28884" result="effect2_dropShadow_2180_28884"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="3"/><feGaussianBlur stdDeviation="3"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/><feBlend mode="normal" in2="effect2_dropShadow_2180_28884" result="effect3_dropShadow_2180_28884"/><feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow_2180_28884" result="shape"/></filter><clipPath id="clip0_2180_28884"><rect x="51.3457" y="83.9258" width="51.1589" height="51.1589" rx="1" fill="white"/></clipPath><clipPath id="clip1_2180_28884"><rect x="104.729" y="83.9258" width="52.271" height="51.1589" rx="1" fill="white"/></clipPath><clipPath id="clip2_2180_28884"><rect x="159.225" y="83.9258" width="51.1589" height="51.1589" rx="1" fill="white"/></clipPath></defs></svg>',
			'masonryCardsIcon'       => '<svg width="263" height="200" viewBox="0 0 263 200" fill="none" xmlns="http://www.w3.org/2000/svg"><g clip-path="url(#clip0_2180_28903)"><rect width="262.5" height="200" transform="translate(0.5)" fill="#F3F4F5"/><g filter="url(#filter0_dd_2180_28903)"><g clip-path="url(#clip1_2180_28903)"><rect x="34.5" y="67" width="63.6871" height="103.927" rx="0.893854" fill="white"/><g clip-path="url(#clip2_2180_28903)"><rect width="63.6871" height="80.8938" transform="translate(34.5 67)" fill="#8C8F9A"/><circle opacity="0.5" cx="107.5" cy="128.5" r="46.5" fill="#D0D1D7"/></g><rect x="38.5" y="151.894" width="55.6871" height="3" rx="0.5" fill="#D0D1D7"/><path d="M38.5 158.394C38.5 158.117 38.7239 157.894 39 157.894H71C71.2761 157.894 71.5 158.117 71.5 158.394V160.394C71.5 160.67 71.2761 160.894 71 160.894H39C38.7239 160.894 38.5 160.67 38.5 160.394V158.394Z" fill="#D0D1D7"/><g clip-path="url(#clip3_2180_28903)"><path d="M40.959 164.021V164.044C41.2753 164.044 41.5325 164.301 41.5325 164.617C41.5325 164.957 41.3308 165.284 41.0705 165.568C40.8157 165.845 40.5078 166.077 40.2886 166.232C40.0694 166.077 39.7615 165.845 39.5067 165.568C39.2463 165.284 39.0446 164.957 39.0446 164.617C39.0446 164.301 39.3019 164.044 39.6182 164.044V164.021L39.6181 164.044C39.7103 164.044 39.801 164.066 39.8826 164.109C39.9643 164.151 40.0346 164.213 40.0875 164.288L40.0875 164.288L40.2703 164.549L40.2886 164.575L40.3069 164.549L40.4897 164.288L40.4897 164.288C40.5425 164.213 40.6128 164.151 40.6945 164.109C40.7761 164.066 40.8669 164.044 40.959 164.044L40.959 164.021ZM40.959 164.021C41.2876 164.021 41.5549 164.289 41.5549 164.617C41.5549 165.313 40.7291 165.949 40.2886 166.259L40.4714 164.276C40.5263 164.197 40.5993 164.133 40.6841 164.089C40.769 164.045 40.8633 164.021 40.959 164.021ZM40.2886 166.559L40.2759 166.55C40.0151 166.37 39.6409 166.104 39.3319 165.774C39.0232 165.446 38.7765 165.052 38.7765 164.617C38.7765 164.394 38.8652 164.18 39.023 164.022C39.1809 163.864 39.395 163.776 39.6182 163.776L40.2886 166.559ZM40.2886 166.559L40.3013 166.55M40.2886 166.559L40.3013 166.55M40.3013 166.55C40.5621 166.37 40.9362 166.104 41.2453 165.774C41.554 165.446 41.8007 165.052 41.8007 164.617C41.8007 164.507 41.7789 164.397 41.7366 164.295C41.6943 164.193 41.6323 164.1 41.5542 164.022C41.476 163.944 41.3832 163.882 41.2811 163.84C41.179 163.797 41.0695 163.776 40.959 163.776C40.8238 163.775 40.6906 163.808 40.5707 163.871M40.3013 166.55L40.5707 163.871M40.5707 163.871C40.46 163.928 40.3637 164.01 40.2886 164.109M40.5707 163.871L40.2886 164.109M40.2886 164.109C40.2135 164.01 40.1172 163.928 40.0065 163.871C39.8866 163.808 39.7534 163.775 39.6182 163.776L40.2886 164.109Z" fill="#434960" stroke="#434960" stroke-width="0.0446927"/></g><g clip-path="url(#clip4_2180_28903)"><path d="M48.3318 163.648C48.1362 163.648 47.9425 163.687 47.7617 163.762C47.581 163.837 47.4168 163.946 47.2784 164.085C46.999 164.364 46.8421 164.743 46.8421 165.138C46.8408 165.482 46.9599 165.816 47.1788 166.081L46.8808 166.379C46.8602 166.4 46.8461 166.427 46.8406 166.456C46.835 166.485 46.8381 166.514 46.8495 166.542C46.8619 166.568 46.882 166.591 46.9072 166.606C46.9324 166.622 46.9616 166.629 46.9911 166.628H48.3318C48.727 166.628 49.1059 166.471 49.3853 166.192C49.6646 165.912 49.8216 165.533 49.8216 165.138C49.8216 164.743 49.6646 164.364 49.3853 164.085C49.1059 163.805 48.727 163.648 48.3318 163.648ZM48.3318 166.33H47.3501L47.4886 166.191C47.5164 166.164 47.532 166.126 47.532 166.086C47.532 166.047 47.5164 166.009 47.4886 165.981C47.2936 165.787 47.1721 165.53 47.1449 165.256C47.1177 164.981 47.1865 164.706 47.3396 164.477C47.4926 164.247 47.7204 164.078 47.9842 163.998C48.248 163.918 48.5315 163.931 48.7863 164.037C49.0411 164.142 49.2515 164.333 49.3816 164.576C49.5117 164.819 49.5536 165.099 49.5 165.37C49.4464 165.64 49.3007 165.884 49.0877 166.059C48.8747 166.234 48.6076 166.33 48.3318 166.33Z" fill="#434960"/></g></g></g><g filter="url(#filter1_dd_2180_28903)"><g clip-path="url(#clip5_2180_28903)"><rect x="34.5" y="174.279" width="63.6871" height="105.475" rx="0.893854" fill="white"/><g clip-path="url(#clip6_2180_28903)"><rect width="63.6871" height="80.8938" transform="translate(34.5 174.279)" fill="#8C8F9A"/></g></g><rect x="34.6117" y="174.391" width="63.4636" height="105.251" rx="0.782122" stroke="#F3F4F5" stroke-width="0.223464"/></g><g filter="url(#filter2_dd_2180_28903)"><g clip-path="url(#clip7_2180_28903)"><rect x="102.656" y="67.001" width="63.6871" height="82.4748" rx="0.893854" fill="white"/><rect width="63.6871" height="59.4413" transform="translate(102.656 67.001)" fill="#8C8F9A"/><rect x="126.119" y="88.2305" width="16.9832" height="16.9832" rx="8.49161" fill="white"/><path d="M132.6 94.6262C132.6 94.4601 132.774 94.3521 132.923 94.4263L137.786 96.8581C137.951 96.9404 137.951 97.1755 137.786 97.2578L132.923 99.6895C132.774 99.7638 132.6 99.6558 132.6 99.4897V94.6262Z" fill="black"/><rect x="106.656" y="130.442" width="55.6871" height="3" rx="0.5" fill="#D0D1D7"/><path d="M106.656 136.942C106.656 136.666 106.88 136.442 107.156 136.442H139.156C139.432 136.442 139.656 136.666 139.656 136.942V138.942C139.656 139.219 139.432 139.442 139.156 139.442H107.156C106.88 139.442 106.656 139.219 106.656 138.942V136.942Z" fill="#D0D1D7"/><g clip-path="url(#clip8_2180_28903)"><path d="M109.115 142.57V142.592C109.432 142.592 109.689 142.85 109.689 143.166C109.689 143.505 109.487 143.833 109.227 144.117C108.972 144.394 108.664 144.626 108.445 144.781C108.226 144.626 107.918 144.394 107.663 144.117C107.403 143.833 107.201 143.505 107.201 143.166C107.201 142.85 107.458 142.592 107.774 142.592V142.57L107.774 142.592C107.867 142.593 107.957 142.615 108.039 142.658C108.121 142.7 108.191 142.762 108.244 142.837L108.244 142.837L108.427 143.097L108.445 143.123L108.463 143.097L108.646 142.837L108.646 142.837C108.699 142.762 108.769 142.7 108.851 142.658C108.932 142.615 109.023 142.593 109.115 142.592L109.115 142.57ZM109.115 142.57C109.444 142.57 109.711 142.837 109.711 143.166C109.711 143.862 108.885 144.498 108.445 144.808L108.628 142.824C108.683 142.746 108.756 142.682 108.84 142.638C108.925 142.593 109.02 142.57 109.115 142.57ZM108.445 145.108L108.432 145.099C108.171 144.919 107.797 144.652 107.488 144.323C107.179 143.995 106.933 143.601 106.933 143.166C106.933 142.943 107.021 142.729 107.179 142.571C107.337 142.413 107.551 142.324 107.774 142.324L108.445 145.108ZM108.445 145.108L108.458 145.099M108.445 145.108L108.458 145.099M108.458 145.099C108.718 144.919 109.092 144.652 109.402 144.323C109.71 143.995 109.957 143.601 109.957 143.166C109.957 143.056 109.935 142.946 109.893 142.844C109.851 142.742 109.789 142.649 109.71 142.571C109.632 142.493 109.539 142.431 109.437 142.388C109.335 142.346 109.226 142.324 109.115 142.324C108.98 142.324 108.847 142.357 108.727 142.419M108.458 145.099L108.727 142.419M108.727 142.419C108.616 142.477 108.52 142.559 108.445 142.658M108.727 142.419L108.445 142.658M108.445 142.658C108.37 142.559 108.273 142.477 108.163 142.419C108.043 142.357 107.91 142.324 107.774 142.324L108.445 142.658Z" fill="#434960" stroke="#434960" stroke-width="0.0446927"/></g><g clip-path="url(#clip9_2180_28903)"><path d="M116.488 142.197C116.292 142.197 116.099 142.236 115.918 142.311C115.737 142.386 115.573 142.495 115.435 142.634C115.155 142.913 114.998 143.292 114.998 143.687C114.997 144.031 115.116 144.365 115.335 144.63L115.037 144.928C115.016 144.949 115.002 144.976 114.997 145.004C114.991 145.033 114.994 145.063 115.006 145.09C115.018 145.117 115.038 145.14 115.063 145.155C115.089 145.17 115.118 145.178 115.147 145.177H116.488C116.883 145.177 117.262 145.02 117.542 144.74C117.821 144.461 117.978 144.082 117.978 143.687C117.978 143.292 117.821 142.913 117.542 142.634C117.262 142.354 116.883 142.197 116.488 142.197ZM116.488 144.879H115.506L115.645 144.74C115.673 144.712 115.688 144.675 115.688 144.635C115.688 144.596 115.673 144.558 115.645 144.53C115.45 144.335 115.328 144.079 115.301 143.805C115.274 143.53 115.343 143.255 115.496 143.025C115.649 142.796 115.877 142.627 116.14 142.547C116.404 142.467 116.688 142.48 116.943 142.586C117.197 142.691 117.408 142.881 117.538 143.124C117.668 143.368 117.71 143.648 117.656 143.919C117.603 144.189 117.457 144.433 117.244 144.608C117.031 144.783 116.764 144.879 116.488 144.879Z" fill="#434960"/></g></g></g><g filter="url(#filter3_dd_2180_28903)"><g clip-path="url(#clip10_2180_28903)"><rect x="102.656" y="152.828" width="63.6871" height="105.475" rx="0.893854" fill="white"/><rect width="63.6871" height="80.8938" transform="translate(102.656 152.828)" fill="#8C8F9A"/></g><rect x="102.768" y="152.94" width="63.4636" height="105.251" rx="0.782122" stroke="#F3F4F5" stroke-width="0.223464"/></g><g filter="url(#filter4_dd_2180_28903)"><g clip-path="url(#clip11_2180_28903)"><rect x="170.812" y="67.001" width="63.6871" height="103.48" rx="0.893854" fill="white"/><g clip-path="url(#clip12_2180_28903)"><rect width="63.6871" height="80.8938" transform="translate(170.812 67.001)" fill="#8C8F9A"/><rect opacity="0.5" x="149" y="147.841" width="62" height="68" transform="rotate(-45 149 147.841)" fill="#D0D1D7"/></g><rect x="174.812" y="151.448" width="55.6871" height="3" rx="0.5" fill="#D0D1D7"/><path d="M174.812 157.948C174.812 157.672 175.036 157.448 175.312 157.448H207.312C207.589 157.448 207.812 157.672 207.812 157.948V159.948C207.812 160.224 207.589 160.448 207.312 160.448H175.312C175.036 160.448 174.812 160.224 174.812 159.948V157.948Z" fill="#D0D1D7"/><g clip-path="url(#clip13_2180_28903)"><path d="M177.271 163.576V163.598C177.588 163.598 177.845 163.856 177.845 164.172C177.845 164.511 177.643 164.839 177.383 165.122C177.128 165.4 176.82 165.632 176.601 165.787C176.382 165.632 176.074 165.4 175.819 165.122C175.559 164.839 175.357 164.511 175.357 164.172C175.357 163.856 175.614 163.598 175.931 163.598V163.576L175.931 163.598C176.023 163.599 176.113 163.621 176.195 163.663C176.277 163.706 176.347 163.768 176.4 163.843L176.4 163.843L176.583 164.103L176.601 164.129L176.619 164.103L176.802 163.843L176.802 163.843C176.855 163.768 176.925 163.706 177.007 163.663C177.089 163.621 177.179 163.599 177.272 163.598L177.271 163.576ZM177.271 163.576C177.6 163.576 177.867 163.843 177.867 164.172C177.867 164.868 177.042 165.504 176.601 165.814L176.784 163.83C176.839 163.752 176.912 163.688 176.997 163.644C177.081 163.599 177.176 163.576 177.271 163.576ZM176.601 166.113L176.588 166.105C176.328 165.925 175.953 165.658 175.644 165.329C175.336 165.001 175.089 164.606 175.089 164.172C175.089 163.949 175.178 163.735 175.336 163.577C175.493 163.419 175.707 163.33 175.931 163.33L176.601 166.113ZM176.601 166.113L176.614 166.105M176.601 166.113L176.614 166.105M176.614 166.105C176.875 165.925 177.249 165.658 177.558 165.329C177.866 165.001 178.113 164.606 178.113 164.172C178.113 164.061 178.091 163.952 178.049 163.85C178.007 163.748 177.945 163.655 177.867 163.577C177.788 163.499 177.696 163.437 177.594 163.394C177.491 163.352 177.382 163.33 177.271 163.33C177.136 163.33 177.003 163.363 176.883 163.425M176.614 166.105L176.883 163.425M176.883 163.425C176.773 163.483 176.676 163.564 176.601 163.664M176.883 163.425L176.601 163.664M176.601 163.664C176.526 163.564 176.43 163.483 176.319 163.425C176.199 163.363 176.066 163.33 175.931 163.33L176.601 163.664Z" fill="#434960" stroke="#434960" stroke-width="0.0446927"/></g><g clip-path="url(#clip14_2180_28903)"><path d="M184.644 163.203C184.449 163.203 184.255 163.242 184.074 163.317C183.893 163.391 183.729 163.501 183.591 163.639C183.312 163.919 183.155 164.298 183.155 164.693C183.153 165.037 183.272 165.371 183.491 165.636L183.193 165.934C183.173 165.955 183.159 165.981 183.153 166.01C183.148 166.039 183.151 166.069 183.162 166.096C183.174 166.123 183.194 166.146 183.22 166.161C183.245 166.176 183.274 166.184 183.304 166.183H184.644C185.039 166.183 185.418 166.026 185.698 165.746C185.977 165.467 186.134 165.088 186.134 164.693C186.134 164.298 185.977 163.919 185.698 163.639C185.418 163.36 185.039 163.203 184.644 163.203ZM184.644 165.885H183.663L183.801 165.746C183.829 165.718 183.844 165.68 183.844 165.641C183.844 165.602 183.829 165.564 183.801 165.536C183.606 165.341 183.485 165.085 183.457 164.81C183.43 164.536 183.499 164.261 183.652 164.031C183.805 163.802 184.033 163.633 184.297 163.553C184.561 163.472 184.844 163.486 185.099 163.591C185.354 163.697 185.564 163.887 185.694 164.13C185.824 164.373 185.866 164.654 185.812 164.925C185.759 165.195 185.613 165.439 185.4 165.614C185.187 165.789 184.92 165.884 184.644 165.885Z" fill="#434960"/></g></g></g><g filter="url(#filter5_dd_2180_28903)"><g clip-path="url(#clip15_2180_28903)"><rect x="170.812" y="173.833" width="63.6871" height="105.475" rx="0.893854" fill="white"/><g clip-path="url(#clip16_2180_28903)"><rect width="63.6871" height="80.8938" transform="translate(170.812 173.833)" fill="#8C8F9A"/></g></g><rect x="170.924" y="173.945" width="63.4636" height="105.251" rx="0.782122" stroke="#F3F4F5" stroke-width="0.223464"/></g><circle cx="45.9821" cy="37.9821" r="10.9821" fill="#8C8F9A"/><path d="M65.2012 29.146C65.2012 28.9114 65.2782 28.717 65.4322 28.563C65.5862 28.409 65.7805 28.332 66.0152 28.332H72.6592C72.8938 28.332 73.0882 28.409 73.2422 28.563C73.3962 28.717 73.4732 28.9114 73.4732 29.146V31.818C73.4732 32.0527 73.3962 32.247 73.2422 32.401C73.0882 32.555 72.8938 32.632 72.6592 32.632H66.0152C65.7805 32.632 65.5862 32.555 65.4322 32.401C65.2782 32.247 65.2012 32.0527 65.2012 31.818V29.146Z" fill="#8C8F9A"/><path d="M71.4961 29.146C71.4961 28.9114 71.5731 28.717 71.7271 28.563C71.8811 28.409 72.0754 28.332 72.3101 28.332H78.9541C79.1888 28.332 79.3831 28.409 79.5371 28.563C79.6911 28.717 79.7681 28.9114 79.7681 29.146V31.818C79.7681 32.0527 79.6911 32.247 79.5371 32.401C79.3831 32.555 79.1888 32.632 78.9541 32.632H72.3101C72.0754 32.632 71.8811 32.555 71.7271 32.401C71.5731 32.247 71.4961 32.0527 71.4961 31.818V29.146Z" fill="#8C8F9A"/><path d="M77.791 29.146C77.791 28.9114 77.868 28.717 78.022 28.563C78.176 28.409 78.3703 28.332 78.605 28.332H82.249C82.4837 28.332 82.678 28.409 82.832 28.563C82.986 28.717 83.063 28.9114 83.063 29.146V31.818C83.063 32.0527 82.986 32.247 82.832 32.401C82.678 32.555 82.4837 32.632 82.249 32.632H78.605C78.3703 32.632 78.176 32.555 78.022 32.401C77.868 32.247 77.791 32.0527 77.791 31.818V29.146Z" fill="#8C8F9A"/><path d="M81.0859 29.146C81.0859 28.9114 81.1629 28.717 81.3169 28.563C81.4709 28.409 81.6653 28.332 81.8999 28.332H88.5439C88.7786 28.332 88.9729 28.409 89.1269 28.563C89.2809 28.717 89.3579 28.9114 89.3579 29.146V31.818C89.3579 32.0527 89.2809 32.247 89.1269 32.401C88.9729 32.555 88.7786 32.632 88.5439 32.632H81.8999C81.6653 32.632 81.4709 32.555 81.3169 32.401C81.1629 32.247 81.0859 32.0527 81.0859 31.818V29.146Z" fill="#8C8F9A"/><path d="M87.3809 29.157C87.3809 28.9224 87.4579 28.728 87.6119 28.574C87.7659 28.42 87.9602 28.343 88.1949 28.343H91.6929C91.9275 28.343 92.1219 28.42 92.2759 28.574C92.4299 28.728 92.5069 28.9224 92.5069 29.157V31.818C92.5069 32.0527 92.4299 32.247 92.2759 32.401C92.1219 32.555 91.9275 32.632 91.6929 32.632H88.1949C87.9602 32.632 87.7659 32.555 87.6119 32.401C87.4579 32.247 87.3809 32.0527 87.3809 31.818V29.157Z" fill="#8C8F9A"/><path d="M90.5283 29.146C90.5283 28.9114 90.6053 28.717 90.7593 28.563C90.9133 28.409 91.1077 28.332 91.3423 28.332H97.9863C98.221 28.332 98.4153 28.409 98.5693 28.563C98.7233 28.717 98.8003 28.9114 98.8003 29.146V31.818C98.8003 32.0527 98.7233 32.247 98.5693 32.401C98.4153 32.555 98.221 32.632 97.9863 32.632H91.3423C91.1077 32.632 90.9133 32.555 90.7593 32.401C90.6053 32.247 90.5283 32.0527 90.5283 31.818V29.146Z" fill="#8C8F9A"/><path d="M65.2012 37.3534C65.2012 37.1454 65.2784 36.9731 65.433 36.8366C65.5876 36.7001 65.7826 36.6318 66.0181 36.6318H72.6859C72.9214 36.6318 73.1164 36.7001 73.271 36.8366C73.4255 36.9731 73.5028 37.1454 73.5028 37.3534V39.8448C73.5028 40.0528 73.4255 40.2251 73.271 40.3616C73.1164 40.4981 72.9214 40.5663 72.6859 40.5663H66.0181C65.7826 40.5663 65.5876 40.4981 65.433 40.3616C65.2784 40.2251 65.2012 40.0528 65.2012 39.8448V37.3534Z" fill="#D0D1D7"/><path d="M71.5186 37.3534C71.5186 37.1454 71.5959 36.9731 71.7505 36.8366C71.905 36.7001 72.1 36.6318 72.3355 36.6318H79.0033C79.2388 36.6318 79.4339 36.7001 79.5884 36.8366C79.743 36.9731 79.8202 37.1454 79.8202 37.3534V39.8448C79.8202 40.0528 79.743 40.2251 79.5884 40.3616C79.4339 40.4981 79.2388 40.5663 79.0033 40.5663H72.3355C72.1 40.5663 71.905 40.4981 71.7505 40.3616C71.5959 40.2251 71.5186 40.0528 71.5186 39.8448V37.3534Z" fill="#D0D1D7"/><path d="M77.8361 37.3534C77.8361 37.1454 77.9134 36.9731 78.0679 36.8366C78.2225 36.7001 78.4175 36.6318 78.653 36.6318H85.3208C85.5563 36.6318 85.7513 36.7001 85.9059 36.8366C86.0604 36.9731 86.1377 37.1454 86.1377 37.3534V39.8448C86.1377 40.0528 86.0604 40.2251 85.9059 40.3616C85.7513 40.4981 85.5563 40.5663 85.3208 40.5663H78.653C78.4175 40.5663 78.2225 40.4981 78.0679 40.3616C77.9134 40.2251 77.8361 40.0528 77.8361 39.8448V37.3534Z" fill="#D0D1D7"/><path d="M84.1535 37.3534C84.1535 37.1454 84.2308 36.9731 84.3854 36.8366C84.5399 36.7001 84.735 36.6318 84.9705 36.6318H91.6382C91.8738 36.6318 92.0688 36.7001 92.2233 36.8366C92.3779 36.9731 92.4552 37.1454 92.4552 37.3534V39.8448C92.4552 40.0528 92.3779 40.2251 92.2233 40.3616C92.0688 40.4981 91.8738 40.5663 91.6382 40.5663H84.9705C84.735 40.5663 84.5399 40.4981 84.3854 40.3616C84.2308 40.2251 84.1535 40.0528 84.1535 39.8448V37.3534Z" fill="#D0D1D7"/><path d="M90.471 37.3631C90.471 37.1551 90.5483 36.9829 90.7028 36.8463C90.8574 36.7098 91.0524 36.6416 91.2879 36.6416H94.7984C95.0339 36.6416 95.229 36.7098 95.3835 36.8463C95.5381 36.9829 95.6154 37.1551 95.6154 37.3631V39.8448C95.6154 40.0528 95.5381 40.2251 95.3835 40.3616C95.229 40.4981 95.0339 40.5663 94.7984 40.5663H91.2879C91.0524 40.5663 90.8574 40.4981 90.7028 40.3616C90.5483 40.2251 90.471 40.0528 90.471 39.8448V37.3631Z" fill="#D0D1D7"/><path d="M93.6297 37.3631C93.6297 37.1551 93.707 36.9829 93.8616 36.8463C94.0161 36.7098 94.2111 36.6416 94.4466 36.6416H97.9572C98.1927 36.6416 98.3877 36.7098 98.5423 36.8463C98.6968 36.9829 98.7741 37.1551 98.7741 37.3631V39.8448C98.7741 40.0528 98.6968 40.2251 98.5423 40.3616C98.3877 40.4981 98.1927 40.5663 97.9572 40.5663H94.4466C94.2111 40.5663 94.0161 40.4981 93.8616 40.3616C93.707 40.2251 93.6297 40.0528 93.6297 39.8448V37.3631Z" fill="#D0D1D7"/><path d="M96.7885 37.3534C96.7885 37.1454 96.8657 36.9731 97.0203 36.8366C97.1748 36.7001 97.3699 36.6318 97.6054 36.6318H104.273C104.509 36.6318 104.704 36.7001 104.858 36.8366C105.013 36.9731 105.09 37.1454 105.09 37.3534V39.8448C105.09 40.0528 105.013 40.2251 104.858 40.3616C104.704 40.4981 104.509 40.5663 104.273 40.5663H97.6054C97.3699 40.5663 97.1748 40.4981 97.0203 40.3616C96.8657 40.2251 96.7885 40.0528 96.7885 39.8448V37.3534Z" fill="#D0D1D7"/><path d="M103.106 37.3631C103.106 37.1551 103.183 36.9829 103.338 36.8463C103.492 36.7098 103.687 36.6416 103.923 36.6416H107.433C107.669 36.6416 107.864 36.7098 108.018 36.8463C108.173 36.9829 108.25 37.1551 108.25 37.3631V39.8448C108.25 40.0528 108.173 40.2251 108.018 40.3616C107.864 40.4981 107.669 40.5663 107.433 40.5663H103.923C103.687 40.5663 103.492 40.4981 103.338 40.3616C103.183 40.2251 103.106 40.0528 103.106 39.8448V37.3631Z" fill="#D0D1D7"/><path d="M106.265 37.3534C106.265 37.1454 106.342 36.9731 106.496 36.8366C106.651 36.7001 106.846 36.6318 107.082 36.6318H113.749C113.985 36.6318 114.18 36.7001 114.334 36.8366C114.489 36.9731 114.566 37.1454 114.566 37.3534V39.8448C114.566 40.0528 114.489 40.2251 114.334 40.3616C114.18 40.4981 113.985 40.5663 113.749 40.5663H107.082C106.846 40.5663 106.651 40.4981 106.496 40.3616C106.342 40.2251 106.265 40.0528 106.265 39.8448V37.3534Z" fill="#D0D1D7"/><path d="M112.582 37.3534C112.582 37.1454 112.659 36.9731 112.814 36.8366C112.968 36.7001 113.164 36.6318 113.399 36.6318H120.067C120.302 36.6318 120.497 36.7001 120.652 36.8366C120.806 36.9731 120.884 37.1454 120.884 37.3534V39.8448C120.884 40.0528 120.806 40.2251 120.652 40.3616C120.497 40.4981 120.302 40.5663 120.067 40.5663H113.399C113.164 40.5663 112.968 40.4981 112.814 40.3616C112.659 40.2251 112.582 40.0528 112.582 39.8448V37.3534Z" fill="#D0D1D7"/><path d="M118.9 37.3534C118.9 37.1454 118.977 36.9731 119.131 36.8366C119.286 36.7001 119.481 36.6318 119.716 36.6318H126.384C126.62 36.6318 126.815 36.7001 126.969 36.8366C127.124 36.9731 127.201 37.1454 127.201 37.3534V39.8448C127.201 40.0528 127.124 40.2251 126.969 40.3616C126.815 40.4981 126.62 40.5663 126.384 40.5663H119.716C119.481 40.5663 119.286 40.4981 119.131 40.3616C118.977 40.2251 118.9 40.0528 118.9 39.8448V37.3534Z" fill="#D0D1D7"/><path d="M65.2012 44.4189C65.2012 44.2109 65.2784 44.0386 65.433 43.9021C65.5876 43.7656 65.7826 43.6974 66.0181 43.6974H72.6859C72.9214 43.6974 73.1164 43.7656 73.271 43.9021C73.4255 44.0386 73.5028 44.2109 73.5028 44.4189V46.9103C73.5028 47.1183 73.4255 47.2906 73.271 47.4271C73.1164 47.5636 72.9214 47.6318 72.6859 47.6318H66.0181C65.7826 47.6318 65.5876 47.5636 65.433 47.4271C65.2784 47.2906 65.2012 47.1183 65.2012 46.9103V44.4189Z" fill="#D0D1D7"/><path d="M71.5186 44.4189C71.5186 44.2109 71.5959 44.0386 71.7505 43.9021C71.905 43.7656 72.1 43.6974 72.3355 43.6974H79.0033C79.2388 43.6974 79.4339 43.7656 79.5884 43.9021C79.743 44.0386 79.8202 44.2109 79.8202 44.4189V46.9103C79.8202 47.1183 79.743 47.2906 79.5884 47.4271C79.4339 47.5636 79.2388 47.6318 79.0033 47.6318H72.3355C72.1 47.6318 71.905 47.5636 71.7505 47.4271C71.5959 47.2906 71.5186 47.1183 71.5186 46.9103V44.4189Z" fill="#D0D1D7"/><path d="M77.8361 44.4189C77.8361 44.2109 77.9134 44.0386 78.0679 43.9021C78.2225 43.7656 78.4175 43.6974 78.653 43.6974H85.3208C85.5563 43.6974 85.7513 43.7656 85.9059 43.9021C86.0604 44.0386 86.1377 44.2109 86.1377 44.4189V46.9103C86.1377 47.1183 86.0604 47.2906 85.9059 47.4271C85.7513 47.5636 85.5563 47.6318 85.3208 47.6318H78.653C78.4175 47.6318 78.2225 47.5636 78.0679 47.4271C77.9134 47.2906 77.8361 47.1183 77.8361 46.9103V44.4189Z" fill="#D0D1D7"/><path d="M84.1535 44.4189C84.1535 44.2109 84.2308 44.0386 84.3854 43.9021C84.5399 43.7656 84.735 43.6974 84.9705 43.6974H91.6382C91.8738 43.6974 92.0688 43.7656 92.2233 43.9021C92.3779 44.0386 92.4552 44.2109 92.4552 44.4189V46.9103C92.4552 47.1183 92.3779 47.2906 92.2233 47.4271C92.0688 47.5636 91.8738 47.6318 91.6382 47.6318H84.9705C84.735 47.6318 84.5399 47.5636 84.3854 47.4271C84.2308 47.2906 84.1535 47.1183 84.1535 46.9103V44.4189Z" fill="#D0D1D7"/><path d="M90.471 44.4189C90.471 44.2109 90.5483 44.0386 90.7028 43.9021C90.8574 43.7656 91.0524 43.6974 91.2879 43.6974H97.9557C98.1912 43.6974 98.3862 43.7656 98.5408 43.9021C98.6953 44.0386 98.7726 44.2109 98.7726 44.4189V46.9103C98.7726 47.1183 98.6953 47.2906 98.5408 47.4271C98.3862 47.5636 98.1912 47.6318 97.9557 47.6318H91.2879C91.0524 47.6318 90.8574 47.5636 90.7028 47.4271C90.5483 47.2906 90.471 47.1183 90.471 46.9103V44.4189Z" fill="#D0D1D7"/><path d="M96.7885 44.4189C96.7885 44.2109 96.8657 44.0386 97.0203 43.9021C97.1748 43.7656 97.3699 43.6974 97.6054 43.6974H104.273C104.509 43.6974 104.704 43.7656 104.858 43.9021C105.013 44.0386 105.09 44.2109 105.09 44.4189V46.9103C105.09 47.1183 105.013 47.2906 104.858 47.4271C104.704 47.5636 104.509 47.6318 104.273 47.6318H97.6054C97.3699 47.6318 97.1748 47.5636 97.0203 47.4271C96.8657 47.2906 96.7885 47.1183 96.7885 46.9103V44.4189Z" fill="#D0D1D7"/><path d="M103.106 44.4189C103.106 44.2109 103.183 44.0386 103.338 43.9021C103.492 43.7656 103.687 43.6974 103.923 43.6974H110.591C110.826 43.6974 111.021 43.7656 111.176 43.9021C111.33 44.0386 111.408 44.2109 111.408 44.4189V46.9103C111.408 47.1183 111.33 47.2906 111.176 47.4271C111.021 47.5636 110.826 47.6318 110.591 47.6318H103.923C103.687 47.6318 103.492 47.5636 103.338 47.4271C103.183 47.2906 103.106 47.1183 103.106 46.9103V44.4189Z" fill="#D0D1D7"/><path d="M109.423 44.4286C109.423 44.2206 109.501 44.0484 109.655 43.9119C109.81 43.7754 110.005 43.7071 110.24 43.7071H113.751C113.986 43.7071 114.181 43.7754 114.336 43.9119C114.49 44.0484 114.568 44.2206 114.568 44.4286V46.9103C114.568 47.1183 114.49 47.2906 114.336 47.4271C114.181 47.5636 113.986 47.6318 113.751 47.6318H110.24C110.005 47.6318 109.81 47.5636 109.655 47.4271C109.501 47.2906 109.423 47.1183 109.423 46.9103V44.4286Z" fill="#D0D1D7"/><rect x="207.271" y="27" width="26.7277" height="10.4911" rx="0.915179" fill="#0096CC"/><g clip-path="url(#clip17_2180_28903)"><path d="M213.219 31.2177C212.647 31.2177 212.192 31.6825 212.192 32.2455C212.192 32.8175 212.647 33.2733 213.219 33.2733C213.782 33.2733 214.247 32.8175 214.247 32.2455C214.247 31.6825 213.782 31.2177 213.219 31.2177ZM213.219 32.9158C212.853 32.9158 212.549 32.6209 212.549 32.2455C212.549 31.8791 212.844 31.5842 213.219 31.5842C213.586 31.5842 213.881 31.8791 213.881 32.2455C213.881 32.6209 213.586 32.9158 213.219 32.9158ZM214.524 31.182C214.524 31.0479 214.417 30.9407 214.283 30.9407C214.149 30.9407 214.042 31.0479 214.042 31.182C214.042 31.3161 214.149 31.4233 214.283 31.4233C214.417 31.4233 214.524 31.3161 214.524 31.182ZM215.203 31.4233C215.186 31.1016 215.114 30.8156 214.882 30.5832C214.649 30.3508 214.363 30.2793 214.042 30.2615C213.711 30.2436 212.719 30.2436 212.388 30.2615C212.066 30.2793 211.789 30.3508 211.548 30.5832C211.316 30.8156 211.244 31.1016 211.226 31.4233C211.208 31.754 211.208 32.746 211.226 33.0767C211.244 33.3984 211.316 33.6755 211.548 33.9168C211.789 34.1492 212.066 34.2207 212.388 34.2385C212.719 34.2564 213.711 34.2564 214.042 34.2385C214.363 34.2207 214.649 34.1492 214.882 33.9168C215.114 33.6755 215.186 33.3984 215.203 33.0767C215.221 32.746 215.221 31.754 215.203 31.4233ZM214.774 33.4253C214.712 33.604 214.569 33.7381 214.399 33.8096C214.131 33.9168 213.505 33.89 213.219 33.89C212.924 33.89 212.299 33.9168 212.04 33.8096C211.861 33.7381 211.727 33.604 211.655 33.4253C211.548 33.1661 211.575 32.5405 211.575 32.2455C211.575 31.9595 211.548 31.3339 211.655 31.0658C211.727 30.896 211.861 30.7619 212.04 30.6904C212.299 30.5832 212.924 30.61 213.219 30.61C213.505 30.61 214.131 30.5832 214.399 30.6904C214.569 30.753 214.703 30.896 214.774 31.0658C214.882 31.3339 214.855 31.9595 214.855 32.2455C214.855 32.5405 214.882 33.1661 214.774 33.4253Z" fill="white"/></g><path d="M217.625 33.7451H218.152V32.5093H219.361V32.0676H218.152V31.2778H219.488V30.836H217.625V33.7451ZM220.777 33.7877C221.417 33.7877 221.823 33.3374 221.823 32.6627C221.823 31.9866 221.417 31.5349 220.777 31.5349C220.138 31.5349 219.732 31.9866 219.732 32.6627C219.732 33.3374 220.138 33.7877 220.777 33.7877ZM220.78 33.3758C220.427 33.3758 220.253 33.0605 220.253 32.6613C220.253 32.2622 220.427 31.9426 220.78 31.9426C221.128 31.9426 221.302 32.2622 221.302 32.6613C221.302 33.0605 221.128 33.3758 220.78 33.3758ZM222.813 30.836H222.299V33.7451H222.813V30.836ZM223.896 30.836H223.382V33.7451H223.896V30.836ZM225.417 33.7877C226.056 33.7877 226.462 33.3374 226.462 32.6627C226.462 31.9866 226.056 31.5349 225.417 31.5349C224.778 31.5349 224.371 31.9866 224.371 32.6627C224.371 33.3374 224.778 33.7877 225.417 33.7877ZM225.42 33.3758C225.066 33.3758 224.893 33.0605 224.893 32.6613C224.893 32.2622 225.066 31.9426 225.42 31.9426C225.768 31.9426 225.941 32.2622 225.941 32.6613C225.941 33.0605 225.768 33.3758 225.42 33.3758ZM227.331 33.7451H227.867L228.277 32.2707H228.306L228.715 33.7451H229.251L229.869 31.5633H229.343L228.965 33.0889H228.944L228.552 31.5633H228.034L227.642 33.0974H227.622L227.238 31.5633H226.714L227.331 33.7451Z" fill="white"/></g><defs><filter id="filter0_dd_2180_28903" x="33.3827" y="66.7765" width="65.9221" height="106.162" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.893854"/><feGaussianBlur stdDeviation="0.558659"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_2180_28903"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.223464"/><feGaussianBlur stdDeviation="0.223464"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="effect1_dropShadow_2180_28903" result="effect2_dropShadow_2180_28903"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_2180_28903" result="shape"/></filter><filter id="filter1_dd_2180_28903" x="33.3827" y="174.056" width="65.9221" height="107.709" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.893854"/><feGaussianBlur stdDeviation="0.558659"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_2180_28903"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.223464"/><feGaussianBlur stdDeviation="0.223464"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="effect1_dropShadow_2180_28903" result="effect2_dropShadow_2180_28903"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_2180_28903" result="shape"/></filter><filter id="filter2_dd_2180_28903" x="101.539" y="66.7775" width="65.9221" height="84.7092" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.893854"/><feGaussianBlur stdDeviation="0.558659"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_2180_28903"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.223464"/><feGaussianBlur stdDeviation="0.223464"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="effect1_dropShadow_2180_28903" result="effect2_dropShadow_2180_28903"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_2180_28903" result="shape"/></filter><filter id="filter3_dd_2180_28903" x="101.539" y="152.605" width="65.9221" height="107.709" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.893854"/><feGaussianBlur stdDeviation="0.558659"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_2180_28903"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.223464"/><feGaussianBlur stdDeviation="0.223464"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="effect1_dropShadow_2180_28903" result="effect2_dropShadow_2180_28903"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_2180_28903" result="shape"/></filter><filter id="filter4_dd_2180_28903" x="169.695" y="66.7775" width="65.9221" height="105.715" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.893854"/><feGaussianBlur stdDeviation="0.558659"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_2180_28903"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.223464"/><feGaussianBlur stdDeviation="0.223464"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="effect1_dropShadow_2180_28903" result="effect2_dropShadow_2180_28903"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_2180_28903" result="shape"/></filter><filter id="filter5_dd_2180_28903" x="169.695" y="173.61" width="65.9221" height="107.709" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.893854"/><feGaussianBlur stdDeviation="0.558659"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_2180_28903"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.223464"/><feGaussianBlur stdDeviation="0.223464"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="effect1_dropShadow_2180_28903" result="effect2_dropShadow_2180_28903"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_2180_28903" result="shape"/></filter><clipPath id="clip0_2180_28903"><rect width="262.5" height="200" fill="white" transform="translate(0.5)"/></clipPath><clipPath id="clip1_2180_28903"><rect x="34.5" y="67" width="63.6871" height="103.927" rx="0.893854" fill="white"/></clipPath><clipPath id="clip2_2180_28903"><rect width="63.6871" height="80.8938" fill="white" transform="translate(34.5 67)"/></clipPath><clipPath id="clip3_2180_28903"><rect width="3.57542" height="3.57542" fill="white" transform="translate(38.5 163.352)"/></clipPath><clipPath id="clip4_2180_28903"><rect width="3.57542" height="3.57542" fill="white" transform="translate(46.5449 163.352)"/></clipPath><clipPath id="clip5_2180_28903"><rect x="34.5" y="174.279" width="63.6871" height="105.475" rx="0.893854" fill="white"/></clipPath><clipPath id="clip6_2180_28903"><rect width="63.6871" height="80.8938" fill="white" transform="translate(34.5 174.279)"/></clipPath><clipPath id="clip7_2180_28903"><rect x="102.656" y="67.001" width="63.6871" height="82.4748" rx="0.893854" fill="white"/></clipPath><clipPath id="clip8_2180_28903"><rect width="3.57542" height="3.57542" fill="white" transform="translate(106.656 141.9)"/></clipPath><clipPath id="clip9_2180_28903"><rect width="3.57542" height="3.57542" fill="white" transform="translate(114.701 141.9)"/></clipPath><clipPath id="clip10_2180_28903"><rect x="102.656" y="152.828" width="63.6871" height="105.475" rx="0.893854" fill="white"/></clipPath><clipPath id="clip11_2180_28903"><rect x="170.812" y="67.001" width="63.6871" height="103.48" rx="0.893854" fill="white"/></clipPath><clipPath id="clip12_2180_28903"><rect width="63.6871" height="80.8938" fill="white" transform="translate(170.812 67.001)"/></clipPath><clipPath id="clip13_2180_28903"><rect width="3.57542" height="3.57542" fill="white" transform="translate(174.812 162.906)"/></clipPath><clipPath id="clip14_2180_28903"><rect width="3.57542" height="3.57542" fill="white" transform="translate(182.857 162.906)"/></clipPath><clipPath id="clip15_2180_28903"><rect x="170.812" y="173.833" width="63.6871" height="105.475" rx="0.893854" fill="white"/></clipPath><clipPath id="clip16_2180_28903"><rect width="63.6871" height="80.8938" fill="white" transform="translate(170.812 173.833)"/></clipPath><clipPath id="clip17_2180_28903"><rect width="4.57589" height="4.57589" fill="white" transform="translate(210.932 29.958)"/></clipPath></defs></svg>',
			'cardGridIcon'           => '<svg width="263" height="200" viewBox="0 0 263 200" fill="none" xmlns="http://www.w3.org/2000/svg"><g clip-path="url(#clip0_2180_29052)"><rect width="262.5" height="200" fill="#F3F4F5"/><g filter="url(#filter0_dd_2180_29052)"><g clip-path="url(#clip1_2180_29052)"><rect x="34.5" y="67" width="63.6871" height="86.0335" rx="0.893854" fill="white"/><g clip-path="url(#clip2_2180_29052)"><rect width="63.6871" height="63" transform="translate(34.5 67)" fill="#8C8F9A"/><circle opacity="0.5" cx="30.6875" cy="139" r="32" fill="#D0D1D7"/></g><rect x="38.5" y="134" width="55.6871" height="3" rx="0.5" fill="#D0D1D7"/><path d="M38.5 140.5C38.5 140.224 38.7239 140 39 140H71C71.2761 140 71.5 140.224 71.5 140.5V142.5C71.5 142.776 71.2761 143 71 143H39C38.7239 143 38.5 142.776 38.5 142.5V140.5Z" fill="#D0D1D7"/><g clip-path="url(#clip3_2180_29052)"><path d="M40.959 146.128V146.15C41.2753 146.15 41.5325 146.407 41.5325 146.724C41.5325 147.063 41.3308 147.391 41.0705 147.674C40.8157 147.952 40.5078 148.183 40.2886 148.338C40.0694 148.183 39.7615 147.952 39.5067 147.674C39.2463 147.391 39.0446 147.063 39.0446 146.724C39.0446 146.407 39.3019 146.15 39.6182 146.15V146.128L39.6181 146.15C39.7103 146.15 39.801 146.173 39.8826 146.215C39.9643 146.258 40.0346 146.319 40.0875 146.395L40.0875 146.395L40.2703 146.655L40.2886 146.681L40.3069 146.655L40.4897 146.395L40.4897 146.395C40.5425 146.319 40.6128 146.258 40.6945 146.215C40.7761 146.173 40.8669 146.15 40.959 146.15L40.959 146.128ZM40.959 146.128C41.2876 146.128 41.5549 146.395 41.5549 146.724C41.5549 147.419 40.7291 148.056 40.2886 148.366L40.4714 146.382C40.5263 146.304 40.5993 146.24 40.6841 146.195C40.769 146.151 40.8633 146.128 40.959 146.128ZM40.2886 148.665L40.2759 148.656C40.0151 148.477 39.6409 148.21 39.3319 147.881C39.0232 147.552 38.7765 147.158 38.7765 146.724C38.7765 146.5 38.8652 146.286 39.023 146.128C39.1809 145.971 39.395 145.882 39.6182 145.882L40.2886 148.665ZM40.2886 148.665L40.3013 148.656M40.2886 148.665L40.3013 148.656M40.3013 148.656C40.5621 148.477 40.9362 148.21 41.2453 147.881C41.554 147.552 41.8007 147.158 41.8007 146.724C41.8007 146.613 41.7789 146.504 41.7366 146.402C41.6943 146.299 41.6323 146.207 41.5542 146.128C41.476 146.05 41.3832 145.988 41.2811 145.946C41.179 145.904 41.0695 145.882 40.959 145.882C40.8238 145.882 40.6906 145.915 40.5707 145.977M40.3013 148.656L40.5707 145.977M40.5707 145.977C40.46 146.035 40.3637 146.116 40.2886 146.216M40.5707 145.977L40.2886 146.216M40.2886 146.216C40.2135 146.116 40.1172 146.035 40.0065 145.977C39.8866 145.915 39.7534 145.882 39.6182 145.882L40.2886 146.216Z" fill="#434960" stroke="#434960" stroke-width="0.0446927"/></g><g clip-path="url(#clip4_2180_29052)"><path d="M48.3318 145.755C48.1362 145.755 47.9425 145.793 47.7617 145.868C47.581 145.943 47.4168 146.053 47.2784 146.191C46.999 146.471 46.8421 146.85 46.8421 147.245C46.8408 147.589 46.9599 147.922 47.1788 148.188L46.8808 148.486C46.8602 148.507 46.8461 148.533 46.8406 148.562C46.835 148.591 46.8381 148.621 46.8495 148.648C46.8619 148.675 46.882 148.697 46.9072 148.713C46.9324 148.728 46.9616 148.736 46.9911 148.734H48.3318C48.727 148.734 49.1059 148.577 49.3853 148.298C49.6646 148.019 49.8216 147.64 49.8216 147.245C49.8216 146.85 49.6646 146.471 49.3853 146.191C49.1059 145.912 48.727 145.755 48.3318 145.755ZM48.3318 148.436H47.3501L47.4886 148.298C47.5164 148.27 47.532 148.232 47.532 148.193C47.532 148.154 47.5164 148.116 47.4886 148.088C47.2936 147.893 47.1721 147.637 47.1449 147.362C47.1177 147.088 47.1865 146.812 47.3396 146.583C47.4926 146.354 47.7204 146.185 47.9842 146.104C48.248 146.024 48.5315 146.038 48.7863 146.143C49.0411 146.249 49.2515 146.439 49.3816 146.682C49.5117 146.925 49.5536 147.206 49.5 147.476C49.4464 147.747 49.3007 147.99 49.0877 148.165C48.8747 148.34 48.6076 148.436 48.3318 148.436Z" fill="#434960"/></g></g></g><g filter="url(#filter1_dd_2180_29052)"><g clip-path="url(#clip5_2180_29052)"><rect x="34.5" y="156.386" width="63.6871" height="87.581" rx="0.893854" fill="white"/><rect width="63.6871" height="63" transform="translate(34.5 156.386)" fill="#8C8F9A"/></g><rect x="34.6117" y="156.497" width="63.4636" height="87.3575" rx="0.782122" stroke="#F3F4F5" stroke-width="0.223464"/></g><g filter="url(#filter2_dd_2180_29052)"><g clip-path="url(#clip6_2180_29052)"><rect x="102.656" y="67.001" width="63.6871" height="86.0335" rx="0.893854" fill="white"/><rect width="63.6871" height="63" transform="translate(102.656 67.001)" fill="#8C8F9A"/><rect x="126.119" y="88.2305" width="16.9832" height="16.9832" rx="8.49161" fill="white"/><path d="M132.6 94.6262C132.6 94.4601 132.774 94.3521 132.923 94.4263L137.786 96.8581C137.951 96.9404 137.951 97.1755 137.786 97.2578L132.923 99.6895C132.774 99.7638 132.6 99.6558 132.6 99.4897V94.6262Z" fill="black"/><rect x="106.656" y="134.001" width="55.6871" height="3" rx="0.5" fill="#D0D1D7"/><path d="M106.656 140.501C106.656 140.225 106.88 140.001 107.156 140.001H139.156C139.432 140.001 139.656 140.225 139.656 140.501V142.501C139.656 142.777 139.432 143.001 139.156 143.001H107.156C106.88 143.001 106.656 142.777 106.656 142.501V140.501Z" fill="#D0D1D7"/><g clip-path="url(#clip7_2180_29052)"><path d="M109.115 146.129V146.151C109.432 146.151 109.689 146.408 109.689 146.725C109.689 147.064 109.487 147.392 109.227 147.675C108.972 147.953 108.664 148.184 108.445 148.339C108.226 148.184 107.918 147.953 107.663 147.675C107.403 147.392 107.201 147.064 107.201 146.725C107.201 146.408 107.458 146.151 107.774 146.151V146.129L107.774 146.151C107.867 146.151 107.957 146.174 108.039 146.216C108.121 146.259 108.191 146.32 108.244 146.396L108.244 146.396L108.427 146.656L108.445 146.682L108.463 146.656L108.646 146.396L108.646 146.396C108.699 146.32 108.769 146.259 108.851 146.216C108.932 146.174 109.023 146.151 109.115 146.151L109.115 146.129ZM109.115 146.129C109.444 146.129 109.711 146.396 109.711 146.725C109.711 147.42 108.885 148.056 108.445 148.367L108.628 146.383C108.683 146.305 108.756 146.241 108.84 146.196C108.925 146.152 109.02 146.129 109.115 146.129ZM108.445 148.666L108.432 148.657C108.171 148.478 107.797 148.211 107.488 147.882C107.179 147.553 106.933 147.159 106.933 146.725C106.933 146.501 107.021 146.287 107.179 146.129C107.337 145.972 107.551 145.883 107.774 145.883L108.445 148.666ZM108.445 148.666L108.458 148.657M108.445 148.666L108.458 148.657M108.458 148.657C108.718 148.478 109.092 148.211 109.402 147.882C109.71 147.553 109.957 147.159 109.957 146.725C109.957 146.614 109.935 146.505 109.893 146.403C109.851 146.3 109.789 146.208 109.71 146.129C109.632 146.051 109.539 145.989 109.437 145.947C109.335 145.905 109.226 145.883 109.115 145.883C108.98 145.883 108.847 145.916 108.727 145.978M108.458 148.657L108.727 145.978M108.727 145.978C108.616 146.036 108.52 146.117 108.445 146.216M108.727 145.978L108.445 146.216M108.445 146.216C108.37 146.117 108.273 146.036 108.163 145.978C108.043 145.916 107.91 145.883 107.774 145.883L108.445 146.216Z" fill="#434960" stroke="#434960" stroke-width="0.0446927"/></g><g clip-path="url(#clip8_2180_29052)"><path d="M116.488 145.756C116.292 145.756 116.099 145.794 115.918 145.869C115.737 145.944 115.573 146.054 115.435 146.192C115.155 146.472 114.998 146.851 114.998 147.246C114.997 147.59 115.116 147.923 115.335 148.189L115.037 148.487C115.016 148.508 115.002 148.534 114.997 148.563C114.991 148.592 114.994 148.622 115.006 148.649C115.018 148.676 115.038 148.698 115.063 148.714C115.089 148.729 115.118 148.737 115.147 148.735H116.488C116.883 148.735 117.262 148.578 117.542 148.299C117.821 148.02 117.978 147.641 117.978 147.246C117.978 146.851 117.821 146.472 117.542 146.192C117.262 145.913 116.883 145.756 116.488 145.756ZM116.488 148.437H115.506L115.645 148.299C115.673 148.271 115.688 148.233 115.688 148.194C115.688 148.154 115.673 148.117 115.645 148.089C115.45 147.894 115.328 147.638 115.301 147.363C115.274 147.089 115.343 146.813 115.496 146.584C115.649 146.355 115.877 146.186 116.14 146.105C116.404 146.025 116.688 146.039 116.943 146.144C117.197 146.25 117.408 146.44 117.538 146.683C117.668 146.926 117.71 147.207 117.656 147.477C117.603 147.748 117.457 147.991 117.244 148.166C117.031 148.341 116.764 148.437 116.488 148.437Z" fill="#434960"/></g></g></g><g filter="url(#filter3_dd_2180_29052)"><g clip-path="url(#clip9_2180_29052)"><rect x="102.656" y="156.387" width="63.6871" height="87.581" rx="0.893854" fill="white"/><g clip-path="url(#clip10_2180_29052)"><rect width="63.6871" height="63" transform="translate(102.656 156.387)" fill="#8C8F9A"/><rect opacity="0.5" x="115" y="206.234" width="47" height="47" transform="rotate(-45 115 206.234)" fill="#D0D1D7"/></g></g><rect x="102.768" y="156.498" width="63.4636" height="87.3575" rx="0.782122" stroke="#F3F4F5" stroke-width="0.223464"/></g><g filter="url(#filter4_dd_2180_29052)"><g clip-path="url(#clip11_2180_29052)"><rect x="170.812" y="67.001" width="64" height="86.0335" rx="0.893854" fill="white"/><rect width="64" height="63" transform="translate(170.812 67.001)" fill="#8C8F9A"/><circle opacity="0.5" cx="231.5" cy="68.5" r="37.5" fill="#D0D1D7"/><rect x="174.812" y="134.001" width="55.6871" height="3" rx="0.5" fill="#D0D1D7"/><path d="M174.812 140.501C174.812 140.225 175.036 140.001 175.312 140.001H207.312C207.589 140.001 207.812 140.225 207.812 140.501V142.501C207.812 142.777 207.589 143.001 207.312 143.001H175.312C175.036 143.001 174.812 142.777 174.812 142.501V140.501Z" fill="#D0D1D7"/><g clip-path="url(#clip12_2180_29052)"><path d="M177.271 146.129V146.151C177.588 146.151 177.845 146.408 177.845 146.725C177.845 147.064 177.643 147.392 177.383 147.675C177.128 147.953 176.82 148.184 176.601 148.339C176.382 148.184 176.074 147.953 175.819 147.675C175.559 147.392 175.357 147.064 175.357 146.725C175.357 146.408 175.614 146.151 175.931 146.151V146.129L175.931 146.151C176.023 146.151 176.113 146.174 176.195 146.216C176.277 146.259 176.347 146.32 176.4 146.396L176.4 146.396L176.583 146.656L176.601 146.682L176.619 146.656L176.802 146.396L176.802 146.396C176.855 146.32 176.925 146.259 177.007 146.216C177.089 146.174 177.179 146.151 177.272 146.151L177.271 146.129ZM177.271 146.129C177.6 146.129 177.867 146.396 177.867 146.725C177.867 147.42 177.042 148.056 176.601 148.367L176.784 146.383C176.839 146.305 176.912 146.241 176.997 146.196C177.081 146.152 177.176 146.129 177.271 146.129ZM176.601 148.666L176.588 148.657C176.328 148.478 175.953 148.211 175.644 147.882C175.336 147.553 175.089 147.159 175.089 146.725C175.089 146.501 175.178 146.287 175.336 146.129C175.493 145.972 175.707 145.883 175.931 145.883L176.601 148.666ZM176.601 148.666L176.614 148.657M176.601 148.666L176.614 148.657M176.614 148.657C176.875 148.478 177.249 148.211 177.558 147.882C177.866 147.553 178.113 147.159 178.113 146.725C178.113 146.614 178.091 146.505 178.049 146.403C178.007 146.3 177.945 146.208 177.867 146.129C177.788 146.051 177.696 145.989 177.594 145.947C177.491 145.905 177.382 145.883 177.271 145.883C177.136 145.883 177.003 145.916 176.883 145.978M176.614 148.657L176.883 145.978M176.883 145.978C176.773 146.036 176.676 146.117 176.601 146.216M176.883 145.978L176.601 146.216M176.601 146.216C176.526 146.117 176.43 146.036 176.319 145.978C176.199 145.916 176.066 145.883 175.931 145.883L176.601 146.216Z" fill="#434960" stroke="#434960" stroke-width="0.0446927"/></g><g clip-path="url(#clip13_2180_29052)"><path d="M184.644 145.756C184.449 145.756 184.255 145.794 184.074 145.869C183.893 145.944 183.729 146.054 183.591 146.192C183.312 146.472 183.155 146.851 183.155 147.246C183.153 147.59 183.272 147.923 183.491 148.189L183.193 148.487C183.173 148.508 183.159 148.534 183.153 148.563C183.148 148.592 183.151 148.622 183.162 148.649C183.174 148.676 183.194 148.698 183.22 148.714C183.245 148.729 183.274 148.737 183.304 148.735H184.644C185.039 148.735 185.418 148.578 185.698 148.299C185.977 148.02 186.134 147.641 186.134 147.246C186.134 146.851 185.977 146.472 185.698 146.192C185.418 145.913 185.039 145.756 184.644 145.756ZM184.644 148.437H183.663L183.801 148.299C183.829 148.271 183.844 148.233 183.844 148.194C183.844 148.154 183.829 148.117 183.801 148.089C183.606 147.894 183.485 147.638 183.457 147.363C183.43 147.089 183.499 146.813 183.652 146.584C183.805 146.355 184.033 146.186 184.297 146.105C184.561 146.025 184.844 146.039 185.099 146.144C185.354 146.25 185.564 146.44 185.694 146.683C185.824 146.926 185.866 147.207 185.812 147.477C185.759 147.748 185.613 147.991 185.4 148.166C185.187 148.341 184.92 148.437 184.644 148.437Z" fill="#434960"/></g></g></g><g filter="url(#filter5_dd_2180_29052)"><g clip-path="url(#clip14_2180_29052)"><rect x="170.812" y="156.387" width="63.6871" height="87.581" rx="0.893854" fill="white"/><rect width="63.6871" height="63" transform="translate(170.812 156.387)" fill="#8C8F9A"/></g><rect x="170.924" y="156.498" width="63.4636" height="87.3575" rx="0.782122" stroke="#F3F4F5" stroke-width="0.223464"/></g><circle cx="45.9821" cy="37.9821" r="10.9821" fill="#8C8F9A"/><path d="M65.2012 29.146C65.2012 28.9114 65.2782 28.717 65.4322 28.563C65.5862 28.409 65.7805 28.332 66.0152 28.332H72.6592C72.8938 28.332 73.0882 28.409 73.2422 28.563C73.3962 28.717 73.4732 28.9114 73.4732 29.146V31.818C73.4732 32.0527 73.3962 32.247 73.2422 32.401C73.0882 32.555 72.8938 32.632 72.6592 32.632H66.0152C65.7805 32.632 65.5862 32.555 65.4322 32.401C65.2782 32.247 65.2012 32.0527 65.2012 31.818V29.146Z" fill="#8C8F9A"/><path d="M71.4961 29.146C71.4961 28.9114 71.5731 28.717 71.7271 28.563C71.8811 28.409 72.0754 28.332 72.3101 28.332H78.9541C79.1888 28.332 79.3831 28.409 79.5371 28.563C79.6911 28.717 79.7681 28.9114 79.7681 29.146V31.818C79.7681 32.0527 79.6911 32.247 79.5371 32.401C79.3831 32.555 79.1888 32.632 78.9541 32.632H72.3101C72.0754 32.632 71.8811 32.555 71.7271 32.401C71.5731 32.247 71.4961 32.0527 71.4961 31.818V29.146Z" fill="#8C8F9A"/><path d="M77.791 29.146C77.791 28.9114 77.868 28.717 78.022 28.563C78.176 28.409 78.3703 28.332 78.605 28.332H82.249C82.4837 28.332 82.678 28.409 82.832 28.563C82.986 28.717 83.063 28.9114 83.063 29.146V31.818C83.063 32.0527 82.986 32.247 82.832 32.401C82.678 32.555 82.4837 32.632 82.249 32.632H78.605C78.3703 32.632 78.176 32.555 78.022 32.401C77.868 32.247 77.791 32.0527 77.791 31.818V29.146Z" fill="#8C8F9A"/><path d="M81.0859 29.146C81.0859 28.9114 81.1629 28.717 81.3169 28.563C81.4709 28.409 81.6653 28.332 81.8999 28.332H88.5439C88.7786 28.332 88.9729 28.409 89.1269 28.563C89.2809 28.717 89.3579 28.9114 89.3579 29.146V31.818C89.3579 32.0527 89.2809 32.247 89.1269 32.401C88.9729 32.555 88.7786 32.632 88.5439 32.632H81.8999C81.6653 32.632 81.4709 32.555 81.3169 32.401C81.1629 32.247 81.0859 32.0527 81.0859 31.818V29.146Z" fill="#8C8F9A"/><path d="M87.3809 29.157C87.3809 28.9224 87.4579 28.728 87.6119 28.574C87.7659 28.42 87.9602 28.343 88.1949 28.343H91.6929C91.9275 28.343 92.1219 28.42 92.2759 28.574C92.4299 28.728 92.5069 28.9224 92.5069 29.157V31.818C92.5069 32.0527 92.4299 32.247 92.2759 32.401C92.1219 32.555 91.9275 32.632 91.6929 32.632H88.1949C87.9602 32.632 87.7659 32.555 87.6119 32.401C87.4579 32.247 87.3809 32.0527 87.3809 31.818V29.157Z" fill="#8C8F9A"/><path d="M90.5283 29.146C90.5283 28.9114 90.6053 28.717 90.7593 28.563C90.9133 28.409 91.1077 28.332 91.3423 28.332H97.9863C98.221 28.332 98.4153 28.409 98.5693 28.563C98.7233 28.717 98.8003 28.9114 98.8003 29.146V31.818C98.8003 32.0527 98.7233 32.247 98.5693 32.401C98.4153 32.555 98.221 32.632 97.9863 32.632H91.3423C91.1077 32.632 90.9133 32.555 90.7593 32.401C90.6053 32.247 90.5283 32.0527 90.5283 31.818V29.146Z" fill="#8C8F9A"/><path d="M65.2012 37.3534C65.2012 37.1454 65.2784 36.9731 65.433 36.8366C65.5876 36.7001 65.7826 36.6318 66.0181 36.6318H72.6859C72.9214 36.6318 73.1164 36.7001 73.271 36.8366C73.4255 36.9731 73.5028 37.1454 73.5028 37.3534V39.8448C73.5028 40.0528 73.4255 40.2251 73.271 40.3616C73.1164 40.4981 72.9214 40.5663 72.6859 40.5663H66.0181C65.7826 40.5663 65.5876 40.4981 65.433 40.3616C65.2784 40.2251 65.2012 40.0528 65.2012 39.8448V37.3534Z" fill="#D0D1D7"/><path d="M71.5186 37.3534C71.5186 37.1454 71.5959 36.9731 71.7505 36.8366C71.905 36.7001 72.1 36.6318 72.3355 36.6318H79.0033C79.2388 36.6318 79.4339 36.7001 79.5884 36.8366C79.743 36.9731 79.8202 37.1454 79.8202 37.3534V39.8448C79.8202 40.0528 79.743 40.2251 79.5884 40.3616C79.4339 40.4981 79.2388 40.5663 79.0033 40.5663H72.3355C72.1 40.5663 71.905 40.4981 71.7505 40.3616C71.5959 40.2251 71.5186 40.0528 71.5186 39.8448V37.3534Z" fill="#D0D1D7"/><path d="M77.8361 37.3534C77.8361 37.1454 77.9134 36.9731 78.0679 36.8366C78.2225 36.7001 78.4175 36.6318 78.653 36.6318H85.3208C85.5563 36.6318 85.7513 36.7001 85.9059 36.8366C86.0604 36.9731 86.1377 37.1454 86.1377 37.3534V39.8448C86.1377 40.0528 86.0604 40.2251 85.9059 40.3616C85.7513 40.4981 85.5563 40.5663 85.3208 40.5663H78.653C78.4175 40.5663 78.2225 40.4981 78.0679 40.3616C77.9134 40.2251 77.8361 40.0528 77.8361 39.8448V37.3534Z" fill="#D0D1D7"/><path d="M84.1535 37.3534C84.1535 37.1454 84.2308 36.9731 84.3854 36.8366C84.5399 36.7001 84.735 36.6318 84.9705 36.6318H91.6382C91.8738 36.6318 92.0688 36.7001 92.2233 36.8366C92.3779 36.9731 92.4552 37.1454 92.4552 37.3534V39.8448C92.4552 40.0528 92.3779 40.2251 92.2233 40.3616C92.0688 40.4981 91.8738 40.5663 91.6382 40.5663H84.9705C84.735 40.5663 84.5399 40.4981 84.3854 40.3616C84.2308 40.2251 84.1535 40.0528 84.1535 39.8448V37.3534Z" fill="#D0D1D7"/><path d="M90.471 37.3631C90.471 37.1551 90.5483 36.9829 90.7028 36.8463C90.8574 36.7098 91.0524 36.6416 91.2879 36.6416H94.7984C95.0339 36.6416 95.229 36.7098 95.3835 36.8463C95.5381 36.9829 95.6154 37.1551 95.6154 37.3631V39.8448C95.6154 40.0528 95.5381 40.2251 95.3835 40.3616C95.229 40.4981 95.0339 40.5663 94.7984 40.5663H91.2879C91.0524 40.5663 90.8574 40.4981 90.7028 40.3616C90.5483 40.2251 90.471 40.0528 90.471 39.8448V37.3631Z" fill="#D0D1D7"/><path d="M93.6297 37.3631C93.6297 37.1551 93.707 36.9829 93.8616 36.8463C94.0161 36.7098 94.2111 36.6416 94.4466 36.6416H97.9572C98.1927 36.6416 98.3877 36.7098 98.5423 36.8463C98.6968 36.9829 98.7741 37.1551 98.7741 37.3631V39.8448C98.7741 40.0528 98.6968 40.2251 98.5423 40.3616C98.3877 40.4981 98.1927 40.5663 97.9572 40.5663H94.4466C94.2111 40.5663 94.0161 40.4981 93.8616 40.3616C93.707 40.2251 93.6297 40.0528 93.6297 39.8448V37.3631Z" fill="#D0D1D7"/><path d="M96.7885 37.3534C96.7885 37.1454 96.8657 36.9731 97.0203 36.8366C97.1748 36.7001 97.3699 36.6318 97.6054 36.6318H104.273C104.509 36.6318 104.704 36.7001 104.858 36.8366C105.013 36.9731 105.09 37.1454 105.09 37.3534V39.8448C105.09 40.0528 105.013 40.2251 104.858 40.3616C104.704 40.4981 104.509 40.5663 104.273 40.5663H97.6054C97.3699 40.5663 97.1748 40.4981 97.0203 40.3616C96.8657 40.2251 96.7885 40.0528 96.7885 39.8448V37.3534Z" fill="#D0D1D7"/><path d="M103.106 37.3631C103.106 37.1551 103.183 36.9829 103.338 36.8463C103.492 36.7098 103.687 36.6416 103.923 36.6416H107.433C107.669 36.6416 107.864 36.7098 108.018 36.8463C108.173 36.9829 108.25 37.1551 108.25 37.3631V39.8448C108.25 40.0528 108.173 40.2251 108.018 40.3616C107.864 40.4981 107.669 40.5663 107.433 40.5663H103.923C103.687 40.5663 103.492 40.4981 103.338 40.3616C103.183 40.2251 103.106 40.0528 103.106 39.8448V37.3631Z" fill="#D0D1D7"/><path d="M106.265 37.3534C106.265 37.1454 106.342 36.9731 106.496 36.8366C106.651 36.7001 106.846 36.6318 107.082 36.6318H113.749C113.985 36.6318 114.18 36.7001 114.334 36.8366C114.489 36.9731 114.566 37.1454 114.566 37.3534V39.8448C114.566 40.0528 114.489 40.2251 114.334 40.3616C114.18 40.4981 113.985 40.5663 113.749 40.5663H107.082C106.846 40.5663 106.651 40.4981 106.496 40.3616C106.342 40.2251 106.265 40.0528 106.265 39.8448V37.3534Z" fill="#D0D1D7"/><path d="M112.582 37.3534C112.582 37.1454 112.659 36.9731 112.814 36.8366C112.968 36.7001 113.164 36.6318 113.399 36.6318H120.067C120.302 36.6318 120.497 36.7001 120.652 36.8366C120.806 36.9731 120.884 37.1454 120.884 37.3534V39.8448C120.884 40.0528 120.806 40.2251 120.652 40.3616C120.497 40.4981 120.302 40.5663 120.067 40.5663H113.399C113.164 40.5663 112.968 40.4981 112.814 40.3616C112.659 40.2251 112.582 40.0528 112.582 39.8448V37.3534Z" fill="#D0D1D7"/><path d="M118.9 37.3534C118.9 37.1454 118.977 36.9731 119.131 36.8366C119.286 36.7001 119.481 36.6318 119.716 36.6318H126.384C126.62 36.6318 126.815 36.7001 126.969 36.8366C127.124 36.9731 127.201 37.1454 127.201 37.3534V39.8448C127.201 40.0528 127.124 40.2251 126.969 40.3616C126.815 40.4981 126.62 40.5663 126.384 40.5663H119.716C119.481 40.5663 119.286 40.4981 119.131 40.3616C118.977 40.2251 118.9 40.0528 118.9 39.8448V37.3534Z" fill="#D0D1D7"/><path d="M65.2012 44.4189C65.2012 44.2109 65.2784 44.0386 65.433 43.9021C65.5876 43.7656 65.7826 43.6974 66.0181 43.6974H72.6859C72.9214 43.6974 73.1164 43.7656 73.271 43.9021C73.4255 44.0386 73.5028 44.2109 73.5028 44.4189V46.9103C73.5028 47.1183 73.4255 47.2906 73.271 47.4271C73.1164 47.5636 72.9214 47.6318 72.6859 47.6318H66.0181C65.7826 47.6318 65.5876 47.5636 65.433 47.4271C65.2784 47.2906 65.2012 47.1183 65.2012 46.9103V44.4189Z" fill="#D0D1D7"/><path d="M71.5186 44.4189C71.5186 44.2109 71.5959 44.0386 71.7505 43.9021C71.905 43.7656 72.1 43.6974 72.3355 43.6974H79.0033C79.2388 43.6974 79.4339 43.7656 79.5884 43.9021C79.743 44.0386 79.8202 44.2109 79.8202 44.4189V46.9103C79.8202 47.1183 79.743 47.2906 79.5884 47.4271C79.4339 47.5636 79.2388 47.6318 79.0033 47.6318H72.3355C72.1 47.6318 71.905 47.5636 71.7505 47.4271C71.5959 47.2906 71.5186 47.1183 71.5186 46.9103V44.4189Z" fill="#D0D1D7"/><path d="M77.8361 44.4189C77.8361 44.2109 77.9134 44.0386 78.0679 43.9021C78.2225 43.7656 78.4175 43.6974 78.653 43.6974H85.3208C85.5563 43.6974 85.7513 43.7656 85.9059 43.9021C86.0604 44.0386 86.1377 44.2109 86.1377 44.4189V46.9103C86.1377 47.1183 86.0604 47.2906 85.9059 47.4271C85.7513 47.5636 85.5563 47.6318 85.3208 47.6318H78.653C78.4175 47.6318 78.2225 47.5636 78.0679 47.4271C77.9134 47.2906 77.8361 47.1183 77.8361 46.9103V44.4189Z" fill="#D0D1D7"/><path d="M84.1535 44.4189C84.1535 44.2109 84.2308 44.0386 84.3854 43.9021C84.5399 43.7656 84.735 43.6974 84.9705 43.6974H91.6382C91.8738 43.6974 92.0688 43.7656 92.2233 43.9021C92.3779 44.0386 92.4552 44.2109 92.4552 44.4189V46.9103C92.4552 47.1183 92.3779 47.2906 92.2233 47.4271C92.0688 47.5636 91.8738 47.6318 91.6382 47.6318H84.9705C84.735 47.6318 84.5399 47.5636 84.3854 47.4271C84.2308 47.2906 84.1535 47.1183 84.1535 46.9103V44.4189Z" fill="#D0D1D7"/><path d="M90.471 44.4189C90.471 44.2109 90.5483 44.0386 90.7028 43.9021C90.8574 43.7656 91.0524 43.6974 91.2879 43.6974H97.9557C98.1912 43.6974 98.3862 43.7656 98.5408 43.9021C98.6953 44.0386 98.7726 44.2109 98.7726 44.4189V46.9103C98.7726 47.1183 98.6953 47.2906 98.5408 47.4271C98.3862 47.5636 98.1912 47.6318 97.9557 47.6318H91.2879C91.0524 47.6318 90.8574 47.5636 90.7028 47.4271C90.5483 47.2906 90.471 47.1183 90.471 46.9103V44.4189Z" fill="#D0D1D7"/><path d="M96.7885 44.4189C96.7885 44.2109 96.8657 44.0386 97.0203 43.9021C97.1748 43.7656 97.3699 43.6974 97.6054 43.6974H104.273C104.509 43.6974 104.704 43.7656 104.858 43.9021C105.013 44.0386 105.09 44.2109 105.09 44.4189V46.9103C105.09 47.1183 105.013 47.2906 104.858 47.4271C104.704 47.5636 104.509 47.6318 104.273 47.6318H97.6054C97.3699 47.6318 97.1748 47.5636 97.0203 47.4271C96.8657 47.2906 96.7885 47.1183 96.7885 46.9103V44.4189Z" fill="#D0D1D7"/><path d="M103.106 44.4189C103.106 44.2109 103.183 44.0386 103.338 43.9021C103.492 43.7656 103.687 43.6974 103.923 43.6974H110.591C110.826 43.6974 111.021 43.7656 111.176 43.9021C111.33 44.0386 111.408 44.2109 111.408 44.4189V46.9103C111.408 47.1183 111.33 47.2906 111.176 47.4271C111.021 47.5636 110.826 47.6318 110.591 47.6318H103.923C103.687 47.6318 103.492 47.5636 103.338 47.4271C103.183 47.2906 103.106 47.1183 103.106 46.9103V44.4189Z" fill="#D0D1D7"/><path d="M109.423 44.4286C109.423 44.2206 109.501 44.0484 109.655 43.9119C109.81 43.7754 110.005 43.7071 110.24 43.7071H113.751C113.986 43.7071 114.181 43.7754 114.336 43.9119C114.49 44.0484 114.568 44.2206 114.568 44.4286V46.9103C114.568 47.1183 114.49 47.2906 114.336 47.4271C114.181 47.5636 113.986 47.6318 113.751 47.6318H110.24C110.005 47.6318 109.81 47.5636 109.655 47.4271C109.501 47.2906 109.423 47.1183 109.423 46.9103V44.4286Z" fill="#D0D1D7"/><rect x="207.271" y="27" width="26.7277" height="10.4911" rx="0.915179" fill="#0096CC"/><g clip-path="url(#clip15_2180_29052)"><path d="M213.219 31.2177C212.647 31.2177 212.192 31.6825 212.192 32.2455C212.192 32.8175 212.647 33.2733 213.219 33.2733C213.782 33.2733 214.247 32.8175 214.247 32.2455C214.247 31.6825 213.782 31.2177 213.219 31.2177ZM213.219 32.9158C212.853 32.9158 212.549 32.6209 212.549 32.2455C212.549 31.8791 212.844 31.5842 213.219 31.5842C213.586 31.5842 213.881 31.8791 213.881 32.2455C213.881 32.6209 213.586 32.9158 213.219 32.9158ZM214.524 31.182C214.524 31.0479 214.417 30.9407 214.283 30.9407C214.149 30.9407 214.042 31.0479 214.042 31.182C214.042 31.3161 214.149 31.4233 214.283 31.4233C214.417 31.4233 214.524 31.3161 214.524 31.182ZM215.203 31.4233C215.186 31.1016 215.114 30.8156 214.882 30.5832C214.649 30.3508 214.363 30.2793 214.042 30.2615C213.711 30.2436 212.719 30.2436 212.388 30.2615C212.066 30.2793 211.789 30.3508 211.548 30.5832C211.316 30.8156 211.244 31.1016 211.226 31.4233C211.208 31.754 211.208 32.746 211.226 33.0767C211.244 33.3984 211.316 33.6755 211.548 33.9168C211.789 34.1492 212.066 34.2207 212.388 34.2385C212.719 34.2564 213.711 34.2564 214.042 34.2385C214.363 34.2207 214.649 34.1492 214.882 33.9168C215.114 33.6755 215.186 33.3984 215.203 33.0767C215.221 32.746 215.221 31.754 215.203 31.4233ZM214.774 33.4253C214.712 33.604 214.569 33.7381 214.399 33.8096C214.131 33.9168 213.505 33.89 213.219 33.89C212.924 33.89 212.299 33.9168 212.04 33.8096C211.861 33.7381 211.727 33.604 211.655 33.4253C211.548 33.1661 211.575 32.5405 211.575 32.2455C211.575 31.9595 211.548 31.3339 211.655 31.0658C211.727 30.896 211.861 30.7619 212.04 30.6904C212.299 30.5832 212.924 30.61 213.219 30.61C213.505 30.61 214.131 30.5832 214.399 30.6904C214.569 30.753 214.703 30.896 214.774 31.0658C214.882 31.3339 214.855 31.9595 214.855 32.2455C214.855 32.5405 214.882 33.1661 214.774 33.4253Z" fill="white"/></g><path d="M217.625 33.7451H218.152V32.5093H219.361V32.0676H218.152V31.2778H219.488V30.836H217.625V33.7451ZM220.777 33.7877C221.417 33.7877 221.823 33.3374 221.823 32.6627C221.823 31.9866 221.417 31.5349 220.777 31.5349C220.138 31.5349 219.732 31.9866 219.732 32.6627C219.732 33.3374 220.138 33.7877 220.777 33.7877ZM220.78 33.3758C220.427 33.3758 220.253 33.0605 220.253 32.6613C220.253 32.2622 220.427 31.9426 220.78 31.9426C221.128 31.9426 221.302 32.2622 221.302 32.6613C221.302 33.0605 221.128 33.3758 220.78 33.3758ZM222.813 30.836H222.299V33.7451H222.813V30.836ZM223.896 30.836H223.382V33.7451H223.896V30.836ZM225.417 33.7877C226.056 33.7877 226.462 33.3374 226.462 32.6627C226.462 31.9866 226.056 31.5349 225.417 31.5349C224.778 31.5349 224.371 31.9866 224.371 32.6627C224.371 33.3374 224.778 33.7877 225.417 33.7877ZM225.42 33.3758C225.066 33.3758 224.893 33.0605 224.893 32.6613C224.893 32.2622 225.066 31.9426 225.42 31.9426C225.768 31.9426 225.941 32.2622 225.941 32.6613C225.941 33.0605 225.768 33.3758 225.42 33.3758ZM227.331 33.7451H227.867L228.277 32.2707H228.306L228.715 33.7451H229.251L229.869 31.5633H229.343L228.965 33.0889H228.944L228.552 31.5633H228.034L227.642 33.0974H227.622L227.238 31.5633H226.714L227.331 33.7451Z" fill="white"/></g><defs><filter id="filter0_dd_2180_29052" x="33.3827" y="66.7765" width="65.9221" height="88.2678" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.893854"/><feGaussianBlur stdDeviation="0.558659"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_2180_29052"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.223464"/><feGaussianBlur stdDeviation="0.223464"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="effect1_dropShadow_2180_29052" result="effect2_dropShadow_2180_29052"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_2180_29052" result="shape"/></filter><filter id="filter1_dd_2180_29052" x="33.3827" y="156.162" width="65.9221" height="89.8157" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.893854"/><feGaussianBlur stdDeviation="0.558659"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_2180_29052"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.223464"/><feGaussianBlur stdDeviation="0.223464"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="effect1_dropShadow_2180_29052" result="effect2_dropShadow_2180_29052"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_2180_29052" result="shape"/></filter><filter id="filter2_dd_2180_29052" x="101.539" y="66.7775" width="65.9221" height="88.2678" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.893854"/><feGaussianBlur stdDeviation="0.558659"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_2180_29052"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.223464"/><feGaussianBlur stdDeviation="0.223464"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="effect1_dropShadow_2180_29052" result="effect2_dropShadow_2180_29052"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_2180_29052" result="shape"/></filter><filter id="filter3_dd_2180_29052" x="101.539" y="156.163" width="65.9221" height="89.8157" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.893854"/><feGaussianBlur stdDeviation="0.558659"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_2180_29052"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.223464"/><feGaussianBlur stdDeviation="0.223464"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="effect1_dropShadow_2180_29052" result="effect2_dropShadow_2180_29052"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_2180_29052" result="shape"/></filter><filter id="filter4_dd_2180_29052" x="169.695" y="66.7775" width="66.2346" height="88.2678" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.893854"/><feGaussianBlur stdDeviation="0.558659"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_2180_29052"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.223464"/><feGaussianBlur stdDeviation="0.223464"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="effect1_dropShadow_2180_29052" result="effect2_dropShadow_2180_29052"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_2180_29052" result="shape"/></filter><filter id="filter5_dd_2180_29052" x="169.695" y="156.163" width="65.9221" height="89.8157" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.893854"/><feGaussianBlur stdDeviation="0.558659"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_2180_29052"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.223464"/><feGaussianBlur stdDeviation="0.223464"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="effect1_dropShadow_2180_29052" result="effect2_dropShadow_2180_29052"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_2180_29052" result="shape"/></filter><clipPath id="clip0_2180_29052"><rect width="262.5" height="200" fill="white"/></clipPath><clipPath id="clip1_2180_29052"><rect x="34.5" y="67" width="63.6871" height="86.0335" rx="0.893854" fill="white"/></clipPath><clipPath id="clip2_2180_29052"><rect width="63.6871" height="63" fill="white" transform="translate(34.5 67)"/></clipPath><clipPath id="clip3_2180_29052"><rect width="3.57542" height="3.57542" fill="white" transform="translate(38.5 145.458)"/></clipPath><clipPath id="clip4_2180_29052"><rect width="3.57542" height="3.57542" fill="white" transform="translate(46.5449 145.458)"/></clipPath><clipPath id="clip5_2180_29052"><rect x="34.5" y="156.386" width="63.6871" height="87.581" rx="0.893854" fill="white"/></clipPath><clipPath id="clip6_2180_29052"><rect x="102.656" y="67.001" width="63.6871" height="86.0335" rx="0.893854" fill="white"/></clipPath><clipPath id="clip7_2180_29052"><rect width="3.57542" height="3.57542" fill="white" transform="translate(106.656 145.459)"/></clipPath><clipPath id="clip8_2180_29052"><rect width="3.57542" height="3.57542" fill="white" transform="translate(114.701 145.459)"/></clipPath><clipPath id="clip9_2180_29052"><rect x="102.656" y="156.387" width="63.6871" height="87.581" rx="0.893854" fill="white"/></clipPath><clipPath id="clip10_2180_29052"><rect width="63.6871" height="63" fill="white" transform="translate(102.656 156.387)"/></clipPath><clipPath id="clip11_2180_29052"><rect x="170.812" y="67.001" width="64" height="86.0335" rx="0.893854" fill="white"/></clipPath><clipPath id="clip12_2180_29052"><rect width="3.57542" height="3.57542" fill="white" transform="translate(174.812 145.459)"/></clipPath><clipPath id="clip13_2180_29052"><rect width="3.57542" height="3.57542" fill="white" transform="translate(182.857 145.459)"/></clipPath><clipPath id="clip14_2180_29052"><rect x="170.812" y="156.387" width="63.6871" height="87.581" rx="0.893854" fill="white"/></clipPath><clipPath id="clip15_2180_29052"><rect width="4.57589" height="4.57589" fill="white" transform="translate(210.932 29.958)"/></clipPath></defs></svg>',
			'highlightIcon'          => '<svg width="263" height="200" viewBox="0 0 263 200" fill="none" xmlns="http://www.w3.org/2000/svg"><g clip-path="url(#clip0_2180_29197)"><rect width="262.5" height="200" transform="translate(0.5)" fill="#F3F4F5"/><g filter="url(#filter0_ddd_2180_29197)"><g clip-path="url(#clip1_2180_29197)"><rect x="29" y="34.1426" width="205" height="175.714" rx="2" fill="white"/><g clip-path="url(#clip2_2180_29197)"><rect width="83.6639" height="76.6214" transform="translate(47 84)" fill="#96CE89"/><circle cx="29" cy="161" r="59" fill="#B6DDAD"/></g><g clip-path="url(#clip3_2180_29197)"><rect width="40.9868" height="37.4656" transform="translate(132.354 84)" fill="#96CE89"/><g clip-path="url(#clip4_2180_29197)"><rect x="167.566" y="85.127" width="3.94375" height="3.94375" rx="1.97188" fill="white"/><path d="M170.215 86.4229V87.437H169.2V86.4229H170.215ZM170.215 86.2539H169.2C169.107 86.2539 169.031 86.33 169.031 86.4229V87.437C169.031 87.53 169.107 87.606 169.2 87.606H170.215C170.307 87.606 170.384 87.53 170.384 87.437V86.4229C170.384 86.33 170.307 86.2539 170.215 86.2539ZM168.693 86.5919V87.7751C168.693 87.868 168.769 87.9441 168.862 87.9441H170.046V87.7751H168.862V86.5919H168.693Z" fill="#141B38"/></g></g><rect width="40.9868" height="37.4656" transform="translate(132.354 123.156)" fill="#96CE89"/><g clip-path="url(#clip5_2180_29197)"><rect width="40.9868" height="37.4656" transform="translate(175.031 84)" fill="#96CE89"/><circle cx="208.5" cy="124.5" r="21.5" fill="#B6DDAD"/></g><rect width="40.9868" height="37.4656" transform="translate(175.031 123.156)" fill="#96CE89"/><g clip-path="url(#clip6_2180_29197)"><rect width="40.9868" height="37.4656" transform="translate(47 162.312)" fill="#96CE89"/><g clip-path="url(#clip7_2180_29197)"><rect x="82.2129" y="163.438" width="3.94375" height="3.94375" rx="1.97188" fill="white"/><path d="M84.861 164.734V165.749H83.8469V164.734H84.861ZM84.861 164.565H83.8469C83.7539 164.565 83.6779 164.641 83.6779 164.734V165.749C83.6779 165.842 83.7539 165.918 83.8469 165.918H84.861C84.954 165.918 85.03 165.842 85.03 165.749V164.734C85.03 164.641 84.954 164.565 84.861 164.565ZM83.3398 164.903V166.087C83.3398 166.18 83.4159 166.256 83.5089 166.256H84.692V166.087H83.5089V164.903H83.3398Z" fill="#141B38"/></g></g><rect width="40.9868" height="37.4656" transform="translate(89.6777 162.312)" fill="#96CE89"/><g clip-path="url(#clip8_2180_29197)"><rect width="83.6639" height="76.6214" transform="translate(132.354 162.312)" fill="#96CE89"/><circle cx="210.354" cy="235.312" r="59" fill="#96CE89"/></g><circle cx="56.9821" cy="59.9821" r="10.9821" fill="#B6DDAD"/><path d="M76.2012 51.146C76.2012 50.9114 76.2782 50.717 76.4322 50.563C76.5862 50.409 76.7805 50.332 77.0152 50.332H83.6592C83.8938 50.332 84.0882 50.409 84.2422 50.563C84.3962 50.717 84.4732 50.9114 84.4732 51.146V53.818C84.4732 54.0527 84.3962 54.247 84.2422 54.401C84.0882 54.555 83.8938 54.632 83.6592 54.632H77.0152C76.7805 54.632 76.5862 54.555 76.4322 54.401C76.2782 54.247 76.2012 54.0527 76.2012 53.818V51.146Z" fill="#8C8F9A"/><path d="M82.4961 51.146C82.4961 50.9114 82.5731 50.717 82.7271 50.563C82.8811 50.409 83.0754 50.332 83.3101 50.332H89.9541C90.1888 50.332 90.3831 50.409 90.5371 50.563C90.6911 50.717 90.7681 50.9114 90.7681 51.146V53.818C90.7681 54.0527 90.6911 54.247 90.5371 54.401C90.3831 54.555 90.1888 54.632 89.9541 54.632H83.3101C83.0754 54.632 82.8811 54.555 82.7271 54.401C82.5731 54.247 82.4961 54.0527 82.4961 53.818V51.146Z" fill="#8C8F9A"/><path d="M88.791 51.146C88.791 50.9114 88.868 50.717 89.022 50.563C89.176 50.409 89.3703 50.332 89.605 50.332H93.249C93.4837 50.332 93.678 50.409 93.832 50.563C93.986 50.717 94.063 50.9114 94.063 51.146V53.818C94.063 54.0527 93.986 54.247 93.832 54.401C93.678 54.555 93.4837 54.632 93.249 54.632H89.605C89.3703 54.632 89.176 54.555 89.022 54.401C88.868 54.247 88.791 54.0527 88.791 53.818V51.146Z" fill="#8C8F9A"/><path d="M92.0859 51.146C92.0859 50.9114 92.1629 50.717 92.3169 50.563C92.4709 50.409 92.6653 50.332 92.8999 50.332H99.5439C99.7786 50.332 99.9729 50.409 100.127 50.563C100.281 50.717 100.358 50.9114 100.358 51.146V53.818C100.358 54.0527 100.281 54.247 100.127 54.401C99.9729 54.555 99.7786 54.632 99.5439 54.632H92.8999C92.6653 54.632 92.4709 54.555 92.3169 54.401C92.1629 54.247 92.0859 54.0527 92.0859 53.818V51.146Z" fill="#8C8F9A"/><path d="M98.3809 51.157C98.3809 50.9224 98.4579 50.728 98.6119 50.574C98.7659 50.42 98.9602 50.343 99.1949 50.343H102.693C102.928 50.343 103.122 50.42 103.276 50.574C103.43 50.728 103.507 50.9224 103.507 51.157V53.818C103.507 54.0527 103.43 54.247 103.276 54.401C103.122 54.555 102.928 54.632 102.693 54.632H99.1949C98.9602 54.632 98.7659 54.555 98.6119 54.401C98.4579 54.247 98.3809 54.0527 98.3809 53.818V51.157Z" fill="#8C8F9A"/><path d="M101.528 51.146C101.528 50.9114 101.605 50.717 101.759 50.563C101.913 50.409 102.108 50.332 102.342 50.332H108.986C109.221 50.332 109.415 50.409 109.569 50.563C109.723 50.717 109.8 50.9114 109.8 51.146V53.818C109.8 54.0527 109.723 54.247 109.569 54.401C109.415 54.555 109.221 54.632 108.986 54.632H102.342C102.108 54.632 101.913 54.555 101.759 54.401C101.605 54.247 101.528 54.0527 101.528 53.818V51.146Z" fill="#8C8F9A"/><path d="M76.2012 59.3534C76.2012 59.1454 76.2784 58.9731 76.433 58.8366C76.5876 58.7001 76.7826 58.6318 77.0181 58.6318H83.6859C83.9214 58.6318 84.1164 58.7001 84.271 58.8366C84.4255 58.9731 84.5028 59.1454 84.5028 59.3534V61.8448C84.5028 62.0528 84.4255 62.2251 84.271 62.3616C84.1164 62.4981 83.9214 62.5663 83.6859 62.5663H77.0181C76.7826 62.5663 76.5876 62.4981 76.433 62.3616C76.2784 62.2251 76.2012 62.0528 76.2012 61.8448V59.3534Z" fill="#D0D1D7"/><path d="M82.5186 59.3534C82.5186 59.1454 82.5959 58.9731 82.7505 58.8366C82.905 58.7001 83.1 58.6318 83.3355 58.6318H90.0033C90.2388 58.6318 90.4339 58.7001 90.5884 58.8366C90.743 58.9731 90.8202 59.1454 90.8202 59.3534V61.8448C90.8202 62.0528 90.743 62.2251 90.5884 62.3616C90.4339 62.4981 90.2388 62.5663 90.0033 62.5663H83.3355C83.1 62.5663 82.905 62.4981 82.7505 62.3616C82.5959 62.2251 82.5186 62.0528 82.5186 61.8448V59.3534Z" fill="#D0D1D7"/><path d="M88.8361 59.3534C88.8361 59.1454 88.9134 58.9731 89.0679 58.8366C89.2225 58.7001 89.4175 58.6318 89.653 58.6318H96.3208C96.5563 58.6318 96.7513 58.7001 96.9059 58.8366C97.0604 58.9731 97.1377 59.1454 97.1377 59.3534V61.8448C97.1377 62.0528 97.0604 62.2251 96.9059 62.3616C96.7513 62.4981 96.5563 62.5663 96.3208 62.5663H89.653C89.4175 62.5663 89.2225 62.4981 89.0679 62.3616C88.9134 62.2251 88.8361 62.0528 88.8361 61.8448V59.3534Z" fill="#D0D1D7"/><path d="M95.1535 59.3534C95.1535 59.1454 95.2308 58.9731 95.3854 58.8366C95.5399 58.7001 95.735 58.6318 95.9705 58.6318H102.638C102.874 58.6318 103.069 58.7001 103.223 58.8366C103.378 58.9731 103.455 59.1454 103.455 59.3534V61.8448C103.455 62.0528 103.378 62.2251 103.223 62.3616C103.069 62.4981 102.874 62.5663 102.638 62.5663H95.9705C95.735 62.5663 95.5399 62.4981 95.3854 62.3616C95.2308 62.2251 95.1535 62.0528 95.1535 61.8448V59.3534Z" fill="#D0D1D7"/><path d="M101.471 59.3631C101.471 59.1551 101.548 58.9829 101.703 58.8463C101.857 58.7098 102.052 58.6416 102.288 58.6416H105.798C106.034 58.6416 106.229 58.7098 106.384 58.8463C106.538 58.9829 106.615 59.1551 106.615 59.3631V61.8448C106.615 62.0528 106.538 62.2251 106.384 62.3616C106.229 62.4981 106.034 62.5663 105.798 62.5663H102.288C102.052 62.5663 101.857 62.4981 101.703 62.3616C101.548 62.2251 101.471 62.0528 101.471 61.8448V59.3631Z" fill="#D0D1D7"/><path d="M104.63 59.3631C104.63 59.1551 104.707 58.9829 104.862 58.8463C105.016 58.7098 105.211 58.6416 105.447 58.6416H108.957C109.193 58.6416 109.388 58.7098 109.542 58.8463C109.697 58.9829 109.774 59.1551 109.774 59.3631V61.8448C109.774 62.0528 109.697 62.2251 109.542 62.3616C109.388 62.4981 109.193 62.5663 108.957 62.5663H105.447C105.211 62.5663 105.016 62.4981 104.862 62.3616C104.707 62.2251 104.63 62.0528 104.63 61.8448V59.3631Z" fill="#D0D1D7"/><path d="M107.788 59.3534C107.788 59.1454 107.866 58.9731 108.02 58.8366C108.175 58.7001 108.37 58.6318 108.605 58.6318H115.273C115.509 58.6318 115.704 58.7001 115.858 58.8366C116.013 58.9731 116.09 59.1454 116.09 59.3534V61.8448C116.09 62.0528 116.013 62.2251 115.858 62.3616C115.704 62.4981 115.509 62.5663 115.273 62.5663H108.605C108.37 62.5663 108.175 62.4981 108.02 62.3616C107.866 62.2251 107.788 62.0528 107.788 61.8448V59.3534Z" fill="#D0D1D7"/><path d="M114.106 59.3631C114.106 59.1551 114.183 58.9829 114.338 58.8463C114.492 58.7098 114.687 58.6416 114.923 58.6416H118.433C118.669 58.6416 118.864 58.7098 119.018 58.8463C119.173 58.9829 119.25 59.1551 119.25 59.3631V61.8448C119.25 62.0528 119.173 62.2251 119.018 62.3616C118.864 62.4981 118.669 62.5663 118.433 62.5663H114.923C114.687 62.5663 114.492 62.4981 114.338 62.3616C114.183 62.2251 114.106 62.0528 114.106 61.8448V59.3631Z" fill="#D0D1D7"/><path d="M117.265 59.3534C117.265 59.1454 117.342 58.9731 117.496 58.8366C117.651 58.7001 117.846 58.6318 118.082 58.6318H124.749C124.985 58.6318 125.18 58.7001 125.334 58.8366C125.489 58.9731 125.566 59.1454 125.566 59.3534V61.8448C125.566 62.0528 125.489 62.2251 125.334 62.3616C125.18 62.4981 124.985 62.5663 124.749 62.5663H118.082C117.846 62.5663 117.651 62.4981 117.496 62.3616C117.342 62.2251 117.265 62.0528 117.265 61.8448V59.3534Z" fill="#D0D1D7"/><path d="M123.582 59.3534C123.582 59.1454 123.659 58.9731 123.814 58.8366C123.968 58.7001 124.164 58.6318 124.399 58.6318H131.067C131.302 58.6318 131.497 58.7001 131.652 58.8366C131.806 58.9731 131.884 59.1454 131.884 59.3534V61.8448C131.884 62.0528 131.806 62.2251 131.652 62.3616C131.497 62.4981 131.302 62.5663 131.067 62.5663H124.399C124.164 62.5663 123.968 62.4981 123.814 62.3616C123.659 62.2251 123.582 62.0528 123.582 61.8448V59.3534Z" fill="#D0D1D7"/><path d="M129.9 59.3534C129.9 59.1454 129.977 58.9731 130.131 58.8366C130.286 58.7001 130.481 58.6318 130.716 58.6318H137.384C137.62 58.6318 137.815 58.7001 137.969 58.8366C138.124 58.9731 138.201 59.1454 138.201 59.3534V61.8448C138.201 62.0528 138.124 62.2251 137.969 62.3616C137.815 62.4981 137.62 62.5663 137.384 62.5663H130.716C130.481 62.5663 130.286 62.4981 130.131 62.3616C129.977 62.2251 129.9 62.0528 129.9 61.8448V59.3534Z" fill="#D0D1D7"/><path d="M76.2012 66.4189C76.2012 66.2109 76.2784 66.0386 76.433 65.9021C76.5876 65.7656 76.7826 65.6974 77.0181 65.6974H83.6859C83.9214 65.6974 84.1164 65.7656 84.271 65.9021C84.4255 66.0386 84.5028 66.2109 84.5028 66.4189V68.9103C84.5028 69.1183 84.4255 69.2906 84.271 69.4271C84.1164 69.5636 83.9214 69.6318 83.6859 69.6318H77.0181C76.7826 69.6318 76.5876 69.5636 76.433 69.4271C76.2784 69.2906 76.2012 69.1183 76.2012 68.9103V66.4189Z" fill="#D0D1D7"/><path d="M82.5186 66.4189C82.5186 66.2109 82.5959 66.0386 82.7505 65.9021C82.905 65.7656 83.1 65.6974 83.3355 65.6974H90.0033C90.2388 65.6974 90.4339 65.7656 90.5884 65.9021C90.743 66.0386 90.8202 66.2109 90.8202 66.4189V68.9103C90.8202 69.1183 90.743 69.2906 90.5884 69.4271C90.4339 69.5636 90.2388 69.6318 90.0033 69.6318H83.3355C83.1 69.6318 82.905 69.5636 82.7505 69.4271C82.5959 69.2906 82.5186 69.1183 82.5186 68.9103V66.4189Z" fill="#D0D1D7"/><path d="M88.8361 66.4189C88.8361 66.2109 88.9134 66.0386 89.0679 65.9021C89.2225 65.7656 89.4175 65.6974 89.653 65.6974H96.3208C96.5563 65.6974 96.7513 65.7656 96.9059 65.9021C97.0604 66.0386 97.1377 66.2109 97.1377 66.4189V68.9103C97.1377 69.1183 97.0604 69.2906 96.9059 69.4271C96.7513 69.5636 96.5563 69.6318 96.3208 69.6318H89.653C89.4175 69.6318 89.2225 69.5636 89.0679 69.4271C88.9134 69.2906 88.8361 69.1183 88.8361 68.9103V66.4189Z" fill="#D0D1D7"/><path d="M95.1535 66.4189C95.1535 66.2109 95.2308 66.0386 95.3854 65.9021C95.5399 65.7656 95.735 65.6974 95.9705 65.6974H102.638C102.874 65.6974 103.069 65.7656 103.223 65.9021C103.378 66.0386 103.455 66.2109 103.455 66.4189V68.9103C103.455 69.1183 103.378 69.2906 103.223 69.4271C103.069 69.5636 102.874 69.6318 102.638 69.6318H95.9705C95.735 69.6318 95.5399 69.5636 95.3854 69.4271C95.2308 69.2906 95.1535 69.1183 95.1535 68.9103V66.4189Z" fill="#D0D1D7"/><path d="M101.471 66.4189C101.471 66.2109 101.548 66.0386 101.703 65.9021C101.857 65.7656 102.052 65.6974 102.288 65.6974H108.956C109.191 65.6974 109.386 65.7656 109.541 65.9021C109.695 66.0386 109.773 66.2109 109.773 66.4189V68.9103C109.773 69.1183 109.695 69.2906 109.541 69.4271C109.386 69.5636 109.191 69.6318 108.956 69.6318H102.288C102.052 69.6318 101.857 69.5636 101.703 69.4271C101.548 69.2906 101.471 69.1183 101.471 68.9103V66.4189Z" fill="#D0D1D7"/><path d="M107.788 66.4189C107.788 66.2109 107.866 66.0386 108.02 65.9021C108.175 65.7656 108.37 65.6974 108.605 65.6974H115.273C115.509 65.6974 115.704 65.7656 115.858 65.9021C116.013 66.0386 116.09 66.2109 116.09 66.4189V68.9103C116.09 69.1183 116.013 69.2906 115.858 69.4271C115.704 69.5636 115.509 69.6318 115.273 69.6318H108.605C108.37 69.6318 108.175 69.5636 108.02 69.4271C107.866 69.2906 107.788 69.1183 107.788 68.9103V66.4189Z" fill="#D0D1D7"/><path d="M114.106 66.4189C114.106 66.2109 114.183 66.0386 114.338 65.9021C114.492 65.7656 114.687 65.6974 114.923 65.6974H121.591C121.826 65.6974 122.021 65.7656 122.176 65.9021C122.33 66.0386 122.408 66.2109 122.408 66.4189V68.9103C122.408 69.1183 122.33 69.2906 122.176 69.4271C122.021 69.5636 121.826 69.6318 121.591 69.6318H114.923C114.687 69.6318 114.492 69.5636 114.338 69.4271C114.183 69.2906 114.106 69.1183 114.106 68.9103V66.4189Z" fill="#D0D1D7"/><path d="M120.423 66.4286C120.423 66.2206 120.501 66.0484 120.655 65.9119C120.81 65.7754 121.005 65.7071 121.24 65.7071H124.751C124.986 65.7071 125.181 65.7754 125.336 65.9119C125.49 66.0484 125.568 66.2206 125.568 66.4286V68.9103C125.568 69.1183 125.49 69.2906 125.336 69.4271C125.181 69.5636 124.986 69.6318 124.751 69.6318H121.24C121.005 69.6318 120.81 69.5636 120.655 69.4271C120.501 69.2906 120.423 69.1183 120.423 68.9103V66.4286Z" fill="#D0D1D7"/><rect x="187.271" y="49" width="26.7277" height="10.4911" rx="0.915179" fill="#0096CC"/><g clip-path="url(#clip9_2180_29197)"><path d="M193.219 53.2177C192.647 53.2177 192.192 53.6825 192.192 54.2455C192.192 54.8175 192.647 55.2733 193.219 55.2733C193.782 55.2733 194.247 54.8175 194.247 54.2455C194.247 53.6825 193.782 53.2177 193.219 53.2177ZM193.219 54.9158C192.853 54.9158 192.549 54.6209 192.549 54.2455C192.549 53.8791 192.844 53.5842 193.219 53.5842C193.586 53.5842 193.881 53.8791 193.881 54.2455C193.881 54.6209 193.586 54.9158 193.219 54.9158ZM194.524 53.182C194.524 53.0479 194.417 52.9407 194.283 52.9407C194.149 52.9407 194.042 53.0479 194.042 53.182C194.042 53.3161 194.149 53.4233 194.283 53.4233C194.417 53.4233 194.524 53.3161 194.524 53.182ZM195.203 53.4233C195.186 53.1016 195.114 52.8156 194.882 52.5832C194.649 52.3508 194.363 52.2793 194.042 52.2615C193.711 52.2436 192.719 52.2436 192.388 52.2615C192.066 52.2793 191.789 52.3508 191.548 52.5832C191.316 52.8156 191.244 53.1016 191.226 53.4233C191.208 53.754 191.208 54.746 191.226 55.0767C191.244 55.3984 191.316 55.6755 191.548 55.9168C191.789 56.1492 192.066 56.2207 192.388 56.2385C192.719 56.2564 193.711 56.2564 194.042 56.2385C194.363 56.2207 194.649 56.1492 194.882 55.9168C195.114 55.6755 195.186 55.3984 195.203 55.0767C195.221 54.746 195.221 53.754 195.203 53.4233ZM194.774 55.4253C194.712 55.604 194.569 55.7381 194.399 55.8096C194.131 55.9168 193.505 55.89 193.219 55.89C192.924 55.89 192.299 55.9168 192.04 55.8096C191.861 55.7381 191.727 55.604 191.655 55.4253C191.548 55.1661 191.575 54.5405 191.575 54.2455C191.575 53.9595 191.548 53.3339 191.655 53.0658C191.727 52.896 191.861 52.7619 192.04 52.6904C192.299 52.5832 192.924 52.61 193.219 52.61C193.505 52.61 194.131 52.5832 194.399 52.6904C194.569 52.753 194.703 52.896 194.774 53.0658C194.882 53.3339 194.855 53.9595 194.855 54.2455C194.855 54.5405 194.882 55.1661 194.774 55.4253Z" fill="white"/></g><path d="M197.625 55.7451H198.152V54.5093H199.361V54.0676H198.152V53.2778H199.488V52.836H197.625V55.7451ZM200.777 55.7877C201.417 55.7877 201.823 55.3374 201.823 54.6627C201.823 53.9866 201.417 53.5349 200.777 53.5349C200.138 53.5349 199.732 53.9866 199.732 54.6627C199.732 55.3374 200.138 55.7877 200.777 55.7877ZM200.78 55.3758C200.427 55.3758 200.253 55.0605 200.253 54.6613C200.253 54.2622 200.427 53.9426 200.78 53.9426C201.128 53.9426 201.302 54.2622 201.302 54.6613C201.302 55.0605 201.128 55.3758 200.78 55.3758ZM202.813 52.836H202.299V55.7451H202.813V52.836ZM203.896 52.836H203.382V55.7451H203.896V52.836ZM205.417 55.7877C206.056 55.7877 206.462 55.3374 206.462 54.6627C206.462 53.9866 206.056 53.5349 205.417 53.5349C204.778 53.5349 204.371 53.9866 204.371 54.6627C204.371 55.3374 204.778 55.7877 205.417 55.7877ZM205.42 55.3758C205.066 55.3758 204.893 55.0605 204.893 54.6613C204.893 54.2622 205.066 53.9426 205.42 53.9426C205.768 53.9426 205.941 54.2622 205.941 54.6613C205.941 55.0605 205.768 55.3758 205.42 55.3758ZM207.331 55.7451H207.867L208.277 54.2707H208.306L208.715 55.7451H209.251L209.869 53.5633H209.343L208.965 55.0889H208.944L208.552 53.5633H208.034L207.642 55.0974H207.622L207.238 53.5633H206.714L207.331 55.7451Z" fill="white"/></g></g></g><defs><filter id="filter0_ddd_2180_29197" x="16" y="27.1426" width="231" height="201.714" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="6"/><feGaussianBlur stdDeviation="6.5"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_2180_29197"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="1"/><feGaussianBlur stdDeviation="1"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/><feBlend mode="normal" in2="effect1_dropShadow_2180_29197" result="effect2_dropShadow_2180_29197"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="3"/><feGaussianBlur stdDeviation="3"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/><feBlend mode="normal" in2="effect2_dropShadow_2180_29197" result="effect3_dropShadow_2180_29197"/><feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow_2180_29197" result="shape"/></filter><clipPath id="clip0_2180_29197"><rect width="262.5" height="200" fill="white" transform="translate(0.5)"/></clipPath><clipPath id="clip1_2180_29197"><rect x="29" y="34.1426" width="205" height="175.714" rx="2" fill="white"/></clipPath><clipPath id="clip2_2180_29197"><rect width="83.6639" height="76.6214" fill="white" transform="translate(47 84)"/></clipPath><clipPath id="clip3_2180_29197"><rect width="40.9868" height="37.4656" fill="white" transform="translate(132.354 84)"/></clipPath><clipPath id="clip4_2180_29197"><rect x="167.566" y="85.127" width="3.94375" height="3.94375" rx="1.97188" fill="white"/></clipPath><clipPath id="clip5_2180_29197"><rect width="40.9868" height="37.4656" fill="white" transform="translate(175.031 84)"/></clipPath><clipPath id="clip6_2180_29197"><rect width="40.9868" height="37.4656" fill="white" transform="translate(47 162.312)"/></clipPath><clipPath id="clip7_2180_29197"><rect x="82.2129" y="163.438" width="3.94375" height="3.94375" rx="1.97188" fill="white"/></clipPath><clipPath id="clip8_2180_29197"><rect width="83.6639" height="76.6214" fill="white" transform="translate(132.354 162.312)"/></clipPath><clipPath id="clip9_2180_29197"><rect width="4.57589" height="4.57589" fill="white" transform="translate(190.932 51.958)"/></clipPath></defs></svg>',
			'singlePostIcon'         => '<svg width="263" height="200" viewBox="0 0 263 200" fill="none" xmlns="http://www.w3.org/2000/svg"><g clip-path="url(#clip0_2180_29258)"><rect width="262.5" height="200" fill="#F3F4F5"/><g filter="url(#filter0_ddd_2180_29258)"><rect x="29" y="19" width="205" height="164" rx="2" fill="white"/><g clip-path="url(#clip1_2180_29258)"><g clip-path="url(#clip2_2180_29258)"><rect width="183" height="136" transform="translate(40 63)" fill="#FFDF99"/><circle cx="187.018" cy="219.929" r="117" fill="#FFD066"/></g></g><circle cx="51.9821" cy="41.9821" r="10.9821" fill="#FFDF99"/><path d="M71.2012 33.146C71.2012 32.9114 71.2782 32.717 71.4322 32.563C71.5862 32.409 71.7805 32.332 72.0152 32.332H78.6592C78.8938 32.332 79.0882 32.409 79.2422 32.563C79.3962 32.717 79.4732 32.9114 79.4732 33.146V35.818C79.4732 36.0527 79.3962 36.247 79.2422 36.401C79.0882 36.555 78.8938 36.632 78.6592 36.632H72.0152C71.7805 36.632 71.5862 36.555 71.4322 36.401C71.2782 36.247 71.2012 36.0527 71.2012 35.818V33.146Z" fill="#8C8F9A"/><path d="M77.4961 33.146C77.4961 32.9114 77.5731 32.717 77.7271 32.563C77.8811 32.409 78.0754 32.332 78.3101 32.332H84.9541C85.1888 32.332 85.3831 32.409 85.5371 32.563C85.6911 32.717 85.7681 32.9114 85.7681 33.146V35.818C85.7681 36.0527 85.6911 36.247 85.5371 36.401C85.3831 36.555 85.1888 36.632 84.9541 36.632H78.3101C78.0754 36.632 77.8811 36.555 77.7271 36.401C77.5731 36.247 77.4961 36.0527 77.4961 35.818V33.146Z" fill="#8C8F9A"/><path d="M83.791 33.146C83.791 32.9114 83.868 32.717 84.022 32.563C84.176 32.409 84.3703 32.332 84.605 32.332H88.249C88.4837 32.332 88.678 32.409 88.832 32.563C88.986 32.717 89.063 32.9114 89.063 33.146V35.818C89.063 36.0527 88.986 36.247 88.832 36.401C88.678 36.555 88.4837 36.632 88.249 36.632H84.605C84.3703 36.632 84.176 36.555 84.022 36.401C83.868 36.247 83.791 36.0527 83.791 35.818V33.146Z" fill="#8C8F9A"/><path d="M87.0859 33.146C87.0859 32.9114 87.1629 32.717 87.3169 32.563C87.4709 32.409 87.6653 32.332 87.8999 32.332H94.5439C94.7786 32.332 94.9729 32.409 95.1269 32.563C95.2809 32.717 95.3579 32.9114 95.3579 33.146V35.818C95.3579 36.0527 95.2809 36.247 95.1269 36.401C94.9729 36.555 94.7786 36.632 94.5439 36.632H87.8999C87.6653 36.632 87.4709 36.555 87.3169 36.401C87.1629 36.247 87.0859 36.0527 87.0859 35.818V33.146Z" fill="#8C8F9A"/><path d="M93.3809 33.157C93.3809 32.9224 93.4579 32.728 93.6119 32.574C93.7659 32.42 93.9602 32.343 94.1949 32.343H97.6929C97.9275 32.343 98.1219 32.42 98.2759 32.574C98.4299 32.728 98.5069 32.9224 98.5069 33.157V35.818C98.5069 36.0527 98.4299 36.247 98.2759 36.401C98.1219 36.555 97.9275 36.632 97.6929 36.632H94.1949C93.9602 36.632 93.7659 36.555 93.6119 36.401C93.4579 36.247 93.3809 36.0527 93.3809 35.818V33.157Z" fill="#8C8F9A"/><path d="M96.5283 33.146C96.5283 32.9114 96.6053 32.717 96.7593 32.563C96.9133 32.409 97.1077 32.332 97.3423 32.332H103.986C104.221 32.332 104.415 32.409 104.569 32.563C104.723 32.717 104.8 32.9114 104.8 33.146V35.818C104.8 36.0527 104.723 36.247 104.569 36.401C104.415 36.555 104.221 36.632 103.986 36.632H97.3423C97.1077 36.632 96.9133 36.555 96.7593 36.401C96.6053 36.247 96.5283 36.0527 96.5283 35.818V33.146Z" fill="#8C8F9A"/><path d="M71.2012 41.3534C71.2012 41.1454 71.2784 40.9731 71.433 40.8366C71.5876 40.7001 71.7826 40.6318 72.0181 40.6318H78.6859C78.9214 40.6318 79.1164 40.7001 79.271 40.8366C79.4255 40.9731 79.5028 41.1454 79.5028 41.3534V43.8448C79.5028 44.0528 79.4255 44.2251 79.271 44.3616C79.1164 44.4981 78.9214 44.5663 78.6859 44.5663H72.0181C71.7826 44.5663 71.5876 44.4981 71.433 44.3616C71.2784 44.2251 71.2012 44.0528 71.2012 43.8448V41.3534Z" fill="#D0D1D7"/><path d="M77.5186 41.3534C77.5186 41.1454 77.5959 40.9731 77.7505 40.8366C77.905 40.7001 78.1 40.6318 78.3355 40.6318H85.0033C85.2388 40.6318 85.4339 40.7001 85.5884 40.8366C85.743 40.9731 85.8202 41.1454 85.8202 41.3534V43.8448C85.8202 44.0528 85.743 44.2251 85.5884 44.3616C85.4339 44.4981 85.2388 44.5663 85.0033 44.5663H78.3355C78.1 44.5663 77.905 44.4981 77.7505 44.3616C77.5959 44.2251 77.5186 44.0528 77.5186 43.8448V41.3534Z" fill="#D0D1D7"/><path d="M83.8361 41.3534C83.8361 41.1454 83.9134 40.9731 84.0679 40.8366C84.2225 40.7001 84.4175 40.6318 84.653 40.6318H91.3208C91.5563 40.6318 91.7513 40.7001 91.9059 40.8366C92.0604 40.9731 92.1377 41.1454 92.1377 41.3534V43.8448C92.1377 44.0528 92.0604 44.2251 91.9059 44.3616C91.7513 44.4981 91.5563 44.5663 91.3208 44.5663H84.653C84.4175 44.5663 84.2225 44.4981 84.0679 44.3616C83.9134 44.2251 83.8361 44.0528 83.8361 43.8448V41.3534Z" fill="#D0D1D7"/><path d="M90.1535 41.3534C90.1535 41.1454 90.2308 40.9731 90.3854 40.8366C90.5399 40.7001 90.735 40.6318 90.9705 40.6318H97.6382C97.8738 40.6318 98.0688 40.7001 98.2233 40.8366C98.3779 40.9731 98.4552 41.1454 98.4552 41.3534V43.8448C98.4552 44.0528 98.3779 44.2251 98.2233 44.3616C98.0688 44.4981 97.8738 44.5663 97.6382 44.5663H90.9705C90.735 44.5663 90.5399 44.4981 90.3854 44.3616C90.2308 44.2251 90.1535 44.0528 90.1535 43.8448V41.3534Z" fill="#D0D1D7"/><path d="M96.471 41.3631C96.471 41.1551 96.5483 40.9829 96.7028 40.8463C96.8574 40.7098 97.0524 40.6416 97.2879 40.6416H100.798C101.034 40.6416 101.229 40.7098 101.384 40.8463C101.538 40.9829 101.615 41.1551 101.615 41.3631V43.8448C101.615 44.0528 101.538 44.2251 101.384 44.3616C101.229 44.4981 101.034 44.5663 100.798 44.5663H97.2879C97.0524 44.5663 96.8574 44.4981 96.7028 44.3616C96.5483 44.2251 96.471 44.0528 96.471 43.8448V41.3631Z" fill="#D0D1D7"/><path d="M99.6297 41.3631C99.6297 41.1551 99.707 40.9829 99.8616 40.8463C100.016 40.7098 100.211 40.6416 100.447 40.6416H103.957C104.193 40.6416 104.388 40.7098 104.542 40.8463C104.697 40.9829 104.774 41.1551 104.774 41.3631V43.8448C104.774 44.0528 104.697 44.2251 104.542 44.3616C104.388 44.4981 104.193 44.5663 103.957 44.5663H100.447C100.211 44.5663 100.016 44.4981 99.8616 44.3616C99.707 44.2251 99.6297 44.0528 99.6297 43.8448V41.3631Z" fill="#D0D1D7"/><path d="M102.788 41.3534C102.788 41.1454 102.866 40.9731 103.02 40.8366C103.175 40.7001 103.37 40.6318 103.605 40.6318H110.273C110.509 40.6318 110.704 40.7001 110.858 40.8366C111.013 40.9731 111.09 41.1454 111.09 41.3534V43.8448C111.09 44.0528 111.013 44.2251 110.858 44.3616C110.704 44.4981 110.509 44.5663 110.273 44.5663H103.605C103.37 44.5663 103.175 44.4981 103.02 44.3616C102.866 44.2251 102.788 44.0528 102.788 43.8448V41.3534Z" fill="#D0D1D7"/><path d="M109.106 41.3631C109.106 41.1551 109.183 40.9829 109.338 40.8463C109.492 40.7098 109.687 40.6416 109.923 40.6416H113.433C113.669 40.6416 113.864 40.7098 114.018 40.8463C114.173 40.9829 114.25 41.1551 114.25 41.3631V43.8448C114.25 44.0528 114.173 44.2251 114.018 44.3616C113.864 44.4981 113.669 44.5663 113.433 44.5663H109.923C109.687 44.5663 109.492 44.4981 109.338 44.3616C109.183 44.2251 109.106 44.0528 109.106 43.8448V41.3631Z" fill="#D0D1D7"/><path d="M112.265 41.3534C112.265 41.1454 112.342 40.9731 112.496 40.8366C112.651 40.7001 112.846 40.6318 113.082 40.6318H119.749C119.985 40.6318 120.18 40.7001 120.334 40.8366C120.489 40.9731 120.566 41.1454 120.566 41.3534V43.8448C120.566 44.0528 120.489 44.2251 120.334 44.3616C120.18 44.4981 119.985 44.5663 119.749 44.5663H113.082C112.846 44.5663 112.651 44.4981 112.496 44.3616C112.342 44.2251 112.265 44.0528 112.265 43.8448V41.3534Z" fill="#D0D1D7"/><path d="M118.582 41.3534C118.582 41.1454 118.659 40.9731 118.814 40.8366C118.968 40.7001 119.164 40.6318 119.399 40.6318H126.067C126.302 40.6318 126.497 40.7001 126.652 40.8366C126.806 40.9731 126.884 41.1454 126.884 41.3534V43.8448C126.884 44.0528 126.806 44.2251 126.652 44.3616C126.497 44.4981 126.302 44.5663 126.067 44.5663H119.399C119.164 44.5663 118.968 44.4981 118.814 44.3616C118.659 44.2251 118.582 44.0528 118.582 43.8448V41.3534Z" fill="#D0D1D7"/><path d="M124.9 41.3534C124.9 41.1454 124.977 40.9731 125.131 40.8366C125.286 40.7001 125.481 40.6318 125.716 40.6318H132.384C132.62 40.6318 132.815 40.7001 132.969 40.8366C133.124 40.9731 133.201 41.1454 133.201 41.3534V43.8448C133.201 44.0528 133.124 44.2251 132.969 44.3616C132.815 44.4981 132.62 44.5663 132.384 44.5663H125.716C125.481 44.5663 125.286 44.4981 125.131 44.3616C124.977 44.2251 124.9 44.0528 124.9 43.8448V41.3534Z" fill="#D0D1D7"/><path d="M71.2012 48.4189C71.2012 48.2109 71.2784 48.0386 71.433 47.9021C71.5876 47.7656 71.7826 47.6974 72.0181 47.6974H78.6859C78.9214 47.6974 79.1164 47.7656 79.271 47.9021C79.4255 48.0386 79.5028 48.2109 79.5028 48.4189V50.9103C79.5028 51.1183 79.4255 51.2906 79.271 51.4271C79.1164 51.5636 78.9214 51.6318 78.6859 51.6318H72.0181C71.7826 51.6318 71.5876 51.5636 71.433 51.4271C71.2784 51.2906 71.2012 51.1183 71.2012 50.9103V48.4189Z" fill="#D0D1D7"/><path d="M77.5186 48.4189C77.5186 48.2109 77.5959 48.0386 77.7505 47.9021C77.905 47.7656 78.1 47.6974 78.3355 47.6974H85.0033C85.2388 47.6974 85.4339 47.7656 85.5884 47.9021C85.743 48.0386 85.8202 48.2109 85.8202 48.4189V50.9103C85.8202 51.1183 85.743 51.2906 85.5884 51.4271C85.4339 51.5636 85.2388 51.6318 85.0033 51.6318H78.3355C78.1 51.6318 77.905 51.5636 77.7505 51.4271C77.5959 51.2906 77.5186 51.1183 77.5186 50.9103V48.4189Z" fill="#D0D1D7"/><path d="M83.8361 48.4189C83.8361 48.2109 83.9134 48.0386 84.0679 47.9021C84.2225 47.7656 84.4175 47.6974 84.653 47.6974H91.3208C91.5563 47.6974 91.7513 47.7656 91.9059 47.9021C92.0604 48.0386 92.1377 48.2109 92.1377 48.4189V50.9103C92.1377 51.1183 92.0604 51.2906 91.9059 51.4271C91.7513 51.5636 91.5563 51.6318 91.3208 51.6318H84.653C84.4175 51.6318 84.2225 51.5636 84.0679 51.4271C83.9134 51.2906 83.8361 51.1183 83.8361 50.9103V48.4189Z" fill="#D0D1D7"/><path d="M90.1535 48.4189C90.1535 48.2109 90.2308 48.0386 90.3854 47.9021C90.5399 47.7656 90.735 47.6974 90.9705 47.6974H97.6382C97.8738 47.6974 98.0688 47.7656 98.2233 47.9021C98.3779 48.0386 98.4552 48.2109 98.4552 48.4189V50.9103C98.4552 51.1183 98.3779 51.2906 98.2233 51.4271C98.0688 51.5636 97.8738 51.6318 97.6382 51.6318H90.9705C90.735 51.6318 90.5399 51.5636 90.3854 51.4271C90.2308 51.2906 90.1535 51.1183 90.1535 50.9103V48.4189Z" fill="#D0D1D7"/><path d="M96.471 48.4189C96.471 48.2109 96.5483 48.0386 96.7028 47.9021C96.8574 47.7656 97.0524 47.6974 97.2879 47.6974H103.956C104.191 47.6974 104.386 47.7656 104.541 47.9021C104.695 48.0386 104.773 48.2109 104.773 48.4189V50.9103C104.773 51.1183 104.695 51.2906 104.541 51.4271C104.386 51.5636 104.191 51.6318 103.956 51.6318H97.2879C97.0524 51.6318 96.8574 51.5636 96.7028 51.4271C96.5483 51.2906 96.471 51.1183 96.471 50.9103V48.4189Z" fill="#D0D1D7"/><path d="M102.788 48.4189C102.788 48.2109 102.866 48.0386 103.02 47.9021C103.175 47.7656 103.37 47.6974 103.605 47.6974H110.273C110.509 47.6974 110.704 47.7656 110.858 47.9021C111.013 48.0386 111.09 48.2109 111.09 48.4189V50.9103C111.09 51.1183 111.013 51.2906 110.858 51.4271C110.704 51.5636 110.509 51.6318 110.273 51.6318H103.605C103.37 51.6318 103.175 51.5636 103.02 51.4271C102.866 51.2906 102.788 51.1183 102.788 50.9103V48.4189Z" fill="#D0D1D7"/><path d="M109.106 48.4189C109.106 48.2109 109.183 48.0386 109.338 47.9021C109.492 47.7656 109.687 47.6974 109.923 47.6974H116.591C116.826 47.6974 117.021 47.7656 117.176 47.9021C117.33 48.0386 117.408 48.2109 117.408 48.4189V50.9103C117.408 51.1183 117.33 51.2906 117.176 51.4271C117.021 51.5636 116.826 51.6318 116.591 51.6318H109.923C109.687 51.6318 109.492 51.5636 109.338 51.4271C109.183 51.2906 109.106 51.1183 109.106 50.9103V48.4189Z" fill="#D0D1D7"/><path d="M115.423 48.4286C115.423 48.2206 115.501 48.0484 115.655 47.9119C115.81 47.7754 116.005 47.7071 116.24 47.7071H119.751C119.986 47.7071 120.181 47.7754 120.336 47.9119C120.49 48.0484 120.568 48.2206 120.568 48.4286V50.9103C120.568 51.1183 120.49 51.2906 120.336 51.4271C120.181 51.5636 119.986 51.6318 119.751 51.6318H116.24C116.005 51.6318 115.81 51.5636 115.655 51.4271C115.501 51.2906 115.423 51.1183 115.423 50.9103V48.4286Z" fill="#D0D1D7"/><rect x="196.271" y="31" width="26.7277" height="10.4911" rx="0.915179" fill="#0096CC"/><g clip-path="url(#clip3_2180_29258)"><path d="M202.219 35.2177C201.647 35.2177 201.192 35.6825 201.192 36.2455C201.192 36.8175 201.647 37.2733 202.219 37.2733C202.782 37.2733 203.247 36.8175 203.247 36.2455C203.247 35.6825 202.782 35.2177 202.219 35.2177ZM202.219 36.9158C201.853 36.9158 201.549 36.6209 201.549 36.2455C201.549 35.8791 201.844 35.5842 202.219 35.5842C202.586 35.5842 202.881 35.8791 202.881 36.2455C202.881 36.6209 202.586 36.9158 202.219 36.9158ZM203.524 35.182C203.524 35.0479 203.417 34.9407 203.283 34.9407C203.149 34.9407 203.042 35.0479 203.042 35.182C203.042 35.3161 203.149 35.4233 203.283 35.4233C203.417 35.4233 203.524 35.3161 203.524 35.182ZM204.203 35.4233C204.186 35.1016 204.114 34.8156 203.882 34.5832C203.649 34.3508 203.363 34.2793 203.042 34.2615C202.711 34.2436 201.719 34.2436 201.388 34.2615C201.066 34.2793 200.789 34.3508 200.548 34.5832C200.316 34.8156 200.244 35.1016 200.226 35.4233C200.208 35.754 200.208 36.746 200.226 37.0767C200.244 37.3984 200.316 37.6755 200.548 37.9168C200.789 38.1492 201.066 38.2207 201.388 38.2385C201.719 38.2564 202.711 38.2564 203.042 38.2385C203.363 38.2207 203.649 38.1492 203.882 37.9168C204.114 37.6755 204.186 37.3984 204.203 37.0767C204.221 36.746 204.221 35.754 204.203 35.4233ZM203.774 37.4253C203.712 37.604 203.569 37.7381 203.399 37.8096C203.131 37.9168 202.505 37.89 202.219 37.89C201.924 37.89 201.299 37.9168 201.04 37.8096C200.861 37.7381 200.727 37.604 200.655 37.4253C200.548 37.1661 200.575 36.5405 200.575 36.2455C200.575 35.9595 200.548 35.3339 200.655 35.0658C200.727 34.896 200.861 34.7619 201.04 34.6904C201.299 34.5832 201.924 34.61 202.219 34.61C202.505 34.61 203.131 34.5832 203.399 34.6904C203.569 34.753 203.703 34.896 203.774 35.0658C203.882 35.3339 203.855 35.9595 203.855 36.2455C203.855 36.5405 203.882 37.1661 203.774 37.4253Z" fill="white"/></g><path d="M206.625 37.7451H207.152V36.5093H208.361V36.0676H207.152V35.2778H208.488V34.836H206.625V37.7451ZM209.777 37.7877C210.417 37.7877 210.823 37.3374 210.823 36.6627C210.823 35.9866 210.417 35.5349 209.777 35.5349C209.138 35.5349 208.732 35.9866 208.732 36.6627C208.732 37.3374 209.138 37.7877 209.777 37.7877ZM209.78 37.3758C209.427 37.3758 209.253 37.0605 209.253 36.6613C209.253 36.2622 209.427 35.9426 209.78 35.9426C210.128 35.9426 210.302 36.2622 210.302 36.6613C210.302 37.0605 210.128 37.3758 209.78 37.3758ZM211.813 34.836H211.299V37.7451H211.813V34.836ZM212.896 34.836H212.382V37.7451H212.896V34.836ZM214.417 37.7877C215.056 37.7877 215.462 37.3374 215.462 36.6627C215.462 35.9866 215.056 35.5349 214.417 35.5349C213.778 35.5349 213.371 35.9866 213.371 36.6627C213.371 37.3374 213.778 37.7877 214.417 37.7877ZM214.42 37.3758C214.066 37.3758 213.893 37.0605 213.893 36.6613C213.893 36.2622 214.066 35.9426 214.42 35.9426C214.768 35.9426 214.941 36.2622 214.941 36.6613C214.941 37.0605 214.768 37.3758 214.42 37.3758ZM216.331 37.7451H216.867L217.277 36.2707H217.306L217.715 37.7451H218.251L218.869 35.5633H218.343L217.965 37.0889H217.944L217.552 35.5633H217.034L216.642 37.0974H216.622L216.238 35.5633H215.714L216.331 37.7451Z" fill="white"/></g></g><defs><filter id="filter0_ddd_2180_29258" x="16" y="12" width="231" height="190" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="6"/><feGaussianBlur stdDeviation="6.5"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_2180_29258"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="1"/><feGaussianBlur stdDeviation="1"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/><feBlend mode="normal" in2="effect1_dropShadow_2180_29258" result="effect2_dropShadow_2180_29258"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="3"/><feGaussianBlur stdDeviation="3"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/><feBlend mode="normal" in2="effect2_dropShadow_2180_29258" result="effect3_dropShadow_2180_29258"/><feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow_2180_29258" result="shape"/></filter><clipPath id="clip0_2180_29258"><rect width="262.5" height="200" fill="white"/></clipPath><clipPath id="clip1_2180_29258"><rect width="183" height="106" fill="white" transform="translate(40 63)"/></clipPath><clipPath id="clip2_2180_29258"><rect width="183" height="136" fill="white" transform="translate(40 63)"/></clipPath><clipPath id="clip3_2180_29258"><rect width="4.57589" height="4.57589" fill="white" transform="translate(199.932 33.958)"/></clipPath></defs></svg>',
			'singlePostCarouselIcon' => '<svg width="263" height="200" viewBox="0 0 263 200" fill="none" xmlns="http://www.w3.org/2000/svg"><g clip-path="url(#clip0_2180_29277)"><rect width="262.5" height="200" transform="translate(0.5)" fill="#F3F4F5"/><g filter="url(#filter0_ddd_2180_29277)"><rect x="30" y="19" width="205" height="164" rx="2" fill="white"/><g clip-path="url(#clip1_2180_29277)"><g clip-path="url(#clip2_2180_29277)"><rect width="183" height="136" transform="translate(41 63)" fill="#FFDF99"/><circle cx="198.018" cy="119.928" r="78.0178" fill="#FFD066"/></g></g><circle cx="115" cy="170" r="2" fill="#434960"/><circle cx="124" cy="170" r="2" fill="#D0D1D7"/><circle cx="133" cy="170" r="2" fill="#D0D1D7"/><circle cx="142" cy="170" r="2" fill="#D0D1D7"/><circle cx="151" cy="170" r="2" fill="#D0D1D7"/><circle cx="52.9821" cy="41.9821" r="10.9821" fill="#FFDF99"/><path d="M72.2012 33.146C72.2012 32.9114 72.2782 32.717 72.4322 32.563C72.5862 32.409 72.7805 32.332 73.0152 32.332H79.6592C79.8938 32.332 80.0882 32.409 80.2422 32.563C80.3962 32.717 80.4732 32.9114 80.4732 33.146V35.818C80.4732 36.0527 80.3962 36.247 80.2422 36.401C80.0882 36.555 79.8938 36.632 79.6592 36.632H73.0152C72.7805 36.632 72.5862 36.555 72.4322 36.401C72.2782 36.247 72.2012 36.0527 72.2012 35.818V33.146Z" fill="#8C8F9A"/><path d="M78.4961 33.146C78.4961 32.9114 78.5731 32.717 78.7271 32.563C78.8811 32.409 79.0754 32.332 79.3101 32.332H85.9541C86.1888 32.332 86.3831 32.409 86.5371 32.563C86.6911 32.717 86.7681 32.9114 86.7681 33.146V35.818C86.7681 36.0527 86.6911 36.247 86.5371 36.401C86.3831 36.555 86.1888 36.632 85.9541 36.632H79.3101C79.0754 36.632 78.8811 36.555 78.7271 36.401C78.5731 36.247 78.4961 36.0527 78.4961 35.818V33.146Z" fill="#8C8F9A"/><path d="M84.791 33.146C84.791 32.9114 84.868 32.717 85.022 32.563C85.176 32.409 85.3703 32.332 85.605 32.332H89.249C89.4837 32.332 89.678 32.409 89.832 32.563C89.986 32.717 90.063 32.9114 90.063 33.146V35.818C90.063 36.0527 89.986 36.247 89.832 36.401C89.678 36.555 89.4837 36.632 89.249 36.632H85.605C85.3703 36.632 85.176 36.555 85.022 36.401C84.868 36.247 84.791 36.0527 84.791 35.818V33.146Z" fill="#8C8F9A"/><path d="M88.0859 33.146C88.0859 32.9114 88.1629 32.717 88.3169 32.563C88.4709 32.409 88.6653 32.332 88.8999 32.332H95.5439C95.7786 32.332 95.9729 32.409 96.1269 32.563C96.2809 32.717 96.3579 32.9114 96.3579 33.146V35.818C96.3579 36.0527 96.2809 36.247 96.1269 36.401C95.9729 36.555 95.7786 36.632 95.5439 36.632H88.8999C88.6653 36.632 88.4709 36.555 88.3169 36.401C88.1629 36.247 88.0859 36.0527 88.0859 35.818V33.146Z" fill="#8C8F9A"/><path d="M94.3809 33.157C94.3809 32.9224 94.4579 32.728 94.6119 32.574C94.7659 32.42 94.9602 32.343 95.1949 32.343H98.6929C98.9275 32.343 99.1219 32.42 99.2759 32.574C99.4299 32.728 99.5069 32.9224 99.5069 33.157V35.818C99.5069 36.0527 99.4299 36.247 99.2759 36.401C99.1219 36.555 98.9275 36.632 98.6929 36.632H95.1949C94.9602 36.632 94.7659 36.555 94.6119 36.401C94.4579 36.247 94.3809 36.0527 94.3809 35.818V33.157Z" fill="#8C8F9A"/><path d="M97.5283 33.146C97.5283 32.9114 97.6053 32.717 97.7593 32.563C97.9133 32.409 98.1077 32.332 98.3423 32.332H104.986C105.221 32.332 105.415 32.409 105.569 32.563C105.723 32.717 105.8 32.9114 105.8 33.146V35.818C105.8 36.0527 105.723 36.247 105.569 36.401C105.415 36.555 105.221 36.632 104.986 36.632H98.3423C98.1077 36.632 97.9133 36.555 97.7593 36.401C97.6053 36.247 97.5283 36.0527 97.5283 35.818V33.146Z" fill="#8C8F9A"/><path d="M72.2012 41.3534C72.2012 41.1454 72.2784 40.9731 72.433 40.8366C72.5876 40.7001 72.7826 40.6318 73.0181 40.6318H79.6859C79.9214 40.6318 80.1164 40.7001 80.271 40.8366C80.4255 40.9731 80.5028 41.1454 80.5028 41.3534V43.8448C80.5028 44.0528 80.4255 44.2251 80.271 44.3616C80.1164 44.4981 79.9214 44.5663 79.6859 44.5663H73.0181C72.7826 44.5663 72.5876 44.4981 72.433 44.3616C72.2784 44.2251 72.2012 44.0528 72.2012 43.8448V41.3534Z" fill="#D0D1D7"/><path d="M78.5186 41.3534C78.5186 41.1454 78.5959 40.9731 78.7505 40.8366C78.905 40.7001 79.1 40.6318 79.3355 40.6318H86.0033C86.2388 40.6318 86.4339 40.7001 86.5884 40.8366C86.743 40.9731 86.8202 41.1454 86.8202 41.3534V43.8448C86.8202 44.0528 86.743 44.2251 86.5884 44.3616C86.4339 44.4981 86.2388 44.5663 86.0033 44.5663H79.3355C79.1 44.5663 78.905 44.4981 78.7505 44.3616C78.5959 44.2251 78.5186 44.0528 78.5186 43.8448V41.3534Z" fill="#D0D1D7"/><path d="M84.8361 41.3534C84.8361 41.1454 84.9134 40.9731 85.0679 40.8366C85.2225 40.7001 85.4175 40.6318 85.653 40.6318H92.3208C92.5563 40.6318 92.7513 40.7001 92.9059 40.8366C93.0604 40.9731 93.1377 41.1454 93.1377 41.3534V43.8448C93.1377 44.0528 93.0604 44.2251 92.9059 44.3616C92.7513 44.4981 92.5563 44.5663 92.3208 44.5663H85.653C85.4175 44.5663 85.2225 44.4981 85.0679 44.3616C84.9134 44.2251 84.8361 44.0528 84.8361 43.8448V41.3534Z" fill="#D0D1D7"/><path d="M91.1535 41.3534C91.1535 41.1454 91.2308 40.9731 91.3854 40.8366C91.5399 40.7001 91.735 40.6318 91.9705 40.6318H98.6382C98.8738 40.6318 99.0688 40.7001 99.2233 40.8366C99.3779 40.9731 99.4552 41.1454 99.4552 41.3534V43.8448C99.4552 44.0528 99.3779 44.2251 99.2233 44.3616C99.0688 44.4981 98.8738 44.5663 98.6382 44.5663H91.9705C91.735 44.5663 91.5399 44.4981 91.3854 44.3616C91.2308 44.2251 91.1535 44.0528 91.1535 43.8448V41.3534Z" fill="#D0D1D7"/><path d="M97.471 41.3631C97.471 41.1551 97.5483 40.9829 97.7028 40.8463C97.8574 40.7098 98.0524 40.6416 98.2879 40.6416H101.798C102.034 40.6416 102.229 40.7098 102.384 40.8463C102.538 40.9829 102.615 41.1551 102.615 41.3631V43.8448C102.615 44.0528 102.538 44.2251 102.384 44.3616C102.229 44.4981 102.034 44.5663 101.798 44.5663H98.2879C98.0524 44.5663 97.8574 44.4981 97.7028 44.3616C97.5483 44.2251 97.471 44.0528 97.471 43.8448V41.3631Z" fill="#D0D1D7"/><path d="M100.63 41.3631C100.63 41.1551 100.707 40.9829 100.862 40.8463C101.016 40.7098 101.211 40.6416 101.447 40.6416H104.957C105.193 40.6416 105.388 40.7098 105.542 40.8463C105.697 40.9829 105.774 41.1551 105.774 41.3631V43.8448C105.774 44.0528 105.697 44.2251 105.542 44.3616C105.388 44.4981 105.193 44.5663 104.957 44.5663H101.447C101.211 44.5663 101.016 44.4981 100.862 44.3616C100.707 44.2251 100.63 44.0528 100.63 43.8448V41.3631Z" fill="#D0D1D7"/><path d="M103.788 41.3534C103.788 41.1454 103.866 40.9731 104.02 40.8366C104.175 40.7001 104.37 40.6318 104.605 40.6318H111.273C111.509 40.6318 111.704 40.7001 111.858 40.8366C112.013 40.9731 112.09 41.1454 112.09 41.3534V43.8448C112.09 44.0528 112.013 44.2251 111.858 44.3616C111.704 44.4981 111.509 44.5663 111.273 44.5663H104.605C104.37 44.5663 104.175 44.4981 104.02 44.3616C103.866 44.2251 103.788 44.0528 103.788 43.8448V41.3534Z" fill="#D0D1D7"/><path d="M110.106 41.3631C110.106 41.1551 110.183 40.9829 110.338 40.8463C110.492 40.7098 110.687 40.6416 110.923 40.6416H114.433C114.669 40.6416 114.864 40.7098 115.018 40.8463C115.173 40.9829 115.25 41.1551 115.25 41.3631V43.8448C115.25 44.0528 115.173 44.2251 115.018 44.3616C114.864 44.4981 114.669 44.5663 114.433 44.5663H110.923C110.687 44.5663 110.492 44.4981 110.338 44.3616C110.183 44.2251 110.106 44.0528 110.106 43.8448V41.3631Z" fill="#D0D1D7"/><path d="M113.265 41.3534C113.265 41.1454 113.342 40.9731 113.496 40.8366C113.651 40.7001 113.846 40.6318 114.082 40.6318H120.749C120.985 40.6318 121.18 40.7001 121.334 40.8366C121.489 40.9731 121.566 41.1454 121.566 41.3534V43.8448C121.566 44.0528 121.489 44.2251 121.334 44.3616C121.18 44.4981 120.985 44.5663 120.749 44.5663H114.082C113.846 44.5663 113.651 44.4981 113.496 44.3616C113.342 44.2251 113.265 44.0528 113.265 43.8448V41.3534Z" fill="#D0D1D7"/><path d="M119.582 41.3534C119.582 41.1454 119.659 40.9731 119.814 40.8366C119.968 40.7001 120.164 40.6318 120.399 40.6318H127.067C127.302 40.6318 127.497 40.7001 127.652 40.8366C127.806 40.9731 127.884 41.1454 127.884 41.3534V43.8448C127.884 44.0528 127.806 44.2251 127.652 44.3616C127.497 44.4981 127.302 44.5663 127.067 44.5663H120.399C120.164 44.5663 119.968 44.4981 119.814 44.3616C119.659 44.2251 119.582 44.0528 119.582 43.8448V41.3534Z" fill="#D0D1D7"/><path d="M125.9 41.3534C125.9 41.1454 125.977 40.9731 126.131 40.8366C126.286 40.7001 126.481 40.6318 126.716 40.6318H133.384C133.62 40.6318 133.815 40.7001 133.969 40.8366C134.124 40.9731 134.201 41.1454 134.201 41.3534V43.8448C134.201 44.0528 134.124 44.2251 133.969 44.3616C133.815 44.4981 133.62 44.5663 133.384 44.5663H126.716C126.481 44.5663 126.286 44.4981 126.131 44.3616C125.977 44.2251 125.9 44.0528 125.9 43.8448V41.3534Z" fill="#D0D1D7"/><path d="M72.2012 48.4189C72.2012 48.2109 72.2784 48.0386 72.433 47.9021C72.5876 47.7656 72.7826 47.6974 73.0181 47.6974H79.6859C79.9214 47.6974 80.1164 47.7656 80.271 47.9021C80.4255 48.0386 80.5028 48.2109 80.5028 48.4189V50.9103C80.5028 51.1183 80.4255 51.2906 80.271 51.4271C80.1164 51.5636 79.9214 51.6318 79.6859 51.6318H73.0181C72.7826 51.6318 72.5876 51.5636 72.433 51.4271C72.2784 51.2906 72.2012 51.1183 72.2012 50.9103V48.4189Z" fill="#D0D1D7"/><path d="M78.5186 48.4189C78.5186 48.2109 78.5959 48.0386 78.7505 47.9021C78.905 47.7656 79.1 47.6974 79.3355 47.6974H86.0033C86.2388 47.6974 86.4339 47.7656 86.5884 47.9021C86.743 48.0386 86.8202 48.2109 86.8202 48.4189V50.9103C86.8202 51.1183 86.743 51.2906 86.5884 51.4271C86.4339 51.5636 86.2388 51.6318 86.0033 51.6318H79.3355C79.1 51.6318 78.905 51.5636 78.7505 51.4271C78.5959 51.2906 78.5186 51.1183 78.5186 50.9103V48.4189Z" fill="#D0D1D7"/><path d="M84.8361 48.4189C84.8361 48.2109 84.9134 48.0386 85.0679 47.9021C85.2225 47.7656 85.4175 47.6974 85.653 47.6974H92.3208C92.5563 47.6974 92.7513 47.7656 92.9059 47.9021C93.0604 48.0386 93.1377 48.2109 93.1377 48.4189V50.9103C93.1377 51.1183 93.0604 51.2906 92.9059 51.4271C92.7513 51.5636 92.5563 51.6318 92.3208 51.6318H85.653C85.4175 51.6318 85.2225 51.5636 85.0679 51.4271C84.9134 51.2906 84.8361 51.1183 84.8361 50.9103V48.4189Z" fill="#D0D1D7"/><path d="M91.1535 48.4189C91.1535 48.2109 91.2308 48.0386 91.3854 47.9021C91.5399 47.7656 91.735 47.6974 91.9705 47.6974H98.6382C98.8738 47.6974 99.0688 47.7656 99.2233 47.9021C99.3779 48.0386 99.4552 48.2109 99.4552 48.4189V50.9103C99.4552 51.1183 99.3779 51.2906 99.2233 51.4271C99.0688 51.5636 98.8738 51.6318 98.6382 51.6318H91.9705C91.735 51.6318 91.5399 51.5636 91.3854 51.4271C91.2308 51.2906 91.1535 51.1183 91.1535 50.9103V48.4189Z" fill="#D0D1D7"/><path d="M97.471 48.4189C97.471 48.2109 97.5483 48.0386 97.7028 47.9021C97.8574 47.7656 98.0524 47.6974 98.2879 47.6974H104.956C105.191 47.6974 105.386 47.7656 105.541 47.9021C105.695 48.0386 105.773 48.2109 105.773 48.4189V50.9103C105.773 51.1183 105.695 51.2906 105.541 51.4271C105.386 51.5636 105.191 51.6318 104.956 51.6318H98.2879C98.0524 51.6318 97.8574 51.5636 97.7028 51.4271C97.5483 51.2906 97.471 51.1183 97.471 50.9103V48.4189Z" fill="#D0D1D7"/><path d="M103.788 48.4189C103.788 48.2109 103.866 48.0386 104.02 47.9021C104.175 47.7656 104.37 47.6974 104.605 47.6974H111.273C111.509 47.6974 111.704 47.7656 111.858 47.9021C112.013 48.0386 112.09 48.2109 112.09 48.4189V50.9103C112.09 51.1183 112.013 51.2906 111.858 51.4271C111.704 51.5636 111.509 51.6318 111.273 51.6318H104.605C104.37 51.6318 104.175 51.5636 104.02 51.4271C103.866 51.2906 103.788 51.1183 103.788 50.9103V48.4189Z" fill="#D0D1D7"/><path d="M110.106 48.4189C110.106 48.2109 110.183 48.0386 110.338 47.9021C110.492 47.7656 110.687 47.6974 110.923 47.6974H117.591C117.826 47.6974 118.021 47.7656 118.176 47.9021C118.33 48.0386 118.408 48.2109 118.408 48.4189V50.9103C118.408 51.1183 118.33 51.2906 118.176 51.4271C118.021 51.5636 117.826 51.6318 117.591 51.6318H110.923C110.687 51.6318 110.492 51.5636 110.338 51.4271C110.183 51.2906 110.106 51.1183 110.106 50.9103V48.4189Z" fill="#D0D1D7"/><path d="M116.423 48.4286C116.423 48.2206 116.501 48.0484 116.655 47.9119C116.81 47.7754 117.005 47.7071 117.24 47.7071H120.751C120.986 47.7071 121.181 47.7754 121.336 47.9119C121.49 48.0484 121.568 48.2206 121.568 48.4286V50.9103C121.568 51.1183 121.49 51.2906 121.336 51.4271C121.181 51.5636 120.986 51.6318 120.751 51.6318H117.24C117.005 51.6318 116.81 51.5636 116.655 51.4271C116.501 51.2906 116.423 51.1183 116.423 50.9103V48.4286Z" fill="#D0D1D7"/><rect x="197.271" y="31" width="26.7277" height="10.4911" rx="0.915179" fill="#0096CC"/><g clip-path="url(#clip3_2180_29277)"><path d="M203.219 35.2177C202.647 35.2177 202.192 35.6825 202.192 36.2455C202.192 36.8175 202.647 37.2733 203.219 37.2733C203.782 37.2733 204.247 36.8175 204.247 36.2455C204.247 35.6825 203.782 35.2177 203.219 35.2177ZM203.219 36.9158C202.853 36.9158 202.549 36.6209 202.549 36.2455C202.549 35.8791 202.844 35.5842 203.219 35.5842C203.586 35.5842 203.881 35.8791 203.881 36.2455C203.881 36.6209 203.586 36.9158 203.219 36.9158ZM204.524 35.182C204.524 35.0479 204.417 34.9407 204.283 34.9407C204.149 34.9407 204.042 35.0479 204.042 35.182C204.042 35.3161 204.149 35.4233 204.283 35.4233C204.417 35.4233 204.524 35.3161 204.524 35.182ZM205.203 35.4233C205.186 35.1016 205.114 34.8156 204.882 34.5832C204.649 34.3508 204.363 34.2793 204.042 34.2615C203.711 34.2436 202.719 34.2436 202.388 34.2615C202.066 34.2793 201.789 34.3508 201.548 34.5832C201.316 34.8156 201.244 35.1016 201.226 35.4233C201.208 35.754 201.208 36.746 201.226 37.0767C201.244 37.3984 201.316 37.6755 201.548 37.9168C201.789 38.1492 202.066 38.2207 202.388 38.2385C202.719 38.2564 203.711 38.2564 204.042 38.2385C204.363 38.2207 204.649 38.1492 204.882 37.9168C205.114 37.6755 205.186 37.3984 205.203 37.0767C205.221 36.746 205.221 35.754 205.203 35.4233ZM204.774 37.4253C204.712 37.604 204.569 37.7381 204.399 37.8096C204.131 37.9168 203.505 37.89 203.219 37.89C202.924 37.89 202.299 37.9168 202.04 37.8096C201.861 37.7381 201.727 37.604 201.655 37.4253C201.548 37.1661 201.575 36.5405 201.575 36.2455C201.575 35.9595 201.548 35.3339 201.655 35.0658C201.727 34.896 201.861 34.7619 202.04 34.6904C202.299 34.5832 202.924 34.61 203.219 34.61C203.505 34.61 204.131 34.5832 204.399 34.6904C204.569 34.753 204.703 34.896 204.774 35.0658C204.882 35.3339 204.855 35.9595 204.855 36.2455C204.855 36.5405 204.882 37.1661 204.774 37.4253Z" fill="white"/></g><path d="M207.625 37.7451H208.152V36.5093H209.361V36.0676H208.152V35.2778H209.488V34.836H207.625V37.7451ZM210.777 37.7877C211.417 37.7877 211.823 37.3374 211.823 36.6627C211.823 35.9866 211.417 35.5349 210.777 35.5349C210.138 35.5349 209.732 35.9866 209.732 36.6627C209.732 37.3374 210.138 37.7877 210.777 37.7877ZM210.78 37.3758C210.427 37.3758 210.253 37.0605 210.253 36.6613C210.253 36.2622 210.427 35.9426 210.78 35.9426C211.128 35.9426 211.302 36.2622 211.302 36.6613C211.302 37.0605 211.128 37.3758 210.78 37.3758ZM212.813 34.836H212.299V37.7451H212.813V34.836ZM213.896 34.836H213.382V37.7451H213.896V34.836ZM215.417 37.7877C216.056 37.7877 216.462 37.3374 216.462 36.6627C216.462 35.9866 216.056 35.5349 215.417 35.5349C214.778 35.5349 214.371 35.9866 214.371 36.6627C214.371 37.3374 214.778 37.7877 215.417 37.7877ZM215.42 37.3758C215.066 37.3758 214.893 37.0605 214.893 36.6613C214.893 36.2622 215.066 35.9426 215.42 35.9426C215.768 35.9426 215.941 36.2622 215.941 36.6613C215.941 37.0605 215.768 37.3758 215.42 37.3758ZM217.331 37.7451H217.867L218.277 36.2707H218.306L218.715 37.7451H219.251L219.869 35.5633H219.343L218.965 37.0889H218.944L218.552 35.5633H218.034L217.642 37.0974H217.622L217.238 35.5633H216.714L217.331 37.7451Z" fill="white"/></g></g><defs><filter id="filter0_ddd_2180_29277" x="17" y="12" width="231" height="190" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="6"/><feGaussianBlur stdDeviation="6.5"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_2180_29277"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="1"/><feGaussianBlur stdDeviation="1"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/><feBlend mode="normal" in2="effect1_dropShadow_2180_29277" result="effect2_dropShadow_2180_29277"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="3"/><feGaussianBlur stdDeviation="3"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/><feBlend mode="normal" in2="effect2_dropShadow_2180_29277" result="effect3_dropShadow_2180_29277"/><feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow_2180_29277" result="shape"/></filter><clipPath id="clip0_2180_29277"><rect width="262.5" height="200" fill="white" transform="translate(0.5)"/></clipPath><clipPath id="clip1_2180_29277"><rect width="183" height="96" fill="white" transform="translate(41 63)"/></clipPath><clipPath id="clip2_2180_29277"><rect width="183" height="136" fill="white" transform="translate(41 63)"/></clipPath><clipPath id="clip3_2180_29277"><rect width="4.57589" height="4.57589" fill="white" transform="translate(200.932 33.958)"/></clipPath></defs></svg>',
			'camera' => '<svg width="28" height="28" viewBox="0 0 28 28" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M10.5007 3.5L8.36565 5.83333H4.66732C3.38398 5.83333 2.33398 6.88333 2.33398 8.16667V22.1667C2.33398 23.45 3.38398 24.5 4.66732 24.5H23.334C24.6173 24.5 25.6673 23.45 25.6673 22.1667V8.16667C25.6673 6.88333 24.6173 5.83333 23.334 5.83333H19.6357L17.5007 3.5H10.5007ZM14.0007 21C10.7807 21 8.16732 18.3867 8.16732 15.1667C8.16732 11.9467 10.7807 9.33333 14.0007 9.33333C17.2207 9.33333 19.834 11.9467 19.834 15.1667C19.834 18.3867 17.2207 21 14.0007 21Z" fill="#0096CC"/><path d="M14.0007 19.8333L15.459 16.625L18.6673 15.1667L15.459 13.7083L14.0007 10.5L12.5423 13.7083L9.33398 15.1667L12.5423 16.625L14.0007 19.8333Z" fill="#0096CC"/></svg>',
			'uploadFile' => '<svg width="12" height="14" viewBox="0 0 12 14" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M7.33268 0.333008H1.99935C1.26602 0.333008 0.672682 0.933008 0.672682 1.66634L0.666016 12.333C0.666016 13.0663 1.25935 13.6663 1.99268 13.6663H9.99935C10.7327 13.6663 11.3327 13.0663 11.3327 12.333V4.33301L7.33268 0.333008ZM9.99935 12.333H1.99935V1.66634H6.66602V4.99967H9.99935V12.333ZM3.33268 9.00634L4.27268 9.94634L5.33268 8.89301V11.6663H6.66602V8.89301L7.72602 9.95301L8.66602 9.00634L6.00602 6.33301L3.33268 9.00634Z" fill="#141B38"/></svg>',
			'addRoundIcon' =>'<svg width="16" height="16" viewBox="0 0 16 16" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M1.33333 8C1.33333 6.13333 2.4 4.53333 4 3.8V2.33333C1.66667 3.2 0 5.4 0 8C0 10.6 1.66667 12.8 4 13.6667V12.2C2.4 11.4667 1.33333 9.86667 1.33333 8ZM10 2C6.66667 2 4 4.66667 4 8C4 11.3333 6.66667 14 10 14C13.3333 14 16 11.3333 16 8C16 4.66667 13.3333 2 10 2ZM13.3333 8.66667H10.6667V11.3333H9.33333V8.66667H6.66667V7.33333H9.33333V4.66667H10.6667V7.33333H13.3333V8.66667Z" fill="#0068A0"/></svg>',
			'loaderSVG'    => '<svg version="1.1" id="loader-1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" x="0px" y="0px" width="20px" height="20px" viewBox="0 0 50 50" style="enable-background:new 0 0 50 50;" xml:space="preserve"><path fill="#fff" d="M43.935,25.145c0-10.318-8.364-18.683-18.683-18.683c-10.318,0-18.683,8.365-18.683,18.683h4.068c0-8.071,6.543-14.615,14.615-14.615c8.072,0,14.615,6.543,14.615,14.615H43.935z"><animateTransform attributeType="xml" attributeName="transform" type="rotate" from="0 25 25" to="360 25 25" dur="0.6s" repeatCount="indefinite"/></path></svg>',
			'checkmarkSVG' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><!-- Font Awesome Pro 5.15.4 by @fontawesome - https://fontawesome.com License - https://fontawesome.com/license (Commercial License) --><path d="M504 256c0 136.967-111.033 248-248 248S8 392.967 8 256 119.033 8 256 8s248 111.033 248 248zM227.314 387.314l184-184c6.248-6.248 6.248-16.379 0-22.627l-22.627-22.627c-6.248-6.249-16.379-6.249-22.628 0L216 308.118l-70.059-70.059c-6.248-6.248-16.379-6.248-22.628 0l-22.627 22.627c-6.248 6.248-6.248 16.379 0 22.627l104 104c6.249 6.249 16.379 6.249 22.628.001z"/></svg>',
			'timesSVG' => '<svg  width="14" height="14" fill="none" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><!-- Font Awesome Pro 5.15.4 by @fontawesome - https://fontawesome.com License - https://fontawesome.com/license (Commercial License) --><path fill="#fff" d="M256 8C119 8 8 119 8 256s111 248 248 248 248-111 248-248S393 8 256 8zm121.6 313.1c4.7 4.7 4.7 12.3 0 17L338 377.6c-4.7 4.7-12.3 4.7-17 0L256 312l-65.1 65.6c-4.7 4.7-12.3 4.7-17 0L134.4 338c-4.7-4.7-4.7-12.3 0-17l65.6-65-65.6-65.1c-4.7-4.7-4.7-12.3 0-17l39.6-39.6c4.7-4.7 12.3-4.7 17 0l65 65.7 65.1-65.6c4.7-4.7 12.3-4.7 17 0l39.6 39.6c4.7 4.7 4.7 12.3 0 17L312 256l65.6 65.1z"/></svg>',
			'times2SVG'		=> '<svg width="10" height="10" viewBox="0 0 10 10" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M9.66671 1.27334L8.72671 0.333344L5.00004 4.06001L1.27337 0.333344L0.333374 1.27334L4.06004 5.00001L0.333374 8.72668L1.27337 9.66668L5.00004 5.94001L8.72671 9.66668L9.66671 8.72668L5.94004 5.00001L9.66671 1.27334Z" fill="#841919"></path></svg>',
			'chevronRight' => '<svg width="7" height="10" viewBox="0 0 7 10" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M1.3332 0L0.158203 1.175L3.97487 5L0.158203 8.825L1.3332 10L6.3332 5L1.3332 0Z" fill="#0068A0"></path></svg>',
			'blockEditorSBILogo' => '<svg width="86" height="83" viewBox="0 0 86 83" fill="none" xmlns="http://www.w3.org/2000/svg"><rect x="1.49414" y="4.43494" width="65.6329" height="65.6329" rx="12" transform="rotate(-3 1.49414 4.43494)" fill="white"/><rect x="1.49414" y="4.43494" width="65.6329" height="65.6329" rx="12" transform="rotate(-3 1.49414 4.43494)" stroke="#CED0D9" stroke-width="2.5003"/><path d="M54.7434 60.4452C54.7434 68.7888 59.7264 75.931 66.5096 77.1715L66.1136 78.4257L65.7119 79.6978L67.0411 79.5848L71.5286 79.2031L73.2178 79.0594L72.1443 77.7473L71.5197 76.9839C77.8707 75.3291 82.445 68.4326 82.445 60.4452C82.445 51.2906 76.4134 43.5316 68.5956 43.5316C60.7779 43.5316 54.7434 51.2904 54.7434 60.4452Z" fill="#FE544F" stroke="white" stroke-width="1.78661"/><path fill-rule="evenodd" clip-rule="evenodd" d="M71.685 50.0573L72.3086 56.491L78.7693 56.6766L74.0957 61.0221L77.787 66.3611L71.5684 65.1921L69.6832 71.4108L66.8207 65.842L61.041 68.4904L63.2644 62.5204L57.6274 59.6772L63.6616 57.7366L61.9965 51.7922L67.7053 55.0468L71.685 50.0573Z" fill="white"/><path d="M35.6888 25.5681C31.0072 25.8135 27.4759 29.8128 27.7174 34.4213C27.9628 39.1029 31.8889 42.6381 36.5706 42.3928C41.1791 42.1512 44.7874 38.2212 44.542 33.5396C44.3005 28.9311 40.2973 25.3266 35.6888 25.5681ZM36.4172 39.4667C33.4181 39.6239 30.8044 37.3403 30.6434 34.268C30.4862 31.2688 32.7737 28.7283 35.846 28.5673C38.8452 28.4101 41.3857 30.6976 41.5429 33.6967C41.7039 36.7691 39.4164 39.3095 36.4172 39.4667ZM46.3535 24.7158C46.296 23.6185 45.3722 22.7867 44.2749 22.8443C43.1777 22.9018 42.3459 23.8256 42.4034 24.9228C42.4609 26.0201 43.3847 26.8519 44.4819 26.7944C45.5792 26.7369 46.411 25.8131 46.3535 24.7158ZM52.0164 26.3995C51.7321 23.7738 51.0242 21.4636 49.0227 19.6614C47.0211 17.8591 44.6496 17.3966 42.0085 17.3883C39.2943 17.3839 31.1745 17.8094 28.4756 18.0975C25.8499 18.3819 23.6129 19.0859 21.7375 21.0913C19.9353 23.0929 19.4727 25.4644 19.4644 28.1055C19.46 30.8197 19.8855 38.9394 20.1737 41.6383C20.458 44.2641 21.162 46.5011 23.1674 48.3765C25.2422 50.1749 27.5405 50.6413 30.1816 50.6495C32.8959 50.654 41.0156 50.2285 43.7145 49.9403C46.3402 49.656 48.6504 48.9481 50.4526 46.9465C52.251 44.8718 52.7174 42.5735 52.7257 39.9324C52.7301 37.2181 52.3046 29.0984 52.0164 26.3995ZM49.364 42.9693C48.9286 44.4591 47.8157 45.6177 46.4565 46.2757C44.308 47.2686 39.1759 47.3175 36.8351 47.4401C34.4211 47.5667 29.3121 48.0545 27.1447 47.2878C25.651 46.7793 24.4963 45.7395 23.8344 44.3072C22.8454 42.2318 22.7965 37.0998 22.67 34.6858C22.5473 32.345 22.0595 27.236 22.8223 24.9955C23.3347 23.5749 24.3744 22.4202 25.8068 21.7583C27.8821 20.7693 33.0142 20.7204 35.4282 20.5939C37.769 20.4712 42.878 19.9834 45.1185 20.7462C46.5352 21.1854 47.6938 22.2983 48.3519 23.6575C49.3447 25.806 49.3936 30.9381 49.5163 33.2789C49.6428 35.6928 50.1306 40.8019 49.364 42.9693Z" fill="url(#paint0_linear_3281_44284)"/><defs><linearGradient id="paint0_linear_3281_44284" x1="33.8101" y1="80.5555" x2="99.1424" y2="6.47942" gradientUnits="userSpaceOnUse"><stop stop-color="white"/><stop offset="0.147864" stop-color="#F6640E"/><stop offset="0.443974" stop-color="#BA03A7"/><stop offset="0.733337" stop-color="#6A01B9"/><stop offset="1" stop-color="#6B01B9"/></linearGradient></defs></svg>'
		);
		return $builder_svg_icons;
	}


	public static function sb_other_plugins_modal() {
		check_ajax_referer( 'sbi_nonce' , 'sbi_nonce');

		if ( ! current_user_can( 'activate_plugins' ) || ! current_user_can( 'install_plugins' ) ) {
			wp_send_json_error();
		}

		$plugin = isset( $_POST['plugin'] ) ? sanitize_key( $_POST['plugin'] ) : '';
		$sb_other_plugins = self::install_plugins_popup();
		$plugin = isset( $sb_other_plugins[ $plugin ] ) ? $sb_other_plugins[ $plugin ] : false;
		if ( ! $plugin ) {
			wp_send_json_error();
		}

		// Build the content for modals
		$output = '<div class="sbi-fb-source-popup sbi-fb-popup-inside sbi-install-plugin-modal">
		<div class="sbi-fb-popup-cls"><svg width="14" height="14" viewBox="0 0 14 14" fill="none" xmlns="http://www.w3.org/2000/svg">
		<path d="M14 1.41L12.59 0L7 5.59L1.41 0L0 1.41L5.59 7L0 12.59L1.41 14L7 8.41L12.59 14L14 12.59L8.41 7L14 1.41Z" fill="#141B38"></path>
		</svg></div>
		<div class="sbi-install-plugin-body sbi-fb-fs">
		<div class="sbi-install-plugin-header">
		<div class="sb-plugin-image">'. $plugin['svgIcon'] .'</div>
		<div class="sb-plugin-name">
		<h3>'. $plugin['name'] .'<span>Free</span></h3>
		<p><span class="sb-author-logo">
		<svg width="13" height="17" viewBox="0 0 13 17" fill="none" xmlns="http://www.w3.org/2000/svg">
		<path fill-rule="evenodd" clip-rule="evenodd" d="M5.72226 4.70098C4.60111 4.19717 3.43332 3.44477 2.34321 3.09454C2.73052 4.01824 3.05742 5.00234 3.3957 5.97507C2.72098 6.48209 1.93286 6.8757 1.17991 7.30453C1.82065 7.93788 2.72809 8.3045 3.45109 8.85558C2.87196 9.57021 1.73414 10.3129 1.45689 10.9606C2.65579 10.8103 4.05285 10.5668 5.16832 10.5174C5.41343 11.7495 5.53984 13.1002 5.88845 14.2288C6.40758 12.7353 6.87695 11.192 7.49488 9.79727C8.44849 10.1917 9.61069 10.6726 10.5416 10.9052C9.88842 9.98881 9.29237 9.01536 8.71356 8.02465C9.57007 7.40396 10.4364 6.79309 11.2617 6.14122C10.0952 6.03375 8.88647 5.96834 7.66107 5.91968C7.46633 4.65567 7.5175 3.14579 7.21791 1.98667C6.76462 2.93671 6.2297 3.80508 5.72226 4.70098ZM6.27621 15.1705C6.12214 15.8299 6.62974 16.1004 6.55318 16.5C6.052 16.3273 5.67498 16.2386 5.00213 16.3338C5.02318 15.8194 5.48587 15.7466 5.3899 15.1151C-1.78016 14.3 -1.79456 1.34382 5.3345 0.546422C14.2483 -0.450627 14.528 14.9414 6.27621 15.1705Z" fill="#FE544F"></path><path fill-rule="evenodd" clip-rule="evenodd" d="M7.21769 1.98657C7.51728 3.1457 7.46611 4.65557 7.66084 5.91955C8.88625 5.96824 10.0949 6.03362 11.2615 6.14113C10.4362 6.79299 9.56984 7.40386 8.71334 8.02454C9.29215 9.01527 9.8882 9.98869 10.5414 10.9051C9.61046 10.6725 8.44827 10.1916 7.49466 9.79716C6.87673 11.1919 6.40736 12.7352 5.88823 14.2287C5.53962 13.1001 5.41321 11.7494 5.16809 10.5173C4.05262 10.5667 2.65558 10.8102 1.45666 10.9605C1.73392 10.3128 2.87174 9.57012 3.45087 8.85547C2.72786 8.30438 1.82043 7.93778 1.17969 7.30443C1.93264 6.8756 2.72074 6.482 3.39547 5.97494C3.05719 5.00224 2.73031 4.01814 2.34299 3.09445C3.43308 3.44467 4.60089 4.19707 5.72204 4.70088C6.22947 3.80499 6.7644 2.93662 7.21769 1.98657Z" fill="white"></path>
		</svg>
		</span>
		<span class="sb-author-name">'. $plugin['author'] .'</span>
		</p></div></div>
		<div class="sbi-install-plugin-content">
		<p>'. $plugin['description'] .'</p>';

		$plugin_install_data = array(
			'step' => 'install',
			'action' => 'sbi_install_addon',
			'nonce' => wp_create_nonce('sbi-admin'),
			'plugin' => $plugin['plugin'],
			'download_plugin' => $plugin['download_plugin'],
		);

		if ( ! $plugin['installed'] ) {
			$output .= sprintf(
				"<button class='sbi-install-plugin-btn sbi-btn-orange' id='sbi_install_op_btn' data-plugin-atts='%s'>%s</button></div></div></div>",
				sbi_json_encode( $plugin_install_data ),
				__('Install', 'instagram-feed')
			);
		}
		if ( $plugin['installed'] && ! $plugin['activated'] ) {
			$plugin_install_data['step'] = 'activate';
			$plugin_install_data['action'] = 'sbi_activate_addon';
			$output .= sprintf(
				"<button class='sbi-install-plugin-btn sbi-btn-orange' id='sbi_install_op_btn' data-plugin-atts='%s'>%s</button></div></div></div>",
				sbi_json_encode( $plugin_install_data ),
				__('Activate', 'instagram-feed')
			);
		}
		if ( $plugin['installed'] && $plugin['activated'] ) {
			$output .= sprintf(
				"<button class='sbi-install-plugin-btn sbi-btn-orange' id='sbi_install_op_btn' disabled='disabled'>%s</button></div></div></div>",
				__('Plugin installed & activated', 'instagram-feed')
			);
		}
		wp_send_json_success( $output );
		wp_die();
	}


	/**
	 * Plugins information for plugin install modal in all feeds page on select source flow
	 *
	 * @since 6.0
	 *
	 * @return array
	 */
	public static function install_plugins_popup() {
		$active_sb_plugins = Util::get_sb_active_plugins_info();

		return array(
			'facebook' => array(
				'displayName'         => __( 'Facebook', 'instagram-feed' ),
				'name'                => __( 'Facebook Feed', 'instagram-feed' ),
				'author'              => __( 'By Smash Balloon', 'instagram-feed' ),
				'description'         => __( 'To display a Facebook feed, our Facebook plugin is required. </br> It provides a clean and beautiful way to add your Facebook posts to your website. Grab your visitors attention and keep them engaged with your site longer.', 'instagram-feed' ),
				'dashboard_permalink' => admin_url( 'admin.php?page=cff-feed-builder' ),
				'svgIcon'             => '<svg viewBox="0 0 14 15"  width="36" height="36"><path d="M7.00016 0.860001C3.3335 0.860001 0.333496 3.85333 0.333496 7.54C0.333496 10.8733 2.7735 13.64 5.96016 14.14V9.47333H4.26683V7.54H5.96016V6.06667C5.96016 4.39333 6.9535 3.47333 8.48016 3.47333C9.20683 3.47333 9.96683 3.6 9.96683 3.6V5.24667H9.12683C8.30016 5.24667 8.04016 5.76 8.04016 6.28667V7.54H9.8935L9.5935 9.47333H8.04016V14.14C9.61112 13.8919 11.0416 13.0903 12.0734 11.88C13.1053 10.6697 13.6704 9.13043 13.6668 7.54C13.6668 3.85333 10.6668 0.860001 7.00016 0.860001Z" fill="rgb(0, 107, 250)"/></svg>',
				'installed'           => $active_sb_plugins['is_facebook_installed'],
				'activated'           => is_plugin_active( $active_sb_plugins['facebook_plugin'] ),
				'plugin'              => $active_sb_plugins['facebook_plugin'],
				'download_plugin'     => 'https://downloads.wordpress.org/plugin/custom-facebook-feed.zip',
			),
			'twitter'  => array(
				'displayName'         => __( 'Twitter', 'instagram-feed' ),
				'name'                => __( 'Twitter Feed', 'instagram-feed' ),
				'author'              => __( 'By Smash Balloon', 'instagram-feed' ),
				'description'         => __( 'Custom Twitter Feeds is a highly customizable way to display tweets from your Twitter account. Promote your latest content and update your site content automatically.', 'instagram-feed' ),
				'dashboard_permalink' => admin_url( 'admin.php?page=ctf-feed-builder' ),
				'svgIcon'             => '<svg width="36" height="36" viewBox="0 0 36 36" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M33.6905 9C32.5355 9.525 31.2905 9.87 30.0005 10.035C31.3205 9.24 32.3405 7.98 32.8205 6.465C31.5755 7.215 30.1955 7.74 28.7405 8.04C27.5555 6.75 25.8905 6 26.0005 6C20.4755 6 17.5955 8.88 17.5955 12.435C17.5955 12.945 17.6555 13.44 17.7605 13.905C12.4205 13.635 7.66555 11.07 4.50055 7.185C3.94555 8.13 3.63055 9.24 3.63055 10.41C3.63055 12.645 4.75555 14.625 6.49555 15.75C5.43055 15.75 4.44055 15.45 3.57055 15V15.045C3.57055 18.165 5.79055 20.775 8.73055 21.36C7.78664 21.6183 6.79569 21.6543 5.83555 21.465C6.24296 22.7437 7.04085 23.8626 8.11707 24.6644C9.19329 25.4662 10.4937 25.9105 11.8355 25.935C9.56099 27.7357 6.74154 28.709 3.84055 28.695C3.33055 28.695 2.82055 28.665 2.31055 28.605C5.16055 30.435 8.55055 31.5 12.1805 31.5C26.0005 31.5 30.4955 21.69 30.4955 13.185C30.4955 12.9 30.4955 12.63 30.4805 12.345C31.7405 11.445 32.8205 10.305 33.6905 9Z" fill="#1B90EF"/></svg>',
				'installed'           => $active_sb_plugins['is_twitter_installed'],
				'activated'           => is_plugin_active( $active_sb_plugins['twitter_plugin'] ),
				'plugin'              => $active_sb_plugins['twitter_plugin'],
				'download_plugin'     => 'https://downloads.wordpress.org/plugin/custom-twitter-feeds.zip',
			),
			'youtube'  => array(
				'displayName'         => __( 'YouTube', 'instagram-feed' ),
				'name'                => __( 'Feeds for YouTube', 'instagram-feed' ),
				'author'              => __( 'By Smash Balloon', 'instagram-feed' ),
				'description'         => __( 'To display a YouTube feed, our YouTube plugin is required. It provides a simple yet powerful way to display videos from YouTube on your website, Increasing engagement with your channel while keeping visitors on your website.', 'instagram-feed' ),
				'dashboard_permalink' => admin_url( 'admin.php?page=sby-feed-builder' ),
				'svgIcon'             => '<svg width="36" height="36" viewBox="0 0 36 36" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M15 22.5L22.785 18L15 13.5V22.5ZM32.34 10.755C32.535 11.46 32.67 12.405 32.76 13.605C32.865 14.805 32.91 15.84 32.91 16.74L33 18C33 21.285 32.76 23.7 32.34 25.245C31.965 26.595 31.095 27.465 29.745 27.84C29.04 28.035 27.75 28.17 25.77 28.26C23.82 28.365 22.035 28.41 20.385 28.41L18 28.5C11.715 28.5 7.8 28.26 6.255 27.84C4.905 27.465 6.035 26.595 3.66 25.245C3.465 24.54 3.33 23.595 3.24 22.395C3.135 21.195 3.09 20.16 3.09 19.26L3 18C3 14.715 3.24 12.3 3.66 10.755C6.035 9.405 4.905 8.535 6.255 8.16C6.96 7.965 8.25 7.83 10.23 7.74C12.18 7.635 13.965 7.59 15.615 7.59L18 7.5C24.285 7.5 28.2 7.74 29.745 8.16C31.095 8.535 31.965 9.405 32.34 10.755Z" fill="#EB2121"/></svg>',
				'installed'           => $active_sb_plugins['is_youtube_installed'],
				'activated'           => is_plugin_active( $active_sb_plugins['youtube_plugin'] ),
				'plugin'              => $active_sb_plugins['youtube_plugin'],
				'download_plugin'     => 'https://downloads.wordpress.org/plugin/feeds-for-youtube.zip',
			),
		);
	}

	/**
	 * Gets a list of info
	 * Used in multiple places in the feed creator
	 * Other Platforms + Social Links
	 * Upgrade links
	 *
	 * @return array
	 *
	 * @since 6.0
	 */
	public static function get_smashballoon_info() {
		$smash_info = array(
			'colorSchemes'   => array(
				'facebook'  => '#006BFA',
				'twitter'   => '#1B90EF',
				'instagram' => '#BA03A7',
				'youtube'   => '#EB2121',
				'linkedin'  => '#007bb6',
				'mail'      => '#666',
				'smash'     => '#EB2121',
			),
			'upgrade'        => array(
				'name' => __( 'Upgrade to Pro', 'instagram-feed' ),
				'icon' => 'instagram',
				'link' => 'https://smashballoon.com/instagram-feed/',
			),
			'platforms'      => array(
				array(
					'name' => __( 'Facebook Feed', 'instagram-feed' ),
					'icon' => 'facebook',
					'link' => 'https://smashballoon.com/instagram-feed/?utm_campaign=instagram-pro&utm_source=balloon&utm_medium=instagram',
				),
				array(
					'name' => __( 'Twitter Feed', 'instagram-feed' ),
					'icon' => 'twitter',
					'link' => 'https://smashballoon.com/custom-twitter-feeds/?utm_campaign=instagram-pro&utm_source=balloon&utm_medium=twitter',
				),
				array(
					'name' => __( 'YouTube Feed', 'instagram-feed' ),
					'icon' => 'youtube',
					'link' => 'https://smashballoon.com/youtube-feed/?utm_campaign=instagram-pro&utm_source=balloon&utm_medium=youtube',
				),
				array(
					'name' => __( 'Social Wall Plugin', 'instagram-feed' ),
					'icon' => 'smash',
					'link' => 'https://smashballoon.com/social-wall/?utm_campaign=instagram-pro&utm_source=balloon&utm_medium=social-wall ',
				),
			),
			'socialProfiles' => array(
				'facebook' => 'https://www.facebook.com/SmashBalloon/',
				'twitter'   => 'https://twitter.com/smashballoon',
			),
			'morePlatforms'  => array( 'instagram', 'youtube', 'twitter' ),
		);

		return $smash_info;
	}

	/**
	 * Text specific to onboarding. Will return an associative array 'active' => false
	 * if onboarding has been dismissed for the user or there aren't any legacy feeds.
	 *
	 * @return array
	 *
	 * @since 4.0
	 */
	public function get_onboarding_text() {
		// TODO: return if no legacy feeds
		$sbi_statuses_option = get_option( 'sbi_statuses', array() );

		if ( ! isset( $sbi_statuses_option['legacy_onboarding'] ) ) {
			return array( 'active' => false );
		}

		if ( $sbi_statuses_option['legacy_onboarding']['active'] === false
			 || self::onboarding_status() === 'dismissed' ) {
			return array( 'active' => false );
		}

		$type = $sbi_statuses_option['legacy_onboarding']['type'];

		$text = array(
			'active'      => true,
			'type'        => $type,
			'legacyFeeds' => array(
				'heading'     => __( 'Legacy Feed Settings', 'instagram-feed' ),
				'description' => sprintf( __( 'These settings will impact %1$s legacy feeds on your site. You can learn more about what legacy feeds are and how they differ from new feeds %2$shere%3$s.', 'instagram-feed' ), '<span class="cff-fb-count-placeholder"></span>', '<a href="https://smashballoon.com/doc/facebook-legacy-feeds/" target="_blank" rel="noopener">', '</a>' ),
			),
			'getStarted'  => __( 'You can now create and customize feeds individually. Click "Add New" to get started.', 'instagram-feed' ),
		);

		if ( $type === 'single' ) {
			$text['tooltips'] = array(
				array(
					'step'    => 1,
					'heading' => __( 'How you create a feed has changed', 'instagram-feed' ),
					'p'       => __( 'You can now create and customize feeds individually without using shortcode options.', 'instagram-feed' ) . ' ' . __( 'Click "Add New" to get started.', 'instagram-feed' ),
					'pointer' => 'top',
				),
				array(
					'step'    => 2,
					'heading' => __( 'Your existing feed is here', 'instagram-feed' ),
					'p'       => __( 'You can edit your existing feed from here, and all changes will only apply to this feed.', 'instagram-feed' ),
					'pointer' => 'top',
				),
			);
		} else {
			$text['tooltips'] = array(
				array(
					'step'    => 1,
					'heading' => __( 'How you create a feed has changed', 'instagram-feed' ),
					'p'       => __( 'You can now create and customize feeds individually without using shortcode options.', 'instagram-feed' ) . ' ' . __( 'Click "Add New" to get started.', 'instagram-feed' ),
					'pointer' => 'top',
				),
				array(
					'step'    => 2,
					'heading' => __( 'Your existing feeds are under "Legacy" feeds', 'instagram-feed' ),
					'p'       => __( 'You can edit the settings for any existing "legacy" feed (i.e. any feed created prior to this update) here.', 'instagram-feed' ) . ' ' . __( 'This works just like the old settings page and affects all legacy feeds on your site.', 'instagram-feed' ),
				),
				array(
					'step'    => 3,
					'heading' => __( 'Existing feeds work as normal', 'instagram-feed' ),
					'p'       => __( 'You don\'t need to update or change any of your existing feeds. They will continue to work as usual.', 'instagram-feed' ) . ' ' . __( 'This update only affects how new feeds are created and customized.', 'instagram-feed' ),
				),
			);
		}

		return $text;
	}

	public function get_customizer_onboarding_text() {

		if ( self::onboarding_status( 'customizer' ) === 'dismissed' ) {
			return array( 'active' => false );
		}

		$text = array(
			'active'   => true,
			'type'     => 'customizer',
			'tooltips' => array(
				array(
					'step'    => 1,
					'heading' => __( 'Embedding a Feed', 'instagram-feed' ),
					'p'       => __( 'After you are done customizing the feed, click here to add it to a page or a widget.', 'instagram-feed' ),
					'pointer' => 'top',
				),
				array(
					'step'    => 2,
					'heading' => __( 'Customize', 'instagram-feed' ),
					'p'       => __( 'Change your feed layout, color scheme, or customize individual feed sections here.', 'instagram-feed' ),
					'pointer' => 'top',
				),
				array(
					'step'    => 3,
					'heading' => __( 'Settings', 'instagram-feed' ),
					'p'       => __( 'Update your feed source, filter your posts, or change advanced settings here.', 'instagram-feed' ),
					'pointer' => 'top',
				),
			),
		);

		return $text;
	}

	/**
	 * Text related to the feed customizer
	 *
	 * @return array
	 *
	 * @since 6.0
	 */
	public function get_customize_screens_text() {
		$text = array(
			'common'              => array(
				'preview'       => __( 'Preview', 'instagram-feed' ),
				'help'          => __( 'Help', 'instagram-feed' ),
				'embed'         => __( 'Embed', 'instagram-feed' ),
				'save'          => __( 'Save', 'instagram-feed' ),
				'sections'      => __( 'Sections', 'instagram-feed' ),
				'enable'        => __( 'Enable', 'instagram-feed' ),
				'background'    => __( 'Background', 'instagram-feed' ),
				'text'          => __( 'Text', 'instagram-feed' ),
				'inherit'       => __( 'Inherit from Theme', 'instagram-feed' ),
				'size'          => __( 'Size', 'instagram-feed' ),
				'color'         => __( 'Color', 'instagram-feed' ),
				'height'        => __( 'Height', 'instagram-feed' ),
				'placeholder'   => __( 'Placeholder', 'instagram-feed' ),
				'select'        => __( 'Select', 'instagram-feed' ),
				'enterText'     => __( 'Enter Text', 'instagram-feed' ),
				'hoverState'    => __( 'Hover State', 'instagram-feed' ),
				'sourceCombine' => __( 'Combine sources from multiple platforms using our Social Wall plugin', 'instagram-feed' ),
			),

			'tabs'                => array(
				'customize' => __( 'Customize', 'instagram-feed' ),
				'settings'  => __( 'Settings', 'instagram-feed' ),
			),
			'overview'            => array(
				'feedLayout'  => __( 'Feed Layout', 'instagram-feed' ),
				'colorScheme' => __( 'Color Scheme', 'instagram-feed' ),
				'header'      => __( 'Header', 'instagram-feed' ),
				'posts'       => __( 'Posts', 'instagram-feed' ),
				'likeBox'     => __( 'Like Box', 'instagram-feed' ),
				'loadMore'    => __( 'Load More Button', 'instagram-feed' ),
			),
			'feedLayoutScreen'    => array(
				'layout'     => __( 'Layout', 'instagram-feed' ),
				'list'       => __( 'List', 'instagram-feed' ),
				'grid'       => __( 'Grid', 'instagram-feed' ),
				'masonry'    => __( 'Masonry', 'instagram-feed' ),
				'carousel'   => __( 'Carousel', 'instagram-feed' ),
				'feedHeight' => __( 'Feed Height', 'instagram-feed' ),
				'number'     => __( 'Number of Posts', 'instagram-feed' ),
				'columns'    => __( 'Columns', 'instagram-feed' ),
				'desktop'    => __( 'Desktop', 'instagram-feed' ),
				'tablet'     => __( 'Tablet', 'instagram-feed' ),
				'mobile'     => __( 'Mobile', 'instagram-feed' ),
				'bottomArea' => array(
					'heading'     => __( 'Tweak Post Styles', 'instagram-feed' ),
					'description' => __( 'Change post background, border radius, shadow etc.', 'instagram-feed' ),
				),
			),
			'colorSchemeScreen'   => array(
				'scheme'        => __( 'Scheme', 'instagram-feed' ),
				'light'         => __( 'Light', 'instagram-feed' ),
				'dark'          => __( 'Dark', 'instagram-feed' ),
				'custom'        => __( 'Custom', 'instagram-feed' ),
				'customPalette' => __( 'Custom Palette', 'instagram-feed' ),
				'background2'   => __( 'Background 2', 'instagram-feed' ),
				'text2'         => __( 'Text 2', 'instagram-feed' ),
				'link'          => __( 'Link', 'instagram-feed' ),
				'bottomArea'    => array(
					'heading'     => __( 'Overrides', 'instagram-feed' ),
					'description' => __( 'Colors that have been overridden from individual post element settings will not change. To change them, you will have to reset overrides.', 'instagram-feed' ),
					'ctaButton'   => __( 'Reset Overrides.', 'instagram-feed' ),
				),
			),
			'headerScreen'        => array(
				'headerType'     => __( 'Header Type', 'instagram-feed' ),
				'visual'         => __( 'Visual', 'instagram-feed' ),
				'coverPhoto'     => __( 'Cover Photo', 'instagram-feed' ),
				'nameAndAvatar'  => __( 'Name and avatar', 'instagram-feed' ),
				'about'          => __( 'About (bio and Likes)', 'instagram-feed' ),
				'displayOutside' => __( 'Display outside scrollable area', 'instagram-feed' ),
				'icon'           => __( 'Icon', 'instagram-feed' ),
				'iconImage'      => __( 'Icon Image', 'instagram-feed' ),
				'iconColor'      => __( 'Icon Color', 'instagram-feed' ),
			),
			// all Lightbox in common
			// all Load More in common
			'likeBoxScreen'       => array(
				'small'                     => __( 'Small', 'instagram-feed' ),
				'large'                     => __( 'Large', 'instagram-feed' ),
				'coverPhoto'                => __( 'Cover Photo', 'instagram-feed' ),
				'customWidth'               => __( 'Custom Width', 'instagram-feed' ),
				'defaultSetTo'              => __( 'By default, it is set to auto', 'instagram-feed' ),
				'width'                     => __( 'Width', 'instagram-feed' ),
				'customCTA'                 => __( 'Custom CTA', 'instagram-feed' ),
				'customCTADescription'      => __( 'This toggles the custom CTA like "Show now" and "Contact"', 'instagram-feed' ),
				'showFans'                  => __( 'Show Fans', 'instagram-feed' ),
				'showFansDescription'       => __( 'Show visitors which of their friends follow your page', 'instagram-feed' ),
				'displayOutside'            => __( 'Display outside scrollable area', 'instagram-feed' ),
				'displayOutsideDescription' => __( 'Make the like box fixed by moving it outside the scrollable area', 'instagram-feed' ),
			),
			'postsScreen'         => array(
				'thumbnail'           => __( 'Thumbnail', 'instagram-feed' ),
				'half'                => __( 'Half width', 'instagram-feed' ),
				'full'                => __( 'Full width', 'instagram-feed' ),
				'useFull'             => __( 'Use full width layout when post width is less than 500px', 'instagram-feed' ),
				'postStyle'           => __( 'Post Style', 'instagram-feed' ),
				'editIndividual'      => __( 'Edit Individual Elements', 'instagram-feed' ),
				'individual'          => array(
					'description'                => __( 'Hide or show individual elements of a post or edit their options', 'instagram-feed' ),
					'name'                       => __( 'Name', 'instagram-feed' ),
					'edit'                       => __( 'Edit', 'instagram-feed' ),
					'postAuthor'                 => __( 'Post Author', 'instagram-feed' ),
					'postText'                   => __( 'Post Text', 'instagram-feed' ),
					'date'                       => __( 'Date', 'instagram-feed' ),
					'photosVideos'               => __( 'Photos/Videos', 'instagram-feed' ),
					'likesShares'                => __( 'Likes, Shares and Comments', 'instagram-feed' ),
					'eventTitle'                 => __( 'Event Title', 'instagram-feed' ),
					'eventDetails'               => __( 'Event Details', 'instagram-feed' ),
					'postAction'                 => __( 'Post Action Links', 'instagram-feed' ),
					'sharedPostText'             => __( 'Shared Post Text', 'instagram-feed' ),
					'sharedLinkBox'              => __( 'Shared Link Box', 'instagram-feed' ),
					'postTextDescription'        => __( 'The main text of the Instagram post', 'instagram-feed' ),
					'maxTextLength'              => __( 'Maximum Text Length', 'instagram-feed' ),
					'characters'                 => __( 'Characters', 'instagram-feed' ),
					'linkText'                   => __( 'Link text to Instagram post', 'instagram-feed' ),
					'postDateDescription'        => __( 'The date of the post', 'instagram-feed' ),
					'format'                     => __( 'Format', 'instagram-feed' ),
					'custom'                     => __( 'Custom', 'instagram-feed' ),
					'learnMoreFormats'           => '<a href="https://smashballoon.com/doc/date-formatting-reference/" target="_blank" rel="noopener">' . __( 'Learn more about custom formats', 'instagram-feed' ) . '</a>',
					'addTextBefore'              => __( 'Add text before date', 'instagram-feed' ),
					'addTextBeforeEG'            => __( 'E.g. Posted', 'instagram-feed' ),
					'addTextAfter'               => __( 'Add text after date', 'instagram-feed' ),
					'addTextAfterEG'             => __( 'E.g. - posted date', 'instagram-feed' ),
					'timezone'                   => __( 'Timezone', 'instagram-feed' ),
					'tzDescription'              => __( 'Timezone settings are global across all feeds. To update it use the global settings.', 'instagram-feed' ),
					'tzCTAText'                  => __( 'Go to Global Settings', 'instagram-feed' ),
					'photosVideosDescription'    => __( 'Any photos or videos in your posts', 'instagram-feed' ),
					'useOnlyOne'                 => __( 'Use only one image per post', 'instagram-feed' ),
					'postActionLinksDescription' => __( 'The "View on Instagram" and "Share" links at the bottom of each post', 'instagram-feed' ),
					'viewOnFBLink'               => __( 'View on Instagram link', 'instagram-feed' ),
					'viewOnFBLinkDescription'    => __( 'Toggle "View on Instagram" link below each post', 'instagram-feed' ),
					'customizeText'              => __( 'Customize Text', 'instagram-feed' ),
					'shareLink'                  => __( 'Share Link', 'instagram-feed' ),
					'shareLinkDescription'       => __( 'Toggle "Share" link below each post', 'instagram-feed' ),
					'likesSharesDescription'     => __( 'The comments box displayed at the bottom of each timeline post', 'instagram-feed' ),
					'iconTheme'                  => __( 'Icon Theme', 'instagram-feed' ),
					'auto'                       => __( 'Auto', 'instagram-feed' ),
					'light'                      => __( 'Light', 'instagram-feed' ),
					'dark'                       => __( 'Dark', 'instagram-feed' ),
					'expandComments'             => __( 'Expand comments box by default', 'instagram-feed' ),
					'hideComment'                => __( 'Hide comment avatars', 'instagram-feed' ),
					'showLightbox'               => __( 'Show comments in lightbox', 'instagram-feed' ),
					'eventTitleDescription'      => __( 'The title of an event', 'instagram-feed' ),
					'eventDetailsDescription'    => __( 'The information associated with an event', 'instagram-feed' ),
					'textSize'                   => __( 'Text Size', 'instagram-feed' ),
					'textColor'                  => __( 'Text Color', 'instagram-feed' ),
					'sharedLinkBoxDescription'   => __( "The link info box that's created when a link is shared in a Instagram post", 'instagram-feed' ),
					'boxStyle'                   => __( 'Box Style', 'instagram-feed' ),
					'removeBackground'           => __( 'Remove background/border', 'instagram-feed' ),
					'linkTitle'                  => __( 'Link Title', 'instagram-feed' ),
					'linkURL'                    => __( 'Link URL', 'instagram-feed' ),
					'linkDescription'            => __( 'Link Description', 'instagram-feed' ),
					'chars'                      => __( 'chars', 'instagram-feed' ),
					'sharedPostDescription'      => __( 'The description text associated with shared photos, videos, or links', 'instagram-feed' ),
				),
				'postType'            => __( 'Post Type', 'instagram-feed' ),
				'boxed'               => __( 'boxed', 'instagram-feed' ),
				'regular'             => __( 'Regular', 'instagram-feed' ),
				'indvidualProperties' => __( 'Indvidual Properties', 'instagram-feed' ),
				'backgroundColor'     => __( 'Background Color', 'instagram-feed' ),
				'borderRadius'        => __( 'Border Radius', 'instagram-feed' ),
				'boxShadow'           => __( 'Box Shadow', 'instagram-feed' ),
			),
			'shoppableFeedScreen' => array(
				'heading1'     => __( 'Make your Instagram Feed Shoppable', 'instagram-feed' ),
				'headingRenew'     => __( 'Renew your license and make your Instagram Feed Shoppable', 'instagram-feed' ),
				'headingActivate'     => __( 'Activate your license and make your Instagram Feed Shoppable', 'instagram-feed' ),
				'description1' => __( 'This feature links the post to the one specificed in your caption.<br/><br/>Don’t want to add links to the caption? You can add links manually to each post.<br/><br/>Enable it to get started.', 'instagram-feed' ),
				'descriptionRenew' => __( 'This feature links the post to the one specified in your caption.<br/><br/>Don’t want to add links to the caption? You can add links manually to each post.<br/><br><br>', 'instagram-feed' ),
				'heading2'     => __( 'Tap “Add” or “Update” on an<br/>image to add/update it’s URL', 'instagram-feed' ),

			),
		);

		$text['onboarding'] = $this->get_customizer_onboarding_text();

		return $text;
	}

	/**
	 * Returns an associate array of all existing sources along with their data
	 *
	 * @param int $page
	 *
	 * @return array
	 *
	 * @since 6.0
	 */

	public static function get_source_list( $page = 1 ) {
		$args['page'] = $page;
		$source_data  = SBI_Db::source_query( $args );

		$encryption   = new \InstagramFeed\SB_Instagram_Data_Encryption();

		$return = array();
		foreach ( $source_data as $source ) {
			$info                  = ! empty( $source['info'] ) ? json_decode( $encryption->decrypt( $source['info'] ), true ) : array();
			$source['header_data'] = $info;

			$settings = array( 'gdpr' => 'no' );

			$avatar = \SB_Instagram_Parse::get_avatar( $info, $settings );

			if ( \SB_Instagram_Connected_Account::local_avatar_exists( $source['username'] ) ) {
				$source['local_avatar_url'] = \SB_Instagram_Connected_Account::get_local_avatar_url( $source['username'] );
				$source['local_avatar']     = \SB_Instagram_Connected_Account::get_local_avatar_url( $source['username'] );
			} else {
				$source['local_avatar'] = false;
			}

			$source['avatar_url']       = $avatar;
			$source['just_added']       = ( ! empty( $_GET['sbi_username'] ) && isset( $info['username'] ) && $info['username'] === $_GET['sbi_username'] );
			$source['error_encryption'] = false;
			if ( isset( $source['access_token'] ) && strpos( $source['access_token'], 'IG' ) === false && strpos( $source['access_token'], 'EA' ) === false && ! $encryption->decrypt( $source['access_token'] ) ) {
				$source['error_encryption'] = true;
			}

			$return[] = $source;
		}

		return $return;
	}

	/**
	 * Check if the account source type is business
	 *
	 * @since 2.0
	 *
	 */
	public static function is_business_source() {
		$source_list   = self::get_source_list();
		$business_type = false;

		foreach( $source_list as $source ) {
			if ( isset( $source['account_type'] ) && $source['account_type'] === 'business' ) {
				$business_type = true;
			}
			if ( isset( $source['type'] ) && $source['type'] === 'business' ) {
				$business_type = true;
			}
		}

		return $business_type;
	}

	/**
	 * Get Links with UTM
	 *
	 * @return array
	 *
	 * @since 4.0
	 */
	public static function get_links_with_utm() {
		$license_key = null;
		if ( get_option( 'sbi_license_key' ) ) {
			$license_key = get_option( 'sbi_license_key' );
		}
		$all_access_bundle       = sprintf( 'https://smashballoon.com/all-access/?edd_license_key=%s&upgrade=true&utm_campaign=instagram-pro&utm_source=all-feeds&utm_medium=footer-banner&utm_content=learn-more', sanitize_key( $license_key ) );
		$all_access_bundle_popup = sprintf( 'https://smashballoon.com/all-access/?edd_license_key=%s&upgrade=true&utm_campaign=instagram-pro&utm_source=balloon&utm_medium=all-access', sanitize_key( $license_key ) );
		$sourceCombineCTA        = sprintf( 'https://smashballoon.com/social-wall/?edd_license_key=%s&upgrade=true&utm_campaign=instagram-pro&utm_source=customizer&utm_medium=sources&utm_content=social-wall', sanitize_key( $license_key ) );

		return array(
			'allAccessBundle'  => $all_access_bundle,
			'popup'            => array(
				'allAccessBundle' => $all_access_bundle_popup,
				'fbProfile'       => 'https://www.facebook.com/SmashBalloon/',
				'twitterProfile'  => 'https://twitter.com/smashballoon',
			),
			'sourceCombineCTA' => $sourceCombineCTA,
			'multifeedCTA'     => 'https://smashballoon.com/extensions/multifeed/?utm_campaign=instagram-pro&utm_source=customizer&utm_medium=sources&utm_content=multifeed',
			'doc'              => 'https://smashballoon.com/docs/instagram/?utm_campaign=instagram-pro&utm_source=support&utm_medium=view-documentation-button&utm_content=view-documentation',
			'blog'             => 'https://smashballoon.com/blog/?utm_campaign=instagram-pro&utm_source=support&utm_medium=view-blog-button&utm_content=view-blog',
			'gettingStarted'   => 'https://smashballoon.com/docs/getting-started/?instagram&utm_campaign=instagram-pro&utm_source=support&utm_medium=getting-started-button&utm_content=getting-started',
		);
	}

	public static function get_social_wall_links() {
		return array(
			'<a href="' . 