<?php

namespace memberpress\quizzes\models;

use memberpress\courses as courses;

if (!defined('ABSPATH')) {
    die('You are not allowed to call this page directly.');
}

class Quiz extends courses\models\Lesson
{
    public static $cpt                       = 'mpcs-quiz';
    public static $permalink_slug            = 'quizzes';
    public static $nonce_str                 = 'mpcs-quiz-nonce';
    public static $require_passing_score_str = '_mpcs_lesson_require_passing_score';
    public static $passing_score_unit_str    = '_mpcs_lesson_passing_score_unit';
    public static $passing_score_str         = '_mpcs_lesson_passing_score';
    public static $failed_message_str        = '_mpcs_lesson_failed_message';
    public static $allow_retakes_str         = '_mpcs_lesson_allow_retakes';
    public static $retake_limit_str          = '_mpcs_lesson_retake_limit';
    public static $retake_method_str         = '_mpcs_lesson_retake_method';

    /**
     * Quiz constructor.
     *
     * @param null $obj The object to initialize the model with.
     */
    public function __construct($obj = null)
    {
        parent::__construct($obj);
        $this->load_cpt(
            $obj,
            self::$cpt,
            [
                'status'                => [
                    'default' => 'enabled',
                    'type'    => 'string',
                ],
                'section_id'            => [
                    'default' => 0,
                    'type'    => 'integer',
                ],
                'lesson_order'          => [
                    'default' => 0,
                    'type'    => 'integer',
                ],
                'require_passing_score' => [
                    'default' => false,
                    'type'    => 'boolean',
                ],
                'passing_score_unit'    => [
                    'default' => 'percent',
                    'type'    => 'string',
                ],
                'passing_score'         => [
                    'default' => 1,
                    'type'    => 'integer',
                ],
                'failed_message'        => [
                    'default' => esc_html__('You did not meet the passing score. Please try again.', 'memberpress-course-quizzes'),
                    'type'    => 'string',
                ],
                'allow_retakes'         => [
                    'default' => false,
                    'type'    => 'boolean',
                ],
                'retake_limit'          => [
                    'default' => 1,
                    'type'    => 'integer',
                ],
                'retake_method'         => [
                    'default' => 'latest',
                    'type'    => 'string',
                ],
                'randomise_questions' => [
                    'default' => false,
                    'type' => 'bool'
                ]
            ]
        );
    }

    /**
     * Get the key for a quiz attribute
     *
     * @param string $attr The attribute to get the key for.
     *
     * @return string
     */
    public function get_attr_key($attr)
    {
        return '_mpcs_lesson_' . $attr;
    }

  /**
   * Get the array of questions within the content of this quiz
   *
   * @param bool $ids_only Get only the question IDs rather than the question instances.
   *
   * @return Question[]|int[]
   */
    public function get_questions($ids_only = false)
    {
        if (!function_exists('parse_blocks')) {
            return [];
        }

        $questions = [];
        $blocks    = parse_blocks($this->post_content);

        $this->extract_questions_from_blocks($blocks, $questions, $ids_only);

        return $questions;
    }

    /**
     * Extract questions from blocks
     *
     * @param array $blocks The blocks to extract questions from.
     * @param array $questions The array to store the questions in.
     * @param bool  $ids_only Get only the question IDs rather than the question instances.
     */
    public function extract_questions_from_blocks($blocks, &$questions, $ids_only)
    {
        foreach ($blocks as $block) {
            if (isset($block['blockName']) && strpos($block['blockName'], 'memberpress-courses') === 0) {
                $id = isset($block['attrs']['questionId']) ? (int) $block['attrs']['questionId'] : 0;

                if ($id > 0) {
                    if ($ids_only) {
                        $questions[] = $id;
                    } else {
                        $question = new Question($id);

                        if ($question->id > 0 && $question->type !== 'placeholder') {
                            $questions[] = $question;
                        }
                    }
                }
            }

            if (isset($block['innerBlocks']) && is_array($block['innerBlocks'])) {
                $this->extract_questions_from_blocks($block['innerBlocks'], $questions, $ids_only);
            }
        }
    }

  /**
   * Does this quiz have any completed attempts?
   *
   * @return bool
   */
    public function has_attempts()
    {
        global $wpdb;
        $db = courses\lib\Db::fetch();

        $table = $db->attempts;
        $where = $wpdb->prepare(" WHERE quiz_id = %d AND status IN ('complete', 'pending')", $this->ID);
        $query = "SELECT COUNT(*) FROM {$table}{$where}";
        $count = $wpdb->get_var($query); // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared

        return $count > 0;
    }

    /**
     * Check if quiz can be retaken
     *
     * @param Attempt $attempt The attempt to check.
     * @return bool
     */
    public function grant_retakes($attempt)
    {
        if (false === $this->allow_retakes) {
            return false;
        }

        if ($attempt->attempts > $this->retake_limit) {
            return false;
        }

        return true;
    }

    /**
     * Get the admin view URL for this quiz
     *
     * @return string
     */
    public function get_admin_view_url()
    {
        if (courses\helpers\App::is_gradebook_addon_active()) {
            $url = admin_url('admin.php?page=memberpress-course-gradebook');
        } else {
            $url = add_query_arg(['id' => $this->ID], admin_url('admin.php?page=mpcs-quiz-attempts'));
        }
        return $url;
    }
}
