<?php

namespace memberpress\quizzes\models;

use memberpress\courses as courses;

if (!defined('ABSPATH')) {
    die('You are not allowed to call this page directly.');
}

/**
 * Attempt model
 *
 * @property int $id The attempt ID
 * @property int $quiz_id The quiz ID
 * @property int $user_id The user ID
 * @property int $points_awarded The total number of points awarded
 * @property int $points_possible The total number of points possible
 * @property int $bonus_points The total number of bonus points
 * @property int $score The score percentage
 * @property string $status The attempt status, 'draft' or 'complete'
 * @property string $feedback Grader's feedback
 * @property string $started_at Datetime in MySQL format
 * @property string $finished_at Datetime in MySQL format
 */
class Attempt extends courses\lib\BaseModel
{
    public static $draft_str    = 'draft';
    public static $pending_str  = 'pending'; // failed status.
    public static $complete_str = 'complete';

    /**
     * Attempt constructor
     *
     * @param mixed $obj The object to initialize the model with.
     */
    public function __construct($obj = null)
    {
        $this->initialize(
            [
                'id'              => [
                    'default' => 0,
                    'type'    => 'integer',
                ],
                'quiz_id'         => [
                    'default' => 0,
                    'type'    => 'integer',
                ],
                'user_id'         => [
                    'default' => 0,
                    'type'    => 'integer',
                ],
                'points_awarded'  => [
                    'default' => 0,
                    'type'    => 'integer',
                ],
                'points_possible' => [
                    'default' => 0,
                    'type'    => 'integer',
                ],
                'bonus_points'    => [
                    'default' => 0,
                    'type'    => 'integer',
                ],
                'score'           => [
                    'default' => 0,
                    'type'    => 'integer',
                ],
                'feedback'        => [
                    'default' => '',
                    'type'    => 'string',
                ],
                'status'          => [
                    'default' => self::$draft_str,
                    'type'    => 'string',
                ],
                'attempts'        => [
                    'default' => 0,
                    'type'    => 'integer',
                ],
                'started_at'      => [
                    'default' => null,
                    'type'    => 'datetime',
                ],
                'finished_at'     => [
                    'default' => null,
                    'type'    => 'datetime',
                ],
            ],
            $obj
        );
    }

    /**
     * Store this attempt
     *
     * @param bool $validate Whether to validate the model before storing.
     * @return int|\WP_Error|false The attempt ID, a WP_Error if validation fails, or false on failure.
     */
    public function store($validate = true)
    {
        if ($validate) {
            try {
                $this->validate();
            } catch (courses\lib\ValidationException $e) {
                return new \WP_Error(get_class($e), $e->getMessage());
            }
        }

        $db    = courses\lib\Db::fetch();
        $attrs = $this->get_values();

        if (isset($this->id) && $this->id > 0) {
            $db->update_record($db->attempts, $this->id, $attrs);
        } else {
            $this->id = $db->create_record($db->attempts, $attrs, false);
        }

        return $this->id;
    }

    /**
     * Destroy this attempt
     *
     * @return int|false The number of affected rows or false if there was an error
     */
    public function destroy()
    {
        $db     = courses\lib\Db::fetch();
        $result = $db->delete_records($db->attempts, ['id' => $this->id]);

        if ($result) {
            // only delete answers if attempt deletion succeeded.
            $db->delete_records($db->answers, ['attempt_id' => $this->id]);

            // delete the UserProgress record too.
            $user_progress = courses\models\UserProgress::find_one_by_user_and_lesson($this->user_id, $this->quiz_id);

            if (!empty($user_progress) && !empty($user_progress->id)) {
                $user_progress->destroy();
            }
        }

        return $result;
    }

    /**
     * Get the quiz associated with this attempt
     *
     * @return Quiz|false
     */
    public function quiz()
    {
        return Quiz::find($this->quiz_id);
    }

    /**
     * Get the user associated with this attempt
     *
     * @return \WP_User|false
     */
    public function user()
    {
        return get_user_by('id', $this->user_id);
    }

    /**
     * Is this attempt complete?
     *
     * @return bool
     */
    public function is_complete()
    {
        return $this->status === self::$complete_str;
    }

    /**
     * Is this attempt a draft?
     *
     * @return bool
     */
    public function is_draft()
    {
        return $this->status === self::$draft_str;
    }

    /**
     * Is this attempt a pending?
     *
     * @return bool
     */
    public function is_pending()
    {
        return $this->status === self::$pending_str;
    }

    /**
     * Get the score for this attempt formatted as "Score: #/# #%"
     *
     * @return string
     */
    public function get_score()
    {
        $score = sprintf(
            /* translators: %1$s: points awarded, %2$s: points possible, %3$s: score percent, %%: literal percent sign */
            __('Score: %1$s/%2$s (%3$s%%)', 'memberpress-course-quizzes'),
            $this->points_awarded + $this->bonus_points,
            $this->points_possible,
            $this->score
        );

        return apply_filters('mpcs_attempt_score', $score, $this);
    }

    /**
     * Get the score for this attempt formatted as "Score: #%"
     *
     * @return string
     */
    public function get_score_percent()
    {
        $score = sprintf(
            /* translators: %s: score percent, %%: literal percent sign */
            __('Score: %s%%', 'memberpress-course-quizzes'),
            $this->score
        );

        return apply_filters('mpcs_attempt_score_percent', $score, $this);
    }

    /**
     * Validate the attempt object
     *
     * @return bool
     * @throws courses\lib\ValidationException If validation fails.
     */
    public function validate()
    {
        $statuses = [self::$draft_str, self::$pending_str, self::$complete_str];

        courses\lib\Validate::is_numeric($this->id, 0, null, __('Id', 'memberpress-course-quizzes'));
        courses\lib\Validate::is_numeric($this->quiz_id, 0, null, __('Quiz Id', 'memberpress-course-quizzes'));
        courses\lib\Validate::is_numeric($this->user_id, 0, null, __('User Id', 'memberpress-course-quizzes'));
        courses\lib\Validate::is_numeric($this->points_awarded, 0, null, __('Points Awarded', 'memberpress-course-quizzes'));
        courses\lib\Validate::is_numeric($this->points_possible, 0, null, __('Points Possible', 'memberpress-course-quizzes'));
        courses\lib\Validate::is_numeric($this->score, 0, null, __('Score', 'memberpress-course-quizzes'));
        courses\lib\Validate::is_in_array($this->status, $statuses, __('Status', 'memberpress-course-quizzes'));

        return true;
    }

    /**
     * Get the number of attempts for a user on a quiz
     *
     * @param string  $order_by The column to order by.
     * @param string  $order The order direction.
     * @param string  $paged The current page.
     * @param string  $search The search term.
     * @param integer $perpage The number of items per page.
     * @param integer $quiz_id  The quiz ID.
     * @return array
     */
    public static function list_table($order_by = '', $order = '', $paged = '', $search = '', $perpage = 10, $quiz_id = null)
    {
        global $wpdb;
        $db = courses\lib\Db::fetch();

        $cols = [
            'id'              => 'att.id',
            'quiz_id'         => 'att.quiz_id',
            'user_id'         => 'att.user_id',
            'user_login'      => 'usr.user_login',
            'user_email'      => 'usr.user_email',
            'first_name'      => 'um_first_name.meta_value',
            'last_name'       => 'um_last_name.meta_value',
            'name'            => "CASE WHEN um_first_name.meta_value IS NULL OR TRIM(um_first_name.meta_value) = '' OR um_last_name.meta_value IS NULL OR TRIM(um_last_name.meta_value) = '' THEN usr.user_login ELSE CONCAT_WS(' ', um_first_name.meta_value, um_last_name.meta_value) END",
            'points_awarded'  => 'att.points_awarded',
            'points_possible' => 'att.points_possible',
            'score'           => 'att.score',
            'attempts'        => 'att.attempts',
            'allow_retakes'   => 'pm_allow_retakes.meta_value',
            'retake_limit'    => 'pm_retake_limit.meta_value',
            'finished_at'     => 'att.finished_at',
        ];

        $search_cols = [
            'um_first_name.meta_value',
            'um_last_name.meta_value',
            'usr.user_login',
            'user_email' => 'usr.user_email',
            'att.score',
        ];

        $from = "{$db->attempts} AS att";

        $joins = [
            "LEFT JOIN {$wpdb->users} AS usr ON att.user_id = usr.ID",
            "LEFT JOIN {$wpdb->usermeta} AS um_first_name ON um_first_name.user_id = usr.ID AND um_first_name.meta_key = 'first_name'",
            "LEFT JOIN {$wpdb->usermeta} AS um_last_name ON um_last_name.user_id = usr.ID AND um_last_name.meta_key = 'last_name'",
            "LEFT JOIN {$wpdb->postmeta} AS pm_allow_retakes ON pm_allow_retakes.post_id = $quiz_id AND pm_allow_retakes.meta_key = '_mpcs_lesson_allow_retakes'",
            "LEFT JOIN {$wpdb->postmeta} AS pm_retake_limit ON pm_retake_limit.post_id = $quiz_id AND pm_retake_limit.meta_key = '_mpcs_lesson_retake_limit'",
        ];

        $args = [$wpdb->prepare('(att.status = %s OR att.status = %s)', self::$complete_str, self::$pending_str)];

        if (is_numeric($quiz_id)) {
            $args[] = $wpdb->prepare('att.quiz_id = %d', (int) $quiz_id);
        }

        return courses\lib\Db::list_table($cols, $from, $joins, $args, $order_by, $order, $paged, $search, $perpage, $search_cols);
    }


    /**
     * Checks if there is any gradeable questions and ungraded answers
     *
     * @return bool
     */
    public function requires_manual_grading()
    {
        $db = courses\lib\Db::fetch();

        $questions = $db->get_records($db->questions, [
            'quiz_id' => $this->quiz_id,
        ]);

        $gradable_question_types = array_filter($questions, function ($question) {
            return courses\helpers\App::is_gradebook_addon_active() && in_array($question->type, ['essay', 'short-answer'], true);
        });

        if (empty($gradable_question_types)) {
            return false;
        }

        // Check if there is any answer that's not graded yet.
        $answer = Answer::get_one([
            'attempt_id' => $this->id,
            'grader'     => 0,
            'graded_at'  => '0000-00-00 00:00:00',
        ]);

        if ($answer) {
            return true;
        }

        return false;
    }

    /**
     * Get attempts with ungraded answers
     *
     * @return array
     */
    public static function get_attempts_with_ungraded_answers()
    {
        global $wpdb;
        $db = courses\lib\Db::fetch();

        $sql = "
        SELECT DISTINCT att.id, quiz_id, p.post_title, u.user_login, u.user_email, um1.meta_value as first_name, um2.meta_value as last_name
        FROM {$db->attempts} AS att
        JOIN {$db->answers} AS ans ON att.id = ans.attempt_id
        JOIN {$wpdb->posts} AS p ON att.quiz_id = p.ID
        JOIN {$wpdb->users} AS u ON att.user_id = u.ID
        LEFT JOIN {$wpdb->usermeta} AS um1 ON u.ID = um1.user_id AND um1.meta_key = 'first_name'
        LEFT JOIN {$wpdb->usermeta} AS um2 ON u.ID = um2.user_id AND um2.meta_key = 'last_name'
        WHERE att.status = %s AND ans.grader = %d AND ans.graded_at = %s
        LIMIT 10
        ";

        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
        $query = $wpdb->prepare($sql, self::$complete_str, 0, '0000-00-00 00:00:00');
        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
        $attempts = $wpdb->get_results($query);

        return $attempts;
    }

    /**
     * Get latest attempts
     *
     * @return array
     */
    public static function get_latest_attempts()
    {
        global $wpdb;
        $db = courses\lib\Db::fetch();

        $sql = "
        SELECT att.id, quiz_id, p.post_title, u.user_login, u.user_email, um1.meta_value as first_name, um2.meta_value as last_name
        FROM {$db->attempts} AS att
        JOIN {$wpdb->posts} AS p ON att.quiz_id = p.ID
        JOIN {$wpdb->users} AS u ON att.user_id = u.ID
        LEFT JOIN {$wpdb->usermeta} AS um1 ON u.ID = um1.user_id AND um1.meta_key = 'first_name'
        LEFT JOIN {$wpdb->usermeta} AS um2 ON u.ID = um2.user_id AND um2.meta_key = 'last_name'
        WHERE att.status = %s
        LIMIT 10
        ";

        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
        $query = $wpdb->prepare($sql, self::$complete_str);
        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
        $attempts = $wpdb->get_results($query);

        return $attempts;
    }

    /**
     * Get answers for this attempt
     *
     * @return array
     */
    public function get_answers()
    {
        $db      = courses\lib\Db::fetch();
        $answers = $db->get_records($db->answers, ['attempt_id' => $this->id]);

        return $answers;
    }
}
