<?php

namespace memberpress\quizzes\controllers;

use memberpress\courses;
use memberpress\quizzes as base;
use memberpress\quizzes\lib;
use memberpress\quizzes\helpers;
use memberpress\quizzes\models;
use memberpress\quizzes\emails;

if (!defined('ABSPATH')) {
    die('You are not allowed to call this page directly.');
}

class Quizzes extends courses\lib\BaseCtrl
{
    /**
     * Load the hooks
     */
    public function load_hooks()
    {
        add_action('wp_enqueue_scripts', [$this, 'enqueue_scripts']);
        add_filter('template_include', [$this, 'override_template'], 999999);
        add_filter('the_content', [$this, 'append_quiz_navigation']);
        add_action('wp_ajax_mpcs_submit_quiz', [$this, 'handle_submit_quiz']);
        add_action('wp_ajax_mpcs_auto_save_question', [$this, 'handle_auto_save_question']);
        add_filter('mpcs_course_quizzes', [$this, 'quizzes_for_course'], 10, 4);
        add_filter('mpcs_classroom_lesson_buttons', [$this, 'add_classroom_lesson_buttons'], 10, 2);
        add_filter('mpcs_check_quiz_guest_view', [$this, 'is_logged_out_quiz'], 10, 2);
        add_action('mpcs_section_lesson_row', [$this, 'render_section_lesson_row'], 10, 6);
        add_action('mpcs_before_course_progress_summary', [$this, 'render_quizzes_progress_for_ca'], 10, 2);
        add_action('mpcs_profile_lesson_progress', [$this, 'render_profile_quiz'], 10, 3);
    }

    /**
     * Enqueue the scripts and styles
     */
    public function enqueue_scripts()
    {
        // phpcs:disable WordPress.WP.EnqueuedResourceParameters.NotInFooter
        global $post;
        if ($post instanceof \WP_Post && $post->post_type == models\Quiz::$cpt && is_single()) {
            $quiz = models\Quiz::find($post->ID);

            if ($quiz instanceof models\Quiz) {
                wp_enqueue_style('mpcs-quiz', base\CSS_URL . '/quiz.css', [], base\VERSION);
                wp_enqueue_script('jquery-scrollto', courses\JS_URL . '/vendor/jquery.scrollTo.min.js', ['jquery'], base\VERSION);
                wp_enqueue_script('sortablejs', courses\JS_URL . '/vendor/Sortable.min.js', ['jquery'], '1.15.1');
                wp_enqueue_script('mpcs-quiz', base\JS_URL . '/quiz.js', ['jquery'], base\VERSION);

                $l10n = [
                    'ajax_url'                 => admin_url('admin-ajax.php'),
                    'submit_quiz_nonce'        => wp_create_nonce('mpcs_submit_quiz'),
                    'auto_save_question_nonce' => wp_create_nonce('mpcs_auto_save_question'),
                    'post_id'                  => $quiz->ID,
                    'error_submitting_quiz'    => esc_html(
                        apply_filters(
                            'mpcs_quiz_error_submitting_quiz',
                            // translators: %s: the failed attempt message.
                            __('An error occurred submitting the quiz: %s', 'memberpress-course-quizzes'),
                            $quiz
                        )
                    ),
                    'attempt_complete'         => false,
                    'auto_save_enabled'        => apply_filters('mpcs_quiz_auto_save_enabled', true),
                    // translators: %d: the number of characters.
                    'character_count'          => esc_html__('%d characters', 'memberpress-course-quizzes'),
                    'scroll_enabled'           => apply_filters('mpcs_quiz_scroll_enabled', true),
                    'scroll_speed'             => apply_filters('mpcs_quiz_scroll_speed', 800),
                    'scroll_offset'            => apply_filters('mpcs_quiz_scroll_offset', -50),
                    'progress_nonce'           => wp_create_nonce('lesson_progress'),
                ];

                if (is_user_logged_in()) {
                    $is_quiz_available = $quiz->is_available();

                    if ($is_quiz_available) {
                        $attempt = models\Attempt::get_one([
                            'quiz_id' => $quiz->ID,
                            'user_id' => get_current_user_id(),
                        ]);

                        if (!$attempt instanceof models\Attempt) {
                            $attempt             = new models\Attempt();
                            $attempt->quiz_id    = $quiz->ID;
                            $attempt->user_id    = get_current_user_id();
                            $attempt->status     = models\Attempt::$draft_str;
                            $attempt->started_at = gmdate('Y-m-d H:i:s');
                            $attempt->store();
                        }

                        $l10n = array_merge($l10n, [
                            'attempt_id'       => $attempt->id,
                            'attempt_complete' => $attempt->is_complete(),
                            'attempt_score'    => apply_filters('mpcs_quiz_score_title', sprintf(
                                '<h4 class="mpcs-quiz-score">%s</h4>',
                                esc_html($attempt->get_score())
                            ), $attempt),
                            'retake_method'    => $quiz->retake_method,
                            'attempt_reset' => sprintf('<div class="mpcs-quiz-reset"><button type="button" id="mpcs-quiz-reset">%s</button></div>', esc_html('Reset Attempt')),
                        ]);

                        $retake_now = isset($_GET['retake']) && 'true' === $_GET['retake']; // phpcs:ignore WordPress.Security.NonceVerification.Recommended

                        // Show errors if there is no retake attempt and quiz allow retakes.
                        if (false == $retake_now || false == $quiz->grant_retakes($attempt)) {
                            if ($quiz->require_passing_score) {
                                $l10n['attempt_pending']       = $attempt->is_pending();
                                $l10n['require_passing_score'] = $quiz->require_passing_score;
                                $l10n['failed_message']        = $quiz->failed_message;
                            }

                            if ($quiz->allow_retakes) {
                                $permalink             = get_permalink($quiz->ID);
                                $retake_button         = lib\View::get_string('retake_button', get_defined_vars());
                                $l10n['retake_button'] = $retake_button;
                            }
                        }
                    }
                }
            }

            wp_localize_script(
                'mpcs-quiz',
                'MpcsQuizL10n',
                ['l10n_print_after' => 'MpcsQuizL10n = ' . wp_json_encode($l10n)]
            );
        }
        // phpcs:enable WordPress.WP.EnqueuedResourceParameters.NotInFooter
    }

    /**
     * Override default template with the quiz page template
     *
     * @param string $template current template.
     * @return string modified template
     */
    public static function override_template($template)
    {
        global $post;

        if ($post instanceof \WP_Post && $post->post_type == models\Quiz::$cpt && is_single()) {
            $quiz   = new models\Quiz($post->ID);
            $course = $quiz->course();

            if ($course instanceof courses\models\Course && courses\lib\Utils::user_can_view_course($course)) {
                $new_template = locate_template($course->page_template);
            } else {
                // If the course is not found, trigger a 404 error.
                global $wp_query;
                $wp_query->set_404();

                status_header(404);
                nocache_headers();

                return get_404_template();
            }

            if (courses\helpers\App::is_classroom()) {
                $template = lib\View::file('/classroom_courses_single_quiz');
            } elseif (isset($new_template) && !empty($new_template)) {
                return $new_template;
            } else {
                $located_template = locate_template(
                    [
                        'single-mpcs-quiz.php',
                        'single-mpcs-course.php',
                        'page.php',
                        'custom_template.php',
                        'single.php',
                        'index.php',
                    ]
                );

                if (!empty($located_template)) {
                    $template = $located_template;
                }
            }
        }

        return $template;
    }

    /**
     * Handle the quiz submission Ajax request
     */
    public function handle_submit_quiz()
    {
        // Check for a valid request.
        if (!courses\lib\Utils::is_post_request() || !isset($_POST['post_id']) || !is_numeric($_POST['post_id'])) {
            wp_send_json_error(__('Bad request', 'memberpress-course-quizzes'));
        }

        // Verify the nonce.
        if (!check_ajax_referer('mpcs_submit_quiz', false, false)) {
            wp_send_json_error(__('Security check failed', 'memberpress-course-quizzes'));
        }

        // Get POST values.
        $raw_values = wp_unslash($_POST);

        // Sanitize and validate quiz ID.
        $quiz_id = isset($raw_values['post_id']) && is_numeric($raw_values['post_id']) ? (int) $raw_values['post_id'] : 0;
        $quiz    = new models\Quiz($quiz_id);

        // Get current user.
        $user = courses\lib\Utils::get_currentuserinfo();
        if (!$user instanceof \WP_User) {
            wp_send_json_error(__('You must be logged in to submit a quiz', 'memberpress-course-quizzes'));
        }

        // Sanitize and validate attempt ID.
        $attempt_id = isset($raw_values['attempt_id']) && is_numeric($raw_values['attempt_id']) ? (int) $raw_values['attempt_id'] : 0;
        $attempt    = models\Attempt::find($attempt_id);

        // Call the method to process the quiz submission.
        $result = $this->process_quiz_submission($raw_values, $quiz, $attempt, $user);

        // Send the result as the AJAX response.
        if (is_wp_error($result)) {
            wp_send_json_error($result->get_error_message());
        }

        wp_send_json_success($result);
    }

    /**
     * Process the quiz submission and return the result.
     *
     * @param array          $values The form values submitted with the quiz.
     * @param models\Quiz    $quiz The quiz instance.
     * @param models\Attempt $attempt The attempt instance.
     * @param \WP_User       $user The user.
     * @return array|\WP_Error The result of the submission process or a WP_Error object.
     */
    public function process_quiz_submission($values, $quiz, $attempt, $user)
    {
        // Check for quiz existence.
        if (empty($quiz->ID)) {
            return new \WP_Error('quiz_not_found', __('Quiz not found', 'memberpress-course-quizzes'));
        }

        // Validate section and course.
        $section = $quiz->section();
        if (!$section instanceof courses\models\Section) {
            return new \WP_Error('section_not_found', __('Course section not found', 'memberpress-course-quizzes'));
        }

        $course = $section->course();
        if (!$course instanceof courses\models\Course) {
            return new \WP_Error('course_not_found', __('Course not found', 'memberpress-course-quizzes'));
        }

        // Retrieve and validate questions.
        $questions = $quiz->get_questions();
        $values    = $this->sanitize_quiz_values($questions, $values, $quiz);
        $errors    = $this->validate_quiz($questions, $values, $quiz);


        if (!empty($errors)) {
            wp_send_json_error(compact('errors'));
        }

        // Fetch attempt and validate.
        $old_attempt = (object)$attempt->get_values();

        if (!$attempt instanceof models\Attempt) {
            return new \WP_Error('attempt_not_found', __('Attempt not found', 'memberpress-course-quizzes'));
        }

        if ($attempt->user_id !== $user->ID) {
            return new \WP_Error('invalid_user', __('This attempt is for a different user', 'memberpress-course-quizzes'));
        }

        if ($attempt->quiz_id !== $quiz->ID) {
            return new \WP_Error('invalid_quiz', __('This attempt is for a different quiz', 'memberpress-course-quizzes'));
        }

        if ($quiz->allow_retakes && !$quiz->grant_retakes($attempt)) {
            return new \WP_Error('no_retakes', __('You cannot retake this quiz at the moment.', 'memberpress-course-quizzes'));
        }

        // Calculate scores.
        $now                   = gmdate('Y-m-d H:i:s');
        $total_points_possible = 0;
        $total_points_awarded  = 0;

        foreach ($questions as $question) {
            $total_points_possible += $question->get_points_possible();
            $total_points_awarded  += $question->get_score($values[$question->id]);
        }

        // Update attempt details.
        $attempt->status          = models\Attempt::$complete_str;
        $attempt->attempts        = ++$attempt->attempts;
        $attempt->points_awarded  = $total_points_awarded;
        $attempt->points_possible = $total_points_possible;
        $attempt->score           = $total_points_possible > 0 ? round(($total_points_awarded / $total_points_possible) * 100) : 0;
        $attempt->started_at      = empty($attempt->started_at) ? $now : $attempt->started_at;
        $attempt->finished_at     = $now;

        $result = $attempt->store();
        if ($result instanceof \WP_Error) {
            return $result;
        }

        // Handle retake method.
        if ('best' === $quiz->retake_method && $old_attempt->score > $attempt->score) {
            $attempt->points_awarded  = $old_attempt->points_awarded;
            $attempt->points_possible = $old_attempt->points_possible;
            $attempt->score           = $old_attempt->score;
        } else {
            foreach ($questions as $question) {
                models\Answer::insert_or_replace_answer(
                    $attempt->id,
                    $question->id,
                    $values[$question->id],
                    $question->get_points_possible(),
                    $question->get_score($values[$question->id]),
                    $question->can_be_manually_graded() ? 0 : $this->get_grader_user_id($quiz),
                    $now,
                    $question->can_be_manually_graded() ? '' : $now
                );
            }
        }

        // Send email notification.
        try {
            courses\lib\Utils::send_notices($attempt, null, emails\AdminQuizAttemptedEmail::class);
            courses\lib\Utils::send_notices($attempt, emails\UserQuizAttemptedEmail::class);
        } catch (\Throwable $t) {
            error_log('Error sending quiz attempt notifications: ' . $t->getMessage());
        }

        // Handle passing score requirement.
        if ($quiz->require_passing_score && ! helpers\Quizzes::meets_passing_score($quiz, $attempt)) {
            $attempt->status = models\Attempt::$pending_str;
            $attempt->store();

            $response = [
                'type'         => 'require_passing_score',
                'message'      => esc_html__('You did not meet the required passing score.', 'memberpress-course-quizzes'),
                'show_results' => $course->show_results == 'enabled',
            ];

            courses\lib\Utils::send_notices($attempt, null, emails\AdminQuizFailedEmail::class);

            if ($quiz->allow_retakes) {
                $response['retake_button'] = lib\View::get_string('retake_button', get_defined_vars());
            }

            if (!empty($quiz->failed_message)) {
                $response['message'] = $quiz->failed_message;
            }

            return new \WP_Error('failed_quiz', $response);
        }

        // Complete the lesson for the user if necessary.
        if (!courses\models\UserProgress::has_completed_lesson($user->ID, $quiz->ID)) {
            $quiz->complete($user->ID);
            $mepr_user = new \MeprUser($user->ID);
            \MeprEvent::record('mpca-quiz-attempt-completed', $mepr_user, [
                'attempt_id' => $attempt->id,
            ]);
        }

        // Return success response.
        return [
            'show_results' => $course->show_results == 'enabled',
        ];
    }

    /**
     * Validate the quiz submission
     *
     * @param  models\Question[] $questions The array of questions.
     * @param  array             $values    The submitted values.
     * @param  models\Quiz       $quiz      The quiz instance.
     * @return array
     */
    protected function sanitize_quiz_values($questions, $values, $quiz)
    {
        $sanitized = [];

        foreach ($questions as $question) {
            $key = "mpcs_quiz_question_{$question->id}";

            switch ($question->type) {
                case 'multiple-choice':
                case 'true-false':
                case 'short-answer':
                case 'likert-scale':
                default:
                    $sanitized[$question->id] = isset($values[$key]) && is_string($values[$key]) ? sanitize_text_field($values[$key]) : '';
                    break;
                case 'multiple-answer':
                case 'fill-blank':
                case 'sort-values':
                case 'match-matrix':
                    $sanitized[$question->id] = isset($values[$key]) && is_array($values[$key]) ? array_map('sanitize_text_field', $values[$key]) : [];
                    break;
                case 'essay':
                    $sanitized[$question->id] = isset($values[$key]) && is_string($values[$key]) ? sanitize_textarea_field($values[$key]) : '';
                    break;
            }
        }

        return apply_filters('mpcs_sanitize_quiz_values', $sanitized, $questions, $values, $quiz);
    }

    /**
     * Validate the quiz submission
     *
     * @param  models\Question[] $questions The array of questions.
     * @param  array             $values    The submitted values.
     * @param  models\Quiz       $quiz      The quiz instance.
     * @return array
     */
    protected function validate_quiz($questions, $values, $quiz)
    {
        $errors = [];

        foreach ($questions as $question) {
            if (!$question->required && $question->is_value_empty($values[$question->id])) {
                continue;
            } elseif ($question->required && $question->is_value_empty($values[$question->id])) {
                $errors[] = [
                    'id'      => $question->id,
                    'message' => __('This field is required', 'memberpress-course-quizzes'),
                ];
            } elseif ($question->type == 'essay') {
                $length     = mb_strlen($values[$question->id]);
                $min_length = isset($question->settings['min']) && is_numeric($question->settings['min']) && $question->settings['min'] > 0 ? (int) $question->settings['min'] : 1;
                $max_length = isset($question->settings['max']) && is_numeric($question->settings['max']) && $question->settings['max'] >= 0 ? (int) $question->settings['max'] : 0;

                if ($length < $min_length) {
                    $errors[] = [
                        'id'      => $question->id,
                        'message' => sprintf(
                            /* translators: %s: the number of characters */
                            _n(
                                'Please enter at least %s character',
                                'Please enter at least %s characters',
                                $min_length,
                                'memberpress-course-quizzes'
                            ),
                            number_format_i18n($min_length)
                        ),
                    ];
                } elseif ($max_length > 0 && $length > $max_length) {
                    $errors[] = [
                        'id'      => $question->id,
                        'message' => sprintf(
                            /* translators: %s: the number of characters */
                            _n(
                                'Please enter no more than %s character',
                                'Please enter no more than %s characters',
                                $max_length,
                                'memberpress-course-quizzes'
                            ),
                            number_format_i18n($max_length)
                        ),
                    ];
                }
            } elseif ($question->type == 'fill-blank') {
                if (apply_filters('mpcs_fill_blank_require_all', true)) {
                    $data    = helpers\Questions::get_fill_blank_data($question);
                    $answers = array_filter(array_map('strlen', $values[$question->id]));

                    if (count($answers) < count($data['answers'])) {
                        $errors[] = [
                            'id'      => $question->id,
                            'message' => __('All fields are required', 'memberpress-course-quizzes'),
                        ];
                    }
                }
            } elseif ($question->type == 'sort-values') {
                if (
                    apply_filters('mpcs_validate_sort_values', true, $question, $values) &&
                    is_array($values[$question->id]) &&
                    is_array($question->options)
                ) {
                    foreach ($values[$question->id] as $value) {
                        if (!in_array($value, $question->options, true)) {
                            $errors[] = [
                                'id'      => $question->id,
                                'message' => __('Invalid value given', 'memberpress-course-quizzes'),
                            ];

                            break;
                        }
                    }
                }
            } elseif ($question->type == 'match-matrix') {
                if (is_array($values[$question->id]) && is_array($question->answer)) {
                    $answers = array_filter(array_map('strlen', $values[$question->id]));

                    if (
                        apply_filters('mpcs_match_matrix_require_all', true, $question, $values) &&
                        count($answers) < count($question->answer)
                    ) {
                        $errors[] = [
                            'id'      => $question->id,
                            'message' => __('All fields are required', 'memberpress-course-quizzes'),
                        ];
                    } elseif (apply_filters('mpcs_validate_match_matrix_values', true)) {
                        foreach ($values[$question->id] as $value) {
                            if ($value !== '' && !in_array($value, $question->answer, true)) {
                                $errors[] = [
                                    'id'      => $question->id,
                                    'message' => __('Invalid value given', 'memberpress-course-quizzes'),
                                ];

                                break;
                            }
                        }
                    }
                }
            }
        }

        return apply_filters('mpcs_validate_quiz', $errors, $questions, $values, $quiz);
    }

    /**
     * Get the user ID of the grader
     *
     * Currently, it is set as the first admin user.
     *
     * @param models\Quiz $quiz The quiz instance.
     * @return int
     */
    protected function get_grader_user_id($quiz)
    {
        $grader_user_id = 0;

        $admins = get_users([
            'role'    => 'administrator',
            'number'  => 1,
            'fields'  => 'ID',
            'orderby' => 'ID',
            'order'   => 'ASC',
        ]);

        if (isset($admins[0])) {
            $grader_user_id = (int) $admins[0];
        }

        return (int) apply_filters('mpcs_get_grader_user_id', $grader_user_id, $quiz);
    }

    /**
     * Append the quiz navigation to the content
     *
     * @param string $content The post content.
     * @return string The modified post content.
     */
    public function append_quiz_navigation($content)
    {
        global $post;

        if ($post instanceof \WP_Post && $post->post_type == models\Quiz::$cpt && is_single() && is_user_logged_in()) {
            $current_user         = courses\lib\Utils::get_currentuserinfo();
            $current_lesson       = new courses\models\Lesson($post->ID);
            $lesson_nav_ids       = $current_lesson->nav_ids();
            $current_lesson_index = array_search($current_lesson->ID, $lesson_nav_ids);
            $current_section      = $current_lesson->section();
            $lesson_available     = $current_lesson->is_available();

            if ($current_section !== false && $lesson_available) {
                if (!courses\helpers\Lessons::has_next_lesson($current_lesson_index, $lesson_nav_ids)) {
                    $current_course        = $current_section->course();
                    $sections              = $current_course->sections();
                    $section_ids           = array_map(function ($section) {
                        return $section->id;
                    }, $sections);
                    $current_section_index = array_search($current_section->id, $section_ids);
                }

                $options     = get_option('mpcs-options');
                $attempt     = models\Attempt::get_one([
                    'quiz_id' => $post->ID,
                    'user_id' => get_current_user_id(),
                ]);
                $quiz        = new models\Quiz($post->ID);
                $retake_args = isset($_GET['retake']) ? wp_unslash(sanitize_text_field($_GET['retake'])) : false; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
                $show_submit = $retake_args === 'true' && $quiz->grant_retakes($attempt);

                ob_start();
                require(lib\View::file('/courses_navigation'));
                $nav_links = ob_get_clean();

                $content .= $nav_links;
            }
        }

        return $content;
    }

    /**
     * Handle the auto save question Ajax request
     */
    public function handle_auto_save_question()
    {
        if (
            !courses\lib\Utils::is_post_request() ||
            !isset($_POST['attempt_id'], $_POST['question_id']) ||
            !is_numeric($_POST['attempt_id']) ||
            !is_numeric($_POST['question_id'])
        ) {
            wp_send_json_error(__('Bad request', 'memberpress-course-quizzes'));
        }

        $user = courses\lib\Utils::get_currentuserinfo();

        if (!$user instanceof \WP_User) {
            wp_send_json_error(__('You must be logged in', 'memberpress-course-quizzes'));
        }

        if (!check_ajax_referer('mpcs_auto_save_question', false, false)) {
            wp_send_json_error(__('Security check failed', 'memberpress-course-quizzes'));
        }

        $values      = wp_unslash($_POST);
        $attempt_id  = (int) $values['attempt_id'];
        $question_id = (int) $values['question_id'];

        if ($attempt_id <= 0 || $question_id <= 0) {
            wp_send_json_error(__('Bad request', 'memberpress-course-quizzes'));
        }

        $attempt = models\Attempt::find($attempt_id);

        if (!$attempt instanceof models\Attempt) {
            wp_send_json_error(__('Attempt not found', 'memberpress-course-quizzes'));
        }

        if ($attempt->user_id !== get_current_user_id()) {
            wp_send_json_error(__('Bad request', 'memberpress-course-quizzes'));
        }

        $quiz = $attempt->quiz();

        if (!$quiz instanceof models\Quiz) {
            wp_send_json_error(__('Quiz not found', 'memberpress-course-quizzes'));
        }

        $question = models\Question::find($question_id);

        if (!$question instanceof models\Question) {
            wp_send_json_error(__('Question not found', 'memberpress-course-quizzes'));
        }

        $now    = gmdate('Y-m-d H:i:s');
        $values = $this->sanitize_quiz_values([$question], $values, $quiz);

        models\Answer::insert_or_replace_answer(
            $attempt->id,
            $question->id,
            $values[$question->id],
            0,
            0,
            0,
            $now,
            $now
        );

        wp_send_json_success();
    }

    /**
     * Get quizzes for a course
     *
     * @param array  $quizzes The quizzes array.
     * @param object $course The course object.
     * @param array  $sections The sections array.
     * @param string $type The type of return.
     * @return array
     */
    public function quizzes_for_course($quizzes, $course, $sections, $type)
    {
        foreach ($sections as $section) {
            $quizzes = array_merge(
                $quizzes,
                models\Quiz::find_all_by_section($section->id, [models\Quiz::$cpt])
            );
        }

        if ($type == 'ids') {
            return array_map(
                function ($quiz) {
                    return $quiz->ID;
                },
                $quizzes
            );
        } else {
            return $quizzes;
        }
    }

    /**
     * Add classroom lesson buttons
     *
     * @param string $nav_links The navigation links.
     * @param array  $data The data array.
     * @return string
     */
    public function add_classroom_lesson_buttons($nav_links, $data)
    {
        $post = isset($data['post']) ? $data['post'] : null;
        if ($post && $post->post_type == models\Quiz::$cpt && is_user_logged_in()) {
            $options               = isset($data['options']) ? $data['options'] : [];
            $lesson_nav_ids        = isset($data['lesson_nav_ids']) ? $data['lesson_nav_ids'] : [];
            $current_lesson_index  = isset($data['current_lesson_index']) ? $data['current_lesson_index'] : 0;
            $current_course        = isset($data['current_course']) ? $data['current_course'] : null;
            $current_section_index = isset($data['current_section_index']) ? $data['current_section_index'] : 0;
            $section_ids           = isset($data['section_ids']) ? $data['section_ids'] : [];

            $attempt = models\Attempt::get_one([
                'quiz_id' => $post->ID,
                'user_id' => get_current_user_id(),
            ]);
            $quiz    = new models\Quiz($post->ID);
            $retake  = isset($_GET['retake']) ? wp_unslash(sanitize_text_field($_GET['retake'])) : 'false'; // phpcs:ignore WordPress.Security.NonceVerification.Recommended

            \ob_start();
            require lib\View::file('/courses_classroom_buttons');
            $nav_links = \ob_get_clean();
        }

        return $nav_links;
    }

    /**
     * Render the section lesson row
     *
     * @param \WP_Post $lesson The lesson object.
     * @param bool     $lesson_available Whether the lesson is available.
     * @param bool     $has_completed_lesson Whether the lesson has been completed.
     * @param bool     $is_sidebar Whether the lesson is in the sidebar.
     * @param bool     $show_bookmark Whether to show the bookmark.
     * @param \WP_Post $next_lesson The next lesson object.
     * @return void
     */
    public function render_section_lesson_row($lesson, $lesson_available, $has_completed_lesson, $is_sidebar, $show_bookmark, $next_lesson)
    {
        if (models\Quiz::$cpt !== $lesson->post_type) {
            return;
        }

        $current_user_id = get_current_user_id();
        $attempt         = $lesson->post_type == models\Quiz::$cpt ? models\Attempt::get_one([
            'user_id' => $current_user_id,
            'quiz_id' => $lesson->ID,
        ]) : false;
        $has_failed_quiz = $attempt instanceof models\Attempt && $attempt->is_pending();

        if (courses\helpers\App::is_classroom()) {
            require lib\View::file('/classroom_section_quiz_row');
        } else {
            require lib\View::file('/section_quiz_row');
        }
    }

    /**
     * Render the quizzes progress for the course admin
     *
     * @param object   $course The course object.
     * @param \WP_User $user The user object.
     * @return void
     */
    public function render_quizzes_progress_for_ca($course, $user)
    {
        foreach ($course->quizzes() as $quiz) {
            if ($quiz->post_status == 'draft') {
                continue;
            }
            $has_completed_lesson = courses\models\UserProgress::has_completed_lesson($user->ID, $quiz->ID);
            $attempt              = $quiz->post_type == models\Quiz::$cpt ? models\Attempt::get_one([
                'user_id' => $user->ID,
                'quiz_id' => $quiz->ID,
            ]) : false;
            $score                = '-';
            if ($attempt instanceof models\Attempt && $has_completed_lesson) {
                $score = sprintf(
                    /* translators: %1$s: points awarded, %2$s: points possible, %3$s: score percent, %%: literal percent sign */
                    __('%1$s/%2$s (%3$s%%)', 'memberpress-course-quizzes'),
                    $attempt->points_awarded,
                    $attempt->points_possible,
                    $attempt->score
                );
            }
            ?>
            <div class="quiz-progress-summary-row">
                <div class="quiz-progress-summary-title"><a href="<?php echo esc_url(get_the_permalink($quiz->ID)); ?>"><?php echo esc_html($quiz->post_title); ?></a></div>
                <div class="quiz-progress-summary">
                    <?php echo esc_html($score); ?>
                </div>
            </div>
            </div>
        <?php }
    }

    /**
     * Render quiz progress on profile's "Course Information" section
     *
     * @param \WP_Post $quiz The quiz object.
     * @param \WP_Post $course The course object.
     * @param \WP_User $user The user object.
     * @return void
     */
    public function render_profile_quiz($quiz, $course, $user)
    {
        if ($quiz->post_type !== models\Quiz::$cpt) {
            return;
        }
        $has_completed_lesson = courses\models\UserProgress::has_completed_lesson($user->ID, $quiz->ID);
        $attempt              = $quiz->post_type == models\Quiz::$cpt ? models\Attempt::get_one([
            'user_id' => $user->ID,
            'quiz_id' => $quiz->ID,
        ]) : false;
        $score                = '-';
        $view_attempt         = esc_attr__('No Attempt', 'memberpress-course-quizzes');
        if ($attempt instanceof models\Attempt && $has_completed_lesson) {
            $score = sprintf(
                /* translators: %1$s: points awarded, %2$s: points possible, %3$s: score percent, %%: literal percent sign */
                __('%1$s/%2$s (%3$s%%)', 'memberpress-course-quizzes'),
                $attempt->points_awarded,
                $attempt->points_possible,
                $attempt->score
            );

            $score = apply_filters('mpcs_attempt_score', $score, $attempt);

            $view_attempt = sprintf(
                '<a href class="mpcs-quiz-attempt-view mpcs-quiz-attempt" data-id="%s">%s</a>',
                esc_attr($attempt->id),
                esc_attr__('View Attempt', 'memberpress-course-quizzes')
            );
        }
        ?>
        <tr>
            <th>
                <div class="mpca-quiz-progress-title"><a target="_blank" href="<?php echo esc_url(add_query_arg(['id' => $quiz->ID], admin_url('admin.php?page=mpcs-quiz-attempts'))); ?>"><?php echo esc_html($quiz->post_title); ?></a></div>
            </th>
            <td class="progress mpca-quiz-progress"><?php echo esc_html($score); ?></td>
            <td><?php echo wp_kses_post($view_attempt); ?>
            </td>
        </tr>
        <?php
    }

    /**
     * Check if the quiz is for logged out users
     *
     * @param bool     $result The result.
     * @param \WP_Post $post The post object.
     * @return bool
     */
    public function is_logged_out_quiz($result, $post)
    {
        $result = !is_user_logged_in() && $post->post_type == models\Quiz::$cpt;
        return $result;
    }
}
