<?php

namespace memberpress\quizzes\controllers;

if (!defined('ABSPATH')) {
    die('You are not allowed to call this page directly.');
}

use memberpress\courses;
use memberpress\quizzes\lib;
use memberpress\quizzes\emails;
use memberpress\quizzes\models;
use memberpress\quizzes\helpers;
use memberpress\quizzes as base;
use MeprAddonUpdates;

class App extends courses\lib\BaseCtrl
{
    /**
     * Load hooks for the plugin.
     */
    public function load_hooks()
    {
        add_filter('mpcs_models_namespace', [$this, 'models_namespace']);
        add_filter('mpcs_lesson_permalink_rules', [$this, 'permalink_rules'], 10, 3);
        add_action('in_admin_header', [$this, 'admin_header'], 0);
        add_filter('mpcs_dripping_post_types', [$this, 'course_dripping_post_types'], 10, 2);
        add_action('mpcs_register_addon_emails', [$this, 'register_emails']);
        add_filter('mpcs_course_email_vars', [$this, 'register_email_vars'], 10, 2);
        add_filter('mpcs_lesson_email_params', [$this, 'register_email_params'], 10, 3);
    }

    /**
     * Register the plugin emails.
     *
     * @param lib\EmailManager $email_manager The email manager object.
     */
    public function register_emails($email_manager)
    {
        $email_manager->register_email('admin_quiz_failed_email', emails\AdminQuizFailedEmail::class);
        $email_manager->register_email('admin_quiz_attempted_email', emails\AdminQuizAttemptedEmail::class);
        $email_manager->register_email('user_quiz_attempted_email', emails\UserQuizAttemptedEmail::class);

        if (courses\helpers\App::is_gradebook_addon_active()) {
            $email_manager->register_email('user_quiz_graded_email', emails\UserQuizGradedEmail::class);
        }
    }

    /**
     * Register the plugin email variables.
     *
     * @param array                 $email_vars The list of email variables.
     * @param courses\lib\BaseEmail $email The email object.
     * @return array
     */
    public function register_email_vars(array $email_vars, courses\lib\BaseEmail $email)
    {
        if (strpos($email->key, 'quiz') === false) {
            return $email_vars;
        }

        $email_vars = array_merge($email_vars, [
            'quiz_url'                    => 'URL to the quiz',
            'quiz_points_possible'        => 'Total points the quiz is worth',
            'quiz_points_scored'          => 'Points scored by the student on the quiz',
            'quiz_score_percentage'       => 'Quiz score as a percentage (points_scored + bonus_points) / points_possible',
            'quiz_bonus_points'           => 'Bonus points received on the quiz',
            'quiz_passing_score_required' => 'Is passing score required to pass the quiz?',
            'quiz_did_pass'               => 'Yes or No if the student passed the quiz',
            'quiz_name'                   => 'Name (post_title) of the quiz',
            'quiz_id'                     => 'ID (post_ID) of the quiz',
            'grade_quiz_url'              => 'URL for the admin to grade or view the quiz (Admin emails only)',
            'quiz_retakes_allowed'        => 'Yes or No if retakes are allowed',
            'quiz_retakes_remaining'      => 'Number of remaining retake attempts',
        ]);

        return $email_vars;
    }

    /**
     * Register the plugin email parameters.
     *
     * @param array  $email_params The list of email parameters.
     * @param object $quiz The object.
     * @param object $attempt The attempt object.
     * @return array
     */
    public function register_email_params($email_params, $quiz, $attempt)
    {
        if (! $quiz instanceof models\Quiz) {
            return $email_params;
        }

        if (!$attempt || $attempt->quiz_id !== $quiz->ID) {
            return $email_params;
        }

        $email_params = array_merge($email_params, [
            'quiz_url'                    => get_permalink($quiz->ID),
            'quiz_points_possible'        => $attempt->points_possible,
            'quiz_points_scored'          => $attempt->points_awarded,
            'quiz_score_percentage'       => $attempt->score,
            'quiz_bonus_points'           => $attempt->bonus_points,
            'quiz_passing_score_required' => $quiz->passing_score . ' ' . $quiz->passing_score_unit,
            'quiz_did_pass'               => helpers\Quizzes::meets_passing_score($quiz, $attempt) ? esc_html__('Yes', 'memberpress-course-quizzes') : esc_html__('No', 'memberpress-course-quizzes'),
            'quiz_name'                   => $quiz->post_title,
            'quiz_id'                     => $quiz->ID,
            'grade_quiz_url'              => $quiz->get_admin_view_url(),
            'quiz_retakes_allowed'        => $quiz->allow_retakes ? esc_html__('Yes', 'memberpress-course-quizzes') : esc_html__('No', 'memberpress-course-quizzes'),
            'quiz_retakes_remaining'      => $quiz->retake_limit - $attempt->attempts,
        ]);

        return $email_params;
    }

    /**
     * Enable updates for this add-on
     */
    public static function addon_updates()
    {
        if (class_exists('MeprAddonUpdates')) {
            new MeprAddonUpdates(
                base\EDITION,
                base\PLUGIN_SLUG,
                'mpcsq_license_key',
                'MemberPress Course Quizzes',
                'Quizzes add-on for MemberPress Courses.'
            );
        }
    }

    /**
     * Make sure the rewrite rules are flushed to prevent issues with accessing the custom post types.
     * All custom post types should be registered by now.
     *
     * @return void
     */
    public function maybe_flush_rewrite_rules()
    {
        if (empty(get_option('mepr_courses_flushed_rewrite_rules', ''))) {
            flush_rewrite_rules();
            update_option('mepr_courses_flushed_rewrite_rules', true);
        }
    }


    /**
     * Add rewrite rules for the assignments permalink.
     *
     * @param  array  $rules                The current rewrite rules.
     * @param  string $courses_cpt          The courses custom post type slug.
     * @param  string $courses_permalink_base The courses permalink base.
     *
     * @return array
     */
    public function permalink_rules($rules, $courses_cpt, $courses_permalink_base)
    {
        $rules[$courses_permalink_base . '/([^/]+)/' . helpers\App::get_quizzes_permalink_base() . '/([^/]+)/?$'] = 'index.php?' . $courses_cpt . '=$matches[2]&' . models\Quiz::$cpt . '=$matches[2]'; // makes /courses/course-name/quizzes/quiz-name/ resolve to quiz post.
        $rules[helpers\App::get_quizzes_permalink_base() . '/([^/]+)/?$'] = 'index.php?' . models\Quiz::$cpt . '=$matches[1]'; // Comment this line if you dont want quizzes/quiz-name to work in /courses/course-name/quizzes/quiz-name/.

        return $rules;
    }

    /**
     * Add models namespace to the list of Courses model namespaces
     *
     * @param array $models List of model namespaces.
     * @return array
     */
    public function models_namespace($models)
    {
        $models[] = base\MODELS_NAMESPACE;
        return $models;
    }

    /**
     * Add the admin header to the Edit Quiz page.
     */
    public static function admin_header()
    {
        global $current_screen;

        if ($current_screen->post_type === models\Quiz::$cpt && $current_screen->base === 'post') {
            lib\View::render('/admin/courses_quiz_header', get_defined_vars());
            self::override_editor_logo();
        }
    }

    /**
     * Replace Logo in Gutenberg Fullscreen Mode
     */
    private static function override_editor_logo()
    {
        global $current_screen;
        if (!$current_screen->is_block_editor) {
            return;
        } ?>
        <style>
            body.is-fullscreen-mode .edit-post-header a.edit-post-fullscreen-mode-close img,
            body.is-fullscreen-mode .edit-post-header a.edit-post-fullscreen-mode-close svg {
                display: none;
            }

            .edit-post-fullscreen-mode-close {
                background-color: #184499 !important;
            }

            body.is-fullscreen-mode .edit-post-header a.edit-post-fullscreen-mode-close:before {
                background-image: url('<?php echo esc_url(base\BRAND_URL . '/images/brand-icon-white.png'); ?>');
                background-size: contain;
                top: 20px;
                right: 10px;
                bottom: 20px;
                left: 10px;
                background-repeat: no-repeat;
            }
        </style>
        <?php
    }

    /**
     * Add the quiz post type to the list of post types that can be dripped.
     *
     * @param array  $dripping_cpts The list of post types that can be dripped.
     * @param object $course The course object.
     * @return array
     */
    public function course_dripping_post_types($dripping_cpts, $course)
    {
        $dripping_cpts[models\Quiz::$cpt] = isset($course->drip_quizzes) ? (int) $course->drip_quizzes : 0;
        return $dripping_cpts;
    }
}
