<?php

namespace memberpress\quizzes\controllers;

if (!defined('ABSPATH')) {
    die('You are not allowed to call this page directly.');
}

use memberpress\courses as courses;
use memberpress\quizzes\models as models;
use memberpress\quizzes\helpers as helpers;

class Api extends courses\lib\BaseCtrl
{
    /**
     * Load the hooks for the controller
     */
    public function load_hooks()
    {
        add_action('mpcs_courses_api_routes', [$this, 'quizzess_api_routes'], 10, 2);
    }

    /**
     * Register the API routes for quizzes
     *
     * @param courses\controllers\Api $courses_api The courses API controller.
     * @param string                  $namespace_str The namespace string for the API.
     *
     * @return void
     */
    public function quizzess_api_routes($courses_api, $namespace_str)
    {
        register_rest_route($namespace_str, '/courses/quizzes', [
            [
                'methods'             => \WP_REST_Server::READABLE,
                'callback'            => function ($request) use ($courses_api) {
                    return $courses_api->fetch_items($request, 'mpcs-quiz', 'quizzes');
                },
                'permission_callback' => [$courses_api, 'fetch_lessons_permissions_check'],
            ],
        ]);

        register_rest_route($namespace_str, '/courses/quizzes/(?P<id>[\d]+)', [
            [
                'methods'             => \WP_REST_Server::CREATABLE,
                'callback'            => [$this, 'duplicate_quiz'],
                'permission_callback' => [$courses_api, 'create_item_permissions_check'],
            ],
        ]);

        register_rest_route($namespace_str, '/courses/question/all', [
            [
                'methods'             => \WP_REST_Server::READABLE,
                'callback'            => [$this, 'fetch_all_questions'],
                'permission_callback' => [$courses_api, 'fetch_lessons_permissions_check'],
            ],
        ]);

        register_rest_route($namespace_str, '/courses/question/(?P<id>[\d]+)/duplicate/(?P<quiz_id>[\d]+)', [
            [
                'methods'             => \WP_REST_Server::CREATABLE,
                'callback'            => [$this, 'duplicate_question'],
                'permission_callback' => [$courses_api, 'create_item_permissions_check'],
            ],
        ]);

        register_rest_route($namespace_str, '/courses/reserveQuestionId/(?P<id>[\d]+)', [
            [
                'methods'             => \WP_REST_Server::READABLE,
                'callback'            => [$this, 'reserve_questionId'],
                'permission_callback' => [$courses_api, 'create_item_permissions_check'],
            ],
        ]);

        register_rest_route($namespace_str, '/courses/releaseQuestion/(?P<id>[\d]+)', [
            [
                'methods'             => \WP_REST_Server::CREATABLE,
                'callback'            => [$this, 'release_question'],
                'permission_callback' => [$courses_api, 'create_item_permissions_check'],
            ],
        ]);

        register_rest_route($namespace_str, '/courses/quiz/(?P<quiz_id>[\d]+)/questions', [
            [
                'methods'             => \WP_REST_Server::CREATABLE,
                'callback'            => [$this, 'save_questions'],
                'permission_callback' => [$courses_api, 'create_item_permissions_check'],
            ],
        ]);
    }

    /**
     * Fetches all questions from custom questions table
     *
     * @param \WP_REST_Request $request Full data about the request.
     * @return \WP_REST_Response
     */
    public function fetch_all_questions($request)
    {
        $params  = $request->get_params();
        $quiz_id = isset($params['id']) ? absint($params['id']) : 0; // If no id provided, use 0.

        $search = $params['search'] ? sanitize_text_field($params['search']) : '';
        $page   = $params['page'] ? max(1, (int) $params['page']) : 1;

        $data = helpers\Questions::questions_with_meta($quiz_id, $search, $page);

        return new \WP_REST_Response($data, 200);
    }


    /**
     * Duplicate a question
     *
     * @param \WP_REST_Request $request Full data about the request.
     * @return \WP_REST_Response|\WP_Error
     */
    public function duplicate_question(\WP_REST_Request $request)
    {
        $question_id = (int) $request->get_param('id');
        $quiz_id     = (int) $request->get_param('quiz_id');

        $original = models\Question::find($question_id);

        if (!$original instanceof models\Question) {
            return new \WP_Error('not-found', __('Question not found', 'memberpress-course-quizzes'), ['status' => 404]);
        }

        $question = new models\Question();
        $question->load_from_array($original->get_values());
        $question->id      = 0;
        $question->quiz_id = $quiz_id;
        $question->store();

        return new \WP_REST_Response(helpers\Questions::prepare_question($question));
    }

    /**
     * Reserves a row in the mepr_questions table for a question block
     *
     * @param \WP_REST_Request $request Full data about the request.
     * @return \WP_REST_Response
     */
    public function reserve_questionId($request)
    {
        $quiz_id = absint($request->get_param('id'));
        $id      = helpers\Questions::save_question_placeholder($quiz_id);

        return new \WP_REST_Response($id, 200);
    }

    /**
     * Releases a reserved id if no data was saved in it.
     * OR
     * If data was saved into the question, sets the quiz_id
     * to 0 to remove it from the quiz. Essentiall the question will be
     * orphaned.
     *
     * @param \WP_REST_Request $request Full data about the request.
     * @return \WP_REST_Response
     */
    public function release_question($request)
    {
        $id = absint($request->get_param('id'));
        helpers\Questions::maybe_orphan_or_delete_question($id);

        return new \WP_REST_Response($id, 200);
    }


    /**
     * Save the questions for a quiz
     *
     * @param \WP_REST_Request $request Full data about the request.
     * @return \WP_REST_Response|\WP_Error
     */
    public function save_questions($request)
    {
        $quiz_id   = (int) $request->get_param('quiz_id');
        $questions = $request->get_param('questions');
        $order     = $request->get_param('order');

        $quiz = models\Quiz::find($quiz_id);

        if (!$quiz instanceof models\Quiz) {
            return new \WP_Error('not-found', __('Quiz not found', 'memberpress-course-quizzes'), ['status' => 404]);
        }

        if (!is_array($questions) || !is_array($order)) {
            return new \WP_Error('bad-request', __('Bad request', 'memberpress-course-quizzes'), ['status' => 400]);
        }

        $question_ids = $quiz->get_questions(true);
        $errors       = [];
        $replaced_ids = [];

        foreach ($questions as $id => $question) {
            if (!is_numeric($id)) { // Skip placeholders.
                continue;
            }

            $index              = array_search($id, $order, true);
            $question['number'] = $index !== false ? $index + 1 : 1;
            $result             = helpers\Questions::save_question($quiz->ID, $question);

            if ($result instanceof \WP_Error) {
                $errors[] = [
                    'id'      => $id,
                    'message' => $result->get_error_message(),
                ];
            } elseif (is_numeric($result) && $result !== $id) {
                // The question ID changed when it was saved, which could happen if the question was deleted
                // beforehand, we need to return the new ID so that the question store can be updated.
                $replaced_ids[] = [
                    'oldId' => $id,
                    'newId' => $result,
                ];

                // Replace the old ID with the new ID within the question IDs from the post content.
                $key = array_search($id, $question_ids, true);
                if ($key !== false) {
                    $question_ids[$key] = $result;
                }
            }
        }

        if (!empty($question_ids)) {
            models\Question::sync_database($quiz->ID, $question_ids);
        }

        return new \WP_REST_Response([
            'errors' => $errors,
            'ids'    => $replaced_ids,
        ]);
    }

    /**
     * Duplicate a lesson
     *
     * @param \WP_REST_Request $request Full data about the request.
     * @return \WP_REST_Response|\WP_Error
     */
    public function duplicate_quiz($request)
    {
        $post_id = absint($request->get_param('id'));
        $post    = get_post($post_id);

        if (!$post instanceof \WP_Post || $post->post_type !== models\Quiz::$cpt) {
            return new \WP_Error('not-found', __('Post not found', 'memberpress-course-quizzes'), ['status' => 404]);
        }

        // args for new post.
        $args = [
            'comment_status' => $post->comment_status,
            'ping_status'    => $post->ping_status,
            'post_author'    => $post->post_author,
            'post_content'   => '', // Saving empty content initially for quizzes (see below).
            'post_excerpt'   => $post->post_excerpt,
            'post_name'      => $post->post_name,
            'post_parent'    => $post->post_parent,
            'post_password'  => $post->post_password,
            'post_status'    => $post->post_status,
            'post_title'     => $post->post_title,
            'post_type'      => $post->post_type,
            'to_ping'        => $post->to_ping,
            'menu_order'     => $post->menu_order,
        ];

        // insert the new post.
        $new_post_id = wp_insert_post($args);

        if (empty($new_post_id)) {
            return new \WP_Error('cant-create', __('Could not create duplicate post', 'memberpress-course-quizzes'), ['status' => 500]);
        }

        // add taxonomy terms to the new post.
        $taxonomies = get_object_taxonomies($post->post_type);
        foreach ($taxonomies as $taxonomy) {
            $post_terms = wp_get_object_terms($post_id, $taxonomy, ['fields' => 'slugs']);
            wp_set_object_terms($new_post_id, $post_terms, $taxonomy, false);
        }

        // Duplicate the questions within the quiz content first, then update the post with the new content.
        // If we save the duplicated post content during wp_insert_post above, it triggers Question::sync_database which
        // detaches the questions from the original post.
        $quiz               = new models\Quiz($new_post_id);
        $quiz->post_content = helpers\Questions::duplicate_quiz_questions($post->post_content, $new_post_id);
        $quiz->store();

        return new \WP_REST_Response($quiz->rec);
    }
}
