<?php

namespace memberpress\quizzes\lib;

use memberpress\quizzes as base;
use memberpress\courses as courses;

/**
 * Ctrls in MemberPress Courses are all singletons, so we can
 * use this factory to churn out objects for us.
 */
class CtrlFactory
{
    /**
     * Fetch a controller object.
     *
     * @param string $class Controller class name.
     * @param array  $args  Arguments to pass to the controller.
     *
     * @return object Controller object.
     * @throws \Exception If the controller class doesn't exist.
     */
    public static function fetch($class, $args = [])
    {
        static $objs;

        if (0 !== strpos($class, base\CTRLS_NAMESPACE) && $args['path'] === base\CTRLS_PATH) {
            $class = base\CTRLS_NAMESPACE . '\\' . courses\lib\Inflector::classify($class);
        }

        if (0 !== strpos($class, base\ADMIN_CTRLS_NAMESPACE) && $args['path'] === base\ADMIN_CTRLS_PATH) {
            $class = base\ADMIN_CTRLS_NAMESPACE . '\\' . courses\lib\Inflector::classify($class);
        }

        if (isset($objs[$class]) && ($objs[$class] instanceof courses\lib\BaseCtrl)) {
            return $objs[$class];
        }

        if (!class_exists($class)) {
            // Translators: %s is the class name that wasn't found.
            throw new \Exception(sprintf(esc_html__('Ctrl: %s wasn\'t found', 'memberpress-course-quizzes'), esc_html($class)));
        }

        // We'll let the autoloader in main.php.
        // handle including files containing these classes.
        $r   = new \ReflectionClass($class);
        $obj = $r->newInstanceArgs();

        $objs[$class] = $obj;

        return $obj;
    }

    /**
     * Get all controllers.
     *
     * @param array $args Arguments to pass to the controllers.
     *
     * @return array Controller objects.
     */
    public static function all($args = [])
    {
        $objs = [];

        foreach (self::paths() as $path) {
            $args['path'] = $path;
            $ctrls        = @glob($path . '/*.php', GLOB_NOSORT); // phpcs:ignore WordPress.PHP.NoSilencedErrors.Discouraged
            foreach ($ctrls as $ctrl) {
                if (basename($ctrl) === 'index.php') {
                    continue; // don't load index.php.
                }
                $class        = preg_replace('#\.php#', '', basename($ctrl));
                $objs[$class] = self::fetch($class, $args);
            }
        }

        return $objs;
    }

    /**
     * Get the paths to look for controllers in.
     *
     * @return array
     */
    public static function paths()
    {
        return apply_filters(
            base\SLUG_KEY . '_ctrls_paths',
            [base\CTRLS_PATH, base\ADMIN_CTRLS_PATH, base\BRAND_CTRLS_PATH]
        );
    }
}
