<?php

namespace memberpress\quizzes\models;

use memberpress\courses as courses;

if (!defined('ABSPATH')) {
    die('You are not allowed to call this page directly.');
}


/**
 * Answer model
 *
 * @property int $id The answer ID
 * @property int $attempt_id The attempt ID
 * @property int $question_id The question ID
 * @property string $answer The answer given
 * @property int $points_possible Highest possible points for this answer
 * @property int $points_awarded Points awarded for this answer
 * @property int $grader The user ID of the grader
 * @property string $answered_at Datetime in MySQL format
 * @property string $graded_at Datetime in MySQL format
 */
class Answer extends courses\lib\BaseModel
{
    /**
     * Answer constructor.
     *
     * @param null $obj The object to initialize the model with.
     */
    public function __construct($obj = null)
    {
        $this->initialize(
            [
                'id'              => [
                    'default' => 0,
                    'type'    => 'integer',
                ],
                'attempt_id'      => [
                    'default' => 0,
                    'type'    => 'integer',
                ],
                'question_id'     => [
                    'default' => 0,
                    'type'    => 'integer',
                ],
                'answer'          => [
                    'default' => '',
                    'type'    => 'string',
                ],
                'points_possible' => [
                    'default' => 0,
                    'type'    => 'integer',
                ],
                'points_awarded'  => [
                    'default' => 0,
                    'type'    => 'integer',
                ],
                'grader'          => [
                    'default' => 0,
                    'type'    => 'integer',
                ],
                'answered_at'     => [
                    'default' => courses\lib\Utils::ts_to_mysql_date(time()),
                    'type'    => 'datetime',
                ],
                'graded_at'       => [
                    'default' => courses\lib\Utils::ts_to_mysql_date(time()),
                    'type'    => 'datetime',
                ],
            ],
            $obj
        );
    }

    /**
     * Store this answer
     *
     * @param bool $validate Whether to validate the model before storing. Default is true.
     *
     * @return int|\WP_Error|false The answer ID, a WP_Error if validation fails, or false on failure
     */
    public function store($validate = true)
    {
        if ($validate) {
            try {
                $this->validate();
            } catch (courses\lib\ValidationException $e) {
                return new \WP_Error(get_class($e), $e->getMessage());
            }
        }

        $db    = courses\lib\Db::fetch();
        $attrs = $this->get_values();

        $attrs['answer'] = is_array($attrs['answer']) ? serialize($attrs['answer']) : $attrs['answer'];

        if (isset($this->id) && $this->id > 0) {
            $db->update_record($db->answers, $this->id, $attrs);
        } else {
            $this->id = $db->create_record($db->answers, $attrs, false);
        }

        return $this->id;
    }

    /**
     * Destroy this answer
     *
     * @return int|false The number of affected rows or false if there was an error
     */
    public function destroy()
    {
        $db = courses\lib\Db::fetch();

        return $db->delete_records($db->answers, ['id' => $this->id]);
    }

    /**
     * Validate the answer object
     *
     * @return bool
     * @throws courses\lib\ValidationException If validation fails.
     */
    public function validate()
    {
        courses\lib\Validate::is_numeric($this->id, 0, null, __('Id', 'memberpress-course-quizzes'));
        courses\lib\Validate::is_numeric($this->attempt_id, 0, null, __('Attempt Id', 'memberpress-course-quizzes'));
        courses\lib\Validate::is_numeric($this->question_id, 0, null, __('Question Id', 'memberpress-course-quizzes'));
        courses\lib\Validate::is_numeric($this->points_possible, 0, null, __('Points Possible', 'memberpress-course-quizzes'));
        courses\lib\Validate::is_numeric($this->points_awarded, 0, null, __('Points Awarded', 'memberpress-course-quizzes'));
        courses\lib\Validate::not_empty($this->answered_at, __('Created At', 'memberpress-course-quizzes'));

        return true;
    }

    /**
     * Insert an answer or update an existing answer
     *
     * @param  int          $attempt_id      The attempt ID.
     * @param  int          $question_id     The question ID.
     * @param  string|array $answer          The answer string, or array for multiple answer (pass it unserialized).
     * @param  int          $points_possible The points possible.
     * @param  int          $points_awarded  The points awarded.
     * @param  int          $grader          The grader user ID.
     * @param  string       $answered_at     The MySQL datetime the answer was created at.
     * @param  string       $graded_at       The MySQL datetime the answer was graded at.
     *
     * @return int|false                     Integer (0, 1 or 2) on success, false on failure
     */
    public static function insert_or_replace_answer(
        $attempt_id,
        $question_id,
        $answer,
        $points_possible,
        $points_awarded,
        $grader,
        $answered_at,
        $graded_at
    ) {
        global $wpdb;
        $db = courses\lib\Db::fetch();

        $query = "INSERT INTO {$db->answers}
            (`attempt_id`, `question_id`, `answer`, `points_possible`, `points_awarded`, `grader`, `answered_at`, `graded_at`)
            VALUES
            (%d, %d, %s, %d, %d, %d, %s, %s)
            ON DUPLICATE KEY UPDATE
            `attempt_id` = VALUES(`attempt_id`),
            `question_id` = VALUES(`question_id`),
            `answer` = VALUES(`answer`),
            `points_possible` = VALUES(`points_possible`),
            `points_awarded` = VALUES(`points_awarded`),
            `grader` = VALUES(`grader`),
            `answered_at` = VALUES(`answered_at`),
            `graded_at` = VALUES(`graded_at`)
        ";


        $prepared_query = $wpdb->prepare(
            $query, // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
            $attempt_id,
            $question_id,
            maybe_serialize($answer),
            $points_possible,
            $points_awarded,
            $grader,
            $answered_at,
            $graded_at
        );

        return $wpdb->query($prepared_query); // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
    }
}
