<?php

namespace memberpress\quizzes\controllers\admin;

if (!defined('ABSPATH')) {
    die('You are not allowed to call this page directly.');
}

use memberpress\courses as courses;
use memberpress\quizzes as base;
use memberpress\quizzes\models as models;
use memberpress\quizzes\helpers as helpers;
use memberpress\courses\controllers\admin as adminmpcs;

class Quizzes extends courses\lib\BaseCptCtrl
{
    /**
     * Load hooks
     *
     * @return void
     */
    public function load_hooks()
    {
        $this->ctaxes = ['course-tags', 'course-categories'];

        add_action('admin_enqueue_scripts', [$this, 'admin_enqueue_scripts']);
        add_action('enqueue_block_editor_assets', [$this, 'enqueue_block_editor_assets']);
        add_action('wp_ajax_mpcs_delete_all_attempts', [$this, 'ajax_delete_all_attempts']);
        add_filter('manage_mpcs-quiz_posts_columns', [$this, 'add_course_column']);
        add_action('manage_mpcs-quiz_posts_custom_column', [$this, 'course_column_content'], 10, 2);
        add_action('admin_footer-edit.php', [$this, 'categories_tags_buttons']);
        add_action('admin_init', [$this, 'register_filter_queries']);
        add_action('restrict_manage_posts', [$this, 'render_additional_filters']);
        add_action('init', [$this, 'register_meta_fields']);
        add_action('updated_post_meta', [$this, 'sync_require_previous_setting'], 10, 4);
        add_action('rest_after_insert_' . models\Quiz::$cpt, [$this, 'save_post_data'], 10, 2);
        add_action('mpcs_lesson_cpts', [$this, 'add_quizzes_to_lesson_cpts'], 10, 2);
        add_action('mpcs_course_admin_settings', [$this, 'course_admin_settings'], 10, 2);
        add_action('mpcs_admin_slugs_options', [$this, 'page_slug_settings'], 10, 2);
        add_action('mpcs_curriculum_lesson', [$this, 'add_quizzes_to_curriculum'], 10, 2);
        add_action('mpcs_reset_course_progress', [$this, 'reset_attempts_for_quiz'], 10, 2);
    }

    /**
     * Register the custom post type
     *
     * @return void
     */
    public function register_post_type()
    {
        $this->cpt = (object)[
            'slug'   => models\Quiz::$cpt,
            'config' => [
                'labels'             => [
                    'name'               => __('Quizzes', 'memberpress-course-quizzes'),
                    'singular_name'      => __('Quiz', 'memberpress-course-quizzes'),
                    'add_new'            => __('Add New', 'memberpress-course-quizzes'),
                    'add_new_item'       => __('Add New Quiz', 'memberpress-course-quizzes'),
                    'edit_item'          => __('Edit Quiz', 'memberpress-course-quizzes'),
                    'new_item'           => __('New Quiz', 'memberpress-course-quizzes'),
                    'view_item'          => __('View Quiz', 'memberpress-course-quizzes'),
                    'search_items'       => __('Search Quizzes', 'memberpress-course-quizzes'),
                    'not_found'          => __('No Quizzes found', 'memberpress-course-quizzes'),
                    'not_found_in_trash' => __('No Quizzes found in Trash', 'memberpress-course-quizzes'),
                    'parent_item_colon'  => __('Parent Quiz:', 'memberpress-course-quizzes'),
                ],
                'public'             => true,
                'publicly_queryable' => true,
                'show_ui'            => true,
                'show_in_rest'       => true,
                'show_in_menu'       => courses\PLUGIN_NAME,
                'has_archive'        => false,
                'capability_type'    => 'page',
                'hierarchical'       => false,
                'rewrite'            => [
                    'slug'       => '/' . courses\helpers\Courses::get_permalink_base() . '/%course_slug%/' . helpers\App::get_quizzes_permalink_base(),
                    'with_front' => false,
                ],
                'supports'           => ['title', 'editor', 'thumbnail', 'custom-fields'],
                'taxonomies'         => [],
            ],
        ];

        if (!empty($this->ctaxes)) {
            $this->cpt->config['taxonomies'] = $this->ctaxes;
        }

        register_post_type(models\Quiz::$cpt, $this->cpt->config);
    }

    /**
     * Enqueue scripts and styles for the admin
     *
     * @return void
     */
    public function admin_enqueue_scripts()
    {
        global $current_screen, $post;

        if ($post instanceof \WP_Post && $current_screen instanceof \WP_Screen && $current_screen->id == models\Quiz::$cpt) {
            $quiz = models\Quiz::find($post->ID);

            if ($quiz instanceof models\Quiz) {
                $course = $quiz->course();

                $courses_url  = '';
                $course_title = '';

                if ($course instanceof courses\models\Course) {
                    $courses_url  = get_edit_post_link($course->ID) . '#curriculum';
                    $course_title = $course->post_title;
                }

                $data = helpers\Questions::questions_with_meta($quiz->ID);

                wp_enqueue_style('vex-css', base\CSS_URL . '/vendor/vex.css', [], base\VERSION);
                wp_enqueue_style('mpcs-quiz-editor', base\CSS_URL . '/admin_quiz_editor.css', [], base\VERSION);
                wp_enqueue_script('mpcs-course-editor-js', courses\JS_URL . '/course-editor.js', ['jquery'], base\VERSION, true);
                wp_enqueue_script('vex-js', courses\JS_URL . '/vendor/vex.combined.js', [], base\VERSION, true);
                wp_enqueue_script('mpcs-quiz-editor-js', base\JS_URL . '/quiz-editor.js', ['jquery', 'vex-js'], base\VERSION, true);

                $back_cta_label = __('Back to Quizzes', 'memberpress-course-quizzes');
                $back_cta_url   = admin_url('edit.php?post_type=' . models\Quiz::$cpt);
                if (isset($_GET['curriculum']) && $course) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
                    // translators: %s is the course title.
                    $back_cta_label = sprintf(__('Back to %s', 'memberpress-course-quizzes'), $course->post_title);
                    $back_cta_url   = get_edit_post_link($course->ID) . '#curriculum';
                }


                wp_localize_script('mpcs-course-editor-js', 'MPCS_Course_Data', [
                    'state'          => [
                        'questions' => (object) $this->get_questions($quiz),
                        'sidebar'   => [
                            'questions'  => $data['questions'],
                            'searchMeta' => $data['searchMeta'],
                        ],
                    ],
                    'imagesUrl'      => base\IMAGES_URL,
                    'coursesUrl'     => $courses_url,
                    'courseTitle'    => $course_title,
                    'back_cta_url'   => $back_cta_url,
                    'back_cta_label' => $back_cta_label,
                    'api'            => [
                        'question'        => courses\controllers\CoursesApi::$namespace_str . '/' . courses\controllers\CoursesApi::$resource_name_str . '/question/',
                        'reserveId'       => courses\controllers\CoursesApi::$namespace_str . '/' . courses\controllers\CoursesApi::$resource_name_str . '/reserveQuestionId/',
                        'releaseQuestion' => courses\controllers\CoursesApi::$namespace_str . '/' . courses\controllers\CoursesApi::$resource_name_str . '/releaseQuestion/',
                    ],
                    'strings'        => [
                        'require_passing_score_str' => models\Quiz::$require_passing_score_str,
                        'passing_score_unit_str'    => models\Quiz::$passing_score_unit_str,
                        'passing_score_str'         => models\Quiz::$passing_score_str,
                        'failed_message_str'        => models\Quiz::$failed_message_str,
                        'allow_retakes_str'         => models\Quiz::$allow_retakes_str,
                        'retake_limit_str'          => models\Quiz::$retake_limit_str,
                        'retake_method_str'         => models\Quiz::$retake_method_str,
                    ],
                ]);

                $quiz_editor_l10n = [
                    'hasAttempts'                  => $quiz->has_attempts(),
                    'quizLockedMessage'            => $this->get_quiz_locked_dialog_html($quiz->ID),
                    'confirmDeleteAllQuizAttempts' => __('Are you sure you want to delete all attempts for this quiz?', 'memberpress-course-quizzes'),
                    'delete'                       => __('Delete', 'memberpress-course-quizzes'),
                    'cancel'                       =>  __('Cancel', 'memberpress-course-quizzes'),
                    'ajaxUrl'                      => admin_url('admin-ajax.php'),
                    'quizId'                       => $post->ID,
                    'deleteAllAttemptsNonce'       => wp_create_nonce('mpcs_delete_all_attempts'),
                    'courseUrl'                    => $courses_url,
                ];

                wp_localize_script(
                    'mpcs-quiz-editor-js',
                    'MpcsQuizEditorL10n',
                    ['l10n_print_after' => 'MpcsQuizEditorL10n = ' . wp_json_encode($quiz_editor_l10n)]
                );
            }
        }
    }


    /**
     * Add assignments to the list of lesson custom post types.
     *
     * @param  array $cpts The list of lesson custom post types.
     * @return array
     */
    public function add_quizzes_to_lesson_cpts($cpts)
    {
        $cpts[models\Quiz::$cpt] = '\\' . models\Quiz::class;
        return $cpts;
    }

    /**
     * Get the questions for the quiz editor default state
     *
     * @param models\Quiz $quiz The quiz object.
     * @return array
     */
    protected function get_questions($quiz)
    {
        $questions = [];

        foreach ($quiz->get_questions() as $question) {
            $questions[$question->id] = helpers\Questions::prepare_question($question);
        }

        return $questions;
    }

    /**
     * Enqueue block editor only JavaScript and CSS.
     */
    public function enqueue_block_editor_assets()
    {
        global $current_screen, $post;

        if ($post instanceof \WP_Post && $current_screen instanceof \WP_Screen && $current_screen->id == models\Quiz::$cpt) {
            $quiz = models\Quiz::find($post->ID);

            if ($quiz instanceof models\Quiz) {
                $course = $quiz->course();

                if ($course instanceof courses\models\Course || apply_filters('mpcs_load_quiz_builder', true, $quiz)) {
                    $asset_file = include(base\PATH . '/build/index.asset.php');
                    wp_enqueue_style('mpcs-quizzes', base\URL . '/build/index.css', [], '0.1');
                    wp_enqueue_style('mpcs-quizzes-style', base\URL . '/build/style-index.css', [], '0.1');
                    wp_enqueue_script(
                        'mpcs-quizzes-builder',
                        base\URL . '/build/index.js',
                        array_merge($asset_file['dependencies'], ['mpcs-course-editor-js', 'regenerator-runtime']),
                        $asset_file['version'],
                        true
                    );
                }
            }
        }
    }

    /**
     * Get the HTML for the popup message shown when a quiz can't be edited
     *
     * @param int $quiz_id The quiz ID.
     * @return string
     */
    public function get_quiz_locked_dialog_html($quiz_id)
    {
        ob_start();
        ?>
        <div class="mpcs-quiz-locked-content">
            <div class="mpcs-quiz-locked-title"><?php esc_html_e("You Can't Edit this Quiz", 'memberpress-course-quizzes'); ?></div>
            <p>
                <?php
                printf(
                    /* translators: %1$s: open link tag to view attempts, %2$s: close link tag to view attempts */
                    esc_html__('This quiz %1$salready has attempts%2$s recorded for it. Do you want to delete all attempts?', 'memberpress-course-quizzes'),
                    sprintf(
                        '<a href="%s" target="_blank">',
                        esc_url(add_query_arg(['id' => $quiz_id], admin_url('admin.php?page=mpcs-quiz-attempts')))
                    ),
                    '</a>'
                );
                ?>
            </p>
            <p><strong><?php esc_html_e('Warning: This cannot be undone and students will lose their scores.', 'memberpress-course-quizzes'); ?></strong></p>
        </div>
        <?php
        return ob_get_clean();
    }

    /**
     * Handle the Ajax request to delete all quiz attempts
     */
    public function ajax_delete_all_attempts()
    {
        courses\lib\Utils::validate_admin_ajax_post_request('mpcs_delete_all_attempts');

        if (!isset($_POST['quiz_id']) || !is_numeric($_POST['quiz_id'])) { // phpcs:ignore WordPress.Security.NonceVerification.Missing
            wp_send_json_error(__('Bad request', 'memberpress-course-quizzes'));
        }

        $quiz_id = (int) $_POST['quiz_id']; // phpcs:ignore WordPress.Security.NonceVerification.Missing

        $attempts = models\Attempt::get_all('', '', ['quiz_id' => $quiz_id]);

        if (is_array($attempts)) {
            foreach ($attempts as $attempt) {
                $attempt->destroy();
            }
        }

        wp_send_json_success();
    }

    /**
     * Add a column to the quizzes list table for the course
     *
     * @param array $columns The columns in the list table.
     * @return array
     */
    public static function add_course_column($columns)
    {
        $columns['course'] = __('Course', 'memberpress-course-quizzes');

        return $columns;
    }

    /**
     * Output the content for the course column in the quizzes list table
     *
     * @param string $column The column being output.
     * @param int    $post_id The ID of the post being output.
     * @return void
     */
    public static function course_column_content($column, $post_id)
    {
        if ($column === 'course') {
            $quiz = models\Quiz::find($post_id);

            if ($quiz instanceof models\Quiz) {
                $course = $quiz->course();

                if ($course instanceof courses\models\Course) {
                    echo esc_html($course->post_title);
                }
            }
        }
    }

    /**
     * Add buttons for categories and tags to the quizzes list table
     *
     * @return void
     */
    public function categories_tags_buttons()
    {
        if (empty($_GET['post_type']) || models\Quiz::$cpt !== $_GET['post_type']) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
            return;
        }
        $new_links  = sprintf('<a href="%2$s" class="page-title-action" style="margin-left: 0;">%1$s</a>', esc_html__('Categories', 'memberpress-course-quizzes'), add_query_arg([
            'taxonomy'  => adminmpcs\CurriculumCategories::$tax,
            'post_type' => models\Quiz::$cpt,
        ], admin_url('edit-tags.php')));
        $new_links .= sprintf('<a href="%2$s" class="page-title-action">%1$s</a>', esc_html__('Tags', 'memberpress-course-quizzes'), add_query_arg([
            'taxonomy'  => adminmpcs\CurriculumTags::$tax,
            'post_type' => models\Quiz::$cpt,
        ], admin_url('edit-tags.php')));
        ?>
        <script>
            jQuery(document).ready(function($) {
                $('.wrap .wp-header-end').before("<?php echo addslashes($new_links); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped ?>");
            });
        </script>
        <?php
    }

    /**
     * Render extra filters.
     *
     * @param string $post_type The post type.
     * @return void
     */
    public static function render_additional_filters($post_type)
    {
        if ($post_type === models\Quiz::$cpt) {
            $taxonomy      = adminmpcs\CurriculumCategories::$tax;
            $selected      = isset($_GET[$taxonomy]) ? $_GET[$taxonomy] : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
            $info_taxonomy = get_taxonomy($taxonomy);

            if (false === $info_taxonomy) {
                return;
            }

            $taxonomy_args = [
                'taxonomy'   => $taxonomy,
                'hide_empty' => false,
                'fields'     => 'ids',
                'number'     => 1,
            ];

            $taxonomy_terms = get_terms($taxonomy_args);

            if (empty($taxonomy_terms)) {
                return;
            }

            wp_dropdown_categories([
                // translators: %s is the taxonomy label.
                'show_option_all' => sprintf(esc_html__('Show all %s', 'memberpress-course-quizzes'), $info_taxonomy->label),
                'taxonomy'        => $taxonomy,
                'name'            => $taxonomy,
                'orderby'         => 'name',
                'selected'        => $selected,
                'show_count'      => true,
                'hide_empty'      => false,
                'walker'          => new courses\lib\WalkerCurriculumCategories($post_type),
            ]);
        }
    }

    /**
     * Register the filter queries.
     *
     * @return void
     */
    public function register_filter_queries()
    {
        add_action('parse_query', [$this, 'filter_post_type_by_taxonomy']);
    }

    /**
     * Filter the courses as per selected taxonomy.
     *
     * @param string $query The query object.
     * @return void
     */
    public function filter_post_type_by_taxonomy($query)
    {
        global $pagenow;
        $taxonomy = adminmpcs\CurriculumCategories::$tax;
        if (
            $pagenow == 'edit.php' && is_admin()
            && isset($query->query_vars['post_type'])
            && $query->query_vars['post_type'] === models\Quiz::$cpt
            && isset($query->query_vars[$taxonomy])
            && is_numeric($query->query_vars[$taxonomy])
            && 0 < absint($query->query_vars[$taxonomy])
        ) {
            $term = get_term_by('id', (int) $query->query_vars[$taxonomy], $taxonomy);
            if ($term && !is_wp_error($term)) {
                $query->query_vars[$taxonomy] = $term->slug;
            }
        }
    }


    /**
     * Save the quiz post data
     *
     * @param \WP_Post $post The post object.
     * @param array    $request The request data.
     * @return void
     */
    public function save_post_data($post, $request)
    {
        // Skip ajax.
        if (defined('DOING_AJAX')) {
            return;
        }

        $quiz      = new models\Quiz($post->ID);
        $unit_str  = models\Quiz::$passing_score_unit_str;
        $score_str = models\Quiz::$passing_score_str;
        $meta      = (array) $request['meta'];

        if (isset($meta[$unit_str]) && 'percent' === $meta[$unit_str]) {
            if (isset($meta[$score_str]) && $meta[$score_str] > 100) {
                $quiz->passing_score = 100;
                $quiz->store();
            }
        }
    }

    /**
     * Register the meta fields for the quiz
     *
     * @return void
     */
    public static function register_meta_fields()
    {
        register_meta('post', models\Quiz::$require_passing_score_str, [
            'object_subtype' => models\Quiz::$cpt,
            'auth_callback'  => '__return_true',
            'default'        => false,
            'show_in_rest'   => true,
            'single'         => true,
            'type'           => 'boolean',
        ]);
        register_meta('post', models\Quiz::$passing_score_unit_str, [
            'object_subtype' => models\Quiz::$cpt,
            'auth_callback'  => '__return_true',
            'default'        => 'percent',
            'show_in_rest'   => true,
            'single'         => true,
            'type'           => 'string',
        ]);
        register_meta('post', models\Quiz::$passing_score_str, [
            'object_subtype' => models\Quiz::$cpt,
            'auth_callback'  => '__return_true',
            'default'        => 1,
            'show_in_rest'   => true,
            'single'         => true,
            'type'           => 'number',
        ]);
        register_meta('post', models\Quiz::$failed_message_str, [
            'object_subtype' => models\Quiz::$cpt,
            'auth_callback'  => '__return_true',
            'default'        => '',
            'show_in_rest'   => true,
            'single'         => true,
            'type'           => 'string',
        ]);
        register_meta('post', models\Quiz::$allow_retakes_str, [
            'object_subtype' => models\Quiz::$cpt,
            'auth_callback'  => '__return_true',
            'default'        => false,
            'show_in_rest'   => true,
            'single'         => true,
            'type'           => 'boolean',
        ]);
        register_meta('post', models\Quiz::$retake_limit_str, [
            'object_subtype' => models\Quiz::$cpt,
            'auth_callback'  => '__return_true',
            'default'        => 1,
            'show_in_rest'   => true,
            'single'         => true,
            'type'           => 'number',
        ]);
        register_meta('post', models\Quiz::$retake_method_str, [
            'object_subtype' => models\Quiz::$cpt,
            'auth_callback'  => '__return_true',
            'default'        => '',
            'show_in_rest'   => true,
            'single'         => true,
            'type'           => 'string',
        ]);
    }

    /**
     * If the user enables require_passing_score setting on the quiz
     * Then ensure course's “Require Previous Lesson/Quiz” is disabled
     *
     * @param int    $meta_id The meta ID.
     * @param int    $object_id The object ID.
     * @param string $meta_key The meta key.
     * @param string $_meta_value The meta value.
     * @return void
     */
    public function sync_require_previous_setting($meta_id, $object_id, $meta_key, $_meta_value)
    {
        if ($meta_key !== models\Quiz::$require_passing_score_str) {
            return;
        }

        if (!$_meta_value || false === $_meta_value) {
            return;
        }

        $quiz   = models\Quiz::find($object_id);
        $course = $quiz->course();

        if (true === $course->require_previous) {
            return;
        }

        $course->require_previous = 'enabled';
        $course->store();
    }

    /**
     * Add quizzes to the course admin settings
     *
     * @param array         $settings The settings array.
     * @param models\Course $course The course object.
     * @return array
     */
    public function course_admin_settings($settings, $course)
    {
        $settings['quiz_requires_passing_score'] = self::any_quiz_with_require_passing_score($course);
        $settings['drip_quizzes']                = [
            'name'  => courses\models\Course::$drip_quizzes_str,
            'value' => isset($course->drip_quizzes) ? $course->drip_quizzes : 0,
        ];

        return $settings;
    }

    /**
     * Check if the course has any quiz with require passing score
     *
     * @param object $course The course object.
     * @return boolean
     */
    public static function any_quiz_with_require_passing_score($course)
    {
        $quizzes = (array) $course->quizzes();
        if (!empty($quizzes) && is_array($quizzes)) {
            foreach ($quizzes as $quiz) {
                $require_passing_score = get_post_meta($quiz->ID, models\Quiz::$require_passing_score_str, true);
                if ($require_passing_score) {
                    return true;
                }
            }
        }

        return false;
    }

    /**
     * Add the quizzes slug settings to the admin settings page
     *
     * @param array $options The options array.
     * @return void
     */
    public function page_slug_settings($options)
    {
        ?>
        <tr valign="top">
            <td>
                <label for="mpcs_options_quizzes_slug"><?php esc_html_e('Quizzes Slug:', 'memberpress-course-quizzes'); ?>
                    <?php
                    courses\helpers\App::info_tooltip(
                        'mepr-quizzes-slug',
                        esc_html__('Quizzes Slug', 'memberpress-course-quizzes'),
                        esc_html__('Use this field to change the permalink base of your quizzes to something other than /quizzes/', 'memberpress-course-quizzes')
                    );
                    ?>
            </td>
            <td>
                <input type="text" id="mpcs_options_quizzes_slug" name="mpcs-options[quizzes-slug]" placeholder="<?php echo esc_attr(models\Quiz::$permalink_slug); ?>" class="regular-text" value="<?php echo esc_attr(courses\helpers\Options::val($options, 'quizzes-slug')); ?>" />
            </td>
        </tr>
        <?php
    }

    /**
     * Add quizzes to the curriculum
     *
     * @param array $curriculum The curriculum array.
     * @param mixed $lesson The lesson object.
     * @return array
     */
    public function add_quizzes_to_curriculum($curriculum, $lesson)
    {
        if ($lesson instanceof models\Quiz) {
            $curriculum['lessons']['section'][$lesson->ID]['hasAttempts'] = $lesson->has_attempts();
        }

        return $curriculum;
    }

    /**
     * Reset the attempts for a quiz
     *
     * @param int $user_id The user ID.
     * @param int $lesson_id The lesson ID.
     * @return void
     */
    public function reset_attempts_for_quiz($user_id, $lesson_id)
    {
        $attempt = models\Attempt::get_one([
            'user_id' => $user_id,
            'quiz_id' => $lesson_id,
        ]);

        if ($attempt instanceof models\Attempt) {
            $answers = $attempt->get_answers();
            foreach ($answers as $answer) {
                $answer = models\Answer::find($answer->id);
                $answer->destroy();
            }

            $attempt->destroy();
        }
    }
}
