<?php
/**
 * Bookings and Appointment Plugin for WooCommerce
 *
 * Class for Outlook Calendar API for WooCommerce Booking and Appointment Plugin
 *
 * @author   Tyche Softwares
 * @package  Bkap_Outlook_Calendar/Outlook-Calendar-OAuth
 * @category Classes
 * @since 1.0.0
 */

require_once BKAPOC_INCLUDE_PATH . '/outlook-vendor/vendor/autoload.php';

use Microsoft\Graph\Graph;
use Microsoft\Graph\Model;
if ( ! class_exists( 'BKAP_Outlook_Calendar_OAuth' ) ) {

	/**
	 * Class for Outlook Calendar API for WooCommerce Booking and Appointment Plugin
	 * BKAP_Outlook_Calendar_OAuth
	 *
	 * @class BKAP_Outlook_Calendar_OAuth
	 */
	class BKAP_Outlook_Calendar_OAuth {

		/**
		 * Product ID
		 *
		 * @var int $product_id.
		 */
		public $product_id;

		/**
		 * User ID
		 *
		 * @var int $user_id.
		 */
		public $user_id;

		/**
		 * Product OR Global
		 *
		 * @var bool $is_product.
		 */
		public $is_product;

		/**
		 * OAuth Provider
		 *
		 * @var obj OAUTH Provider.
		 */
		public $provider;

		/**
		 * OAuth Provider
		 *
		 * @var obj OAUTH Provider.
		 */
		public $graph = null;

		/**
		 * Construct
		 *
		 * @param int $product_id Product ID.
		 * @param int $user_id User ID.
		 */
		public function __construct( $product_id = 0, $user_id = 1 ) {

			$this->product_id = $product_id;
			$this->user_id    = $user_id;
			$this->is_product = ( $product_id ) ? true : false; // If called for product the true else false.

			$this->local_time      = current_time( 'timestamp' ); // phpcs:ignore
			$global_settings       = json_decode( get_option( 'woocommerce_booking_global_settings' ) );
			$this->time_format     = ( isset( $global_settings->booking_time_format ) ) ? $global_settings->booking_time_format : 'H:i';
			$this->date_format     = ( isset( $global_settings->booking_date_format ) ) ? $global_settings->booking_date_format : 'Y-m-d';
			$this->datetime_format = $this->date_format . ' ' . $this->time_format;
			$uploads               = wp_upload_dir(); // Set log file location.
			$this->uploads_dir     = isset( $uploads['basedir'] ) ? $uploads['basedir'] . '/' : WP_CONTENT_DIR . '/uploads/';
			$this->log_file        = $this->uploads_dir . 'bkap-log.txt';
		}

		/**
		 * This function is for connecting the Microsoft Graph API.
		 *
		 * @since 1.0.0
		 */
		public function bkap_outlook_graph() {

			$graph      = new Graph();
			$connection = $this->bkap_outlook_connect();

			if ( $connection ) {

				$access_token = $this->bkap_get_access_token();
				$token_data   = $this->bkap_get_token_data();

				if ( empty( $access_token ) && $token_data ) {

					if ( $token_data->hasExpired() ) {
						if ( ! is_null( $token_data->getRefreshToken() ) ) {

							$token_data = $this->provider->getAccessToken(
								'refresh_token',
								array(
									'scope'         => $this->provider->scope,
									'refresh_token' => $token_data->getRefreshToken()
								)
							);

							$access_token = $token_data->getToken();
							$this->bkap_set_access_token( $access_token );

							$graph->setAccessToken( $access_token );

							$this->graph = $graph;
						} else {
							$token_data = null;
						}
					}
				} else {
					if ( $access_token ) {
						$graph->setAccessToken( $access_token );
						$this->graph = $graph;
					}
				}
			}

			return $graph;
		}

		/**
		 * This function returns the calendar list.
		 *
		 * @since 1.0.0
		 */
		public function bkap_outlook_calendar_list() {

			$this->bkap_outlook_graph();

			if ( ! is_null( $this->graph ) ) {
				$calendar_data = array( array( 'name' => __( 'Select a calendar from the list', 'bkap-outlook-calendar' ), 'id' => '' ) );

				$calendars = $this->graph
							->createRequest( 'GET', '/me/calendars' )
							->setReturnType( Model\Event::class )
							->execute();

				foreach ( $calendars as $calendar ) {
					$properties      = $calendar->getProperties();
					$calendar_data[] = array(
						'name' => $properties['name'],
						'id'   => $properties['id']
					);
				}
			} else {
				$calendar_data = array();
			}

			return $calendar_data;
		}

		/**
		 * This function inserts the event to the calendar.
		 *
		 * @param array $event_details This array contains all the booking details.
		 * @param int   $item_id Item ID.
		 * @param int   $user_id User ID.
		 * @param int   $product_id Product ID.
		 * @param bool  $test False if Testing.
		 * @param int   $item_number Number of Item.
		 *
		 * @since 1.0.0
		 */
		public function bkap_outlook_insert_event( $event_details, $item_id, $user_id, $product_id = 0, $test = false, $item_number = -1 ) {

			$this->bkap_outlook_graph();

			$bkap = bkap_event_data( $event_details, $item_id );

			if ( ! isset( $bkap ) ) {
				return false;
			}

			$timezone_string = bkap_booking_get_timezone_string();
			if ( $bkap->start_time == '' && $bkap->end_time == '' ) {
				$start = bkap_get_date_as_per_utc_timezone( $bkap->start, $timezone_string );
				$end   = bkap_get_date_as_per_utc_timezone( $bkap->end, $timezone_string );
			} elseif ( $bkap->end_time == '' ) {
				$start = bkap_get_date_as_per_utc_timezone( $bkap->start . ' ' . $bkap->start_time, $timezone_string );
				$end   = bkap_get_date_as_per_utc_timezone( $bkap->start . ' ' . $bkap->start_time, $timezone_string );
			} else {
				$start = bkap_get_date_as_per_utc_timezone( $bkap->start . ' ' . $bkap->start_time, $timezone_string );
				$end   = bkap_get_date_as_per_utc_timezone( $bkap->end . ' ' . $bkap->end_time, $timezone_string );
			}

			$event_loc_desc_sum = bkap_get_event_location_description_summary( $bkap, 'outlook' );
			$location           = $event_loc_desc_sum['location'];
			$summary            = $event_loc_desc_sum['summary'];
			$description        = $event_loc_desc_sum['description'];

			$data = array(
				'Subject'  => $summary,
				'Body'     => array(
					'ContentType' => 'HTML',
					'Content'     => $description,
				),
				'location' => array(
					'displayName' => $location,
				),
				'Start'    => array(
					'DateTime' => $start,
					'TimeZone' => $timezone_string,
				),
				'End'      => array(
					'DateTime' => $end,
					'TimeZone' => $timezone_string,
				),
			);

			$calendar_id = $this->bkap_calendar_id();
			$url         = '/me/calendars/' . $calendar_id . '/events';

			try {
				$response = $this->graph->createRequest( 'POST', $url )
				->attachBody( $data )
				->setReturnType( Model\Event::class )
				->execute();

				$response_properties = $response->getProperties();
				$uid                 = $response_properties['id'];

				bkap_update_event_item_uid_data( $uid, $product_id, $user_id, $item_id, $item_number, 'outlook' );

				return true;
			} catch ( Exception $e ) {
				$this->log( 'Insert went wrong: ' . $e->getMessage() );
				return false;
			}
		}

		/**
		 * This functions deletes the event from the Outlook Calendar.
		 *
		 * @param int    $item_id Item ID.
		 * @param string $event_uid Event ID.
		 * @param int    $item_number Item Number.
		 * @since 1.0.0
		 */
		public function bkap_outlook_delete_event( $item_id, $event_uid = '', $item_number = -1 ) {

			$event_id = ''; // get the event UID.
			$user     = new WP_User( $this->user_id );
			if ( $this->is_product ) {
				$event_uids = get_post_meta( $this->product_id, 'bkap_outlook_event_uids_ids', true );
			} elseif ( isset( $user->roles[0] ) && 'tour_operator' === $user->roles[0] ) {
				$event_uids = get_the_author_meta( 'tours_event_uids_ids', $this->user_id );
			} else {
				$event_uids = get_option( 'bkap_outlook_event_uids_ids' );
			}

			$this->bkap_outlook_graph();

			if ( '' !== $event_uid ) {
				$event_uid = array( $event_uid );
			} else {

				if ( is_array( $event_uids ) && count( $event_uids ) > 0 ) {
					if ( isset( $item_id ) && array_key_exists( $item_id, $event_uids ) ) {

						$event_id  = $event_uids[ $item_id ];
						$ids       = array();
						$event_ids = explode( ',', $event_id );

						foreach ( $event_ids as $event_key => $event_value ) {
							if ( $item_number < 0 ) {
								$ids[] = $event_value;
							} else {
								if ( $item_number == $event_key ) {
									$ids[] = $event_value;
									break;
								}
							}
						}
						$event_uid = $ids;
					} else {
						$event_uid = array();
					}
				}
			}

			$calendar_id = $this->bkap_calendar_id();

			if ( ! empty( $event_uid ) && $calendar_id != '' ) {
				try {
					foreach ( $event_uid as $uid ) {
						$calendar_id = $this->bkap_calendar_id();
						$url         = '/me/calendars/' . $calendar_id . '/events/' . $uid;
						$response    = $this->graph->createRequest( 'DELETE', $url )
							->setReturnType( Model\Event::class )
							->execute();
					}
				} catch ( Exception $e ) {
					//echo '<pre>'; print_r( $e->getMessage() ); echo '</pre>';
					$this->log( 'Event does\'t found in selected calendar: ' . $e->getMessage() );
				}
			}
		}

		/**
		 * This functions return the selected Calendar ID.
		 *
		 * @since 1.0.0
		 */
		public function bkap_calendar_id() {

			if ( $this->is_product ) {
				$calendar_id = get_post_meta( $this->product_id, '_bkap_outlook_calendar_id', true );				
			} else {
				$calendar_id = get_option( 'bkap_outlook_calendar_id', '' );
			}

			return $calendar_id;
		}

		/**
		 * This functions will make the OAuth connection if the Client ID and Secret is set.
		 *
		 * @since 1.0.0
		 */
		public function bkap_outlook_connect() {

			$data = $this->bkap_outlook_client_id_secret_data();

			if ( '' !== $data['client_id'] && '' !== $data['client_secret'] ) {

				$provider = new TheNetworg\OAuth2\Client\Provider\Azure(
					array(
						'clientId'               => $data['client_id'],
						'clientSecret'           => $data['client_secret'],
						'redirectUri'            => admin_url(),
						'scopes'                 => array( 'openid' ), // Optional.
						'defaultEndPointVersion' => '2.0', // Optional.
					)
				);

				// Set to use v2 API, skip the line or set the value to Azure::ENDPOINT_VERSION_1_0 if willing to use v1 API.
				$provider->defaultEndPointVersion = TheNetworg\OAuth2\Client\Provider\Azure::ENDPOINT_VERSION_2_0;
				$base_graph_uri                   = $provider->getRootMicrosoftGraphUri( null );
				$provider->scope                  = 'openid profile email offline_access Calendars.ReadWrite ' . $base_graph_uri . '/User.Read';
				$this->provider                   = $provider;

				return true;
			} else {
				return false;
			}
		}

		/**
		 * This functions prepares the Authorization URL.
		 *
		 * @since 1.0.0
		 */
		public function bkap_authorization_url() {

			$query = str_replace( admin_url(), '', $this->bkap_get_redirect_uri() );

			if ( is_null( $this->provider ) ) {
				$authorization_url = '';
			} else {
				$authorization_url = $this->provider->getAuthorizationUrl(
					array(
						'scope' => $this->provider->scope,
						'state' => $query,
					)
				);
			}

			return $authorization_url;
		}

		/**
		 * This functions prepares the logout url.
		 *
		 * @since 1.0.0
		 */
		public function bkap_logout_url() {

			if ( $this->is_product ) {
				$product_edit_url = get_edit_post_link( $this->product_id );
				$redirect_args    = array( 'bkap_outlook_logout' => $this->product_id );
				$logout_url       = add_query_arg( $redirect_args, $product_edit_url );
			} else {
				$redirect_args = array(
					'page'                => 'woocommerce_booking_page',
					'action'              => 'calendar_sync_settings',
					'section'             => 'outlook_calendar',
					'bkap_outlook_logout' => 0,
				);
				$logout_url    = add_query_arg( $redirect_args, admin_url( '/admin.php?' ) );
			}

			return $logout_url;
		}

		/**
		 * This will return the uri to which the user will be redirected after consent screen.
		 *
		 * @since 1.0.0
		 */
		public function bkap_get_redirect_uri() {

			$redirect_uri = '';
			if ( $this->is_product ) {
				$query_args   = array(
					'post'               => $this->product_id,
					'action'             => 'edit',
					'bkap-outlook-oauth' => $this->product_id,
				);
				$redirect_uri = add_query_arg( $query_args, admin_url( 'post.php' ) );

				if ( ! is_admin() && current_user_can( 'dokan_edit_product' ) && function_exists( 'dokan_edit_product_url' ) ) {
					$query_args   = array( 'bkap-outlook-oauth' => $this->product_id );
					$redirect_uri = add_query_arg( $query_args, dokan_edit_product_url( $this->product_id ) );
				}
			} else {
				$query_args   = array(
					'page'               => 'woocommerce_booking_page',
					'action'             => 'calendar_sync_settings',
					'section'            => 'outlook_calendar',
					'bkap-outlook-oauth' => '1',
				);
				$redirect_uri = add_query_arg( $query_args, admin_url( 'admin.php' ) );
			}

			return $redirect_uri;
		}

		/**
		 * This function is fetching the access token and its data when user redirect to the redirect uri after the consent.
		 *
		 * @since 1.0.0
		 */
		public function bkap_outlook_oauth_redirect() {

			$this->bkap_outlook_connect();

			$token        = $this->provider->getAccessToken(
				'authorization_code',
				array(
					'scope' => $this->provider->scope,
					'code'  => $_GET['code'],
				)
			);
			$access_token = $token->getToken();
			try {
				$user_id    = get_current_user_id();
				$product_id = $this->product_id; // phpcs:ignore

				if ( $this->is_product ) {
					set_transient( 'bkap_outlook_access_token_' . $this->product_id, $access_token, 3500 );
					update_post_meta( $this->product_id, '_bkap_outlook_token_data', $token );
				} else {
					set_transient( 'bkap_outlook_access_token', $access_token, 3500 );
					update_option( 'bkap_outlook_token_data', $token );
				}
			} catch ( Exception $e ) {
				$this->log( 'Error while doing OAuth: ' . $e->getMessage() );
			}

			if ( ! empty( $access_token ) ) {
				$status = 'success';
			} else {
				$status        = 'fail';
				$error_message = sprintf( __( 'Outlook OAuth failed with "%1$s", "%2$s"', 'bkap-outlook-calendar' ), isset( $_GET['error'] ) ? $_GET['error'] : '', isset( $_GET['error_description'] ) ? $_GET['error_description'] : '' ); // phpcs:ignore
				$this->log( $error_message );
			}

			// Redirecting user to appropriate page.
			if ( $this->product_id ) {
				$redirect_args = array(
					'post'                    => $this->product_id,
					'action'                  => 'edit',
					'bkap_outlook_con_status' => $status,
				);
				$url           = add_query_arg( $redirect_args, admin_url( '/post.php?' ) );

				if ( ! is_admin() && current_user_can( 'dokan_edit_product' ) && function_exists( 'dokan_edit_product_url' ) ) {
					$query_args = array( 'bkap_outlook_con_status' => $status );
					$url        = add_query_arg( $query_args, dokan_edit_product_url( $this->product_id ) );
				}
			} else {
				$redirect_args = array(
					'page'                    => 'woocommerce_booking_page',
					'action'                  => 'calendar_sync_settings',
					'section'                 => 'outlook_calendar',
					'bkap_outlook_con_status' => $status,
				);
				$url           = add_query_arg( $redirect_args, admin_url( '/admin.php?' ) );
			}

			wp_safe_redirect( $url );
			exit;
		}

		/**
		 * This function will be called on the logout. Token and Token Data will be deleted upon the logout.
		 *
		 * @since 1.0.0
		 */
		public function bkap_outlook_oauth_logout() {

			if ( $this->is_product ) {
				delete_post_meta( $this->product_id, '_bkap_outlook_calendar_id' );
				delete_post_meta( $this->product_id, '_bkap_outlook_token_data' );
				delete_transient( 'bkap_outlook_access_token_' . $this->product_id );
				$this->log( 'Product : Left the Outlook Calendar App. successfully - ' . $this->product_id );
			} else {
				delete_option( 'bkap_outlook_calendar_id' );
				delete_option( 'bkap_outlook_token_data' );
				delete_transient( 'bkap_outlook_access_token' );
				$this->log( 'Global : Left the Outlook Calendar App. successfully' );
			}

			return true;
		}

		/**
		 * This will set the Access Token information in transient.
		 *
		 * @param string $access_token Access Token.
		 * @since 1.0.0
		 */
		public function bkap_set_access_token( $access_token ) {

			$global_product = ( $this->is_product ) ? '_' . $this->product_id : '';
			set_transient( 'bkap_outlook_access_token' . $global_product, $access_token, 3600 );
		}

		/**
		 * This will return the the Access Token information.
		 *
		 * @since 1.0.0
		 */
		public function bkap_get_access_token() {

			$product_global = ( $this->is_product ) ? '_' . $this->product_id : '';
			$access_token   = get_transient( 'bkap_outlook_access_token' . $product_global );

			return $access_token;
		}

		/**
		 * This will return the Token Data information.
		 *
		 * @since 1.0.0
		 */
		public function bkap_get_token_data() {

			if ( $this->is_product ) {
				$token_data = get_post_meta( $this->product_id, '_bkap_outlook_token_data', true );
			} else {
				$token_data = get_option( 'bkap_outlook_token_data' );
			}

			return $token_data;
		}

		/**
		 * This function checks if the Client ID and Client Secret is set or not.
		 *
		 * @since 1.0.0
		 */
		public function bkap_outlook_client_id_secret_data() {

			$data = array(
				'client_id'     => '',
				'client_secret' => '',
			);

			if ( $this->is_product ) {
				$status = get_post_meta( $this->product_id, '_bkap_outlook_calendar', '' );
				if ( '' !== $status ) {
					$oauth_settings['client_id']     = get_post_meta( $this->product_id, '_bkap_outlook_calendar_client_id', true );
					$oauth_settings['client_secret'] = get_post_meta( $this->product_id, '_bkap_outlook_calendar_client_secret', true );
				}
			} else {
				$oauth_settings['client_id']     = get_option( 'bkap_outlook_calendar_client_key', '' );
				$oauth_settings['client_secret'] = get_option( 'bkap_outlook_calendar_client_secret', '' );
				// if we are storing in two options then change this.
			}

			if ( isset( $oauth_settings ) && ! empty( $oauth_settings['client_id'] ) && ! empty( $oauth_settings['client_secret'] ) ) {
				$data['client_id']     = $oauth_settings['client_id'];
				$data['client_secret'] = $oauth_settings['client_secret'];
			}

			return $data;
		}

		/**
		 * Used to log messages in the bkap-log file.
		 *
		 * @param string $message Message to be added in log file.
		 * @since 1.0.0
		 */
		public function log( $message = '' ) {

			if ( $message ) {
				$to_put = '<b>[' . date_i18n( $this->datetime_format, $this->local_time ) . ']</b> ' . $message;
				// Prevent multiple messages with same text and same timestamp.
				if ( ! file_exists( $this->log_file ) || strpos( @file_get_contents( $this->log_file ), $to_put ) === false ) {
					@file_put_contents( $this->log_file, $to_put . chr( 10 ) . chr( 13 ), FILE_APPEND );
				}
			}
		}
	}
}
