<?php

/**
 * The admin-specific functionality of the plugin.
 *
 * @link       http://www.tychesoftwares.com/
 * @since      1.0.0
 *
 * @package    Bkap_Outlook_Calendar
 * @subpackage Bkap_Outlook_Calendar/admin
 */

/**
 * The admin-specific functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the admin-specific stylesheet and JavaScript.
 *
 * @package    Bkap_Outlook_Calendar
 * @subpackage Bkap_Outlook_Calendar/admin
 * @author     Tyche Softwares <vishal@tychesoftwares.com>
 */
class Bkap_Outlook_Calendar_Admin {

	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $plugin_name    The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 * @param      string $plugin_name       The name of this plugin.
	 * @param      string $version    The version of this plugin.
	 */
	public function __construct( $plugin_name, $version ) {

		$this->plugin_name = $plugin_name;
		$this->version     = $version;
	}

	/**
	 * Settings Link on Plugin meta.
	 *
	 * @param array $links Plugin Meta links.
	 * @since    1.0.0
	 */
	public function bkap_outlook_calendar_plugin_settings_link( $links ) {

		$setting_link['settings'] = '<a href="' . esc_url( get_admin_url( null, 'admin.php?page=woocommerce_booking_page&action=calendar_sync_settings&section=outlook_calendar' ) ) . '">Settings</a>';
		$links                    = $setting_link + $links;
		return $links;
	}

	/**
	 * This function will show notice if WooCommerce and Booking & Appointment Plugin for WooCommerce is not active.
	 *
	 * @since    1.0.0
	 */
	public function bkap_outlook_calendar_require_plugin_notice() {

		$page = get_post_type();
		if ( ! in_array( $page, array( 'page', 'post' ), true ) ) {
			$msg = __( 'Outlook Calendar Sync Addon is enabled but not effective. It requires Booking & Appointment Plugin for WooCommerce in order to work.', 'bkap-outlook-calendar' );
			printf( '<div class="error"><p>%s</p></div>', esc_html( $msg ) );
		}
	}

	/**
	 * OAuth Redirect for Connecting and getting Access Token.
	 * Also, Handled the logout action by removing the data.
	 *
	 * @since    1.0.0
	 */
	public function bkap_outlook_oauth_redirect() {

		if ( isset( $_GET['code'] ) && '' !== $_GET['code'] ) { // phpcs:ignore

			if ( isset( $_GET['state'] ) ) { // phpcs:ignore

				$state = $_GET['state'];
				if ( strpos( $state, 'outlook_calendar') !== false ) {
					$product_id = 0;
				} else {
					$explode    = explode( '&', $state );
					$explode    = explode( '=', $explode[0] );
					$product_id = (int) $explode[1];
				}
				$user_id            = get_current_user_id();
				$bkap_outlook_oauth = new BKAP_Outlook_Calendar_OAuth( $product_id, $user_id );
				$bkap_outlook_oauth->bkap_outlook_oauth_redirect();
			}
		}

		if ( isset( $_GET['bkap_outlook_logout'] ) ) { // phpcs:ignore
			if ( '' != $_GET['bkap_outlook_logout'] ) { // phpcs:ignore
				$user_id         = get_current_user_id();
				$product_id      = ( 0 != $_GET['bkap_outlook_logout'] ) ? $_GET['bkap_outlook_logout'] : 0; // phpcs:ignore
				$bkap_oauth_gcal = new BKAP_Outlook_Calendar_OAuth( $product_id, $user_id );
				$bkap_oauth_gcal->bkap_outlook_oauth_logout();
			}
		}
	}

	/**
	 * Displaying the notice based on the successful and failed connection.
	 *
	 * @since    1.0.0
	 */
	public function bkap_outlook_success_fail_notice() {
		if ( isset( $_GET['bkap_outlook_con_status'] ) ) { // phpcs:ignore
			$status = $_GET['bkap_outlook_con_status']; // phpcs:ignore
			switch ( $status ) {
				case 'success':
					$message = __( 'Successfully connected to Outlook.', 'bkap-outlook-calendar' );
					$class   = 'notice notice-success';
					break;
				case 'fail':
					$uploads     = wp_upload_dir(); // Set log file location.
					$uploads_dir = isset( $uploads['basedir'] ) ? $uploads['basedir'] . '/' : WP_CONTENT_DIR . '/uploads/';
					$log_file    = $uploads_dir . 'bkap-log.txt';
					/* translators: %s: Bkap Log file url. */
					$message = sprintf( __( 'Failed to connect to your account, please try again, if the problem persists, please check the log in the %s file and see what is happening or please contact Support team.', 'bkap-outlook-calendar' ), $log_file );
					$class   = 'notice notice-error';
					break;
			}

			printf( '<div class="%s"><p>%s</p></div>', esc_attr( $class ), esc_html( $message ) );
		}

		if ( isset( $_GET['bkap_outlook_logout'] ) ) { // phpcs:ignore
			$message = __( 'Outlook Calendar Account disconnected successfully!', 'bkap-outlook-calendar' );
			$class   = 'notice notice-success';
			printf( '<div class="%s"><p>%s</p></div>', esc_attr( $class ), esc_html( $message ) );
		}
	}

	/**
	 * Adding Outlook Calendar Link on Integration page.
	 *
	 * @param string $section Section.
	 * @since    1.0.0
	 */
	public function bkap_outlook_calendar_link( $section ) {

		$outlook = ( 'outlook_calendar' === $section ) ? 'current' : '';
		?>
		<li>
		| <a href="admin.php?page=woocommerce_booking_page&action=calendar_sync_settings&section=outlook_calendar"class="<?php echo esc_attr( $outlook ); ?>"><?php esc_html_e( 'Outlook Calendar Sync', 'bkap-outlook-calendar' ); ?></a>
		</li>
		<?php
	}

	/**
	 * Adding Settings API on the Outlook Calendar page.
	 *
	 * @param string $section Section.
	 * @since    1.0.0
	 */
	public function bkap_outlook_calendar_options( $section ) {

		if ( 'outlook_calendar' === $section ) {
			print( '<div id="content"><form method="post" action="options.php">' );
			settings_errors();
			settings_fields( 'bkap_outlook_calendar_settings' );
			do_settings_sections( 'bkap_outlook_calendar_settings_page' );
			submit_button( __( 'Save Settings', 'bkap-outlook-calendar' ), 'primary', 'save', true );
			print( '</form></div>' );
		}
	}

	/**
	 * Registering the option to store the data of Outlook Calendar Options.
	 *
	 * @since    1.0.0
	 */
	public function bkap_outlook_calendar_register_setting() {

		register_setting(
			'bkap_outlook_calendar_settings',
			'bkap_outlook_calendar_event_location'
		);
		register_setting(
			'bkap_outlook_calendar_settings',
			'bkap_outlook_calendar_event_summary'
		);
		register_setting(
			'bkap_outlook_calendar_settings',
			'bkap_outlook_calendar_event_description'
		);

		register_setting(
			'bkap_outlook_calendar_settings',
			'bkap_outlook_calendar_integration'
		);
		register_setting(
			'bkap_outlook_calendar_settings',
			'bkap_outlook_calendar_client_key'
		);

		register_setting(
			'bkap_outlook_calendar_settings',
			'bkap_outlook_calendar_client_secret'
		);

		register_setting(
			'bkap_outlook_calendar_settings',
			'bkap_outlook_calendar_id'
		);
	}

	/**
	 * Adding the Section and Fields of the Outlook Calendar Options.
	 *
	 * @param string $section Section of Integration.
	 * @since    1.0.0
	 */
	public function bkap_outlook_calendar_settings( $section ) {

		if ( isset( $_GET['action'] ) && 'calendar_sync_settings' === $_GET['action'] && isset( $_GET['section'] ) && 'outlook_calendar' === $_GET['section'] ) {

			add_settings_section(
				'bkap_outlook_sync_general_settings_section',       // ID used to identify this section and with which to register options.
				__( 'General Settings', 'bkap-outlook-calendar' ),     // Title to be displayed on the administration page.
				array( 'Bkap_Outlook_Calendar_Settings', 'bkap_outlook_sync_general_settings_callback' ),      // Callback used to render the description of the section.
				'bkap_outlook_calendar_settings_page'               // Page on which to add this section of options.
			);

			add_settings_field(
				'bkap_outlook_calendar_event_location',
				__( 'Event Location', 'bkap-outlook-calendar' ),
				array( 'Bkap_Outlook_Calendar_Settings', 'bkap_outlook_calendar_event_location_callback' ),
				'bkap_outlook_calendar_settings_page',
				'bkap_outlook_sync_general_settings_section',
				array( __( '<br>Enter the text that will be used as location field in event of the Calendar. If left empty, website description is sent instead. <br><i>Note: You can use ADDRESS and CITY placeholders which will be replaced by their real values.</i>', 'bkap-outlook-calendar' ) )
			);

			add_settings_field(
				'bkap_outlook_calendar_event_summary',
				__( 'Event summary (name)', 'bkap-outlook-calendar' ),
				array( 'Bkap_Outlook_Calendar_Settings', 'bkap_outlook_calendar_event_summary_callback' ),
				'bkap_outlook_calendar_settings_page',
				'bkap_outlook_sync_general_settings_section'
			);

			add_settings_field(
				'bkap_outlook_calendar_event_description',
				__( 'Event Description', 'bkap-outlook-calendar' ),
				array( 'Bkap_Outlook_Calendar_Settings', 'bkap_outlook_calendar_event_description_callback' ),
				'bkap_outlook_calendar_settings_page',
				'bkap_outlook_sync_general_settings_section',
				array( '<br>For the above 2 fields, you can use the following placeholders which will be replaced by their real values:&nbsp;SITE_NAME, CLIENT, PRODUCT_NAME, PRODUCT_WITH_QTY, RESOURCE, ORDER_DATE_TIME, ORDER_DATE, ORDER_NUMBER, PRICE, PHONE, NOTE, ADDRESS, EMAIL (Client\'s email), ZOOM_MEETING', 'bkap-outlook-calendar' )
			);

			add_settings_section(
				'bkap_outlook_calendar_settings_section', // ID used to identify this section and with which to register options.
				__( 'Outlook Calendar Sync Settings', 'bkap-outlook-calendar' ), // Title to be displayed on the administration page.
				array( 'Bkap_Outlook_Calendar_Settings', 'bkap_outlook_calendar_settings_callback' ), // Callback used to render the description of the section.
				'bkap_outlook_calendar_settings_page' // Page on which to add this section of options.
			);

			add_settings_field(
				'bkap_outlook_calendar_integration',
				__( 'Integration Mode', 'bkap-outlook-calendar' ),
				array( 'Bkap_Outlook_Calendar_Settings', 'bkap_outlook_calendar_integration_callback' ),
				'bkap_outlook_calendar_settings_page',
				'bkap_outlook_calendar_settings_section',
				array()
			);

			add_settings_field(
				'bkap_outlook_calendar_instructions',
				__( 'Instructions', 'bkap-outlook-calendar' ),
				array( 'Bkap_Outlook_Calendar_Settings', 'bkap_outlook_calendar_instructions_callback' ),
				'bkap_outlook_calendar_settings_page',
				'bkap_outlook_calendar_settings_section',
				array( 'class' => 'bkap_outlook_calendar_instructions' )
			);

			$outlook         = new BKAP_Outlook_Calendar_OAuth();
			$outlook_connect = $outlook->bkap_outlook_connect();
			$connect_link    = $outlook->bkap_authorization_url();
			$redirect_uri    = $outlook->bkap_get_redirect_uri();
			$calendars       = $outlook->bkap_outlook_calendar_list();
			$readonly        = ( ! empty( $calendars ) ) ? true : false;

			add_settings_field(
				'bkap_outlook_calendar_client_key',
				__( 'Client ID', 'bkap-outlook-calendar' ),
				array( 'Bkap_Outlook_Calendar_Settings', 'bkap_outlook_calendar_client_key_callback' ),
				'bkap_outlook_calendar_settings_page',
				'bkap_outlook_calendar_settings_section',
				array( 'readonly' => $readonly )
			);

			add_settings_field(
				'bkap_outlook_calendar_client_secret',
				__( 'Client Secret', 'bkap-outlook-calendar' ),
				array( 'Bkap_Outlook_Calendar_Settings', 'bkap_outlook_calendar_client_secret_callback' ),
				'bkap_outlook_calendar_settings_page',
				'bkap_outlook_calendar_settings_section',
				array( 'readonly' => $readonly )
			);

			add_settings_field(
				'bkap_outlook_calendar_redirect_uri',
				__( 'Redirect URI', 'bkap-outlook-calendar' ),
				array( 'Bkap_Outlook_Calendar_Settings', 'bkap_outlook_calendar_redirect_uri_callback' ),
				'bkap_outlook_calendar_settings_page',
				'bkap_outlook_calendar_settings_section',
				array( 'label_for' => 'bkap_outlook_calendar_redirect_uri', 'redirect_uri' => admin_url()/* $redirect_uri */ )
			);

			if ( empty( $calendars ) ) {
				add_settings_field(
					'bkap_outlook_calendar_connect',
					'',
					array( 'Bkap_Outlook_Calendar_Settings', 'bkap_outlook_calendar_connect_callback' ),
					'bkap_outlook_calendar_settings_page',
					'bkap_outlook_calendar_settings_section',
					array( 'label_for' => 'bkap_outlook_calendar_redirect_uri', 'connect_link' => $connect_link )
				);
			} else {
				add_settings_field(
					'bkap_outlook_calendar_id',
					'Calendar to be used',
					array( 'Bkap_Outlook_Calendar_Settings', 'bkap_outlook_calendar_id_callback' ),
					'bkap_outlook_calendar_settings_page',
					'bkap_outlook_calendar_settings_section',
					array( 'label_for' => 'bkap_outlook_calendar_id', 'calendars' => $calendars )
				);

				$logout = $outlook->bkap_logout_url();

				add_settings_field(
					'bkap_outlook_calendar_logout',
					'',
					array( 'Bkap_Outlook_Calendar_Settings', 'bkap_outlook_calendar_logout_callback' ),
					'bkap_outlook_calendar_settings_page',
					'bkap_outlook_calendar_settings_section',
					array( 'label_for' => 'bkap_outlook_calendar_id', 'logout_link' => $logout )
				);
			}
		}
	}

	/**
	 * Adding the Outlook Calendar Settings.
	 *
	 * @param int   $product_id Product ID.
	 * @param array $booking_settings Booking Settings.
	 * @since    1.0.0
	 */
	public function bkap_outlook_calendar_product_settings( $product_id, $booking_settings ) {

		$user_id         = get_current_user_id();
		$outlook         = new BKAP_Outlook_Calendar_OAuth( $product_id, $user_id );
		$outlook_connect = $outlook->bkap_outlook_connect();
		$connect_link    = $outlook->bkap_authorization_url();
		$redirect_uri    = $outlook->bkap_get_redirect_uri();
		$calendars       = $outlook->bkap_outlook_calendar_list();

		$enable_outlook_calendar = '';
		if ( isset( $booking_settings['bkap_outlook_calendar'] ) && 'on' === $booking_settings['bkap_outlook_calendar'] ) {
			$enable_outlook_calendar = 'checked';
		}

		$client_id = '';
		if ( isset( $booking_settings['bkap_outlook_calendar_client_id'] ) && '' !== $booking_settings['bkap_outlook_calendar_client_id'] ) {
			$client_id = $booking_settings['bkap_outlook_calendar_client_id'];
		}

		$client_secret = '';
		if ( isset( $booking_settings['bkap_outlook_calendar_client_secret'] ) && '' !== $booking_settings['bkap_outlook_calendar_client_secret'] ) {
			$client_secret = $booking_settings['bkap_outlook_calendar_client_secret'];
		}

		$calendar_id = '';
		if ( isset( $booking_settings['bkap_outlook_calendar_id'] ) && '' !== $booking_settings['bkap_outlook_calendar_id'] ) {
			$calendar_id = $booking_settings['bkap_outlook_calendar_id'];
		}
		?>
		<!-- Outlook Calendar -->
		<button type="button" class="bkap-integrations-accordion"><b><?php esc_html_e( 'Outlook Calendar', 'bkap-outlook-calendar' ); ?></b></button>
		<div class="bkap_google_sync_settings_content bkap_integrations_panel">
			<table class='form-table bkap-form-table'>
				<tr>
					<th>
						<?php esc_html_e( 'Enable Outlook Calendar', 'bkap-outlook-calendar' ); ?>
					</th>
					<td>
						<label class="bkap_switch">
							<input id="bkap_outlook_calendar" name= "bkap_outlook_calendar" type="checkbox" <?php esc_attr_e( $enable_outlook_calendar ); ?>/>
						<div class="bkap_slider round"></div> 
					</td>
					<td>
						<img class="help_tip" width="16" height="16" data-tip="<?php esc_attr_e( 'Enable Outlook Calendar.', 'bkap-outlook-calendar' ); ?>" src="<?php echo plugins_url(); ?>/woocommerce/assets/images/help.png"/>
					</td>
				</tr>

				<tr>
					<th >
						<label for="bkap_outlook_calendar_client_id"><?php esc_html_e( 'Client ID', 'bkap-outlook-calendar' ); ?></label>
					</th>
					<td>
						<input id="bkap_outlook_calendar_client_id" name= "bkap_outlook_calendar_client_id" value="<?php echo $client_id; ?>" size="40" type="text" />
					</td>
					<td>
						<img class="help_tip" width="16" height="16" data-tip="<?php _e( 'Client ID.', 'bkap-outlook-calendar' ); ?>" src="<?php echo plugins_url(); ?>/woocommerce/assets/images/help.png"/>
					</td>
				</tr>

				<tr>
					<th>
						<label for="bkap_outlook_calendar_client_secret"><?php esc_html_e( 'Client Secret', 'bkap-outlook-calendar' ); ?></label>
					</th>
					<td>
						<input id="bkap_outlook_calendar_client_secret" name= "bkap_outlook_calendar_client_secret" value="<?php echo $client_secret; ?>" size="40" type="text" />
					</td>
					<td>
						<img class="help_tip" width="16" height="16" data-tip="<?php _e( 'Client Secret.', 'bkap-outlook-calendar' ); ?>" src="<?php echo plugins_url(); ?>/woocommerce/assets/images/help.png"/>
					</td>
				</tr>

				<tr>
					<th >
						<label for="bkap_outlook_calendar_client_id"><?php esc_html_e( 'Redirect URI', 'bkap-outlook-calendar' ); ?></label>
					</th>
					<td>
					<input readonly="readonly" class="bkap-outlook-redirect-uri" style="width:89%;" type="text" id="bkap-outlook-redirect-uri" name="bkap-outlook-redirect-uri" value="<?php echo admin_url(); ?>">
						<em><a href="javascript:void(0)" style="border: 1px solid #eee;padding: 4px;" id="bkap_copy_outlook_redirect_uri" data-selector-to-copy="#bkap-outlook-redirect-uri" data-tip="Copied!" class="dashicons dashicons-admin-page bkap-outlook-url-copy-to-clipboard"></a><span id="bkap_outlook_redirect_uri_copied"></span></em>
					</td>
					<td>
					</td>
				</tr>

				<?php if ( ! empty( $calendars ) ) { ?>
					<tr>
						<th>
							<label for="bkap_outlook_calendar_id"><?php esc_html_e( 'Calendar to be used', 'bkap-outlook-calendar' ); ?></label>
						</th>
						<td>
						<select name="bkap_outlook_calendar_id" id="bkap_outlook_calendar_id">
							<?php
							foreach ( $calendars as $key => $value ) {
								$selected = '';
								if ( $value['id'] == $calendar_id ) {
									$selected = 'selected';
								}
								echo '<option value="' . $value['id'] . '" ' . $selected . '>' . $value['name'] . '</option>';
							}
							?>
						</select>
						</td>
						<td>
							<img class="help_tip" width="16" height="16" data-tip="<?php _e( 'Booking Event will be created in the selected Calendar ID', 'bkap-outlook-calendar' ); ?>" src="<?php echo plugins_url(); ?>/woocommerce/assets/images/help.png"/>
						</td>
					</tr>

					<?php $logout = $outlook->bkap_logout_url(); ?>

					<tr>
						<th></th>
						<td>
							<?php echo sprintf( '<a style="" href="%s" class="button-primary">%s</a>', $logout, __( 'Disconnect', 'bkap-outlook-calendar' ) ); ?>
						</td>
						<td>
						</td>
					</tr>
					<?php
				} else {
					$disabled = 'disabled="disabled"';
					if ( '' !== $client_id && '' !== $client_secret ) {
						$disabled = '';
					}
					?>
					<tr>
						<th></th>
						<td>
							<?php echo sprintf( '<a style="" href="%s" class="button-primary" %s>%s</a>', $connect_link, $disabled, __( 'Connect to Outlook', 'bkap-outlook-calendar' ) ); ?>
						</td>
						<td>
						</td>
					</tr>
				<?php } ?>

			</table>
			<br>
		</div>
		<?php
	}

	/**
	 * Collecting the outlook calendar data for saving it individually and in serialized.
	 *
	 * @param array $integration_data All Booking Data.
	 * @param int   $product_id Product ID.
	 * @param obj   $clean_integration_data Data being return from the js.
	 *
	 * @since    1.0.0
	 */
	public function bkap_save_outlook_calendar_product_settings( $integration_data, $product_id, $clean_integration_data ) {

		$integration_data['_bkap_outlook_calendar']               = isset( $clean_integration_data->bkap_outlook_calendar ) ? $clean_integration_data->bkap_outlook_calendar : '';
		$integration_data['_bkap_outlook_calendar_client_id']     = isset( $clean_integration_data->bkap_outlook_calendar_client_id ) ? $clean_integration_data->bkap_outlook_calendar_client_id : '';
		$integration_data['_bkap_outlook_calendar_client_secret'] = isset( $clean_integration_data->bkap_outlook_calendar_client_secret ) ? $clean_integration_data->bkap_outlook_calendar_client_secret : '';
		$integration_data['_bkap_outlook_calendar_id']            = isset( $clean_integration_data->bkap_outlook_calendar_id ) ? $clean_integration_data->bkap_outlook_calendar_id : '';

		return $integration_data;
	}

	/**
	 * Saving the outlook calendar settings data in Booking Settings.
	 *
	 * @param array $updated_settings Booking Settings.
	 * @param array $integration_data Array of collected settings.
	 *
	 * @since    1.0.0
	 */
	public function bkap_update_serialized_post_meta_outlook_calendar( $updated_settings, $integration_data ) {

		if ( isset( $integration_data['_bkap_outlook_calendar'] ) ) {
			$updated_settings['bkap_outlook_calendar'] = $integration_data['_bkap_outlook_calendar'];
		} else {
			$updated_settings['bkap_outlook_calendar'] = '';
		}

		if ( isset( $integration_data['_bkap_outlook_calendar_client_id'] ) ) {
			$updated_settings['bkap_outlook_calendar_client_id'] = $integration_data['_bkap_outlook_calendar_client_id'];
		} else {
			$updated_settings['bkap_outlook_calendar_client_id'] = '';
		}

		if ( isset( $integration_data['_bkap_outlook_calendar_client_secret'] ) ) {
			$updated_settings['bkap_outlook_calendar_client_secret'] = $integration_data['_bkap_outlook_calendar_client_secret'];
		} else {
			$updated_settings['bkap_outlook_calendar_client_secret'] = '';
		}

		if ( isset( $integration_data['_bkap_outlook_calendar_id'] ) ) {
			$updated_settings['bkap_outlook_calendar_id'] = $integration_data['_bkap_outlook_calendar_id'];
		} else {
			$updated_settings['bkap_outlook_calendar_id'] = '';
		}
		return $updated_settings;
	}

	/**
	 * Register the script file being used to collect the outlook calendar data using js.
	 *
	 * @param int $post_id Product ID.
	 * @since    1.0.0
	 */
	public function bkap_outlook_calendar_metabox_js( $post_id ) {
		wp_enqueue_script( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'js/bkap-outlook-calendar-admin.js', array( 'jquery' ), $this->version, false );
	}

	/**
	 * Adding Submenu for Outlook Calendar License.
	 *
	 * @since    1.0.0
	 */
	public function bkap_outlook_calendar_license_menu() {
		$page = add_submenu_page(
			'edit.php?post_type=bkap_booking',
			__( 'Activate Outlook Calendar License', 'bkap-outlook-calendar' ),
			__( 'Activate Outlook Calendar License', 'bkap-outlook-calendar' ),
			'manage_woocommerce',
			'outlook_calendar_license_page',
			array( 'Bkap_Outlook_Calendar_License', 'bkap_get_edd_outlook_calendar_license_page' )
		);
	}

	/**
	 * Register the stylesheets for the admin area.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_styles() {

		/**
		 * This function is provided for demonstration purposes only.
		 *
		 * An instance of this class should be passed to the run() function
		 * defined in Bkap_Outlook_Calendar_Loader as all of the hooks are defined
		 * in that particular class.
		 *
		 * The Bkap_Outlook_Calendar_Loader will then create the relationship
		 * between the defined hooks and the functions defined in this
		 * class.
		 */

		wp_enqueue_style( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'css/bkap-outlook-calendar-admin.css', array(), $this->version, 'all' );

	}

	/**
	 * Register the JavaScript for the admin area.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_scripts() {

		/**
		 * This function is provided for demonstration purposes only.
		 *
		 * An instance of this class should be passed to the run() function
		 * defined in Bkap_Outlook_Calendar_Loader as all of the hooks are defined
		 * in that particular class.
		 *
		 * The Bkap_Outlook_Calendar_Loader will then create the relationship
		 * between the defined hooks and the functions defined in this
		 * class.
		 */

		wp_enqueue_script( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'js/bkap-outlook-calendar-admin.js', array( 'jquery' ), $this->version, false );

	}

}
