<?php

/**
 * The file that defines the core plugin class
 *
 * A class definition that includes attributes and functions used across both the
 * public-facing side of the site and the admin area.
 *
 * @link       http://www.tychesoftwares.com/
 * @since      1.0.0
 *
 * @package    Bkap_Outlook_Calendar
 * @subpackage Bkap_Outlook_Calendar/includes
 */

/**
 * The core plugin class.
 *
 * This is used to define internationalization, admin-specific hooks, and
 * public-facing site hooks.
 *
 * Also maintains the unique identifier of this plugin as well as the current
 * version of the plugin.
 *
 * @since      1.0.0
 * @package    Bkap_Outlook_Calendar
 * @subpackage Bkap_Outlook_Calendar/includes
 * @author     Tyche Softwares <vishal@tychesoftwares.com>
 */
class Bkap_Outlook_Calendar {

	/**
	 * The unique identifier of this plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string    $plugin_name    The string used to uniquely identify this plugin.
	 */
	protected $plugin_name;

	/**
	 * The current version of the plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string    $version    The current version of the plugin.
	 */
	protected $version;

	/**
	 * Define the core functionality of the plugin.
	 *
	 * Set the plugin name and the plugin version that can be used throughout the plugin.
	 * Load the dependencies, define the locale, and set the hooks for the admin area and
	 * the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function __construct() {
		if ( defined( 'BKAP_OUTLOOK_CALENDAR_VERSION' ) ) {
			$this->version = BKAP_OUTLOOK_CALENDAR_VERSION;
		} else {
			$this->version = '1.0.0';
		}
		$this->plugin_name = 'bkap-outlook-calendar';

		$this->bkapoc_define_constants();
		$this->bkapoc_load_dependencies();
		$this->bkapoc_set_locale();

		$status = $this->bkap_is_woocommerce_bkap_activated();
		$this->bkapoc_define_admin_hooks( $status );
		$this->bkapoc_define_public_hooks( $status );

		$this->bkapoc_license();

	}

	/**
	 * Check if WooCommerce and Booking is activated.
	 *
	 * @since 1.10
	 * @return bool $check true is Booking and WooCommerce is active else false.
	 */
	public function bkap_is_woocommerce_bkap_activated() {

		$active_plugins       = get_option( 'active_plugins', array() );
		$site_plugins         = get_site_option( 'active_sitewide_plugins', array() );
		$woocommerce_basename = 'woocommerce-update/woocommerce.php';
		$bkap_basename        = 'woocommerce-booking/woocommerce-booking.php';
		$check                = false;

		if ( ( in_array( $woocommerce_basename, $active_plugins, true ) || isset( $site_plugins[ $woocommerce_basename ] ) ) ) {
			if ( in_array( $bkap_basename, $active_plugins, true ) || isset( $site_plugins[ $bkap_basename ] ) ) {
				return true;
			}
		}
		return $check;
	}

	/**
	 * Defining Constants for the plugin.
	 *
	 * Include the following files that make up the plugin:
	 *
	 * - Bkap_Outlook_Calendar_Loader. Orchestrates the hooks of the plugin.
	 * - Bkap_Outlook_Calendar_i18n. Defines internationalization functionality.
	 * - Bkap_Outlook_Calendar_Admin. Defines all hooks for the admin area.
	 * - Bkap_Outlook_Calendar_Public. Defines all hooks for the public side of the site.
	 *
	 * Create an instance of the loader which will be used to register the hooks
	 * with WordPress.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function bkapoc_define_constants() {

		if ( ! defined( 'BKAPOC_PLUGIN_PATH' ) ) {
			define( 'BKAPOC_PLUGIN_PATH', untrailingslashit( plugin_dir_path( BKAPOC_FILE ) ) );
		}

		if ( ! defined( 'BKAPOC_PLUGIN_URL' ) ) {
			define( 'BKAPOC_PLUGIN_URL', untrailingslashit( plugins_url( '/', BKAPOC_FILE ) ) );
		}

		define( 'BKAPOC_INCLUDE_PATH', BKAPOC_PLUGIN_PATH . '/includes/' );
	}

	/**
	 * Load the required dependencies for this plugin.
	 *
	 * Include the following files that make up the plugin:
	 *
	 * - Bkap_Outlook_Calendar_Loader. Orchestrates the hooks of the plugin.
	 * - Bkap_Outlook_Calendar_i18n. Defines internationalization functionality.
	 * - Bkap_Outlook_Calendar_Admin. Defines all hooks for the admin area.
	 * - Bkap_Outlook_Calendar_Public. Defines all hooks for the public side of the site.
	 *
	 * Create an instance of the loader which will be used to register the hooks
	 * with WordPress.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function bkapoc_load_dependencies() {

		require_once BKAPOC_INCLUDE_PATH . 'plugin-updates/EDD_Plugin_Updater.php';

		require_once BKAPOC_INCLUDE_PATH . 'bkap-outlook-calendar-functions.php';

		/**
		 * The class responsible for orchestrating the actions and filters of the
		 * core plugin.
		 */
		require_once BKAPOC_INCLUDE_PATH . 'class-bkap-outlook-calendar-loader.php';

		/**
		 * The class responsible for defining internationalization functionality
		 * of the plugin.
		 */
		require_once BKAPOC_INCLUDE_PATH . 'class-bkap-outlook-calendar-i18n.php';

		/**
		 * The class responsible for orchestrating the actions and filters of the
		 * core plugin.
		 */
		require_once BKAPOC_INCLUDE_PATH . 'outlook-vendor/class-bkap-outlook-calendar-oauth.php';

		/**
		 * The class responsible for defining all functions that occur in the global settings area.
		 */
		require_once BKAPOC_PLUGIN_PATH . '/admin/class-bkap-outlook-calendar-settings.php';

		/**
		 * The class responsible for defining all actions that occur in the admin area.
		 */
		require_once BKAPOC_PLUGIN_PATH . '/admin/class-bkap-outlook-calendar-admin.php';

		/**
		 * The class responsible for defining all actions that occur in the public-facing
		 * side of the site.
		 */
		require_once BKAPOC_PLUGIN_PATH . '/public/class-bkap-outlook-calendar-public.php';

		/**
		 * The class responsible for defining all actions that occur for the licensing side of the site.
		 */
		require_once BKAPOC_INCLUDE_PATH . 'class-bkap-outlook-calendar-license.php';

		$this->loader = new Bkap_Outlook_Calendar_Loader();

	}

	/**
	 * Define the locale for this plugin for internationalization.
	 *
	 * Uses the Bkap_Outlook_Calendar_i18n class in order to set the domain and to register the hook
	 * with WordPress.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function bkapoc_set_locale() {

		$plugin_i18n = new Bkap_Outlook_Calendar_i18n();

		$this->loader->add_action( 'plugins_loaded', $plugin_i18n, 'load_plugin_textdomain' );

	}

	/**
	 * Register all of the hooks related to the admin area functionality of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function bkapoc_define_admin_hooks( $status ) {

		$plugin_admin = new Bkap_Outlook_Calendar_Admin( $this->get_plugin_name(), $this->get_version() );

		if ( $status ) {
			$this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_styles' );
			$this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_scripts' );
			$this->loader->add_action( 'wp_enqueue_scripts', $plugin_admin, 'enqueue_scripts' );
			$this->loader->add_action( 'bkap_integration_links', $plugin_admin, 'bkap_outlook_calendar_link', 10, 1 );
			$this->loader->add_action( 'bkap_global_integration_settings', $plugin_admin, 'bkap_outlook_calendar_options', 10, 1 );
			$this->loader->add_action( 'admin_init', $plugin_admin, 'bkap_outlook_calendar_settings' );
			$this->loader->add_action( 'bkap_register_setting', $plugin_admin, 'bkap_outlook_calendar_register_setting', 10, 1 );

			$this->loader->add_action( 'admin_init', $plugin_admin, 'bkap_outlook_oauth_redirect', 9 );
			$this->loader->add_action( 'admin_notices', $plugin_admin, 'bkap_outlook_success_fail_notice' );

			$this->loader->add_action( 'after_bkap_load_product_scripts_js', $plugin_admin, 'bkap_outlook_calendar_metabox_js', 10, 1 );
			$this->loader->add_filter( 'bkap_product_integration_data', $plugin_admin, 'bkap_save_outlook_calendar_product_settings', 10, 3 );
			$this->loader->add_filter( 'bkap_update_serialized_post_meta_integration_data', $plugin_admin, 'bkap_update_serialized_post_meta_outlook_calendar', 10, 2 );

			$this->loader->add_action( 'bkap_after_zoom_meeting_settings_product', $plugin_admin, 'bkap_outlook_calendar_product_settings', 10, 2 );

			$this->loader->add_action( 'bkap_add_submenu', $plugin_admin, 'bkap_outlook_calendar_license_menu', 10, 2 );
		} else {
			$this->loader->add_action( 'admin_init', $plugin_admin, 'bkap_outlook_calendar_require_plugin_notice', 10 );
		}
	}

	/**
	 * Register all of the hooks related to the public-facing functionality of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function bkapoc_define_public_hooks( $status ) {

		if ( $status ) {
			$plugin_public = new Bkap_Outlook_Calendar_Public( $this->get_plugin_name(), $this->get_version() );

			$this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_styles' );
			$this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_scripts' );

			// Insert Event to Outlook Calendar.
			$this->loader->add_action( 'after_bkap_create_booking_post', $plugin_public, 'bkap_outlook_calendar_update_order_meta', 10, 6 );
			$this->loader->add_action( 'bkap_update_booking_tables_confirmed_integration', $plugin_public, 'bkap_outlook_calendar_update_booking_tables_confirmed', 10, 6 );
			$this->loader->add_action( 'bkap_restore_trashed_booking', $plugin_public, 'bkap_outlook_calendar_restore_trashed_booking', 10, 6 );
			$this->loader->add_action( 'bkap_insert_event_to_calendar', $plugin_public, 'bkap_outlook_calendar_insert_event_to_calendar', 10, 6 );
			$this->loader->add_action( 'bkap_create_order_event', $plugin_public, 'bkap_outlook_calendar_create_order_booking', 10, 5 );
			$this->loader->add_action( 'bkap_create_booking_event', $plugin_public, 'bkap_outlook_calendar_create_order_booking', 10, 5 );

			// Delete Event from Outlook Calendar.
			$this->loader->add_action( 'bkap_reallocation_of_booking', $plugin_public, 'bkap_outlook_calendar_reallocation_of_booking', 10, 4 );
			$this->loader->add_action( 'bkap_delete_event_from_calendar', $plugin_public, 'bkap_outlook_calendar_delete_event_from_calendar', 10, 3 );
		}
	}

	/**
	 * Licensing Related Code.
	 *
	 * @since    1.0.0
	 */
	private function bkapoc_license() {

		$plugin_public = new Bkap_Outlook_Calendar_License( $this->get_plugin_name(), $this->get_version() );
		$this->loader->add_action( 'admin_init', $plugin_public, 'bkap_edd_sample_register_option', 9, 3 );
		$this->loader->add_action( 'admin_init', $plugin_public, 'bkap_edd_sample_deactivate_license', 10, 3 );
		$this->loader->add_action( 'admin_init', $plugin_public, 'bkap_edd_sample_activate_license', 10, 3 );

	}

	/**
	 * Run the loader to execute all of the hooks with WordPress.
	 *
	 * @since    1.0.0
	 */
	public function run() {
		$this->loader->run();
	}

	/**
	 * The name of the plugin used to uniquely identify it within the context of
	 * WordPress and to define internationalization functionality.
	 *
	 * @since     1.0.0
	 * @return    string    The name of the plugin.
	 */
	public function get_plugin_name() {
		return $this->plugin_name;
	}

	/**
	 * The reference to the class that orchestrates the hooks with the plugin.
	 *
	 * @since     1.0.0
	 * @return    Bkap_Outlook_Calendar_Loader    Orchestrates the hooks of the plugin.
	 */
	public function get_loader() {
		return $this->loader;
	}

	/**
	 * Retrieve the version number of the plugin.
	 *
	 * @since     1.0.0
	 * @return    string    The version number of the plugin.
	 */
	public function get_version() {
		return $this->version;
	}

}
