<?php

/**
 * The public-facing functionality of the plugin.
 *
 * @link       http://www.tychesoftwares.com/
 * @since      1.0.0
 *
 * @package    Bkap_Outlook_Calendar
 * @subpackage Bkap_Outlook_Calendar/public
 */

/**
 * The public-facing functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the public-facing stylesheet and JavaScript.
 *
 * @package    Bkap_Outlook_Calendar
 * @subpackage Bkap_Outlook_Calendar/public
 * @author     Tyche Softwares <vishal@tychesoftwares.com>
 */
class Bkap_Outlook_Calendar_Public {

	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $plugin_name    The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 * @param    string $plugin_name       The name of the plugin.
	 * @param    string $version           The version of this plugin.
	 */
	public function __construct( $plugin_name, $version ) {

		$this->plugin_name = $plugin_name;
		$this->version     = $version;
	}

	/**
	 * This function will be called on the order placed and it will insert the event to the Outlook Calendar.
	 *
	 * @since    1.0.0
	 * @param    int   $item_id       Item ID.
	 * @param    int   $post_id       Product ID.
	 * @param    int   $quantity      Quantity.
	 * @param    array $booking_data  Array of Booking data.
	 * @param    int   $variation_id  Variation ID.
	 * @param    int   $order_id      Order ID.
	 *
	 * @since 1.0.0
	 */
	public function bkap_outlook_calendar_update_order_meta( $item_id, $post_id, $quantity, $booking_data, $variation_id, $order_id ) {

		$product_id        = $post_id;
		$booking_settings  = bkap_setting( $product_id );
		$active_connection = bkap_outlook_calendar_connected( $booking_settings );

		if ( '' !== $active_connection ) {

			$post_id          = ( 'product' === $active_connection ) ? $post_id : 0;
			$user_id          = bkap_get_user_id();
			$outlook_calendar = new BKAP_Outlook_Calendar_OAuth( $post_id, $user_id );

			// check the booking status, if pending confirmation, then do not insert event in the calendar.
			$booking_status = wc_get_order_item_meta( $item_id, '_wapbk_booking_status' );

			if ( ( isset( $booking_status ) && 'pending-confirmation' != $booking_status ) || ( ! isset( $booking_status ) ) ) {

				$event_details = bkap_event_details_upon_order_placed( $post_id, $booking_settings, $order_id, $item_id, $variation_id, $booking_data, $quantity );
				if ( isset( $event_details['hidden_booking_date'] ) ) {

					$time_exploded = array();
					// Checking for multiple timeslots.
					if ( isset( $booking_data['time_slot'] ) && $booking_data['time_slot'] != '' ) {

						if ( strpos( $booking_data['time_slot'], '<br>' ) !== false ) {
							$time_exploded = explode( '<br>', $booking_data['time_slot'] );
							array_shift( $time_exploded );
						} else {
							$time_exploded[] = $booking_data['time_slot'];
						}
					}

					if ( count( $time_exploded ) > 0 ) {
						$timezone_check = bkap_timezone_check( $global_settings );
						$offset         = ( $timezone_check ) ? bkap_get_offset( $_COOKIE['bkap_offset'] ) : '';

						foreach ( $time_exploded as $key_time => $value_time ) {

							if ( $timezone_check ) {

								$from_time       = $to_time = '';
								$value_time_slot = explode( '-', $value_time );
								$from_time       = bkap_time_convert_asper_timezone( $value_time_slot[0], $offset, 'H:i' );
								$value_time      = $from_time;
								if ( isset( $value_time_slot[1] ) ) {
									$to_time    = bkap_time_convert_asper_timezone( $value_time_slot[1], $offset, 'H:i' );
									$value_time = $from_time . ' - ' . $to_time;
								}

								$event_details['time_slot'] = $value_time;
							} else {
								$event_details['time_slot'] = $value_time;
							}

							$outlook_calendar->bkap_outlook_insert_event( $event_details, $item_id, $user_id, $post_id, false );
						}
					} else {

						$outlook_calendar->bkap_outlook_insert_event( $event_details, $item_id, $user_id, $post_id, false );
					}

					// add an order note, mentioning an event has been created for the item.
					$order      = new WC_Order( $order_id );
					$post_title = $event_details['product_name'];
					$order_note = __( "Booking details for $post_title have been exported to the Outlook Calendar", 'bkap-outlook-calendar' );
					$order->add_order_note( $order_note );
				}
			}
		}
	}

	/**
	 * This function will be called on booking confirmed and it will insert the event to the Outlook Calendar.
	 *
	 * @since    1.0.0
	 * @param    string $_status       Booking Status.
	 * @param    int    $item_id       Item ID.
	 * @param    array  $item_value    Item Value.
	 * @param    int    $order_id      Order ID.
	 * @param    obj    $order         Order Object.
	 * @param    array  $event_details Array of Event Data.
	 */
	public function bkap_outlook_calendar_update_booking_tables_confirmed( $_status, $item_id, $item_value, $order_id, $order, $event_details ) {
		$this->bkap_outlook_calendar_common_insert_event( $item_id, $item_value['product_id'], $order, $event_details );
	}

	/**
	 * This function will be called upon restoring the trashed booking and it will insert the event to the Outlook Calendar.
	 *
	 * @since    1.0.0
	 * @param    int   $item_id       Item ID.
	 * @param    array $item_value    Item Value.
	 * @param    int   $order_id      Order ID.
	 * @param    obj   $order         Order Object.
	 * @param    array $event_details Array of Event Data.
	 */
	public function bkap_outlook_calendar_restore_trashed_booking( $item_id, $item_value, $order_id, $order, $event_details ) {
		bkap_outlook_calendar_common_insert_event( $item_id, $item_value['product_id'], $order, $event_details );
	}

	/**
	 * This function will insert the event to the Outlook Calendar.
	 *
	 * @since    1.0.0
	 * @param    int   $item_id       Item ID.
	 * @param    array $item_value    Item Value.
	 * @param    int   $order_id      Order ID.
	 * @param    obj   $order         Order Object.
	 * @param    array $event_details Array of Event Data.
	 * @param    int   $item_num      Item Number.
	 */
	public function bkap_outlook_calendar_insert_event_to_calendar( $item_id, $item_value, $order_id, $order, $event_details, $item_num ) {
		$this->bkap_outlook_calendar_common_insert_event( $item_id, $item_value['product_id'], $order, $event_details );
	}

	/**
	 * This function will insert the event to the Outlook Calendar when manually creating the booking.
	 *
	 * @since    1.0.0
	 * @param    int   $item_id       Item ID.
	 * @param    int   $product_id    Product ID.
	 * @param    int   $order_id      Order ID.
	 * @param    obj   $order         Order Object.
	 * @param    array $event_details Array of Event Data.
	 */
	public function bkap_outlook_calendar_create_order_booking( $item_id, $product_id, $order_id, $order, $event_details ) {
		$this->bkap_outlook_calendar_common_insert_event( $item_id, $product_id, $order, $event_details );
	}

	/**
	 * This function will delete the event from Outlook Calendar when reallocating the booking.
	 *
	 * @param int   $item_id Item ID.
	 * @param array $item_value Item Value.
	 * @param int   $product_id Product ID.
	 * @param int   $order_id Order ID.
	 * @since    1.0.0
	 */
	public function bkap_outlook_calendar_reallocation_of_booking( $item_id, $item_value, $product_id, $order_id ) {

		$booking_settings  = bkap_setting( $product_id );
		$active_connection = bkap_outlook_calendar_connected( $booking_settings );

		if ( '' !== $active_connection ) {
			$product_id       = ( 'product' === $active_connection ) ? $product_id : 0;
			$user_id          = bkap_get_user_id();
			$outlook_calendar = new BKAP_Outlook_Calendar_OAuth( $product_id, $user_id );
			$outlook_calendar->bkap_outlook_delete_event( $item_id );
		}
	}

	/**
	 * This function will delete the event from Outlook Calendar.
	 *
	 * @param int $item_id Item ID.
	 * @param int $product_id Product ID.
	 * @param int $item_number Item Number.
	 * @since    1.0.0
	 */
	public function bkap_outlook_calendar_delete_event_from_calendar( $item_id, $product_id, $item_number ) {

		$booking_settings  = bkap_setting( $product_id );
		$active_connection = bkap_outlook_calendar_connected( $booking_settings );

		if ( '' !== $active_connection ) {
			$product_id       = ( 'product' === $active_connection ) ? $product_id : 0;
			$user_id          = bkap_get_user_id();
			$outlook_calendar = new BKAP_Outlook_Calendar_OAuth( $product_id, $user_id );
			$outlook_calendar->bkap_outlook_delete_event( $item_id, '', $item_number );
		}
	}

	/**
	 * Register the stylesheets for the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_styles() {

		/**
		 * This function is provided for demonstration purposes only.
		 *
		 * An instance of this class should be passed to the run() function
		 * defined in Bkap_Outlook_Calendar_Loader as all of the hooks are defined
		 * in that particular class.
		 *
		 * The Bkap_Outlook_Calendar_Loader will then create the relationship
		 * between the defined hooks and the functions defined in this
		 * class.
		 */

		wp_enqueue_style( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'css/bkap-outlook-calendar-public.css', array(), $this->version, 'all' );

	}

	/**
	 * Register the JavaScript for the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_scripts() {

		/**
		 * This function is provided for demonstration purposes only.
		 *
		 * An instance of this class should be passed to the run() function
		 * defined in Bkap_Outlook_Calendar_Loader as all of the hooks are defined
		 * in that particular class.
		 *
		 * The Bkap_Outlook_Calendar_Loader will then create the relationship
		 * between the defined hooks and the functions defined in this
		 * class.
		 */

		wp_enqueue_script( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'js/bkap-outlook-calendar-public.js', array( 'jquery' ), $this->version, false );

	}

}
