<?php
namespace ACFWP\Models;

use ACFWP\Abstracts\Abstract_Main_Plugin_Class;

use ACFWP\Interfaces\Model_Interface;

use ACFWP\Models\Objects\Advanced_Coupon;

use ACFWP\Helpers\Plugin_Constants;
use ACFWP\Helpers\Helper_Functions;

if ( !defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Model that houses the logic of the Sort_Coupons module.
 *
 * @since 2.5
 */
class Sort_Coupons implements Model_Interface {

    /*
    |--------------------------------------------------------------------------
    | Class Properties
    |--------------------------------------------------------------------------
    */

    /**
     * Property that holds the single main instance of URL_Coupon.
     *
     * @since 2.5
     * @access private
     * @var Sort_Coupons
     */
    private static $_instance;

    /**
     * Model that houses all the plugin constants.
     *
     * @since 2.5
     * @access private
     * @var Plugin_Constants
     */
    private $_constants;

    /**
     * Property that houses all the helper functions of the plugin.
     *
     * @since 2.5
     * @access private
     * @var Helper_Functions
     */
    private $_helper_functions;

    /**
     * Property that holds the check value if applied coupons on cart already sorted or not.
     * 
     * @since 2.5
     * @access private
     * @var bool
     */
    private $_is_applied_coupons_sorted = false;




    /*
    |--------------------------------------------------------------------------
    | Class Methods
    |--------------------------------------------------------------------------
    */

    /**
     * Class constructor.
     *
     * @since 2.5
     * @access public
     *
     * @param Abstract_Main_Plugin_Class $main_plugin      Main plugin object.
     * @param Plugin_Constants           $constants        Plugin constants object.
     * @param Helper_Functions           $helper_functions Helper functions object.
     */
    public function __construct( Abstract_Main_Plugin_Class $main_plugin , Plugin_Constants $constants , Helper_Functions $helper_functions ) {

        $this->_constants        = $constants;
        $this->_helper_functions = $helper_functions;

        $main_plugin->add_to_all_plugin_models( $this );
        $main_plugin->add_to_public_models( $this );

    }

    /**
     * Ensure that only one instance of this class is loaded or can be loaded ( Singleton Pattern ).
     *
     * @since 2.5
     * @access public
     *
     * @param Abstract_Main_Plugin_Class $main_plugin      Main plugin object.
     * @param Plugin_Constants           $constants        Plugin constants object.
     * @param Helper_Functions           $helper_functions Helper functions object.
     * @return Sort_Coupons
     */
    public static function get_instance( Abstract_Main_Plugin_Class $main_plugin , Plugin_Constants $constants , Helper_Functions $helper_functions ) {

        if ( !self::$_instance instanceof self )
            self::$_instance = new self( $main_plugin , $constants , $helper_functions );

        return self::$_instance;

    }




    /*
    |--------------------------------------------------------------------------
    | Sort Coupons implementation
    |--------------------------------------------------------------------------
    */

    /**
     * Sort coupons before cart totals is calculated.
     * 
     * @since 2.5
     * @access public
     */
    public function sort_coupons_before_calculate_totals() {

        $coupons = \WC()->cart->get_applied_coupons();

        // don't proceed if there are no coupons to sort.
        if ( empty( $coupons ) || $this->_is_applied_coupons_sorted ) return;

        // sort coupons.
        $sorted_coupons = $this->_sort_coupons( $coupons );

        // overwrite cart applied coupons value.
        \WC()->cart->set_applied_coupons( $sorted_coupons );

        // make sure sort only happens once as calculate totals maybe run multiple times in one refresh.
        $this->_is_applied_coupons_sorted = true;
    }

    /**
     * Sort auto apply coupons.
     * 
     * @since 2.5
     * @access public
     * 
     * @param array $auto_coupons List of auto apply coupon IDs.
     * @return array Filtered list of auto apply coupon IDs.
     */
    public function sort_auto_apply_coupons( $auto_coupons ) {

        if ( apply_filters( 'acfw_skip_sort_auto_apply_coupons' , false , $auto_coupons ) || empty( $auto_coupons ) )
            return $auto_coupons;

        return $this->_sort_coupons( $auto_coupons );
    }

    /**
     * Sort auto apply coupons.
     * 
     * @since 2.5
     * @access public
     * 
     * @param array $auto_coupons List of auto apply coupon IDs.
     * @return array Filtered list of auto apply coupon IDs.
     */
    public function sort_apply_notification_coupons( $apply_notif_coupons ) {

        if ( apply_filters( 'acfw_skip_sort_apply_notification_coupons' , false , $apply_notif_coupons ) || empty( $apply_notif_coupons ) )
            return $apply_notif_coupons;

        return $this->_sort_coupons( $apply_notif_coupons );
    }

    /**
     * Sort coupons based on priority (from 1 to 99 or greater).
     * 
     * @since 2.5
     * @access private
     * 
     * @param array $coupons List of coupon codes or ID.
     * @return array Sorted list of coupon codes.
     */
    private function _sort_coupons( $coupons ) {

        $sort = array();

        foreach ( $coupons as $coupon_code ) {

            $coupon = new Advanced_Coupon( $coupon_code );
            $temp   = $coupon->get_advanced_prop( 'coupon_sort_priority' );
            
            $sort[ $coupon_code ] = $temp ? $temp : 30;
        }

        uasort( $sort , function( $a, $b ) {
            if ( $a == $b ) return 0;
            return ( $a > $b ) ? 1 : -1;
        } );

        return array_keys( $sort );
    }




    /*
    |--------------------------------------------------------------------------
    | Fulfill implemented interface contracts
    |--------------------------------------------------------------------------
    */

    /**
     * Execute Sort_Coupons class.
     *
     * @since 2.5
     * @access public
     * @inherit ACFWP\Interfaces\Model_Interface
     */
    public function run() {

        if ( ! \ACFWF()->Helper_Functions->is_module( Plugin_Constants::SORT_COUPONS_MODULE ) )
            return;

        add_action( 'woocommerce_before_calculate_totals' , array( $this , 'sort_coupons_before_calculate_totals' ) , 1 );
        add_filter( 'acfwp_auto_apply_coupons' , array( $this , 'sort_auto_apply_coupons' ) );
        add_filter( 'acfwp_apply_notification_coupons' , array( $this , 'sort_apply_notification_coupons' ) );
    }

}
