<?php
/*
Plugin Name: Blue2 301 Redirects
Plugin URI: https://bitbucket.org/blue2digital/301-redirects
Description: Create a list of URLs that you would like to 301 redirect to another page or site. Now actually works with urlencodable characters.
Version: 2.0.1
Author: Blue2
Author URI: https://blue2.co.uk
*/

new RedirectController();

class RedirectController
{
    public function __construct()
    {
        add_action('admin_menu', [$this, 'createMenu']);
        add_action('init', [$this, 'redirect'], 1);

        if (isset($_POST['301_redirects'])) {
            add_action('admin_init', [$this, 'saveRedirects']);
        }
    }


    /**
     * Generate the link to the options page under settings
     *
     * @access public
     * @return void
     */
    public function createMenu()
    {
        add_options_page('301 Redirects', '301 Redirects', 'manage_options', '301options', [$this, 'addOptionsPage']);
    }

    /**
     * Generate the options page in the wordpress admin
     *
     * @access public
     * @return void
     */
    function addOptionsPage()
    {
        ?>
        <div class="wrap simple_301_redirects">
            <script>
                //todo: This should be enqued
                jQuery(document).ready(function ()
                {
                    jQuery('.wps301-delete')
                        .css({'position': 'relative'})
                        .css({'padding': 0})
                        .css({'margin-top': '5px'})
                        .click(function ()
                        {
                            let isConfirmed = confirm('Delete This Redirect?');
                            if (isConfirmed) {
                                jQuery(this).parent().parent().remove();
                                jQuery('#simple_301_redirects_form').submit();
                            }
                        });

                    jQuery('.simple_301_redirects .documentation').hide().before('<p><a class="reveal-documentation" href="#">Documentation</a></p>');
                    jQuery('.reveal-documentation').click(function ()
                    {
                        jQuery(this).parent().siblings('.documentation').slideToggle();
                        return false;
                    });
                })
                ;
            </script>

            <?php
            if (isset($_POST['301_redirects'])) {
                echo '<div id="message" class="updated"><p>Settings Saved</p></div>';
            }
            ?>

            <h2>301 Redirects</h2>

            <form method="post" id="simple_301_redirects_form" action="options-general.php?page=301options&savedata=true">

                <?php wp_nonce_field('save_redirects', '301_nonce'); ?>

                <table class="widefat">
                    <thead>
                    <tr>
                        <th colspan="2">Request</th>
                        <th colspan="2">Destination</th>
                    </tr>
                    </thead>
                    <tbody>
                    <tr>
                        <td colspan="2">
                            <small>example: /about.htm</small>
                        </td>
                        <td colspan="2">
                            <small>example: <?= get_option('home'); ?>/about/</small>
                        </td>
                    </tr>
                    <?= $this->expand_redirects(); ?>
                    <tr>
                        <td style="width:35%;"><input type="text" name="301_redirects[request][]" value="" style="width:99%;"/></td>
                        <td style="width:2%;">&raquo;</td>
                        <td style="width:60%;"><input type="text" name="301_redirects[destination][]" value="" style="width:99%;"/></td>
                        <td></td>
                    </tr>
                    </tbody>
                </table>

                <?php $wildcard_checked = (get_option('301_redirects_wildcard') === 'true' ? ' checked="checked"' : ''); ?>
                <p><input type="checkbox" name="301_redirects[wildcard]" id="wps301-wildcard"<?= $wildcard_checked; ?> /><label for="wps301-wildcard"> Use Wildcards?</label></p>
                <?php $this->documentationLayout(); ?>
                <p class="submit"><input type="submit" name="submit_301" class="button-primary" value="Save Changes"/></p>
            </form>

        </div>
        <?php
    }

    private function documentationLayout()
    {
        ?>
        <div class="documentation">
            <h2>Documentation</h2>
            <h3>Simple Redirects</h3>
            <p>Simple redirects work similar to the format that Apache uses: the request should be relative to your WordPress root. The destination can be either a full URL to
                any page on the web, or relative to your WordPress root.</p>
            <h4>Example</h4>
            <ul>
                <li><strong>Request:</strong> /old-page/</li>
                <li><strong>Destination:</strong> /new-page/</li>
            </ul>

            <h3>Wildcards</h3>
            <p>To use wildcards, put an asterisk (*) after the folder name that you want to redirect.</p>
            <h4>Example</h4>
            <ul>
                <li><strong>Request:</strong> /old-folder/*</li>
                <li><strong>Destination:</strong> /redirect-everything-here/</li>
            </ul>

            <p>You can also use the asterisk in the destination to replace whatever it matched in the request if you like. Something like this:</p>
            <h4>Example</h4>
            <ul>
                <li><strong>Request:</strong> /old-folder/*</li>
                <li><strong>Destination:</strong> /some/other/folder/*</li>
            </ul>
            <p>Or:</p>
            <ul>
                <li><strong>Request:</strong> /old-folder/*/content/</li>
                <li><strong>Destination:</strong> /some/other/folder/*</li>
            </ul>
        </div>
        <?php
    }

    /**
     * expand_redirects function
     * utility function to return the current list of redirects as form fields
     * @access public
     * @return string <html>
     */
    function expand_redirects()
    {
        $redirects = get_option('301_redirects');
        $output = '';
        if (!empty($redirects)) {
            foreach ($redirects as $request => $destination) {
                $output .= '
					<tr>
						<td><input type="text" name="301_redirects[request][]" value="' . $request . '" style="width:99%" /></td>
						<td>&raquo;</td>
						<td><input type="text" name="301_redirects[destination][]" value="' . $destination . '" style="width:99%;" /></td>
						<td><button type="button" class="notice-dismiss wps301-delete"><span class="screen-reader-text">Delete</span></button></td>
					</tr>
					';
            }
        }
        return $output;
    }

    /**
     * Save the redirects from the options page to the database
     *
     * @access public
     * @return void
     */
    function saveRedirects()
    {
        if (!current_user_can('manage_options')) {
            wp_die('You do not have sufficient permissions to access this page.');
        }

        if (check_admin_referer('save_redirects', '301_nonce') !== 1) {
            wp_die('Saving redirects must be initiated from the 301 Redirect page.');
        }


        $data = $_POST['301_redirects'];

        $redirects = [];

        for ($i = 0; $i < sizeof($data['request']); ++$i) {

            $request = trim(sanitize_text_field($data['request'][$i]));
            $destination = trim(sanitize_text_field($data['destination'][$i]));

            if ($request !== '' && $destination !== '') {
                $redirects[$request] = $destination;
            }
        }

        update_option('301_redirects', $redirects);

        if (isset($data['wildcard'])) {
            update_option('301_redirects_wildcard', 'true');
        } else {
            delete_option('301_redirects_wildcard');
        }
    }

    /**
     * Read the list of redirects and if the current page is found in the list, redirect
     *
     * @access public
     * @return void
     */
    function redirect()
    {
        $requestedPage = rtrim($_SERVER['REQUEST_URI'], '/');

        if ($this->isAdminPage($requestedPage)) {
            return;
        }

        $redirects = get_option('301_redirects');
        if (!empty($redirects)) {

            $isWildcardAllowed = get_option('301_redirects_wildcard') === 'true';


            $redirectTo = '';

            foreach ($redirects as $request => $destination) {

                if ($isWildcardAllowed && strpos($request, '*') !== false) {

                    $pattern = rtrim($request, '/');
                    $pattern = str_replace('*', '(.*)', $pattern);
                    $pattern = str_replace('/', '\/', $pattern);
                    $pattern = '/^' . $pattern . '/';

                    $destination = str_replace('*', '$1', $destination);
                    $output = preg_replace($pattern, $destination, $requestedPage);

                    if ($output !== $requestedPage) {
                        // pattern matched, perform redirect
                        $redirectTo = $output;
                    }

                } elseif (urldecode($requestedPage) == urldecode(rtrim($request, '/'))) {
                    // simple comparison redirect
                    $redirectTo = $destination;
                }

                // redirect. the second condition here prevents redirect loops as a result of wildcards.
                if ($redirectTo !== '' && trim($redirectTo, '/') !== trim($requestedPage, '/')) {
                    // check if destination needs the domain prepended
                    if (strpos($redirectTo, '/') === 0) {
                        $redirectTo = home_url() . $redirectTo;
                    }
                    header('HTTP/1.1 301 Moved Permanently');
                    header('Location: ' . $redirectTo);
                    exit();
                } else {
                    unset($redirects);
                }
            }
        }
    }

    private function isAdminPage($requestedPage)
    {
        return strpos($requestedPage, '/wp-login') !== false || strpos($requestedPage, '/wp-admin') !== false;
    }

}

